/*=================================================================================================
Filename:    vos_dlink.c
Description: API of vos double link 
Data:         Initial revision. Bob@2020-03-30
==================================================================================================*/
#ifdef __cplusplus 
    extern "C" { 
#endif

#include "vos_if.h"
#include "vos_innr.h"

int vos_dlkShow(HVOSDLK hDlk, DLK_NODE *pNode)
{
	int nNodeCnt=0;
	VosDlk_S *pVosDlk = NULL;
	DLK_NODE *pShow, *pHead;
	unsigned long *pDataAddr;

	if((hDlk == 0) && (pNode == NULL))return 0;

	if(hDlk != 0)
	{
		pVosDlk = (VosDlk_S *)hDlk;
		VOS_ASSERT(pVosDlk->nMagicId == VOS_DLK_MAGIC_ID);
		vosPrintf("----DLK: 0x%08x, size=%d, pHead=0x%x\r\n", pVosDlk, pVosDlk->nDataSize, pVosDlk->pHead);
		pHead = pVosDlk->pHead;
	}
	else
	{
		pHead = pNode;
	}
	
	DLK_WALK_LOOP_START(pShow, pHead)
	{
		VOS_ASSERT(pShow->nHeadId == VOS_DLK_HEAD_ID);
		nNodeCnt++;
		pDataAddr = (unsigned long *)(pShow + 1);
		vosPrintf("    node%03d[0x%08x]: prev=0x%08x, next=0x%08x [data]:0x%08x 0x%08x\r\n", nNodeCnt, pShow, pShow->prev, pShow->next, *pDataAddr, *(pDataAddr+1));
	}
	DLK_WALK_LOOP_END(pShow, pHead)

	return nNodeCnt;
}

HVOSDLK vos_dlkCreate(int nDataSize, int mutexFlag, pfDlFunc pfGetKey)
{
	VosDlk_S *pVosDlk = NULL;
	char dlqName[8] = {0,};
	static int g_nDlqCnt = 0;

	pVosDlk = (VosDlk_S *)vos_malloc(sizeof(VosDlk_S));
	VOS_ASSERT(pVosDlk != NULL);
	memset(pVosDlk, 0, sizeof(VosDlk_S));

	sprintf(dlqName, "dlq_%d", g_nDlqCnt);g_nDlqCnt++;
	pVosDlk->nMagicId = VOS_DLK_MAGIC_ID;
	pVosDlk->nDataSize = nDataSize;
	pVosDlk->pfGetKey = pfGetKey;
	pVosDlk->nNodeCnt = 0;
	if(mutexFlag)
	{
		pVosDlk->hMutex = vos_mutexCreate(dlqName, 0);
		VOS_ASSERT(pVosDlk->hMutex != 0);
	}

	return (HVOSDLK)pVosDlk;
}

//nSortFlag: 0-no sort; 1-increase; 2-decrease
void vos_dlkSetSort(HVOSDLK hDlk, pfDlFunc pfGetKey, unsigned long nSortFlag)
{
	VosDlk_S *pVosDlk = NULL;

	pVosDlk = (VosDlk_S *)hDlk;
	VOS_ASSERT(pVosDlk->nMagicId == VOS_DLK_MAGIC_ID);
	
	pVosDlk->pfGetKey = pfGetKey;
	pVosDlk->nSortFlag = nSortFlag;
	return;
}

void vos_dlkAddMutex(HVOSDLK hDlk, char *pName)
{
	VosDlk_S *pVosDlk = NULL;

	pVosDlk = (VosDlk_S *)hDlk;
	VOS_ASSERT(pVosDlk->nMagicId == VOS_DLK_MAGIC_ID);

	if(pVosDlk->hMutex == 0)
	{
		pVosDlk->hMutex = vos_mutexCreate(pName, 0);
		VOS_ASSERT(pVosDlk->hMutex != 0);
	}

}

int vos_dlkClean(HVOSDLK hDlk)
{
	VosDlk_S *pVosDlk = NULL;
	DLK_NODE *pNext, *pDel;

	pVosDlk = (VosDlk_S *)hDlk;
	VOS_ASSERT(pVosDlk->nMagicId == VOS_DLK_MAGIC_ID);

	if(pVosDlk->hMutex)vos_mutexLock(pVosDlk->hMutex);

	pDel = pVosDlk->pHead;
	do
	{
		if(pDel == NULL)break;
		pNext = pDel->next;
		pDel->nHeadId == VOS_DLK_HEAD_ID + 1;
		vos_free(pDel);
		pDel = pNext;		
	}while(pDel != pVosDlk->pHead);
		
	pVosDlk->pHead = NULL;
	if(pVosDlk->hMutex)vos_mutexUnlock(pVosDlk->hMutex);
	return 0;
}

int vos_dlkDestroy(HVOSDLK hDlk)
{
	VosDlk_S *pVosDlk = NULL;

	pVosDlk = (VosDlk_S *)hDlk;
	VOS_ASSERT(pVosDlk->nMagicId == VOS_DLK_MAGIC_ID);

	vos_dlkClean(hDlk);
	
	if(pVosDlk->hMutex)vos_mutexClose(pVosDlk->hMutex);
	pVosDlk->nMagicId = VOS_DLK_MAGIC_ID+1;
	vos_free(pVosDlk);
	return 0;
}


DLK_NODE * vos_dlkIn ( DLK_NODE * a,  DLK_NODE * b)
{
   a->next->prev = b->prev;
   b->prev->next = a->next;
   a->next = b;
   b->prev = a;
   return b;
}

void vos_dlkRm(DLK_NODE *qp)
{
	qp->next->prev=qp->prev;
	qp->prev->next=qp->next;
	return;
}

//if pNode is NULL, insert to tail
DLK_NODE * vos_dlkInsert(HVOSDLK hDlk, DLK_NODE *pNode, void *pData)
{
	VosDlk_S *pVosDlk = NULL;
	DLK_NODE *pNew;
	unsigned long nDataAddr;

	pVosDlk = (VosDlk_S *)hDlk;
	VOS_ASSERT(pVosDlk->nMagicId == VOS_DLK_MAGIC_ID);
	VOS_ASSERT(pData != NULL);

	//make new node
	pNew = (DLK_NODE *)vos_malloc(sizeof(DLK_NODE) + pVosDlk->nDataSize);
	VOS_ASSERT(pNew != NULL);
	pNew->prev = pNew;
	pNew->next = pNew;
	pNew->nHeadId = VOS_DLK_HEAD_ID;
	nDataAddr = (unsigned long)(pNew + 1);
	memcpy((void *)nDataAddr, pData, pVosDlk->nDataSize);

	if(pVosDlk->hMutex)vos_mutexLock(pVosDlk->hMutex);
	
	//add to doubleLinkQue
	if(pVosDlk->pHead == NULL) //empty
	{
		pVosDlk->pHead = pNew;
	}
	else
	{
		if(pNode == (DLK_NODE *)-1) //insert to head		
		{
			pNode = pVosDlk->pHead->prev; //tail			
			vos_dlkIn(pNode, pNew);
			pVosDlk->pHead = pNew;
		}
		else
		{
			if(pNode == NULL)pNode = pVosDlk->pHead->prev; //insert to tail			
			vos_dlkIn(pNode, pNew);
		}
	}

	if(pVosDlk->hMutex)vos_mutexUnlock(pVosDlk->hMutex);
	
	return pNew;
}

void * vos_dlkInsertData(HVOSDLK hDlk, void *pData)
{
	DLK_NODE *pNewNode;
	void *pNewData = NULL;

	pNewNode = vos_dlkInsert(hDlk, NULL, pData);
	if(NULL != pNewNode)pNewData = (void *)(pNewNode + 1);

	return pNewData;
}

void * vos_dlkInsertDataSort(HVOSDLK hDlk, void *pData, unsigned long nSortKey)
{
	VosDlk_S *pVosDlk = NULL;
	DLK_NODE *pNewNode, *pSortNode;
	void *pNewData = NULL;
	
	pVosDlk = (VosDlk_S *)hDlk;
	VOS_ASSERT(pVosDlk->nMagicId == VOS_DLK_MAGIC_ID);	
	if(pVosDlk->hMutex)vos_mutexLock(pVosDlk->hMutex);
	
	pSortNode = vos_dlkFind(hDlk, nSortKey);
	if(NULL == pSortNode)
	{
		pSortNode = (DLK_NODE *)-1; //insert to head
	}
	pNewNode = vos_dlkInsert(hDlk, pSortNode, pData);
	if(NULL != pNewNode)pNewData = (void *)(pNewNode + 1);

	if(pVosDlk->hMutex)vos_mutexUnlock(pVosDlk->hMutex);
	return pNewData;
}

DLK_NODE * vos_dlkInsertNode(HVOSDLK hDlk, DLK_NODE *pNode, DLK_NODE *pNew)
{
	VosDlk_S *pVosDlk = NULL;
	unsigned long nDataAddr;

	pVosDlk = (VosDlk_S *)hDlk;
	VOS_ASSERT(pVosDlk->nMagicId == VOS_DLK_MAGIC_ID);
	VOS_ASSERT(pNew != NULL);

	if(pVosDlk->hMutex)vos_mutexLock(pVosDlk->hMutex);
	
	//add to doubleLinkQue
	if(pNode == NULL) //empty
	{
		if(pVosDlk->pHead == NULL)
			pVosDlk->pHead = pNew;
		else
			vos_dlkIn(pVosDlk->pHead->prev, pNew); //add to tail
	}
	else
	{
		vos_dlkIn(pNode, pNew);
	}

	if(pVosDlk->hMutex)vos_mutexUnlock(pVosDlk->hMutex);
	
	return pNew;
}

DLK_NODE * vos_dlkFind(HVOSDLK hDlk, unsigned long nKey)
{
	VosDlk_S *pVosDlk = NULL;
	DLK_NODE *pHead, *pNode, *pResult = NULL;

	pVosDlk = (VosDlk_S *)hDlk;
	VOS_ASSERT(pVosDlk != NULL);
	VOS_ASSERT(pVosDlk->nMagicId == VOS_DLK_MAGIC_ID);
	if(pVosDlk->pfGetKey == NULL)return NULL;

	pHead = pVosDlk->pHead;	
	if(pVosDlk->hMutex)vos_mutexLock(pVosDlk->hMutex);

	DLK_WALK_LOOP_START(pNode, pHead)
	{
		VOS_ASSERT(pNode->nHeadId == VOS_DLK_HEAD_ID);
		if(0 == pVosDlk->nSortFlag) //no sort
		{
			if(pVosDlk->pfGetKey((void *)(pNode+1), 0) == nKey)
			{
				pResult = pNode;
				break;
			}
		}
		else if(1 == pVosDlk->nSortFlag) //increase
		{	
			if(pVosDlk->pfGetKey((void *)(pNode+1), 0) < (nKey+1))
				pResult = pNode;
			else
				break;
		}
		else //decrease
		{	
			if(pVosDlk->pfGetKey((void *)(pNode+1), 0) > nKey)
				pResult = pNode;
			else
				break;
		}
	}
	DLK_WALK_LOOP_END(pNode, pHead)
	
	if(pVosDlk->hMutex)vos_mutexUnlock(pVosDlk->hMutex);

	return pResult;
}

void vos_dlkDelete(HVOSDLK hDlk, DLK_NODE *pNode)
{
	VosDlk_S *pVosDlk = NULL;
	DLK_NODE *pHead;

	pVosDlk = (VosDlk_S *)hDlk;
	VOS_ASSERT(pVosDlk->nMagicId == VOS_DLK_MAGIC_ID);
	VOS_ASSERT(pNode != NULL);
	VOS_ASSERT(pNode->nHeadId == VOS_DLK_HEAD_ID);

	if(pVosDlk->pHead == NULL)return;

	if(pVosDlk->hMutex)vos_mutexLock(pVosDlk->hMutex);
	if(pNode == pVosDlk->pHead)
	{
		if(pNode->next == pNode)
			pVosDlk->pHead = NULL; //last one
		else
			pVosDlk->pHead = pNode->next;
	}
	vos_dlkRm(pNode);
	if(pVosDlk->hMutex)vos_mutexUnlock(pVosDlk->hMutex);

	pNode->nHeadId == VOS_DLK_HEAD_ID + 1;
	vos_free((void *)pNode);
	return;
}

int vos_dlkDeleteByKey(HVOSDLK hDlk, unsigned long nKey)
{
	DLK_NODE *pNode;

	pNode = vos_dlkFind(hDlk, nKey);
	if(pNode == NULL)return 1;

	vos_dlkDelete(hDlk, pNode);
	return 0;	
}

DLK_NODE * vos_dlkGetHead(HVOSDLK hDlk)
{
	VosDlk_S *pVosDlk = NULL;

	pVosDlk = (VosDlk_S *)hDlk;
	VOS_ASSERT(pVosDlk->nMagicId == VOS_DLK_MAGIC_ID);
	return pVosDlk->pHead;
}

DLK_NODE * vos_dlkGetTail(HVOSDLK hDlk)
{
	VosDlk_S *pVosDlk = NULL;

	pVosDlk = (VosDlk_S *)hDlk;
	VOS_ASSERT(pVosDlk->nMagicId == VOS_DLK_MAGIC_ID);

	if(pVosDlk->pHead == NULL)return NULL;
	
	return pVosDlk->pHead->prev;
}

DLK_NODE * vos_dlkGetNext(HVOSDLK hDlk, DLK_NODE *pNode)
{
	VosDlk_S *pVosDlk = NULL;

	pVosDlk = (VosDlk_S *)hDlk;
	VOS_ASSERT(pVosDlk->nMagicId == VOS_DLK_MAGIC_ID);

	if(pVosDlk->pHead == pNode->next)return NULL;
	
	return pNode->next;
}

int vos_dlkGetCnt(HVOSDLK hDlk)
{
	int nNodeCnt=0;
	VosDlk_S *pVosDlk = NULL;
	DLK_NODE *pShow;

	if(hDlk == 0)return 0;
	pVosDlk = (VosDlk_S *)hDlk;
	VOS_ASSERT(pVosDlk->nMagicId == VOS_DLK_MAGIC_ID);

	DLK_WALK_LOOP_START(pShow, pVosDlk->pHead)
	nNodeCnt++;
	DLK_WALK_LOOP_END(pShow, pVosDlk->pHead)

	return nNodeCnt;
}

//return total memory size
int vos_dlkSave2Mem(HVOSDLK hDlk, void *pMem)
{
	VosDlk_S *pSrcDlk = NULL;
	VosDlk_S *pDesDlk = NULL;
	DLK_NODE *pSrc, *pDes;
	int nNodeCnt=0, nTotalSize;

	VOS_ASSERT(hDlk != 0);
	VOS_ASSERT(pMem != NULL);
	pSrcDlk = (VosDlk_S *)hDlk;
	pDesDlk = (VosDlk_S *)pMem;
	VOS_ASSERT(pSrcDlk->nMagicId == VOS_DLK_MAGIC_ID);

	memcpy((void *)pDesDlk, (void *)pSrcDlk, sizeof(VosDlk_S));
	pDesDlk->pHead = NULL;
	
	pDes = (DLK_NODE *)(pDesDlk + 1);

	if(pSrcDlk->hMutex)vos_mutexLock(pSrcDlk->hMutex);
	DLK_WALK_LOOP_START(pSrc, pSrcDlk->pHead)
	{
		memcpy((void *)pDes, (void *)pSrc, sizeof(DLK_NODE)+pSrcDlk->nDataSize);
		pDes->next = pDes;pDes->prev = pDes;
		vos_dlkInsertNode((HVOSDLK)pDesDlk, NULL, pDes);
		pDes = (DLK_NODE *)((unsigned int)pDes +(sizeof(DLK_NODE)+pSrcDlk->nDataSize));
		nNodeCnt++;
	}
	DLK_WALK_LOOP_END(pSrc, pSrcDlk->pHead)
	if(pSrcDlk->hMutex)vos_mutexUnlock(pSrcDlk->hMutex);

	nTotalSize = nNodeCnt*(sizeof(DLK_NODE)+pSrcDlk->nDataSize) + sizeof(VosDlk_S);
	return nTotalSize;
}

HVOSDLK vos_dlkRecoverFromMem(void *pMem)
{
	VosDlk_S *pSrcDlk = NULL;
	VosDlk_S *pDesDlk = NULL;
	DLK_NODE *pSrc, *pDes;
	int nNodeCnt=0;

	VOS_ASSERT(pMem != NULL);
	pSrcDlk = (VosDlk_S *)pMem;
	VOS_ASSERT(pSrcDlk->nMagicId == VOS_DLK_MAGIC_ID);
	
	pDesDlk = (VosDlk_S *)vos_malloc(sizeof(VosDlk_S));
	VOS_ASSERT(pDesDlk != NULL);
	memcpy((void *)pDesDlk, (void *)pSrcDlk, sizeof(VosDlk_S));
	pDesDlk->pHead = NULL;

	if(pSrcDlk->hMutex)vos_mutexLock(pSrcDlk->hMutex);
	DLK_WALK_LOOP_START(pSrc, pSrcDlk->pHead)
	{
		vos_dlkInsert((HVOSDLK)pDesDlk, NULL, (void *)(pSrc+1));
		nNodeCnt++;
	}
	DLK_WALK_LOOP_END(pSrc, pSrcDlk->pHead)
	if(pSrcDlk->hMutex)vos_mutexUnlock(pSrcDlk->hMutex);

	return (HVOSDLK)pDesDlk;
}

//copy dlk(include all nodes, pNode is the head) to a new one,  return the head node of new dlk.
DLK_NODE * vos_dlkCp2New(HVOSDLK hDlk, DLK_NODE *pNode)
{
	VosDlk_S *pSrcDlk = (VosDlk_S *)hDlk;
	VosDlk_S sNewDlk;
	HVOSDLK hDlkNew;
	DLK_NODE *pSrc, *pDes;

	VOS_ASSERT(pSrcDlk != NULL);
	
	if(pSrcDlk->hMutex)vos_mutexLock(pSrcDlk->hMutex);
	memcpy(&sNewDlk, pSrcDlk, sizeof(VosDlk_S));
	sNewDlk.pHead = NULL;
	hDlkNew = (HVOSDLK)&sNewDlk;

	DLK_WALK_LOOP_START(pSrc, pNode)
	{
		VOS_ASSERT(pSrc->nHeadId == VOS_DLK_HEAD_ID);
		pDes = vos_dlkInsert(hDlkNew, NULL, (void *)(pSrc+1));
	}
	DLK_WALK_LOOP_END(pSrc, pNode)
	if(pSrcDlk->hMutex)vos_mutexUnlock(pSrcDlk->hMutex);
		
	return sNewDlk.pHead;
}

int test_dlkFind(void *pData, unsigned long nKey)
{
	if(*(unsigned long *)pData == nKey)return 1;
	return 0;
}

int test_dlk(int nFlag, int nPar1, int nPar2, int nPar3)
{
	HVOSDLK hDlk;
	DLK_NODE *pNode, *pNode1;
	int nRet;
	int buff[64] = {0,};

	switch(nFlag)
	{
		case 0:
			nRet = vos_dlkShow((HVOSDLK)nPar1, NULL);
			break;
			
		case 1:
			nRet = vos_dlkShow(0, (DLK_NODE *)nPar1);
			break;
			
		case 2:
			hDlk = (int)vos_dlkCreate(32, 0, test_dlkFind);
			buff[0] = 0x11; buff[1] = ~buff[0];
			pNode = vos_dlkInsert(hDlk, NULL, (void *)buff);
			buff[0] = 0x13; buff[1] = ~buff[0];
			pNode1 = vos_dlkInsert(hDlk, NULL, (void *)buff);
			buff[0] = 0x12; buff[1] = ~buff[0];
			pNode1 = vos_dlkInsert(hDlk, pNode, (void *)buff);
			buff[0] = 0x14; buff[1] = ~buff[0];
			pNode1 = vos_dlkInsert(hDlk, NULL, (void *)buff);

			vos_dlkShow(hDlk, NULL);
			nRet = (int)hDlk;
			break;

		case 3:
			pNode = vos_dlkFind((HVOSDLK)nPar1, 0x12);
			vos_dlkShow(0, pNode);
			break;
			
		default:
			break;
	}

    return nRet;
}


#ifdef __cplusplus 
    }
#endif

