/******************************************************************************
 * Copyright (c) 2007-2016 Realtek Semiconductor Corp. All Rights Reserved.
 * 
 * This program is dual-licensed under both the GPL version 2 and BSD
 * license. Either license may be used at your option.
 * 
 * 
 * License
 * 
 * 
 * GPL v2:
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 * 
 * 
 * Alternatively, this software may be distributed, used, and modified
 * under the terms of BSD license:
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * 
 * 3. Neither the name(s) of the above-listed copyright holder(s) nor the
 * names of its contributors may be used to endorse or promote products
 * derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************/

#include "wmac.h"
#include "pltfm_api.h"
#include "ui_info.h"
#include "../Header/GeneralDef.h"

enum wmac_ofld_feature_id {
	OFLD_F_CFG_PARAM,
};

struct wmac_ctrl_info_fmt {
	u16 macid;
	u32 gtab_id;
	u32 antnum;
};

u8 wmac_init_halmac_api(struct ui_adapter_e *adapter)
{
	enum halmac_interface intf;
	enum halmac_ret_status ret;

	switch (adapter->intf_info.intf) {
	case HOST_INTF_USB:
		intf = HALMAC_INTERFACE_USB;
		break;
	case HOST_INTF_SDIO:
		intf = HALMAC_INTERFACE_SDIO;
		break;
	case HOST_INTF_PCI:
		intf = HALMAC_INTERFACE_PCIE;
		break;
	case HOST_INTF_MASS_STORAGE:
		intf = HALMAC_INTERFACE_USB;
		break;
	default:
		MFC_MSG_ERR("undefined intf!!\n");
		return 0;
	}

	ret = halmac_init_adapter(adapter, &pltfm_api_cb, intf,
				  &(adapter->halmac_adapter),
				  &(adapter->halmac_api));
	if (ret != HALMAC_RET_SUCCESS) {
		MFC_MSG_ERR("halmac_init_adapter!!\n");
		return 0;
	}

	return 1;
}

u8 wmac_fine_tune_intf(struct ui_adapter_e *adapter)
{
	struct halmac_adapter *mac_adapter = adapter->halmac_adapter;
	struct halmac_api *api = adapter->halmac_api;
	enum halmac_ret_status ret;

	ret = api->halmac_interface_integration_tuning(mac_adapter);
	if (ret != HALMAC_RET_SUCCESS) {
		MFC_MSG_ERR("intf integration tuning!!\n");
		return 0;
	}

	ret = api->halmac_phy_cfg(mac_adapter, HALMAC_INTF_PHY_PLATFORM_ALL);
	if (ret != HALMAC_RET_SUCCESS) {
		MFC_MSG_ERR("phy cfg!!\n");
		return 0;
	}

	return 1;
}

static u8 wmac_intf_info(struct ui_adapter_e *adapter)
{
	struct halmac_adapter *mac_adapter = adapter->halmac_adapter;
	struct halmac_api *api = adapter->halmac_api;
	enum halmac_ret_status ret = HALMAC_RET_SUCCESS;

	if (adapter->intf_info.intf == HOST_INTF_USB) {
		u8 bulkout_num;

		bulkout_num = adapter->intf_info.usb_handle.bulkout_num;
		api->halmac_set_bulkout_num(mac_adapter, bulkout_num);
	} else if (adapter->intf_info.intf == HOST_INTF_SDIO) {
#if HALMAC_SDIO_SUPPORT
		struct halmac_sdio_hw_info sdio_info;
		enum halmac_sdio_cmd53_4byte_mode mode;

		sdio_info.spec_ver = HALMAC_SDIO_SPEC_VER_3_00;
		sdio_info.clock_speed = 208;
		sdio_info.block_size = 512;
		ret = api->halmac_sdio_hw_info(mac_adapter, &sdio_info);
		if (ret != HALMAC_RET_SUCCESS) {
			MFC_MSG_ERR("sdio hw info!!\n");
			return 0;
		}

		mode = HALMAC_SDIO_CMD53_4BYTE_MODE_RW;
		ret = api->halmac_sdio_cmd53_4byte(mac_adapter, mode);
		if (ret != HALMAC_RET_SUCCESS) {
			MFC_MSG_ERR("cmd53 4byte!!\n");
			return 0;
		}
#endif
	}

	return 1;
}

static u8 wmac_reset_mac(struct ui_adapter_e *adapter)
{
	struct halmac_adapter *mac_adapter = adapter->halmac_adapter;
	struct halmac_api *api = adapter->halmac_api;
	enum halmac_ret_status ret = HALMAC_RET_SUCCESS;

	ret = api->halmac_mac_power_switch(mac_adapter, HALMAC_MAC_POWER_OFF);
	if(ret != HALMAC_RET_SUCCESS) {
		MFC_MSG_ERR("mac reset -- off!!\n");
		return 0;
	}

	ret = api->halmac_mac_power_switch(mac_adapter, HALMAC_MAC_POWER_ON);
	if(ret != HALMAC_RET_SUCCESS) {
		MFC_MSG_ERR("mac reset -- on!!\n");
		return 0;
	}

	return 1;
}

static u8 wmac_init_system(struct ui_adapter_e *adapter)
{
	struct halmac_adapter *mac_adapter = adapter->halmac_adapter;
	struct halmac_api *api = adapter->halmac_api;
	enum halmac_ret_status ret = HALMAC_RET_SUCCESS;

	ret = api->halmac_pre_init_system_cfg(mac_adapter);
	if (ret != HALMAC_RET_SUCCESS) {
		if (ret == HALMAC_RET_WLAN_MODE_FAIL) {
			MFC_MSG_BOX("[WARN] In test mode");
		} else {
			MFC_MSG_ERR("pre-init system!!\n");
			return 0;
		}
	}

	ret = api->halmac_mac_power_switch(mac_adapter, HALMAC_MAC_POWER_ON);
	if(ret != HALMAC_RET_SUCCESS) {
		if (ret == HALMAC_RET_PWR_UNCHANGE) {
			if(!wmac_reset_mac(adapter))
				return 0;
		} else {
			MFC_MSG_ERR("power on fail!!\n");
			return 0;
		}
	}

	ret = api->halmac_init_system_cfg(mac_adapter);
	if (ret != HALMAC_RET_SUCCESS) {
		MFC_MSG_ERR("init system!!\n");
		return 0;
	}

	return 1;
}

static void wmac_dcpu_fwconf(struct ui_adapter_e *adapter)
{
	struct fwcfg_info cfg_info;
	struct halmac_adapter *mac_adapter = adapter->halmac_adapter;
	struct halmisc_api *misc_api = adapter->halmac_api->halmisc_api;
	struct mac_setting_info *mac_info = &(adapter->mac_info);

	memcpy(cfg_info.mac_addr.addr, mac_info->addr.src_addr, 6);

	switch (mac_info->test_opt.lb_mode) {
	case MFC_TRX_MODE_NORM:
		cfg_info.mode = HALMAC_TRX_MODE_NORMAL;
		break;
	case MFC_TRX_MODE_LB:
		cfg_info.mode = HALMAC_TRX_MODE_LOOPBACK;
		break;
	case MFC_TRX_MODE_DLLB:
		cfg_info.mode = HALMAC_TRX_MODE_DELAY_LOOPBACK;
		break;
	default:
		return;
	}

	if (misc_api->halmac_set_fwconf_sec)
		misc_api->halmac_set_fwconf_sec(mac_adapter, &cfg_info);
}

static u8 wmac_download_fw(struct ui_adapter_e *adapter)
{
	u8 *buf;
	u32 size;
	TCHAR *file_dir;
	CFile file_ctrl;
	enum halmac_ret_status ret;
	struct halmac_adapter *mac_adapter = adapter->halmac_adapter;
	struct halmac_api *api = adapter->halmac_api;
	struct halmisc_api *misc_api = adapter->halmac_api->halmisc_api;

	adapter->dlfw_ok = 0;

	switch (adapter->chip_type.chip_id) {
	case _CHIP_ID_8822B:
		file_dir = TEXT("./HWVerifyData/RTL8822B/RTL8822Bfw_NIC.bin");
		break;
	case _CHIP_ID_8821C:
		file_dir = TEXT("./HWVerifyData/RTL8821C/RTL8821Cfw_NIC.bin");
		break;
	case _CHIP_ID_8824B:
		file_dir = TEXT("./HWVerifyData/RTL8824B/RTL8824Bfw_NIC.bin");
		break;
	case _CHIP_ID_8197F:
		file_dir = TEXT("./HWVerifyData/RTL8197F/RTL8197Ffw_NIC.bin");
		break;
	case _CHIP_ID_8822C:
		file_dir = TEXT("./HWVerifyData/RTL8822C/RTL8822Cfw_NIC.bin");
		break;
	case _CHIP_ID_8812F:
		file_dir = TEXT("./HWVerifyData/RTL8812F/RTL8812Ffw_NIC.bin");
		break;
	default:
		MFC_MSG_BOX("dlfw -- unsupported chip!!");
		return 0;
	}
#if defined(HALMAC_DATA_CPU_EN)
	wmac_dcpu_fwconf(adapter);
#endif
	if (!file_ctrl.Open(file_dir, CFile::modeRead|CFile::typeBinary)) {
		MFC_MSG_BOX("Open FW bin file!!");
		return 0;
	}

	size = (u32)file_ctrl.GetLength();
	buf = (u8 *)malloc(size);
	if (!buf) {
		MFC_MSG_ERR("malloc fw buf!!\n");
		return 0;
	}

	file_ctrl.Read(buf, size);
	file_ctrl.Close();

	ret = api->halmac_download_firmware(mac_adapter, buf, size);
	if(ret != HALMAC_RET_SUCCESS) {
		MFC_MSG_ERR("Download FW!!\n");
		return 0;
	}

	adapter->dlfw_ok = 1;

	return 1;
}

static void wmac_clear_ofld_feature_status(struct ui_adapter_e *adapter,
					      enum wmac_ofld_feature_id f_id)
{
	struct c2h_event_info *c2h_info = &(adapter->c2h_info);

	switch (f_id) {
	case OFLD_F_CFG_PARAM:
		c2h_info->cfg_param = 0;
		break;
	default:
		break;
	}
}

static u8 wmac_c2h_flag_polling(u8 *flag, u32 cnt)
{
	while (!(*flag)) {
		cnt--;
		if (!cnt)
			return 0;
		Sleep(1);
	}

	return 1;
}

static u8 wmac_get_ofld_feature_status(struct ui_adapter_e *adapter,
				       enum wmac_ofld_feature_id f_id)
{
	u32 cnt = 2000;
	enum halmac_cmd_process_status proc_status;
	enum halmac_ret_status ret;
	struct halmac_adapter *mac_adapter = adapter->halmac_adapter;
	struct halmac_api *api = adapter->halmac_api;

	if (adapter->fw_info.query_status) {
		enum halmac_feature_id id;

		switch (f_id) {
		case OFLD_F_CFG_PARAM:
			id = HALMAC_FEATURE_CFG_PARA;
			break;
		default:
			MFC_MSG_ERR("ofld feature id (query)!!\n");
			return 0;
		}

		do {
			ret = api->halmac_query_status(mac_adapter, id,
						       &proc_status,
						       NULL, NULL);
			if (ret == HALMAC_RET_SUCCESS &&
			    proc_status == HALMAC_CMD_PROCESS_DONE)
				break;
			cnt--;
			if (!cnt)
				return 0;
			Sleep(1);
		} while (1);

	} else {
		u8 *flag;

		struct c2h_event_info *c2h_info = &(adapter->c2h_info);

		switch (f_id) {
		case OFLD_F_CFG_PARAM:
			flag = &c2h_info->cfg_param;
			break;
		default:
			MFC_MSG_ERR("ofld feature id (idc)!!\n");
			return 0;
		}

		if (!wmac_c2h_flag_polling(flag, cnt))
			return 0;
	}

	return 1;
}

static u8 wmac_cfg_mac_param(struct ui_adapter_e *adapter)
{
#define READ_FILE_LINE_BUF 256
	u32 offset;
	u32 data;
	u8 full_fifo = 1;
	enum halmac_ret_status ret;
	struct halmac_adapter *mac_adapter = adapter->halmac_adapter;
	struct halmac_api *api = adapter->halmac_api;
	enum wmac_ofld_feature_id f_id = OFLD_F_CFG_PARAM;
	struct halmac_phy_parameter_info info;
	TCHAR *file_dir;
	TCHAR w_buf[READ_FILE_LINE_BUF];
	CStdioFile file_ctrl;

	switch (adapter->chip_type.chip_id) {
	case _CHIP_ID_8822B:
		file_dir = TEXT("./HWVerifyData/RTL8822B/MAC_REG.txt");
		break;
	case _CHIP_ID_8821C:
		file_dir = TEXT("./HWVerifyData/RTL8821C/MAC_REG.txt");
		break;
	case _CHIP_ID_8824B:
		file_dir = TEXT("./HWVerifyData/RTL8824B/MAC_REG.txt");
		break;
	case _CHIP_ID_8197F:
		file_dir = TEXT("./HWVerifyData/RTL8197F/MAC_REG.txt");
		break;
	case _CHIP_ID_8822C:
		file_dir = TEXT("./HWVerifyData/RTL8822C/MAC_REG.txt");
		break;
	case _CHIP_ID_8812F:
		file_dir = TEXT("./HWVerifyData/RTL8812F/MAC_REG.txt");
		break;
	default:
		MFC_MSG_BOX("cfg mac param -- unsupported chip!!");
		return 0;
	}

	if (!file_ctrl.Open(file_dir, CFile::modeRead|CFile::typeText)) {
		MFC_MSG_ERR("Open MAC_REG file!!\n");
		return 0;
	}

	wmac_clear_ofld_feature_status(adapter, f_id);
	do {
		file_ctrl.ReadString(w_buf, READ_FILE_LINE_BUF - 1);
		if (*w_buf == '/' || *w_buf == 0x0A || *w_buf == NULL)
			continue;

		swscanf_s(w_buf, TEXT("%lx %lx"), &offset, &data);
		if (offset == 0xFF || offset == 0xFFFF) {
			info.cmd_id = HALMAC_PARAMETER_CMD_END;
			if (adapter->dlfw_ok) {
				ret = api->halmac_cfg_parameter(mac_adapter,
								&info,
								full_fifo);
				if (ret != HALMAC_RET_SUCCESS) {
					MFC_MSG_ERR("cfg mac param!!\n");
					file_ctrl.Close();
					return 0;
				}
			}
			break;
		}

		info.cmd_id = HALMAC_PARAMETER_CMD_MAC_W8;
		info.content.MAC_REG_W.value = data;
		info.content.MAC_REG_W.offset = (u16)offset;
		info.content.MAC_REG_W.msk_en = 0;

		if (adapter->dlfw_ok) {
			ret = api->halmac_cfg_parameter(mac_adapter,
							&info,
							full_fifo);
			if (ret != HALMAC_RET_SUCCESS &&
			    ret != HALMAC_RET_PARA_SENDING) {
				MFC_MSG_ERR("cfg mac param!!\n");
				file_ctrl.Close();
				return 0;
			}

			if (ret == HALMAC_RET_PARA_SENDING) {
				if (!wmac_get_ofld_feature_status(adapter,
								  f_id)) {
					MFC_MSG_ERR("ofld feature status!!\n");
					file_ctrl.Close();
					return 0;
				}
				wmac_clear_ofld_feature_status(adapter, f_id);
			}
		} else {
			api->halmac_reg_write_8(mac_adapter, offset, (u8)data);
		}
	} while (1);

	file_ctrl.Close();

	if (adapter->dlfw_ok) {
		if (!wmac_get_ofld_feature_status(adapter, f_id)) {
			MFC_MSG_ERR("ofld feature status!!\n");
			return 0;
		}
	}

	return 1;
#undef READ_FILE_LINE_BUF
}

static u8 wmac_cfg_mac_fpga_param(struct ui_adapter_e *adapter)
{
#define READ_FILE_LINE_BUF 256
	u32 offset;
	u32 data;
	u8 full_fifo = 1;
	enum halmac_ret_status ret;
	struct halmac_adapter *mac_adapter = adapter->halmac_adapter;
	struct halmac_api *api = adapter->halmac_api;
	enum wmac_ofld_feature_id f_id = OFLD_F_CFG_PARAM;
	struct halmac_phy_parameter_info info;
	TCHAR *file_dir;
	TCHAR w_buf[READ_FILE_LINE_BUF];
	CStdioFile file_ctrl;

	switch (adapter->chip_type.chip_id) {
	case _CHIP_ID_8822B:
		file_dir = TEXT("./HWVerifyData/RTL8822B/MAC_REG_FPGA.txt");
		break;
	case _CHIP_ID_8821C:
		file_dir = TEXT("./HWVerifyData/RTL8821C/MAC_REG_FPGA.txt");
		break;
	case _CHIP_ID_8824B:
		file_dir = TEXT("./HWVerifyData/RTL8824B/MAC_REG_FPGA.txt");
		break;
	case _CHIP_ID_8197F:
		file_dir = TEXT("./HWVerifyData/RTL8197F/MAC_REG_FPGA.txt");
		break;
	case _CHIP_ID_8822C:
		file_dir = TEXT("./HWVerifyData/RTL8822C/MAC_REG_FPGA.txt");
		break;
	case _CHIP_ID_8812F:
		file_dir = TEXT("./HWVerifyData/RTL8812F/MAC_REG_FPGA.txt");
		break;
	default:
		MFC_MSG_BOX("cfg mac fpga param -- unsupported chip!!");
		return 0;
	}

	if (!file_ctrl.Open(file_dir, CFile::modeRead | CFile::typeText)) {
		MFC_MSG_ERR("Open MAC_REG_FPGA file!!\n");
		return 0;
	}

	wmac_clear_ofld_feature_status(adapter, f_id);
	do {
		file_ctrl.ReadString(w_buf, READ_FILE_LINE_BUF - 1);
		if (*w_buf == '/' || *w_buf == 0x0A || *w_buf == NULL)
			continue;

		swscanf_s(w_buf, TEXT("%lx %lx"), &offset, &data);
		if (offset == 0xFF || offset == 0xFFFF) {
			info.cmd_id = HALMAC_PARAMETER_CMD_END;
			if (adapter->dlfw_ok) {
				ret = api->halmac_cfg_parameter(mac_adapter,
								&info,
								full_fifo);
				if (ret != HALMAC_RET_SUCCESS) {
					MFC_MSG_ERR("cfg mac param!!\n");
					file_ctrl.Close();
					return 0;
				}
			}
			break;
		}

		info.cmd_id = HALMAC_PARAMETER_CMD_MAC_W8;
		info.content.MAC_REG_W.value = data;
		info.content.MAC_REG_W.offset = (u16)offset;
		info.content.MAC_REG_W.msk_en = 0;

		if (adapter->dlfw_ok) {
			ret = api->halmac_cfg_parameter(mac_adapter,
							&info,
							full_fifo);
			if (ret != HALMAC_RET_SUCCESS &&
			    ret != HALMAC_RET_PARA_SENDING) {
				MFC_MSG_ERR("cfg mac param!!\n");
				file_ctrl.Close();
				return 0;
			}

			if (ret == HALMAC_RET_PARA_SENDING) {
				if (!wmac_get_ofld_feature_status(adapter,
								  f_id)) {
					MFC_MSG_ERR("ofld feature status!!\n");
					file_ctrl.Close();
					return 0;
				}
				wmac_clear_ofld_feature_status(adapter, f_id);
			}
		} else {
			api->halmac_reg_write_8(mac_adapter, offset, (u8)data);
		}
	} while (1);

	file_ctrl.Close();

	if (adapter->dlfw_ok) {
		if (!wmac_get_ofld_feature_status(adapter, f_id)) {
			MFC_MSG_ERR("ofld feature status!!\n");
			return 0;
		}
	}

	return 1;
#undef READ_FILE_LINE_BUF
}

static u8 wmac_cfg_bb_param(struct ui_adapter_e *adapter)
{
#define READ_FILE_LINE_BUF 256
	u32 offset;
	u32 data;
	u8 full_fifo = 1;
	enum halmac_ret_status ret;
	struct halmac_adapter *mac_adapter = adapter->halmac_adapter;
	struct halmac_api *api = adapter->halmac_api;
	struct halmisc_api *misc_api = adapter->halmac_api->halmisc_api;
	enum wmac_ofld_feature_id f_id = OFLD_F_CFG_PARAM;
	struct halmac_phy_parameter_info info;
	TCHAR *file_dir;
	TCHAR w_buf[READ_FILE_LINE_BUF];
	CStdioFile file_ctrl;

	misc_api->halmac_bb_preconfig(mac_adapter);

	switch (adapter->chip_type.chip_id) {
	case _CHIP_ID_8822B:
		file_dir = TEXT("./HWVerifyData/RTL8822B/PHY_REG.txt");
		break;
	case _CHIP_ID_8821C:
		file_dir = TEXT("./HWVerifyData/RTL8821C/PHY_REG.txt");
		break;
	case _CHIP_ID_8824B:
		file_dir = TEXT("./HWVerifyData/RTL8824B/PHY_REG.txt");
		break;
	case _CHIP_ID_8197F:
		file_dir = TEXT("./HWVerifyData/RTL8197F/PHY_REG.txt");
		break;
	case _CHIP_ID_8822C:
		file_dir = TEXT("./HWVerifyData/RTL8822C/PHY_REG.txt");
		break;
	case _CHIP_ID_8812F:
		file_dir = TEXT("./HWVerifyData/RTL8812F/PHY_REG.txt");
		break;
	default:
		MFC_MSG_BOX("cfg phy param -- unsupported chip!!");
		return 0;
	}

	if (!file_ctrl.Open(file_dir, CFile::modeRead|CFile::typeText)) {
		MFC_MSG_ERR("Open PHY_REG file!!\n");
		return 0;
	}

	wmac_clear_ofld_feature_status(adapter, f_id);
	do {
		file_ctrl.ReadString(w_buf, READ_FILE_LINE_BUF - 1);
		if (*w_buf == '/' || *w_buf == 0x0A || *w_buf == NULL)
			continue;

		swscanf_s(w_buf, TEXT("%lx %lx"), &offset, &data);
		if (offset == 0xFF || offset == 0xFFFF) {
			info.cmd_id = HALMAC_PARAMETER_CMD_END;
			if (adapter->dlfw_ok) {
				ret = api->halmac_cfg_parameter(mac_adapter,
								&info,
								full_fifo);
				if (ret != HALMAC_RET_SUCCESS) {
					MFC_MSG_ERR("cfg phy param!!\n");
					file_ctrl.Close();
					return 0;
				}
			}
			break;
		}

		switch (offset) {
		case 0xF9:
			info.cmd_id = HALMAC_PARAMETER_CMD_DELAY_US;
			info.content.DELAY_TIME.delay_time = 1;
			break;
		case 0xFA:
			info.cmd_id = HALMAC_PARAMETER_CMD_DELAY_US;
			info.content.DELAY_TIME.delay_time = 5;
			break;
		case 0xFB:
			info.cmd_id = HALMAC_PARAMETER_CMD_DELAY_US;
			info.content.DELAY_TIME.delay_time = 50;
			break;
		case 0xFC:
			info.cmd_id = HALMAC_PARAMETER_CMD_DELAY_MS;
			info.content.DELAY_TIME.delay_time = 1;
			break;
		case 0xFD:
			info.cmd_id = HALMAC_PARAMETER_CMD_DELAY_MS;
			info.content.DELAY_TIME.delay_time = 5;
			break;
		case 0xFE:
			info.cmd_id = HALMAC_PARAMETER_CMD_DELAY_MS;
			info.content.DELAY_TIME.delay_time = 50;
			break;
		default:
			info.cmd_id = HALMAC_PARAMETER_CMD_BB_W32;
			info.content.BB_REG_W.value = data;
			info.content.BB_REG_W.offset = (u16)offset;
			info.content.BB_REG_W.msk_en = 0;
			break;
		}

		if (adapter->dlfw_ok) {
			ret = api->halmac_cfg_parameter(mac_adapter,
							&info,
							full_fifo);
			if (ret != HALMAC_RET_SUCCESS &&
			    ret != HALMAC_RET_PARA_SENDING) {
				MFC_MSG_ERR("cfg phy param!!\n");
				file_ctrl.Close();
				return 0;
			}

			if (ret == HALMAC_RET_PARA_SENDING) {
				if (!wmac_get_ofld_feature_status(adapter,
								  f_id)) {
					MFC_MSG_ERR("ofld feature status!!\n");
					file_ctrl.Close();
					return 0;
				}
				wmac_clear_ofld_feature_status(adapter, f_id);
			}
		} else {
			api->halmac_reg_write_32(mac_adapter, offset, data);
		}
	} while (1);

	file_ctrl.Close();

	if (adapter->dlfw_ok) {
		if (!wmac_get_ofld_feature_status(adapter, f_id)) {
			MFC_MSG_ERR("ofld feature status!!\n");
			return 0;
		}
	}

	switch (adapter->chip_type.chip_id) {
	case _CHIP_ID_8822B:
		file_dir = TEXT("./HWVerifyData/RTL8822B/AGC_TAB.txt");
		break;
	case _CHIP_ID_8821C:
		file_dir = TEXT("./HWVerifyData/RTL8821C/AGC_TAB.txt");
		break;
	case _CHIP_ID_8824B:
		file_dir = TEXT("./HWVerifyData/RTL8824B/AGC_TAB.txt");
		break;
	case _CHIP_ID_8197F:
		file_dir = TEXT("./HWVerifyData/RTL8197F/AGC_TAB.txt");
		break;
	case _CHIP_ID_8822C:
		file_dir = TEXT("./HWVerifyData/RTL8822C/AGC_TAB.txt");
		break;
	case _CHIP_ID_8812F:
		file_dir = TEXT("./HWVerifyData/RTL8812F/AGC_TAB.txt");
		break;
	default:
		MFC_MSG_BOX("cfg agc param -- unsupported chip!!");
		return 0;
	}

	if (!file_ctrl.Open(file_dir, CFile::modeRead|CFile::typeText)) {
		MFC_MSG_ERR("Open AGC_TAB file!!\n");
		return 0;
	}

	wmac_clear_ofld_feature_status(adapter, f_id);
	do {
		file_ctrl.ReadString(w_buf, READ_FILE_LINE_BUF - 1);
		if (*w_buf == '/' || *w_buf == 0x0A || *w_buf == NULL)
			continue;

		swscanf_s(w_buf, TEXT("%lx %lx"), &offset, &data);
		if (offset == 0xFF || offset == 0xFFFF) {
			info.cmd_id = HALMAC_PARAMETER_CMD_END;
			if (adapter->dlfw_ok) {
				ret = api->halmac_cfg_parameter(mac_adapter,
								&info,
								full_fifo);
				if (ret != HALMAC_RET_SUCCESS) {
					MFC_MSG_ERR("cfg phy param!!\n");
					file_ctrl.Close();
					return 0;
				}
			}
			break;
		}

		switch (offset) {
		case 0xFE:
			info.cmd_id = HALMAC_PARAMETER_CMD_DELAY_US;
			info.content.DELAY_TIME.delay_time = 100;
			break;
		case 0xFFE:
			info.cmd_id = HALMAC_PARAMETER_CMD_DELAY_MS;
			info.content.DELAY_TIME.delay_time = 50;
			break;
		default:
			info.cmd_id = HALMAC_PARAMETER_CMD_BB_W32;
			info.content.BB_REG_W.value = data;
			info.content.BB_REG_W.offset = (u16)offset;
			info.content.BB_REG_W.msk_en = 0;
			break;
		}

		if (adapter->dlfw_ok) {
			ret = api->halmac_cfg_parameter(mac_adapter,
							&info,
							full_fifo);
			if (ret != HALMAC_RET_SUCCESS &&
			    ret != HALMAC_RET_PARA_SENDING) {
				MFC_MSG_ERR("cfg phy param!!\n");
				file_ctrl.Close();
				return 0;
			}

			if (ret == HALMAC_RET_PARA_SENDING) {
				if (!wmac_get_ofld_feature_status(adapter,
								  f_id)) {
					MFC_MSG_ERR("ofld feature status!!\n");
					file_ctrl.Close();
					return 0;
				}
				wmac_clear_ofld_feature_status(adapter, f_id);
			}
		} else {
			api->halmac_reg_write_32(mac_adapter, offset, data);
		}
	} while (1);

	file_ctrl.Close();

	if (adapter->dlfw_ok) {
		if (!wmac_get_ofld_feature_status(adapter, f_id)) {
			MFC_MSG_ERR("ofld feature status!!\n");
			return 0;
		}
	}

	misc_api->halmac_init_crystal_capacity(mac_adapter);

	return 1;
#undef READ_FILE_LINE_BUF
}

static u8 wmac_cfg_rf_param(struct ui_adapter_e *adapter)
{
#define READ_FILE_LINE_BUF 256
#define RF_PARAM_FILE_MAX_NUM 7
	u32 offset;
	u32 data;
	u8 full_fifo = 1;
	enum halmac_ret_status ret;
	struct halmac_adapter *mac_adapter = adapter->halmac_adapter;
	struct halmac_api *api = adapter->halmac_api;
	struct halmisc_api *misc_api = adapter->halmac_api->halmisc_api;
	enum wmac_ofld_feature_id f_id = OFLD_F_CFG_PARAM;
	struct halmac_phy_parameter_info info;
	TCHAR *file_dir[RF_PARAM_FILE_MAX_NUM + 1];
	u8 cnt = 0;
	TCHAR w_buf[READ_FILE_LINE_BUF];
	CStdioFile file_ctrl;

	/* Please don't change sort of file, or you will regret*/
	/* Sort :  RadioA -> RadioB -> RadioC -> RadioD -> */
	/* RadioSyn0 -> RadioSyn1 */
	switch (adapter->chip_type.chip_id) {
	case _CHIP_ID_8822B:
		file_dir[0] = TEXT("./HWVerifyData/RTL8822B/RadioA.txt");
		file_dir[1] = NULL;
		break;
	case _CHIP_ID_8821C:
		file_dir[0] = TEXT("./HWVerifyData/RTL8821C/RadioA.txt");
		file_dir[1] = NULL;
		break;
	case _CHIP_ID_8824B:
		file_dir[0] = TEXT("./HWVerifyData/RTL8824B/RadioA.txt");
		file_dir[1] = TEXT("./HWVerifyData/RTL8824B/RadioB.txt");
		file_dir[2] = TEXT("./HWVerifyData/RTL8824B/RadioC.txt");
		file_dir[3] = TEXT("./HWVerifyData/RTL8824B/RadioD.txt");
		file_dir[4] = TEXT("./HWVerifyData/RTL8824B/RadioSyn0.txt");
		file_dir[5] = TEXT("./HWVerifyData/RTL8824B/RadioSyn1.txt");
		file_dir[6] = NULL;
		break;
	case _CHIP_ID_8197F:
		file_dir[0] = TEXT("./HWVerifyData/RTL8197F/RadioA.txt");
		file_dir[1] = NULL;
		break;
	case _CHIP_ID_8822C:
		file_dir[0] = TEXT("./HWVerifyData/RTL8822C/RadioA.txt");
		file_dir[1] = TEXT("./HWVerifyData/RTL8822C/RadioB.txt");
		file_dir[2] = NULL;
		break;
	case _CHIP_ID_8812F:
		file_dir[0] = TEXT("./HWVerifyData/RTL8197F/RadioA.txt");
		file_dir[1] = TEXT("./HWVerifyData/RTL8822C/RadioB.txt");
		file_dir[2] = NULL;
		break;
	default:
		MFC_MSG_BOX("cfg rf param -- unsupported chip!!");
		return 0;
	}

CFG_RF_PARAM_FILE:
	if (!file_ctrl.Open(file_dir[cnt], CFile::modeRead|CFile::typeText)) {
		MFC_MSG_ERR("Open %s file!!\n", file_dir[cnt]);
		return 0;
	}

	wmac_clear_ofld_feature_status(adapter, f_id);
	do {
		file_ctrl.ReadString(w_buf, READ_FILE_LINE_BUF - 1);
		if (*w_buf == '/' || *w_buf == 0x0A || *w_buf == NULL)
			continue;

		swscanf_s(w_buf, TEXT("%lx %lx"), &offset, &data);
		if (offset == 0xFF || offset == 0xFFFF) {
			info.cmd_id = HALMAC_PARAMETER_CMD_END;
			if (adapter->dlfw_ok) {
				ret = api->halmac_cfg_parameter(mac_adapter,
								&info,
								full_fifo);
				if (ret != HALMAC_RET_SUCCESS) {
					MFC_MSG_ERR("cfg rf param!!\n");
					file_ctrl.Close();
					return 0;
				}
			}
			break;
		}

		switch (offset) {
		case 0xFE:
			info.cmd_id = HALMAC_PARAMETER_CMD_DELAY_US;
			info.content.DELAY_TIME.delay_time = 100;
			break;
		case 0xFFE:
			info.cmd_id = HALMAC_PARAMETER_CMD_DELAY_MS;
			info.content.DELAY_TIME.delay_time = 50;
			break;
		default:
			info.cmd_id = HALMAC_PARAMETER_CMD_RF_W;
			info.content.RF_REG_W.rf_path = cnt;
			info.content.RF_REG_W.value = data;
			info.content.RF_REG_W.offset = (u8)offset;
			info.content.RF_REG_W.msk_en = 0;
			break;
		}

		if (adapter->dlfw_ok) {
			ret = api->halmac_cfg_parameter(mac_adapter,
							&info,
							full_fifo);
			if (ret != HALMAC_RET_SUCCESS &&
			    ret != HALMAC_RET_PARA_SENDING) {
				MFC_MSG_ERR("cfg mac param!!\n");
				file_ctrl.Close();
				return 0;
			}

			if (ret == HALMAC_RET_PARA_SENDING) {
				if (!wmac_get_ofld_feature_status(adapter,
								  f_id)) {
					MFC_MSG_ERR("ofld feature status!!\n");
					file_ctrl.Close();
					return 0;
				}
				wmac_clear_ofld_feature_status(adapter, f_id);
			}
		} else {
			struct hal_rf_reg_info rf_info;

			if(offset == 0xFFE){
				Sleep(50);
			} else if(offset == 0xFE){
				Sleep(1);
			} else {
				rf_info.rf_path = (enum halmac_rf_path)cnt;
				rf_info.offset = offset;
				rf_info.bit_mask = 0xFFFFF;
				rf_info.data = data;
				misc_api->halmac_rf_reg_write(mac_adapter,
							      &rf_info);
			}
		}
	} while (1);

	file_ctrl.Close();

	if (adapter->dlfw_ok) {
		if (!wmac_get_ofld_feature_status(adapter, f_id)) {
			MFC_MSG_ERR("ofld feature status!!\n");
			return 0;
		}
	}

	cnt++;
	if (file_dir[cnt])
		goto CFG_RF_PARAM_FILE;

	return 1;
#undef READ_FILE_LINE_BUF
}

/* Simply extern start trx API temporarily, */
/*  and need to re-arch DataPath later... */
extern void datapath_start_mac_trx(u8 start_tx, u8 start_rx);

static u8 wmac_cfg_param(struct ui_adapter_e *adapter)
{
	u8 en;
	enum halmac_ret_status ret;
	struct halmac_adapter *mac_adapter = adapter->halmac_adapter;
	struct halmac_api *api = adapter->halmac_api;
	struct halmisc_api *misc_api = adapter->halmac_api->halmisc_api;

	/* Testprogram will use FW_CFG_PARAM feature */
	/* if firmware download is successful. */
	/* And need to recive C2H packets to get status */
	if (adapter->dlfw_ok)
		datapath_start_mac_trx(0, 1);

	if (adapter->fw_info.bb_param || adapter->fw_info.rf_param) {
	    	en = 1;
		ret = api->halmac_set_hw_value(mac_adapter,
					       HALMAC_HW_EN_BB_RF, &en);
		if (ret != HALMAC_RET_SUCCESS)
			MFC_MSG_ERR("enable BB/RF!!\n");


		DM_Odm.RFType = ODM_2T2R;
		DM_Odm.mp_mode = 1;
		config_phydm_parameter_init(&DM_Odm, ODM_PRE_SETTING);
	}

	if (adapter->fw_info.mac_param) {
		if (!wmac_cfg_mac_param(adapter))
			MFC_MSG_BOX("Config MAC_REG!!\n");
	}

	if (adapter->fw_info.mac_fpga_param) {
		if (!wmac_cfg_mac_fpga_param(adapter))
			MFC_MSG_BOX("Config MAC_REG_FPGA!!\n");
	}

	if (adapter->fw_info.bb_param) {
		if (!wmac_cfg_bb_param(adapter))
			MFC_MSG_BOX("Config PHY_REG/AGC_TAB!!\n");
	}

	if (adapter->fw_info.rf_param) {
		if (!wmac_cfg_rf_param(adapter))
			MFC_MSG_BOX("Config RF_PATHX!!\n");
	}

	if (adapter->fw_info.bb_param || adapter->fw_info.rf_param) {
		config_phydm_parameter_init(&DM_Odm, ODM_POST_SETTING);
		config_phydm_trx_mode(&DM_Odm,
				      adapter->mac_info.test_opt.rf_path,
				      adapter->mac_info.test_opt.rf_path,
				      1);
	}

    	Sleep(100);

	if(adapter->chip_type.chip_id == _CHIP_ID_8822C ||
	   adapter->chip_type.chip_id == _CHIP_ID_8824B){
		phy_init_haldm(&DM_Odm);
	}
	config_phydm_switch_channel_bw(&DM_Odm, adapter->bb_rf.central_ch,
				       adapter->bb_rf.pri_ch_idx,
				       adapter->bb_rf.bw);
	api->halmac_cfg_ch_bw(mac_adapter,
			      adapter->bb_rf.central_ch,
			      (enum halmac_pri_ch_idx)adapter->bb_rf.pri_ch_idx,
			      (enum halmac_bw)adapter->bb_rf.bw);

	return 1;
}

static u8 wmac_init_trx_path(struct ui_adapter_e *adapter)
{
	u8 en;
	u32 tmp;
	enum halmac_rx_fifo_expanding_mode rx_fifo_mode;
	struct halmac_rxagg_cfg rxagg_cfg;
	struct halmac_chip_type chip_t;
	struct halmac_adapter *mac_adapter = adapter->halmac_adapter;
	struct halmac_api *api = adapter->halmac_api;
	struct halmisc_api *misc_api = adapter->halmac_api->halmisc_api;
	enum halmac_ret_status ret = HALMAC_RET_SUCCESS;

	misc_api->halmac_get_chip_type(mac_adapter, &chip_t);

	MFC_MSG_ALWAYS("=====CHIP INFO=====\n");
	switch (chip_t.chip_id) {
	case HALMAC_CHIP_ID_8822B:
		MFC_MSG_ALWAYS("ID : 8822B\n");
		break;
	case HALMAC_CHIP_ID_8821C:
		MFC_MSG_ALWAYS("ID : 8821C\n");
		break;
	case HALMAC_CHIP_ID_8814B:
		MFC_MSG_ALWAYS("ID : 8814B\n");
		break;
	case HALMAC_CHIP_ID_8822C:
		MFC_MSG_ALWAYS("ID : 8822C\n");
		break;
	case HALMAC_CHIP_ID_8812F:
		MFC_MSG_ALWAYS("ID : 8812F\n");
		break;
	default:
		MFC_MSG_ALWAYS("ID : (@_@)\n");
		break;
	}

	switch (chip_t.chip_version) {
	case HALMAC_CHIP_VER_A_CUT:
		MFC_MSG_ALWAYS("VER : A-CUT\n");
		break;
	case HALMAC_CHIP_VER_B_CUT:
		MFC_MSG_ALWAYS("VER : B-CUT\n");
		break;
	case HALMAC_CHIP_VER_C_CUT:
		MFC_MSG_ALWAYS("VER : C-CUT\n");
		break;
	case HALMAC_CHIP_VER_D_CUT:
		MFC_MSG_ALWAYS("VER : D-CUT\n");
		break;
	case HALMAC_CHIP_VER_E_CUT:
		MFC_MSG_ALWAYS("VER : E-CUT\n");
		break;
	case HALMAC_CHIP_VER_F_CUT:
		MFC_MSG_ALWAYS("VER : F-CUT\n");
		break;
	case HALMAC_CHIP_VER_TEST:
		MFC_MSG_ALWAYS("VER : TEST\n");
		break;
	default:
		MFC_MSG_ALWAYS("VER : (X_X)\n");
		break;
	}

	adapter->chip_type.chip_id = (enum _chip_id)chip_t.chip_id;
	DM_Odm.chip_id = chip_t.chip_id;

	ret = misc_api->halmac_pre_init_trx_cfg(mac_adapter);
	if (ret != HALMAC_RET_SUCCESS) {
		MFC_MSG_ERR("pre-init trx!!\n");
		return 0;
	}

	/* Set LTE MUX control to wifi */
	en = 1;
	ret = misc_api->halmisc_set_hw_value(mac_adapter,
					     HALMAC_HW_LTE_MUX_CTRL, &en);
	if (ret != HALMAC_RET_SUCCESS) {
		MFC_MSG_ERR("lte mux control!!\n");
		//return 0;
	}

	en = adapter->mac_info.test_opt.txdesc_txfr_en;
	ret = misc_api->halmac_tx_desc_transfer(mac_adapter, en);
	if (ret != HALMAC_RET_SUCCESS)
		MFC_MSG_ERR("txdesc transfer!!\n");

	en = adapter->mac_info.test_opt.txdesc_chksum_en;
	ret = api->halmac_set_hw_value(mac_adapter,
				       HALMAC_HW_TXDESC_CHECKSUM, &en);
	if (ret != HALMAC_RET_SUCCESS) {
		MFC_MSG_ERR("txdesc chksum!!\n");
		return 0;
	}

	tmp = adapter->txdesc.tx_agg_mum;
	ret = misc_api->halmac_set_agg_num(mac_adapter, (u8)tmp);
	if (ret != HALMAC_RET_SUCCESS) {
		MFC_MSG_ERR("txagg num!!\n");
		return 0;
	}

	if (adapter->fw_info.dlfw)
		wmac_download_fw(adapter);

	ret = api->halmac_cfg_la_mode(mac_adapter, HALMAC_LA_MODE_DISABLE);
	if (ret != HALMAC_RET_SUCCESS) {
		MFC_MSG_ERR("la mode!!\n");
		return 0;
	}

	switch (adapter->mac_info.test_opt.fifo_expanding_cfg) {
	case MFC_FIFO_EXPANDING_DIS:
	default:
		rx_fifo_mode = HALMAC_RX_FIFO_EXPANDING_MODE_DISABLE;
		break;
	case MFC_FIFO_EXPANDING_1_BLK:
		rx_fifo_mode = HALMAC_RX_FIFO_EXPANDING_MODE_1_BLOCK;
		break;
	case MFC_FIFO_EXPANDING_2_BLK:
		rx_fifo_mode = HALMAC_RX_FIFO_EXPANDING_MODE_2_BLOCK;
		break;
	case MFC_FIFO_EXPANDING_3_BLK:
		rx_fifo_mode = HALMAC_RX_FIFO_EXPANDING_MODE_3_BLOCK;
		break;
	case MFC_FIFO_EXPANDING_4_BLK:
		rx_fifo_mode = HALMAC_RX_FIFO_EXPANDING_MODE_4_BLOCK;
		break;
	}

	ret = api->halmac_cfg_rxff_expand_mode(mac_adapter, rx_fifo_mode);
	if (ret != HALMAC_RET_SUCCESS) {
		MFC_MSG_ERR("rx fifo expand mode mode!!\n");
		return 0;
	}

	switch (adapter->mac_info.test_opt.lb_mode) {
	case MFC_TRX_MODE_NORM:
		tmp = HALMAC_TRX_MODE_NORMAL;
		break;
	case MFC_TRX_MODE_DLLB:
		tmp = HALMAC_TRX_MODE_DELAY_LOOPBACK;
		break;
	case MFC_TRX_MODE_LB:
	default:
		tmp = HALMAC_TRX_MODE_LOOPBACK;
		break;
	}

	ret = api->halmac_init_mac_cfg(mac_adapter, (enum halmac_trx_mode)tmp);
	if (ret != HALMAC_RET_SUCCESS) {
		MFC_MSG_ERR("init mac cfg!!\n");
		return 0;
	}

	/* Set LTE COEX setting to wifi */
	misc_api->halmac_coex_cfg(mac_adapter);

	switch (adapter->mac_info.test_opt.rx_agg_mode) {
	case MFC_RXAGG_MODE_NONE:
	default:
		tmp = HALMAC_RX_AGG_MODE_NONE;
		break;
	case MFC_RXAGG_MODE_DMA:
		tmp = HALMAC_RX_AGG_MODE_DMA;
		break;
	case MFC_RXAGG_MODE_USB:
		tmp = HALMAC_RX_AGG_MODE_USB;
		break;
	}
	en = adapter->mac_info.test_opt.rx_agg_size_limit_en;

	rxagg_cfg.threshold.size_limit_en = en;
	rxagg_cfg.mode = (enum halmac_rx_agg_mode)tmp;
	rxagg_cfg.threshold.drv_define = 0;
	rxagg_cfg.threshold.size = 0xFF;
	rxagg_cfg.threshold.timeout = 0x01;

	ret = api->halmac_cfg_rx_aggregation(mac_adapter, &rxagg_cfg);
	if (ret != HALMAC_RET_SUCCESS) {
		MFC_MSG_ERR("cfg rxagg!!\n");
		return 0;
	}

	ret = api->halmac_cfg_operation_mode(mac_adapter,
					     HALMAC_WIRELESS_MODE_AC);
	if (ret != HALMAC_RET_SUCCESS) {
		MFC_MSG_ERR("cfg operation mode!!\n");
		return 0;
	}

	if (adapter->dlfw_ok) {
		struct halmac_general_info gen_info;
		gen_info.rfe_type = 0;
		gen_info.rf_type = HALMAC_RF_2T2R;
		gen_info.tx_ant_status = (ODM_RF_A | ODM_RF_B);
		gen_info.rx_ant_status = (ODM_RF_A | ODM_RF_B);
		ret = api->halmac_send_general_info(mac_adapter, &gen_info);
		if (ret != HALMAC_RET_SUCCESS) {
			MFC_MSG_ERR("send general info!\n");
			return 0;
		}
	}

	tmp = (u32)adapter->mac_info.test_opt.drv_info;
	api->halmac_cfg_drv_info(mac_adapter, (enum halmac_drv_info)tmp);

	if (adapter->fw_info.param_file) {
		if (!wmac_cfg_param(adapter)) {
			MFC_MSG_ERR("cfg param!\n");
			return 0;
		}
	}

	api->halmac_init_interface_cfg(mac_adapter);

	en = adapter->mac_info.test_opt.txagg_align_en;
	tmp = adapter->mac_info.test_opt.txagg_align_size;
	ret = api->halmac_cfg_tx_agg_align(mac_adapter, en, tmp);
	if (ret != HALMAC_RET_SUCCESS)
		MFC_MSG_ERR("tx agg align!!\n");

	return 1;
}

static u8 _wmac_init_mac(struct ui_adapter_e *adapter)
{
	union halmac_wlan_addr wlan_addr;
	u8 mc_addr[6] = {0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF};
	struct halmac_adapter *mac_adapter = adapter->halmac_adapter;
	struct halmac_api *api = adapter->halmac_api;
	struct halmisc_api *misc_api = adapter->halmac_api->halmisc_api;
	enum halmac_ret_status ret;

	if (!wmac_init_trx_path(adapter))
		return 0;

	if (adapter->intf_info.intf == HOST_INTF_SDIO) {
		struct halmac_sdio_himr_info sdio_himr;
		enum halmac_sdio_tx_format sdio_tx_fmt;

		sdio_himr.rx_request = 1;
		sdio_himr.aval_msk = 0;
		misc_api->halmac_himr_setting_sdio(mac_adapter, &sdio_himr);

		if (adapter->intf_info.sdio_handle.tx_addr_dummy_mode)
			sdio_tx_fmt = HALMAC_SDIO_DUMMY_AUTO_MODE;
		else
			sdio_tx_fmt = HALMAC_SDIO_AGG_MODE;

		ret = api->halmac_set_hw_value(mac_adapter,
					       HALMAC_HW_SDIO_TX_FORMAT,
					       &sdio_tx_fmt);
		if(ret != HALMAC_RET_SUCCESS)
			MFC_MSG_ERR("SDIO tx addr dummy mode!!");
	}

	memcpy(wlan_addr.addr,
	       adapter->mac_info.addr.src_addr, 6);
	ret = api->halmac_cfg_mac_addr(mac_adapter, 0, &wlan_addr);
	if (ret != HALMAC_RET_SUCCESS)
		MFC_MSG_ERR("config mac addr!!\n");

	memcpy(wlan_addr.addr, adapter->mac_info.addr.bssid, 6);
	ret = api->halmac_cfg_bssid(mac_adapter, 0, &wlan_addr);
	if (ret != HALMAC_RET_SUCCESS)
		MFC_MSG_ERR("config bssid addr!!\n");

	memcpy(wlan_addr.addr, mc_addr, 6);
	ret = api->halmac_cfg_multicast_addr(mac_adapter, &wlan_addr);
	if (ret != HALMAC_RET_SUCCESS)
		MFC_MSG_ERR("config mc addr fail!!\n");

	return 1;
}

static u8 wmac_cfg_wowlan(struct ui_adapter_e *adapter)
{
	return 1;
}

static u8 wmac_cfg_security(struct ui_adapter_e *adapter)
{
#define SEC_KEY_LEN	16
	struct halmac_security_setting cfg;
	struct security_info *info = &adapter->mac_info.security;
	struct halmac_cam_entry_info cam_info;
	struct halmac_adapter *mac_adapter = adapter->halmac_adapter;
	struct halmac_api *api = adapter->halmac_api;
	enum halmac_ret_status ret;

	cfg.tx_encryption = (info->encrypt_method == MFC_SECURITY_METHOD_HW) ? 1 : 0;
	cfg.rx_decryption = (info->decrypt_method == MFC_SECURITY_METHOD_HW) ? 1 : 0;
	cfg.bip_enable = (info->security_type == MFC_SECURITY_TYPE_BIP) ? 1 : 0;
	cfg.compare_keyid = 1;
	ret = api->halmac_config_security(mac_adapter, &cfg);
	if (ret != HALMAC_RET_SUCCESS) {
		MFC_MSG_ERR("cfg security!!\n");
		return 0;
	}

	memset(&cam_info, 0x00, sizeof(cam_info));
	cam_info.key_id = (u16)info->key_id;
	cam_info.security_type = (enum hal_security_type)info->security_type;
	cam_info.valid = 1;
	cam_info.unicast = (adapter->mac_info.addr.cast == MFC_UNICAST) ? 1 : 0;
	memcpy(cam_info.mac_address, adapter->mac_info.addr.dest_addr, MAC_ADDR_LEN);
	memcpy(cam_info.key, info->key, SEC_KEY_LEN);
	memcpy(cam_info.key_ext, info->key_ext, SEC_KEY_LEN);

	ret = api->halmac_write_cam(mac_adapter, 0, &cam_info);
	if (ret != HALMAC_RET_SUCCESS) {
		MFC_MSG_ERR("write cam!!\n");
		return 0;
	}

	return 1;
#undef SEC_KEY_LEN
}

static u8 wmac_init_feature(struct ui_adapter_e *adapter)
{
	u8 en;
	u32 tmp;
	struct halmac_adapter *mac_adapter = adapter->halmac_adapter;
	struct halmac_api *api = adapter->halmac_api;
	struct halmisc_api *misc_api = adapter->halmac_api->halmisc_api;
	enum halmac_ret_status ret;
	struct halmac_cut_amsdu_cfg cut_amsdu;
	struct halmac_rqpn_ch_map map;

	en = adapter->mac_info.test_opt.rx_phy_fifo;
	tmp = adapter->mac_info.test_opt.rx_phy_fifo_part_len;
	misc_api->halmac_init_phy_fifo(mac_adapter, tmp, en);

	en = adapter->mac_info.test_opt.rx_fw_fifo;
	misc_api->halmac_init_fw_fifo(mac_adapter, en);

	cut_amsdu.cut_amsdu_en = adapter->mac_info.test_opt.rx_amsdu_cut.en;
	cut_amsdu.chk_len_en = adapter->mac_info.test_opt.rx_amsdu_cut.chk_len.enable;
	cut_amsdu.chk_len_def_val = adapter->mac_info.test_opt.rx_amsdu_cut.chk_len.default_val;
	cut_amsdu.chk_len_l_th = (u8)adapter->mac_info.test_opt.rx_amsdu_cut.chk_len.low_th;
	cut_amsdu.chk_len_h_th = (u16)adapter->mac_info.test_opt.rx_amsdu_cut.chk_len.hi_th;
	ret = api->halmac_rx_cut_amsdu_cfg(mac_adapter, &cut_amsdu);
	if (ret != HALMAC_RET_SUCCESS)
		MFC_MSG_ERR("config cut a-msdu!!\n");

	en= adapter->mac_info.test_opt.rx_shift;
	api->halmac_set_hw_value(mac_adapter, HALMAC_HW_RX_SHIFT, &en);

	api->halmac_get_hw_value(mac_adapter,
				 HALMAC_HW_RQPN_CH_MAPPING, &map);
	adapter->rqpn_ch_id[RQPN_2_CH_VO] = map.dma_map_vo;
	adapter->rqpn_ch_id[RQPN_2_CH_VI] = map.dma_map_vi;
	adapter->rqpn_ch_id[RQPN_2_CH_BE] = map.dma_map_be;
	adapter->rqpn_ch_id[RQPN_2_CH_BK] = map.dma_map_bk;
	adapter->rqpn_ch_id[RQPN_2_CH_MG] = map.dma_map_mg;
	adapter->rqpn_ch_id[RQPN_2_CH_HI] = map.dma_map_hi;

	en= adapter->mac_info.test_opt.txsc_en;
	misc_api->halmisc_set_txscen(mac_adapter, en);

	if (!wmac_cfg_security(adapter))
		return 0;

	if (!wmac_cfg_wowlan(adapter))
		return 0;

	return 1;
}

static void wmac_get_group_bitmap(struct ui_adapter_e *adapter, u16 *bitmap)
{
	u8 sta_ida;
	u8 sta_idb;
	u8 tmp;

	sta_ida = adapter->mac_info.mumimo.mu.mu_sta_ida;
	sta_idb = adapter->mac_info.mumimo.mu.mu_sta_idb;

	if (sta_ida > sta_idb){
		tmp = sta_ida;
		sta_ida = sta_idb;
		sta_idb = tmp;
	}

	switch (sta_ida){
	case MFC_MU_STA_ID0:
		if (sta_idb >= MFC_MU_STA_ID0 && sta_idb <= MFC_MU_STA_ID5)
			*bitmap = BIT(sta_idb - 1);
		break;
	case MFC_MU_STA_ID1:
		if (sta_idb >= MFC_MU_STA_ID1 && sta_idb <= MFC_MU_STA_ID5)
			*bitmap = BIT(sta_idb + 3);
		break;
	case MFC_MU_STA_ID2:
		if (sta_idb >= MFC_MU_STA_ID2 && sta_idb <= MFC_MU_STA_ID5)
			*bitmap = BIT(sta_idb + 6);
		break;
	case MFC_MU_STA_ID3:
		if (sta_idb >= MFC_MU_STA_ID3 && sta_idb <= MFC_MU_STA_ID5)
			*bitmap = BIT(sta_idb + 8);
		break;
	case MFC_MU_STA_ID4:
		if (sta_idb >= MFC_MU_STA_ID4 && sta_idb <= MFC_MU_STA_ID5)
			*bitmap = BIT(sta_idb + 9);
		break;
	case MFC_MU_STA_ID5:
		/* Already covered */
		break;
	default :
		break;
	}
}

/* NOTE: No err handling of size check may cause seg fault */
void wmac_get_gid_valid(u16 *bitmap, u32 *gid_valid)
{
	u8 idx;
	u8 id0;
	u8 id1;
	u8 gid;

	for (idx = 0; idx < 15; idx++) {
		if (idx < 5) {
			/*grouping_bitmap bit0~4, MU_STA0 with MUSTA1~5*/
			id0 = 0;
			id1 = (u8)(idx + 1);
		} else if (idx < 9) {
			/*grouping_bitmap bit5~8, MU_STA1 with MUSTA2~5*/
			id0 = 1;
			id1 = (u8)(idx - 3);
		} else if (idx < 12) {
			/*grouping_bitmap bit9~11, MU_STA2 with MUSTA3~5*/
			id0 = 2;
			id1 = (u8)(idx - 6);
		} else if (idx < 14) {
			/*grouping_bitmap bit12~13, MU_STA3 with MUSTA4~5*/
			id0 = 3;
			id1 = (u8)(idx - 8);
		} else {
			/*grouping_bitmap bit14, MU_STA4 with MUSTA5*/
			id0 = 4;
			id1 = (u8)(idx - 9);
		}
		if (*bitmap & BIT(idx)) {
			/*Pair 1*/
			gid = (idx << 1) + 1;
			*(gid_valid + id0) |= (BIT(gid));
			*(gid_valid + id1) |= (BIT(gid));
			/*Pair 2*/
			gid += 1;
			*(gid_valid + id0) |= (BIT(gid));
			*(gid_valid + id1) |= (BIT(gid));
		} else {
			/*Pair 1*/
			gid = (idx << 1) + 1;
			*(gid_valid + id0) &= ~(BIT(gid));
			*(gid_valid + id1) &= ~(BIT(gid));
			/*Pair 2*/
			gid += 1;
			*(gid_valid + id0) &= ~(BIT(gid));
			*(gid_valid + id1) &= ~(BIT(gid));
		}
	}
}

static void upd_ctrlinfo(struct ui_adapter_e *adapter,
			 struct wmac_ctrl_info_fmt *info)
{
	u32 val32;
	u32 offset;
	u8 i;
	struct halmac_adapter *mac_adapter = adapter->halmac_adapter;
	struct halmisc_api *misc_api = adapter->halmac_api->halmisc_api;

	offset = info->macid << 5;

	for (i = 0; i < 32; i+= 4)
		misc_api->halmac_ctrl_info_write(mac_adapter, offset + i, 0);

	val32 = misc_api->halmac_ctrl_info_read(mac_adapter, offset);
	val32 &= 0xfff8ffff;
	val32 |= info->gtab_id << 16;
	misc_api->halmac_ctrl_info_write(mac_adapter, offset, val32);
}

static u8 wmac_cfg_mumimo(struct ui_adapter_e *adapter)
{
	u8 role = adapter->mac_info.mumimo.mu.role;
	u32 idx;
	u32 val32;
	struct _mumimo_opt *mimo = &adapter->mac_info.mumimo;
	struct halmac_adapter *mac_adapter = adapter->halmac_adapter;
	struct halmac_api *api = adapter->halmac_api;
	struct halmac_cfg_mumimo_para mimo_param;

	if (role == MFC_MU_ROLE_NONE) {
		MFC_MSG_ERR("mumimo role!!");
		return 1;
	}

	wmac_get_group_bitmap(adapter, &mimo_param.grouping_bitmap);

	if (role == MFC_MU_ROLE_AP) {
		struct halmac_mu_bfee_init_para bfee_param;
		
		val32 = api->halmac_reg_read_32(mac_adapter, 0x19E0);
		val32 &= 0x0000FFFF; 
		val32 |= BIT(31) | ((u32)mimo_param.grouping_bitmap << 16);
		api->halmac_reg_write_32(mac_adapter, 0x19E0, val32);

		bfee_param.userid = mimo->mu.mu_sta_ida + 2;
		bfee_param.paid = mimo->snding_pair.bfee_paid[mimo->mu.mu_sta_ida + 2];
		bfee_param.user_position_l = mimo->snding_pair.user_pos_l[mimo->mu.mu_sta_ida];
		bfee_param.user_position_h = mimo->snding_pair.user_pos_h[mimo->mu.mu_sta_ida] ;
		api->halmac_mu_bfee_entry_init(mac_adapter, &bfee_param);
		
		bfee_param.userid = mimo->mu.mu_sta_idb + 2;
		bfee_param.paid = mimo->snding_pair.bfee_paid[mimo->mu.mu_sta_idb + 2];
		bfee_param.user_position_l = mimo->snding_pair.user_pos_l[mimo->mu.mu_sta_idb];
		bfee_param.user_position_h = mimo->snding_pair.user_pos_h[mimo->mu.mu_sta_idb] ;
		api->halmac_mu_bfee_entry_init(mac_adapter, &bfee_param);

		mimo_param.role = HAL_BFER;						
		mimo_param.sounding_sts[mimo->mu.mu_sta_ida] = 1;	
		mimo_param.sounding_sts[mimo->mu.mu_sta_idb] = 1;			
	} else {
		u32 gid_valid[6] = {0};
		
		mimo_param.role = HAL_BFEE;
		wmac_get_gid_valid(&mimo_param.grouping_bitmap, gid_valid);
		memcpy(mimo_param.given_gid_tab, &gid_valid[mimo->mu.mu_sta_ida], 8);
		mimo_param.given_user_pos[0] = mimo->snding_pair.user_pos_l[mimo->mu.mu_sta_ida];
		mimo_param.given_user_pos[1] = mimo->snding_pair.user_pos_h[mimo->mu.mu_sta_ida];
		mimo_param.given_user_pos[2] = 0;
		mimo_param.given_user_pos[3] = 0;
	}

	api->halmac_cfg_mumimo(mac_adapter, &mimo_param);

	if (role == MFC_MU_ROLE_AP) {
		struct wmac_ctrl_info_fmt ctrl_info;

		ctrl_info.macid = mimo->mu.mu_sta_ida;
		ctrl_info.gtab_id = mimo->mu.mu_sta_ida + 1;
		ctrl_info.antnum = mimo->mu.mu_staa_antnum;
		upd_ctrlinfo(adapter, &ctrl_info);

		ctrl_info.macid = mimo->mu.mu_sta_idb;
		ctrl_info.gtab_id = mimo->mu.mu_sta_idb + 1;
		ctrl_info.antnum = mimo->mu.mu_stab_antnum;
		upd_ctrlinfo(adapter, &ctrl_info);

		val32 = (u32)api->halmac_reg_read_16(mac_adapter, 0x80);
		api->halmac_reg_write_16(mac_adapter, 0x80, val32 | BIT(15));
	}

	if (adapter->mac_info.multi_dest.multi_dest_enable) {
		struct multi_dest_opt *multi_dest;

		multi_dest = &adapter->mac_info.multi_dest;

		if (role == MFC_MU_ROLE_AP) {
			if (mimo->mu.mu_staa_antnum > 0 &&
			    mimo->mu.mu_staa_antnum < 3) {
				(*multi_dest->info)[0].macid = mimo->mu.mu_sta_ida;
				(*multi_dest->info)[0].paid = mimo->snding_pair.bfee_paid[mimo->mu.mu_sta_ida + 2];
				(*multi_dest->info)[1].macid = mimo->mu.mu_sta_idb;
				(*multi_dest->info)[1].paid = mimo->snding_pair.bfee_paid[mimo->mu.mu_sta_idb + 2];
			}
		} else {
			for (idx = 0; idx < multi_dest->multi_dest_num; idx++)
				(*multi_dest->info)[idx].macid = idx;
		}
	}


	return 1;
}

static u8 wmac_cfg_mumimo_new(struct ui_adapter_e *adapter)
{
#define MU_INFO_NUM	4
	u8 idx;
	u32 mu_id[MU_INFO_NUM];
	u32 mu_ant_num[MU_INFO_NUM];
	u32 val32;
	u8 gid = adapter->mac_info.mumimo.mu.mu_group_id;
	u8 role = adapter->mac_info.mumimo.mu.role;
	struct _mumimo_opt *mimo = &adapter->mac_info.mumimo;
	struct halmac_cfg_mumimo_para mimo_param;
	struct halmac_adapter *mac_adapter = adapter->halmac_adapter;
	struct halmac_api *api = adapter->halmac_api;

	if (role == MFC_MU_ROLE_NONE) {
		MFC_MSG_ERR("mumimo(new) role!!");
		return 1;
	}

	switch(gid) {
	case MFC_MU_GROUP_ID32:
		gid = 32;
		break;
	case MFC_MU_GROUP_ID48:
		gid = 48;
		break;
	default:
		break;
	}

	mu_id[0] = mimo->mu.mu_sta_ida;
	mu_id[1] = mimo->mu.mu_sta_idb;
	mu_id[2] = mimo->mu.mu_sta_idc;
	mu_id[3] = mimo->mu.mu_sta_idd;
	mu_ant_num[0] = mimo->mu.mu_staa_antnum;
	mu_ant_num[1] = mimo->mu.mu_stab_antnum;
	mu_ant_num[2] = mimo->mu.mu_stac_antnum;
	mu_ant_num[3] = mimo->mu.mu_stad_antnum;

	if (role == MFC_MU_ROLE_AP) {
		struct halmac_mu_bfee_init_para bfee_param;

		for (idx = 0; idx < MU_INFO_NUM; idx++) {
			bfee_param.userid = mu_id[idx] + 2;
			bfee_param.paid = mimo->snding_pair.bfee_paid[mu_id[idx] + 2];
			bfee_param.user_position_l = 0;
			bfee_param.user_position_h = 0;
			bfee_param.user_position_l_1= 0;
			bfee_param.user_position_h_1 = 0;
			if (gid < 16)
				bfee_param.user_position_l = idx << gid * 2;
			else if (gid < 32)
				bfee_param.user_position_h = idx << (gid % 16) * 2;
			else if (gid < 48)
				bfee_param.user_position_l_1= idx << (gid % 16) * 2;
			else if (gid < 64)
				bfee_param.user_position_h_1 = idx << (gid % 16) * 2;
			api->halmac_mu_bfee_entry_init(mac_adapter, &bfee_param);
		}

		mimo_param.role = HAL_BFER;
		mimo_param.grouping_bitmap = gid;		
		mimo_param.mu_tx_en = 1;
		for (idx = 0; idx < MU_INFO_NUM; idx++) {
			if (mu_ant_num[idx] > 0 && mu_ant_num[idx] < 3)
				mimo_param.sounding_sts[mu_id[idx]] = 1; 
		}
	} else {
		mimo_param.role = HAL_BFEE;
		if (gid < 32) {
			mimo_param.given_gid_tab[0] = BIT(gid);
			mimo_param.given_gid_tab[1] = 0;
		} else {
			mimo_param.given_gid_tab[0] = 0;
			mimo_param.given_gid_tab[1] = BIT(gid -32);
		}			
		memset(&mimo_param.given_user_pos[0], 0, 16);
		switch (mimo->mu.role) {
		case MFC_MU_ROLE_STAA:
			break;
		case MFC_MU_ROLE_STAB:
			mimo_param.given_user_pos[gid >> 4] |=
			~(BIT((gid % 16) * 2 + 1)) & BIT((gid % 16) * 2);					
			break;
		case MFC_MU_ROLE_STAC:
			/*STA_C always in user position 2*/
			mimo_param.given_user_pos[gid >> 4] |=
			BIT((gid % 16) * 2 + 1) & ~(BIT((gid % 16) * 2));
			break;
		case MFC_MU_ROLE_STAD:
			/*STA_D always in user position 3*/
			mimo_param.given_user_pos[gid >> 4] |=
			BIT((gid % 16) * 2 + 1) & (BIT((gid % 16) * 2));
			break;
		default:
			break;
		}
	}

	api->halmac_cfg_mumimo(mac_adapter, &mimo_param);

	if (role == MFC_MU_ROLE_AP){
		struct wmac_ctrl_info_fmt ctrl_info;

		for (idx = 0; idx < MU_INFO_NUM; idx++) {
			ctrl_info.macid = mu_id[idx];
			ctrl_info.gtab_id = mu_id[idx] + 1;
			ctrl_info.antnum = mu_ant_num[idx];
			upd_ctrlinfo(adapter, &ctrl_info);
		}

		val32 = (u32)api->halmac_reg_read_16(mac_adapter, 0x80);
		api->halmac_reg_write_16(mac_adapter, 0x80, val32 | BIT(15));
	}

	if (adapter->mac_info.multi_dest.multi_dest_enable) {	
		struct multi_dest_opt *multi_dest;

		multi_dest = &adapter->mac_info.multi_dest;

		if (role == MFC_MU_ROLE_AP) {
			for (idx = 0; idx < MU_INFO_NUM; idx++) {
				if (mimo->mu.mu_staa_antnum > 0 &&
				    mimo->mu.mu_staa_antnum < 3) {
					(*multi_dest->info)[idx].macid = mu_id[idx];
					(*multi_dest->info)[idx].paid = mimo->snding_pair.bfee_paid[mu_id[idx] + 2];
				}
			}
		} else {
			for (idx = 0; idx < multi_dest->multi_dest_num; idx++)
				(*multi_dest->info)[idx].macid = idx;
		}
	}

	return 1;
#undef MU_INFO_NUM
}

static u8 wmac_cfg_bfer_snding(struct ui_adapter_e *adapter)
{
	u8 rate;
	u16 paid;
	struct _mumimo_opt *mimo = &adapter->mac_info.mumimo;
	struct halmac_adapter *mac_adapter = adapter->halmac_adapter;
	struct halmac_api *api = adapter->halmac_api;
	enum halmac_ret_status ret;

	rate = mimo->snding.ptcl_rate + HALMAC_OFDM6;

	ret = api->halmac_cfg_sounding(mac_adapter, HAL_BFER,
				       (enum halmac_data_rate)rate);
	if (ret != HALMAC_RET_SUCCESS) {
		MFC_MSG_ERR("config bfer snding!!\n");
		return 0;
	}

	if (mimo->snding.fb_type == MFC_FB_TYPE_SU) {
		paid = mimo->snding_pair.bfee_paid[mimo->snding.bfee_id0];
		api->halmac_su_bfee_entry_init(mac_adapter,
					       mimo->snding.bfee_id0, paid);
	} else {
		u32 i;
		u8 bfee_id;
		struct halmac_mu_bfee_init_para param;

		for (i = 0; i < mimo->snding.bfee_id_num; i++) {
			switch (i) {
			case 0:
			default:
				bfee_id = mimo->snding.bfee_id0;
				break;
			case 1:
				bfee_id = mimo->snding.bfee_id1;
				break;
			case 2:
				bfee_id = mimo->snding.bfee_id2;
				break;
			case 3:
				bfee_id = mimo->snding.bfee_id3;
				break;
			}

			param.userid = bfee_id;
			param.paid = mimo->snding_pair.bfee_paid[bfee_id];
			param.user_position_l = mimo->snding_pair.user_pos_l[bfee_id - 2];
			param.user_position_h = mimo->snding_pair.user_pos_h[bfee_id - 2] ;
			api->halmac_mu_bfee_entry_init(mac_adapter, &param);
		}
	}

	return 1;
}

static u8 wmac_cfg_bfee_snding(struct ui_adapter_e *adapter)
{
	u8 rate;
	u8 bfer_id;
	struct _mumimo_opt *mimo = &adapter->mac_info.mumimo;
	struct halmac_adapter *mac_adapter = adapter->halmac_adapter;
	struct halmac_api *api = adapter->halmac_api;
	enum halmac_ret_status ret;

	rate = mimo->snding.ptcl_rate + HALMAC_OFDM6;

	ret = api->halmac_cfg_sounding(mac_adapter, HAL_BFEE,
				       (enum halmac_data_rate)rate);
	if (ret != HALMAC_RET_SUCCESS) {
		MFC_MSG_ERR("config bfee snding!!\n");
		return 0;
	}

	bfer_id = mimo->snding.bfer_id;
	if (mimo->snding.fb_type == MFC_FB_TYPE_SU) {
		struct halmac_su_bfer_init_para param;

		param.userid = bfer_id;
		param.paid = mimo->snding_pair.bfer_paid[bfer_id];
		param.csi_para = mimo->snding_pair.csi_param[bfer_id];
		memcpy(param.bfer_address.addr,
		       adapter->mac_info.addr.dest_addr, MAC_ADDR_LEN);
		api->halmac_su_bfer_entry_init(mac_adapter, &param);
	} else {
		struct halmac_mu_bfer_init_para param;
	
		param.paid = mimo->snding_pair.bfer_paid[bfer_id];
		param.csi_para = mimo->snding_pair.csi_param[bfer_id];
		param.my_aid = mimo->snding_pair.sta_aid[mimo->snding.bfee_id0];
		param.csi_length_sel = HAL_CSI_SEG_4K;
		memcpy(param.bfer_address.addr,
		       adapter->mac_info.addr.dest_addr, MAC_ADDR_LEN);
		api->halmac_mu_bfer_entry_init(mac_adapter, &param);
	}

	return 1;
}

static u8 wmac_cfg_snding(struct ui_adapter_e *adapter)
{
	u8 ret;
	u8 role = adapter->mac_info.mumimo.snding.role;
	
	if (role == MFC_ROLE_BEAMFORMER)
		ret = wmac_cfg_bfer_snding(adapter);
	else
		ret = wmac_cfg_bfee_snding(adapter);

	return ret;
}

/* The arch and coding style of MIMO init is not good. */
/* Therefor, should re-arrange it later. */
static u8 wmac_init_mimo(struct ui_adapter_e *adapter)
{	
	u8 ret;
	u8 val8;
	struct halmac_adapter *mac_adapter = adapter->halmac_adapter;
	struct halmac_api *api = adapter->halmac_api;
	struct mac_setting_info *mac_info = &adapter->mac_info;

	if (adapter->mac_info.mumimo.snding.en) {
		ret = wmac_cfg_snding(adapter);
		if (!ret)
			return ret;
	}

	if (mac_info->multi_dest.multi_dest_enable &&
	    mac_info->mumimo.mu.role == MFC_MU_ROLE_AP &&
	    mac_info->test_opt.lb_mode == MFC_TRX_MODE_DLLB) {
		val8 = api->halmac_reg_read_8(mac_adapter, REG_RCR);  
		api->halmac_reg_write_8(mac_adapter, REG_RCR, val8 | BIT_AAP);
		if (adapter->chip_type.chip_id != _CHIP_ID_8822B)
			api->halmac_reg_write_8(mac_adapter, 0x167e, 0x50);
	}

	if (adapter->chip_type.chip_id != _CHIP_ID_8824B) {
		ret = wmac_cfg_mumimo(adapter);
		if (!ret)
			return ret;
	} else {
		ret = wmac_cfg_mumimo_new(adapter);
		if (!ret)
			return ret;
	}

	return 1;
}

u8 wmac_init_mac(struct ui_adapter_e *adapter,
		 u8 (*intf_rst_cb)(struct ui_adapter_e *adapter))
{
	/* Reset interface before mac init */
	/* to avoid unsynced hardware setting issue */
	if (intf_rst_cb)
		intf_rst_cb(adapter);

	if (!wmac_intf_info(adapter))
		return 0;

	if (!wmac_init_system(adapter))
		return 0;

	if (!_wmac_init_mac(adapter))
		return 0;

	if (!wmac_init_feature(adapter))
		return 0;

	if (!wmac_init_mimo(adapter))
		return 0;

	return 1;
}

/* Only support read 1byte currently */
/* Reference to usbphy_read_88xx */
u8 wmac_usb_phy_read8(struct ui_adapter_e *adapter, enum wmac_usb_type type,
		      u32 offset)
{
	return 1;
}

/* Only support write 1byte currently */
/* Reference to usbphy_write_88xx */
u8 wmac_usb_phy_write8(struct ui_adapter_e *adapter, enum wmac_usb_type type,
		       u32 offset, u8 val)
{
	return 1;
}

