/*
 * qcomposit.c -- composite USB device driver for Quasar
 *
 * Copyright (c) 2014, 2015, The Linux Foundation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */
#include <linux/kernel.h>
#include <linux/utsname.h>
#include <linux/device.h>
#include <linux/version.h>

#include "quasargadget.h"
#include <linux/usb/composite.h>
#if LINUX_VERSION_CODE <= KERNEL_VERSION(4,1,0)
#include "../drivers/usb/gadget/gadget_chips.h"
#include "../drivers/usb/gadget/u_serial.h"
#else
#include "gadget_chips.h"
#include "u_serial.h"
#endif

/*-------------------------------------------------------------------------*/
/*
 * Kbuild is not very cooperative with respect to linking separately
 * compiled library objects into one module.  So for now we won't use
 * separate compilation ... ensuring init/exit sections work to shrink
 * the runtime footprint, and giving us at least some parts of what
 * a "gcc --combine ... part1.c part2.c part3.c ... " build would.
 */
#include "../drivers/usb/gadget/composite.c"
#include "../drivers/usb/gadget/usbstring.c"
#include "../drivers/usb/gadget/config.c"
#if LINUX_VERSION_CODE <= KERNEL_VERSION(4,1,0)
#include "../drivers/usb/gadget/epautoconf.c"
#else
#include "epautoconf.c"
#endif

extern int __init quasar_printer_add(struct usb_configuration *c);
extern int __init quasar_scanner_add(struct usb_configuration *c);
extern int __init quasar_debug_add(struct usb_configuration *c);
extern int __init quasar_storage_add(struct usb_configuration *c);
extern int __init quasar_ippusb_add(struct usb_configuration *c, int devminor);
extern int __init quasar_storage_init(struct usb_composite_dev *cdev);
extern int __init quasar_storage_put(struct usb_composite_dev *cdev);
extern void quasar_storage_unmap(void);

/*-------------------------------------------------------------------------*/

#if USB_COMPLIANCE_TEST
static int hasPrinter = 0;
#else
static int hasPrinter = 1;
#endif
module_param(hasPrinter, int, 0);
MODULE_PARM_DESC(hasPrinter, "Printer");

#if USB_COMPLIANCE_TEST
static int hasScanner = 0;
#else
static int hasScanner = 1;
#endif
module_param(hasScanner, int, 0);
MODULE_PARM_DESC(hasScanner, "Scanner");

#if USB_COMPLIANCE_TEST
static int hasDebug = 0;
#else
static int hasDebug = 1;
#endif
module_param(hasDebug, int, S_IRUGO);
MODULE_PARM_DESC(hasDebug, "Debug");

#if USB_COMPLIANCE_TEST
static int hasStorage = 1;
#else
static int hasStorage = 0;
#endif
module_param(hasStorage, int, 0);
MODULE_PARM_DESC(hasStorage, "Storage");

#define MAX_IPPUSB_INTF 4
#if USB_COMPLIANCE_TEST
unsigned nIPPUSB = 0;
#else
unsigned nIPPUSB = 0;
#endif
module_param(nIPPUSB, int, 0);
MODULE_PARM_DESC(nIPPUSB, "Number of IPP-USB interfaces, default=0");

#if 0
/* 6 digits for the unique part of serial number */
char serialNumber[6];
int nCount = 0;
module_param_array(serialNumber, byte, &nCount, 0);
MODULE_PARM_DESC(serialNumber, "User-defined USB serial number");
#endif

#define DRIVER_DESC	    "Ricoh"
#define DRIVER_NAME	    "pmxusb"
#define DRIVER_VERSION	"2021.12.02"

#define DEVICE_DESC	"Quasar-USB-Device"
#define DEVICE_NAME "JiaoS MFP USB Device"

static const char longname[40]  = DEVICE_NAME;
static const char shortname[40] = DEVICE_DESC;

/* These are VID/PID defaults.  The composite module code
 * (see linux.../drivers/usb/gadget/composite.c) has module parameters
 * that, if used, will override these here
 */
USB_GADGET_COMPOSITE_OPTIONS();

#ifndef DRIVER_VENDOR_ID
	#define DRIVER_VENDOR_ID	QBIT_USB_VID
#endif
#ifndef DRIVER_PRODUCT_ID
	#define DRIVER_PRODUCT_ID	QUASAR_USB_PID_AIO
/*	#define DRIVER_PRODUCT_ID	QUASAR_USB_PID_PRINT */
/*	#define DRIVER_PRODUCT_ID	QUASAR_USB_PID_SCAN  */
#endif

struct class *pmxusb_gadget_class;
dev_t usb_gadget_devno;
int major = -1;
int minors = 1;

/*-------------------------------------------------------------------------*/

static struct usb_device_descriptor device_desc = {
	.bLength =		sizeof device_desc,
	.bDescriptorType =	USB_DT_DEVICE,

	/* Must be 0x0210 for a SS device in HS/FS/LS connection */
	.bcdUSB =		__constant_cpu_to_le16(0x0210),
	.bDeviceClass =		USB_CLASS_PER_INTERFACE,

	.bDeviceSubClass =	0,
	.bDeviceProtocol =	0,

	.idVendor =		__constant_cpu_to_le16(DRIVER_VENDOR_ID),
	.idProduct =		__constant_cpu_to_le16(DRIVER_PRODUCT_ID),
	.bNumConfigurations =	1,
};

#ifdef CONFIG_USB_OTG
static struct usb_otg_descriptor otg_descriptor = {
	.bLength =		sizeof otg_descriptor,
	.bDescriptorType =	USB_DT_OTG,
	.bmAttributes =		USB_OTG_SRP | USB_OTG_HNP,
};

const struct usb_descriptor_header *otg_desc[] = {
	(struct usb_descriptor_header *) &otg_descriptor,
	NULL,
};
#endif

/* string IDs are assigned dynamically */

static char manufacturer[64];

//char serial_header[8] = "SN:";

/* default 32 hex digit serial number (ASCII)
 * The kernel will convert the string of serial number from ASCII to UTF-16.
 */
char serial[] = "1234567890ABCDEF1234567890ABCDEF";

#define STRING_MANUFACTURER	0
#define STRING_PRODUCT		1
#define STRING_SERIAL		2
#define STRING_CONFIG		3

static struct usb_string strings_dev[] = {
	{STRING_MANUFACTURER,	manufacturer},
	{STRING_PRODUCT,	longname},
	{STRING_SERIAL,		serial},
	{STRING_CONFIG,		shortname},
	{  }			/* end of list */
};

static struct usb_gadget_strings stringtab_dev = {
	.language	= 0x0409,	/* en-us */
	.strings	= strings_dev,
};

static struct usb_gadget_strings *dev_strings[] = {
	&stringtab_dev,
	NULL,
};

static int __init quasar_bind_config(struct usb_configuration *c)
{
	int rc = 0;
	int i;
	
	if (hasPrinter) {
		rc += quasar_printer_add(c);
	}
	if (hasScanner) {
		rc += quasar_scanner_add(c);
	}
	if (hasStorage) {
		rc += quasar_storage_add(c);
	}
	for (i = 0; i < nIPPUSB && i < MAX_IPPUSB_INTF; i++) {
		rc += quasar_ippusb_add(c, i + 4);
	}	
 
	if (hasDebug) {
		rc += quasar_debug_add(c);
	}
 
	//printk("%s, version: %s\n", DRIVER_NAME, DRIVER_VERSION);
	return rc;
}

/*
 * The setup() callback implements all the ep0 functionality that's not
 * handled lower down and not handled by the functions.  Requests
 * to interfaces go directly to the functions not here, so this is
 * a no-op for now
 */
static int
quasar_setup(struct usb_configuration *c,
		const struct usb_ctrlrequest *ctrl)
{
	struct usb_request	*req = c->cdev->req;
	int			value = -EOPNOTSUPP;
	u16			wIndex = le16_to_cpu(ctrl->wIndex);
	u16			wValue = le16_to_cpu(ctrl->wValue);
	u16			wLength = le16_to_cpu(ctrl->wLength);

	switch (ctrl->bRequestType&USB_TYPE_MASK) {

	default:
		VDBG(c->cdev,
			"unknown ctrl req%02x.%02x v%04x i%04x l%d\n",
			ctrl->bRequestType, ctrl->bRequest,
			wValue, wIndex, wLength);
		break;
	}
	/* respond with data transfer before status phase? */
	if (value >= 0) {
		req->length = value;
		req->zero = value < wLength;
		value = usb_ep_queue(c->cdev->gadget->ep0, req, GFP_ATOMIC);
		if (value < 0) {
			DBG(c->cdev, "ep_queue --> %d\n", value);
			req->status = 0;
		}
	}
	/* host either stalls (value < 0) or reports success */
	return value;
}

static struct usb_configuration qcomposite_driver_ops = {
	.label		= "Quasar-Device",
	.strings	= dev_strings,
	.setup		= quasar_setup,
	.bConfigurationValue = 1,
	.bmAttributes	= USB_CONFIG_ATT_SELFPOWER,
};

/*-------------------------------------------------------------------------*/

static int __init quasar_bind(struct usb_composite_dev *cdev)
{
	struct usb_gadget	*gadget = cdev->gadget;
	int			id;
	int			rc;

	set_gadget_data(cdev->gadget, cdev);

#if 0 // tony
	if (nCount) {
		/* Use variable USB serial number. Format is
		 * - Fixed prefix (12 hex digits): Use the default one.
		 * - USB VID (4 hex digits)
		 * - USB PID (4 hex digits)
		 * - Unique MAC address (12 hex digits)
		 */
		snprintf(&serial[12], sizeof(serial)-12,
			"%04X%04X%02X%02X%02X%02X%02X%02X",
			device_desc.idVendor, device_desc.idProduct,
			serialNumber[0], serialNumber[1], serialNumber[2],
			serialNumber[3], serialNumber[4], serialNumber[5]);
	}
#endif
	/* Allocate string descriptor numbers ... note that string
	 * contents can be overridden by the composite_dev glue.
	 */
	id = usb_string_id(cdev);
	if (id < 0)
		return id;
	strings_dev[STRING_MANUFACTURER].id = id;
	device_desc.iManufacturer = id;

	id = usb_string_id(cdev);
	if (id < 0)
		return id;
	strings_dev[STRING_PRODUCT].id = id;
	device_desc.iProduct = id;

	id = usb_string_id(cdev);
	if (id < 0)
		return id;
	strings_dev[STRING_SERIAL].id = id;
	device_desc.iSerialNumber = id;

	id = usb_string_id(cdev);
	if (id < 0)
		return id;
	strings_dev[STRING_CONFIG].id = id;
	qcomposite_driver_ops.iConfiguration = id;
	
#ifdef CONFIG_USB_OTG
	/* support OTG systems */
	if (gadget_is_otg(cdev->gadget)) {
		qcomposite_driver_ops.descriptors = otg_desc;
		qcomposite_driver_ops.bmAttributes |= USB_CONFIG_ATT_WAKEUP;
	}
#endif
	device_desc.bcdDevice = __constant_cpu_to_le16(0x0100);
	snprintf(manufacturer, sizeof manufacturer, "%s %s with %s",
		init_utsname()->sysname, init_utsname()->release,
		gadget->name);

	if (hasStorage) {
		rc = quasar_storage_init(cdev);
		if (rc) {
			ERROR(cdev, "Can't setup storage\n"); 
		}
	}
	/* Register configuration(s)
	 */
	rc = usb_add_config(cdev, &qcomposite_driver_ops, quasar_bind_config);
	if (rc) {
		ERROR(cdev, "Can't add config\n"); 
	}
	if (hasStorage) {
		rc = quasar_storage_put(cdev);
		if (rc) {
			ERROR(cdev, "Can't put storage\n"); 
		}
	}

	/* Override VID and PID if specified by module parameters */
	usb_composite_overwrite_options(cdev, &coverwrite);
	
	INFO(cdev, "%s, version: " DRIVER_VERSION "\n", longname);

	return rc;
}

static struct usb_composite_driver quasar_driver_ops = {
	.name		= "Quasar-USB",
	.dev		= &device_desc,
	.strings	= dev_strings,
	.bind		= quasar_bind,
};

MODULE_LICENSE("GPL v2");

static int __init init(void)
{
	int rc;

	pmxusb_gadget_class = class_create(THIS_MODULE, "primax_usb_gadget");
	if (IS_ERR(pmxusb_gadget_class)) {
		rc = PTR_ERR(pmxusb_gadget_class);
		return rc;
	}

	rc = alloc_chrdev_region(&usb_gadget_devno, 0, 1,
			"USB mfp gadget");
	if (rc) {
		class_destroy(pmxusb_gadget_class);
		return rc;
	}
	major = MAJOR(usb_gadget_devno);

	rc = usb_composite_probe(&quasar_driver_ops);

	return rc;
}
module_init(init);

static void __exit cleanup(void)
{
	if (hasStorage)
		quasar_storage_unmap();

	usb_composite_unregister(&quasar_driver_ops);
	unregister_chrdev_region(usb_gadget_devno, 1);
	class_destroy(pmxusb_gadget_class);
}
module_exit(cleanup);

MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_AUTHOR("Tony Tan");
MODULE_VERSION(DRIVER_VERSION);

//#endif // version hack
