/*
 * f_printer.c -- Printer gadget function driver
 *
 * Copyright (c) 2014, 2015, The Linux Foundation. All rights reserved.
 *
 * based in part on printer gadget driver by Craig W. Nadler
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/ioport.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/version.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/timer.h>
#include <linux/list.h>
#include <linux/interrupt.h>
#include <linux/utsname.h>
#include <linux/device.h>
#include <linux/moduleparam.h>
#include <linux/fs.h>
#include <linux/poll.h>
#include <linux/types.h>
#include <linux/ctype.h>
#include <linux/cdev.h>

#include <asm/byteorder.h>
#include <linux/io.h>
#include <linux/irq.h>
#include <linux/uaccess.h>
#include <asm/unaligned.h>

/* include first here so DUALSPEED is defined for composite, etc.
*/
#include "quasargadget.h"
#include <linux/usb/ch9.h>
#include <linux/usb/gadget.h>
#include <linux/usb/composite.h>
#include <asm/ioctls.h>		/* for FIONBIO */
#if LINUX_VERSION_CODE <= KERNEL_VERSION(4,1,0)
#include "../drivers/usb/gadget/gadget_chips.h"
#else
#include "gadget_chips.h"
#endif
#include "quasardev.h"

extern int pmxdev_buf_flush(struct quasar_dev *dev, int rx);

static int ipp_printer_buf_flush(struct quasar_dev *dev, int rx);

/*-------------------------------------------------------------------------*/

/* Number of requests to allocate per endpoint, not used for ep0. */
static unsigned qlenIPPUSB = 8;
module_param(qlenIPPUSB, uint, S_IRUGO|S_IWUSR);

#define QLEN	qlenIPPUSB

/*
 * DESCRIPTORS ... most are static, but strings and interface numbers
 * are built on demand.
 */
static const struct usb_interface_descriptor printer_intf = {
	.bLength =		sizeof printer_intf,
	.bDescriptorType =	USB_DT_INTERFACE,
	.bAlternateSetting =	0,
	.bInterfaceNumber =	0,	/* filled in by bind */
	.bNumEndpoints =	2,
	.bInterfaceClass =	USB_CLASS_PRINTER,
	.bInterfaceSubClass =	1,	/* Printer Sub-Class */
	.bInterfaceProtocol =	4,	/* IPP-USB */
	.iInterface =		0
};

static const struct usb_endpoint_descriptor fs_ep_in_desc = {
	.bLength =		USB_DT_ENDPOINT_SIZE,
	.bDescriptorType =	USB_DT_ENDPOINT,
	.bEndpointAddress =	USB_DIR_IN,
	.bmAttributes =		USB_ENDPOINT_XFER_BULK,
	/* autoconfig .wMaxPacketSize =	__constant_cpu_to_le16(512) */
};

static const struct usb_endpoint_descriptor fs_ep_out_desc = {
	.bLength =		USB_DT_ENDPOINT_SIZE,
	.bDescriptorType =	USB_DT_ENDPOINT,
	.bEndpointAddress =	USB_DIR_OUT,
	.bmAttributes =		USB_ENDPOINT_XFER_BULK,
	/* autoconfig .wMaxPacketSize =	__constant_cpu_to_le16(512) */
};

/*
 * usb 2.0 devices need to expose both high speed and full speed
 * descriptors, unless they only run at full speed.
 */
static const struct usb_endpoint_descriptor hs_ep_in_desc = {
	.bLength =		USB_DT_ENDPOINT_SIZE,
	.bDescriptorType =	USB_DT_ENDPOINT,
	.bmAttributes =		USB_ENDPOINT_XFER_BULK,
	.wMaxPacketSize =	__constant_cpu_to_le16(512)
};

static const struct usb_endpoint_descriptor hs_ep_out_desc = {
	.bLength =		USB_DT_ENDPOINT_SIZE,
	.bDescriptorType =	USB_DT_ENDPOINT,
	.bmAttributes =		USB_ENDPOINT_XFER_BULK,
	.wMaxPacketSize =	__constant_cpu_to_le16(512)
};

static struct usb_qualifier_descriptor dev_qualifier = {
	.bLength =		sizeof dev_qualifier,
	.bDescriptorType =	USB_DT_DEVICE_QUALIFIER,
	.bcdUSB =		__constant_cpu_to_le16(0x0200),
	.bDeviceClass =		USB_CLASS_PRINTER,
	.bNumConfigurations =	1
};

/*
 * usb 3.0 descriptors.
 */
static struct usb_endpoint_descriptor ss_ep_in_desc = {
	.bLength =		USB_DT_ENDPOINT_SIZE,
	.bDescriptorType =	USB_DT_ENDPOINT,
	.bmAttributes =		USB_ENDPOINT_XFER_BULK,
	.wMaxPacketSize =	cpu_to_le16(1024),
};

static struct usb_ss_ep_comp_descriptor ss_ep_in_comp_desc = {
	.bLength =		sizeof ss_ep_in_comp_desc,
	.bDescriptorType =	USB_DT_SS_ENDPOINT_COMP,
	.bMaxBurst =	15,   /* enable burst mode */
	.bmAttributes =	0,
	.wBytesPerInterval =	0,
};

static struct usb_endpoint_descriptor ss_ep_out_desc = {
	.bLength =		USB_DT_ENDPOINT_SIZE,
	.bDescriptorType =	USB_DT_ENDPOINT,
	.bmAttributes =		USB_ENDPOINT_XFER_BULK,
	.wMaxPacketSize =	cpu_to_le16(1024),
};

static struct usb_ss_ep_comp_descriptor ss_ep_out_comp_desc = {
	.bLength =		sizeof ss_ep_out_comp_desc,
	.bDescriptorType =	USB_DT_SS_ENDPOINT_COMP,
	.bMaxBurst =	15,   /* enable burst mode */
	.bmAttributes =	0,
	.wBytesPerInterval =	0,
};

/* One instance of an ippusb interface has this info
*/
struct ippusb_intf {
	int devminor;
	struct usb_interface_descriptor interface;
	struct usb_endpoint_descriptor   fs_ep_in_desc;
	struct usb_endpoint_descriptor   fs_ep_out_desc;
	struct usb_descriptor_header    *fs_printer_function[4];
	struct usb_endpoint_descriptor   hs_ep_in_desc;
	struct usb_endpoint_descriptor   hs_ep_out_desc;
	struct usb_descriptor_header    *hs_printer_function[4];
	struct usb_endpoint_descriptor   ss_ep_in_desc;
	struct usb_ss_ep_comp_descriptor ss_ep_in_comp_desc;
	struct usb_endpoint_descriptor   ss_ep_out_desc;
	struct usb_ss_ep_comp_descriptor ss_ep_out_comp_desc;
	struct usb_descriptor_header    *ss_printer_function[6];
};

/* function-specific strings: */

static struct usb_string strings_printer[] = {
	[0].s = "IPPUSB Data1",
	[1].s = "IPPUSB Data2",
	[2].s = "IPPUSB Data3",
	[3].s = "IPPUSB Data4",
	{  }			/* end of list */
};

static struct usb_gadget_strings stringtab_printer = {
	.language	= 0x0409,	/* en-us */
	.strings	= strings_printer,
};

static struct usb_gadget_strings *printer_strings[] = {
	&stringtab_printer,
	NULL,
};

/*-------------------------------------------------------------------------*/
extern int major;
extern int minors;
extern struct class *pmxusb_gadget_class;

/*-------------------------------------------------------------------------*/

static int
printer_open(struct inode *inode, struct file *fd)
{
	struct quasar_dev	*dev;
	int			ret = -EBUSY;

	dev = container_of(inode->i_cdev, struct quasar_dev, my_chardev);

	if (down_interruptible(&dev->sem)) {
		return -EINTR;
	}
	if (!dev->cdev_open) {
		dev->cdev_open = 1;
		fd->private_data = dev;
		ret = 0;
	}
	up(&dev->sem);

	DBG(dev->cdev, "printer_open returned %x\n", ret);
	return ret;
}

static int
printer_close(struct inode *inode, struct file *fd)
{
	struct quasar_dev	*dev = fd->private_data;

	if (down_interruptible(&dev->sem)) {
		return -EINTR;
	}
	dev->cdev_open = 0;
	fd->private_data = NULL;
	up(&dev->sem);

	DBG(dev->cdev, "printer_close\n");

	return 0;
}

static ssize_t
printer_read(struct file *fd, char __user *buf, size_t len, loff_t *ptr)
{
	return qdev_read(fd, buf, len, ptr);
}

static ssize_t
printer_write(struct file *fd, const char __user *buf, size_t len, loff_t *ptr)
{
	return qdev_write(fd, buf, len, ptr);
}

static int
printer_fsync(struct file *fd, loff_t start, loff_t end, int datasync)
{
	return qdev_fsync(fd, datasync);
}

static unsigned int
printer_poll(struct file *fd, poll_table *wait)
{
	return qdev_poll(fd, wait);
}

static long
printer_ioctl(struct file *fd, unsigned int code, unsigned long arg)
{
	struct quasar_dev	*dev = fd->private_data;
	int			status = 0;
	int flushFlag = 0;

	DBG(dev->cdev, "printer_ioctl: cmd=0x%4.4x, arg=%lu\n", code, arg);

	/* handle ioctls */

	switch (code) {
	case FIONBIO:
		if (arg) {
			fd->f_flags |= O_NONBLOCK;
		} else {
			fd->f_flags &= ~O_NONBLOCK;
		}
		break;

	case FIOPMXFLUSHBUF:
		status = 0;
		flushFlag = 1;
		break;

	default:
		/* could not handle ioctl */
		DBG(dev->cdev, "printer_ioctl: ERROR cmd=0x%4.4xis not supported\n",
				code);
		status = -ENOTTY;
	}

	if( flushFlag) {
		ipp_printer_buf_flush(dev, arg);
	}
	
	return status;
}

static int ipp_printer_buf_flush(struct quasar_dev *dev, int rx)
{
	INFO(dev, "IPP received buffer flush Request\n");
	return pmxdev_buf_flush(dev, rx);
}


/* used after endpoint configuration */
static struct file_operations printer_io_operations = {
	.owner =	THIS_MODULE,
	.open =		printer_open,
	.read =		printer_read,
	.write =	printer_write,
	.fsync =	printer_fsync,
	.poll =		printer_poll,
	.unlocked_ioctl = printer_ioctl,
	.compat_ioctl	= printer_ioctl,
	.release =	printer_close
};

/*-------------------------------------------------------------------------*/

static int
enable_printer(struct usb_composite_dev *cdev, struct quasar_dev *dev)
{
	int			result = 0;

	if (config_ep_by_speed(dev->gadget, &dev->function, dev->in_ep)) {
		dev->in_ep->desc = NULL;
		return -EINVAL;
	}
	dev->in_ep->driver_data = dev;

	if (config_ep_by_speed(dev->gadget, &dev->function, dev->out_ep)) {
		dev->out_ep->desc = NULL;
		return -EINVAL;
	}
	dev->out_ep->driver_data = dev;

	result = usb_ep_enable(dev->in_ep);
	if (result != 0) {
		DBG(cdev, "enable %s --> %d\n", dev->in_ep->name, result);
		goto done;
	}

	result = usb_ep_enable(dev->out_ep);
	if (result != 0) {
		DBG(cdev, "enable %s --> %d\n", dev->in_ep->name, result);
		goto done;
	}
done:
	/* on error, disable any endpoints  */
	if (result != 0) {
		(void) usb_ep_disable(dev->in_ep);
		(void) usb_ep_disable(dev->out_ep);
		dev->in_ep = NULL;
		dev->out_ep = NULL;
	}
	if (! result)
		dev->enabled = 1;

	/* caller is responsible for cleanup on error */
	return result;
}

static void disable_printer(struct quasar_dev *dev)
{
	if (! dev->enabled)
		return;

	DBG(dev->cdev, "%s\n", __func__);

	if (dev->in_ep)
		usb_ep_disable(dev->in_ep);

	if (dev->out_ep)
		usb_ep_disable(dev->out_ep);
	dev->enabled = 0;
}

static void printer_soft_reset(struct quasar_dev *dev)
{
	struct usb_request	*req;

	INFO(dev, "Received Printer Reset Request\n");

	if (usb_ep_disable(dev->in_ep))
		DBG(dev->cdev, "Failed to disable USB in_ep\n");
	if (usb_ep_disable(dev->out_ep))
		DBG(dev->cdev, "Failed to disable USB out_ep\n");

	if (dev->current_rx_req != NULL) {
		list_add(&dev->current_rx_req->list, &dev->rx_reqs);
		dev->current_rx_req = NULL;
	}
	dev->current_rx_bytes = 0;
	dev->current_rx_buf = NULL;
	dev->reset_request = 1;

	while (likely(!(list_empty(&dev->rx_buffers)))) {
		req = container_of(dev->rx_buffers.next, struct usb_request,
				list);
		list_del_init(&req->list);
		list_add(&req->list, &dev->rx_reqs);
	}

	while (likely(!(list_empty(&dev->rx_reqs_active)))) {
		req = container_of(dev->rx_buffers.next, struct usb_request,
				list);
		list_del_init(&req->list);
		list_add(&req->list, &dev->rx_reqs);
	}

	while (likely(!(list_empty(&dev->tx_reqs_active)))) {
		req = container_of(dev->tx_reqs_active.next,
				struct usb_request, list);
		list_del_init(&req->list);
		list_add(&req->list, &dev->tx_reqs);
	}

	if (usb_ep_enable(dev->in_ep))
		DBG(dev->cdev, "Failed to enable USB in_ep\n");
	if (usb_ep_enable(dev->out_ep))
		DBG(dev->cdev, "Failed to enable USB out_ep\n");

	wake_up_interruptible(&dev->rx_wait);
	wake_up_interruptible(&dev->tx_wait);
	wake_up_interruptible(&dev->tx_flush_wait);
}

static int
printer_setup(struct usb_function *f, const struct usb_ctrlrequest *ctrl)
{
	struct quasar_dev	*dev = func_to_qdev(f);
	struct usb_composite_dev *cdev = dev->cdev;
	struct usb_request	*req = cdev->req;
	struct ippusb_intf	*itf = (struct ippusb_intf*)dev->priv;
	int			value = -EOPNOTSUPP;
	u16			wIndex = le16_to_cpu(ctrl->wIndex);
	u16			wValue = le16_to_cpu(ctrl->wValue);
	u16			wLength = le16_to_cpu(ctrl->wLength);

	DBG(cdev, "prt ctrl req%02x.%02x v%04x i%04x l%d\n",
		ctrl->bRequestType, ctrl->bRequest, wValue, wIndex, wLength);

	switch (ctrl->bRequestType&USB_TYPE_MASK) {

	case USB_TYPE_CLASS:
		switch (ctrl->bRequest) {
		case 0: /* Get the IEEE-1284 PNP String */
			value = 0;
			break;

		case 1: /* Get Port Status */
			if (wIndex != itf->interface.bInterfaceNumber)
				break;
			*(u8 *)req->buf = 0;
			value = min(wLength, (u16) 1);
			break;

		case 2: /* Soft Reset */
			if (wIndex != itf->interface.bInterfaceNumber)
				break;
			printer_soft_reset(dev);
			value = 0;
			break;

		default:
			goto unknown;
		}
		break;

	default:
unknown:
		VDBG(cdev,
			"unknown ctrl req%02x.%02x v%04x i%04x l%d\n",
			ctrl->bRequestType, ctrl->bRequest,
			wValue, wIndex, wLength);
		break;
	}
	/* respond with data transfer before status phase? */
	if (value >= 0) {
		req->length = value;
		req->zero = value < wLength;
		value = usb_ep_queue(cdev->gadget->ep0, req, GFP_ATOMIC);
		if (value < 0) {
			DBG(cdev, "ep_queue --> %d\n", value);
			req->status = 0;
		}
	}
	/* host either stalls (value < 0) or reports success */
	return value;
}

static void
printer_unbind(struct usb_configuration *c, struct usb_function *f)
{
	struct usb_composite_dev *cdev;
	struct quasar_dev	*dev;
	struct usb_request	*req;

	cdev = c->cdev;
	dev = func_to_qdev(f);

	DBG(cdev, "%s\n", __func__);

	/* Remove sysfs files */
	device_destroy(pmxusb_gadget_class, dev->cdevno);

	/* Remove Character Device */
	cdev_del(&dev->my_chardev);

	/* we must already have been disconnected ... no i/o may be active */
	WARN_ON(!list_empty(&dev->tx_reqs_active));
	WARN_ON(!list_empty(&dev->rx_reqs_active));

	/* Free all memory for this driver. */
	while (!list_empty(&dev->tx_reqs)) {
		req = container_of(dev->tx_reqs.next, struct usb_request,
				list);
		list_del(&req->list);
		qdev_req_free(dev->in_ep, req);
	}

	if (dev->current_rx_req != NULL)
		qdev_req_free(dev->out_ep, dev->current_rx_req);

	while (!list_empty(&dev->rx_reqs)) {
		req = container_of(dev->rx_reqs.next,
				struct usb_request, list);
		list_del(&req->list);
		qdev_req_free(dev->out_ep, req);
	}
	while (!list_empty(&dev->rx_buffers)) {
		req = container_of(dev->rx_buffers.next,
				struct usb_request, list);
		list_del(&req->list);
		qdev_req_free(dev->out_ep, req);
	}
	usb_free_descriptors(f->fs_descriptors);
	if (f->hs_descriptors)
		usb_free_descriptors(f->hs_descriptors);
#if defined(CONFIG_SOC_QUASAR6300) || defined(BOARD_TYPE_EP4) || defined(BOARD_TYPE_EP3L)
	if (f->ss_descriptors)
		usb_free_descriptors(f->ss_descriptors);
#endif
}

static int __init
printer_bind(struct usb_configuration *c, struct usb_function *f)
{
	struct usb_composite_dev *cdev;
	struct quasar_dev	*dev;
	struct ippusb_intf	*itf;
	struct usb_ep		*in_ep, *out_ep;
	int			status = -ENOMEM;
	u32			i;
	struct usb_request	*req;
	int strNo = 0;
	const char *ippDev[] = {
		"j_ippusb1",
		"j_ippusb2",
		"j_ippusb3",
		"j_ippusb4",
	};

	cdev = c->cdev;
	dev = func_to_qdev(f);
	dev->cdev = cdev;
	
	/* allocate instance-specific interface IDs */
	status = usb_interface_id(c, f);
	if (status < 0)
		goto fail;
	itf = (struct ippusb_intf *)dev->priv;
	itf->interface.bInterfaceNumber = status;
	strNo = itf->devminor - 4;
	status = -ENODEV;

	/* create the char device for this printer as qusb4 
	*/
	dev->cdevno = MKDEV(major, minors++);
	
	/* create the char device for this printer as qusb4 
	*/
	//dev->cdevno = MKDEV(QUSB_MAJOR, itf->devminor);
	dev->cdevno = MKDEV(major, minors++);

	/* Setup the sysfs files for the printer gadget. */
	dev->pdev = device_create(pmxusb_gadget_class, NULL, dev->cdevno,
				  NULL, ippDev[strNo]);

	if (IS_ERR(dev->pdev)) {
		ERROR(dev, "Failed to create device: j_ipp_printer\n");
		status = PTR_ERR(dev->pdev);
		goto fail;
	}
	
	/*
	 * Register a character device as an interface to a user mode
	 * program that handles the printer specific functionality.
	 */
	cdev_init(&dev->my_chardev, &printer_io_operations);
	dev->my_chardev.owner = THIS_MODULE;
	
	status = cdev_add(&dev->my_chardev, dev->cdevno, 1);
	if (status) {
		ERROR(cdev, "Failed to open char device\n");
		goto fail;
	}
	DBG(cdev, "usb printer added on dev %d (%d)\n", major, dev->cdevno);

	/* all we really need is bulk IN/OUT */
	in_ep = usb_ep_autoconfig(cdev->gadget, &itf->fs_ep_in_desc);
	if (!in_ep) {
		dev_err(&cdev->gadget->dev, "can't autoconfigure on %s\n",
			cdev->gadget->name);
		return -ENODEV;
	}
	in_ep->driver_data = dev;	/* claim */

	out_ep = usb_ep_autoconfig(cdev->gadget, &itf->fs_ep_out_desc);
	if (!out_ep) {
		dev_err(&cdev->gadget->dev, "can't autoconfigure on %s\n",
			cdev->gadget->name);
		return -ENODEV;
	}
	out_ep->driver_data = dev;	/* claim */

	itf->hs_ep_in_desc.bEndpointAddress = itf->fs_ep_in_desc.bEndpointAddress;
	itf->hs_ep_out_desc.bEndpointAddress = itf->fs_ep_out_desc.bEndpointAddress;
	itf->ss_ep_in_desc.bEndpointAddress = itf->fs_ep_in_desc.bEndpointAddress;
	itf->ss_ep_out_desc.bEndpointAddress = itf->fs_ep_out_desc.bEndpointAddress;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4,19,0)
	status = usb_assign_descriptors(f, itf->fs_printer_function, itf->hs_printer_function,
			itf->ss_printer_function, NULL);
#else     
	status = usb_assign_descriptors(f, itf->fs_printer_function,
			itf->hs_printer_function, itf->ss_printer_function);
#endif
	if (status)
		goto fail;

	dev->in_ep = in_ep;
	dev->out_ep = out_ep;

	sema_init(&dev->sem, 1);
	spin_lock_init(&dev->lock_io);
	INIT_LIST_HEAD(&dev->tx_reqs);
	INIT_LIST_HEAD(&dev->tx_reqs_active);
	INIT_LIST_HEAD(&dev->rx_reqs);
	INIT_LIST_HEAD(&dev->rx_reqs_active);
	INIT_LIST_HEAD(&dev->rx_buffers);
	init_waitqueue_head(&dev->rx_wait);
	init_waitqueue_head(&dev->tx_wait);
	init_waitqueue_head(&dev->tx_flush_wait);

	dev->enabled = 0;
	dev->cdev_open = 0;
	dev->current_rx_req = NULL;
	dev->current_rx_bytes = 0;
	dev->current_rx_buf = NULL;

	/* alloc buffer rings
	*/
	for (i = 0; i < QLEN; i++) {
		req = qdev_req_alloc(dev->in_ep, USB_BUFSIZE, GFP_KERNEL, i);
		if (!req) {
			while (!list_empty(&dev->tx_reqs)) {
				req = container_of(dev->tx_reqs.next,
						struct usb_request, list);
				list_del(&req->list);
				qdev_req_free(dev->in_ep, req);
			}
			return -ENOMEM;
		}
		list_add(&req->list, &dev->tx_reqs);
	}
	for (i = 0; i < QLEN; i++) {
		req = qdev_req_alloc(dev->out_ep, USB_BUFSIZE, GFP_KERNEL, i);
		if (!req) {
			while (!list_empty(&dev->rx_reqs)) {
				req = container_of(dev->rx_reqs.next,
						struct usb_request, list);
				list_del(&req->list);
				qdev_req_free(dev->out_ep, req);
			}
			return -ENOMEM;
		}
		list_add(&req->list, &dev->rx_reqs);
	}
	INFO(cdev, "using %s, OUT %s IN %s\n", cdev->gadget->name, out_ep->name,
			in_ep->name);

	return 0;

fail:
	printer_unbind(c, f);
	return status;
}

/*-------------------------------------------------------------------------*/

static int printer_set_alt(struct usb_function *f,
		unsigned intf, unsigned alt)
{
	struct quasar_dev	*dev = func_to_qdev(f);
	struct usb_composite_dev *cdev = f->config->cdev;

	/* we know alt is zero */
	if (dev->in_ep->driver_data)
		disable_printer(dev);
	return enable_printer(cdev, dev);
}

static void printer_disable(struct usb_function *f)
{
	struct quasar_dev *dev = func_to_qdev(f);
	disable_printer(dev);
}

static void printer_suspend(struct usb_function *f)
{
	struct usb_composite_dev *cdev = f->config->cdev;

	if (cdev->gadget->speed == USB_SPEED_UNKNOWN)
		return;

	DBG(cdev, "%s\n", __func__);
}

static void printer_resume(struct usb_function *f)
{
	struct usb_composite_dev *cdev = f->config->cdev;

	DBG(cdev, "%s\n", __func__);
}

int __init quasar_ippusb_add(struct usb_configuration *c, int devminor)
{
	struct quasar_dev	*dev;
	struct ippusb_intf	*itf;
	int			status;
	int			id;

	/* allocate string ID(s) */
	id = usb_string_id(c->cdev);
	if (id < 0)
		return id;
	strings_printer[devminor-4].id = id;

	dev = kzalloc(sizeof *dev, GFP_KERNEL);
	if (!dev)
		return -ENOMEM;

	dev->gadget = c->cdev->gadget;
	dev->priv = kzalloc(sizeof(struct ippusb_intf), GFP_KERNEL);
	if (! dev->priv) {
		kfree(dev);
		return -ENOMEM;
	}
	itf = (struct ippusb_intf*)dev->priv;
	
	/* init itf from static initializers */
	memcpy(&itf->interface, &printer_intf, sizeof(printer_intf));
	memcpy(&itf->fs_ep_in_desc, &fs_ep_in_desc, sizeof(fs_ep_in_desc));
	memcpy(&itf->fs_ep_out_desc, &fs_ep_out_desc, sizeof(fs_ep_out_desc));
	itf->fs_printer_function[0] =
		(struct usb_descriptor_header *)&itf->interface;
	itf->fs_printer_function[1] =
		(struct usb_descriptor_header *)&itf->fs_ep_out_desc;
	itf->fs_printer_function[2] =
		(struct usb_descriptor_header *)&itf->fs_ep_in_desc;
	itf->fs_printer_function[3] = NULL;
	memcpy(&itf->hs_ep_in_desc, &hs_ep_in_desc, sizeof(hs_ep_in_desc));
	memcpy(&itf->hs_ep_out_desc, &hs_ep_out_desc, sizeof(hs_ep_out_desc));
	itf->hs_printer_function[0] =
		(struct usb_descriptor_header *)&itf->interface;
	itf->hs_printer_function[1] =
		(struct usb_descriptor_header *)&itf->hs_ep_out_desc;
	itf->hs_printer_function[2] =
		(struct usb_descriptor_header *)&itf->hs_ep_in_desc;
	itf->hs_printer_function[3] = NULL;
	memcpy(&itf->ss_ep_in_desc, &ss_ep_in_desc, sizeof(ss_ep_in_desc));
	memcpy(&itf->ss_ep_in_comp_desc, &ss_ep_in_comp_desc,
		sizeof(ss_ep_in_comp_desc));
	memcpy(&itf->ss_ep_out_desc, &ss_ep_out_desc,
		sizeof(ss_ep_out_desc));
	memcpy(&itf->ss_ep_out_comp_desc, &ss_ep_out_comp_desc,
		sizeof(ss_ep_out_comp_desc));
	itf->ss_printer_function[0] =
		(struct usb_descriptor_header *)&itf->interface;
	itf->ss_printer_function[1] =
		(struct usb_descriptor_header *)&itf->ss_ep_out_desc;
	itf->ss_printer_function[2] =
		(struct usb_descriptor_header *)&itf->ss_ep_out_comp_desc;
	itf->ss_printer_function[3] =
		(struct usb_descriptor_header *)&itf->ss_ep_in_desc;
	itf->ss_printer_function[4] =
		(struct usb_descriptor_header *)&itf->ss_ep_in_comp_desc;
	itf->ss_printer_function[5] = NULL;
	itf->devminor = devminor;

	itf->interface.iInterface = id;
	
	dev->function.name = "IPPUSB";
	dev->function.strings = printer_strings;
	dev->function.fs_descriptors = itf->fs_printer_function;
	dev->function.hs_descriptors = itf->hs_printer_function;
	dev->function.ss_descriptors = itf->ss_printer_function;
	dev->function.bind    = printer_bind;
	dev->function.unbind  = printer_unbind;
	dev->function.setup   = printer_setup;
	dev->function.set_alt = printer_set_alt;
	dev->function.disable = printer_disable;
	dev->function.suspend = printer_suspend;
	dev->function.resume  = printer_resume;

	status = usb_add_function(c, &dev->function);
	if (status) {
		kfree(dev);
		kfree(itf);
	}
	return status;
}

