/*
 * Copyright (c) 2016-2019, QBit Semiconductor LTD.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/jiffies.h>
#include <linux/platform_device.h>
#include <linux/of_platform.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/pagemap.h>
#include <linux/io.h>
#include <linux/dma-mapping.h>
#include <linux/vmalloc.h>
#include <linux/mman.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/device.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/poll.h>
#include <linux/mutex.h>
#include <asm/io.h>
#include <asm/mman.h>
#include <asm/uaccess.h>
#include <asm/dma.h>
#include <asm/pgalloc.h>
#include <linux/watchdog.h>

/* minimum and maximum watchdog trigger timeout, in cycle unit */
#define MIN_WDT_TIMEOUT			0x0000FFFF
#define MAX_WDT_TIMEOUT			0xFFFFFFFF

/* WDT registers offset */
#define WDTTC_OFFSET            0x00
#define WDTCNT_OFFSET           0x04
#define WDTEN_OFFSET            0x08
#define WDTLD_OFFSET            0x0C
#define WDTBEND_OFFSET          0x10
#define WDT_ENABLE              0x01

#define WDT_INITIAL_TICK_VALUE  0xFFFFFFFF
#define WDT_RELOAD_COUNTER      0x7A1C

static int heartbeat = WDT_INITIAL_TICK_VALUE;
module_param(heartbeat, int, 0644);
MODULE_PARM_DESC(heartbeat, "Quasar watchdog timer heatbit count in cycle unit");

struct quasar_wdt {
	struct watchdog_device	wdd;
	void __iomem		*wdt_base;
};


static int quasar_wdt_start(struct watchdog_device *wdd)
{
	struct quasar_wdt *wdt = watchdog_get_drvdata(wdd);

    if (readl(wdt->wdt_base + WDTEN_OFFSET) & WDT_ENABLE)
    {
        return -EPERM;
    }
	//  Set watchdog count
    writel(wdd->timeout, wdt->wdt_base + WDTTC_OFFSET);
    // Load the watchdog timer first, then start it.
    writel(wdd->timeout, wdt->wdt_base + WDTBEND_OFFSET);
    writel(WDT_ENABLE, wdt->wdt_base + WDTEN_OFFSET);
    writel(WDT_RELOAD_COUNTER, wdt->wdt_base + WDTLD_OFFSET);

	return 0;
}

static int quasar_wdt_stop(struct watchdog_device *wdd)
{
	return 0;
}

static int quasar_wdt_ping(struct watchdog_device *wdd)
{
	struct quasar_wdt *wdt = watchdog_get_drvdata(wdd);

	writel(WDT_RELOAD_COUNTER, wdt->wdt_base + WDTLD_OFFSET);

	return 0;
}

static int quasar_wdt_set_timeout(struct watchdog_device *wdd,
				 unsigned int timeout)
{
    struct quasar_wdt *wdt = watchdog_get_drvdata(wdd);
	if (readl(wdt->wdt_base + WDTEN_OFFSET) & WDT_ENABLE)
    {
        return -EPERM;
    }
    
	wdd->timeout = timeout;
	return 0;
}

static unsigned int quasar_wdt_get_timeleft(struct watchdog_device *wdd)
{
	struct quasar_wdt *wdt = watchdog_get_drvdata(wdd);
    u32 val;
    
	val = readl(wdt->wdt_base + WDTCNT_OFFSET);
    
	return val;
}

static const struct watchdog_info quasar_wdt_info = {
	.options	= WDIOF_SETTIMEOUT |
			  WDIOF_KEEPALIVEPING,
	.firmware_version = 0,
	.identity	= "Quasar Watchdog",
};

static struct watchdog_ops quasar_wdt_ops = {
	.owner = THIS_MODULE,
	.start = quasar_wdt_start,
    .stop = quasar_wdt_stop,
	.ping = quasar_wdt_ping,
	.set_timeout = quasar_wdt_set_timeout,
	.get_timeleft = quasar_wdt_get_timeleft,
};

static int quasar_wdt_probe(struct platform_device *pdev)
{
	struct watchdog_device *wdd;
	struct quasar_wdt *wdt;
	struct resource *res;
	void __iomem *regs;
	int ret;

	/* This is the timer base. */
	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	regs = devm_ioremap_resource(&pdev->dev, res);
    
	if (regs == NULL) {
		dev_err(&pdev->dev, "Unable to remap IO\n");
		return -ENXIO;
	}

	/*
	 * Allocate our watchdog driver data, which has the
	 * struct watchdog_device nested within it.
	 */
	wdt = devm_kzalloc(&pdev->dev, sizeof(*wdt), GFP_KERNEL);
	if (!wdt)
		return -ENOMEM;

	/* Initialize struct quasar_wdt. */
	wdt->wdt_base = regs;

	/* Initialize struct watchdog_device. */
	wdd = &wdt->wdd;
	wdd->timeout = heartbeat;
	wdd->info = &quasar_wdt_info;
	wdd->ops = &quasar_wdt_ops;
	wdd->min_timeout = MIN_WDT_TIMEOUT;
	wdd->max_timeout = MAX_WDT_TIMEOUT;

	watchdog_set_drvdata(wdd, wdt);
	ret = watchdog_register_device(wdd);
	if (ret) {
		dev_err(&pdev->dev,
			"failed to register watchdog device\n");
		return ret;
	}

	platform_set_drvdata(pdev, wdt);

	dev_info(&pdev->dev,
		 "initialized (heartbeat = %u cycles)\n", heartbeat);

	return 0;
}

static int quasar_wdt_remove(struct platform_device *pdev)
{
	struct quasar_wdt *wdt = platform_get_drvdata(pdev);
	watchdog_unregister_device(&wdt->wdd);

	dev_info(&pdev->dev, "removed wdt\n");

	return 0;
}

#ifdef CONFIG_PM_SLEEP
static int quasar_wdt_runtime_suspend(struct device *dev)
{
	return 0;
}

static int quasar_wdt_runtime_resume(struct device *dev)
{
	return 0;
}
#endif

static const struct of_device_id quasar_wdt_of_match[] = {
	{ .compatible = "qbit,quasar-watchdog", },
	{ },
};
MODULE_DEVICE_TABLE(of, quasar_wdt_of_match);

static const struct dev_pm_ops quasar_wdt_pm_ops = {
	SET_SYSTEM_SLEEP_PM_OPS(quasar_wdt_runtime_suspend,
				quasar_wdt_runtime_resume)
};

static struct platform_driver quasar_wdt_driver = {
	.probe		= quasar_wdt_probe,
	.remove		= quasar_wdt_remove,
	.driver		= {
		.owner	= THIS_MODULE,
		.name	= "quasar-wdt",
		.pm	= &quasar_wdt_pm_ops,
		.of_match_table = quasar_wdt_of_match,
	},
};
module_platform_driver(quasar_wdt_driver);

MODULE_DESCRIPTION("Quasar Watchdog Driver");
MODULE_LICENSE("GPL v2");
