// SPDX-License-Identifier: GPL-2.0
/*
 * xhci-plat.c - xHCI host controller driver platform Bus Glue.
 *
 * Copyright (C) 2012 Texas Instruments Incorporated - http://www.ti.com
 * Author: Sebastian Andrzej Siewior <bigeasy@linutronix.de>
 *
 * A lot of code borrowed from the Linux xHCI driver.
 */

#include <linux/clk.h>
#include <linux/dma-mapping.h>
#include <linux/module.h>
#include <linux/pci.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/platform_device.h>
#include <linux/usb/phy.h>
#include <linux/slab.h>
#include <linux/acpi.h>

#include "xhci.h"
#include "xhci-plat.h"
#include "xhci-mvebu.h"
#include "xhci-rcar.h"

#ifdef CONFIG_SOC_QUASAR6300
#include "../../../../drivers/quasar/6300/qbsoc_sys_regs_B0.h"
#include "../../../../drivers/quasar/6300/qbsoc_ddr_regs_B0.h"
//#include "../../../../drivers/quasar/qbsoc_sys_regs_B0.h"
//#include "../../../../drivers/quasar/qbsoc_ddr_regs_B0.h"
#else
#include "../../../../drivers/quasar/6600/qbsoc_sys_regs.h"
#include "../../../../drivers/quasar/6600/qbsoc_ddr_regs.h"
#endif

#ifdef CONFIG_SOC_QUASAR6300
//#define ENABLE_HOST_USDR    /* USDR is not host controller */
#define ENABLE_HOST_UH2P      /* UH2P as host controller, QB63xx default */
#elif defined(BOARD_TYPE_RC4)
#define ENABLE_HOST_USDR      /* USDR as host controller */
#define ENABLE_HOST_UHDR0     /* UHDR0 as host controller */
//#define ENABLE_HOST_UHDR1   /* UHDR1 is not host controller */
#elif defined(BOARD_TYPE_RC3L)
#define ENABLE_HOST_USDR      /* USDR as host controller */
//#define ENABLE_HOST_UHDR0   /* UHDR0 is not host controller */
#define ENABLE_HOST_UHDR1     /* UHDR1 as host controller */
#elif defined(BOARD_TYPE_EP4)
//#define ENABLE_HOST_USDR    /* USDR is not host controller */
#define ENABLE_HOST_UHDR0     /* UHDR0 as host controller */
#define ENABLE_HOST_UHDR1     /* UHDR1 as host controller */
#elif defined(BOARD_TYPE_EP3L)
//#define ENABLE_HOST_USDR    /* USDR is not host controller */
#define ENABLE_HOST_UHDR0     /* UHDR0 as host controller */
#define ENABLE_HOST_UHDR1     /* UHDR1 as host controller */
#endif

#define USB_USDR    (0)
#define USB_UH2P    (1)
#define USB_UHDR0   (USB_UH2P)
#define USB_UHDR1   (2)

#ifdef CONFIG_SOC_QUASAR6300
#define uint_port u32
#else
#define uint_port u64
#endif

u32 usbh_ctrl = 0xff;   /* USB host controller per definitions above */

static int xhci_plat_setup(struct usb_hcd *hcd);
static int xhci_plat_start(struct usb_hcd *hcd);

static struct hc_driver __read_mostly xhci_plat_hc_driver0;
static const struct xhci_driver_overrides xhci_plat_overrides0 __initconst = {
	.extra_priv_size = sizeof(struct xhci_plat_priv),
	.reset = xhci_plat_setup,
	.start = xhci_plat_start,
};

static struct hc_driver __read_mostly xhci_plat_hc_driver1;
static const struct xhci_driver_overrides xhci_plat_overrides1 __initconst = {
	.extra_priv_size = sizeof(struct xhci_plat_priv),
	.reset = xhci_plat_setup,
	.start = xhci_plat_start,
};

static struct hc_driver __read_mostly xhci_plat_hc_driver2;
static const struct xhci_driver_overrides xhci_plat_overrides2 __initconst = {
	.extra_priv_size = sizeof(struct xhci_plat_priv),
	.reset = xhci_plat_setup,
	.start = xhci_plat_start,
};

/* SuperSpeed PHY parameters */
#define SS_PHY_PARAM00_ADDR     0x00
#define SS_PHY_PARAM01_ADDR     0x01
#define SS_PHY_PARAM02_ADDR     0x02
#define SS_PHY_PARAM03_ADDR     0x03
#define SS_PHY_PARAM04_ADDR     0x04
#define SS_PHY_PARAM05_ADDR     0x05
#define SS_PHY_PARAM06_ADDR     0x06
#define SS_PHY_PARAM07_ADDR     0x07
#define SS_PHY_PARAM08_ADDR     0x08
#define SS_PHY_PARAM09_ADDR     0x09
#define SS_PHY_PARAM10_ADDR     0x0A
#define SS_PHY_PARAM11_ADDR     0x0B
#define SS_PHY_PARAM12_ADDR     0x0C
#define SS_PHY_PARAM13_ADDR     0x0D
#define SS_PHY_PARAM14_ADDR     0x0E
#define SS_PHY_PARAM15_ADDR     0x0F
#define SS_PHY_PARAM16_ADDR     0x10
#define SS_PHY_PARAM17_ADDR     0x11
#define SS_PHY_PARAM18_ADDR     0x12
#define SS_PHY_PARAM19_ADDR     0x13
#define SS_PHY_PARAM20_ADDR     0x14
#define SS_PHY_PARAM21_ADDR     0x15
#define SS_PHY_PARAM22_ADDR     0x16
#define SS_PHY_PARAM23_ADDR     0x17
#define SS_PHY_PARAM24_ADDR     0x18
#define SS_PHY_PARAM25_ADDR     0x19
#define SS_PHY_PARAM26_ADDR     0x1A
#define SS_PHY_PARAM27_ADDR     0x1B
#define SS_PHY_PARAM28_ADDR     0x1C
#define SS_PHY_PARAM29_ADDR     0x1D
#define SS_PHY_PARAM30_ADDR     0x1E
#define SS_PHY_PARAM31_ADDR     0x1F

static u8 phy_ss_test_read(void __iomem *regs_gpf, u8 addr)
{
	u32 reg = 0;

	if (addr > 0x1f) {
		printk("wrong addr 0x%02X\n", addr);
	} else {
		reg = readl(regs_gpf + USDR_CSR2_OFF);
		reg &= USDR_CSR2__TEST_ADDR__INV_MASK;
		reg |= 0x01;   // set TEST_I[0] for read
		reg |= ((u32)addr << USDR_CSR2__TEST_ADDR__SHIFT);
		writel(reg, regs_gpf + USDR_CSR2_OFF);
		udelay(1);
		reg = (readl(regs_gpf + USDR_CSR2_OFF) >>
				USDR_CSR2__TEST_O__SHIFT);
	}
	return (u8)reg;
}

static void phy_ss_test_write(void __iomem *regs_gpf, u8 addr, u8 value)
{
	u32 reg = 0;

	if (addr > 0x1f) {
		printk("wrong addr 0x%02X\n", addr);
	} else {
		reg = readl(regs_gpf + USDR_CSR2_OFF);
		reg &= (USDR_CSR2__TEST_ADDR__INV_MASK &
				USDR_CSR2__TEST_I__INV_MASK &
				USDR_CSR2__TEST_WRITE__INV_MASK);
		reg |= ((u32)addr << USDR_CSR2__TEST_ADDR__SHIFT);
		reg |= ((u32)value << USDR_CSR2__TEST_I__SHIFT);
		writel(reg, regs_gpf + USDR_CSR2_OFF);
		udelay(1);
		reg |= USDR_CSR2__TEST_WRITE__MASK;     /* assert TEST_WRITE */
		writel(reg, regs_gpf + USDR_CSR2_OFF);
		udelay(1);
		reg &= USDR_CSR2__TEST_WRITE__INV_MASK; /* de-assert TEST_WRITE */
		writel(reg, regs_gpf + USDR_CSR2_OFF);
	}
}

static void phy_ss_cdr_cpd_trim(void __iomem *regs_gpf, u8 val)
{
	u32 reg = 0;

	if (val > 0x0f) {
		printk("wrong CDR_CPD_TRIM value %d\n", val);
		return;
	}
	reg = (u32)phy_ss_test_read(regs_gpf, SS_PHY_PARAM07_ADDR);
	reg &= ~0x0f;
	reg |= val;
	phy_ss_test_write(regs_gpf, SS_PHY_PARAM07_ADDR, (u8)reg);
}

static void phy_ss_cdr_trim(void __iomem *regs_gpf, u8 val)
{
	u32 reg = 0;

	if (val > 0x03) {
		printk("wrong CDR_TRIM value %d\n", val);
		return;
	}
	reg = (u32)phy_ss_test_read(regs_gpf, SS_PHY_PARAM13_ADDR);
	reg &= ~0x60;
	reg |= (val << 5);
	phy_ss_test_write(regs_gpf, SS_PHY_PARAM13_ADDR, (u8)reg);
}

static void phy_ss_cdr_cpf_trim(void __iomem *regs_gpf, u8 val)
{
	u32 reg = 0;

	if (val > 0x0f) {
		printk("wrong CDR_CPF_TRIM value %d\n", val);
		return;
	}
	reg = (u32)phy_ss_test_read(regs_gpf, SS_PHY_PARAM08_ADDR);
	reg &= ~0x0f;
	reg |= val;
	phy_ss_test_write(regs_gpf, SS_PHY_PARAM08_ADDR, (u8)reg);
}

static void phy_ss_bgr_trim(void __iomem *regs_gpf, u8 val)
{
	u32 reg = 0;

	if (val > 0x0f) {
		printk("wrong BGR_TRIM value %d\n", val);
		return;
	}
	reg = (u32)phy_ss_test_read(regs_gpf, SS_PHY_PARAM11_ADDR);
	reg &= ~0x0f;
	reg |= val;
	phy_ss_test_write(regs_gpf, SS_PHY_PARAM11_ADDR, (u8)reg);
}

static void phy_init_ss(void __iomem *regs_gpf)
{
	/* If this is not USDR, return */
	if (usbh_ctrl != USB_USDR)
		return;

	// addr 0x07[3:0]
	phy_ss_cdr_cpd_trim(regs_gpf, 0x6);   // 0x6 or 0xa

	// addr 0x0d[6:5]
	phy_ss_cdr_trim(regs_gpf, 0x02);   // ori 2

	// addr 0x08[3:0]
	phy_ss_cdr_cpf_trim(regs_gpf, 0x01);   // default 0x3

	phy_ss_bgr_trim(regs_gpf, 0x0f);
}

/*static*/
u8 phy_hs_test_read(void __iomem *regs_gpf, u8 port_no, u16 addr)
{
	u32 reg = 0, val = 0;

	if (addr > 0xfff) {
		printk("wrong addr 0x%x\n", addr);
	} else {
#ifdef CONFIG_SOC_QUASAR6300   /* QB63xx UH2P has two ports */
		switch (port_no) {
		case 1:
#endif
			reg = readl(regs_gpf + USDR_CSR4_OFF);
			reg &= 0x8000ffff;   // mask PHY_CONFIG_MONITOR[62:48]
			reg |= (addr << 16);   // prepare address
			writel(reg, regs_gpf + USDR_CSR4_OFF);
			udelay(1);
			reg |= 0x10000000;   // assert addr strobe
			writel(reg, regs_gpf + USDR_CSR4_OFF);
			udelay(1);
			reg &= ~0x10000000;   // de-assert addr strobe
			writel(reg, regs_gpf + USDR_CSR4_OFF);
			udelay(1);
			reg |= 0x40000000;   // assert read strobe
			writel(reg, regs_gpf + USDR_CSR4_OFF);
			udelay(1);
			val = readl(regs_gpf + USDR_CSR4_OFF);
			val &= 0xff;   // get data
			reg &= ~0x40000000;   // de-assert read strobe
			writel(reg, regs_gpf + USDR_CSR4_OFF);
#ifdef CONFIG_SOC_QUASAR6300   /* QB63xx UH2P has two ports */
			break;
		case 2:
			reg = readl(regs_gpf + UH2P_CSR104_OFF);
			reg &= 0x8000ffff;   // mask PHY_CONFIG_MONITOR[62:48]
			reg |= (addr << 16);   // prepare address
			writel(reg, regs_gpf + UH2P_CSR104_OFF);
			udelay(1);
			reg |= 0x10000000;   // assert addr strobe
			writel(reg, regs_gpf + UH2P_CSR104_OFF);
			udelay(1);
			reg &= ~0x10000000;   // de-assert addr strobe
			writel(reg, regs_gpf + UH2P_CSR104_OFF);
			udelay(1);
			reg |= 0x40000000;   // assert read strobe
			writel(reg, regs_gpf + UH2P_CSR104_OFF);
			udelay(1);
			val = readl(regs_gpf + UH2P_CSR105_OFF);
			val &= 0xff;   // get data
			reg &= ~0x40000000;   // de-assert read strobe
			writel(reg, regs_gpf + UH2P_CSR104_OFF);
			break;
		default:
			printk("Wrong port number %d\n", port_no);
			break;
		}
#endif
	}
	return (u8)val;
}

/*static*/
void phy_hs_test_write(void __iomem *regs_gpf, u8 port_no, u16 addr, u8 value)
{
	u32 reg = 0;

	if (addr > 0xfff) {
		printk("wrong addr 0x%x\n", addr);
	} else {
#ifdef CONFIG_SOC_QUASAR6300   /* QB63xx UH2P has two ports */
		switch (port_no) {
		case 1:
#endif
			reg = readl(regs_gpf + USDR_CSR4_OFF);
			reg &= 0x8000ffff;   // mask PHY_CONFIG_MONITOR[62:48]
			reg |= (addr << 16);   // prepare address
			writel(reg, regs_gpf + USDR_CSR4_OFF);
			udelay(1);
			reg |= 0x10000000;   // assert addr strobe
			writel(reg, regs_gpf + USDR_CSR4_OFF);
			udelay(1);
			reg &= ~0x10000000;   // de-assert addr strobe
			writel(reg, regs_gpf + USDR_CSR4_OFF);
			udelay(1);
			reg &= 0xff00ffff;
			reg |= (value << 16);   // prepare write data
			writel(reg, regs_gpf + USDR_CSR4_OFF);
			udelay(1);
			reg |= 0x20000000;   // assert write strobe
			writel(reg, regs_gpf + USDR_CSR4_OFF);
			udelay(1);
			reg &= ~0x20000000;   // de-assert write strobe
			writel(reg, regs_gpf + USDR_CSR4_OFF);
#ifdef CONFIG_SOC_QUASAR6300   /* QB63xx UH2P has two ports */
			break;
		case 2:
			reg = readl(regs_gpf + UH2P_CSR104_OFF);
			reg &= 0x8000ffff;   // mask PHY_CONFIG_MONITOR[62:48]
			reg |= (addr << 16);   // prepare address
			writel(reg, regs_gpf + UH2P_CSR104_OFF);
			udelay(1);
			reg |= 0x10000000;   // assert addr strobe
			writel(reg, regs_gpf + UH2P_CSR104_OFF);
			udelay(1);
			reg &= ~0x10000000;   // de-assert addr strobe
			writel(reg, regs_gpf + UH2P_CSR104_OFF);
			udelay(1);
			reg &= 0xff00ffff;
			reg |= (value << 16);   // prepare write data
			writel(reg, regs_gpf + UH2P_CSR104_OFF);
			udelay(1);
			reg |= 0x20000000;   // assert write strobe
			writel(reg, regs_gpf + UH2P_CSR104_OFF);
			udelay(1);
			reg &= ~0x20000000;   // de-assert write strobe
			writel(reg, regs_gpf + UH2P_CSR104_OFF);
			break;
		default:
			printk("Wrong port number %d\n", port_no);
			break;
		}
#endif
	}
}

/*static*/
void phy_hs_default(void __iomem *regs_gpf, u8 use_efuse,
			u8 port_no, u32 hi, u32 lo)
{
	u32 reg = 0;
#ifndef CONFIG_SOC_QUASAR6300   /* QB66xx */
	u32 trimr, trimi;
#endif

#ifdef CONFIG_SOC_QUASAR6300   /* QB63xx UH2P has two ports */
	switch (port_no) {
	case 1:
#endif
		if (use_efuse) {
			reg = readl(regs_gpf + USDR_CSR5_OFF);
#ifdef CONFIG_SOC_QUASAR6300
			reg |= USDR_CSR5__HS_PHY_P0_EFUSE_SEL__MASK;
#else
			reg &= (USDR_CSR5__HS_PHY_EFUSE_TRIMR_SEL__INV_MASK &
					USDR_CSR5__HS_PHY_EFUSE_TRIMI_SEL__INV_MASK);
#endif
			writel(reg, regs_gpf + USDR_CSR5_OFF);
		} else	{
			reg = readl(regs_gpf + USDR_CSR5_OFF);
#ifdef CONFIG_SOC_QUASAR6300
			reg &= USDR_CSR5__HS_PHY_P0_EFUSE_SEL__INV_MASK;
#else
			reg |= (USDR_CSR5__HS_PHY_EFUSE_TRIMR_SEL__MASK |
					USDR_CSR5__HS_PHY_EFUSE_TRIMI_SEL__MASK);
			reg &= (USDR_CSR5__HS_PHY_TRIMR__INV_MASK &
					USDR_CSR5__HS_PHY_TRIMI__INV_MASK);
			/* TRIMR & TRIMI */
			trimr = ((lo & 0x000000C0) >> 2) + ((lo & 0x0000F000) >> 12);
			trimi = ((lo & 0xF0000000) >> 28);
			reg |= (trimr << USDR_CSR5__HS_PHY_TRIMR__SHIFT) +
				(trimi << USDR_CSR5__HS_PHY_TRIMI__SHIFT);
#endif
			writel(reg, regs_gpf + USDR_CSR5_OFF);

			writel(lo, regs_gpf + USDR_CSR3_OFF);
			writel(hi, regs_gpf + USDR_CSR4_OFF);
		}
#ifdef CONFIG_SOC_QUASAR6300   /* QB63xx UH2P has two ports */
		break;
	case 2:
		if (use_efuse) {
			reg = readl(regs_gpf + UH2P_CSR105_OFF);
			reg |= UH2P_CSR105__HS_PHY1_P0_EFUSE_SEL__MASK;
			writel(reg, regs_gpf + UH2P_CSR105_OFF);
		} else	{
			reg = readl(regs_gpf + UH2P_CSR105_OFF);
			reg &= UH2P_CSR105__HS_PHY1_P0_EFUSE_SEL__INV_MASK;
			writel(reg, regs_gpf + UH2P_CSR105_OFF);

			writel(lo, regs_gpf + UH2P_CSR103_OFF);
			writel(hi, regs_gpf + UH2P_CSR104_OFF);
		}
		break;
	default:
		printk("Wrong port number %d\n", port_no);
		break;
	}
#endif
}

// [7:6]
/*static*/
void phy_hs_rterm_trim(void __iomem *regs_gpf, u8 use_efuse,
			u8 port_no, u8 rterm)
{
	u32 reg = 0;
#ifndef CONFIG_SOC_QUASAR6300   /* QB66xx */
	u32 trimr;
#endif

	if (0x03 < rterm) {
		printk("wrong rterm 0x%02X\n", rterm);
		return;
	}

#ifdef CONFIG_SOC_QUASAR6300   /* QB63xx UH2P has two ports */
	switch (port_no) {
	case 1:
#endif
		if (use_efuse) {
			reg = readl(regs_gpf + USDR_CSR5_OFF);
#ifdef CONFIG_SOC_QUASAR6300
			reg |= USDR_CSR5__HS_PHY_P0_EFUSE_SEL__MASK;
#else
			reg &= USDR_CSR5__HS_PHY_EFUSE_TRIMR_SEL__INV_MASK;
#endif
			writel(reg, regs_gpf + USDR_CSR5_OFF);
		} else	{
			reg = readl(regs_gpf + USDR_CSR5_OFF);
#ifdef CONFIG_SOC_QUASAR6300
			reg &= USDR_CSR5__HS_PHY_P0_EFUSE_SEL__INV_MASK;
#else
			reg |= USDR_CSR5__HS_PHY_EFUSE_TRIMR_SEL__MASK;
			reg &= USDR_CSR5__HS_PHY_TRIMR__INV_MASK;
			/* TRIMR & TRIMI */
			trimr = (rterm << 4) + ((readl(regs_gpf + USDR_CSR3_OFF) & 0x0000F000) >> 12);
			reg |= (trimr << USDR_CSR5__HS_PHY_TRIMR__SHIFT);
#endif
			writel(reg, regs_gpf + USDR_CSR5_OFF);

			reg = readl(regs_gpf + USDR_CSR3_OFF);
			reg &= ~0x000000C0;
            reg |= (rterm << 6);
			writel(reg, regs_gpf + USDR_CSR3_OFF);
		}
#ifdef CONFIG_SOC_QUASAR6300   /* QB63xx UH2P has two ports */
		break;
	case 2:
		if (use_efuse) {
			reg = readl(regs_gpf + UH2P_CSR105_OFF);
			reg |= UH2P_CSR105__HS_PHY1_P0_EFUSE_SEL__MASK;
			writel(reg, regs_gpf + UH2P_CSR105_OFF);
		} else	{
			reg = readl(regs_gpf + UH2P_CSR105_OFF);
			reg &= UH2P_CSR105__HS_PHY1_P0_EFUSE_SEL__INV_MASK;
			writel(reg, regs_gpf + UH2P_CSR105_OFF);

			reg = readl(regs_gpf + UH2P_CSR103_OFF);
			reg &= ~0x000000C0;
			reg |= (rterm << 6);
			writel(reg, regs_gpf + UH2P_CSR103_OFF);
		}
		break;
	default:
		printk("Wrong port number %d\n", port_no);
		break;
	}
#endif
}

// [15:12]
/*static*/
void phy_hs_term_resistor_trim(void __iomem *regs_gpf, u8 use_efuse,
			u8 port_no, u8 term_resistor)
{
	u32 reg = 0;
#ifndef CONFIG_SOC_QUASAR6300   /* QB66xx */
	u32 trimr;
#endif

	if (0x0F < term_resistor) {
		printk("wrong term_resistor 0x%02X\n", term_resistor);
		return;
	}

#ifdef CONFIG_SOC_QUASAR6300   /* QB63xx UH2P has two ports */
	switch (port_no) {
	case 1:
#endif
		if (use_efuse) {
			reg = readl(regs_gpf + USDR_CSR5_OFF);
#ifdef CONFIG_SOC_QUASAR6300
			reg |= USDR_CSR5__HS_PHY_P0_EFUSE_SEL__MASK;
#else
			reg &= USDR_CSR5__HS_PHY_EFUSE_TRIMR_SEL__INV_MASK;
#endif
			writel(reg, regs_gpf + USDR_CSR5_OFF);
		} else	{
			reg = readl(regs_gpf + USDR_CSR5_OFF);
#ifdef CONFIG_SOC_QUASAR6300
			reg &= USDR_CSR5__HS_PHY_P0_EFUSE_SEL__INV_MASK;
#else
			reg |= USDR_CSR5__HS_PHY_EFUSE_TRIMR_SEL__MASK;
			reg &= USDR_CSR5__HS_PHY_TRIMR__INV_MASK;
			/* TRIMR & TRIMI */
			trimr = ((readl(regs_gpf + USDR_CSR3_OFF) & 0x000000C0) >> 2) + term_resistor;
			reg |= (trimr << USDR_CSR5__HS_PHY_TRIMR__SHIFT);
#endif
			writel(reg, regs_gpf + USDR_CSR5_OFF);

			reg = readl(regs_gpf + USDR_CSR3_OFF);
			reg &= ~0x0000F000;
			reg |= (term_resistor << 12);
			writel(reg, regs_gpf + USDR_CSR3_OFF);
		}
#ifdef CONFIG_SOC_QUASAR6300   /* QB63xx UH2P has two ports */
		break;
	case 2:
		if (use_efuse) {
			reg = readl(regs_gpf + UH2P_CSR105_OFF);
			reg |= UH2P_CSR105__HS_PHY1_P0_EFUSE_SEL__MASK;
			writel(reg, regs_gpf + UH2P_CSR105_OFF);
		} else	{
			reg = readl(regs_gpf + UH2P_CSR105_OFF);
			reg &= UH2P_CSR105__HS_PHY1_P0_EFUSE_SEL__INV_MASK;
			writel(reg, regs_gpf + UH2P_CSR105_OFF);

			reg = readl(regs_gpf + UH2P_CSR103_OFF);
			reg &= ~0x0000F000;
			reg |= (term_resistor << 12);
			writel(reg, regs_gpf + UH2P_CSR103_OFF);
		}
		break;
	default:
		printk("Wrong port number %d\n", port_no);
		break;
	}
#endif
}

// [17:16]
/*static*/
void phy_hs_swing(void __iomem *regs_gpf, u8 port_no, u8 swing)
{
	u32 reg = 0;

	if (0x03 < swing) {
		printk("wrong swing 0x%02X\n", swing);
		return;
	}

#ifdef CONFIG_SOC_QUASAR6300   /* QB63xx UH2P has two ports */
	switch (port_no) {
	case 1:
#endif
		reg = readl(regs_gpf + USDR_CSR3_OFF);
		reg &= ~0x00030000;
		reg |= (swing << 16);
		writel(reg, regs_gpf + USDR_CSR3_OFF);
#ifdef CONFIG_SOC_QUASAR6300   /* QB63xx UH2P has two ports */
		break;
	case 2:
		reg = readl(regs_gpf + UH2P_CSR103_OFF);
		reg &= ~0x00030000;
		reg |= (swing << 16);
		writel(reg, regs_gpf + UH2P_CSR103_OFF);
		break;
	default:
		printk("Wrong port number %d\n", port_no);
		break;
	}
#endif
}

// [25:23]
/*static*/
void phy_hs_band_gap(void __iomem *regs_gpf, u8 port_no, u8 bandgap)
{
	u32 reg = 0;

	if (0x07 < bandgap) {
		printk("wrong bandgap 0x%02X\n", bandgap);
		return;
	}

#ifdef CONFIG_SOC_QUASAR6300   /* QB63xx UH2P has two ports */
	switch (port_no) {
	case 1:
#endif
		reg = readl(regs_gpf + USDR_CSR3_OFF);
		reg &= ~0x03800000;
		reg |= (bandgap << 23);
		writel(reg, regs_gpf + USDR_CSR3_OFF);
#ifdef CONFIG_SOC_QUASAR6300   /* QB63xx UH2P has two ports */
		break;
	case 2:
		reg = readl(regs_gpf + UH2P_CSR103_OFF);
		reg &= ~0x03800000;
		reg |= (bandgap << 23);
		writel(reg, regs_gpf + UH2P_CSR103_OFF);
		break;
	default:
		printk("Wrong port number %d\n", port_no);
		break;
	}
#endif
}

// [31:28]
/*static*/
void phy_hs_current_trim(void __iomem *regs_gpf, u8 use_efuse, u8 port_no, u8 curr)
{
	u32 reg = 0;

	if (0x0F < curr) {
		printk("wrong current 0x%02X\n", curr);
		return;
	}

#ifdef CONFIG_SOC_QUASAR6300   /* QB63xx UH2P has two ports */
	switch (port_no) {
	case 1:
#endif
		if (use_efuse) {
			reg = readl(regs_gpf + USDR_CSR5_OFF);
#ifdef CONFIG_SOC_QUASAR6300
			reg |= USDR_CSR5__HS_PHY_P0_EFUSE_SEL__MASK;
#else
			reg &= USDR_CSR5__HS_PHY_EFUSE_TRIMI_SEL__INV_MASK;
#endif
			writel(reg, regs_gpf + USDR_CSR5_OFF);
		} else	{
			reg = readl(regs_gpf + USDR_CSR5_OFF);
#ifdef CONFIG_SOC_QUASAR6300
			reg &= USDR_CSR5__HS_PHY_P0_EFUSE_SEL__INV_MASK;
#else
			reg |= USDR_CSR5__HS_PHY_EFUSE_TRIMI_SEL__MASK;
			reg &= USDR_CSR5__HS_PHY_TRIMI__INV_MASK;
			/* TRIMI */
			reg |= (curr << USDR_CSR5__HS_PHY_TRIMI__SHIFT);
#endif
			writel(reg, regs_gpf + USDR_CSR5_OFF);

			reg = readl(regs_gpf + USDR_CSR3_OFF);
			reg &= ~0xF0000000;
			reg |= (curr << 28);
			writel(reg, regs_gpf + USDR_CSR3_OFF);
		}
#ifdef CONFIG_SOC_QUASAR6300   /* QB63xx UH2P has two ports */
		break;
	case 2:
		if (use_efuse) {
			reg = readl(regs_gpf + UH2P_CSR105_OFF);
			reg |= UH2P_CSR105__HS_PHY1_P0_EFUSE_SEL__MASK;
			writel(reg, regs_gpf + UH2P_CSR105_OFF);
		} else	{
			reg = readl(regs_gpf + UH2P_CSR105_OFF);
			reg &= UH2P_CSR105__HS_PHY1_P0_EFUSE_SEL__INV_MASK;
			writel(reg, regs_gpf + UH2P_CSR105_OFF);

			reg = readl(regs_gpf + UH2P_CSR103_OFF);
			reg &= ~0xF0000000;
			reg |= (curr << 28);
			writel(reg, regs_gpf + UH2P_CSR103_OFF);
		}
		break;
	default:
		printk("Wrong port number %d\n", port_no);
		break;
	}
#endif
}

// [36]
/*static*/
void phy_hs_driver_slew(void __iomem *regs_gpf, u8 port_no, u8 high_slew)
{
	u32 reg = 0;

	if (1 < high_slew) {
		printk("wrong current 0x%02X\n", high_slew);
		return;
	}

#ifdef CONFIG_SOC_QUASAR6300   /* QB63xx UH2P has two ports */
	switch (port_no) {
	case 1:
#endif
		reg = readl(regs_gpf + USDR_CSR4_OFF);
		reg &= ~0x00000010;
		reg |= (high_slew << 4);
		writel(reg, regs_gpf + USDR_CSR4_OFF);
#ifdef CONFIG_SOC_QUASAR6300   /* QB63xx UH2P has two ports */
		break;
	case 2:
		reg = readl(regs_gpf + UH2P_CSR104_OFF);
		reg &= ~0x00000010;
		reg |= (high_slew << 4);
		writel(reg, regs_gpf + UH2P_CSR104_OFF);
		break;
	default:
		printk("Wrong port number %d\n", port_no);
		break;
	}
#endif
}

// [40:39]
/*static*/
void phy_hs_sq_trim(void __iomem *regs_gpf, u8 port_no, u8 squelch)
{
	u32 reg = 0;

	if (3 < squelch) {
		printk("wrong current 0x%02X\n", squelch);
		return;
	}

#ifdef CONFIG_SOC_QUASAR6300   /* QB63xx UH2P has two ports */
	switch (port_no) {
	case 1:
#endif
		reg = readl(regs_gpf + USDR_CSR4_OFF);
		reg &= ~0x00000180;
		reg |= (squelch << 7);
		writel(reg, regs_gpf + USDR_CSR4_OFF);
#ifdef CONFIG_SOC_QUASAR6300   /* QB63xx UH2P has two ports */
		break;
	case 2:
		reg = readl(regs_gpf + UH2P_CSR104_OFF);
		reg &= ~0x00000180;
		reg |= (squelch << 7);
		writel(reg, regs_gpf + UH2P_CSR104_OFF);
		break;
	default:
		printk("Wrong port number %d\n", port_no);
		break;
	}
#endif
}

static void phy_init_hs(void __iomem *regs_gpf)
{
	phy_hs_default(regs_gpf, 0, 1, 0x00000106, 0x92316680);   // 0000_0106_9231_6680
	if (usbh_ctrl == USB_UH2P)   /* QB63xx UH2P has two ports */
		phy_hs_default(regs_gpf, 0, 2, 0x00000106, 0x92316680);   // 0000_0106_9231_6680

    // [7:6]
	//phy_hs_rterm_trim(regs_gpf, 0, 1, 0);
	//if (usbh_ctrl == USB_UH2P)   /* QB63xx UH2P has two ports */
		//phy_hs_rterm_trim(regs_gpf, 0, 2, 0);

    // [15:12]
    // 0001 Resistance large
    // 0110 Resistance (default)
    // 1111 Resistance small
	//phy_hs_term_resistor_trim(regs_gpf, 0, 1, 6);   // for FS
	//if (usbh_ctrl == USB_UH2P)   /* QB63xx UH2P has two ports */
		//phy_hs_term_resistor_trim(regs_gpf, 0, 2, 6);   // for FS

    // [17:16]
    // 00 = 125%
    // 01 = 100% (recommended)
    // 10 = 110%
    // 11 =  90%
	//phy_hs_swing(regs_gpf, 0, 1, 1);
	//if (usbh_ctrl == USB_UH2P)   /* QB63xx UH2P has two ports */
		//phy_hs_swing(regs_gpf, 0, 2, 1);

    // [25:23]
    // 000 =  80%
    // 001 =  85%
    // 010 =  90%
    // 011 =  95%
    // 100 = 100% (recommended)
    // 101 = 105%
    // 110 = 110%
    // 111 = 115%
	//phy_hs_band_gap(regs_gpf, 0, 1, 4);
	//if (usbh_ctrl == USB_UH2P)   /* QB63xx UH2P has two ports */
		//phy_hs_band_gap(regs_gpf, 0, 2, 4);

    // [31:28]
	//phy_hs_current_trim(regs_gpf, 0, 1, 0x9);   // if device HS eye too small or too large
	//if (usbh_ctrl == USB_UH2P)   /* QB63xx UH2P has two ports */
		//phy_hs_current_trim(regs_gpf, 0, 2, 0x9);   // if device HS eye too small or too large

    // [36]
    // 0 = high slew rate
    // 1 = low slew rate
	//phy_hs_driver_slew(regs_gpf, 0, 1, 0);
	//if (usbh_ctrl == USB_UH2P)   /* QB63xx UH2P has two ports */
		//phy_hs_driver_slew(regs_gpf, 0, 2, 0);

    // [40:39]
    // 00 = 110mV
    // 01 = 122mV
    // 10 = 134mV (recommended)
    // 11 = 146mV
	//phy_hs_sq_trim(regs_gpf, 1, 2);
	//if (usbh_ctrl == USB_UH2P)   /* QB63xx UH2P has two ports */
		//phy_hs_sq_trim(regs_gpf, 2, 2);

#ifdef CONFIG_SOC_QUASAR6300   /* These are default settings for QB66xx HS PHY */
	phy_hs_test_write(regs_gpf, 1, 0, 0x20);   // RX_CHK_SYNC=1
	phy_hs_test_write(regs_gpf, 1, 1, 0x01);   // RX_SYNC_SEL=00b=6 bits; 01b=5 bits; 10b=4 bits
	if (usbh_ctrl == USB_UH2P) {   /* QB63xx UH2P has two ports */
		phy_hs_test_write(regs_gpf, 2, 0, 0x20);   // RX_CHK_SYNC=1
		phy_hs_test_write(regs_gpf, 2, 1, 0x01);   // RX_SYNC_SEL=00b=6 bits; 01b=5 bits; 10b=4 bits
	}
#endif
}

static void phy_reset_ss(void __iomem *regs_gpf, u8 reset)
{
	u32 reg;

	/* If this is not USDR, return */
	if (usbh_ctrl != USB_USDR)
		return;

	udelay(5);
	if (reset) {
		reg = readl(regs_gpf + USDR_RST_OFF);
		reg &= (USDR_RST__SS_PHY_RESET_N_SEL__INV_MASK &
			USDR_RST__SS_PHY_RESET_N__INV_MASK);
		writel(reg, regs_gpf + USDR_RST_OFF);

		reg = readl(regs_gpf + USDR_CSR2_OFF);
		reg &= USDR_CSR2__TEST_RESET_N__INV_MASK;
		writel(reg, regs_gpf + USDR_CSR2_OFF);
	} else {
		reg = readl(regs_gpf + USDR_CSR2_OFF);
		reg |= USDR_CSR2__TEST_RESET_N__MASK;
		writel(reg, regs_gpf + USDR_CSR2_OFF);
		udelay(5);

		/* Init PHY here if needed */
		phy_init_ss(regs_gpf);   /* must set after TEST_RESET_N is high */
		udelay(5);

		reg = readl(regs_gpf + USDR_RST_OFF);
		reg |= USDR_RST__SS_PHY_RESET_N__MASK;
		writel(reg, regs_gpf + USDR_RST_OFF);
	}
	udelay(5);
}

static void phy_reset_hs(void __iomem *regs_gpf, u8 reset)
{
	u32 reg;

	if (reset) {
		/* Put PHY in suspend */
		reg = readl(regs_gpf + USDR_CSR6_OFF);
		reg &= (USDR_CSR6__HS_PHY_P0_SUSPENDM_SEL__INV_MASK &
				USDR_CSR6__HS_PHY_P0_SUSPENDM__INV_MASK);
		reg |= (2 << USDR_CSR6__HS_PHY_P0_SUSPENDM_SEL__SHIFT);
		writel(reg, regs_gpf + USDR_CSR6_OFF);
#ifdef CONFIG_SOC_QUASAR6300   /* QB63xx the second UH2P port */
		if (usbh_ctrl == USB_UH2P) {
			reg = readl(regs_gpf + UH2P_CSR106_OFF);
			reg &= (UH2P_CSR106__HS_PHY1_P0_SUSPENDM_SEL__INV_MASK &
				UH2P_CSR106__HS_PHY1_P0_SUSPENDM__INV_MASK);
			reg |= (2 << UH2P_CSR106__HS_PHY1_P0_SUSPENDM_SEL__SHIFT);
			writel(reg, regs_gpf + UH2P_CSR106_OFF);
		}
#endif

		/* Reset PHY */
#ifdef CONFIG_SOC_QUASAR6300
		if (usbh_ctrl == USB_USDR) {
			reg = readl(regs_gpf + USDR_RST_OFF);
			reg |=  USDR_RST__HS_PHY_P0_RESET__MASK;   /* active high */
			reg &= (USDR_RST__HS_PHY_P0_RESET_SEL__INV_MASK &
				USDR_RST__HS_PHY_PWON_RESETN__INV_MASK);   /* active low */
			writel(reg, regs_gpf + USDR_RST_OFF);
		} else {   /* QB63xx UH2P has two ports */
			reg = readl(regs_gpf + UH2P_RST_OFF);
			reg |= (UH2P_RST__HS_PHY0_P0_RESET__MASK |
				UH2P_RST__HS_PHY1_P0_RESET__MASK);   /* active high */
			reg &= (UH2P_RST__HS_PHY0_P0_RESET_SEL__INV_MASK &
				UH2P_RST__HS_PHY1_P0_RESET_SEL__INV_MASK &
				UH2P_RST__HS_PHY0_PWON_RESETN__INV_MASK &
				UH2P_RST__HS_PHY1_PWON_RESETN__INV_MASK);   /* active low */
			writel(reg, regs_gpf + UH2P_RST_OFF);
		}
#else   /* QB66xx: Each dual-role controller has only one port */
		reg = readl(regs_gpf + USDR_RST_OFF);
		reg |=  USDR_RST__HS_PHY_P0_RESET__MASK;   /* active high */
		reg &= (USDR_RST__HS_PHY_P0_RESET_SEL__INV_MASK &
				USDR_RST__HS_PHY_PWON_RESETN__INV_MASK);   /* active low */
		writel(reg, regs_gpf + USDR_RST_OFF);
#endif
	} else {
        /* Release reset signals */
#ifdef CONFIG_SOC_QUASAR6300
		if (usbh_ctrl == USB_USDR) {
			reg = readl(regs_gpf + USDR_RST_OFF);
			reg |= USDR_RST__HS_PHY_PWON_RESETN__MASK;   /* active low */
			writel(reg, regs_gpf + USDR_RST_OFF);
		} else {   /* QB63xx UH2P has two ports */
			reg = readl(regs_gpf + UH2P_RST_OFF);
			reg |= (UH2P_RST__HS_PHY0_PWON_RESETN__MASK |
				UH2P_RST__HS_PHY1_PWON_RESETN__MASK);   /* active low */
			writel(reg, regs_gpf + UH2P_RST_OFF);
		}
#else   /* QB66xx: Each dual-role controller has only one port */
		reg = readl(regs_gpf + USDR_RST_OFF);
		reg |= USDR_RST__HS_PHY_PWON_RESETN__MASK;   /* active low */
		writel(reg, regs_gpf + USDR_RST_OFF);
#endif
		udelay(5);

		/* Init PHY here if needed */
		phy_init_hs(regs_gpf);   /* must be after PWON_RESETN is high */
		udelay(5);
#ifdef CONFIG_SOC_QUASAR6300
		if (usbh_ctrl == USB_USDR) {
			reg = readl(regs_gpf + USDR_RST_OFF);
			reg &= USDR_RST__HS_PHY_P0_RESET__INV_MASK;   /* active high */
			writel(reg, regs_gpf + USDR_RST_OFF);
		} else {   /* QB63xx UH2P has two ports */
			reg = readl(regs_gpf + UH2P_RST_OFF);
			reg &= (UH2P_RST__HS_PHY0_P0_RESET__INV_MASK &
				UH2P_RST__HS_PHY1_P0_RESET__INV_MASK);   /* active high */
			writel(reg, regs_gpf + UH2P_RST_OFF);
		}
#else   /* QB66xx: Each dual-role controller has only one port */
		reg = readl(regs_gpf + USDR_RST_OFF);
		reg &= USDR_RST__HS_PHY_P0_RESET__INV_MASK;   /* active high */
		writel(reg, regs_gpf + USDR_RST_OFF);
#endif
		udelay(5);

		/* Release Pn_SUSPENDM */
		reg = readl(regs_gpf + USDR_CSR6_OFF);
		reg |= USDR_CSR6__HS_PHY_P0_SUSPENDM__MASK;
		writel(reg, regs_gpf + USDR_CSR6_OFF); 
#ifdef CONFIG_SOC_QUASAR6300   /* QB63xx: the second UH2P ports */
		if (usbh_ctrl == USB_UH2P) {
			reg = readl(regs_gpf + UH2P_CSR106_OFF);
			reg |= UH2P_CSR106__HS_PHY1_P0_SUSPENDM__MASK;
			writel(reg, regs_gpf + UH2P_CSR106_OFF);
		}
#endif
		/* QB66xx: Each dual-role controller has only one port
		   Set dppulldown and dmpulldown as 1 for host port
		   QB63xx UH2P is fixed as host, no need to do it */
		reg = readl(regs_gpf + USDR_CSR5_OFF);
		reg |= (USDR_CSR5__HS_PHY_P0_DMPULLDOWN__MASK |
			USDR_CSR5__HS_PHY_P0_DPPULLDOWN__MASK);
		writel(reg, regs_gpf + USDR_CSR5_OFF);
	}
}

static void usbh_hw_reset(void __iomem *regs_gpf)
{
	volatile u8	__iomem	*rstgenswrststatic2;
	u32 reg;

	rstgenswrststatic2 = ioremap(RSTGEN_SWRSTSTATIC2, 4);
	reg = readl(rstgenswrststatic2);
	switch (usbh_ctrl) {
	case USB_USDR:
		reg |= RSTGEN_SWRSTSTATIC2__SYS_USDR__MASK;
		break;
#ifdef CONFIG_SOC_QUASAR6300
	case USB_UH2P:
		reg |= RSTGEN_SWRSTSTATIC2__SYS_UH2P__MASK;
		break;
#else
	case USB_UHDR0:
		reg |= RSTGEN_SWRSTSTATIC2__SYS_UHDR0__MASK;
		break;
	case USB_UHDR1:
		reg |= RSTGEN_SWRSTSTATIC2__SYS_UHDR1__MASK;
		break;
#endif
	default:
		printk("unknown USB host controller 0x%d\n", (u32)usbh_ctrl);
		break;
	}
	writel(reg, rstgenswrststatic2);
	udelay(5);

	reg = readl(rstgenswrststatic2);
	switch (usbh_ctrl) {
	case USB_USDR:
		reg &= RSTGEN_SWRSTSTATIC2__SYS_USDR__INV_MASK;
		break;
#ifdef CONFIG_SOC_QUASAR6300
	case USB_UH2P:
		reg &= RSTGEN_SWRSTSTATIC2__SYS_UH2P__INV_MASK;
		break;
#else
	case USB_UHDR0:
		reg &= RSTGEN_SWRSTSTATIC2__SYS_UHDR0__INV_MASK;
		break;
	case USB_UHDR1:
		reg &= RSTGEN_SWRSTSTATIC2__SYS_UHDR1__INV_MASK;
		break;
#endif
	default:
		printk("unknown USB host controller 0x%d\n", (u32)usbh_ctrl);
		break;
	}
	writel(reg, rstgenswrststatic2);
	udelay(5);

	switch (usbh_ctrl) {
	case USB_USDR:
		reg = readl(regs_gpf + USDR_RST_OFF);
		reg &= USDR_RST__CNTRL_VCC_RESET_N__INV_MASK;
		writel(reg, regs_gpf + USDR_RST_OFF);
		udelay(5);
		reg = readl(regs_gpf + USDR_RST_OFF);
		reg |= USDR_RST__CNTRL_VCC_RESET_N__MASK;
		writel(reg, regs_gpf + USDR_RST_OFF);
		break;
#ifdef CONFIG_SOC_QUASAR6300
	case USB_UH2P:
		reg = readl(regs_gpf + UH2P_RST_OFF);
		reg &= UH2P_RST__CNTRL_VCC_RESET_N__INV_MASK;
		writel(reg, regs_gpf + UH2P_RST_OFF);
		udelay(5);
		reg = readl(regs_gpf + UH2P_RST_OFF);
		reg |= UH2P_RST__CNTRL_VCC_RESET_N__MASK;
		writel(reg, regs_gpf + UH2P_RST_OFF);
		break;
#else
	case USB_UHDR0:
		reg = readl(regs_gpf + UHDR0_UHDR_RST_OFF);
		reg &= UHDR0_UHDR_RST__CNTRL_VCC_RESET_N__INV_MASK;
		writel(reg, regs_gpf + UHDR0_UHDR_RST_OFF);
		udelay(5);
		reg = readl(regs_gpf + UHDR0_UHDR_RST_OFF);
		reg |= UHDR0_UHDR_RST__CNTRL_VCC_RESET_N__MASK;
		writel(reg, regs_gpf + UHDR0_UHDR_RST_OFF);
		break;
	case USB_UHDR1:
		reg = readl(regs_gpf + UHDR1_UHDR_RST_OFF);
		reg &= UHDR1_UHDR_RST__CNTRL_VCC_RESET_N__INV_MASK;
		writel(reg, regs_gpf + UHDR1_UHDR_RST_OFF);
		udelay(5);
		reg = readl(regs_gpf + UHDR1_UHDR_RST_OFF);
		reg |= UHDR1_UHDR_RST__CNTRL_VCC_RESET_N__MASK;
		writel(reg, regs_gpf + UHDR1_UHDR_RST_OFF);
		break;
#endif
	default:
		printk("unknown USB host controller 0x%d\n", (u32)usbh_ctrl);
		break;
	}
	udelay(5);

	phy_reset_ss(regs_gpf, 1);
	phy_reset_hs(regs_gpf, 1);
	udelay(5);
	phy_reset_ss(regs_gpf, 0);
	phy_reset_hs(regs_gpf, 0);

	iounmap(rstgenswrststatic2);
}

static void xhci_priv_plat_start(struct usb_hcd *hcd)
{
	struct xhci_plat_priv *priv = hcd_to_xhci_priv(hcd);

	if (priv->plat_start)
		priv->plat_start(hcd);
}

static int xhci_priv_init_quirk(struct usb_hcd *hcd)
{
	struct xhci_plat_priv *priv = hcd_to_xhci_priv(hcd);

	if (!priv->init_quirk)
		return 0;

	return priv->init_quirk(hcd);
}

static int xhci_priv_resume_quirk(struct usb_hcd *hcd)
{
	struct xhci_plat_priv *priv = hcd_to_xhci_priv(hcd);

	if (!priv->resume_quirk)
		return 0;

	return priv->resume_quirk(hcd);
}

static void xhci_plat_quirks(struct device *dev, struct xhci_hcd *xhci)
{
	/*
	 * As of now platform drivers don't provide MSI support so we ensure
	 * here that the generic code does not try to make a pci_dev from our
	 * dev struct in order to setup MSI
	 */
	xhci->quirks |= XHCI_PLAT;
}

/* called during probe() after chip reset completes */
static int xhci_plat_setup(struct usb_hcd *hcd)
{
	int ret;


	ret = xhci_priv_init_quirk(hcd);
	if (ret)
		return ret;

	return xhci_gen_setup(hcd, xhci_plat_quirks);
}

static int xhci_plat_start(struct usb_hcd *hcd)
{
	xhci_priv_plat_start(hcd);
	return xhci_run(hcd);
}

#ifdef CONFIG_OF
static const struct xhci_plat_priv xhci_plat_marvell_armada = {
	.init_quirk = xhci_mvebu_mbus_init_quirk,
};

static const struct xhci_plat_priv xhci_plat_renesas_rcar_gen2 = {
	.firmware_name = XHCI_RCAR_FIRMWARE_NAME_V1,
	.init_quirk = xhci_rcar_init_quirk,
	.plat_start = xhci_rcar_start,
	.resume_quirk = xhci_rcar_resume_quirk,
};

static const struct xhci_plat_priv xhci_plat_renesas_rcar_gen3 = {
	.firmware_name = XHCI_RCAR_FIRMWARE_NAME_V3,
	.init_quirk = xhci_rcar_init_quirk,
	.plat_start = xhci_rcar_start,
	.resume_quirk = xhci_rcar_resume_quirk,
};

static const struct of_device_id usb_xhci_of_match0[] = {
#ifdef ENABLE_HOST_USDR
#ifdef CONFIG_SOC_QUASAR6300
	{
		.compatible = "qbit,quasar-usbd",
	},
#else
	{
		.compatible = "qbit,quasar-usdr",
	},
#endif
#endif
	{
		.compatible = "xhci-platform",
	}, {
		.compatible = "marvell,armada-375-xhci",
		.data = &xhci_plat_marvell_armada,
	}, {
		.compatible = "marvell,armada-380-xhci",
		.data = &xhci_plat_marvell_armada,
	}, {
		.compatible = "renesas,xhci-r8a7790",
		.data = &xhci_plat_renesas_rcar_gen2,
	}, {
		.compatible = "renesas,xhci-r8a7791",
		.data = &xhci_plat_renesas_rcar_gen2,
	}, {
		.compatible = "renesas,xhci-r8a7793",
		.data = &xhci_plat_renesas_rcar_gen2,
	}, {
		.compatible = "renesas,xhci-r8a7795",
		.data = &xhci_plat_renesas_rcar_gen3,
	}, {
		.compatible = "renesas,xhci-r8a7796",
		.data = &xhci_plat_renesas_rcar_gen3,
	}, {
		.compatible = "renesas,rcar-gen2-xhci",
		.data = &xhci_plat_renesas_rcar_gen2,
	}, {
		.compatible = "renesas,rcar-gen3-xhci",
		.data = &xhci_plat_renesas_rcar_gen3,
	},
	{},
};
MODULE_DEVICE_TABLE(of, usb_xhci_of_match0);

static const struct of_device_id usb_xhci_of_match1[] = {
#ifdef ENABLE_HOST_UH2P
	{
		.compatible = "generic-xhci",
	},
#endif
#ifdef ENABLE_HOST_UHDR0
	{
		.compatible = "qbit,quasar-uhdr0",
	},
#endif
	{},
};
MODULE_DEVICE_TABLE(of, usb_xhci_of_match1);

/* only for QB66xx */
static const struct of_device_id usb_xhci_of_match2[] = {
#ifdef ENABLE_HOST_UHDR1
	{
		.compatible = "qbit,quasar-uhdr1",
	},
#endif
	{},
};
MODULE_DEVICE_TABLE(of, usb_xhci_of_match2);
#endif

static int xhci_plat_probe(struct platform_device *pdev)
{
	const struct xhci_plat_priv *priv_match;
	const struct hc_driver	*driver;
	struct device		*sysdev, *tmpdev;
	struct xhci_hcd		*xhci;
	struct resource         *res;
	struct usb_hcd		*hcd;
	struct clk              *clk;
	struct clk              *reg_clk;
	int			ret;
	int			irq;
	volatile u8	__iomem	*uh2preg;

	struct device		*dev = &pdev->dev;
	void __iomem		*regs_gpf;

	u32			reg;
	volatile u8	__iomem	*gctl;   /* For QBit xHCI driver configuring port direction */
#ifdef CONFIG_SOC_QUASAR6300
	volatile u8	__iomem	*piodir;   /* For QBit xHCI driver turning on VBUS if needed */
	volatile u8	__iomem	*piomsk;   /* For QBit xHCI driver turning on VBUS if needed */
	volatile u8	__iomem	*piodat;   /* For QBit xHCI driver turning on VBUS if needed */
#endif

	if (usb_disabled())
		return -ENODEV;

	res = platform_get_resource(pdev, IORESOURCE_MEM, 1);
	if (!res) {
		dev_err(dev, "missing memory resource\n");
		return -ENODEV;
	}

	switch ((u32)res->start) {
	case USDRGP_BASE:
		usbh_ctrl = USB_USDR;
		driver = &xhci_plat_hc_driver0;
        gctl = ioremap(USDR_GCTL, 4);
		break;
#ifdef CONFIG_SOC_QUASAR6300
	case UH2PGP_BASE:
		usbh_ctrl = USB_UH2P;
		driver = &xhci_plat_hc_driver1;
        gctl = ioremap(UH2P_GCTL, 4);
		break;
#else
	case UHDR0GP_BASE:
		usbh_ctrl = USB_UHDR0;
		driver = &xhci_plat_hc_driver1;
        gctl = ioremap(UHDR0_GCTL, 4);
		break;
	case UHDR1GP_BASE:
		usbh_ctrl = USB_UHDR1;
		driver = &xhci_plat_hc_driver2;
        gctl = ioremap(UHDR1_GCTL, 4);
		break;
#endif
	default:
		printk("unknown resource 0x%x for USB host controller\n", (u32)res->start);
		return -1;
	}

#ifdef CONFIG_SOC_QUASAR6300
	/* GPIO10 to enable VBUS power (QB63xx 19U DDR4 rev. A EVM) */
	if (usbh_ctrl == USB_USDR) {
		piodir = ioremap(SYS_PIODIRA, 4);
		piomsk = ioremap(SYS_PIOMSKA, 4);
		piodat = ioremap(SYS_PIODATA, 4);

		reg = readl(piodir);
		reg &= ~(1 << 10);
		writel(reg, piodir);
		reg = readl(piomsk);
		reg |= (1 << 10);
		writel(reg, piomsk);

		reg = readl(piodat);
		reg |= (1 << 10);
		writel(reg, piodat);

		iounmap(piodir);
		iounmap(piomsk);
		iounmap(piodat);
	}
#endif

	regs_gpf = devm_ioremap_resource(dev, res);
	if (IS_ERR(regs_gpf))
		return PTR_ERR(regs_gpf);

	usbh_hw_reset(regs_gpf);

	/* Disable filters since it is not needed for host ports */
	writel(0xf, regs_gpf + USDR_BUS_FILTER_BYPASS_OFF);

	irq = platform_get_irq(pdev, 0);
	if (irq < 0)
		return irq;

	/*
	 * sysdev must point to a device that is known to the system firmware
	 * or PCI hardware. We handle these three cases here:
	 * 1. xhci_plat comes from firmware
	 * 2. xhci_plat is child of a device from firmware (dwc3-plat)
	 * 3. xhci_plat is grandchild of a pci device (dwc3-pci)
	 */
	for (sysdev = &pdev->dev; sysdev; sysdev = sysdev->parent) {
		if (is_of_node(sysdev->fwnode) ||
			is_acpi_device_node(sysdev->fwnode))
			break;
#ifdef CONFIG_PCI
		else if (sysdev->bus == &pci_bus_type)
			break;
#endif
	}

	if (!sysdev)
		sysdev = &pdev->dev;

	/* Try to set 64-bit DMA first */
	if (WARN_ON(!sysdev->dma_mask))
		/* Platform did not initialize dma_mask */
		ret = dma_coerce_mask_and_coherent(sysdev,
						   DMA_BIT_MASK(64));
	else
		ret = dma_set_mask_and_coherent(sysdev, DMA_BIT_MASK(64));

	/* If seting 64-bit DMA mask fails, fall back to 32-bit DMA mask */
	if (ret) {
		ret = dma_set_mask_and_coherent(sysdev, DMA_BIT_MASK(32));
		if (ret)
			return ret;
	}

	pm_runtime_set_active(&pdev->dev);
	pm_runtime_enable(&pdev->dev);
	pm_runtime_get_noresume(&pdev->dev);

	hcd = __usb_create_hcd(driver, sysdev, &pdev->dev,
			       dev_name(&pdev->dev), NULL);
	if (!hcd) {
		ret = -ENOMEM;
		goto disable_runtime;
	}

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	hcd->regs = devm_ioremap_resource(&pdev->dev, res);
	if (IS_ERR(hcd->regs)) {
		ret = PTR_ERR(hcd->regs);
		goto put_hcd;
	}

	hcd->rsrc_start = res->start;
	hcd->rsrc_len = resource_size(res);

	/*
	 * Not all platforms have clks so it is not an error if the
	 * clock do not exist.
	 */
	reg_clk = devm_clk_get(&pdev->dev, "reg");
	if (!IS_ERR(reg_clk)) {
		ret = clk_prepare_enable(reg_clk);
		if (ret)
			goto put_hcd;
	} else if (PTR_ERR(reg_clk) == -EPROBE_DEFER) {
		ret = -EPROBE_DEFER;
		goto put_hcd;
	}

	clk = devm_clk_get(&pdev->dev, NULL);
	if (!IS_ERR(clk)) {
		ret = clk_prepare_enable(clk);
		if (ret)
			goto disable_reg_clk;
	} else if (PTR_ERR(clk) == -EPROBE_DEFER) {
		ret = -EPROBE_DEFER;
		goto disable_reg_clk;
	}

	/* For QBit xHCI driver configuring port direction as DFP (host) */
	reg = readl(gctl);
	reg &= USDR_GCTL__PRTCAPDIR__INV_MASK;
	reg |= (1 << USDR_GCTL__PRTCAPDIR__SHIFT);   /* 1 as host; 2 as device */
	writel(reg, gctl);
	iounmap(gctl);

	xhci = hcd_to_xhci(hcd);
	priv_match = of_device_get_match_data(&pdev->dev);
	if (priv_match) {
		struct xhci_plat_priv *priv = hcd_to_xhci_priv(hcd);

		/* Just copy data for now */
		if (priv_match)
			*priv = *priv_match;
	}

	device_wakeup_enable(hcd->self.controller);

	xhci->clk = clk;
	xhci->reg_clk = reg_clk;
	xhci->main_hcd = hcd;
	xhci->shared_hcd = __usb_create_hcd(driver, sysdev, &pdev->dev,
			dev_name(&pdev->dev), hcd);
	if (!xhci->shared_hcd) {
		ret = -ENOMEM;
		goto disable_clk;
	}

	/* imod_interval is the interrupt moderation value in nanoseconds. */
	xhci->imod_interval = 40000;

	/* Iterate over all parent nodes for finding quirks */
	for (tmpdev = &pdev->dev; tmpdev; tmpdev = tmpdev->parent) {

		if (device_property_read_bool(tmpdev, "usb2-lpm-disable"))
			xhci->quirks |= XHCI_HW_LPM_DISABLE;

		if (device_property_read_bool(tmpdev, "usb3-lpm-capable"))
			xhci->quirks |= XHCI_LPM_SUPPORT;

		if (device_property_read_bool(tmpdev, "quirk-broken-port-ped"))
			xhci->quirks |= XHCI_BROKEN_PORT_PED;

		device_property_read_u32(tmpdev, "imod-interval-ns",
					 &xhci->imod_interval);
	}

	hcd->usb_phy = devm_usb_get_phy_by_phandle(sysdev, "usb-phy", 0);
	if (IS_ERR(hcd->usb_phy)) {
		ret = PTR_ERR(hcd->usb_phy);
		if (ret == -EPROBE_DEFER)
			goto put_usb3_hcd;
		hcd->usb_phy = NULL;
	} else {
		ret = usb_phy_init(hcd->usb_phy);
		if (ret)
			goto put_usb3_hcd;
		hcd->skip_phy_initialization = 1;
	}

	ret = usb_add_hcd(hcd, irq, IRQF_SHARED);
	if (ret)
		goto disable_usb_phy;

	if (HCC_MAX_PSA(xhci->hcc_params) >= 4)
		xhci->shared_hcd->can_do_streams = 1;

	ret = usb_add_hcd(xhci->shared_hcd, irq, IRQF_SHARED);
	if (ret)
		goto dealloc_usb2_hcd;

	device_enable_async_suspend(&pdev->dev);
	pm_runtime_put_noidle(&pdev->dev);

	/*
	 * Prevent runtime pm from being on as default, users should enable
	 * runtime pm using power/control in sysfs.
	 */
	pm_runtime_forbid(&pdev->dev);

	/* Program GSBUSCFG0 to disable INCRBRSTENA */
	switch (usbh_ctrl) {
	case USB_USDR:
		uh2preg = ioremap(USDR_GSBUSCFG0, 4);
		break;
#ifdef CONFIG_SOC_QUASAR6300
	case USB_UH2P:
		uh2preg = ioremap(UH2P_GSBUSCFG0, 4);
		break;
#else
	case USB_UHDR0:
		uh2preg = ioremap(UHDR0_GSBUSCFG0, 4);
		break;
	case USB_UHDR1:
		uh2preg = ioremap(UHDR1_GSBUSCFG0, 4);
		break;
#endif
	default:
		printk("unknown USB host controller 0x%d\n", (u32)usbh_ctrl);
		return -1;
	}
	writel(4, uh2preg);
	iounmap(uh2preg);

	return 0;

dealloc_usb2_hcd:
	usb_remove_hcd(hcd);

disable_usb_phy:
	usb_phy_shutdown(hcd->usb_phy);

put_usb3_hcd:
	usb_put_hcd(xhci->shared_hcd);

disable_clk:
	clk_disable_unprepare(clk);

disable_reg_clk:
	clk_disable_unprepare(reg_clk);

put_hcd:
	usb_put_hcd(hcd);

disable_runtime:
	pm_runtime_put_noidle(&pdev->dev);
	pm_runtime_disable(&pdev->dev);

	return ret;
}

static int xhci_plat_remove(struct platform_device *dev)
{
	struct usb_hcd	*hcd = platform_get_drvdata(dev);
	struct xhci_hcd	*xhci = hcd_to_xhci(hcd);
	struct clk *clk = xhci->clk;
	struct clk *reg_clk = xhci->reg_clk;

	xhci->xhc_state |= XHCI_STATE_REMOVING;

	usb_remove_hcd(xhci->shared_hcd);
	usb_phy_shutdown(hcd->usb_phy);

	usb_remove_hcd(hcd);
	usb_put_hcd(xhci->shared_hcd);

	clk_disable_unprepare(clk);
	clk_disable_unprepare(reg_clk);
	usb_put_hcd(hcd);

	pm_runtime_set_suspended(&dev->dev);
	pm_runtime_disable(&dev->dev);

	return 0;
}

/* To reduce power consumption during low-power mode, set 1 to reset host PHY.
 * To detect host port status during low-power mode, set 0 to keep PHY alive.
 */
#define LP_RESET_PHY    1

#if LP_RESET_PHY
static int phy_reset = 0;
#endif

static int __maybe_unused xhci_plat_suspend(struct device *dev)
{
	int rc = 0;
	struct usb_hcd	*hcd = dev_get_drvdata(dev);
	struct xhci_hcd	*xhci = hcd_to_xhci(hcd);
#if LP_RESET_PHY
	void __iomem		*reg;
#endif

	/*
	 * xhci_suspend() needs `do_wakeup` to know whether host is allowed
	 * to do wakeup during suspend. Since xhci_plat_suspend is currently
	 * only designed for system suspend, device_may_wakeup() is enough
	 * to dertermine whether host is allowed to do wakeup. Need to
	 * reconsider this when xhci_plat_suspend enlarges its scope, e.g.,
	 * also applies to runtime suspend.
	 */
	//return xhci_suspend(xhci, device_may_wakeup(dev));
	rc = xhci_suspend(xhci, device_may_wakeup(dev));
#if LP_RESET_PHY
	if (rc == 0) {
		/* Reset PHY to save power */
		switch (usbh_ctrl) {
		case USB_USDR:
			reg = ioremap(USDR_Slave_Endianess_reg, 4);
			phy_reset_ss(reg, 1);
			break;
#ifdef CONFIG_SOC_QUASAR6300
		case USB_UH2P:
			reg = ioremap(UH2P_Slave_Endianess_reg, 4);
			break;
#else
		case USB_UHDR0:
			reg = ioremap(UHDR0_UHDR_Slave_Endianess_reg, 4);
			break;
		case USB_UHDR1:
			reg = ioremap(UHDR1_UHDR_Slave_Endianess_reg, 4);
			break;
#endif
		default:
			printk("unknown USB host controller %d\n", usbh_ctrl);
			return -1;
		}
		phy_reset_hs(reg, 1);
		phy_reset = 1;
		iounmap(reg);
	}
#endif
	return rc;
}

static int __maybe_unused xhci_plat_resume(struct device *dev)
{
	struct usb_hcd	*hcd = dev_get_drvdata(dev);
	struct xhci_hcd	*xhci = hcd_to_xhci(hcd);
#if LP_RESET_PHY
	void __iomem		*reg;
#endif
	int ret;

#if LP_RESET_PHY
	/* Enable PHY if it was reset */
	if (phy_reset) {
		switch (usbh_ctrl) {
		case USB_USDR:
			reg = ioremap(USDR_Slave_Endianess_reg, 4);
			phy_reset_ss(reg, 0);
			break;
#ifdef CONFIG_SOC_QUASAR6300
		case USB_UH2P:
			reg = ioremap(UH2P_Slave_Endianess_reg, 4);
			break;
#else
		case USB_UHDR0:
			reg = ioremap(UHDR0_UHDR_Slave_Endianess_reg, 4);
			break;
		case USB_UHDR1:
			reg = ioremap(UHDR1_UHDR_Slave_Endianess_reg, 4);
			break;
#endif
		default:
			printk("unknown USB host controller %d\n", usbh_ctrl);
			return -1;
		}
		phy_reset_hs(reg, 0);
		phy_reset = 0;
		iounmap(reg);
	}
#endif

	ret = xhci_priv_resume_quirk(hcd);
	if (ret)
		return ret;

	return xhci_resume(xhci, 0);
}

static int __maybe_unused xhci_plat_runtime_suspend(struct device *dev)
{
	struct usb_hcd  *hcd = dev_get_drvdata(dev);
	struct xhci_hcd *xhci = hcd_to_xhci(hcd);

	return xhci_suspend(xhci, true);
}

static int __maybe_unused xhci_plat_runtime_resume(struct device *dev)
{
	struct usb_hcd  *hcd = dev_get_drvdata(dev);
	struct xhci_hcd *xhci = hcd_to_xhci(hcd);

	return xhci_resume(xhci, 0);
}

static const struct dev_pm_ops xhci_plat_pm_ops = {
	SET_SYSTEM_SLEEP_PM_OPS(xhci_plat_suspend, xhci_plat_resume)

	SET_RUNTIME_PM_OPS(xhci_plat_runtime_suspend,
			   xhci_plat_runtime_resume,
			   NULL)
};

static const struct acpi_device_id usb_xhci_acpi_match[] = {
	/* XHCI-compliant USB Controller */
	{ "PNP0D10", },
	{ }
};
MODULE_DEVICE_TABLE(acpi, usb_xhci_acpi_match);

/* the 1st host controller */
static struct platform_driver usb_xhci_driver0 = {
	.probe	= xhci_plat_probe,
	.remove	= xhci_plat_remove,
	.driver	= {
		.name = "xhci-hcd0",
		.pm = &xhci_plat_pm_ops,
		.of_match_table = of_match_ptr(usb_xhci_of_match0),
		.acpi_match_table = ACPI_PTR(usb_xhci_acpi_match),
	},
};
MODULE_ALIAS("platform:xhci-hcd0");

static int __init xhci_plat_init0(void)
{
	xhci_init_driver(&xhci_plat_hc_driver0, &xhci_plat_overrides0);
	return platform_driver_register(&usb_xhci_driver0);
}
module_init(xhci_plat_init0);

static void __exit xhci_plat_exit0(void)
{
	platform_driver_unregister(&usb_xhci_driver0);
}
module_exit(xhci_plat_exit0);

/* the 2nd host controller */
static struct platform_driver usb_xhci_driver1 = {
	.probe	= xhci_plat_probe,
	.remove	= xhci_plat_remove,
	.driver	= {
		.name = "xhci-hcd1",
		.pm = &xhci_plat_pm_ops,
		.of_match_table = of_match_ptr(usb_xhci_of_match1),
		.acpi_match_table = ACPI_PTR(usb_xhci_acpi_match),
	},
};
MODULE_ALIAS("platform:xhci-hcd1");

static int __init xhci_plat_init1(void)
{
	xhci_init_driver(&xhci_plat_hc_driver1, &xhci_plat_overrides1);
	return platform_driver_register(&usb_xhci_driver1);
}
module_init(xhci_plat_init1);

static void __exit xhci_plat_exit1(void)
{
	platform_driver_unregister(&usb_xhci_driver1);
}
module_exit(xhci_plat_exit1);

/* the 3rd host controller */
static struct platform_driver usb_xhci_driver2 = {
	.probe	= xhci_plat_probe,
	.remove	= xhci_plat_remove,
	.driver	= {
		.name = "xhci-hcd2",
		.pm = &xhci_plat_pm_ops,
		.of_match_table = of_match_ptr(usb_xhci_of_match2),
		.acpi_match_table = ACPI_PTR(usb_xhci_acpi_match),
	},
};
MODULE_ALIAS("platform:xhci-hcd2");

static int __init xhci_plat_init2(void)
{
	xhci_init_driver(&xhci_plat_hc_driver2, &xhci_plat_overrides2);
	return platform_driver_register(&usb_xhci_driver2);
}
module_init(xhci_plat_init2);

static void __exit xhci_plat_exit2(void)
{
	platform_driver_unregister(&usb_xhci_driver2);
}
module_exit(xhci_plat_exit2);

MODULE_DESCRIPTION("xHCI Platform Host Controller Driver");
MODULE_LICENSE("GPL");
