/*
 * drivers/rtc/rtc-quasar.c
 * A driver for the RTC embedded in the QBit SOCs
 *
 * Copyright (c) 2014, 2015, The Linux Foundation. All rights reserved.
 *
 * Copyright (c) 2014 Cambridge Silicon Radio Ltd.
 * 
 * Copyright (c) 2017-2021, QBit Semiconductor LTD.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */
// =========================================================
//
//  $DateTime: 2021/10/28 10:29:17 $
//  $Change: 54504 $
//
// =========================================================
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/jiffies.h>
#include <linux/platform_device.h>
#include <linux/rtc.h>
#include <linux/io.h>
#include <linux/mutex.h>
#include <linux/of.h>
#include <linux/slab.h>

// operation modes
#define V_RTC_MODE_NORMAL	0
#define V_RTC_MODE_READ		1
#define V_RTC_MODE_WRITE	2

struct rtc_ctl_regs {
	volatile u32 rtc_cac;
	volatile u32 rtc_ca1;
	volatile u32 rtc_ca2;
	volatile u32 rtc_ca3;
	volatile u32 rtc_ca4;
	volatile u32 rtc_ca5;
	volatile u32 rtc_ca6;
	volatile u32 rtc_ca7;
    volatile u32 rtc_al1;
    volatile u32 rtc_al2;
    volatile u32 rtc_al3;
    volatile u32 rtc_al4;
    volatile u32 rtc_tst;
	volatile u32 reserved[0xcc];
	volatile u32 rtc_gpr;
};

struct rtc_quasar {
	struct rtc_device	*rtc;
    struct rtc_ctl_regs __iomem *regs;
	unsigned long		irq; 
	struct mutex		lock;
};

static inline u32 rtc_readl(struct rtc_quasar *rtc, int reg)
{
	return readl(rtc->regs + reg);
}

static inline void rtc_writel(struct rtc_quasar *rtc, int reg, u32 val)
{
	writel(val, rtc->regs + reg);
}
		
static void rtc_setmode(struct rtc_quasar *rtc, struct rtc_ctl_regs *regs, u32 mode)
{
	u32 value;
    
	value = readl(&regs->rtc_cac);
	value &= (~0x00000003);
	value |= mode;
	writel(value, &regs->rtc_cac);	
}

/*
static void rtc_reset(struct rtc_quasar *rtc, struct rtc_ctl_regs *regs, u32 recover)
{
	u32 value;
    
	value = readl(&regs->rtc_cac);
	value |= 0x00000080;
	writel(value, &reg->rtc_cac);
	if(recover)
	{
		value &= ~0x00000080;
        writel(value, &regs->rtc_cac);
	}
} */

static void rtc_timeset(struct rtc_quasar *rtc, struct rtc_time *tm)
{
	struct rtc_ctl_regs *regs = rtc->regs;
    
	rtc_setmode(rtc, regs, V_RTC_MODE_WRITE);
	udelay(100);
    
	//  update date/time parameters
	writel(tm->tm_sec, &regs->rtc_ca1);
	writel(tm->tm_min, &regs->rtc_ca2);
	writel(tm->tm_hour, &regs->rtc_ca3);
	writel(tm->tm_mday, &regs->rtc_ca4);
	writel(tm->tm_wday, &regs->rtc_ca5);
	writel(tm->tm_mon, &regs->rtc_ca6);
	writel(tm->tm_year, &regs->rtc_ca7);
	
    rtc_setmode(rtc, regs, V_RTC_MODE_NORMAL);
//printk("    sec     %d\n", tm->tm_sec);    
//printk("    min     %d\n", tm->tm_min);  
//printk("    hour    %d\n", tm->tm_hour);  
//printk("    mday    %d\n", tm->tm_mday);  
//printk("    wday    %d\n", tm->tm_wday);  
//printk("    mon     %d\n", tm->tm_mon);  
//printk("    year    %d\n", tm->tm_year);     
}

static void rtc_timeget(struct rtc_quasar *rtc, struct rtc_time *tm)
{   
	struct rtc_ctl_regs *regs = rtc->regs;
    
	rtc_setmode(rtc, regs, V_RTC_MODE_READ);

	// Get date/time parameter
	tm->tm_sec = readl(&regs->rtc_ca1);
	tm->tm_min = readl(&regs->rtc_ca2);
	tm->tm_hour= readl(&regs->rtc_ca3);
	tm->tm_mday= readl(&regs->rtc_ca4);
	tm->tm_wday= readl(&regs->rtc_ca5);
	tm->tm_mon = readl(&regs->rtc_ca6);
	tm->tm_year= readl(&regs->rtc_ca7);
	
//printk("    sec     %d\n", tm->tm_sec);    
//printk("    min     %d\n", tm->tm_min);  
//printk("    hour    %d\n", tm->tm_hour);  
//printk("    mday    %d\n", tm->tm_mday);  
//printk("    wday    %d\n", tm->tm_wday);  
//printk("    mon     %d\n", tm->tm_mon);  
//printk("    year    %d\n", tm->tm_year);  

	rtc_setmode(rtc, regs, V_RTC_MODE_NORMAL);
}

static int quasar_rtc_readtime(struct device *dev, struct rtc_time *tm)
{
	struct rtc_quasar *rtc = dev_get_drvdata(dev);
//printk("----- RTC: read time\n");
	mutex_lock(&rtc->lock);

	rtc_timeget(rtc, tm);
    
	mutex_unlock(&rtc->lock);
	return 0;
}

static int quasar_rtc_settime(struct device *dev, struct rtc_time *tm)
{
	struct rtc_quasar *rtc = dev_get_drvdata(dev);
//printk("----- RTC: set time\n");

	mutex_lock(&rtc->lock);

	rtc_timeset(rtc, tm);

	mutex_unlock(&rtc->lock);

	return 0;
}

//static irqreturn_t quasar_rtc_interrupt(int irq, void *dev_id)
//{
//	return IRQ_NONE;
//}

static struct rtc_class_ops quasar_rtc_ops = {
	.read_time	= quasar_rtc_readtime,
	.set_time	= quasar_rtc_settime,
};

static int __init quasar_rtc_probe(struct platform_device *pdev)
{
	struct rtc_quasar *rtc;
	/*struct device_node *np = pdev->dev.of_node;*/
	struct resource	*regs;
	int irq, ret;
	
	rtc = devm_kzalloc(&pdev->dev, sizeof(struct rtc_quasar), GFP_KERNEL);
	if (rtc == NULL) {
		dev_dbg(&pdev->dev, "out of memory\n");
		return -ENOMEM;
	}

	platform_set_drvdata(pdev, rtc);

	mutex_init(&rtc->lock);
	
	irq = platform_get_irq(pdev, 0);
	if (irq < 0) {
		dev_dbg(&pdev->dev, "could not get irq\n");
		ret = -ENXIO;
		goto out;
	}
	rtc->irq = irq;

	regs = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (regs == NULL) {
		dev_dbg(&pdev->dev, "no mmio resource defined\n");
		ret = -ENXIO;
		goto out;
	}
	rtc->regs = devm_ioremap_resource(&pdev->dev, regs);
	if (rtc->regs == NULL) {
		ret = -ENOMEM;
		dev_dbg(&pdev->dev, "could not map I/O memory\n");
		goto out;
	}
    
	smp_wmb();		/* Paranoid. */

	//ret = request_irq(irq, quasar_rtc_interrupt, IRQF_DISABLED, "rtc", rtc);
	//if (ret) {
	//	dev_dbg(&pdev->dev, "could not request irq %d\n", irq);
	//	goto out_iounmap;
	//}

	rtc->rtc = rtc_device_register(pdev->name, &pdev->dev,
				       &quasar_rtc_ops, THIS_MODULE);
	if (IS_ERR(rtc->rtc)) {
		dev_dbg(&pdev->dev, "could not register rtc device\n");
		ret = PTR_ERR(rtc->rtc);
		goto out_free_irq;
	}
	device_init_wakeup(&pdev->dev, 1);

	dev_info(&pdev->dev, "Quasar RTC at MMIO %08lx irq %ld\n",
		 (unsigned long)rtc->regs, rtc->irq);

	return 0;

out_free_irq:
	free_irq(irq, rtc);
//out_iounmap:
	iounmap(rtc->regs);
out:
	kfree(rtc);
	return ret;
}

static int __exit quasar_rtc_remove(struct platform_device *pdev)
{
	struct rtc_quasar *rtc = platform_get_drvdata(pdev);

	device_init_wakeup(&pdev->dev, 0);

	free_irq(rtc->irq, rtc);
	iounmap(rtc->regs);
	rtc_device_unregister(rtc->rtc);
	kfree(rtc);
	platform_set_drvdata(pdev, NULL);

	return 0;
}

static const struct of_device_id quasar_rtc_of_match[] = {
	{ .compatible = "qbit,quasar-rtc"},
	{},
};
MODULE_DEVICE_TABLE(of, sirfsoc_rtc_of_match);

static struct platform_driver quasar_rtc_driver_probe = {
	.driver = {
		.name = "quasar-rtc",
		.owner = THIS_MODULE,
		/*.pm = &quasar_rtc_pm_ops,*/
		.of_match_table = quasar_rtc_of_match,
	},
	.probe = quasar_rtc_probe,
	.remove = quasar_rtc_remove,
};
module_platform_driver(quasar_rtc_driver_probe);

MODULE_DESCRIPTION("Quasar SoC RTC driver");
MODULE_AUTHOR("QBit Semiconductor LTD.");
MODULE_DESCRIPTION("Real time clock for QBit SOCs");
MODULE_LICENSE("GPL");
