/*
 * QBit Quasar PCIe host controller driver
 *
 * Copyright (c) 2014, 2015, The Linux Foundation. All rights reserved.
 *
 * Copyright (c) 2018, 2019, 2020, 2021 QBit Semiconductor LTD.
 *
 * Based on pci-host-generic.c
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <linux/kernel.h>
#include <linux/interrupt.h>
#include <linux/delay.h>
#include <linux/module.h>
#include <linux/of_address.h>
#include <linux/of_pci.h>
#include <linux/pci.h>
#include <linux/platform_device.h>
#include <asm/pci.h>
#include "../pci.h"
#include "pcie-quasar.h"
#ifdef CONFIG_SOC_QUASAR6300
//#include "../../../../drivers/quasar/qbsoc_ddr_regs_B0.h"
//#include "../../../../drivers/quasar/qbsoc_sys_regs_B0.h"
//#include "../../../../drivers/quasar/qbsoc_pci_regs_B0.h"
#include "../../../../drivers/quasar/6300/qbsoc_ddr_regs_B0.h"
#include "../../../../drivers/quasar/6300/qbsoc_sys_regs_B0.h"
#include "../../../../drivers/quasar/6300/qbsoc_pci_regs_B0.h"
#else
#include "../../../../drivers/quasar/6600/qbsoc_ddr_regs.h"
#include "../../../../drivers/quasar/6600/qbsoc_sys_regs.h"
#include "../../../../drivers/quasar/6600/qbsoc_pci_regs.h"
#endif

#ifndef IN_LPAE
#define IN_LPAE 0
#endif

#ifdef CONFIG_SOC_QUASAR6300
#define uint_port u32
#else
#define uint_port u64
#endif

#define DEBUG_PCIE (0)

#define SPEED_GEN1 (0)   /* 0 for Gen2, 1 for Gen 1 */

#define COHERENCY  (0)   /* 1 to enable coherency */

#ifdef BOARD_TYPE_RC3L
#define EVM_TWO_RC   /* defined when QB66xx enables two PCIe host controllers */
#endif

#define DEVICE_IS_QUASAR (0)   /* 1 when the device is QB63xx or QB66xx EP */

struct quasar_pcie_cfg_bus_ops {
	u32 bus_shift;
	void __iomem *(*map_bus)(struct pci_bus *, unsigned int, int);
};

struct quasar_pcie_cfg_windows {
	struct resource				res;
	struct resource				bus_range;
	void __iomem				**win;

	const struct quasar_pcie_cfg_bus_ops	*ops;
};

/* Definitions for ctrl_no */
#define QUASAR_PCIE1 (1)
#define QUASAR_PCIE2 (2)

struct quasar_pcie {
	struct pci_host_bridge			host;
	struct device		*dev;

	void __iomem		*base_rstgen;
	void __iomem		*base_clkgen;
	void __iomem		*base_ctrl;
	void __iomem		*base_wrap;
	void __iomem		*base_gpf;
	struct resource		*rsc_rstgen;
	struct resource		*rsc_clkgen;
	struct resource		*rsc_ctrl;
	struct resource		*rsc_wrap;
	struct resource		*rsc_gpf;
	int			irq;

	/* PCIEWRAP registers */
	void __iomem		*sw_rst;
	void __iomem		*misc_ctrl;
	void __iomem		*misc_stat;
	void __iomem		*sw_bootstrap;
	void __iomem		*int_flag;
	void __iomem		*int_en;
	void __iomem		*int_clr;
	void __iomem		*phy_rst_ctrl;
	void __iomem		*phy_clk_ctrl;
	void __iomem		*phy_pipe_stat;
#ifdef CONFIG_SOC_QUASAR6300
	void __iomem		*phy_test_ctrl;
#else
	void __iomem		*phy_test_ctrl0;
	void __iomem		*phy_test_ctrl1;
	void __iomem		*phy_test_ctrl2;
	void __iomem		*phy_test_ctrl3;
#endif
	void __iomem		*clkrx_ctrl1;

	struct quasar_pcie_cfg_windows	cfg;
	struct list_head	resources;
	u32	src_pci_clk_mhz;

	u8	ctrl_no;   /* QUASAR_PCIE1 (1) or QUASAR_PCIE2 (2) */
};

struct quasar_pcie_map {
	u32 axi_amap_ctrl0;
	u32 axi_amap_axi_base0;
	u32 axi_amap_axi_base0x;
	u32 axi_amap_pex_basel0;
	u32 axi_amap_pex_baseh0;
#ifdef CONFIG_SOC_QUASAR6600
	u32 axi_amap_size_eam0x;
#endif

	u32 axi_amap_ctrl1;
	u32 axi_amap_axi_base1;
	u32 axi_amap_axi_base1x;
	u32 axi_amap_pex_basel1;
	u32 axi_amap_pex_baseh1;
#ifdef CONFIG_SOC_QUASAR6600
	u32 axi_amap_size_eam1x;
#endif

	u32 axi_amap_ctrl2;
	u32 axi_amap_axi_base2;
	u32 axi_amap_axi_base2x;
	u32 axi_amap_pex_basel2;
	u32 axi_amap_pex_baseh2;
#ifdef CONFIG_SOC_QUASAR6600
	u32 axi_amap_size_eam2x;
#endif

	u32 axi_amap_ctrl3;
	u32 axi_amap_axi_base3;
	u32 axi_amap_axi_base3x;
	u32 axi_amap_pex_basel3;
	u32 axi_amap_pex_baseh3;
#ifdef CONFIG_SOC_QUASAR6600
	u32 axi_amap_size_eam3x;
#endif

	u32 pex_amap_ctrl0;
	u32 pex_amap_axi_base0;
	u32 pex_amap_axi_base0x;
	u32 pex_amap_pex_basel0;
	u32 pex_amap_pex_baseh0;
#ifdef CONFIG_SOC_QUASAR6600
	u32 pex_amap_axi_eam0x;
#endif

	u32 pex_amap_ctrl1;
	u32 pex_amap_axi_base1;
	u32 pex_amap_axi_base1x;
	u32 pex_amap_pex_basel1;
	u32 pex_amap_pex_baseh1;
#ifdef CONFIG_SOC_QUASAR6600
	u32 pex_amap_axi_eam1x;
#endif

	u32 pex_amap_ctrl2;
	u32 pex_amap_axi_base2;
	u32 pex_amap_axi_base2x;
	u32 pex_amap_pex_basel2;
	u32 pex_amap_pex_baseh2;
#ifdef CONFIG_SOC_QUASAR6600
	u32 pex_amap_axi_eam2x;
#endif

	u32 pex_amap_ctrl3;
	u32 pex_amap_axi_base3;
	u32 pex_amap_axi_base3x;
	u32 pex_amap_pex_basel3;
	u32 pex_amap_pex_baseh3;
#ifdef CONFIG_SOC_QUASAR6600
	u32 pex_amap_axi_eam3x;
#endif
};
static struct quasar_pcie_map pcie_map;

static u32 bar_like_size(u32 x)
{
#ifndef CONFIG_SOC_QUASAR6300
    x = ~x;
    x++;
#endif
    return x;
}

static void __iomem *quasar_pcie_map_cfg_bus_addr(struct pci_bus *bus,
					     unsigned int devfn,
					     int where)
{
	struct quasar_pcie *pcie = bus->sysdata;
	resource_size_t idx = bus->number - pcie->cfg.bus_range.start;

	return pcie->cfg.win[idx - 1] + ((devfn << 16) | where);
}

static struct quasar_pcie_cfg_bus_ops quasar_pcie_cfg_cam_bus_ops = {
	.bus_shift	= 16,
	.map_bus	= quasar_pcie_map_cfg_bus_addr,
};

static int quasar_pcie_config_read(struct pci_bus *bus, unsigned int devfn,
				int where, int size, u32 *val)
{
	void __iomem *addr;
	struct quasar_pcie *pcie = bus->sysdata;

	if (devfn) {
		return PCIBIOS_DEVICE_NOT_FOUND;   /* Only support one function for now */
	}

	switch (bus->number) {
	case 0:
		addr = (void __iomem *)((uint_port)pcie->base_ctrl + where);

		switch (size) {
#ifdef CONFIG_SOC_QUASAR6300
		case 1:   /* 63XXA0 cannot do byte or word access to PCIe registers */
			*val = readl((void __iomem *)((uint_port)addr - ((uint_port)addr%4)));
			*val = *val >> (((uint_port)addr%4) * 8);
			*val &= 0xff;
			break;
		case 2:   /* 63XXA0 cannot do byte or word access to PCIe registers */
			*val = readl((void __iomem *)((uint_port)addr - ((uint_port)addr%4)));
			*val = *val >> (((uint_port)addr%4) * 8);
			*val &= 0xffff;
			break;
#else
		case 1:
			*val = readb(addr);
			break;
		case 2:
			*val = readw(addr);
			break;
#endif
		default:
			*val = readl(addr);
			break;
		}
		break;

	case 1:
		addr = pcie->cfg.ops->map_bus(bus, devfn, where);

		switch (size) {
		case 1:
			*val = readb(addr);
			break;
		case 2:
			*val = readw(addr);
			break;
		default:
			*val = readl(addr);
			break;
		}
		break;

	default:
		return PCIBIOS_DEVICE_NOT_FOUND;   /* Don't support other bus */
	}

#if DEBUG_PCIE
#ifdef CONFIG_SOC_QUASAR6300
	dev_warn(pcie->dev, "read bus%d where(0x%08x) addr(0x%08x) val(0x%08x)\n",
			bus->number, where, (uint_port)addr, *val);
#else
	dev_warn(pcie->dev, "read bus%d where(0x%08x) addr(0x%llx) val(0x%08x)\n",
			bus->number, where, (uint_port)addr, *val);
#endif
#endif
	return PCIBIOS_SUCCESSFUL;
}

static int quasar_pcie_config_write(struct pci_bus *bus, unsigned int devfn,
				 int where, int size, u32 val)
{
	void __iomem *addr;
	struct quasar_pcie *pcie = bus->sysdata;

#ifdef CONFIG_SOC_QUASAR6300
    u32 tmp;
#endif

	if (bus->number) {
		addr = pcie->cfg.ops->map_bus(bus, devfn, where);
#if DEBUG_PCIE
#ifdef CONFIG_SOC_QUASAR6300
	dev_warn(pcie->dev, "write bus%d where(0x%08x) addr(0x%08x) val(0x%08x)\n",
			bus->number, where, (uint_port)addr, val);
#else
	dev_warn(pcie->dev, "write bus%d where(0x%08x) addr(0x%llx) val(0x%08x)\n",
			bus->number, where, (uint_port)addr, val);
#endif
#endif

		switch (size) {
		case 1:
			writeb(val, addr);
			break;
		case 2:
			writew(val, addr);
			break;
		default:
			writel(val, addr);
		}
	} else { /* 63XXA0 cannot do byte or word access to PCIe registers */
		addr = (void __iomem *)((uint_port)pcie->base_ctrl + where);
#if DEBUG_PCIE
#ifdef CONFIG_SOC_QUASAR6300
	dev_warn(pcie->dev, "write bus%d where(0x%08x) addr(0x%08x) val(0x%08x)\n",
			bus->number, where, (uint_port)addr, val);
#else
	dev_warn(pcie->dev, "write bus%d where(0x%08x) addr(0x%llx) val(0x%08x)\n",
			bus->number, where, (uint_port)addr, val);
#endif
#endif

		switch (size) {
#ifdef CONFIG_SOC_QUASAR6300
		case 1:
			tmp = readl((void __iomem *)((uint_port)addr - ((uint_port)addr%4)));
			val = val << (((uint_port)addr%4) * 8);
			tmp &= (0xffffffff) - (0xff << (((uint_port)addr%4) * 8));
			tmp = tmp + val;
			writel(tmp, (void __iomem *)((uint_port)addr - ((uint_port)addr%4)));
			break;
		case 2:
			tmp = readl((void __iomem *)((uint_port)addr - ((uint_port)addr%4)));
			val = val << (((uint_port)addr%4) * 8);
			tmp &= (0xffffffff) - (0xffff << (((uint_port)addr%4) * 8));
			tmp = tmp + val;
			writel(tmp, (void __iomem *)((uint_port)addr - ((uint_port)addr%4)));
			break;
#else
		case 1:
			writeb(val, addr);
			break;
		case 2:
			writew(val, addr);
			break;
#endif
		default:
			writel(val, addr);
		}
	}

	return PCIBIOS_SUCCESSFUL;
}

static struct pci_ops quasar_pcie_ops = {
	.map_bus	= quasar_pcie_map_cfg_bus_addr,
	.read	= quasar_pcie_config_read,
	.write	= quasar_pcie_config_write,
};

static const struct of_device_id quasar_pcie_of_match1[] = {
#ifdef CONFIG_SOC_QUASAR6300
	{ .compatible = "qbit,quasar-pcie",
#else
	{ .compatible = "qbit,quasar-pcie1",
#endif
	  .data = &quasar_pcie_cfg_cam_bus_ops },
	{ },
};
MODULE_DEVICE_TABLE(of, quasar_pcie_of_match1);

#ifdef EVM_TWO_RC
static const struct of_device_id quasar_pcie_of_match2[] = {
	{ .compatible = "qbit,quasar-pcie2",
	  .data = &quasar_pcie_cfg_cam_bus_ops },
	{ },
};
MODULE_DEVICE_TABLE(of, quasar_pcie_of_match2);
#endif

static int quasar_pcie_calc_io_offset(struct device *dev,
				  struct of_pci_range *range,
				  struct resource *res,
				  resource_size_t *offset)
{
	static atomic_t wins = ATOMIC_INIT(0);
#ifdef CONFIG_SOC_QUASAR6300
	int err;
#endif
	int idx, max_win;
	unsigned int window;

	if (!PAGE_ALIGNED(range->cpu_addr))
		return -EINVAL;

	max_win = (IO_SPACE_LIMIT + 1) / SZ_64K;
	idx = atomic_inc_return(&wins);
	if (idx > max_win)
		return -ENOSPC;

	window = (idx - 1) * SZ_64K;
#ifdef CONFIG_SOC_QUASAR6300
	err = pci_ioremap_io(window, range->cpu_addr);
	if (err)
		return err;
#endif

	of_pci_range_to_resource(range, dev->of_node, res);
	res->start = window;
	res->end = res->start + range->size - 1;
	*offset = window - range->pci_addr;
	return 0;
}

static int quasar_pcie_calc_mem_offset(struct device *dev,
				   struct of_pci_range *range,
				   struct resource *res,
				   resource_size_t *offset)
{
	of_pci_range_to_resource(range, dev->of_node, res);
	*offset = range->cpu_addr - range->pci_addr;
	return 0;
}

static int quasar_pcie_map_dma_ranges(struct quasar_pcie *pcie)
{
	struct of_pci_range range;
	struct of_pci_range_parser parser;
	struct device *dev = pcie->host.dev.parent;
	struct device_node *np = dev->of_node;
	const int na = 3, ns = 2;
	int len;
	u32 reg_offset = 0x10, idx = 0;

	parser.node = np;
	parser.pna = of_n_addr_cells(parser.node);
	parser.np = parser.pna + na + ns;

	parser.range = of_get_property(parser.node, "dma-ranges", &len);
	if (!parser.range)
		return -ENOENT;

	parser.end = parser.range + len / sizeof(__be32);

	for_each_of_pci_range(&parser, &range) {
		u32 restype = range.flags & IORESOURCE_TYPE_BITS;
		u32 val;

		if (idx == 3)
			val = bar_like_size(range.size) | 1;   /* window 3 is for MSI */
		else
			val = bar_like_size(range.size) | ((restype>>7) | 1);

#if DEBUG_PCIE
		dev_warn(pcie->dev, "dma-ranges: restype(0x%08x) val(0x%08x)\n",
				restype, val);
		dev_warn(pcie->dev, "dma-ranges: flags(0x%08x) pci(0x%llx) cpu(0x%llx) size(0x%llx)\n",
				range.flags, range.pci_addr, range.cpu_addr, range.size);
#endif

		/* Window x: memory space, starting from register index 0 */
		writel(val, pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_CTRL0_OFF + reg_offset*idx);
		writel((u32)range.cpu_addr,
				pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_AXI_BASE0_OFF + reg_offset*idx);
		writel((u32)((u64)range.cpu_addr>>32) | (COHERENCY << 4),
				pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_AXI_BASE0X_OFF + 4*idx);
		writel((u32)range.pci_addr,
				pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_PEX_BASEL0_OFF + reg_offset*idx);
		writel((u32)((u64)range.pci_addr>>32),
				pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_PEX_BASEH0_OFF + reg_offset*idx);
#ifdef CONFIG_SOC_QUASAR6600
#if IN_LPAE
		writel((COHERENCY << 4) | 0xF, pcie->base_ctrl + 0x0000BEF0 + 4*idx);
#else
		writel((COHERENCY << 4), pcie->base_ctrl + 0x0000BEF0 + 4*idx);
#endif
#endif
#if DEBUG_PCIE
		dev_warn(pcie->dev, "IN: PCIE1_PAB_PEX_AMAP_CTRL%d     =0x%08X\n",
			idx, readl(pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_CTRL0_OFF + reg_offset*idx));
		dev_warn(pcie->dev, "    PCIE1_PAB_PEX_AMAP_AXI_BASE%d =0x%08X\n",
			idx, readl(pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_AXI_BASE0_OFF + reg_offset*idx));
		dev_warn(pcie->dev, "    PCIE1_PAB_PEX_AMAP_AXI_BASE%dX=0x%08X\n",
			idx, readl(pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_AXI_BASE0X_OFF + 4*idx));
		dev_warn(pcie->dev, "    PCIE1_PAB_PEX_AMAP_PEX_BASEL%d=0x%08X\n",
			idx, readl(pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_PEX_BASEL0_OFF + reg_offset*idx));
		dev_warn(pcie->dev, "    PCIE1_PAB_PEX_AMAP_PEX_BASEH%d=0x%08X\n",
			idx, readl(pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_PEX_BASEH0_OFF + reg_offset*idx));
#ifdef CONFIG_SOC_QUASAR6600
		dev_warn(pcie->dev, "    PCIE1_PAB_PEX_AMAP_AXI_EAM%dX=0x%08X\n",
			idx, readl(pcie->base_ctrl + 0x0000BEF0 + 4*idx));
#endif
#endif
		if (idx == 3)
			break;
		idx++;
	}

	writel(PCIE1_PAB_PEX_PIO_CTRL0__PIO_ENABLE__MASK,
			pcie->base_ctrl + PCIE1_PAB_PEX_PIO_CTRL0_OFF);
	return 0;
}

static int quasar_pcie_map_ranges(struct quasar_pcie *pcie)
{
	struct of_pci_range range;
	struct of_pci_range_parser parser;
	int err = 0;
	struct device *dev = pcie->host.dev.parent;
	struct device_node *np = dev->of_node;
	u32 reg_offset = 0x10, idx = 0;

	if (of_pci_range_parser_init(&parser, np)) {
		dev_err(dev, "missing \"ranges\" property\n");
		return -EINVAL;
	}

	for_each_of_pci_range(&parser, &range) {
		struct resource *parent, *res;
		resource_size_t offset = 0;
		u32 restype = range.flags & IORESOURCE_TYPE_BITS;
		u32 val = bar_like_size(range.size) | ((restype>>7) | 1);

		res = devm_kmalloc(dev, sizeof(*res), GFP_KERNEL);
		if (!res) {
			err = -ENOMEM;
			goto out_release_res;
		}

		switch (restype) {
		case IORESOURCE_IO:
			parent = &ioport_resource;
			err = quasar_pcie_calc_io_offset(dev, &range, res, &offset);
			break;
		case IORESOURCE_MEM:
			parent = &iomem_resource;
			err = quasar_pcie_calc_mem_offset(dev, &range, res, &offset);
			break;
		default:
			err = -EINVAL;
			continue;
		}

		if (err) {
			dev_warn(dev,
				 "error %d: failed to add resource [type 0x%x, %lld bytes]\n",
				 err, restype, range.size);
			continue;
		}

#if DEBUG_PCIE
		dev_warn(pcie->dev, "ranges: restype(0x%08x) val(0x%08x)\n",
				restype, val);
		dev_warn(pcie->dev, "ranges: flags(0x%08x) pci(0x%llx) cpu(0x%llx) size(0x%llx)\n",
				range.flags, range.pci_addr, range.cpu_addr, range.size);
#endif

		/* Window x: memory space, starting from register index 1 */
		writel(val,
			pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_CTRL1_OFF + reg_offset*idx);
		writel((u32)range.cpu_addr,
			pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_AXI_BASE1_OFF + reg_offset*idx);
		writel((u32)((u64)range.cpu_addr>>32),
			pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_AXI_BASE1X_OFF + 4*idx);
		writel((u32)range.pci_addr,
			pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_PEX_BASEL1_OFF + reg_offset*idx);
		writel((u32)((u64)range.pci_addr>>32),
			pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_PEX_BASEH1_OFF + reg_offset*idx);
#ifdef CONFIG_SOC_QUASAR6600
#if IN_LPAE
		writel((COHERENCY << 4) | 0xF, pcie->base_ctrl + 0x0000BAF0 + 4*idx);
#else
		writel((COHERENCY << 4), pcie->base_ctrl + 0x0000BAF0 + 4*idx);
#endif
#endif
#if DEBUG_PCIE
		dev_warn(pcie->dev, "OUT: PCIE1_PAB_AXI_AMAP_CTRL%d     =0x%08X\n",
			(idx+1), readl(pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_CTRL1_OFF + reg_offset*idx));
		dev_warn(pcie->dev, "     PCIE1_PAB_AXI_AMAP_AXI_BASE%d =0x%08X\n",
			(idx+1), readl(pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_AXI_BASE1_OFF + reg_offset*idx));
		dev_warn(pcie->dev, "     PCIE1_PAB_AXI_AMAP_AXI_BASE%dX=0x%08X\n",
			(idx+1), readl(pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_AXI_BASE1X_OFF + 4*idx));
		dev_warn(pcie->dev, "     PCIE1_PAB_AXI_AMAP_PEX_BASEL%d=0x%08X\n",
			(idx+1), readl(pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_PEX_BASEL1_OFF + reg_offset*idx));
		dev_warn(pcie->dev, "     PCIE1_PAB_AXI_AMAP_PEX_BASEH%d=0x%08X\n",
			(idx+1), readl(pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_PEX_BASEH1_OFF + reg_offset*idx));
#ifdef CONFIG_SOC_QUASAR6600
		dev_warn(pcie->dev, "     PCIE1_PAB_AXI_AMAP_SIZE_EAM%dX=0x%08X\n",
			(idx+1), readl(pcie->base_ctrl + 0x0000BAF0 /*PCIE1_PAB_AXI_AMAP_SIZE_EAM0X_OFF*/ + 4*idx));
#endif
#endif

		pci_add_resource_offset(&pcie->resources, res, offset);
		if (idx == 2)
			break;
		idx++;
	}

	return 0;

out_release_res:
	return err;
}

static int quasar_pcie_map_cfg_windows(struct quasar_pcie *pcie)
{
	int err;
	u8 bus_max;
	resource_size_t busn;
	struct resource *bus_range;
	struct device *dev = pcie->host.dev.parent;
	struct device_node *np = dev->of_node;
	u32 val;

	if (of_pci_parse_bus_range(np, &pcie->cfg.bus_range))
		pcie->cfg.bus_range = (struct resource) {
			.name	= np->name,
			.start	= 0,
			.end	= 1,
			.flags	= IORESOURCE_BUS,
		};

	err = of_address_to_resource(np, 0, &pcie->cfg.res);
	if (err) {
		dev_err(dev, "missing \"reg\" property\n");
		return err;
	}

	pcie->cfg.win = devm_kcalloc(dev, resource_size(&pcie->cfg.bus_range),
				    sizeof(*pcie->cfg.win), GFP_KERNEL);
	if (!pcie->cfg.win)
		return -ENOMEM;

	/* Limit the bus-range to fit within reg */
	bus_max = pcie->cfg.bus_range.start +
			(resource_size(&pcie->cfg.res) >> pcie->cfg.ops->bus_shift) - 1;
	pcie->cfg.bus_range.end = min_t(resource_size_t, pcie->cfg.bus_range.end,
			bus_max);

	/* Map our Configuration Space windows */
	if (!devm_request_mem_region(dev, pcie->cfg.res.start,
				     resource_size(&pcie->cfg.res),
				     "Configuration Space"))
		return -ENOMEM;

	bus_range = &pcie->cfg.bus_range;
	for (busn = bus_range->start; busn <= bus_range->end; ++busn) {
		u32 idx = busn - bus_range->start;
		u32 sz = 1 << pcie->cfg.ops->bus_shift;

		pcie->cfg.win[idx] = devm_ioremap(dev,
						 pcie->cfg.res.start + busn * sz,
						 sz);
		if (!pcie->cfg.win[idx])
			return -ENOMEM;
	}

	/* Register bus resource */
	pci_add_resource(&pcie->resources, bus_range);

	/* Program window 0 for configuration mapping */
	val = bar_like_size((u32)resource_size(&pcie->cfg.res)) | 1;
	writel(val, pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_CTRL0_OFF);
	writel((u32)pcie->cfg.res.start,
			pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_AXI_BASE0_OFF);
	writel((u32)((u64)pcie->cfg.res.start>>32),
			pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_AXI_BASE0X_OFF);
	writel(0, pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_PEX_BASEL0_OFF);
	writel(0, pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_PEX_BASEH0_OFF);
#ifdef CONFIG_SOC_QUASAR6600
#if IN_LPAE
	writel((COHERENCY << 4) | 0xF, pcie->base_ctrl + 0x0000BAF0);
#else
	writel((COHERENCY << 4), pcie->base_ctrl + 0x0000BAF0);
#endif
#endif
#if DEBUG_PCIE
	dev_warn(pcie->dev, "CFG: PCIE1_PAB_AXI_AMAP_CTRL0     =0x%08X\n",
		readl(pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_CTRL0_OFF));
	dev_warn(pcie->dev, "     PCIE1_PAB_AXI_AMAP_AXI_BASE0 =0x%08X\n",
		readl(pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_AXI_BASE0_OFF));
	dev_warn(pcie->dev, "     PCIE1_PAB_AXI_AMAP_AXI_BASE0X=0x%08X\n",
		readl(pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_AXI_BASE0X_OFF));
	dev_warn(pcie->dev, "     PCIE1_PAB_AXI_AMAP_PEX_BASEL0=0x%08X\n",
		readl(pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_PEX_BASEL0_OFF));
	dev_warn(pcie->dev, "     PCIE1_PAB_AXI_AMAP_PEX_BASEH0=0x%08X\n",
		readl(pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_PEX_BASEH0_OFF));
#ifdef CONFIG_SOC_QUASAR6600
	dev_warn(pcie->dev, "     PCIE1_PAB_AXI_AMAP_SIZE_EAM0X=0x%08X\n",
		readl(pcie->base_ctrl + 0x0000BAF0 /*PCIE1_PAB_AXI_AMAP_SIZE_EAM0X_OFF*/));
#endif
#endif
	/* Enable AXI PIO access */
	val = (PCIE1_PAB_AXI_PIO_CTRL0__PIO_ENABLE__MASK |
		PCIE1_PAB_AXI_PIO_CTRL0__MEM_WINDOW_ENABLE__MASK |
		PCIE1_PAB_AXI_PIO_CTRL0__IO_WINDOW_ENABLE__MASK |
		PCIE1_PAB_AXI_PIO_CTRL0__CFG_WINDOW_ENABLE__MASK);
	writel(val, pcie->base_ctrl + PCIE1_PAB_AXI_PIO_CTRL0_OFF);

	return 0;
}

static void quasar_pcie_power_down(struct quasar_pcie *pcie)
{
	u32 val, timeout;
	u32 mask;

	/* PCI block power down sequence: RSTs -> CLKs -> ISO -> PWR
	   reset block logic */
	val = readl(pcie->base_rstgen + RSTGEN_SWRSTSTATIC4_OFF);
	val |= (RSTGEN_SWRSTSTATIC4__PCI_PCIEWRAP__MASK | 
			RSTGEN_SWRSTSTATIC4__PCI_PCIE1__MASK | 
#ifndef CONFIG_SOC_QUASAR6300
			RSTGEN_SWRSTSTATIC4__PCI_PCIE2__MASK | 
#endif
			RSTGEN_SWRSTSTATIC4__PCI__MASK);
	writel(val, pcie->base_rstgen + RSTGEN_SWRSTSTATIC4_OFF);
	/* assert PCI_SW_MEDIUM reset */
	val = readl(pcie->base_rstgen + RSTGEN_SWRSTSTATIC11_OFF);
	val |= RSTGEN_SWRSTSTATIC11__PCI_SW_MEDIUM__MASK;
	writel(val, pcie->base_rstgen + RSTGEN_SWRSTSTATIC11_OFF);
	/* disable source clocks */
	val = readl(pcie->base_clkgen + SYSCG_CLKDISCTRL1_OFF);
	val |= SYSCG_CLKDISCTRL1__PCI__MASK;
	writel(val, pcie->base_clkgen + SYSCG_CLKDISCTRL1_OFF);
	timeout = WAIT_TIME;
	while (!(readl(pcie->base_clkgen + SYSCG_CLKDISSTAT1_OFF) &
			SYSCG_CLKDISSTAT1__PCI__MASK)) {
		PWR_WAIT_TIMEOUT_LOOP
	}
	/* disable IB clocks */
	val = readl(pcie->base_clkgen + SYSCG_CLKDISCTRL2_OFF);
	val |= (SYSCG_CLKDISCTRL2__IB29__MASK |
			SYSCG_CLKDISCTRL2__IB30__MASK);
	writel(val, pcie->base_clkgen + SYSCG_CLKDISCTRL2_OFF);
	timeout = WAIT_TIME;
	mask = (SYSCG_CLKDISSTAT2__IB29__MASK |
			SYSCG_CLKDISSTAT2__IB30__MASK);
	while ((readl(pcie->base_clkgen + SYSCG_CLKDISSTAT2_OFF) &
			mask) != mask) {
		PWR_WAIT_TIMEOUT_LOOP
	}
	val = readl(pcie->base_clkgen + SYSCG_CLKDISCTRL3_OFF);
	val |= SYSCG_CLKDISCTRL3__IB36__MASK;
	writel(val, pcie->base_clkgen + SYSCG_CLKDISCTRL3_OFF);
	timeout = WAIT_TIME;
	mask = SYSCG_CLKDISSTAT3__IB36__MASK;
	while ((readl(pcie->base_clkgen + SYSCG_CLKDISSTAT3_OFF) &
			mask) != mask) {
		PWR_WAIT_TIMEOUT_LOOP
	}
	/* isolate PCI */
	val = readl(pcie->base_rstgen + RSTGEN_PWR_ISOLATE_OFF);
	val &= RSTGEN_PWR_ISOLATE__PCI_N__INV_MASK;
	writel(val, pcie->base_rstgen + RSTGEN_PWR_ISOLATE_OFF);
	udelay(5);
	/* powerdown PCI */
	writel(0, pcie->base_rstgen + RSTGEN_PWR_ON_PCI_OFF);
}

static void quasar_pcie_power_on(struct quasar_pcie *pcie)
{
	u32 val;

	val = readl(pcie->base_rstgen + RSTGEN_PWR_ON_PCI_OFF);
	val |= PWR_ON_W1;
	writel(val, pcie->base_rstgen + RSTGEN_PWR_ON_PCI_OFF);
	udelay(5);

	val = readl(pcie->base_rstgen + RSTGEN_PWR_ON_PCI_OFF);
	val |= PWR_ON_W2;
	writel(val, pcie->base_rstgen + RSTGEN_PWR_ON_PCI_OFF);
	udelay(5);

	val = readl(pcie->base_rstgen + RSTGEN_PWR_ON_PCI_OFF);
	val |= PWR_ON_S1;
	writel(val, pcie->base_rstgen + RSTGEN_PWR_ON_PCI_OFF);
	udelay(5);
}

static void quasar_pcie_power_up(struct quasar_pcie *pcie)
{
	u32 val, timeout;
	u32 mask;

	mask = (PWR_ON_W1 | PWR_ON_W2 | PWR_ON_S1);
	val = readl(pcie->base_rstgen + RSTGEN_PWR_ON_PCI_OFF);
	if ((val & mask) == mask)
		return;

	quasar_pcie_power_on(pcie);

	val = readl(pcie->base_clkgen + SYSCG_CLKDISCTRL1_OFF);
	val &= SYSCG_CLKDISCTRL1__PCI__INV_MASK;
	writel(val, pcie->base_clkgen + SYSCG_CLKDISCTRL1_OFF);
	timeout = WAIT_TIME;
	while (readl(pcie->base_clkgen + SYSCG_CLKDISSTAT1_OFF)
		& SYSCG_CLKDISSTAT1__PCI__MASK) {
			PWR_WAIT_TIMEOUT_LOOP
	}

	/* Enable IB clocks with common power domains
	   DDR_SW - 29,30
	   IPM1 - 36 */
	mask = (PWR_ON_W1 | PWR_ON_W2 | PWR_ON_S1);
	val = readl(pcie->base_rstgen + RSTGEN_PWR_ON_DDR_OFF);
	if ((val & mask) == mask) {
		mask = (SYSCG_CLKDISCTRL2__IB29__INV_MASK &
				SYSCG_CLKDISCTRL2__IB30__INV_MASK);
		val = readl(pcie->base_clkgen + SYSCG_CLKDISCTRL2_OFF);
		writel(val & mask, pcie->base_clkgen + SYSCG_CLKDISCTRL2_OFF);
		timeout = WAIT_TIME;
		mask = (SYSCG_CLKDISSTAT2__IB29__MASK | SYSCG_CLKDISSTAT2__IB30__MASK);
		while (readl(pcie->base_clkgen + SYSCG_CLKDISSTAT2_OFF) & mask) {
			PWR_WAIT_TIMEOUT_LOOP
		}
	}

	mask = (PWR_ON_W1 | PWR_ON_W2 | PWR_ON_S1);
	val = readl(pcie->base_rstgen + RSTGEN_PWR_ON_IPM1_OFF);
	if ((val & mask) == mask) {
		val = readl(pcie->base_clkgen + SYSCG_CLKDISCTRL3_OFF);
		val &= SYSCG_CLKDISCTRL3__IB36__INV_MASK;
		writel(val, pcie->base_clkgen + SYSCG_CLKDISCTRL3_OFF);
		timeout = WAIT_TIME;
		while ((readl(pcie->base_clkgen + SYSCG_CLKDISSTAT3_OFF) &
				SYSCG_CLKDISSTAT3__IB36__MASK) != 0) {
			PWR_WAIT_TIMEOUT_LOOP
		}
	}

	/* Release PCI_SW_MEDIUM reset */
	val = readl(pcie->base_rstgen + RSTGEN_SWRSTSTATIC11_OFF);
	val &= RSTGEN_SWRSTSTATIC11__PCI_SW_MEDIUM__INV_MASK;
	writel(val, pcie->base_rstgen + RSTGEN_SWRSTSTATIC11_OFF);
	/* Change RSTGEN_PWR_ISOLATE.BLOCKNAME_N to 1 to connect the block */
	val = readl(pcie->base_rstgen + RSTGEN_PWR_ISOLATE_OFF);
	val |= RSTGEN_PWR_ISOLATE__PCI_N__MASK;
	writel(val, pcie->base_rstgen + RSTGEN_PWR_ISOLATE_OFF);
	/* Deassert the block reset
	   Release all PCI block reset */
	val = readl(pcie->base_rstgen + RSTGEN_SWRSTSTATIC4_OFF);
	val &= RSTGEN_SWRSTSTATIC4__PCI__INV_MASK;
	writel(val, pcie->base_rstgen + RSTGEN_SWRSTSTATIC4_OFF);
	val = readl(pcie->base_rstgen + RSTGEN_SWRSTSTATIC4_OFF);
	val &= (RSTGEN_SWRSTSTATIC4__PCI__INV_MASK &
			RSTGEN_SWRSTSTATIC4__PCI_PCIE1__INV_MASK &
#ifndef CONFIG_SOC_QUASAR6300
			RSTGEN_SWRSTSTATIC4__PCI_PCIE2__INV_MASK & 
#endif
			RSTGEN_SWRSTSTATIC4__PCI_PCIEWRAP__INV_MASK);
	writel(val, pcie->base_rstgen + RSTGEN_SWRSTSTATIC4_OFF);
}

/*static*/
void quasar_tx_deemp_manual(struct quasar_pcie *pcie, u8 deemp)
{
	u32 val = 0;

	if (deemp > 2) {
		printk("wrong deemp %d\n", deemp);
	} else {
		val = readl(pcie->base_ctrl + PCIE1_GPEXD_GEN2_CTRL_OFF);
		val &= 0xfffffff3;   /* [3:2] */
		val |= ((deemp & 1) << 2);   /* bit 2 = 0 for -6dB, 1 for -3,5dB */
		val |= (1 << 3);   /* bit 3 = 1 to forcedly using bit 2 for de-emphasis */
		writel(val, pcie->base_ctrl + PCIE1_GPEXD_GEN2_CTRL_OFF);
	}
}

/*static*/
void quasar_tx_margin_manual(struct quasar_pcie *pcie, u8 margin)
{
	u32 val = 0;

	if (margin > 7) {
		printk("wrong margin %d\n", margin);
	} else {
		val = readl(pcie->base_ctrl + PCIE1_GPEXP_CFG_LINKCTRL2_OFF);
		val &= 0xfffffc7f;   /* [9:7] */
		val |= (margin << 7);
		writel(val, pcie->base_ctrl + PCIE1_GPEXP_CFG_LINKCTRL2_OFF);
	}
}

static void quasar_pcie_hw_init(struct quasar_pcie *pcie)
{
	u32 val;
	void __iomem	*tmp;

	if (pcie->ctrl_no == QUASAR_PCIE1) {   /* only do once */
		tmp = ioremap(RSTGEN_SWRSTSTATIC4, 4);
		val = readl(tmp);
		val &= (RSTGEN_SWRSTSTATIC4__PCI_PCIEWRAP__INV_MASK &
			RSTGEN_SWRSTSTATIC4__PCI_PCIE1__INV_MASK &
#ifndef CONFIG_SOC_QUASAR6300
			RSTGEN_SWRSTSTATIC4__PCI_PCIE2__INV_MASK & 
#endif
			RSTGEN_SWRSTSTATIC4__PCI__INV_MASK);
		writel(val, tmp);
		iounmap(tmp);
	}

#ifdef CONFIG_SOC_QUASAR6300
	/* If syspll is 2400MHz */
	//pcie->src_pci_clk_mhz = 240;
	/* If syspll is 2000MHz */
	pcie->src_pci_clk_mhz = 250;
#else   /* QB66xx */
	pcie->src_pci_clk_mhz = 300;
#endif
	udelay(100);

	if (pcie->ctrl_no == QUASAR_PCIE1) {   /* only do once */
		val  = readl(pcie->clkrx_ctrl1);
		val &= 0xfffffffe;   // enable
		writel(val, pcie->clkrx_ctrl1);

		/* Wait for CLKRX output to be stable before
		   using that output as ref_clk for PHY.
		   Put a simple wait loop here as a placeholder */
		udelay(100);

		/* Tell PHY to use ref_clk from pad before you take it out of reset */
		val  = readl(pcie->phy_clk_ctrl);
		val |= PCIE_PHY_CLK_CTRL__REF_USE_PAD__MASK;
		writel(val, pcie->phy_clk_ctrl);
	}

	/* Reset PHY and controller */
	val  = readl(pcie->phy_rst_ctrl);
#ifdef CONFIG_SOC_QUASAR6300
	val &= PCIE_PHY_RST_CTRL__RESET_N__INV_MASK;
#else   // QB66xx
	if (pcie->ctrl_no == QUASAR_PCIE1)
		val &= PCIE_PHY_RST_CTRL__PIPE0_RESETN__INV_MASK;
	else
		val &= PCIE_PHY_RST_CTRL__PIPE1_RESETN__INV_MASK;
#endif
	writel(val, pcie->phy_rst_ctrl);

#ifdef CONFIG_SOC_QUASAR6300
	/* Configure GPIO13 to PERSTN */
	tmp = ioremap(SYS_OVLCTL5, 4);
	val = readl(tmp);
	val &= 0xff0fffff;
	val |= 0x00200000;
	writel(val, tmp);
	iounmap(tmp);
#else   // LCD14 and LCD13 for QB66xx
	if (pcie->ctrl_no == QUASAR_PCIE1) {   /* only do once */
		tmp = ioremap(OVLCTL31, 4);
		val = readl(tmp);
		if (pcie->ctrl_no == QUASAR_PCIE1) {
			val &= OVLCTL31__LCD14__INV_MASK;   // link 1
			val |= (4 << OVLCTL31__LCD14__SHIFT);
		} else {
			val &= OVLCTL31__LCD13__INV_MASK;   // link 2
			val |= (4 << OVLCTL31__LCD13__SHIFT);
		}
		writel(val, tmp);
	}
#endif

	val = readl(pcie->sw_rst);
	val &= (PCIE1_SW_RST__PAB_N__INV_MASK &
			PCIE1_SW_RST__AMBA_N__INV_MASK &
			PCIE1_SW_RST__PBUS_N__INV_MASK &
			PCIE1_SW_RST__LINK_N__INV_MASK);
	writel(val, pcie->sw_rst);
	udelay(5);

	/* Set role as rc or ep */
#ifdef CONFIG_SOC_QUASAR6300
	val  = readl(pcie->sw_bootstrap);
	val |= PCIE_SW_BOOTSTRAP__PCIE1_EP_RC_SEL__MASK;
	writel(val, pcie->sw_bootstrap);
#else   // QB66xx depending on the board
	if (pcie->ctrl_no == QUASAR_PCIE1) {   /* only do once */
		val  = readl(pcie->sw_bootstrap);
		val &= (PCIE_SW_BOOTSTRAP__PCIE2_EP_RC_SEL__INV_MASK &
				PCIE_SW_BOOTSTRAP__PCIE1_EP_RC_SEL__INV_MASK &
				PCIE_SW_BOOTSTRAP__PCIEPHY_PIPE_PORT_SEL__INV_MASK &
				PCIE_SW_BOOTSTRAP__PCIEPHY_LANE_SYNC4__INV_MASK);   // clear all fields first
	#ifdef EVM_TWO_RC   // TODO: depending on board types
		val |= (PCIE_SW_BOOTSTRAP__PCIE1_EP_RC_SEL__MASK |
				PCIE_SW_BOOTSTRAP__PCIE2_EP_RC_SEL__MASK);   // RC
		val |= PCIE_SW_BOOTSTRAP__PCIEPHY_PIPE_PORT_SEL__MASK;   // 2 links
		val |= (2 << PCIE_SW_BOOTSTRAP__PCIEPHY_LANE_SYNC4__SHIFT);   // link1 x2 & link2 x2
		//val |= (1 << PCIE_SW_BOOTSTRAP__PCIEPHY_LANE_SYNC4__SHIFT);   // link1 x2 & link2 x1
	#else
		val |= PCIE_SW_BOOTSTRAP__PCIE1_EP_RC_SEL__MASK;   // RC
		val &= PCIE_SW_BOOTSTRAP__PCIEPHY_PIPE_PORT_SEL__INV_MASK;
		val |= (3 << PCIE_SW_BOOTSTRAP__PCIEPHY_LANE_SYNC4__SHIFT);   // link1 x4
		//val |= (2 << PCIE_SW_BOOTSTRAP__PCIEPHY_LANE_SYNC4__SHIFT);   // link1 x2 & link2 x2
		//val |= (1 << PCIE_SW_BOOTSTRAP__PCIEPHY_LANE_SYNC4__SHIFT);   // link1 x2 & link2 x1
		//val |= (0 << PCIE_SW_BOOTSTRAP__PCIEPHY_LANE_SYNC4__SHIFT);   // link1 x1
	#endif
		writel(val, pcie->sw_bootstrap);
	}
#endif

	/* Release TEST_RESET_N */
#ifdef CONFIG_SOC_QUASAR6300
	val  = readl(pcie->phy_test_ctrl);
	val |= PCIE_PHY_TEST_CTRL__TEST_RESET_N__MASK;
	writel(val, pcie->phy_test_ctrl);
#else   // QB66xx
	if (pcie->ctrl_no == QUASAR_PCIE1) {   /* only do once */
		val = readl(pcie->phy_test_ctrl0);
		val |= PCIE_PHY_TEST_CTRL0__TEST_RESET_N_0__MASK;
		writel(val, pcie->phy_test_ctrl0);
		val = readl(pcie->phy_test_ctrl1);
		val |= PCIE_PHY_TEST_CTRL1__TEST_RESET_N_1__MASK;
		writel(val, pcie->phy_test_ctrl1);
		val = readl(pcie->phy_test_ctrl2);
		val |= PCIE_PHY_TEST_CTRL2__TEST_RESET_N_2__MASK;
		writel(val, pcie->phy_test_ctrl2);
		val = readl(pcie->phy_test_ctrl3);
		val |= PCIE_PHY_TEST_CTRL3__TEST_RESET_N_3__MASK;
		writel(val, pcie->phy_test_ctrl3);
	}
#endif
	udelay(5);

	/* Release PHY reset */
	val  = readl(pcie->phy_rst_ctrl);
#ifdef CONFIG_SOC_QUASAR6300
	val |= PCIE_PHY_RST_CTRL__RESET_N__MASK;
#else   // QB66xx
	if (pcie->ctrl_no == QUASAR_PCIE1)
		val |= PCIE_PHY_RST_CTRL__PIPE0_RESETN__MASK;
	else
		val |= PCIE_PHY_RST_CTRL__PIPE1_RESETN__MASK;
#endif
	writel(val, pcie->phy_rst_ctrl);

#ifdef CONFIG_SOC_QUASAR6300
	/* Wait pipe is ready */
	val = PCIE_PHY_PIPE_STAT__PHYSTATUS__MASK;
	while (val != 0) {
		val  = readl(pcie->phy_pipe_stat);
		val &= PCIE_PHY_PIPE_STAT__PHYSTATUS__MASK;
	}
#else   // QB66xx
	if (pcie->ctrl_no == QUASAR_PCIE1) {
		val = PCIE_PHY_PIPE_STAT__PIPE0_PHYSTATUS__MASK;
		while (val) {
			val = readl(pcie->phy_pipe_stat);
			val &= PCIE_PHY_PIPE_STAT__PIPE0_PHYSTATUS__MASK;
		}
	} else {
		val = PCIE_PHY_PIPE_STAT__PIPE1_PHYSTATUS__MASK;
		while (val) {
			val = readl(pcie->phy_pipe_stat);
			val &= PCIE_PHY_PIPE_STAT__PIPE1_PHYSTATUS__MASK;
		}
	}
#endif

	/* Release controller reset */
	val = readl(pcie->sw_rst);
	val |= (PCIE1_SW_RST__PAB_N__MASK |
			PCIE1_SW_RST__AMBA_N__MASK |
			PCIE1_SW_RST__PBUS_N__MASK |
			PCIE1_SW_RST__LINK_N__MASK);
	writel(val, pcie->sw_rst);

#if SPEED_GEN1   /* Fix speed to Gen 1 if needed */
	val = readl(pcie->base_ctrl + PCIE1_GPEXD_GEN2_CTRL_OFF);
	val &= 0xffffff0f;
	val |= 0x10;   /* 0x10 for Gen 1 only, 0x20 (default) for Gen 2 */
	writel(val, pcie->base_ctrl + PCIE1_GPEXD_GEN2_CTRL_OFF);
	val = readl(pcie->base_ctrl + PCIE1_GPEXD_GEN2_CTRL_OFF);
	dev_warn(pcie->dev, "Quasar PCIe is configured as Gen %d\n", (val & 0xf0) >> 4);
#else
	//quasar_tx_deemp_manual(pcie, 0);   /* program de-emphasis if needed */
#endif

	/* Set the ratio of GPEX core clk to be the expected 300 MHz */
	/* GPEXD_CORE_CLK_RATIO = 16 * src_pci_clk/300MHz rounded to nearest integer */
	val = pcie->src_pci_clk_mhz/(300*1000*1000/16);
	writel(val, pcie->base_ctrl + PCIE1_GPEXD_CORE_CLK_RATIO_OFF);

	mdelay(300);
}

static void quasar_pcie_reset_device(struct quasar_pcie *pcie, u8 reset)
{
	u32 val;

	if (reset) {
		val = readl(pcie->sw_rst);
		val &= PCIE1_SW_RST__PERST_N_PIN__INV_MASK;
		writel(val, pcie->sw_rst);
	} else {
		val = readl(pcie->sw_rst);
		val |= PCIE1_SW_RST__PERST_N_PIN__MASK;
		writel(val, pcie->sw_rst);
	}
}

static void quasar_pcie_rc_init(struct quasar_pcie *pcie)
{
	u32 val;

	/* Config RC registers */
	val = (PCIE1_PAB_CTRL__ENABLE_AMBA_PIO__MASK |
			PCIE1_PAB_CTRL__ENABLE_PEX_PIO__MASK |
			PCIE1_PAB_CTRL__ENABLE_WDMA__MASK |
			PCIE1_PAB_CTRL__ENABLE_RDMA__MASK |
#ifdef CONFIG_SOC_QUASAR6300   /* QB63xx only supports 4 bursts */
			PCIE1_PAB_CTRL__AXI_MAX_BURST_4__VALUE |  
#else
			PCIE1_PAB_CTRL__AXI_MAX_BURST_16__VALUE |
#endif
			PCIE1_PAB_CTRL__MAX_DMA_128__VALUE /*PCIE1_PAB_CTRL__MAX_DMA_512__VALUE*/); 
	writel(val, pcie->base_ctrl + PCIE1_PAB_CTRL_OFF);

	/* EP MRRS    - PCIE1_PAB_CTRL[12:10], set above... */
	/*              PCIE1_GPEXP_CFG_DEVCTRL[14:12] */
	/* EP/RC MPS  - PCIE1_GPEXP_CFG_DEVCTRL[7:5] */
	val = readl(pcie->base_ctrl + PCIE1_GPEXP_CFG_DEVCTRL_OFF);
	val &= 0xffff8f1f;
	val |= (0 << 5);   // MPS must be the same for RC and EP: 0-128, 1-256, maybe depending on the device.
	writel(val, pcie->base_ctrl + PCIE1_GPEXP_CFG_DEVCTRL_OFF);

	/* PCIE1_MISC_CTRL.CSR_CONV_DIS must be 0 for 63XX */

	val = readl(pcie->base_ctrl + 0x474);   // PCIE1_GPEXD_CLASSCODE
	val &= 0x000000ff;
	val |= 0x06040000;   // set calss as PCI bridge
	writel(val, pcie->base_ctrl + 0x474);

	writel(0x00010100, pcie->base_ctrl + PCIE1_GPEXP_CFG_BASE2_PRIBUS_OFF);

#ifndef CONFIG_SOC_QUASAR6300   /* QB66xx PCIe host must set PIOS_CONV=1 */
	val = readl(pcie->misc_ctrl);
	val &= PCIE1_MISC_CTRL__PIOS_CONV_SEL__INV_MASK;
	val |= (0x1 << PCIE1_MISC_CTRL__PIOS_CONV_SEL__SHIFT);   // REQUIRED FOR IO/CFG access on PIO slave port
#if COHERENCY
	val |= PCIE1_MISC_CTRL__PIO_AXI_AR_COH_SEL__MASK;   // PIO AXI read coherency
	val |= PCIE1_MISC_CTRL__PIO_AXI_AW_COH_SEL__MASK;   // PIO AXI write coherency
#endif
	writel(val, pcie->misc_ctrl);
#endif

	/* Set config ready */
	writel(PCIE1_GPEXD_CFG_RDY__CFG_RDY__MASK,
			pcie->base_ctrl + PCIE1_GPEXD_CFG_RDY_OFF);

	/* Set command register */
	val = readl(pcie->base_ctrl + PCIE1_GPEXP_CFG_COMMAND_OFF);
	val |= (PCIE1_CFG_COMMAND_STATUS__BUS_MASTER__VALUE |
			PCIE1_CFG_COMMAND_STATUS__MEMORY_SPACE__VALUE |
			PCIE1_CFG_COMMAND_STATUS__IO_SPACE__VALUE);
	writel(val, pcie->base_ctrl + PCIE1_GPEXP_CFG_COMMAND_OFF);
}

static int quasar_pcie_wait_link_up(struct quasar_pcie *pcie)
{
	u32 val = 0, i = 0;

	while (PCIE1_MISC_STAT__GDA_PAB_DL_UP__MASK != val) {
		dev_warn(pcie->dev, "Wait for PCIe link...\n");
		val = readl(pcie->misc_stat);
		val &= PCIE1_MISC_STAT__GDA_PAB_DL_UP__MASK;
        msleep(1000);
		if (i++ > 3)
			return -ENODEV;
	}
	val = readl(pcie->base_ctrl + PCIE1_GPEXP_CFG_LINKCTRL_OFF);
	if (((val & 0x000f0000) >> 16) == 1)
		dev_warn(pcie->dev, "PCIe link speed is Gen 1\n");
	else if (((val & 0x000f0000) >> 16) == 2)
		dev_warn(pcie->dev, "PCIe link speed is Gen 2\n");
	else {
		dev_warn(pcie->dev, "PCIe link speed is unknown 0x%02x\n",
			(val & 0x000f0000) >> 16);
		return -ENODEV;
	}

	return 0;
}

static irqreturn_t quasar_pcie_interrupt(int irq, void *arg)
{
	struct quasar_pcie *pcie = arg;
	u32 val, fifo1, fifo2, fifo3, fifo4, fifo5;

#if DEBUG_PCIE
	dev_warn(pcie->dev, "h_int\n");
#endif

	/* PCIE1_PAB_PEX_INT_STAT
		[ 0] => AMBA PIO completed
		[ 1] => AMBA PIO aborted
		[ 2] => PEX PIO completed
		[ 3] => PEX PIO aborted
		[ 4] => WDMA last descriptor in a chain completed
		[ 5] => WDMA for a descriptor completed
		[ 6] => WDMA aborted
		[ 7] => RDMA last descriptor in a chain completed
		[ 8] => RDMA for a descriptor completed
		[ 9] => RDMA aborted
		[10] => AMBA-to-PCIE mailbox is ready to send
		[11] => AMBA bus reset
	*/
	val = readl(pcie->base_ctrl + PCIE1_PAB_PEX_INT_STAT_OFF);
	if (val) {
#if DEBUG_PCIE
		dev_warn(pcie->dev,	"PAB_PEX_INT_STAT=0x%08x\n", val);
#endif
		writel(val, pcie->base_ctrl + PCIE1_PAB_PEX_INT_STAT_OFF);
	}

	val = readl(pcie->base_ctrl + PCIE1_PAB_AXI_INT_WDMA_STAT_OFF);
	if (val) {
#if DEBUG_PCIE
		dev_warn(pcie->dev,	"PAB_AXI_INT_WDMA_STAT=0x%08x\n", val);
#endif
		writel(val, pcie->base_ctrl + PCIE1_PAB_AXI_INT_WDMA_STAT_OFF);
	}

	val = readl(pcie->base_ctrl + PCIE1_PAB_AXI_INT_RDMA_STAT_OFF);
	if (val) {
#if DEBUG_PCIE
		dev_warn(pcie->dev,	"PAB_AXI_INT_RDMA_STAT=0x%08x\n", val);
#endif
		writel(val, pcie->base_ctrl + PCIE1_PAB_AXI_INT_RDMA_STAT_OFF);
	}

	val = readl(pcie->base_ctrl + PCIE1_PAB_AXI_INT_PIO_STAT_OFF);
	if (val) {
#if DEBUG_PCIE
		dev_warn(pcie->dev,	"PAB_AXI_INT_PIO_STAT=0x%08x\n", val);
#endif
		writel(val, pcie->base_ctrl + PCIE1_PAB_AXI_INT_PIO_STAT_OFF);
	}

	val = readl(pcie->base_ctrl + PCIE1_PAB_AXI_INT_MISC_STAT_OFF);
	if (val) {
#if DEBUG_PCIE
		dev_warn(pcie->dev,	"PAB_AXI_INT_MISC_STAT=0x%08x\n", val);
#endif
		writel(val, pcie->base_ctrl + PCIE1_PAB_AXI_INT_MISC_STAT_OFF);
	}

	if (val & 0x08) {   /* MSI */
		val = readl(pcie->base_ctrl + PCIE1_PAB_MSI_IB_STAT_OFF);
		if (val & 0x01) {
#if DEBUG_PCIE
			dev_warn(pcie->dev,	"MSI status 0x%08X:\n", val);
#endif
			// MSI available
			fifo1 = readl(pcie->base_ctrl + PCIE1_PAB_MSI_IB_FIFO_DW1_OFF);
			fifo2 = readl(pcie->base_ctrl + PCIE1_PAB_MSI_IB_FIFO_DW2_OFF);
			fifo3 = readl(pcie->base_ctrl + PCIE1_PAB_MSI_IB_FIFO_DW3_OFF);
			fifo4 = readl(pcie->base_ctrl + PCIE1_PAB_MSI_IB_FIFO_DW4_OFF);
			fifo5 = readl(pcie->base_ctrl + PCIE1_PAB_MSI_IB_FIFO_DW5_OFF);
#if DEBUG_PCIE
			dev_warn(pcie->dev,	"0x%08X, 0x%08X, 0x%08X, 0x%08X, 0x%08X\n",
				fifo1, fifo2, fifo3, fifo4, fifo5);
#endif
		}
	} else {   /* INTx */
#if DEBUG_PCIE
		switch (val & 0x1e0) {
		case 0x20:
			dev_warn(pcie->dev,	"INTA\n");
			break;
		case 0x40:
			dev_warn(pcie->dev,	"INTB\n");
			break;
		case 0x80:
			dev_warn(pcie->dev,	"INTC\n");
			break;
		case 0x100:
			dev_warn(pcie->dev,	"INTD\n");
			break;
		default:
			break;
		}
#endif
	}

	val = readl(pcie->int_flag);
	if (val) {
#if DEBUG_PCIE
		dev_warn(pcie->dev,	"INT_CLR=0x%08x\n", val);
#endif
		writel(val, pcie->int_clr);
	}

#if DEBUG_PCIE
	dev_warn(pcie->dev,	"h_int(end)\n");
#endif

	return IRQ_HANDLED;
}

static void quasar_pcie_init_interrupt(struct quasar_pcie *pcie)
{
	/* Don't enable PERST# interrupt */
	writel(PCIE1_INT_EN__GDA_PAB__MASK,
		pcie->int_en);

	writel((
#if defined(CONFIG_SOC_QUASAR6300) && defined(CONFIG_PCI_MSI)
			PCIE1_PAB_PEX_INT_STAT__MAILBOX_READY__MASK |
#endif
			PCIE1_PAB_PEX_INT_STAT__WDMA_COMPLETE__MASK |
			PCIE1_PAB_PEX_INT_STAT__RDMA_COMPLETE__MASK),
		pcie->base_ctrl + PCIE1_PAB_PEX_INT_EN_OFF);

	writel(PCIE1_PAB_AXI_INT_WDMA_STAT__WDMA_COMPLETE__MASK,
		pcie->base_ctrl + PCIE1_PAB_AXI_INT_WDMA_EN_OFF);
	writel(PCIE1_PAB_AXI_INT_RDMA_STAT__RDMA_COMPLETE__MASK,
		pcie->base_ctrl + PCIE1_PAB_AXI_INT_RDMA_EN_OFF);

#if defined(CONFIG_SOC_QUASAR6300) && defined(CONFIG_PCI_MSI)
	writel((PCIE1_PAB_AXI_INT_MISC_EN__MAILBOX__MASK |
			PCIE1_PAB_AXI_INT_MISC_EN__MSI__MASK /*|
			PCIE1_PAB_AXI_INT_MISC_EN__PEX_INTA__MASK*/),
		pcie->base_ctrl + PCIE1_PAB_AXI_INT_MISC_EN_OFF);
#else   /* use INTx */
	writel(0x1e0, pcie->base_ctrl + PCIE1_PAB_AXI_INT_MISC_EN_OFF);
#endif
	writel(0x3fffffff, pcie->base_ctrl + PCIE1_PAB_AXI_INT_MISC_STAT_OFF);
}

static int __init quasar_pcie_probe(struct platform_device *pdev)
{
	int err = 0;
	struct device *dev = &pdev->dev;
	struct quasar_pcie *pcie;
	struct pci_bus *bus;
	struct pci_bus *child;
	struct pci_host_bridge *host;
#if DEVICE_IS_QUASAR
	struct pci_dev *p_dev;
#endif
	resource_size_t iobase = 0;
	const struct of_device_id *of_id;
	struct device_node *np = dev->of_node;
	LIST_HEAD(res);

	host = devm_pci_alloc_host_bridge(dev, sizeof(*pcie));
	if (!host)
		return -ENODEV;

	pcie = pci_host_bridge_priv(host);

	pcie->dev = dev;
	platform_set_drvdata(pdev, pcie);

	err = devm_of_pci_get_host_bridge_resources(dev, 0, 0xff, &res, &iobase);
	if (err) {
		dev_err(dev, "Getting host resources failed\n");
		return err;
	}

	err = devm_request_pci_bus_resources(dev, &res);
	if (err) {
		dev_err(dev, "Getting bus resources failed\n");
		goto error;
	}

	pcie->host.dev.parent = dev;
	INIT_LIST_HEAD(&pcie->host.windows);
	INIT_LIST_HEAD(&pcie->resources);

	pcie->rsc_rstgen = platform_get_resource(pdev, IORESOURCE_MEM, 1);
	if (!pcie->rsc_rstgen) {
		dev_err(dev, "No rsc_rstgen base resource\n");
		return -EINVAL;
	}
	pcie->base_rstgen = devm_ioremap_resource(dev, pcie->rsc_rstgen);

	pcie->rsc_clkgen = platform_get_resource(pdev, IORESOURCE_MEM, 2);
	if (!pcie->rsc_clkgen) {
		dev_err(dev, "No rsc_clkgen base resource\n");
		return -EINVAL;
	}
	pcie->base_clkgen = devm_ioremap_resource(dev, pcie->rsc_clkgen);

	pcie->rsc_ctrl = platform_get_resource(pdev, IORESOURCE_MEM, 3);
	if (!pcie->rsc_ctrl) {
		dev_err(dev, "No rsc_ctrl base resource\n");
		return -EINVAL;
	}
	pcie->ctrl_no = 0;
	if (pcie->rsc_ctrl->start == PCIE1_GPEXP_CFG_VENDORID)
		pcie->ctrl_no = QUASAR_PCIE1;
#ifndef CONFIG_SOC_QUASAR6300
	else if (pcie->rsc_ctrl->start == PCIE2_GPEXP_CFG_VENDORID)
		pcie->ctrl_no = QUASAR_PCIE2;
#endif
	else {
		dev_err(dev, "Unknown PCIe controller wrap resource 0x%08x\n",
			(u32)pcie->rsc_ctrl->start);
		return -EINVAL;
	}
	pcie->base_ctrl = devm_ioremap_resource(dev, pcie->rsc_ctrl);

	pcie->rsc_wrap = platform_get_resource(pdev, IORESOURCE_MEM, 4);
	if (!pcie->rsc_wrap) {
		dev_err(dev, "No rsc_wrap base resource\n");
		return -EINVAL;
	}
	pcie->base_wrap = devm_ioremap_resource(dev, pcie->rsc_wrap);
	pcie->sw_bootstrap   = ioremap(pcie->rsc_wrap->start + PCIE_SW_BOOTSTRAP_OFF, 4);
	pcie->phy_rst_ctrl   = ioremap(pcie->rsc_wrap->start + PCIE_PHY_RST_CTRL_OFF, 4);
	pcie->phy_clk_ctrl   = ioremap(pcie->rsc_wrap->start + PCIE_PHY_CLK_CTRL_OFF, 4);
	pcie->phy_pipe_stat  = ioremap(pcie->rsc_wrap->start + PCIE_PHY_PIPE_STAT_OFF, 4);
	pcie->clkrx_ctrl1    = ioremap(pcie->rsc_wrap->start + PCIE_CLKRX_CTRL1_OFF, 4);

	switch (pcie->ctrl_no) {
	case QUASAR_PCIE1:
		pcie->sw_rst	 = ioremap(pcie->rsc_wrap->start + PCIE1_SW_RST_OFF, 4);
		pcie->misc_ctrl	 = ioremap(pcie->rsc_wrap->start + PCIE1_MISC_CTRL_OFF, 4);
		pcie->misc_stat	 = ioremap(pcie->rsc_wrap->start + PCIE1_MISC_STAT_OFF, 4);
		pcie->int_flag   = ioremap(pcie->rsc_wrap->start + PCIE1_INT_FLAG_OFF, 4);
		pcie->int_en     = ioremap(pcie->rsc_wrap->start + PCIE1_INT_EN_OFF, 4);
		pcie->int_clr    = ioremap(pcie->rsc_wrap->start + PCIE1_INT_CLR_OFF, 4);
		break;
#ifndef CONFIG_SOC_QUASAR6300
	case QUASAR_PCIE2:
		pcie->sw_rst	 = ioremap(pcie->rsc_wrap->start + PCIE2_SW_RST_OFF, 4);
		pcie->misc_ctrl	 = ioremap(pcie->rsc_wrap->start + PCIE2_MISC_CTRL_OFF, 4);
		pcie->misc_stat	 = ioremap(pcie->rsc_wrap->start + PCIE2_MISC_STAT_OFF, 4);
		pcie->int_flag   = ioremap(pcie->rsc_wrap->start + PCIE2_INT_FLAG_OFF, 4);
		pcie->int_en     = ioremap(pcie->rsc_wrap->start + PCIE2_INT_EN_OFF, 4);
		pcie->int_clr    = ioremap(pcie->rsc_wrap->start + PCIE2_INT_CLR_OFF, 4);
		break;
#endif
	default:
		break;
	}

#ifdef CONFIG_SOC_QUASAR6300
	pcie->phy_test_ctrl  = ioremap(pcie->rsc_wrap->start + PCIE_PHY_TEST_CTRL_OFF, 4);
#else
	pcie->phy_test_ctrl0 = ioremap(pcie->rsc_wrap->start + PCIE_PHY_TEST_CTRL0_OFF, 4);
	pcie->phy_test_ctrl1 = ioremap(pcie->rsc_wrap->start + PCIE_PHY_TEST_CTRL1_OFF, 4);
	pcie->phy_test_ctrl2 = ioremap(pcie->rsc_wrap->start + PCIE_PHY_TEST_CTRL2_OFF, 4);
	pcie->phy_test_ctrl3 = ioremap(pcie->rsc_wrap->start + PCIE_PHY_TEST_CTRL3_OFF, 4);
#endif

	pcie->rsc_gpf = platform_get_resource(pdev, IORESOURCE_MEM, 5);
	if (!pcie->rsc_gpf) {
		dev_err(dev, "No rsc_gpf base resource\n");
		return -EINVAL;
	}
	pcie->base_gpf = devm_ioremap_resource(dev, pcie->rsc_gpf);

	pcie->irq = platform_get_irq(pdev, 0);
	if (!pcie->irq) {
		dev_err(dev, "No irq\n");
		return -EINVAL;
	}
#if DEBUG_PCIE
	dev_warn(dev,	"irq=%d\n", pcie->irq);
#endif

#ifdef EVM_TWO_RC
	if (pcie->ctrl_no == QUASAR_PCIE1)
		of_id = of_match_node(quasar_pcie_of_match1, np);
	else
		of_id = of_match_node(quasar_pcie_of_match2, np);
#else
	of_id = of_match_node(quasar_pcie_of_match1, np);
#endif
	if (!of_id)
		return -EINVAL;
    else
		pcie->cfg.ops = of_id->data;

	/* Initialize PCIe hardware */
	if (pcie->ctrl_no == QUASAR_PCIE1) {   /* only do once */
		quasar_pcie_power_down(pcie);
		quasar_pcie_power_up(pcie);
	}
	quasar_pcie_hw_init(pcie);

	quasar_pcie_init_interrupt(pcie);
#ifdef CONFIG_SOC_QUASAR6300
	err = devm_request_irq(&pdev->dev, pcie->irq, quasar_pcie_interrupt,
			IRQF_SHARED, "qbit,quasar-pcie", pcie);
	if (err) {
		dev_err(pcie->dev, "failed to request PCIe irq\n");
		return err;
	}
#else
#ifdef EVM_TWO_RC
	if (pcie->ctrl_no == QUASAR_PCIE1) {
		err = devm_request_irq(&pdev->dev, pcie->irq, quasar_pcie_interrupt,
				IRQF_SHARED, "qbit,quasar-pcie1", pcie);
		if (err) {
			dev_err(pcie->dev, "failed to request irq for PCIE1\n");
			return err;
		}
	} else {
		err = devm_request_irq(&pdev->dev, pcie->irq, quasar_pcie_interrupt,
				IRQF_SHARED, "qbit,quasar-pcie2", pcie);
		if (err) {
			dev_err(pcie->dev, "failed to request irq for PCIE2\n");
			return err;
		}
	}
#else
	err = devm_request_irq(&pdev->dev, pcie->irq, quasar_pcie_interrupt,
			IRQF_SHARED, "qbit,quasar-pcie1", pcie);
	if (err) {
		dev_err(pcie->dev, "failed to request irq for PCIE1\n");
		return err;
	}
#endif
#endif

	quasar_pcie_reset_device(pcie, 1);
	msleep(100);
	quasar_pcie_reset_device(pcie, 0);

	/* Wait for device link up */
	err = quasar_pcie_wait_link_up(pcie);
	if (err) {
		dev_warn(pcie->dev, "PCIe link up failed, probably no device\n");
		return err;
	}

	/* Map configuration space windows */
	err = quasar_pcie_map_cfg_windows(pcie);
	if (err) {
		return err;
	}

	/* Map outbound ranges */
	err = quasar_pcie_map_ranges(pcie);
	if (err)
		return err;

	/* Map inbound ranges */
	err = quasar_pcie_map_dma_ranges(pcie);
	if (err)
		return err;

	/* Initialize PCIe host controller */
	quasar_pcie_rc_init(pcie);

	list_splice_init(&res, &host->windows);
	host->dev.parent = dev;
	host->sysdata = pcie;
	host->busnr = 0;
	host->ops = &quasar_pcie_ops;
	host->swizzle_irq = pci_common_swizzle;
	err = pci_scan_root_bus_bridge(host);
	if (err)
		goto error;

	bus = host->bus;

	pci_assign_unassigned_bus_resources(bus);
	list_for_each_entry(child, &bus->children, node)
		pcie_bus_configure_settings(child);

#if DEVICE_IS_QUASAR
	/* If PCIe device is QB63xx/QB66xx, specify host interrupt number
	 * to the device
	 */
	p_dev = pci_scan_single_device(bus, 0);
	if (p_dev)
		p_dev->irq = pcie->irq;
#endif

	pci_bus_add_devices(bus);

	/* Release RSTGEN registers since other hardware might need them */
	iounmap(pcie->base_rstgen);
	release_mem_region(pcie->rsc_rstgen->start, resource_size(pcie->rsc_rstgen));

	/* Release CLKGEN-SYS registers since other hardware might need them */
	iounmap(pcie->base_clkgen);
	release_mem_region(pcie->rsc_clkgen->start, resource_size(pcie->rsc_clkgen));

#ifndef CONFIG_SOC_QUASAR6300   /* QB66xx has two controllers, so release common resources. */
	/* Release PCIEWRAP registers since another PCIe controiller needs them */
	iounmap(pcie->base_wrap);
	release_mem_region(pcie->rsc_wrap->start, resource_size(pcie->rsc_wrap));

	/* Release GPF-PCI registers since another PCIe controiller needs them */
	iounmap(pcie->base_gpf);
	release_mem_region(pcie->rsc_gpf->start, resource_size(pcie->rsc_gpf));
#endif

	return 0;

error:
	pci_free_resource_list(&res);
	return err;
}

#ifdef CONFIG_PM_SLEEP
static void quasar_pcie_backup_map(struct quasar_pcie *pcie)
{
	pcie_map.axi_amap_ctrl0      = readl(pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_CTRL0_OFF);
	pcie_map.axi_amap_axi_base0  = readl(pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_AXI_BASE0_OFF);
	pcie_map.axi_amap_axi_base0x = readl(pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_AXI_BASE0X_OFF);
	pcie_map.axi_amap_pex_basel0 = readl(pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_PEX_BASEL0_OFF);
	pcie_map.axi_amap_pex_baseh0 = readl(pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_PEX_BASEH0_OFF);
#ifdef CONFIG_SOC_QUASAR6600
	pcie_map.axi_amap_size_eam0x = readl(pcie->base_ctrl + 0x0000BAF0 /*PCIE1_PAB_AXI_AMAP_SIZE_EAM0X_OFF*/);
#endif

	pcie_map.axi_amap_ctrl1      = readl(pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_CTRL1_OFF);
	pcie_map.axi_amap_axi_base1  = readl(pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_AXI_BASE1_OFF);
	pcie_map.axi_amap_axi_base1x = readl(pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_AXI_BASE1X_OFF);
	pcie_map.axi_amap_pex_basel1 = readl(pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_PEX_BASEL1_OFF);
	pcie_map.axi_amap_pex_baseh1 = readl(pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_PEX_BASEH1_OFF);
#ifdef CONFIG_SOC_QUASAR6600
	pcie_map.axi_amap_size_eam1x = readl(pcie->base_ctrl + 0x0000BAF4 /*PCIE1_PAB_AXI_AMAP_SIZE_EAM1X_OFF*/);
#endif

	pcie_map.axi_amap_ctrl2      = readl(pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_CTRL2_OFF);
	pcie_map.axi_amap_axi_base2  = readl(pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_AXI_BASE2_OFF);
	pcie_map.axi_amap_axi_base2x = readl(pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_AXI_BASE2X_OFF);
	pcie_map.axi_amap_pex_basel2 = readl(pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_PEX_BASEL2_OFF);
	pcie_map.axi_amap_pex_baseh2 = readl(pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_PEX_BASEH2_OFF);
#ifdef CONFIG_SOC_QUASAR6600
	pcie_map.axi_amap_size_eam2x = readl(pcie->base_ctrl + 0x0000BAF8 /*PCIE1_PAB_AXI_AMAP_SIZE_EAM2X_OFF*/);
#endif

	pcie_map.axi_amap_ctrl3      = readl(pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_CTRL3_OFF);
	pcie_map.axi_amap_axi_base3  = readl(pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_AXI_BASE3_OFF);
	pcie_map.axi_amap_axi_base3x = readl(pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_AXI_BASE3X_OFF);
	pcie_map.axi_amap_pex_basel3 = readl(pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_PEX_BASEL3_OFF);
	pcie_map.axi_amap_pex_baseh3 = readl(pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_PEX_BASEH3_OFF);
#ifdef CONFIG_SOC_QUASAR6600
	pcie_map.axi_amap_size_eam3x = readl(pcie->base_ctrl + 0x0000BAFC /*PCIE1_PAB_AXI_AMAP_SIZE_EAM3X_OFF*/);
#endif

	pcie_map.pex_amap_ctrl0      = readl(pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_CTRL0_OFF);
	pcie_map.pex_amap_axi_base0  = readl(pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_AXI_BASE0_OFF);
	pcie_map.pex_amap_axi_base0x = readl(pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_AXI_BASE0X_OFF);
	pcie_map.pex_amap_pex_basel0 = readl(pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_PEX_BASEL0_OFF);
	pcie_map.pex_amap_pex_baseh0 = readl(pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_PEX_BASEH0_OFF);
#ifdef CONFIG_SOC_QUASAR6600
	pcie_map.pex_amap_axi_eam0x  = readl(pcie->base_ctrl + 0x0000BEF0 /*PCIE1_PAB_PEX_AMAP_AXI_EAM0X_OFF*/);
#endif

	pcie_map.pex_amap_ctrl1      = readl(pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_CTRL1_OFF);
	pcie_map.pex_amap_axi_base1  = readl(pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_AXI_BASE1_OFF);
	pcie_map.pex_amap_axi_base1x = readl(pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_AXI_BASE1X_OFF);
	pcie_map.pex_amap_pex_basel1 = readl(pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_PEX_BASEL1_OFF);
	pcie_map.pex_amap_pex_baseh1 = readl(pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_PEX_BASEH1_OFF);
#ifdef CONFIG_SOC_QUASAR6600
	pcie_map.pex_amap_axi_eam1x  = readl(pcie->base_ctrl + 0x0000BEF4 /*PCIE1_PAB_PEX_AMAP_AXI_EAM1X_OFF*/);
#endif

	pcie_map.pex_amap_ctrl2      = readl(pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_CTRL2_OFF);
	pcie_map.pex_amap_axi_base2  = readl(pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_AXI_BASE2_OFF);
	pcie_map.pex_amap_axi_base2x = readl(pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_AXI_BASE2X_OFF);
	pcie_map.pex_amap_pex_basel2 = readl(pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_PEX_BASEL2_OFF);
	pcie_map.pex_amap_pex_baseh2 = readl(pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_PEX_BASEH2_OFF);
#ifdef CONFIG_SOC_QUASAR6600
	pcie_map.pex_amap_axi_eam2x  = readl(pcie->base_ctrl + 0x0000BEF8 /*PCIE1_PAB_PEX_AMAP_AXI_EAM2X_OFF*/);
#endif

	pcie_map.pex_amap_ctrl3      = readl(pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_CTRL3_OFF);
	pcie_map.pex_amap_axi_base3  = readl(pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_AXI_BASE3_OFF);
	pcie_map.pex_amap_axi_base3x = readl(pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_AXI_BASE3X_OFF);
	pcie_map.pex_amap_pex_basel3 = readl(pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_PEX_BASEL3_OFF);
	pcie_map.pex_amap_pex_baseh3 = readl(pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_PEX_BASEH3_OFF);
#ifdef CONFIG_SOC_QUASAR6600
	pcie_map.pex_amap_axi_eam3x  = readl(pcie->base_ctrl + 0x0000BEFC /*PCIE1_PAB_PEX_AMAP_AXI_EAM3X_OFF*/);
#endif
}

static void quasar_pcie_restore_map(struct quasar_pcie *pcie)
{
	u32 val;

	writel(pcie_map.axi_amap_ctrl0,
		pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_CTRL0_OFF);
	writel(pcie_map.axi_amap_axi_base0,
		pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_AXI_BASE0_OFF);
	writel(pcie_map.axi_amap_axi_base0x,
		pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_AXI_BASE0X_OFF);
	writel(pcie_map.axi_amap_pex_basel0,
		pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_PEX_BASEL0_OFF);
	writel(pcie_map.axi_amap_pex_baseh0,
		pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_PEX_BASEH0_OFF);
#ifdef CONFIG_SOC_QUASAR6600
	writel(pcie_map.axi_amap_size_eam0x,
		pcie->base_ctrl + 0x0000BAF0 /*PCIE1_PAB_AXI_AMAP_SIZE_EAM0X_OFF*/);
#endif

	/* Enable AXI PIO access */
	val = (PCIE1_PAB_AXI_PIO_CTRL0__PIO_ENABLE__MASK |
		PCIE1_PAB_AXI_PIO_CTRL0__MEM_WINDOW_ENABLE__MASK |
		PCIE1_PAB_AXI_PIO_CTRL0__IO_WINDOW_ENABLE__MASK |
		PCIE1_PAB_AXI_PIO_CTRL0__CFG_WINDOW_ENABLE__MASK);
	writel(val, pcie->base_ctrl + PCIE1_PAB_AXI_PIO_CTRL0_OFF);

	writel(pcie_map.axi_amap_ctrl1,
		pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_CTRL1_OFF);
	writel(pcie_map.axi_amap_axi_base1,
		pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_AXI_BASE1_OFF);
	writel(pcie_map.axi_amap_axi_base1x,
		pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_AXI_BASE1X_OFF);
	writel(pcie_map.axi_amap_pex_basel1,
		pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_PEX_BASEL1_OFF);
	writel(pcie_map.axi_amap_pex_baseh1,
		pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_PEX_BASEH1_OFF);
#ifdef CONFIG_SOC_QUASAR6600
	writel(pcie_map.axi_amap_size_eam1x,
		pcie->base_ctrl + 0x0000BAF4 /*PCIE1_PAB_AXI_AMAP_SIZE_EAM1X_OFF*/);
#endif

	writel(pcie_map.axi_amap_ctrl2,
		pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_CTRL2_OFF);
	writel(pcie_map.axi_amap_axi_base2,
		pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_AXI_BASE2_OFF);
	writel(pcie_map.axi_amap_axi_base2x,
		pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_AXI_BASE2X_OFF);
	writel(pcie_map.axi_amap_pex_basel2,
		pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_PEX_BASEL2_OFF);
	writel(pcie_map.axi_amap_pex_baseh2,
		pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_PEX_BASEH2_OFF);
#ifdef CONFIG_SOC_QUASAR6600
	writel(pcie_map.axi_amap_size_eam2x,
		pcie->base_ctrl + 0x0000BAF8 /*PCIE1_PAB_AXI_AMAP_SIZE_EAM2X_OFF*/);
#endif

	writel(pcie_map.axi_amap_ctrl3,
		pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_CTRL3_OFF);
	writel(pcie_map.axi_amap_axi_base3,
		pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_AXI_BASE3_OFF);
	writel(pcie_map.axi_amap_axi_base3x,
		pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_AXI_BASE3X_OFF);
	writel(pcie_map.axi_amap_pex_basel3,
		pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_PEX_BASEL3_OFF);
	writel(pcie_map.axi_amap_pex_baseh3,
		pcie->base_ctrl + PCIE1_PAB_AXI_AMAP_PEX_BASEH3_OFF);
#ifdef CONFIG_SOC_QUASAR6600
	writel(pcie_map.axi_amap_size_eam3x,
		pcie->base_ctrl + 0x0000BAFC /*PCIE1_PAB_AXI_AMAP_SIZE_EAM3X_OFF*/);
#endif

	writel(pcie_map.pex_amap_ctrl0,
			pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_CTRL0_OFF);
	writel(pcie_map.pex_amap_axi_base0,
			pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_AXI_BASE0_OFF);
	writel(pcie_map.pex_amap_axi_base0x,
			pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_AXI_BASE0X_OFF);
	writel(pcie_map.pex_amap_pex_basel0,
			pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_PEX_BASEL0_OFF);
	writel(pcie_map.pex_amap_pex_baseh0,
			pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_PEX_BASEH0_OFF);
#ifdef CONFIG_SOC_QUASAR6600
	writel(pcie_map.pex_amap_axi_eam0x,
		pcie->base_ctrl + 0x0000BEF0 /*PCIE1_PAB_PEX_AMAP_AXI_EAM0X_OFF*/);
#endif

	writel(pcie_map.pex_amap_ctrl1,
			pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_CTRL1_OFF);
	writel(pcie_map.pex_amap_axi_base1,
			pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_AXI_BASE1_OFF);
	writel(pcie_map.pex_amap_axi_base1x,
			pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_AXI_BASE1X_OFF);
	writel(pcie_map.pex_amap_pex_basel1,
			pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_PEX_BASEL1_OFF);
	writel(pcie_map.pex_amap_pex_baseh1,
			pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_PEX_BASEH1_OFF);
#ifdef CONFIG_SOC_QUASAR6600
	writel(pcie_map.pex_amap_axi_eam1x,
		pcie->base_ctrl + 0x0000BEF4 /*PCIE1_PAB_PEX_AMAP_AXI_EAM1X_OFF*/);
#endif

	writel(pcie_map.pex_amap_ctrl2,
			pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_CTRL2_OFF);
	writel(pcie_map.pex_amap_axi_base2,
			pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_AXI_BASE2_OFF);
	writel(pcie_map.pex_amap_axi_base2x,
			pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_AXI_BASE2X_OFF);
	writel(pcie_map.pex_amap_pex_basel2,
			pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_PEX_BASEL2_OFF);
	writel(pcie_map.pex_amap_pex_baseh2,
			pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_PEX_BASEH2_OFF);
#ifdef CONFIG_SOC_QUASAR6600
	writel(pcie_map.pex_amap_axi_eam2x,
		pcie->base_ctrl + 0x0000BEF8 /*PCIE1_PAB_PEX_AMAP_AXI_EAM2X_OFF*/);
#endif

	writel(pcie_map.pex_amap_ctrl3,
			pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_CTRL3_OFF);
	writel(pcie_map.pex_amap_axi_base3,
			pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_AXI_BASE3_OFF);
	writel(pcie_map.pex_amap_axi_base3x,
			pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_AXI_BASE3X_OFF);
	writel(pcie_map.pex_amap_pex_basel3,
			pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_PEX_BASEL3_OFF);
	writel(pcie_map.pex_amap_pex_baseh3,
			pcie->base_ctrl + PCIE1_PAB_PEX_AMAP_PEX_BASEH3_OFF);
#ifdef CONFIG_SOC_QUASAR6600
	writel(pcie_map.pex_amap_axi_eam3x,
		pcie->base_ctrl + 0x0000BEFC /*PCIE1_PAB_PEX_AMAP_AXI_EAM3X_OFF*/);
#endif

	/* Enable PEX PIO access */
	writel(PCIE1_PAB_PEX_PIO_CTRL0__PIO_ENABLE__MASK,
			pcie->base_ctrl + PCIE1_PAB_PEX_PIO_CTRL0_OFF);
}

static int quasar_pcie_suspend(struct device *dev)
{
	struct quasar_pcie *pcie;

	pcie = dev_get_drvdata(dev);

	/* Assert PERST# */
	quasar_pcie_reset_device(pcie, 1);   /* assert PERST# */

	/* Backup memory map settings */
	quasar_pcie_backup_map(pcie);   /* backup memory map */

	return 0;
}

static int quasar_pcie_resume(struct device *dev)
{
	struct quasar_pcie *pcie;
	int err;

	pcie = dev_get_drvdata(dev);

	/* Re-initialize PCIe host hardware */
	quasar_pcie_hw_init(pcie);

	/* De-assert PERST# */
	quasar_pcie_reset_device(pcie, 0);

	/* Wait for device link up */
	err = quasar_pcie_wait_link_up(pcie);
	if (err) {
		dev_warn(pcie->dev, "PCIe link up failed, probably no device\n");
		return err;
	}

	/* Restore memory map settings */
	quasar_pcie_restore_map(pcie);

	/* Re-initialize PCIe host settings */
	quasar_pcie_rc_init(pcie);

	/* Re-initialize interrupt settings */
	quasar_pcie_init_interrupt(pcie);

	return 0;
}
#endif

static const struct dev_pm_ops quasar_pcie_pm_ops1 = {
	SET_NOIRQ_SYSTEM_SLEEP_PM_OPS(quasar_pcie_suspend, quasar_pcie_resume)
};

static const struct dev_pm_ops quasar_pcie_pm_ops2 = {
	SET_NOIRQ_SYSTEM_SLEEP_PM_OPS(quasar_pcie_suspend, quasar_pcie_resume)
};

static struct platform_driver quasar_pcie_driver1 = {
	.driver = {
#ifdef CONFIG_SOC_QUASAR6300
		.name = "quasar-pcie",
#else
		.name = "quasar-pcie1",
#endif
		.owner = THIS_MODULE,
		.of_match_table = quasar_pcie_of_match1,
		.pm = &quasar_pcie_pm_ops1,
	},
	.probe = quasar_pcie_probe,
};
module_platform_driver(quasar_pcie_driver1);

#ifdef EVM_TWO_RC
static struct platform_driver quasar_pcie_driver2 = {
	.driver = {
		.name = "quasar-pcie2",
		.owner = THIS_MODULE,
		.of_match_table = quasar_pcie_of_match2,
		.pm = &quasar_pcie_pm_ops2,
	},
	.probe = quasar_pcie_probe,
};
module_platform_driver(quasar_pcie_driver2);
#endif

MODULE_DESCRIPTION("Quasar PCIe host driver");
MODULE_LICENSE("GPLv2");
