/*
 * MTD driver for serial (SPI) flash chips via the 
 * Alma Technologies SPI-MEM-CTRL (FCSPI) controller
 *
 * Copyright (c) 2012, 2015 Linux Foundation
 *
 * Portions copyright (c) 2005 Intec Automation Inc.
 *
 * This code is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */

#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/platform_device.h>
#include <linux/module.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/partitions.h>
#include <linux/of_platform.h>
#include <linux/mm.h>
#include <linux/dma-mapping.h>
#include <linux/interrupt.h>
#include <linux/spi/flash.h>
#include <linux/sched.h>
#include <linux/workqueue.h>
#include <linux/delay.h>

#if defined(CONFIG_SOC_QUASAR6600)
#define QUASAR_DDRGPF_EXTADDRMODE                   0x0410908C
#else	// QB63XX
#define QUASAR_DDRGPF_EXTADDRMODE                   0x04109080
#endif
#define DDRGPF_EXTENDED_ADDRESS_MODE__EAM__MASK     0x00000001

#define DRIVER_NAME "fcspi"
//#define DEBUG_FCSPI
//#define DEBUG_FCSPI_WRITE
//#define DEBUG_FCSPI_READ

#define UBUFFSIZE PAGE_SIZE

/* Define max times to check status register before we give up. */
#define	MAX_READY_WAIT_JIFFIES	(40 * HZ)

#define SUSPND		(1<<0)
#define FCSPIBUSY	(1<<1)
#define RDBUSY		(1<<2)
#define WRBUSY		(1<<3)
#define ERASEBUSY	(1<<4)
#define FCSPIERR	(1<<5)

#define FCSPI_READ_MSG  0
#define FCSPI_WRITE_MSG 1
#define FCSPI_ERASE_MSG 2

#ifdef CONFIG_SOC_QUASAR6300
	/* 63xx has cache-coherency and we assume if it's on
	 * that fcspi will be coherent too
	 */
	#define FCSPI_COHERENT_DMA	(1<<28)
#else
	#define FCSPI_COHERENT_DMA	(1<<28)
#endif

/****************************************************************************/

struct fcspi_ctl_regs {
	volatile u32 fcspi_ctrl;
	volatile u32 fcspi_stat;
	volatile u32 fcspi_accrr0;
	volatile u32 fcspi_accrr1;
	volatile u32 fcspi_accrr2;
	volatile u32 fcspi_ddpm;
	volatile u32 fcspi_rwdata;
	volatile u32 fcspi_ffstat;
	volatile u32 fcspi_defmem;     /* Write only!! */
	volatile u32 fcspi_exaddr;
	volatile u32 fcspi_memspec;
	volatile u32 fcspi_intmsk;
	volatile u32 fcspi_intreq;
	volatile u32 fcspi_cicfg;
	volatile u32 fcspi_cidr0;
	volatile u32 fcspi_cidr1;    
	volatile u32 fcspi_rdcr;                     
	volatile u32 fcspi_reserved1[47];
	volatile u32 fcspi_cfgram[64]; /* Write only!! */
	volatile u32 fcspi_reserved2[8064];//384];
	volatile u32 fcspi_dma_saddr;
	volatile u32 fcspi_dma_faddr;
	volatile u32 fcspi_dma_len;
	volatile u32 fcspi_dma_cst;
	volatile u32 fcspi_xotf_cst;    
	volatile u32 fcspi_xotf_offset;    
	volatile u32 fcspi_dma_debug;
	volatile u32 fcspi_dma_spare;
};

struct fcspi_bbm_ctrl {
	int                            lba;
	int                            pba;
};

struct fcspi {
	struct platform_device        *pdev;
	struct flash_platform_data    *fdata;
	struct mtd_partition	      *parts;
	struct mtd_info		       mtd;
	struct fcspi_ctl_regs __iomem *regs;
	struct completion              xfer_completion;
	struct workqueue_struct	      *workqueue;
	struct work_struct             work;
	struct list_head               queue;
	spinlock_t                     lock;
	unsigned long                  rregs;
	unsigned long                  rregs_sz;
	u32                            irq;
	volatile unsigned              state;
	int                            flags;
	int                            paeshift;
	int                            nandmode;
	u32                            block_size;
	u32                            sector_size;
	uint8_t                       *buffer;
	struct fcspi_bbm_ctrl         *bbmctl;
	u32                           size;
};

struct fcspi_message {
	struct fcspi                  *ctl;
	int                            type;
	int                            result;
	loff_t                         flash_offset;
	size_t                         len;
	uint8_t                       *buf;
	struct completion              completion;
	struct list_head               queue;
};

static inline struct fcspi *mtd_to_fcspi(struct mtd_info *mtd)
{
	return container_of(mtd, struct fcspi, mtd);
}

#define FCSPI_ERASE_SECTOR 0
#define FCSPI_ERASE_BLOCK  1
#define FCSPI_ERASE_CHIP   2

#define CHIP_NAME_SZ 32

/* Spec sheets usually specify the flash size in megabits, but
 * we want bytes.  This macro converts from megabits to bytes
 */
#define MEGABITS(x) (((1024*1024)/8)*x)

/* flags:
 * bit 0:     NO_CHIP_ERASE
 * bit 1:     NO_BLOCK_ERASE 
 * bit 2:     ONCHIP_BBM
 *  ... 
 * bit 31-16: ONCHIP BBM LUT entry number
 */
#define NO_CHIP_ERASE  1
#define NO_BLOCK_ERASE 2
#define ONCHIP_BBM     4
#define BBMENTRY_WINBOND    (20 << 16)
#define BBM_ENTRIES(flags)  (flags >> 16)

struct fcspi_flash_info {
	u32 id;
	char name[CHIP_NAME_SZ];
	u32 size;               /* size of the chip in bytes */
	u32 erase_size;		/* smallest erase size */
	u32 block_size;		/* block erase size, if there is one */
	int flags;
};

static const struct fcspi_flash_info fcspi_info[] = {
/* Cypress/Spansion */
	{ 0x016017, "s25fl064L", MEGABITS(64),  4096, 64*1024, 0 },
	{ 0x010217, "s25fs064s", MEGABITS(64),  4096, 64*1024, 0 },
	{ 0x016018, "s25fl128l", MEGABITS(128), 4096, 64*1024, 0 },
	{ 0x012018, "s25fl128s", MEGABITS(128), 4096, 64*1024, 0 },
	{ 0x016019, "s25fl256l", MEGABITS(256), 4096, 64*1024, 0 },
	{ 0x010219, "s25fl256s", MEGABITS(256), 4096, 64*1024, 0 },  
	{ 0x010220, "s25fl512s", MEGABITS(512), 4096, 64*1024, 0 }, 
	{ 0x010221, "s70fs01gs", MEGABITS(1024),4096, 64*1024, 0 },                   
/* Gigadevice */
	{ 0xC84017, "gd25q64c",  MEGABITS(64),  4096, 64*1024, 0 },
	{ 0xC86017, "gd25lq64c", MEGABITS(64),  4096, 64*1024, 0 },
	{ 0xC84018, "gd25q127c", MEGABITS(128), 4096, 64*1024, 0 },
	{ 0xC86018, "gd25lq128d",MEGABITS(128), 4096, 64*1024, 0 },
	{ 0xC84019, "gd25q256d", MEGABITS(256), 4096, 64*1024, 0 },
	{ 0xC86019, "gd25lq256d",MEGABITS(256), 4096, 64*1024, 0 },
	{ 0xC8461A, "GD25T512ME",MEGABITS(512), 4096, 64*1024, 0 },
	{ 0xC8661A, "GD55LT512WE",MEGABITS(512), 4096, 64*1024, 0 },
	{ 0xC8471B, "GD55B01GE", MEGABITS(1024), 4096, 64*1024, 0 },
	{ 0xC8671B, "GD55LB01GE",MEGABITS(1024), 4096, 64*1024, 0 },
/* Macronix */
	{ 0xC22017, "mx25l6433f",  MEGABITS(64),  4096, 64*1024, 0 },
	{ 0xC22018, "mx25l12833f", MEGABITS(128), 4096, 64*1024, 0 },
	{ 0xC22019, "mx25l25645g", MEGABITS(256), 4096, 64*1024, 0 },
	{ 0xC2201A, "mx25l51245g", MEGABITS(512), 4096, 64*1024, 0 },
	{ 0xC2201B, "mx66l1g45g",  MEGABITS(1024),4096, 64*1024, 0 },
	{ 0xC2201C, "mx66l2g45g",  MEGABITS(2048),4096, 64*1024, 0 },        
	{ 0xC22537, "mx25u6432f",  MEGABITS(64), 4096, 64*1024, 0 },
	{ 0xC22538, "mx25u12835f", MEGABITS(128), 4096, 64*1024, 0 },
	{ 0xC22539, "mx25u25645g", MEGABITS(256), 4096, 64*1024, 0 },
	{ 0xC2253A, "mx25u51245g", MEGABITS(512), 4096, 64*1024, 0 },
	{ 0xC2253B, "mx66u1g45g",  MEGABITS(1024),4096, 64*1024, 0 },     
	{ 0xC2253C, "mx66u2g45g",  MEGABITS(2048),4096, 64*1024, 0 },             
/* Micron */
	{ 0x20BA18, "MT25ql128aba",MEGABITS(128), 4096, 64*1024, 0 }, 
	{ 0x20BB18, "MT25qu128aba",MEGABITS(128), 4096, 64*1024, 0 }, 
	{ 0x20BA19, "MT25ql256aba",MEGABITS(256), 4096, 64*1024, 0 },    
	{ 0x20BB19, "MT25qu256aba",MEGABITS(256), 4096, 64*1024, 0 },        
	{ 0x20BA20, "MT25ql512aba",MEGABITS(512), 4096, 64*1024, 0 },   
	{ 0x20BB20, "MT25qu512abb",MEGABITS(512), 4096, 64*1024, 0 },      
	{ 0x20BA21, "MT25ql01gbbb",MEGABITS(1024),4096, 64*1024, 0 },   
	{ 0x20BB21, "MT25qu01gbbb",MEGABITS(1024),4096, 64*1024, 0 },     
/* Winbond */
	{ 0xEF6017, "w25q64jw",    MEGABITS(64),  4096, 64*1024, 0 },
	{ 0xEF4017, "w25q64jv",    MEGABITS(64),  4096, 64*1024, 0 },    
	{ 0xEF6018, "w25q128jw",   MEGABITS(128), 4096, 64*1024, 0 },
	{ 0xEF4018, "w25q128jv",   MEGABITS(128), 4096, 64*1024, 0 },   
	{ 0xEF6019, "w25q256jw",   MEGABITS(256), 4096, 64*1024, 0 },
	{ 0xEF4019, "w25q256jv",   MEGABITS(256), 4096, 64*1024, 0 },      
	{ 0xEF6119, "W25M512JW",   MEGABITS(512), 4096, 64*1024, 0 },
	{ 0xEF4020, "W25Q512JV",   MEGABITS(512), 4096, 64*1024, 0 },
	{ 0xEF4021, "W25Q01JV",    MEGABITS(1024), 4096, 64*1024, 0 },
	{ 0xEF8A16, "W77q",        MEGABITS(64), 4096, 64*1024, 0 },
/* ISSI */
	{ 0x9D6019, "IS25LP256E",  MEGABITS(256),  4096, 64*1024, 0 },
	{ 0x9D7018, "IS25WP128F",  MEGABITS(128),  4096, 64*1024, 0 },
	{ 0x9D7019, "IS25WP256E",  MEGABITS(256),  4096, 64*1024, 0 },

	{ 0},
};

static irqreturn_t fcspi_intr(int irq, void *param)
{
	struct fcspi *ctl = param;
	struct fcspi_ctl_regs *regs = ctl->regs;

	writel(1<<24, &regs->fcspi_dma_cst);
	complete(&ctl->xfer_completion);
	return IRQ_HANDLED;
}

static int wait_for_ready(struct fcspi *ctl)
{
	struct fcspi_ctl_regs *regs = ctl->regs;
	unsigned long deadline = jiffies + MAX_READY_WAIT_JIFFIES;

	do {
		if ((readl(&regs->fcspi_stat) & (1<<3)))
			return 0;
		cond_resched();
	} while (!time_after_eq(jiffies, deadline));
	return 1;
}

////////////////////////////////////////////////////////////////////////////
// FCSPI SPI NAND support
//      
static const struct fcspi_flash_info fcspi_nand_info[] = {
/* Macronix */
    { 0x00C212, "MX35LF1GE4AB",MEGABITS(1024), 128*1024, 128*1024, 0 },
    { 0x00C214, "MX35LF1G24AD",MEGABITS(1024), 128*1024, 128*1024, 0 },
    { 0x00C224, "MX35LF2G24AD",MEGABITS(2048), 128*1024, 128*1024, 0 },
    { 0x00C226, "MX35LF2GE4AD",MEGABITS(2048), 128*1024, 128*1024, 0 },
    { 0x00C235, "MX35LF4G24AD",MEGABITS(4096), 128*1024, 128*1024, 0 },
    { 0x00C237, "MX35LF4GE4AD",MEGABITS(4096), 128*1024, 128*1024, 0 },
    { 0x00C294, "MX35UF1G24AD",MEGABITS(1024), 128*1024, 128*1024, 0 },
    { 0x00C296, "MX35UF1GE4AD",MEGABITS(1024), 128*1024, 128*1024, 0 },
    { 0x00C2A4, "MX35UF2G24AD",MEGABITS(2048), 128*1024, 128*1024, 0 },
    { 0x00C2A6, "MX35UF2GE4AD",MEGABITS(2048), 128*1024, 128*1024, 0 }, 
    { 0x00C2B5, "MX35UF4G24AD",MEGABITS(4096), 128*1024, 128*1024, 0 },
    { 0x00C2B7, "MX35UF4GE4AD",MEGABITS(4096), 128*1024, 128*1024, 0 },           
/* Micron */
    { 0x002C14, "MT29F1G01ABA",MEGABITS(1024), 128*1024, 128*1024, 0 },
    { 0x002C15, "MT29F1G01ABB",MEGABITS(1024), 128*1024, 128*1024, 0 },
    { 0x002C24, "MT29F2G01ABA",MEGABITS(2048), 128*1024, 128*1024, 0 },  
    { 0x002C25, "MT29F2G01ABB",MEGABITS(2048), 128*1024, 128*1024, 0 },      
    { 0x002C36, "MT29F4G01ADA",MEGABITS(4096), 128*1024, 128*1024, 0 },               
/* Winbond */
    { 0x00EFAA, "W25N01GV",    MEGABITS(1024), 128*1024, 128*1024, ONCHIP_BBM | BBMENTRY_WINBOND},
    { 0x00EFAB, "W25M02GV",    MEGABITS(2048), 128*1024, 128*1024, ONCHIP_BBM | BBMENTRY_WINBOND},
    { 0x00EFBA, "W25N01GW",    MEGABITS(1024), 128*1024, 128*1024, ONCHIP_BBM | BBMENTRY_WINBOND},
    { 0x00EFBB, "W25M02GW",    MEGABITS(2048), 128*1024, 128*1024, ONCHIP_BBM | BBMENTRY_WINBOND},
    { 0x00EFBC, "W25N01JW",    MEGABITS(1024), 128*1024, 128*1024, ONCHIP_BBM | BBMENTRY_WINBOND},
    { 0x00EFBF, "W25N02JW",    MEGABITS(1024), 128*1024, 128*1024, ONCHIP_BBM | BBMENTRY_WINBOND},
/* GigaDevice */
    { 0x00C8D1, "GD5F1GQ4UE",  MEGABITS(1024), 128*1024, 128*1024, 0 },
    { 0x00C851, "GD5F1GQ5UE",  MEGABITS(1024), 128*1024, 128*1024, 0 },
    { 0x00C8D2, "GD5F2GQ4UE",  MEGABITS(2048), 128*1024, 128*1024, 0 },
    { 0x00C852, "GD5F2GQ5UE",  MEGABITS(2048), 128*1024, 128*1024, 0 },
    { 0x00C855, "GD5F4GQ6UE",  MEGABITS(4096), 128*1024, 128*1024, 0 },
    { 0x00C8C1, "GD5F1GQ4RE",  MEGABITS(1024), 128*1024, 128*1024, 0 },
    { 0x00C841, "GD5F1GQ5RE",  MEGABITS(1024), 128*1024, 128*1024, 0 },
    { 0x00C8C2, "GD5F2GQ4RE",  MEGABITS(2048), 128*1024, 128*1024, 0 },
    { 0x00C842, "GD5F2GQ5RE",  MEGABITS(2048), 128*1024, 128*1024, 0 },
    { 0x00C845, "GD5F4GQ6RE",  MEGABITS(4096), 128*1024, 128*1024, 0 },
/* ISSI */    
    { 0x00C821, "IS37SML01G1", MEGABITS(1024), 128*1024, 128*1024, 0 },    
/* ESMT */
    { 0x00C801, "F50L1G41LB",  MEGABITS(1024), 128*1024, 128*1024, 0 },
    { 0x00C80A, "F50L2G41LB",  MEGABITS(2048), 128*1024, 128*1024, 0 },
    { 0x002C34, "F50L4G41XB",  MEGABITS(4096), 128*1024, 128*1024, 0 },
    { 0x00C811, "F50D1G41LB",  MEGABITS(1024), 128*1024, 128*1024, 0 },
    { 0x00C81A, "F50D2G41LB",  MEGABITS(2048), 128*1024, 128*1024, 0 },
    { 0x002C35, "F50D4G41XB",  MEGABITS(4096), 128*1024, 128*1024, 0 },
    { 0},
};

#define NAND_PAGE_SIZE          2048
#define NAND_PAGE_MASK          ~(NAND_PAGE_SIZE - 1)

#define GETBBMLUT_COMMAND       0xA5
#define SETBBMLUT_COMMAND       0xA1
/// SPI Nand read ID command
#define READID_COMMAND          0x9F
/// SPI Nand page read command
#define PAGEREAD_COMMAND        0x13
/// SPI Nand program execute command
#define PROGRAM_EXECUTE_COMMAND 0x10
/// SPI Nand read status register command
#define RDSR_COMMAND            0x0F
/// SPI Nand write statuc register command
#define WRSR_COMMAND            0x1F
/// SPI Nand erase command
#define ERASEBLOCK_COMMAND      0xD8
/// code for get status register SR_A0 
#define SR_A0                   0xA0
/// code for get status register SR_B0 
#define SR_B0                   0xB0
/// code for get status register SR_C0 
#define SR_C0                   0xC0
/// SR_C0.bit2: E-FAIL bit 
#define SR_C0_EFAIL             0x04    // bit2: Erase Failed bit 
/// SR_C0.bit3: P-FAIL bit 
#define SR_C0_PFAIL             0x08    // bit3: Program Failed bit 
/// SR_B0.bit4: ECC-EN bit 
#define SR_B0_ECC_EN            0x10    // bit4: ECC-EN bit 
/// SR_B0.bit3: BUF bit
#define SR_B0_BUF               0x08    // bit3: BUF bit   
/// SR_B0.bit0: QE bit for Macronix
#define SR_B0_QE                0x01    // bit0: QE bit for Macronix  
/// SR_C0.bit0: OIP (Operation In Progress, Busy)       
#define SR_C0_OIP               0x01    // bit0: OIP (Operation In Progress, Busy)
//#define SR_C0_ECC_MASK          0x70    // bit4-6: ECC status bits (bit-6 for Micron only)
/// SR_C0.bit4-5: ECC status bits
#define SR_C0_ECC_MASK          0x30    // bit4-5: ECC status bits

#define FCSPI_CICFG__HOLDN__MASK        0x00002000
#define FCSPI_CICFG__LEN__SHIFT         8
#define FCSPI_CTRL__MTYPE__MASK         0x08000000
#define FCSPI_CICFG__WREN__MASK         0x00008000
#define FCSPI_CICFG__WIP__MASK          0x00004000
#define FCSPI_CICFG__HOLDN__MASK        0x00002000
#define FCSPI_CICFG__LFEN__MASK         0x00010000
#define FCSPI_CICFG__LFSTOP__MASK       0x00020000
#define FCSPI_CICFG__LEN__SHIFT         8

static int fcspi_nand_mode(struct fcspi *ctl)
{
    struct fcspi_ctl_regs *regs = ctl->regs;
    int nandmode=0;
    
    if (readl(&regs->fcspi_ctrl) & FCSPI_CTRL__MTYPE__MASK)
    {
        nandmode = 1;
    }

    return nandmode;
}

static int fcspi_custom_execute(struct fcspi *ctl, u32 cicfg, u32 cidr0, u32 cidr1, u32* pdr0, u32* pdr1)
{
	struct fcspi_ctl_regs *regs = ctl->regs;
        
	if (wait_for_ready(ctl))
		return EIO;  

	writel(cidr0, &regs->fcspi_cidr0);
	writel(cidr1, &regs->fcspi_cidr1);
	writel(cicfg, &regs->fcspi_cicfg);

	if (wait_for_ready(ctl))
		return EIO;
        
	*pdr0 = readl(&regs->fcspi_cidr0);
	*pdr1 = readl(&regs->fcspi_cidr1);
     
	return 0;
}

static void fcspi_nand_set_feature(struct fcspi *ctl, u32 dr0)
{
    u32 cicfg, cidr0, cidr1;
    u32 rddr0, rddr1;
    
    cicfg = FCSPI_CICFG__HOLDN__MASK | 3 << FCSPI_CICFG__LEN__SHIFT | WRSR_COMMAND;
    cidr0 = dr0;  
    cidr1 = 0;
    fcspi_custom_execute(ctl, cicfg, cidr0, cidr1, &rddr0, &rddr1);
}

static u32 fcspi_nand_get_feature(struct fcspi *ctl, u32 dr0)
{
    u32 cicfg, cidr0, cidr1;
    u32 rddr0, rddr1;
    
    cicfg = FCSPI_CICFG__HOLDN__MASK | 3 << FCSPI_CICFG__LEN__SHIFT | RDSR_COMMAND;
    cidr0 = dr0;
    cidr1 = 0;
    fcspi_custom_execute(ctl, cicfg, cidr0, cidr1, &rddr0, &rddr1);
    
    return ((rddr0 >> 8) & 0xFF);
}    

static u32 fcspi_nand_get_id(struct fcspi *ctl)
{
    u32 cicfg, cidr0, cidr1;
    u32 rddr0, rddr1;
    
    cicfg = FCSPI_CICFG__WREN__MASK | FCSPI_CICFG__WIP__MASK | 
            FCSPI_CICFG__HOLDN__MASK | 4 << FCSPI_CICFG__LEN__SHIFT | READID_COMMAND;
    cidr0 = 0;
    cidr1 = 0;
    fcspi_custom_execute(ctl, cicfg, cidr0, cidr1, &rddr0, &rddr1); 
    
    // format: manufactureID deviceID 00 00
    return (rddr0 & 0x0000FF00) | ((rddr0 & 0x00FF0000) >> 16);   
}

#if 0
static u32 fcspi_hwecc_enable(struct fcspi *ctl, int enable)
{
	u32 srb0;
	
	srb0 = fcspi_nand_get_feature(ctl, SR_B0);
	if (enable)
	{
		// Enable HW-ECC 
		srb0 |= SR_B0_ECC_EN;
	}
	else
	{
		// Disable HW-ECC 
		srb0 &= ~SR_B0_ECC_EN;
	}

	fcspi_nand_set_feature(ctl, SR_B0 | (srb0<< 8));
	
	return 0;
}
#endif

// using the last blocks in flash as the replacement blocks
static int fcspi_bbm_replace(struct fcspi *ctl, u32 flash_offset)
{
	int i, found=0;
	int entries=BBM_ENTRIES(ctl->flags);
	u32 lba, pba;
	u32 cicfg, cidr0=0, cidr1=0;
	u32 rddr0, rddr1;
	
	if (flash_offset & (ctl->block_size - 1))
	{
		// flash offset is not in block boundary
		return -1;
	}
	
	// Last physical block no.
	pba = (ctl->size / ctl->block_size) - 1;
	lba = flash_offset / ctl->block_size;
	while (!found)
	{
		for (i=0; i<entries; i++)
		{
			if (!ctl->bbmctl[i].pba)
			{
				// no conflict with the entries in LUT
				found = 1;
				break;
			}
			else if (ctl->bbmctl[i].lba == lba)
			{
				// replacement block already in bbm LUT !!!
				// failed to replace it twice
				pr_err("MTD_FCSPI: lpb %d is already in bbm LUT !!!", lba);
				return -3;
			}
			else if (ctl->bbmctl[i].pba == pba)
			{
				// the pba already in the LUT entries, try next one
				pba--;
				break;
			}
		}
		
		if (i == entries)
		{
			// failed, no available entries in LUT
			return -1;
		}
	}
	
	// Find one replacement block pba, update the LUT
	cicfg = FCSPI_CICFG__WIP__MASK | FCSPI_CICFG__WREN__MASK |
			FCSPI_CICFG__HOLDN__MASK | 5 << FCSPI_CICFG__LEN__SHIFT | SETBBMLUT_COMMAND;
	// Sequence: A1h, lpb[15:8], lpa[7:0, pba[15:8], pba[7:0]
	cidr0 = ((lba & 0xFF00) >> 8) | ((lba & 0x00FF) << 8) | 
			((pba & 0xFF00) << 8) | ((pba & 0x00FF) << 24);
	fcspi_custom_execute(ctl, cicfg, cidr0, cidr1, &rddr0, &rddr1);
	
	// Update the bbm entry table
	ctl->bbmctl[i].pba = pba;
	ctl->bbmctl[i].lba = lba;

	return 0;
}

/*
 * Erase one sector or block or the entire chip at ``offset'' which is any
 * address within the region which should be erased.
 *
 * Returns 0 if successful, non-zero otherwise.
 */
static int erase_region(struct fcspi *ctl, u32 offset, int type)
{
	struct fcspi_ctl_regs *regs = ctl->regs;

#ifdef DEBUG_FCSPI_WRITE
	printk("%s: offset %08x type %d\n", __func__, offset, type);
#endif

	if (wait_for_ready(ctl))
		return EIO;

	if (ctl->nandmode)
	{
		unsigned long deadline = jiffies + MAX_READY_WAIT_JIFFIES;
        
		// Unprotect BP blocks, data byte is 0 means clear
		fcspi_nand_set_feature(ctl, SR_A0);

		// wait for device not busy
		do {
			if (!(fcspi_nand_get_feature(ctl, SR_C0) & SR_C0_OIP))
				break;
			if (time_after_eq(jiffies, deadline))
			{
				pr_debug("  ### NAND erase: not ready !!!!\n");                 
				break;
			}
			cond_resched();
		} while (1);              
	}

	writel(offset, &regs->fcspi_accrr0);
	writel(type, &regs->fcspi_accrr1);
	writel(2, &regs->fcspi_accrr2);

	if (wait_for_ready(ctl))
		return EIO;
		
	if (ctl->nandmode && (ctl->flags & ONCHIP_BBM) && 
		(fcspi_nand_get_feature(ctl, SR_C0) & SR_C0_EFAIL))
	{
		printk(" ### NAND Erase failed, addr= 0x%08X\n", offset);
		fcspi_bbm_replace(ctl, offset);
		
		return erase_region(ctl, offset, type);        
	}

	return 0;
}

static int fcspi_erase_work(struct fcspi_message *msg)
{
	struct fcspi *ctl = msg->ctl;
	u32 addr,len;

	addr = msg->flash_offset;
	len = msg->len;

#ifdef DEBUG_FCSPI_WRITE
	printk("%s: addr %08x len %08x\n", 
	       __func__, addr, len);
#endif

	if (len == ctl->mtd.size && !(ctl->flags & NO_CHIP_ERASE)) {
		if (erase_region(ctl, 0, FCSPI_ERASE_CHIP)) {
			return -EIO;
		}
		return 0;
	}
	while (len >= ctl->block_size && !(ctl->flags & NO_BLOCK_ERASE)) {
		if (erase_region(ctl, addr, FCSPI_ERASE_BLOCK)) {
			return -EIO;
		}
		addr += ctl->block_size;
		len -= ctl->block_size;
	}
	/* "sector"-at-a-time erase */
	while (len) {
		if (erase_region(ctl, addr, FCSPI_ERASE_SECTOR)) {
			return -EIO;
		}
		addr += ctl->sector_size;
		len -= ctl->sector_size;
	}
	return 0;
}

static int fcspi_write_work(struct fcspi_message *msg)
{
	struct fcspi *ctl = msg->ctl;
	struct fcspi_ctl_regs *regs = ctl->regs;
	struct device *dev = &ctl->pdev->dev;
	dma_addr_t tx_dma;

#ifdef DEBUG_FCSPI_WRITE
	printk("%s: to %llx len %d buf %p\n", __func__, 
           msg->flash_offset, msg->len, msg->buf);
#endif

	if ((msg->flash_offset & 3) || (msg->len & 3)
			|| (size_t)(msg->buf) & (3 | ((1 << ctl->paeshift) - 1))) {
		loff_t aligned_to = msg->flash_offset & ~3;
		unsigned int adj2, adj1 = msg->flash_offset - aligned_to;
		size_t aligned_len = msg->len + adj1;

		adj2 = (4 - (aligned_len & 3)) & 3;
		aligned_len += adj2;
        
		while (msg->len) {
			int copylen;
			int this_len;
			int retval;
			struct fcspi_message this_msg;
			if (aligned_len > UBUFFSIZE) {
				this_len = UBUFFSIZE;
				copylen = this_len;
			} else {
				this_len = aligned_len;
				copylen = this_len - adj1 - adj2;
			}
#ifdef DEBUG_FCSPI_WRITE
			printk("  %s: to %llx len %x buf %p\n", 
			       __func__, msg->flash_offset, 
			       msg->len, msg->buf);
			printk("      aligned_to %llx aligned_len %x "
			       "adj1 %d adj2 %d copylen %x\n",
			       aligned_to, aligned_len, 
			       adj1, adj2, 
			       copylen);
#endif
			*(u32 *)ctl->buffer = 0xffffffff;
			*(u32 *)(ctl->buffer + this_len - 4) = 0xffffffff;
			memcpy(ctl->buffer + adj1, msg->buf, copylen);
			this_msg.ctl = ctl;
			this_msg.type = FCSPI_WRITE_MSG;
			this_msg.result = 0;
			this_msg.flash_offset = aligned_to;
			this_msg.len = this_len;
			this_msg.buf = ctl->buffer;
			retval = fcspi_write_work(&this_msg);
			if (retval)
				return retval;
			msg->buf += copylen;
			msg->len -= copylen;
			aligned_to += this_len;
			adj1 = 0;
		}
		return 0;
	}

	tx_dma = dma_map_single(dev, (void *)msg->buf, msg->len, DMA_TO_DEVICE);
	if (dma_mapping_error(dev, tx_dma)) {
		dev_err(dev, "dma_map_single Rx failed\n");
		return -ENOMEM;
	}

	if (wait_for_ready(ctl))
		return -EIO;

#ifdef DEBUG_FCSPI_WRITE
	printk("%s: tx_dma %08x\n", __func__, tx_dma);
#endif
	writel(tx_dma >> ctl->paeshift, &regs->fcspi_dma_saddr);
	writel(msg->flash_offset, &regs->fcspi_dma_faddr);
	writel(msg->len, &regs->fcspi_dma_len);
	reinit_completion(&ctl->xfer_completion);
	writel(FCSPI_COHERENT_DMA | 1<<16 | 1<<4 | 1, &regs->fcspi_dma_cst);
	wait_for_completion(&ctl->xfer_completion);

	dma_unmap_single(dev, tx_dma, msg->len, DMA_TO_DEVICE);

	return 0;
}

static int fcspi_nand_read_work(struct fcspi_message *msg);
static int fcspi_nand_write_work(struct fcspi_message *msg);
static int fcspi_nand_replace_page(struct fcspi *ctl, loff_t flash_addr, uint8_t *source, size_t size)
{
	loff_t pageaddr; 
	uint8_t *blockbuf;
	uint8_t *pagebuf;
	uint8_t *pageoffset;
	struct fcspi_message msg;
	u32 pageIndex, pagesperblk, replacePage;
	int ret=0;
	
	blockbuf = (uint8_t*)kmalloc(ctl->block_size, GFP_KERNEL);
	if (!blockbuf)
	{
		// cannot allocate block buffer
		return -3;
	}
	
	msg.ctl = ctl;
	/// step 1 -- read all pages in the replacement block except the bad page
	pageaddr = flash_addr & ~(ctl->block_size - 1);
	replacePage = (flash_addr & (ctl->block_size - 1)) / NAND_PAGE_SIZE;
	pagesperblk = ctl->block_size / NAND_PAGE_SIZE;
	
	// copy the bad page data to the block buffer
	// copy replace page content to block buffer
	pagebuf = blockbuf + replacePage * NAND_PAGE_SIZE;
	pageoffset = pagebuf + (flash_addr & (NAND_PAGE_SIZE - 1)); 
	printk("MTDFCSPI, bad page: pageaddr= 0x%llX, pagebuf= %p, pageoffset= %p, size= %ld\n", pageaddr, pagebuf, pageoffset, size); 
#if 1
	// Assume content in the bad page still available, read it first and then update the new one 
	msg.flash_offset = pageaddr + replacePage * NAND_PAGE_SIZE;
	msg.len = NAND_PAGE_SIZE;
	msg.buf = pagebuf;
	ret = fcspi_nand_read_work(&msg);
#else          
	// Assume device reports bad page in the first write operation, so no old content need to preserved in the bad page  
	memset((void*)pagebuf, 0xFF, NAND_PAGE_SIZE);
#endif	
	memcpy((void*)pageoffset, (void*)source, size);

	pagebuf = blockbuf;
	for (pageIndex=0; pageIndex<pagesperblk; pageIndex++)
	{
		if (pageIndex != replacePage)
		{
			msg.flash_offset = pageaddr;
			msg.len = NAND_PAGE_SIZE;
			msg.buf = pagebuf;
			ret = fcspi_nand_read_work(&msg);
			// check ret
		}
		pageaddr += NAND_PAGE_SIZE;
		pagebuf += NAND_PAGE_SIZE;
	}
	  
	/// step 2 -- find a replace block
	pageaddr = flash_addr & ~(ctl->block_size - 1);
	ret = fcspi_bbm_replace(ctl, pageaddr);
	if (!ret)
	{
		/// step 3 -- erase the replacement block
		ret = erase_region(ctl, pageaddr, FCSPI_ERASE_BLOCK);
		
		/// step 4 -- update pages before the bad page (include the bad page)  
		pagebuf = blockbuf;
		for (pageIndex=0; pageIndex<pagesperblk; pageIndex++)
		{
			msg.flash_offset = pageaddr;
			msg.len = NAND_PAGE_SIZE;
			msg.buf = pagebuf;
			ret = fcspi_nand_write_work(&msg);             
			// check ret
			pageaddr += NAND_PAGE_SIZE;
			pagebuf += NAND_PAGE_SIZE;
		}
	}
	
	kfree(blockbuf);
	return ret;
}

static int fcspi_nand_write_work(struct fcspi_message *msg)
{
	struct fcspi *ctl = msg->ctl;
	struct fcspi_ctl_regs *regs = ctl->regs;
	struct device *dev = &ctl->pdev->dev;
	dma_addr_t tx_dma;    
	loff_t  page_offset;
	int copylen; 

#ifdef DEBUG_FCSPI_WRITE
	printk("%s: to %llx len %d buf %p\n", __func__, 
           msg->flash_offset, msg->len, msg->buf);
#endif
            
	page_offset = msg->flash_offset & (NAND_PAGE_SIZE - 1);
	msg->flash_offset &= NAND_PAGE_MASK;        
	while (msg->len) {
		if ((page_offset + msg->len) > NAND_PAGE_SIZE) {
			copylen = NAND_PAGE_SIZE - page_offset;         
		} else {
			copylen = msg->len;
		}

		if (wait_for_ready(ctl))
		{
			return -EIO;
		}

		memset(ctl->buffer, 0xFF, NAND_PAGE_SIZE);
		memcpy(ctl->buffer + page_offset, msg->buf, copylen);

		tx_dma = dma_map_single(dev, (void *)ctl->buffer, NAND_PAGE_SIZE, DMA_TO_DEVICE);
		if (dma_mapping_error(dev, tx_dma)) {
			dev_err(dev, "dma_map_single Rx failed\n");
			return -ENOMEM;
		}

		// Unprotect BP blocks, data byte is 0 means clear
		fcspi_nand_set_feature(ctl, SR_A0);

		writel(tx_dma >> ctl->paeshift, &regs->fcspi_dma_saddr);
		writel(msg->flash_offset, &regs->fcspi_dma_faddr);
		writel(NAND_PAGE_SIZE, &regs->fcspi_dma_len);
		reinit_completion(&ctl->xfer_completion);
		writel(FCSPI_COHERENT_DMA | 1<<16 | 1<<4 | 1, &regs->fcspi_dma_cst);
		wait_for_completion(&ctl->xfer_completion);
		dma_unmap_single(dev, tx_dma, msg->len, DMA_TO_DEVICE);

		if ((ctl->flags & ONCHIP_BBM) && 
			(fcspi_nand_get_feature(ctl, SR_C0) & SR_C0_PFAIL))
		{
			//printk(" ### NAND write failed, addr= 0x%llX, SR_C0: 0x%02X\n", msg->flash_offset, fcspi_nand_get_feature(ctl, SR_C0));
			if (fcspi_nand_replace_page(ctl, msg->flash_offset + page_offset, &ctl->buffer[page_offset], copylen))
			{
				// error -- TBD
			}
		}
                
		msg->buf += copylen;
		msg->len -= copylen;
		msg->flash_offset += NAND_PAGE_SIZE;
		page_offset = 0;
	}

	return 0;
}

static int fcspi_read_work(struct fcspi_message *msg)
{
	struct fcspi *ctl = msg->ctl;
	struct device *dev = &ctl->pdev->dev;
	struct fcspi_ctl_regs *regs = ctl->regs;
	dma_addr_t rx_dma;
	
	if ((msg->flash_offset & 3) || (msg->len & 3)
			|| (size_t)(msg->buf) & (3 | ((1 << ctl->paeshift) - 1))) {
		loff_t aligned_from = msg->flash_offset & ~3;
		unsigned int adj2, adj1 = msg->flash_offset - aligned_from;
		size_t aligned_len = msg->len + adj1;

		adj2 = (4 - (aligned_len & 3)) & 3;
		aligned_len += adj2;
		while (msg->len) {
			int copylen;
			int this_len;
			int retval;
			struct fcspi_message this_msg;
#ifdef DEBUG_FCSPI_READ
			printk("%s: from %llx msg->len %x buf %p\n", 
			       __func__, msg->flash_offset, msg->len, msg->buf);
#endif
			if (aligned_len > UBUFFSIZE) {
				this_len = UBUFFSIZE;
				copylen = this_len - adj1;
			} else {
				this_len = aligned_len;
				copylen = this_len - adj1 - adj2;
			}
#ifdef DEBUG_FCSPI_READ
			printk("len %08x aligned_from %llx aligned_len %x "
			       "adj1 %d adj2 %d copylen %x\n",
			       msg->len, aligned_from, aligned_len, adj1, adj2, 
			       copylen);
#endif
			this_msg.ctl = ctl;
			this_msg.type = FCSPI_READ_MSG;
			this_msg.result = 0;
			this_msg.flash_offset = aligned_from;
			this_msg.len = this_len;
			this_msg.buf = ctl->buffer;
			retval = fcspi_read_work(&this_msg);
			if (retval)
				return retval;
			memcpy(msg->buf, ctl->buffer+adj1, copylen);
			adj1 = 0;
			msg->buf += copylen;
			msg->len -= copylen;
			aligned_from += this_len;
		}
		return 0;
	}

	rx_dma = dma_map_single(dev, msg->buf, msg->len, DMA_FROM_DEVICE);
	if (dma_mapping_error(dev, rx_dma)) {
		dev_err(dev, "dma_map_single Rx failed\n");
		return -ENOMEM;
	}

#ifdef DEBUG_FCSPI_READ
	printk("%s: from %llx len %x buf %p rx_dma %08x\n", 
	       __func__, msg->flash_offset, msg->len, msg->buf, rx_dma);
#endif
	if (wait_for_ready(ctl))
		return -EIO;

	writel(msg->flash_offset, &regs->fcspi_dma_faddr);
	writel(rx_dma >> ctl->paeshift, &regs->fcspi_dma_saddr);
	writel(msg->len, &regs->fcspi_dma_len);

	reinit_completion(&ctl->xfer_completion);
	writel(FCSPI_COHERENT_DMA | 1<<16 | 1<<4, &regs->fcspi_dma_cst);
	wait_for_completion(&ctl->xfer_completion);

	dma_unmap_single(dev, rx_dma, msg->len, DMA_FROM_DEVICE);
	/*
	printk("%s: from %llx len %x buf %p %02X %02X %02X %02X\n", 
		__func__, msg->flash_offset, msg->len, msg->buf,
		msg->buf[0], msg->buf[1], msg->buf[2], msg->buf[3]);
	*/
	return 0;
}

static int fcspi_nand_read_work(struct fcspi_message *msg)
{
	struct fcspi *ctl = msg->ctl;
	struct device *dev = &ctl->pdev->dev;
	struct fcspi_ctl_regs *regs = ctl->regs;
	dma_addr_t rx_dma;
	loff_t  page_offset;
    
	page_offset = msg->flash_offset & (NAND_PAGE_SIZE - 1);
	if ((msg->flash_offset & 3) || (msg->len & 3) 
			|| ((page_offset + msg->len) > NAND_PAGE_SIZE)
			|| (size_t)(msg->buf) & (3 | ((1 << ctl->paeshift) - 1))) {
		loff_t aligned_from = msg->flash_offset & ~3;
		unsigned int adj2, adj1 = msg->flash_offset - aligned_from;
		size_t aligned_len = msg->len + adj1;

		adj2 = (4 - (aligned_len & 3)) & 3;
		aligned_len += adj2;
		while (msg->len) {
			int copylen;
			int this_len;
			int retval;
			struct fcspi_message this_msg;
#ifdef DEBUG_FCSPI_READ
			printk("%s: from %llx msg->len %x buf %p\n", 
			       __func__, msg->flash_offset, msg->len, msg->buf);
#endif
			if ((aligned_len + (page_offset & ~3)) > NAND_PAGE_SIZE) {
				this_len = NAND_PAGE_SIZE - (page_offset & ~3);
				if (this_len > msg->len)
				{
					this_len  = msg->len;
				}
				copylen = this_len - adj1;
			} else {
				this_len = aligned_len;
				if (this_len > (msg->len + adj1 + adj2))
				{
					this_len  = msg->len + adj1 + adj2;
				}                
				copylen = this_len - adj1 - adj2;
			}
#ifdef DEBUG_FCSPI_READ
			printk("len %08x aligned_from %llx aligned_len %x "
			       "adj1 %d adj2 %d copylen %x\n",
			       msg->len, aligned_from, aligned_len, adj1, adj2, 
			       copylen);
#endif
			this_msg.ctl = ctl;
			this_msg.type = FCSPI_READ_MSG;
			this_msg.result = 0;
			this_msg.flash_offset = aligned_from;
			this_msg.len = this_len;
			this_msg.buf = ctl->buffer;
			retval = fcspi_nand_read_work(&this_msg);
			if (retval)
				return retval;
			memcpy(msg->buf, ctl->buffer+adj1, copylen);
			adj1 = 0;
			msg->buf += copylen;
			msg->len -= copylen;
			aligned_from += this_len;
			page_offset = 0;
		}
		return 0;
	}

	rx_dma = dma_map_single(dev, msg->buf, msg->len, DMA_FROM_DEVICE);
	if (dma_mapping_error(dev, rx_dma)) {
		dev_err(dev, "dma_map_single Rx failed\n");
		return -ENOMEM;
	}

#ifdef DEBUG_FCSPI_READ
	printk("%s: from %llx len %x buf %p rx_dma %08x\n", 
	       __func__, msg->flash_offset, msg->len, msg->buf, rx_dma);
#endif
	if (wait_for_ready(ctl))
		return -EIO;
            
	writel(msg->flash_offset, &regs->fcspi_dma_faddr);
	writel(rx_dma >> ctl->paeshift, &regs->fcspi_dma_saddr);
	writel(msg->len, &regs->fcspi_dma_len);

	reinit_completion(&ctl->xfer_completion);
	writel(FCSPI_COHERENT_DMA | 1<<16 | 1<<4, &regs->fcspi_dma_cst);
	wait_for_completion(&ctl->xfer_completion);

	dma_unmap_single(dev, rx_dma, msg->len, DMA_FROM_DEVICE);
	/*
	printk("%s: from %llx len %x buf %p %02X %02X %02X %02X\n", 
		__func__, msg->flash_offset, msg->len, msg->buf,
		msg->buf[0], msg->buf[1], msg->buf[2], msg->buf[3]);
	*/
	return 0;
}

static void handle_msg(struct fcspi_message *msg)
{
	switch (msg->type) {
	case FCSPI_READ_MSG:
		if (msg->ctl->nandmode)
		{
			//fcspi_hwecc_enable(msg->ctl, msg->flash_offset >= 0x03b00000 ? 0 : 1);
			msg->result = fcspi_nand_read_work(msg);
		}
		else
		{    
			msg->result = fcspi_read_work(msg);
		}    
		break;
	case FCSPI_WRITE_MSG:
		if (msg->ctl->nandmode)
		{
			//fcspi_hwecc_enable(msg->ctl, msg->flash_offset >= 0x03b00000 ? 0 : 1);
			msg->result = fcspi_nand_write_work(msg);
		}
		else
		{    
			msg->result = fcspi_write_work(msg);
		}
		break;
	case FCSPI_ERASE_MSG:
		msg->result = fcspi_erase_work(msg);
		break;
	default:
		msg->result = -EINVAL;
		break;
	}
	complete(&msg->completion);
}

static void fcspi_work(struct work_struct *work)
{
	struct fcspi *ctl = container_of(work, struct fcspi, work);
	unsigned long flags;

	spin_lock_irqsave(&ctl->lock, flags);

	while (!list_empty(&ctl->queue)
	       && !(ctl->state & SUSPND)) {

		struct fcspi_message *msg;

		msg = container_of(ctl->queue.next, 
				   struct fcspi_message, queue);

		list_del_init(&msg->queue);

		/* Set Xfer busy flag */
		ctl->state |= FCSPIBUSY;

		spin_unlock_irqrestore(&ctl->lock, flags);

		handle_msg(msg);

		spin_lock_irqsave(&ctl->lock, flags);

		ctl->state &= ~FCSPIBUSY;
	}

	spin_unlock_irqrestore(&ctl->lock, flags);
}


static int fcspi_queue_work(struct fcspi_message *msg)
{
	struct fcspi *ctl = msg->ctl;
	unsigned long flags;

	init_completion(&msg->completion);
	spin_lock_irqsave(&ctl->lock, flags);
	if (ctl->state & SUSPND) {
		spin_unlock_irqrestore(&ctl->lock, flags);
		return -ESHUTDOWN;
	}
	list_add_tail(&msg->queue, &ctl->queue);
	queue_work(ctl->workqueue, &ctl->work);
	spin_unlock_irqrestore(&ctl->lock, flags);
	wait_for_completion(&msg->completion);
	return msg->result;
}

/**
 * fcspi_erase - [MTD Interface] erase block(s)
 * @mtd:	MTD device structure
 * @instr:	erase instruction
 *
 * Erase one or more regions
 */
static int fcspi_erase(struct mtd_info *mtd, struct erase_info *instr)
{
	struct fcspi *ctl = mtd_to_fcspi(mtd);
	struct fcspi_message msg;
	int    result;
	uint32_t rem;

#ifdef DEBUG_FCSPI
	printk("%s: %llx len %lld \n", 
	       __func__, (long long)instr->addr, (long long)instr->len);
#endif

	/* sanity checks */
	if (instr->addr + instr->len > ctl->mtd.size)
		return -EINVAL;
	div_u64_rem(instr->len, mtd->erasesize, &rem);
	if (rem)
		return -EINVAL;

	msg.ctl = ctl;
	msg.type = FCSPI_ERASE_MSG;
	msg.result = 0;
	msg.flash_offset = instr->addr;
	msg.len = instr->len;
	msg.buf = NULL;
	result = fcspi_queue_work(&msg);
#if 0    
	if (result) {
		instr->state = MTD_ERASE_FAILED;
	}
	else {
		instr->state = MTD_ERASE_DONE;
		mtd_erase_callback(instr);
	}
#endif    
	return result;;
}

/**
 * fcspi_write - [MTD Interface] write to flash part
 * @mtd:	MTD device structure
 * @to:		offset to write to
 * @len:	number of bytes to write
 * @retlen:	pointer to variable to store the number of written bytes
 * @buf:	the data to write
 *
 */
static int fcspi_write(struct mtd_info *mtd, loff_t to, size_t len,
		       size_t *retlen, const uint8_t *buf)
{
	struct fcspi *ctl = mtd_to_fcspi(mtd);
	struct device *dev = &ctl->pdev->dev;
	struct fcspi_message msg;

#ifdef DEBUG_FCSPI
	printk("%s: to %llx len %d buf %p\n", __func__, to, len, buf);
#endif

	if (retlen)
		*retlen = len;

	/* sanity checks */
	if (len == 0)
		return(0);

	if (to + len > ctl->mtd.size) {
		dev_err(dev, "Write request overflow\n");
		return -EINVAL;
	}

	msg.ctl = ctl;
	msg.type = FCSPI_WRITE_MSG;
	msg.result = 0;
	msg.flash_offset = to;
	msg.len = len;
	msg.buf = (uint8_t *)buf;
	return (fcspi_queue_work(&msg));
}


/**
 * fcspi_read - [MTD Interface] read from flash part
 * @mtd:	MTD device structure
 * @from:	offset to read from
 * @len:	number of bytes to read
 * @retlen:	pointer to variable to store the number of read bytes
 * @buf:	the databuffer to put data
 *
 */
static int fcspi_read(struct mtd_info *mtd, loff_t from, size_t len,
		      size_t *retlen, uint8_t *buf)
{
	struct fcspi *ctl = mtd_to_fcspi(mtd);
	struct fcspi_message msg;

	if (retlen)
		*retlen = len;

	if (!len)
		return 0;

	if (from + len > ctl->mtd.size)
		return -EINVAL;

	msg.ctl = ctl;
	msg.type = FCSPI_READ_MSG;
	msg.result = 0;
	msg.flash_offset = from;
	msg.len = len;
	msg.buf = buf;
	return (fcspi_queue_work(&msg));
}

/**
 * get Quasar extaddrmode - returns whether we are in extended
 *                          addressing, 0 if we are not, 4 (the DMA
 *                          address shift for some devices) is we are.
 *
 */
static int __init get_paeshift(void)
{
	void __iomem* mapped_address;
	int extaddrmode=0;

	mapped_address = ioremap(QUASAR_DDRGPF_EXTADDRMODE, 0x10);
	if (mapped_address) {
		if (readl((void *)mapped_address) & DDRGPF_EXTENDED_ADDRESS_MODE__EAM__MASK) {
			extaddrmode = 4;
			printk(KERN_INFO "Quasar in extended addressing mode.\n");
		}
		else {
			extaddrmode = 0;
			printk(KERN_INFO "Quasar in 32-bit addressing mode.\n");
		}

		iounmap(mapped_address);
	}
	return extaddrmode;
}

static int __init fcspi_probe(struct platform_device *pdev)
{
	struct fcspi *ctl;
	struct device_node *np = pdev->dev.of_node;
	struct flash_platform_data *fdata;
	struct mtd_part_parser_data	ppdata;
	struct mtd_partition *parts;
	struct mtd_info *mtd;
	struct fcspi_flash_info *info;
	unsigned nr_parts;
	int ret = -ENODEV;
	struct resource	*rregs;
	struct resource	*rirq;
	u32 val;

	fdata = dev_get_platdata(&pdev->dev);
	if (fdata == NULL && np == NULL) {
		dev_err(&pdev->dev, "platform_data missing!\n");
		return -ENODEV;
	}
	//ppdata.of_node = pdev->dev.of_node;

	/* Check for availability of necessary resource */
	rregs = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (rregs == NULL) {
		dev_err(&pdev->dev, "Unable to get SPI MEM resource\n");
		return -ENXIO;
	}

	/* Check for availability of necessary resource */
	rirq = platform_get_resource(pdev, IORESOURCE_IRQ, 0);
	if (rirq == NULL) {
		dev_err(&pdev->dev, "Unable to get SPI IRQ resource\n");
		return -ENXIO;
	}

	ctl = devm_kzalloc(&pdev->dev, sizeof(struct fcspi), GFP_KERNEL);
	if (!ctl) {
		dev_err(&pdev->dev, "Can't allocate control structure\n");
		return -ENOMEM;
	}
	
	ctl->paeshift = get_paeshift();

	platform_set_drvdata(pdev, ctl);

	ctl->irq = rirq->start;
	ctl->pdev = pdev;
	ctl->rregs = rregs->start;
	ctl->rregs_sz = resource_size(rregs);

	init_completion(&ctl->xfer_completion);
	INIT_WORK(&ctl->work, fcspi_work);
	INIT_LIST_HEAD(&ctl->queue);

	ctl->regs = devm_ioremap_resource(&pdev->dev, rregs);
	if (ctl->regs == NULL) {
		dev_err(&pdev->dev, "Unable to remap IO\n");
		ret = -ENXIO;
		goto err1;
	}

	if (devm_request_irq(&pdev->dev, rirq->start, fcspi_intr, 0,
			pdev->name, ctl)) {
		dev_err(&pdev->dev, "Unable to get fcspi IRQ\n");
		ret = -ENXIO;
		goto err2;
	}
	if (fdata) {
		parts = fdata->parts;
		nr_parts = fdata->nr_parts;
	}
	else {
		parts = NULL;
		nr_parts = 0;
	}

	mtd = &ctl->mtd;
	ctl->nandmode = fcspi_nand_mode(ctl);
	if (ctl->nandmode)
	{       
		val = fcspi_nand_get_id(ctl);
		info = (void *)fcspi_nand_info;
	}
	else
	{
		volatile u32 __iomem *memspec;

		memspec = &ctl->regs->fcspi_memspec;
		val = readl(memspec);
		info = (void *)fcspi_info;
	}

	for (; info->id; info++) {
		if (info->id == val) {
			break;
		}
	}
	if (!info->id) {
		dev_err(&pdev->dev, "Unknown flash device %08x\n", val);
		ret =  -ENODEV;
		goto err3;
	}

	/* allocate a buffer for non-aligned accesses */
	if (!(ctl->buffer = kmalloc(UBUFFSIZE, GFP_KERNEL))) {
		dev_err(&pdev->dev, "Can't allocate buffer\n");
		ret = -ENOMEM;
		goto err3;
	}

	printk("	FCSPI: nandmode: %d, id: %X, name: %s, size: %d, erase_size: %d, block_size: %d, flags: %X\n", 
			ctl->nandmode, info->id, info->name, info->size, info->erase_size, info->block_size, info->flags);
	if (ctl->nandmode)
	{
		// Disable HW-ECC 
		//fcspi_hwecc_enable(ctl, 0);
		printk("	Nand, SR_A0: %08X, SR_B0: %08X, SR_C0: %08X\n", fcspi_nand_get_feature(ctl, SR_A0), 
				fcspi_nand_get_feature(ctl, SR_B0), fcspi_nand_get_feature(ctl, SR_C0));
		ctl->bbmctl = NULL;
		if (info->flags & ONCHIP_BBM)
		{
			/* allocate an on-chip bbm management buffer */
			if ((ctl->bbmctl = kmalloc(sizeof(struct fcspi_bbm_ctrl) * BBM_ENTRIES(info->flags), GFP_KERNEL))) {
				u32 cicfg, cidr0=0, cidr1=0;
				u32 rddr0, rddr1;
				u32 bbm_lba, bbm_pba;
				int i;
				
				printk("	Nand on chip bbm lut search, entries= %d\n", BBM_ENTRIES(info->flags));
				/* get BBM LT entries from device */
				// Send the Read BBM LUT command and one dummy byte with long frame mode
				cicfg = FCSPI_CICFG__LFEN__MASK | FCSPI_CICFG__WIP__MASK | 
						FCSPI_CICFG__HOLDN__MASK | 2 << FCSPI_CICFG__LEN__SHIFT | GETBBMLUT_COMMAND;
				fcspi_custom_execute(ctl, cicfg, cidr0, cidr1, &rddr0, &rddr1);
				for (i=0; i<BBM_ENTRIES(info->flags); i++)
				{
					char mskname[4][16]={"available", "not valid", "valid link", "N/A"};
					u32 lpbmsk;
					
					cicfg = FCSPI_CICFG__LFEN__MASK | FCSPI_CICFG__WIP__MASK | 
							FCSPI_CICFG__HOLDN__MASK | 5 << FCSPI_CICFG__LEN__SHIFT;
					fcspi_custom_execute(ctl, cicfg, cidr0, cidr1, &rddr0, &rddr1);
					if (rddr0)
					{
						bbm_lba = ((rddr0 & 0x3F) << 8) | ((rddr0 & 0xFF00) >> 8);
						bbm_pba = ((rddr0 & 0x00FF0000) >> 8) | ((rddr0 & 0xFF000000) >> 24);
						lpbmsk = (rddr0 & 0xC0) >> 6;
						ctl->bbmctl[i].lba = bbm_lba;
						ctl->bbmctl[i].pba = bbm_pba;
						printk("	lut-%02d: lpa_msk: %d(%s), lpa: %d, pba: %d\n", i, lpbmsk, mskname[lpbmsk], bbm_lba, bbm_pba);
					}
					else
					{
						ctl->bbmctl[i].lba = 0;
						ctl->bbmctl[i].pba = 0;
					}
				} 
				// Finish the long frame mode
				cicfg = FCSPI_CICFG__LFEN__MASK | FCSPI_CICFG__WIP__MASK | FCSPI_CICFG__LFSTOP__MASK |
							FCSPI_CICFG__HOLDN__MASK | 1 << FCSPI_CICFG__LEN__SHIFT;
				fcspi_custom_execute(ctl, cicfg, cidr0, cidr1, &rddr0, &rddr1);
			}
			else {
				dev_warn(&pdev->dev, "Can't allocate onchip bbm management buffer\n");
			}
		}
	}
	
	ctl->workqueue = 
		create_singlethread_workqueue(DRIVER_NAME);

	if (ctl->workqueue == NULL) {
		dev_err(&pdev->dev, "Unable to create workqueue\n");
		ret = -ENOMEM;
		goto err4;
	}
	ctl->flags = info->flags;
	ctl->sector_size = info->erase_size;
	ctl->block_size = info->block_size;
	ctl->size = info->size;
	mtd->dev.of_node = pdev->dev.of_node;
	mtd->type = MTD_NORFLASH;
	mtd->writesize = ctl->nandmode ? NAND_PAGE_SIZE : 1;
	mtd->flags = ctl->nandmode ? MTD_CAP_NANDFLASH : MTD_CAP_NORFLASH;
	mtd->_erase = fcspi_erase;
	mtd->_read = fcspi_read;
	mtd->_write = fcspi_write;
	mtd->name = info->name;
	mtd->owner = THIS_MODULE;
	mtd->dev.parent = &pdev->dev;
	mtd->size = info->size;
	mtd->erasesize = info->erase_size;
	if (mtd->erasesize < 16 * 1024)
		mtd->erasesize = 16 * 1024;

	spin_lock_init(&ctl->lock);

	if (!mtd_device_parse_register(mtd, NULL, &ppdata,
			parts, nr_parts)) {
		return 0;
	}

err4:
	kfree(ctl->buffer);
	kfree(ctl->bbmctl);
err3:
	free_irq(ctl->irq, ctl);
	iounmap(ctl->regs);
err2:
	release_mem_region(rregs->start, resource_size(rregs));
err1:
	kfree(ctl);
	platform_set_drvdata(pdev, NULL);
	return ret;
}

static int fcspi_remove(struct platform_device *pdev)
{
	struct fcspi *ctl;
	unsigned long flags;
	printk(KERN_INFO "%s\n", __func__);
	ctl = platform_get_drvdata(pdev);

	spin_lock_irqsave(&ctl->lock, flags);
	ctl->state |= SUSPND;
	spin_unlock_irqrestore(&ctl->lock, flags);

	while (ctl->state & FCSPIBUSY)
		msleep(10);

	destroy_workqueue(ctl->workqueue);
	free_irq(ctl->irq, ctl);
	iounmap(ctl->regs);
	release_mem_region(ctl->rregs, ctl->rregs_sz);
	kfree(ctl->bbmctl);
	kfree(ctl->buffer);
	kfree(ctl);
	platform_set_drvdata(pdev, NULL);
	return 0;
}

static int fcspi_suspend(struct device *dev)
{
	//printk(KERN_INFO "%s\n", __func__);
	return 0;
}

static int fcspi_resume(struct device *dev)
{
	//printk(KERN_INFO "%s\n", __func__);
	return 0;
}

static SIMPLE_DEV_PM_OPS(quasar_fcspi_pm_ops, fcspi_suspend, fcspi_resume);

static const struct of_device_id quasar_fcspi_id_table[] = {
	{ .compatible = "qbit,qbfcspi" },
	{}
};
MODULE_DEVICE_TABLE(of, quasar_fcspi_id_table);

static struct platform_driver quasar_fcspi_driver_probe = {
	.driver = {
		.name = "fcspi",
		.owner	= THIS_MODULE,
		.bus = &platform_bus_type,
		.owner = THIS_MODULE,
		.of_match_table = of_match_ptr(quasar_fcspi_id_table),
		.pm = &quasar_fcspi_pm_ops,
	},
	.probe = fcspi_probe,
	.remove = fcspi_remove,
};
module_platform_driver(quasar_fcspi_driver_probe);

MODULE_AUTHOR("QBit Semiconductor LTD.");
MODULE_DESCRIPTION("MTD SPI driver for FCSPI controller");
MODULE_LICENSE("GPL v2");
