/*
 * SDHCI Support for Quasar Soc
 *
 * Copyright (c) 2014, 2015, The Linux Foundation. All rights reserved.
 * 
 * Portions Copyright (c) 2009 Intel Corporation
 * Portions Copyright (c) 2007, 2011 Freescale Semiconductor, Inc.
 * Portions Copyright (c) 2009 MontaVista Software, Inc.
 * Portions Copyright (C) 2010 ST Microelectronics
 * Copyright (c) 2014 Cambridge Silicon Radio Ltd.
 * 
 * Copyright (c) 2017-2018, QBit Semiconductor LTD.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */
// =========================================================
//
//  $DateTime: 2022/01/06 10:11:28 $
//  $Change: 56812 $
//
// =========================================================
#include <linux/delay.h>
#include <linux/highmem.h>
#include <linux/platform_device.h>

#include <linux/mmc/host.h>

#include <linux/io.h>
#include "sdhci.h"
#include "sdhci-quasar.h"

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/jiffies.h>
#include <linux/platform_device.h>
#include <linux/of_platform.h>
#include <linux/io.h>
#ifdef __KERNEL__
#include <linux/interrupt.h>
#include <linux/device.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/poll.h>
#include <linux/mutex.h>
#include <linux/sched.h>
#include <linux/slab.h>
#endif
#include <linux/dma-mapping.h>
#include <linux/time.h>
#include <linux/string.h>

#define SDIO_XIN0_CLK			0
#define SDIO_SYSPLL_CLKOUTD1	1

#define SDM_CLK			125000000	//125MHz
#define SYS_CLK			2500000000	//2500MHz
#define MAX_HOST_CLK 	200000000	//200MHz


#define SETIOCTL_P(PIN, VAL) do {				\
		u32 ioctl = ioread32(PIN);				\
		ioctl &= IOCTL_SD0CMD__P__INV_MASK;		\
		ioctl |= VAL << IOCTL_SD0CMD__P__SHIFT; \
		iowrite32(ioctl, PIN);					\
	} while (0)

#define SETIOCTL_E(PIN, VAL) do {				\
		u32 ioctl = ioread32(PIN);				\
		ioctl &= IOCTL_SD0CMD__E__INV_MASK;		\
		ioctl |= VAL << IOCTL_SD0CMD__E__SHIFT;	\
		iowrite32(ioctl, PIN);					\
	} while (0)

#define SETIOCTL_SR(PIN, VAL) do {				\
		u32 ioctl = ioread32(PIN);				\
		ioctl &= IOCTL_SD0CMD__SR__INV_MASK;	\
		ioctl |= VAL << IOCTL_SD0CMD__SR__SHIFT;\
		iowrite32(ioctl, PIN);					\
	} while (0)

#define SETIOCTL_SMT(PIN, VAL) do {					\
		u32 ioctl = ioread32(PIN);					\
		ioctl &= IOCTL_SD0CMD__SMT__INV_MASK;		\
		ioctl |= VAL << IOCTL_SD0CMD__SMT__SHIFT;	\
		iowrite32(ioctl, PIN);						\
	} while (0)

struct sdhci_qb63 {
	struct platform_device	*pdev;
	unsigned int sdmclk;
	unsigned char id;
};

#define MAX_DIVISOR 17
typedef struct tag_clk_div_ctrl {
	unsigned char	neg_pwl_start;
	unsigned char	neg_pwh_start;
	unsigned char	pos_pwl_start;
	unsigned char	div_value;
} CLK_DIV_CTRL, *PCLK_DIV_CTRL;

static CLK_DIV_CTRL clkdivctrl_table[MAX_DIVISOR]={
	{0, 0, 0, 0},// padding and not applicable
	{0, 0, 0, 1},
	{0, 0, 2, 2},
	{2, 1, 2, 3},
	{0, 0, 3, 4},
	{3, 1, 3, 5},
	{0, 0, 4, 6},
	{4, 1, 4, 7},
	{0, 0, 5, 8},
	{5, 1, 5, 9},
	{0, 0, 6,10},
	{6, 1, 6,11},
	{0, 0, 7,12},
	{7, 1, 7,13},
	{0, 0, 8,14},
	{8, 1, 8,15},
	{4, 3, 2, 5}
};

static unsigned char core_initialized[3] = {0, 0, 0};

/*****************************************************************************\
 *                                                                           *
 * SDHCI core callbacks                                                      *
 *                                                                           *
\*****************************************************************************/
static unsigned int qb63sd_get_max_clk(struct sdhci_host *host)
{
	unsigned int qb63sd_sdmclk = SDM_CLK;

	return qb63sd_sdmclk;
}

void qb63sd_reset(struct sdhci_host *host, u8 mask)
{
	unsigned int val = 0;

	val = ioread32(host->ioaddr + SDHCI_CLOCK_CONTROL);
	val |= SDHCI_CLOCK_INT_EN;
	iowrite32(val, host->ioaddr + SDHCI_CLOCK_CONTROL);
	while (1) {
		val = ioread32(host->ioaddr + SDHCI_CLOCK_CONTROL);
		if (val & SDHCI_CLOCK_INT_STABLE) {
			break;
		}
	}

	sdhci_reset(host, mask);
}


static void qb63sd_set_hostclk(struct sdhci_host *host, u32 host_clk)
{
	struct sdhci_qb63 *qb63_host = sdhci_priv(host);
	struct platform_device *pdev = qb63_host->pdev;
	unsigned int reg_val = 0;
	void __iomem * clk_dis_ctl = NULL;
	void __iomem * clk_mux_ctl = NULL;
	void __iomem * clk_div_ctl = NULL;
	struct resource *iomem1;
	struct resource *iomem2;
	struct resource *iomem3;
	unsigned char source = SDIO_SYSPLL_CLKOUTD1, div_value = 5;
	unsigned char slot = 0;

	if (!strcmp(pdev->name,"4010000.sdhci")) {
		slot = SD0;
	} else if (!strcmp(pdev->name,"40c0000.sdhci")){
		slot = SD1;
	} else if (!strcmp(pdev->name,"4150000.sdhci")){
		slot = SD2;
	} else {
		dev_err(&pdev->dev, "device address is not supported\n");
		return;
	}

	if (qb63_host->sdmclk == host_clk)
		return;

	for (div_value = 1; div_value < MAX_DIVISOR; div_value++) {
		if ((SYS_CLK / div_value) / 4 <= host_clk) {
			break;
		}
	}

	if (host_clk >= MAX_HOST_CLK) {
		div_value = 16;
		host_clk = MAX_HOST_CLK;
	}

	iomem1 = platform_get_resource(pdev, IORESOURCE_MEM, 1);
	if (!iomem1) {
		dev_err(&pdev->dev, "cannot get resource\n");
		return;
	}
	if (!request_mem_region(iomem1->start, resource_size(iomem1),
		mmc_hostname(host->mmc))) {
		dev_err(&pdev->dev, "cannot request region\n");
		return;
	}
	clk_dis_ctl = ioremap(iomem1->start, resource_size(iomem1));

	iomem2 = platform_get_resource(pdev, IORESOURCE_MEM, 2);
	if (!iomem2) {
		dev_err(&pdev->dev, "cannot get resource\n");
		return;
	}
	if (!request_mem_region(iomem2->start, resource_size(iomem2),
		mmc_hostname(host->mmc))) {
		dev_err(&pdev->dev, "cannot request region\n");
		return;
	}
	clk_mux_ctl = ioremap(iomem2->start, resource_size(iomem2));

	iomem3 = platform_get_resource(pdev, IORESOURCE_MEM, 3);
	if (!iomem3) {
		dev_err(&pdev->dev, "cannot get resource\n");
		return;
	}
	if (!request_mem_region(iomem3->start, resource_size(iomem3),
		mmc_hostname(host->mmc))) {
		dev_err(&pdev->dev, "cannot request region\n");
		return;
	}
	clk_div_ctl = ioremap(iomem3->start, resource_size(iomem3));

	reg_val = ioread32(clk_dis_ctl);
	reg_val |= SDIO0CLKDISCTRL__SDIO0_SDM4__MASK;
	iowrite32(reg_val, clk_dis_ctl);
	do {
		reg_val = ioread32(clk_dis_ctl + 4);
	}
	while (!(reg_val & SDIO0CLKDISSTAT__SDIO0_SDM4__MASK));

	reg_val = ioread32(clk_div_ctl);
	reg_val &= SYSCG_CLKDIVCTRL_SDIO0_SDM4__DIV_VALUE__INV_MASK;
	iowrite32(reg_val, clk_div_ctl);

	reg_val = ioread32(clk_mux_ctl);
	reg_val &= (~(SYSCG_CLKMUXCTRL1__SDIO0_SDM4__MASK << slot));
	reg_val |= (source << (SYSCG_CLKMUXCTRL1__SDIO0_SDM4__SHIFT + slot));
	iowrite32(reg_val, clk_mux_ctl);

	reg_val = ioread32(clk_div_ctl);
	if ((div_value >= 1) && (div_value <= (MAX_DIVISOR - 1))) {
		reg_val &= (~SYSCG_CLKDIVCTRL_SDIO0_SDM4__NEG_PWL_START__MASK);
		reg_val &= (~SYSCG_CLKDIVCTRL_SDIO0_SDM4__NEG_PWH_START__MASK);
		reg_val &= (~SYSCG_CLKDIVCTRL_SDIO0_SDM4__POS_PWL_START__MASK);
		reg_val |= (clkdivctrl_table[div_value].neg_pwl_start <<
					SYSCG_CLKDIVCTRL_SDIO0_SDM4__NEG_PWL_START__SHIFT);
		reg_val |= (clkdivctrl_table[div_value].neg_pwh_start <<
					SYSCG_CLKDIVCTRL_SDIO0_SDM4__NEG_PWH_START__SHIFT);
		reg_val |= (clkdivctrl_table[div_value].pos_pwl_start <<
					SYSCG_CLKDIVCTRL_SDIO0_SDM4__POS_PWL_START__SHIFT);
	}
	else {
		dev_err(&pdev->dev, "unsupported divisor value.\n");
	}
	iowrite32(reg_val, clk_div_ctl);
	reg_val |= (clkdivctrl_table[div_value].div_value << SYSCG_CLKDIVCTRL_SDIO0_SDM4__DIV_VALUE__SHIFT);
	iowrite32(reg_val, clk_div_ctl);

	reg_val = ioread32(clk_dis_ctl);
	reg_val &= SDIO0CLKDISCTRL__SDIO0_SDM4__INV_MASK;
	iowrite32(reg_val, clk_dis_ctl);
	do {
		reg_val = ioread32(clk_dis_ctl + 4);
	}
	while (reg_val & (SDIO0CLKDISSTAT__SDIO0_SDM4__MASK));

	iounmap(clk_div_ctl);
	release_mem_region(iomem3->start, resource_size(iomem3));
	iounmap(clk_mux_ctl);
	release_mem_region(iomem2->start, resource_size(iomem2));
	iounmap(clk_dis_ctl);
	release_mem_region(iomem1->start, resource_size(iomem1));

	qb63_host->sdmclk = host_clk;

	return;
}

static struct sdhci_ops sdhci_qb63_ops = {
	.get_max_clock = qb63sd_get_max_clk,
	.set_clock = sdhci_set_clock,
	.set_bus_width = sdhci_set_bus_width,
	.reset = qb63sd_reset,
	.set_uhs_signaling = sdhci_set_uhs_signaling,
	.set_host_clk = qb63sd_set_hostclk,
};

/*****************************************************************************\
 *                                                                           *
 * Device probing/removal                                                    *
 *                                                                           *
\*****************************************************************************/

static int qb63_sdio_init(struct platform_device *pdev, void __iomem * ioaddr,
							struct sdhci_host *host)
{
	unsigned int reg_val = 0;
	void __iomem * ioaddr0 = NULL;
	void __iomem * clk_dis_ctl = NULL;
	void __iomem * clk_mux_ctl = NULL;
	void __iomem * clk_div_ctl = NULL;
	struct resource *iomem0;
	struct resource *iomem1;
	struct resource *iomem2;
	struct resource *iomem3;
	unsigned char source = SDIO_SYSPLL_CLKOUTD1, div_value = 5;
	unsigned char slot = 0;

	if (!strcmp(pdev->name,"4010000.sdhci")) {
		slot = SD0;
	} else if (!strcmp(pdev->name,"40c0000.sdhci")){
		slot = SD1;
	} else if (!strcmp(pdev->name,"4150000.sdhci")){
		slot = SD2;
	} else {
		dev_err(&pdev->dev, "device address is not supported\n");
		goto err_reset;
	}

	if (!core_initialized[slot]) {

		core_initialized[slot] = 1;

		/* set sdmclk */
		iomem1 = platform_get_resource(pdev, IORESOURCE_MEM, 1);
		if (!iomem1) {
			goto err_reset;
		}
		if (!request_mem_region(iomem1->start, resource_size(iomem1),
			mmc_hostname(host->mmc))) {
			dev_err(&pdev->dev, "cannot request region\n");
			goto err_reset;
		}
		clk_dis_ctl = ioremap(iomem1->start, resource_size(iomem1));

		iomem2 = platform_get_resource(pdev, IORESOURCE_MEM, 2);
		if (!iomem2) {
			goto err_reset;
		}
		if (!request_mem_region(iomem2->start, resource_size(iomem2),
			mmc_hostname(host->mmc))) {
			dev_err(&pdev->dev, "cannot request region\n");
			goto err_reset;
		}
		clk_mux_ctl = ioremap(iomem2->start, resource_size(iomem2));

		iomem3 = platform_get_resource(pdev, IORESOURCE_MEM, 3);
		if (!iomem3) {
			goto err_reset;
		}
		if (!request_mem_region(iomem3->start, resource_size(iomem3),
			mmc_hostname(host->mmc))) {
			dev_err(&pdev->dev, "cannot request region\n");
			goto err_reset;
		}
		clk_div_ctl = ioremap(iomem3->start, resource_size(iomem3));

		reg_val = ioread32(clk_dis_ctl);
		reg_val |= SDIO0CLKDISCTRL__SDIO0_SDM4__MASK;
		iowrite32(reg_val, clk_dis_ctl);
		do {
			reg_val = ioread32(clk_dis_ctl + 4);
		}
		while (!(reg_val & SDIO0CLKDISSTAT__SDIO0_SDM4__MASK));

		reg_val = ioread32(clk_div_ctl);
		reg_val &= SYSCG_CLKDIVCTRL_SDIO0_SDM4__DIV_VALUE__INV_MASK;
		iowrite32(reg_val, clk_div_ctl);

		reg_val = ioread32(clk_mux_ctl);
		reg_val &= (~(SYSCG_CLKMUXCTRL1__SDIO0_SDM4__MASK << slot));
		reg_val |= (source << (SYSCG_CLKMUXCTRL1__SDIO0_SDM4__SHIFT + slot));
		iowrite32(reg_val, clk_mux_ctl);

		reg_val = ioread32(clk_div_ctl);
		if ((div_value >= 1) && (div_value <= (MAX_DIVISOR - 1))) {
			reg_val &= (~SYSCG_CLKDIVCTRL_SDIO0_SDM4__NEG_PWL_START__MASK);
			reg_val &= (~SYSCG_CLKDIVCTRL_SDIO0_SDM4__NEG_PWH_START__MASK);
			reg_val &= (~SYSCG_CLKDIVCTRL_SDIO0_SDM4__POS_PWL_START__MASK);
			reg_val |= (clkdivctrl_table[div_value].neg_pwl_start <<
						SYSCG_CLKDIVCTRL_SDIO0_SDM4__NEG_PWL_START__SHIFT);
			reg_val |= (clkdivctrl_table[div_value].neg_pwh_start <<
						SYSCG_CLKDIVCTRL_SDIO0_SDM4__NEG_PWH_START__SHIFT);
			reg_val |= (clkdivctrl_table[div_value].pos_pwl_start <<
						SYSCG_CLKDIVCTRL_SDIO0_SDM4__POS_PWL_START__SHIFT);
		}
		else {
			dev_err(&pdev->dev, "unsupported divisor value.\n");
		}
		iowrite32(reg_val, clk_div_ctl);
		reg_val |= (div_value << SYSCG_CLKDIVCTRL_SDIO0_SDM4__DIV_VALUE__SHIFT);
		iowrite32(reg_val, clk_div_ctl);

		reg_val = ioread32(clk_dis_ctl);
		reg_val &= SDIO0CLKDISCTRL__SDIO0_SDM4__INV_MASK;
		iowrite32(reg_val, clk_dis_ctl);
		do {
			reg_val = ioread32(clk_dis_ctl + 4);
		}
		while (reg_val & (SDIO0CLKDISSTAT__SDIO0_SDM4__MASK));

		iounmap(clk_div_ctl);
		release_mem_region(iomem3->start, resource_size(iomem3));
		iounmap(clk_mux_ctl);
		release_mem_region(iomem2->start, resource_size(iomem2));
		iounmap(clk_dis_ctl);
		release_mem_region(iomem1->start, resource_size(iomem1));
	}

	/* set SDIOX_EXTCTL */
	iomem0 = platform_get_resource(pdev, IORESOURCE_MEM, 4);
	if (!iomem0) {
		goto err_reset;
	}
	if (!request_mem_region(iomem0->start, resource_size(iomem0),
		mmc_hostname(host->mmc))) {
		dev_err(&pdev->dev, "cannot request region\n");
		goto err_reset;
	}
	ioaddr0 = ioremap(iomem0->start, resource_size(iomem0));

	reg_val = ioread32(ioaddr0);
	reg_val |= (0 << SDIO0_EXTCTL__CLE_OVL__SHIFT);
	reg_val |= (1 << SDIO0_EXTCTL__CD_OVL__SHIFT);
	reg_val |= (1 << SDIO0_EXTCTL__WP_OVL__SHIFT);

	iowrite32(reg_val, ioaddr0);
	iounmap(ioaddr0);
	release_mem_region(iomem0->start, resource_size(iomem0));

	//set drive strength 
	iomem0 = platform_get_resource(pdev, IORESOURCE_MEM, 5);
	if (!iomem0) {
		goto err_reset;
	}
	if (!request_mem_region(iomem0->start, resource_size(iomem0),
		mmc_hostname(host->mmc))) {
		dev_err(&pdev->dev, "cannot request region\n");
	} 
	ioaddr0 = ioremap(iomem0->start, resource_size(iomem0));
/*	set ipad in boot code.
	#if 0
	//12mA is set here for default
	SETIOCTL_E(ioaddr0     , 2);
	SETIOCTL_E(ioaddr0 +  4, 2);
	SETIOCTL_E(ioaddr0 +  8, 2);
	SETIOCTL_E(ioaddr0 + 12, 2);
	SETIOCTL_E(ioaddr0 + 16, 2);
	SETIOCTL_E(ioaddr0 + 20, 2);	

	SETIOCTL_SR(ioaddr0     , 1);
	SETIOCTL_SR(ioaddr0 +  4, 1);
	SETIOCTL_SR(ioaddr0 +  8, 1);
	SETIOCTL_SR(ioaddr0 + 12, 1);
	SETIOCTL_SR(ioaddr0 + 16, 1);
	SETIOCTL_SR(ioaddr0 + 20, 1);

	SETIOCTL_SMT(ioaddr0     , 0);
	SETIOCTL_SMT(ioaddr0 +  4, 0);
	SETIOCTL_SMT(ioaddr0 +  8, 0);
	SETIOCTL_SMT(ioaddr0 + 12, 0);
	SETIOCTL_SMT(ioaddr0 + 16, 0);
	SETIOCTL_SMT(ioaddr0 + 20, 0); //*/

	iounmap(ioaddr0);
	release_mem_region(iomem0->start, resource_size(iomem0));

	//set VDL 
	if (slot == 0) {
		ioaddr0 = ioremap(SYS_SPARE, 0x4);
		reg_val = ioread32(ioaddr0);
		iowrite32(reg_val | 0x00000009, ioaddr0);
		iounmap(ioaddr0);
	} else if (slot == 1) {
		ioaddr0 = ioremap(SYS_SPARE, 0x4);
		reg_val = ioread32(ioaddr0);
		iowrite32(reg_val | 0x00000090, ioaddr0);
		iounmap(ioaddr0);
	} else { //slot == 2
		ioaddr0 = ioremap(DDRGPF_DDR_SPARE, 0x4);
		reg_val = ioread32(ioaddr0);
		iowrite32(reg_val | 0x00000009, ioaddr0);
		iounmap(ioaddr0);
	}

	return 0;

err_reset:
	dev_err(&pdev->dev, "qb63_sdio_init failed.\n");
	return -1;
}

static int sdhci_qb63_probe(struct platform_device *pdev)
{
	struct sdhci_host *host;
	struct sdhci_qb63 *qb63_host;
	struct resource *iomem;
	void __iomem * ioaddr = NULL;
	int ret;
	static unsigned char instance = SD0;

	iomem = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!iomem) {
		ret = -ENOMEM;
		goto err;
	}

	if (resource_size(iomem) < 0x100)
		dev_err(&pdev->dev, "Invalid iomem size. You may "
			"experience problems.\n");

	if (!request_mem_region(iomem->start, resource_size(iomem),
		"mmc")) {
		dev_err(&pdev->dev, "cannot request region\n");
		ret = -EBUSY;
		goto err;
	}

	ioaddr = ioremap(iomem->start, resource_size(iomem));

	//add vendor specific settings
	iowrite32((5 << VENDOR_CDR_DTIMEC_SHIFT), ioaddr + VENDOR_MACRO_CARD_DETECT);
	iowrite32(0x3, ioaddr + VENDOR_RESET_CONTROL_OFFSET);
	while ((ioread32(ioaddr + VENDOR_INIT_STATUS_OFFSET) & RST_STATUS) != RST_STATUS) {
	}
	while ((ioread32(ioaddr + VENDOR_INIT_STATUS_OFFSET) & BCLK_10_CYCLE) != BCLK_10_CYCLE) {
	}
	iowrite32(0x7, ioaddr + VENDOR_AHB_CONFIG_OFFSET);

	if (pdev->dev.parent) {
		pdev->dev.parent->coherent_dma_mask = DMA_BIT_MASK(32);
		host = sdhci_alloc_host(pdev->dev.parent, sizeof(struct sdhci_qb63));
	}
	else
		host = sdhci_alloc_host(&pdev->dev, sizeof(struct sdhci_qb63));

	if (IS_ERR(host)) {
		ret = PTR_ERR(host);
		goto err;
	}

	host->hw_name = pdev->name;
	host->ops = &sdhci_qb63_ops;
	host->quirks = SDHCI_QUIRK_CAP_CLOCK_BASE_BROKEN
					| SDHCI_QUIRK_BROKEN_ADMA_ZEROLEN_DESC
					| SDHCI_QUIRK_NO_ENDATTR_IN_NOPDESC;
	host->quirks2 = SDHCI_QUIRK2_NO_1_8_V 
					| SDHCI_QUIRK2_BROKEN_HS200
					| SDHCI_QUIRK2_SUPPORT_SINGLE
					| SDHCI_QUIRK2_TUNING_WORK_AROUND;
	host->irq = platform_get_irq(pdev, 0);

	host->ioaddr = ioaddr;
	if (!host->ioaddr) {
		dev_err(&pdev->dev, "failed to remap registers\n");
		ret = -ENOMEM;
		goto err_remap;
	}

	qb63_host = sdhci_priv(host);
	qb63_host->pdev = pdev;
	qb63_host->sdmclk = SDM_CLK;			
	qb63_host->id = instance++;

	ret = qb63_sdio_init(pdev, ioaddr, host);
	if(ret)
		goto err_add_host;

	//MMC always has power
	host->mmc->pm_caps |= MMC_PM_KEEP_POWER;
	host->mmc->caps |= MMC_CAP_8_BIT_DATA;	//Add it for 8-bit bus width.

	ret = sdhci_add_host(host);
	if (ret)
		goto err_add_host;

	platform_set_drvdata(pdev, host);

	return 0;

err_add_host:
	iounmap(ioaddr);
err_remap:
	release_mem_region(iomem->start, resource_size(iomem));
	sdhci_free_host(host);
err:
	printk(KERN_ERR"Probing of sdhci-qb63 %s failed: %d\n", pdev->name, ret);
	return ret;
}

static int sdhci_qb63_remove(struct platform_device *pdev)
{
	struct sdhci_host *host = platform_get_drvdata(pdev);
	struct resource *iomem = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	int dead;
	u32 scratch;

	dead = 0;
	scratch = readl(host->ioaddr + SDHCI_INT_STATUS);
	if (scratch == (u32)-1)
		dead = 1;

	sdhci_remove_host(host, dead);
	iounmap(host->ioaddr);
	release_mem_region(iomem->start, resource_size(iomem));
	sdhci_free_host(host);
	platform_set_drvdata(pdev, NULL);

	return 0;
}

static const struct of_device_id qb63_id_table[] = {
	{ .compatible = "qbit,qb63-sdhci" },
	{}
};

MODULE_DEVICE_TABLE(of, qb63_id_table);

static struct platform_driver sdhci_qb63_driver = {
	.driver = {
		.name	= "qb63-sdhci",
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(qb63_id_table),
	},
	.probe		= sdhci_qb63_probe,
	.remove		= sdhci_qb63_remove,
};

/*****************************************************************************\
 *                                                                           *
 * Driver init/exit                                                          *
 *                                                                           *
\*****************************************************************************/
static int __init sdhci_qb63_init(void)
{
	return platform_driver_register(&sdhci_qb63_driver);
}

static void __exit sdhci_qb63_exit(void)
{
	platform_driver_unregister(&sdhci_qb63_driver);
}

module_init(sdhci_qb63_init);
module_exit(sdhci_qb63_exit);

MODULE_DESCRIPTION("Secure Digital Host Controller Interface QB63 driver");
MODULE_LICENSE("GPL v2");
MODULE_ALIAS("qb63:sdhci");

