/*
 * drivers/clocksource/quasar_timer.c
 *
 * Copyright (c) 2014, 2015, The Linux Foundation. All rights reserved.
 *
 * Copyright (c) 2014 Cambridge Silicon Radio Ltd.
 * 
 * Copyright (c) 2017-2018, QBit Semiconductor LTD.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */
// =========================================================
//
//  $DateTime: 2022/01/27 08:19:40 $
//  $Change: 57409 $
//
// =========================================================

#if 0
#include <linux/clockchips.h>
#include <linux/clocksource.h>
#include <linux/interrupt.h>
#include <linux/irqreturn.h>
#include <linux/sched_clock.h>

#include <linux/of_address.h>
#include <linux/of_irq.h>

#include <linux/slab.h>
#include "timer-of.h"

#else
#include <linux/bitops.h>
#include <linux/clockchips.h>
#include <linux/clocksource.h>
#include <linux/interrupt.h>
#include <linux/irqreturn.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/of_address.h>
#include <linux/of_irq.h>
#include <linux/of_platform.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/sched_clock.h>
#include <linux/clocksource.h>
#include <linux/bitops.h>
#include <asm/irq.h>
#endif

/* Quasar timer block has:

   - High resolution timer 0 (64-bit) - used for HRT if configured
   - High resolution timer 1 (32-bit) - used as clocksource (free running schedule clock)
   - System timer 0 - used as clockevent (global timer)
   - System timer 1 - CPU 0 local timer
   - System timer 2 - CPU 1 local timer (if we have that many CPUs),
   - System timer 3 -  ...
   - System timer 4 -  ...
   - System timer 5 -  ...
   - System timer 6 -  ... up to 5 cpus if needed
   - Watchdog timer - unused by this driver.

*/
#define QUASAR_TIMER_MAX_CPU 5 /* for local timer purposes */

#define XTAL_MHZ 25
#define XIN0_CLK (XTAL_MHZ*1000*1000)

// TBD: modify for 6600
#if defined(CONFIG_SOC_QUASAR6300) || defined(CONFIG_SOC_QUASAR6600)
// SYSCG flags
#define SYSCG_BASE       	                        0x04111000
#define SYSCG_SIZE                                  0x00000300
#define SYSCG_CLKSWCTRL1_OFF	                    0x04
#define SYSCG_CLKSWCTRL1__CCI__SHIFT                6
#define SYSCG_CLKSWCTRL1__CCI__MASK                 0x000000C0
#define SYSCG_CLKSWSTAT1__SYS_IS_MUX_CLK__MASK	    0x00040000
#define SYSCG_CLKSWSTAT1__SYS_IS_LP_CLK__MASK	    0x00020000
#define SYSCG_CLKSWSTAT1__SYS_IS_XIN0_CLK__MASK	    0x00010000
#define SYSCG_CLKMUXCTRL1__SYS__SHIFT       	    0
#define SYSCG_CLKMUXCTRL1__SYS__MASK        	    0x00000003
#define SYSCG_CLKDIVCTRL_CCI_OFF                    0x68    
#define SYSCG_CLKDIVCTRL_CCI__DIV_VALUE__SHIFT      0
#define SYSCG_CLKDIVCTRL_CCI__DIV_VALUE__MASK       0x0000000F
#define SYSCG_CLKDIVCTRL_CCI__NEG_PWH_START__SHIFT  16
#define SYSCG_CLKDIVCTRL_CCI__NEG_PWH_START__MASK   0x000F0000
#define SYSCG_CLKMUXCTRL1_OFF                       0x50
#define SYSCG_CLKMUXCTRL1__CCI__SHIFT               1
#define SYSCG_CLKMUXCTRL1__CCI__MASK                0x00000002
#define SYSCG_CLKDIVCTRL_LP_OFF                     0x60
#define SYSCG_CLKDIVCTRL_LP__DIV_VALUE__SHIFT       0
#define SYSCG_CLKDIVCTRL_LP__DIV_VALUE__MASK        0x0000003F
#define SYSCG_CLKDIVCTRL_LP__NEG_PWH_START__SHIFT   16
#define SYSCG_CLKDIVCTRL_LP__NEG_PWH_START__MASK    0x003F0000  
#define SYSCG_SYSSSCCTRL1_OFF                       0x280
#define SYSCG_SYSSSCCTRL1__NRSTDS__SHIFT            11
#define SYSCG_SYSSSCCTRL1__NRSTDS__MASK             0x00000800
#define SYSCG_SYSSSCCTRL1__SSC_EN__SHIFT            5
#define SYSCG_SYSSSCCTRL1__SSC_EN__MASK             0x00000020
#define SYSCG_SYSPLLCTRL1_OFF                       0x260
#define SYSCG_SYSPLLCTRL1__SEL_FDIV4__SHIFT         15
#define SYSCG_SYSPLLCTRL1__SEL_FDIV4__MASK          0x00008000
#define SYSCG_SYSSSCCTRL2_OFF                       0x284
#define SYSCG_SYSSSCCTRL2__SSC_J__SHIFT             24
#define SYSCG_SYSSSCCTRL2__SSC_J__MASK              0x7F000000
#define SYSCG_SYSSSCCTRL2__SSC_K__SHIFT             0
#define SYSCG_SYSSSCCTRL2__SSC_K__MASK              0x000FFFFF

// DDRCG flags
#define DDRCG_BASE       	                        0x04115000
#define DDRCG_SIZE                                  0x00000080
#define DDRCG_DDRSSCCTRL1_OFF                       0x60
#define DDRCG_DDRSSCCTRL1__NRSTDS__SHIFT            11
#define DDRCG_DDRSSCCTRL1__NRSTDS__MASK             0x00000800
#define DDRCG_DDRSSCCTRL1__SSC_EN__SHIFT            5
#define DDRCG_DDRSSCCTRL1__SSC_EN__MASK             0x00000020
#define DDRCG_DDRPLLCTRL1_OFF                       0x40
#define DDRCG_DDRPLLCTRL1__SEL_FDIV4__SHIFT         15
#define DDRCG_DDRPLLCTRL1__SEL_FDIV4__MASK          0x00008000
#define DDRCG_DDRSSCCTRL2_OFF                       0x64
#define DDRCG_DDRSSCCTRL2__SSC_J__SHIFT             24
#define DDRCG_DDRSSCCTRL2__SSC_J__MASK              0x7F000000
#define DDRCG_DDRSSCCTRL2__SSC_K__SHIFT             0
#define DDRCG_DDRSSCCTRL2__SSC_K__MASK              0x000FFFFF

// TBD: consider fraction mode pll calculation in line 157, 161
//*****************************************************************************
static u32 sysPllOutputFrequency(int div)
{
    u8 __iomem *sscgbase;
    //u32 frac;
    u32 Fpllclk = 0;
    u32 Ssc = 0;     // default disable
    u32 FraMode = 0;
    u32 SelFdiv4 = 0, Ssc_j = 0, Ssc_k = 0;
    u32 Frefclk = XIN0_CLK;

    // This register controls the SNI MANT28PLL3041DSV18 PLL 
    // that is used in conjunction with the SNI SSC_CTL56T Spread Spectrum Controller 
    // to implement the SYSPLL.

    // Frefclk = 25 MHz for normal operation
    // Fpllclk = 1380-3010 MHz range
    // The following equations apply for all modes:
    // 
    // syspll_clkoutd1 = Fpllclk
    // syspll_clkoutd2 = Fpllclk/2
    // syspll_clkoutd4 = Fpllclk/4
    // 
    sscgbase = ioremap(SYSCG_BASE, SYSCG_SIZE);    
    
    FraMode = (readl(sscgbase + SYSCG_SYSSSCCTRL1_OFF) & SYSCG_SYSSSCCTRL1__NRSTDS__MASK) 
                >> SYSCG_SYSSSCCTRL1__NRSTDS__SHIFT;
    Ssc = (readl(sscgbase + SYSCG_SYSSSCCTRL1_OFF) & SYSCG_SYSSSCCTRL1__SSC_EN__MASK) 
                >> SYSCG_SYSSSCCTRL1__SSC_EN__SHIFT;
    SelFdiv4 = (readl(sscgbase + SYSCG_SYSPLLCTRL1_OFF) & SYSCG_SYSPLLCTRL1__SEL_FDIV4__MASK) 
                >> SYSCG_SYSPLLCTRL1__SEL_FDIV4__SHIFT;
    Ssc_j = (readl(sscgbase + SYSCG_SYSSSCCTRL2_OFF) & SYSCG_SYSSSCCTRL2__SSC_J__MASK) 
                >> SYSCG_SYSSSCCTRL2__SSC_J__SHIFT;
    Ssc_k = (readl(sscgbase + SYSCG_SYSSSCCTRL2_OFF) & SYSCG_SYSSSCCTRL2__SSC_K__MASK) 
                >> SYSCG_SYSSSCCTRL2__SSC_K__SHIFT;

    iounmap(sscgbase);
    
    // PLL EQUATIONS FOR FRACTIONAL MODE, SS DISABLED
    //frac = Ssc_k / (float)(1024 * 1024);
    //printk("Ssc_j=%u, SSsc_k=0x%x, frac=%d\n", Ssc_j, Ssc_k, frac);
    if (FraMode && !Ssc)
    {
        Fpllclk = Frefclk * 2 * (SelFdiv4 + 1) * (Ssc_j + (Ssc_k / (1024 * 1024)));//frac);
    }
    // PLL EQUATIONS FOR INTEGER MODE
    else
    {
        Fpllclk = Frefclk * 2 * (SelFdiv4 + 1) * Ssc_j;
    }

    return Fpllclk / div;
}

//*****************************************************************************
static u32 ddrPllOutputFrequency(int div)
{
    u8 __iomem *ddrcgbase;
    u32 Frefclk = XIN0_CLK;
    u32 Fpllclk, Ssc, FraMode, SelFdiv4, Ssc_j, Ssc_k;

    // Frefclk = 25 MHz for normal operation
    // Fpllclk = 1380-3010 MHz range
    // The following equations apply for all modes:
    // 
    // ddrpll_clkoutd1 = Fpllclk
    // ddrpll_clkoutd2 = Fpllclk/2
    // ddrpll_clkoutd4 = Fpllclk/4

    ddrcgbase = ioremap(DDRCG_BASE, DDRCG_SIZE);
    
    FraMode = (readl(ddrcgbase + DDRCG_DDRSSCCTRL1_OFF) & DDRCG_DDRSSCCTRL1__NRSTDS__MASK) 
                >> DDRCG_DDRSSCCTRL1__NRSTDS__SHIFT;
    Ssc = (readl(ddrcgbase + DDRCG_DDRSSCCTRL1_OFF) & DDRCG_DDRSSCCTRL1__SSC_EN__MASK) 
                >> DDRCG_DDRSSCCTRL1__SSC_EN__SHIFT;
    SelFdiv4 = (readl(ddrcgbase + DDRCG_DDRPLLCTRL1_OFF) & DDRCG_DDRPLLCTRL1__SEL_FDIV4__MASK) 
                >> DDRCG_DDRPLLCTRL1__SEL_FDIV4__SHIFT;
    Ssc_j = (readl(ddrcgbase + DDRCG_DDRSSCCTRL2_OFF) & DDRCG_DDRSSCCTRL2__SSC_J__MASK) 
                >> DDRCG_DDRSSCCTRL2__SSC_J__SHIFT;
    Ssc_k = (readl(ddrcgbase + DDRCG_DDRSSCCTRL2_OFF) & DDRCG_DDRSSCCTRL2__SSC_K__MASK) 
                >> DDRCG_DDRSSCCTRL2__SSC_K__SHIFT;

    iounmap(ddrcgbase);
    
    // PLL EQUATIONS FOR FRACTIONAL MODE, SS DISABLED
    if (FraMode && !Ssc)
    {
        Fpllclk = Frefclk * 2 * (SelFdiv4 + 1) * Ssc_j + (Ssc_k / 10486) * 1000000;
    }
    // PLL EQUATIONS FOR INTEGER MODE
    else
    {
        Fpllclk = Frefclk * 2 * (SelFdiv4 + 1) * Ssc_j;
    }

    return Fpllclk / div;
}

//*****************************************************************************
static u32 lowPowerFrequency(void)
{
    u8 __iomem *sscgbase;
    u32 lp_clk, lp_div, lp_mul;
 
    // The source clock for this lp_clk is syspll_clkout4.
    // Please check the doc of reg:SYSCG_CLKDIVCTRL_LP for more detail
    // 
    sscgbase = ioremap(SYSCG_BASE, SYSCG_SIZE);

    lp_div = (readl(sscgbase + SYSCG_CLKDIVCTRL_LP_OFF) & SYSCG_CLKDIVCTRL_LP__DIV_VALUE__MASK) 
                >> SYSCG_CLKDIVCTRL_LP__DIV_VALUE__SHIFT;
    lp_div = lp_div ? lp_div : 1;
    lp_mul = (readl(sscgbase + SYSCG_CLKDIVCTRL_LP_OFF) & SYSCG_CLKDIVCTRL_LP__NEG_PWH_START__MASK) 
                >> SYSCG_CLKDIVCTRL_LP__NEG_PWH_START__SHIFT;
    lp_mul = (lp_mul <= 1) ? 1 : 2;

    iounmap(sscgbase);

    lp_clk = lp_mul * sysPllOutputFrequency(4) / lp_div;

    return lp_clk;
}

//*****************************************************************************
static u32 getCCIfrequency(void)
{
    u8 __iomem *sscgbase;
    u32 cci_sw, cci_div, cci_mul, cci_mux;
    u32 cci_clk;
    
    // 1. SYSCG_CLKDISCTRL1.CCI: disable or not
    // 2. SYSCG_CLKSWCTRL1.CCI: select source for sw_cci_clk.
    // 00: xin0_clk
    // 01: lp_clk
    // 10: divxh_cci_clk, see SYSCG_CLKDIVCTRL_CCI
    //     => SYSCG_CLKMUXCTRL1.CCI: select source for cci_divxh_clkin.
    //        0: syspll_clkoutd1
    //        1: ddrpll_clkoutd1
    // 11: reserved

    sscgbase = ioremap(SYSCG_BASE, SYSCG_SIZE);

    cci_sw = (readl(sscgbase + SYSCG_CLKSWCTRL1_OFF) & SYSCG_CLKSWCTRL1__CCI__MASK) 
                >> SYSCG_CLKSWCTRL1__CCI__SHIFT;
    cci_div = (readl(sscgbase + SYSCG_CLKDIVCTRL_CCI_OFF) & SYSCG_CLKDIVCTRL_CCI__DIV_VALUE__MASK) 
                >> SYSCG_CLKDIVCTRL_CCI__DIV_VALUE__SHIFT;
    cci_div = cci_div ? cci_div : 1;
    cci_mul = (readl(sscgbase + SYSCG_CLKDIVCTRL_CCI_OFF) & SYSCG_CLKDIVCTRL_CCI__NEG_PWH_START__MASK) 
                >> SYSCG_CLKDIVCTRL_CCI__NEG_PWH_START__SHIFT;
    cci_mul = (cci_mul <= 1) ? 1 : 2;
    cci_mux = (readl(sscgbase + SYSCG_CLKMUXCTRL1_OFF) & SYSCG_CLKMUXCTRL1__CCI__MASK) 
                >> SYSCG_CLKMUXCTRL1__CCI__SHIFT;
                
    iounmap(sscgbase);
    switch (cci_sw)
    {
    case 1:
        cci_clk = lowPowerFrequency();
        break;
    case 2:
        if (0 == cci_mux)
        {
            cci_clk = cci_mul * sysPllOutputFrequency(1) / cci_div;
        }
        else
        {
            cci_clk = cci_mul * ddrPllOutputFrequency(1) / cci_div;
        }
        break;
    case 0:
    default:  // default: 0x0
        cci_clk = XIN0_CLK;
        break;
    }

    printk("cci clock is %d MHz\n", cci_clk / 1000000);
    return cci_clk;
}

static unsigned sysFrequency(void)
{
    // timer clock is half of CCI clock
    return (getCCIfrequency() >> 1);
}
#endif /* CONFIG_SOC_QUASAR6600 */

/**** System Timer functions, see bottom section for HRT (simple clock, hrt, function)
*/
#define SYT_LOAD		0x00000001
#define SYT_START   	0x00000002
#define SYT_CONT_MODE   0x00000004

#define SYT_TCN_OFF 0
#define SYT_CNT_OFF 4
#define SYT_CTL_OFF 8

/* offset between each timer, in register space
*/
#define SYT_OFF_PERCPU(n) ((n + 1) * 12)

struct quasar_timer {
	u8 __iomem *base;
	u32 mode;
	u32 freq;
	int irq;
	u32 cpj; /* cycles per jiffy */
	struct clock_event_device evt;
	struct irqaction act;
	char name[32];
};

static DEFINE_PER_CPU(struct quasar_timer, percpu_quasar_tick);

static int quasar_time_set_next_event(unsigned long event,
	struct clock_event_device *evt_dev)
{
	struct quasar_timer *timer = container_of(evt_dev,
		struct quasar_timer, evt);

	writel_relaxed(0, timer->base + SYT_CTL_OFF);
	writel_relaxed(event, timer->base + SYT_TCN_OFF);
	writel_relaxed(timer->mode | SYT_LOAD | SYT_START, timer->base + SYT_CTL_OFF);
	return 0;
}

static int quasar_time_shutdown(struct clock_event_device *evt_dev)
{
	struct quasar_timer *timer = container_of(evt_dev,
		struct quasar_timer, evt);

    timer->mode = 0;
    writel_relaxed(0, timer->base + SYT_CTL_OFF);
    return 0;
}

static int quasar_time_periodic(struct clock_event_device *evt_dev)
{
	struct quasar_timer *timer = container_of(evt_dev,
		struct quasar_timer, evt);

    timer->mode = SYT_CONT_MODE;
    quasar_time_set_next_event(timer->cpj, evt_dev);
    return 0;
}

static irqreturn_t quasar_time_interrupt(int irq, void *dev_id)
{
	struct quasar_timer *timer = dev_id;
	void (*event_handler)(struct clock_event_device *);
	
	/* printk(KERN_CRIT "ti b=%p i=%d cpu=%d\n", timer->base, timer->irq, smp_processor_id()); */
	event_handler = READ_ONCE(timer->evt.event_handler);
	if (event_handler)
		event_handler(&timer->evt);
	return IRQ_HANDLED;
}

static int quasar_local_timer_setup(unsigned int cpu)
{
	//unsigned int cpu = smp_processor_id();
	struct quasar_timer *timer=raw_cpu_ptr(&percpu_quasar_tick);

	timer->evt.name = timer->name;
	timer->evt.cpumask = cpumask_of(cpu);
	timer->evt.set_state_shutdown = quasar_time_shutdown;
	timer->evt.set_state_periodic = quasar_time_periodic;
	timer->evt.set_next_event = quasar_time_set_next_event;
	timer->evt.features = CLOCK_EVT_FEAT_PERIODIC | CLOCK_EVT_FEAT_ONESHOT | CLOCK_EVT_FEAT_PERCPU;
	timer->evt.rating = 450; /* higher rating lets linux pick this timer */

	timer->evt.irq = timer->irq;
	if (request_irq(timer->evt.irq, quasar_time_interrupt,
				IRQF_TIMER | IRQF_NOBALANCING,
				timer->evt.name, timer)) {
		pr_err("quasar_local_timer: cannot register IRQ %d\n",
			timer->evt.irq);
		return -EIO;
	}
	irq_force_affinity(timer->evt.irq, cpumask_of(cpu));
	clockevents_config_and_register(&timer->evt, timer->freq,
					0xf, 0x7fffffff);
	pr_info("quasar: local timer registered at %d Hz\n", timer->freq);
	return 0;
}

static int quasar_local_timer_stop(unsigned int cpu)
{
	struct quasar_timer *timer=raw_cpu_ptr(&percpu_quasar_tick);
    
	timer->evt.set_state_shutdown(&timer->evt);
	free_irq(timer->evt.irq, timer);
	return 0;    
}

static int __init quasar_timer_init(struct device_node *node)
{
	void __iomem *base;
	u32 freq, cycles_per_jiffy;
	int irq, nr_irqs, i;
	struct quasar_timer *timer;
    int ret=0;

	base = of_iomap(node, 0);
	if (!base)
		panic("Can't remap registers");

	freq = sysFrequency();

	/* ticks / jiffy = ticks/second * second/jiffy
	*/
	cycles_per_jiffy = (freq + (HZ / 2)) / HZ;

	clocksource_mmio_init(base + SYT_CNT_OFF, node->name,
		freq, 300, 32, clocksource_mmio_readl_up);

	nr_irqs = of_irq_count(node);
	if (nr_irqs < 1)
		panic("No IRQ for timers\n");
	irq = irq_of_parse_and_map(node, 0);
	if (irq <= 0)
		panic("Can't parse IRQ");

	timer = kzalloc(sizeof(*timer), GFP_KERNEL);
	if (!timer)
		panic("Can't allocate timer struct\n");

	strncpy(timer->name, node->name, sizeof(timer->name));
	timer->base = base;
	timer->mode = 0;
	timer->freq = freq;
	timer->cpj = cycles_per_jiffy;
	timer->evt.name = node->name;
	timer->evt.rating = 300;
	timer->evt.features = CLOCK_EVT_FEAT_PERIODIC | CLOCK_EVT_FEAT_ONESHOT;
	timer->evt.set_state_shutdown = quasar_time_shutdown;
	timer->evt.set_state_periodic = quasar_time_periodic;
	timer->evt.set_next_event = quasar_time_set_next_event;
	timer->evt.cpumask = cpumask_of(smp_processor_id());

	timer->act.name = node->name;
	timer->act.flags = IRQF_TIMER | IRQF_IRQPOLL;
	timer->act.dev_id = timer;
	timer->act.handler = quasar_time_interrupt;

	writel_relaxed(0, timer->base + SYT_CTL_OFF);

	if (request_irq(irq, quasar_time_interrupt,
				IRQF_TIMER | IRQF_NOBALANCING,
				timer->evt.name, timer))
		panic("quasar_timer: cannot register IRQ %d\n", irq);

	clockevents_config_and_register(&timer->evt, freq, 0xf, 0xffffffff);

	pr_info("quasar: system timer cpj=%d (irq = %d) freq=%d\n", cycles_per_jiffy, irq, freq);
	
	/* pre-configure each local timer, each local timer is specified
	 * simply as an extra irq in the device tree
	 */
	for (i = 0; i < nr_irqs - 2; i++) {
		timer = &per_cpu(percpu_quasar_tick, i);
		//pr_info("quasar: timer %d-%d, timer=%p, name=%s, base=0x%08X\n", i, nr_irqs, timer, timer->name, timer->base);
		if (! timer) {
			/*panic("Too many irqs for cpu count\n");*/
			break;
		}
		snprintf(timer->name, sizeof(timer->name), "qt lcl %d\n", i);
		timer->irq  = irq_of_parse_and_map(node, i + 1);
		timer->freq = freq;
		timer->mode = 0;
		timer->base = base + SYT_OFF_PERCPU(i);
		timer->cpj  = cycles_per_jiffy;
	}
	/* setup local timer on this boot cpu if more than one timer */
	if (nr_irqs > 1)
		ret = cpuhp_setup_state(CPUHP_AP_QCOM_TIMER_STARTING,
					"Quasar timer notify starting",
					quasar_local_timer_setup,
					quasar_local_timer_stop);

    return ret;
}

/***** HRT/CLK 
*/

#define HRT_STOP		0x00000000
#define HRT_CLEAR   	0x00000001
#define HRT_START   	0x00000002

#define HRTPRE1_OFF 	0x00000000
#define HRTCNT1_OFF 	0x00000004
#define HRTCTL1_OFF 	0x00000008

#define HRTPRE0_OFF 	0x00000000
#define HRTCNT0H_OFF	0x00000004
#define HRTCNT0L_OFF	0x00000008
#define HRTCTL0_OFF 	0x0000000C

struct quasar_clock {
	u8 __iomem *base;
	struct clocksource cs;
};

static int __init quasar_hrt1_init(struct device_node *node)
{
	u8 __iomem *base;
	u32 freq;
	struct quasar_clock *clock;

	base = of_iomap(node, 0);
	if (!base)
		panic("Can't remap registers");

	freq = sysFrequency();

	clocksource_mmio_init(base + HRTCNT1_OFF, node->name,
		freq / 32, 300, 32, clocksource_mmio_readl_up);

	clock = kzalloc(sizeof(*clock), GFP_KERNEL);
	if (!clock)
		panic("Can't allocate clock struct\n");

	clock->base = base;
	clock->cs.name = node->name;
	clock->cs.rating = 300;
	clock->cs.mask = CLOCKSOURCE_MASK(32);
	clock->cs.flags = CLOCK_SOURCE_IS_CONTINUOUS;

	/* start hrt1 free-running at about 8Mhz (1/32 of system clock) */
	writel_relaxed(HRT_CLEAR, clock->base + HRTCTL1_OFF);
	writel_relaxed(31, clock->base + HRTPRE1_OFF);
	writel_relaxed(HRT_START, clock->base + HRTCTL1_OFF);

	clocksource_register_hz(&clock->cs, freq / 32);

	pr_info("quasar: system clock at %u Hz\n", freq / 32);
    
    return 0;
}

static struct quasar_clock *hrt0_sched_clock;

static u64 quasar_sched_clock_read(void)
{
	u32 lo, hi;
	u32 hi2 = readl_relaxed(hrt0_sched_clock->base + HRTCNT0H_OFF);

	do {
		hi = hi2;
		lo = readl_relaxed(hrt0_sched_clock->base + HRTCNT0L_OFF);
		hi2 = readl_relaxed(hrt0_sched_clock->base + HRTCNT0H_OFF);
	} while (hi != hi2);

	return ((u64)hi << 32) | lo;	
}

static u64 quasar_hrt0_read(struct clocksource *cs)
{
	return quasar_sched_clock_read();
}

static int __init quasar_hrt0_init(struct device_node *node)
{
	u8 __iomem *base;
	u32 freq;
	struct quasar_clock *clock;

	base = of_iomap(node, 0);
	if (!base)
		panic("Can't remap registers");

	freq = sysFrequency();

	clock = kzalloc(sizeof(*clock), GFP_KERNEL);
	if (!clock)
		panic("Can't allocate clock struct\n");

	clock->base = base;
	clock->cs.name = node->name;
	clock->cs.rating = 300;
	clock->cs.mask = CLOCKSOURCE_MASK(64);
	clock->cs.flags = CLOCK_SOURCE_IS_CONTINUOUS;
	clock->cs.read = quasar_hrt0_read;
	hrt0_sched_clock = clock; /* Need a copy for
				   * quasar_sched_clock_read() */

	/* start hrt0 free-running at system clock) */
	writel_relaxed(HRT_CLEAR, clock->base + HRTCTL0_OFF);
	writel_relaxed(0, clock->base + HRTPRE0_OFF);
	writel_relaxed(HRT_START, clock->base + HRTCTL0_OFF);

	clocksource_register_hz(&clock->cs, freq);

	sched_clock_register(quasar_sched_clock_read, 64, freq);

	pr_info("quasar: hrt at %u Hz\n", freq);
    return 0;
}
TIMER_OF_DECLARE(quasar, "qbit,quasar-sys-timer", quasar_timer_init);
TIMER_OF_DECLARE(quasarhrt1, "qbit,quasar-hrt32", quasar_hrt1_init);
TIMER_OF_DECLARE(quasarhrt0, "qbit,quasar-hrt64", quasar_hrt0_init);


