/*
 * A driver for the low power for Quasar SOCs
 *
 *
 * Copyright (c) 2015, The Linux Foundation.
 * All rights reserved.
 *
 * Redistribution and use
 * in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer
 * below) provided that the following conditions are met :
 *   *Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *   *Redistributions in binary form must reproduce the
 *    above copyright notice, this list of conditions and
 *    the following disclaimer
 *    in the documentation and/or other materials provided
 *    with the distribution.
 *
 *  NO EXPRESS OR IMPLIED LICENSES TO ANY PARTYS PATENT
 *  RIGHTS ARE GRANTED BY THIS LICENSE.
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS
 *  AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 *  AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 *  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 *  OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 *  OR PROFITS;
 *  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 *  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 *  OF SUCH DAMAGE
 *
 */
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/jiffies.h>
#include <linux/platform_device.h>
#include <linux/of_platform.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/pagemap.h>
#include <linux/io.h>
#include <linux/dma-mapping.h>
#include <linux/vmalloc.h>
#include <linux/mman.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/device.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/poll.h>
#include <linux/mutex.h>
#include <asm/io.h>
#include <asm/mman.h>
#include <asm/uaccess.h>
#include <asm/dma.h>
#include <asm/pgalloc.h>

#define Q6300
#define Q6300B0
#include "../../../../drivers/quasar/qbsocregs.h"
#include "quasar.h"
#include "core.h"

#define SHARE_MEMORY_START  0x88aF0000
#define SHARE_MEMORY_SIZE   0x00010000

#if 0
#define SLEEP_INFO(...)
#else
#define SLEEP_INFO              printk
#endif

#define MAX_LOPIOREGS    	    4

struct quasar_io {
	u32						start;
	u32						length;
};

struct quasar_io_region {
	u32						start;
	u32						end;
	volatile u8	__iomem		*base;
};

static struct quasar_io quasariomap[MAX_LOPIOREGS]={
    {SHARE_MEMORY_START, SHARE_MEMORY_SIZE},     /* Share memory */
    {0x05443800, 0x8000},       /* M30 registers */
    {0x05683800, 0x8000},       /* M31 registers */
    {0x04310010, 0x78}};        /* A7GPF registers */

static struct quasar_io_region	ioregs[MAX_LOPIOREGS];
static u32 nioregs=0;

static volatile u8 __iomem* qlomap_addr(u32 reg)
{
	volatile u8 __iomem* addr = 0;
 	int      i;

	//printk(KERN_WARNING  "qlomap_addr - reg %08x, <%08X-%08X>--<%08X-%08X>--<%08X-%08X>--<%08X-%08X>\n", 
	//       reg, ioregs[0].start, ioregs[0].end, ioregs[1].start, ioregs[1].end, ioregs[2].start, ioregs[2].end, ioregs[3].start, ioregs[3].end);
	/* check for existing mapping */
	for (i = 0; i < nioregs; i++) {
		if(reg >= ioregs[i].start && reg < ioregs[i].end) {
			return(ioregs[i].base + (reg - ioregs[i].start));
		}
	}
	printk(KERN_WARNING "qlomap_addr - could not map reg %08x\n", reg);
	return addr;
}

static int qlomap_readl(u32 reg, unsigned* val)
{
	volatile u32 rv = 0;
	int ret = 0;
	volatile u8 __iomem* addr = qlomap_addr(reg);
	
	if(addr == NULL)
		return -1;
	rv = readl(addr);
	//printk("readl %08X [%08X] -> %08X\n", reg, addr, rv);
	*val = (unsigned long)rv;
	return ret;
}

static int qlomap_writel(u32 reg, u32 val)
{
	volatile u8 __iomem* addr = qlomap_addr(reg);
	if(addr == NULL)
		return -1;
	//printk("writel %08X [%08X] <= %08x\n", reg, addr, val);
	writel(val, addr);
	return 0;
}

// UART2 as serial output port 
#define UART_THR                UART2THR
#define UART_LSR                UART2LSR
#define THRE                    (1<<5) /* Transmitter Hold Register Empty */

#define IOWRREG_U32(p,c)        (*((volatile u32*)(p)) = ((u32)(c)))
#define IORDREG_U32(p)          (*((volatile u32*)p))
#define IOWRREG_U8(p,c)         (*((volatile u8 *)(p)) = ((u8)(c)))
#define IODREG_U8(p)            (*((volatile u8 *)p))
  
/* CP15 control register bits */
#define CR_MMU_EN    (1<<0)     /* enable MMU */
#define CR_DCACHE_EN (1<<2)     /* enable level-1 data cache */
#define CR_Z_EN      (1<<11)    /* program flow prediction enable (disable!) */
#define CR_ICACHE_EN (1<<12)    /* enable level-1 instruction cache */

///////////////////////////////////////////////////////////////////////////////
// Context stotage
//
// keep cpsr and sp in share memory
#define SLEEPINFO_BASE          SHARE_MEMORY_START
#define SLEEPM31INFO_LOCATION   SLEEPINFO_BASE
#define SLEEPM31INFO_LENGTH     8
#define SLEEPINFO_LOCATION      (SLEEPM31INFO_LOCATION + SLEEPM31INFO_LENGTH)
#define SLEEPINFO_LENGTH        128
#define SLEEPINFO_SP_LOCATION   ((SHARE_MEMORY_START + SHARE_MEMORY_SIZE - 1) & ~31)

typedef struct tagSLEEPINFO
{
    u32     spsr;
    u32     sp; 
    u32     m30reg[8];
    u32     m31reg[8];
} SLEEPINFO, *PSLEEPINFO;

SLEEPINFO sinfo;

//////////////////////////////////////////////////////////////////////////////
// Macro definition 
//    

#define ISB()     asm volatile("isb")
#define DSB()     asm volatile("dsb")

// INT on/off
#define INTON() \
    asm volatile("push {r0}             \n\t"       \
                 "mrs r0, CPSR          \n\t"       \
                 "bic r0, r0, #0xC0     \n\t"       \
                 "msr CPSR_c, r0        \n\t"       \
                 "pop {r0}              \n\t" )
 
#define INTOFF() \
    asm volatile("push {r0, r1}         \n\t"       \
                 "mrs r0, CPSR          \n\t"       \
                 "orr r1, r0, #0xC0     \n\t"       \
                 "msr CPSR_c, r1        \n\t"       \
                 "pop {r0, r1}          \n\t" )
    
//////////////////////////////////////////////////////////////////////////////
// Extern 
//
extern void BIOSInvalidateTLBs(void);

// Declare functions in direct mapping section
// those functions can be accessed on either MMU is on or off
void quasarOutHex(u32 number) __attribute__ ((section (".idmap.text")));
void quasarOutStr(char* pstr) __attribute__ ((section (".idmap.text")));
void quasarRestore(void) __attribute__ ((section (".idmap.text")));
void quasarSleep(void) __attribute__ ((section (".idmap.text")));

// APIs for message output -- physical mapping (MMU disabled)
void quasarOutStr(char* pstr)
{
    while (*pstr)
    {
        while (!(IORDREG_U32(UART_LSR) & THRE))
        {
            // wait till transmit buffer empty
        }
        IOWRREG_U32(UART_THR, *pstr);
        pstr++;
    }
}

void quasarOutHex(u32 number)
{
    char outch;
    char outstr[12];
    int i=8, index=0;

    do
    {
        char nybble = (number >> 28) + '0';

        outch = nybble <= '9' ? nybble : nybble - 0xA + 'A' - '0';

        number <<= 4;
        outstr[index++] = outch;
    } while (--i);

    outstr[index++] = '\n';
    outstr[index++] = '\0';
    quasarOutStr(outstr);
}

// Sleep mode restore handler
// This function is called from ROM code directly with d-cache disabled
#define M3_WAIT_WFI                 1
#define QM3_SLEEP_MASK              (QBM3_1_STAT__SLEEP__MASK       |   \
                                     QBM3_1_STAT__DIFIDLE__MASK     |   \
                                     QBM3_1_STAT__IIFIDLE__MASK     |   \
                                     QBM3_1_STAT__DLBIDLE__MASK     |   \
                                     QBM3_1_STAT__ILBIDLE__MASK)
#define WAITFORWFI_TIMEOUT_US       (20 * 1000)    // micro second unit
extern void BIOSSetCp15MVBARReg(u32 monVector, u32 monSP);
extern void BIOSMonContextRestore(void);
extern void BIOSSetCp15Nonsecure(void);
void quasarRestore(void)
{   
    // Get restoring cpsr and sp
    PSLEEPINFO psinfo=(PSLEEPINFO)SLEEPINFO_LOCATION;
    u32 val, tStart;
    u32 monexception_vector = 0x80000040;   // monitor vector is at a fixed location
    u32 monSP = 0x80003000; // monitor sp will be restored after repower
    
    

	/* insure all caches are off */
    asm volatile("push  {r0}                            \n\t"   \
                 "dmb                                   \n\t"   \
                 "mrc 	p15, #0, r0, c1, c0, #0         \n\t"   \
                 "bic 	r0, r0, #(1<<12)                \n\t"   \
                 "bic 	r0, r0, #(1<<11)                \n\t"   \
                 "bic 	r0, r0, #((1<<2)|(1<<0))        \n\t"   \
                 "mcr 	p15, #0, r0, c1, c0, #0         \n\t"   \
                 "pop   {r0}");

    /* turn on SMP bit, enable performance counters and turn on I */
    asm volatile("push  {r0}                            \n\t"   \
                 "mrc	p15, #0, r0, c1, c0, #1         \n\t"   \
                 "orr	r0, r0, #(1<<6)                 \n\t"   \
                 "mcr 	p15, #0, r0, c1, c0, #1         \n\t"   \
                 "pop   {r0}");
                 
    /* Turn on I */                 
    asm volatile("push  {r0}                            \n\t"   \
                 "dmb                                   \n\t"   \
                 "mrc 	p15, #0x0, r0, c1, c0, #0       \n\t"   \
                 "orr 	r0, r0, #(1<<12)                \n\t"   \
                 "orr 	r0, r0, #(1<<11)                \n\t"   \
                 "mcr 	p15, #0x0, r0, c1, c0, #0       \n\t"   \
                 "pop   {r0}");  
                                  
    /* Invalidate TBLs */
    asm volatile("push  {r0}                            \n\t"   \
                 "mov   r0, #0                          \n\t"   \
                 "mcr   p15, #0x0, r0, c8, c7,  #0      \n\t"   \
                 "pop   {r0}");
                
    // Enable Vector Floating point and NEON
    asm volatile("push  {r0, r1}                        \n\t"   \
                 "mrc     p15, 0, r1, c1, c0, 2         \n\t"   \
                 "orr     r1, r1, #(0xf << 20)          \n\t"   \
                 "mcr     p15, 0, r1, c1, c0, 2         \n\t"   \
                 "mov     r1, #0                        \n\t"   \
                 "mcr     p15, 0, r1, c7, c5, 4         \n\t"   \
                 "mcr     p15, 0, r1, c1, c1, 2         \n\t"   \
                 "orr     r1, r1, #(0x3 << 10)          \n\t"   \
                 "mcr     p15, 0, r1, c1, c1, 2         \n\t"   \
                 "pop   {r0, r1}");      
                 
    while(IORDREG_U32(SLEEPM31INFO_LOCATION) == 0);
    //quasarOutHex(IORDREG_U32(HRTCNTL0)); 

    // enable cache-coherent for A7
    IOWRREG_U32(CCI_SNOOP_S3,
                CCI_SNOOP_S3__DVMEN__MASK | CCI_SNOOP_S3__SNPEN__MASK);

    /* Enter non-secure world */
    // Before entering Non-Secure world, we have to setup something 
    // which is only acessible in privilege secure world

    // Set GIC as Group1 interrupt
    for (val=GICD_GROUPR0; val<=GICD_GROUPR6; val+=4)
    {
        IOWRREG_U32(val, 0xFFFFFFFF);
    }
    IOWRREG_U32(GICC_PMR, 0xFF);

    // Setup minotor vector
    BIOSSetCp15MVBARReg((u32)monexception_vector, (u32)monSP);
    // Restore context on Monitor mode
    BIOSMonContextRestore();
    
    // Enter Non-Secure world
    BIOSSetCp15Nonsecure();      

    // Wait for M31 going to wfi
    tStart = IORDREG_U32(HRTCNT1);
#if M3_WAIT_WFI   
    while (1)
    {
        if ((IORDREG_U32(HRTCNT1) - tStart) > WAITFORWFI_TIMEOUT_US)
        {
            quasarOutStr("\nWait for M31 going to WFI TIMEOUT !!!\n");  
            break;
        }
        if ((IORDREG_U32(QBM3_1_STAT) & QM3_SLEEP_MASK) == QM3_SLEEP_MASK)
        {
            break;
        }
    }
#endif
    // Assert both H reset and PO reset.
    val = IORDREG_U32(RSTGEN_SWRSTSTATIC6) |
            RSTGEN_SWRSTSTATIC6__PRT_M3_1_H__MASK |
            RSTGEN_SWRSTSTATIC6__PRT_M3_1_POR__MASK;
    IOWRREG_U32(RSTGEN_SWRSTSTATIC6, val);
    //Release PO reset
    val = IORDREG_U32(RSTGEN_SWRSTSTATIC6) &
            RSTGEN_SWRSTSTATIC6__PRT_M3_1_POR__INV_MASK;
    IOWRREG_U32(RSTGEN_SWRSTSTATIC6, val);
    
#ifdef CONFIG_SOC_QUASAR_M30_POWER_DOWN   
    // Restore M30 registers
    IOWRREG_U32(QBM3_0_IPC, psinfo->m30reg[0]);
    IOWRREG_U32(QBM3_0_ISP, psinfo->m30reg[1]);
    IOWRREG_U32(QBM3_0_PAGE0, psinfo->m30reg[2]);
    IOWRREG_U32(QBM3_0_PAGE1, psinfo->m30reg[3]);
    IOWRREG_U32(QBM3_0_PAGE2, psinfo->m30reg[4]);
    IOWRREG_U32(QBM3_0_PAGE3, psinfo->m30reg[5]);
    IOWRREG_U32(QBM3_0_PPAGE, psinfo->m30reg[6]);
    IOWRREG_U32(QBM3_0_M3IEN, psinfo->m30reg[7]);
#endif
    // Restore M30 registers
    IOWRREG_U32(QBM3_1_IPC, psinfo->m31reg[0]);
    IOWRREG_U32(QBM3_1_ISP, psinfo->m31reg[1]);
    IOWRREG_U32(QBM3_1_PAGE0, psinfo->m31reg[2]);
    IOWRREG_U32(QBM3_1_PAGE1, psinfo->m31reg[3]);
    IOWRREG_U32(QBM3_1_PAGE2, psinfo->m31reg[4]);
    IOWRREG_U32(QBM3_1_PAGE3, psinfo->m31reg[5]);
    IOWRREG_U32(QBM3_1_PPAGE, psinfo->m31reg[6]);
    IOWRREG_U32(QBM3_1_M3IEN, psinfo->m31reg[7]);
        
    // Restore cpsr and sp 
    asm volatile("msr cpsr_cxsf, %0"::"r"(psinfo->spsr));
    asm volatile("mov sp, %0"::"r"(psinfo->sp));
    
    // pop registers of running CPU mode from stack
    asm volatile("ldmfd sp!, {r0-r12,lr}");

    // Return to normal code  
    asm volatile("bx lr"); 
}

/* 
SCAN_POWER_DOWN force M30 entering low power
* 
|    bits |`31 .... 24`|`23 .... 16`|`15 ..... 8`|`7 ...... 0`|
|--------:|:----------:|:----------:|:----------:|:----------:|
|  header |    0xAA    |    0x06    |    Seq.    |    size    |
| text[0] |    0x00    |    0x00    |    0x00    |    0x00    |
        
The response of this command is
|    bits |`31 .... 24`|`23 .... 16`|`15 ..... 8`|`7 ...... 0`|
|--------:|:----------:|:----------:|:----------:|:----------:|
|  header |    0xAA    |    0x06    |    Seq.    |    0x01    |
| text[0] |                                    |||    status  |
*/

/* SCAN_POWER_DOWN force M30 leaving low power
* 
|    bits |`31 .... 24`|`23 .... 16`|`15 ..... 8`|`7 ...... 0`|
|--------:|:----------:|:----------:|:----------:|:----------:|
|  header |    0xAA    |    0x07    |    Seq.    |    size    |
| text[0] |    0x00    |    0x00    |    0x00    |    0x00    |
        
The response of this command is
|    bits |`31 .... 24`|`23 .... 16`|`15 ..... 8`|`7 ...... 0`|
|--------:|:----------:|:----------:|:----------:|:----------:|
|  header |    0xAA    |    0x07    |    Seq.    |    0x01    |
| text[0] |                                    |||    status  |
*/
#define SCAN_POWER_DOWN             0xAA060000
#define SCAN_POWER_UP               0xAA070000
#define QBM3_0_MB0                  0x05447000
#define QBM3_0_MBMC0                0x0544700C
#define QBM3_0_MB1                  0x05447010
#define QBM3_0_MBS1                 0x05447014
#define QBM3_0_MBMC1                0x0544701C
#define SCA_RSP_OK                  0

/* 
PRINT_ENTER_SLEEP force M31 goto sleep mode routine

|    bits |`31 .... 24`|`23 .... 16`|`15 ..... 8`|`7 ...... 0`|
|--------:|:----------:|:----------:|:----------:|:----------:|
|  header |    0xAA    |    0x93    |    Seq.    |    0x01    |
| text[0] |    0x00    |    0x00    |    0x00    |    0x00    | 

The response of this command is:
|    bits |`31 .... 24`|`23 .... 16`|`15 ..... 8`|`7 ...... 0`|
|--------:|:----------:|:----------:|:----------:|:----------:|
|  header |    0xAA    |    0x93    |    Seq.    |    0x02    |
| text[0] |    0x00    |    0x00    |    0x00    |    0x00    | 

*/
#define PRINT_ENTER_SLEEP           0xAA930000
#define QBM3_1_MB0                  0x05687000
#define QBM3_1_MBMC0                0x0568700C
#define QBM3_1_MB1                  0x05687010
#define QBM3_1_MBS1                 0x05687014
#define QBM3_1_MBMC1                0x0568701C
#define QM3_MAX_ARGS                16
#define QM3_SLEEP_RESP_MSG_COUNT    2 
#define PRINT_RSP_OK                0

#define WAITFORSLEEP_TIMEOUT_US     2000000    // micro second unit
extern void BIOSshutdownA7(void);
extern void BIOSMonContextSave(void);
extern void BIOSSetCp15Secure(void);
void quasarSleep(void) __attribute__ ((section (".idmap.text")));
void quasarSleep(void)
{   
    u32 sleep=1;
    u32 entries, msgcnt;
    u32 tStart;
    u32 val;
#ifdef CONFIG_SOC_QUASAR_M30_POWER_DOWN
#endif            
    printk("quasarSleep: disable MMU\n");
    /* disable MMU  */
    asm volatile(
        "mrc     p15, #0, r1, c1, c0, #0        \n\t"   \
        "dsb                                    \n\t"   \
        "bic     r1, r1, #(1 << 0)              \n\t"   \
        "mcr     p15, #0, r1, c1, c0, #0        \n\t"   \
        "isb                                    \n\t");

    // Keeps monitor mode context
    BIOSMonContextSave();
    
#ifdef CONFIG_SOC_QUASAR_M30_POWER_DOWN
    // Notify M30 to enter sleep mode
    // MMU is off, diret access to M30 register (physical address) 
    // 1. Check pending notify/reponse messages from mailbox (mailbox1)
    entries = IORDREG_U32(QBM3_0_MBS1);
    msgcnt = (entries >> 16) & 0x1F;        
    entries &= 0x1F;
    if (msgcnt || entries)
    {
        // Have pending notify/reponse, wake up
        sleep = 0;
    }
    else
    {
        // 2. Send SCAN_POWER_DOWN command to M30 (mailbox0)
        quasarOutStr("Force M30 to enter sleep .....\n");
        //quasarOutStr("      QBM3_0_MBS0: "); 
        //quasarOutHex(IORDREG_U32(QBM3_0_MBS0));
        IOWRREG_U32(QBM3_0_MB0, SCAN_POWER_DOWN | 0x01);
        //quasarOutStr("      QBM3_0_MB0 done\n");
        IOWRREG_U32(QBM3_0_MBMC0, 0);
        //quasarOutStr("      QBM3_0_MBMC0 done\n");
        // 3. Get response (mailbox1)
        tStart = IORDREG_U32(HRTCNT1);
        //quasarOutStr("  Wait for response .....\n");
        while (sleep)
        {
            u32 vals[QM3_MAX_ARGS];
            u32 header, msgtype, msgid, msgcount;
            u32 i;
          
            entries = IORDREG_U32(QBM3_0_MBS1);
            msgcnt = (entries >> 16) & 0x1F;        
            entries &= 0x1F;
            if (! msgcnt || ! entries)
            {
                if ((IORDREG_U32(HRTCNT1) - tStart) > WAITFORSLEEP_TIMEOUT_US)
                {
                    // Timeout, failed
                    quasarOutStr("\nWait for sleep timeout !!!, QBM3_0_MBS1: ");   
                    quasarOutHex(IORDREG_U32(QBM3_0_MBS1));                   
                    sleep = 0;
                    break;
                }
                
                continue;
            }
            header = IORDREG_U32(QBM3_0_MB1);  
            entries--;
            msgtype  = header >> 24;
            msgid    = (header >> 16) & 0xFF;
            msgcount = (header & 0xFF); /* number of dwords, excluding header */

            if (entries < msgcount)
            {
                // Protocol error
                quasarOutStr("\nProtocol error !!!\n");  
                sleep = 0;
                break;
            }
            
            // read msg arguments into buffer
            //
            for (i = 0; i < (msgcount - 1) && (entries > 0) && (i < QM3_MAX_ARGS); i++)
            {
                vals[i] = IORDREG_U32(QBM3_0_MB1);
                entries--;
            }
            // read last msg word and dec msg count
            //
            vals[i] = IORDREG_U32(QBM3_0_MBMC1);
            entries--;
            
            if ((msgtype != 0xAA) || (msgid != 0x06) || (vals[0] != SCA_RSP_OK))
            {
                // protocol error
                quasarOutStr("\nPesponse error !!!\n");  
                quasarOutHex(header);            
                quasarOutHex(vals[0]);             
                sleep = 0;
            }
            break;
        }        
    }

    // Wait for M30 going to wfi
#if M3_WAIT_WFI    
    tStart = IORDREG_U32(HRTCNT1);
    while (1)
    {
        if ((IORDREG_U32(HRTCNT1) - tStart) > WAITFORWFI_TIMEOUT_US)
        {
            quasarOutStr("\nWait for M30 going to WFI TIMEOUT !!!\n");  
            break;
        }
        if ((IORDREG_U32(QBM3_0_STAT) & QM3_SLEEP_MASK) == QM3_SLEEP_MASK)
        {
            break;
        }
    }
#endif
    // Hold reset of m30. m1 is the low power master so it cannot be reset
    val = IORDREG_U32(RSTGEN_SWRSTSTATIC5) | RSTGEN_SWRSTSTATIC5__SCA_M3_0_H__MASK;
    IOWRREG_U32(RSTGEN_SWRSTSTATIC5, val);    
    if (!sleep)
    {
        // wakeup immediately
        IOWRREG_U32(SLEEPM31INFO_LOCATION, 1);
        BIOSSetCp15Secure();
        quasarRestore();
    }
#endif

    // Notify M31 to enter sleep mode
    // MMU is off, diret access to M31 register (physical address) 
    // 1. Check pending notify/reponse messages from mailbox (mailbox1)
    entries = IORDREG_U32(QBM3_1_MBS1);
    msgcnt = (entries >> 16) & 0x1F;        
    entries &= 0x1F;
    if (msgcnt || entries)
    {
        // Have pending notify/reponse, wake up
        quasarOutStr("M31 mailbox is not empty, entries: ");
        quasarOutHex(entries); 
        quasarOutStr(", msg: ");
        quasarOutHex(IORDREG_U32(QBM3_1_MB1)); 
        quasarOutStr("\n");
        sleep = 0;
    }
    else
    {
        // 2. Send PRINT_ENTER_SLEEP command to M31 (mailbox0)
		val = IORDREG_U32(GMAC_CFG);
		IOWRREG_U32(GMAC_CFG, val); 
        quasarOutStr("Force M31 to enter sleep .....\n");
		
        // Clear M31 done information
        IOWRREG_U32(SLEEPM31INFO_LOCATION, 0);
        IOWRREG_U32(QBM3_1_MB0, PRINT_ENTER_SLEEP | 0x01);
        IOWRREG_U32(QBM3_1_MBMC0, 0);
        // 3. Get response (mailbox1)
        tStart = IORDREG_U32(HRTCNT1);
        //quasarOutStr("  Wait for response .....\n");
        while (sleep)
        {
            u32 vals[QM3_MAX_ARGS];
            u32 header, msgtype, msgid, msgcount;
            u32 i;
          
            entries = IORDREG_U32(QBM3_1_MBS1);
            msgcnt = (entries >> 16) & 0x1F;        
            entries &= 0x1F;
            if (! msgcnt || ! entries)
            {
                if ((IORDREG_U32(HRTCNT1) - tStart) > WAITFORSLEEP_TIMEOUT_US)
                {
                    // Timeout, failed
                    quasarOutStr("\nWait for sleep timeout !!!, QBM3_1_MBS1: ");   
                    quasarOutHex(IORDREG_U32(QBM3_1_MBS1));                   
                    sleep = 0;
                    break;
                }
                
                continue;
            }
            header = IORDREG_U32(QBM3_1_MB1);  
            entries--;
            msgtype  = header >> 24;
            msgid    = (header >> 16) & 0xFF;
            msgcount = (header & 0xFF); /* number of dwords, excluding header */

            if (entries < msgcount)
            {
                // Protocol error
                quasarOutStr("\nProtocol error !!!\n");  
                sleep = 0;
                break;
            }
            
            // read msg arguments into buffer
            //
            for (i = 0; i < (msgcount - 1) && (entries > 0) && (i < QM3_MAX_ARGS); i++)
            {
                vals[i] = IORDREG_U32(QBM3_1_MB1);
                entries--;
            }
            // read last msg word and dec msg count
            //
            vals[i] = IORDREG_U32(QBM3_1_MBMC1);
            entries--;
            
            if ((msgtype != 0xAA) || (msgid != 0x93) || (vals[0] != PRINT_RSP_OK))
            {
                // protocol error
                quasarOutStr("\nPesponse error !!!\n");  
                quasarOutHex(header);            
                quasarOutHex(vals[0]);             
                sleep = 0;
            }
            break;
        }        
    }

    if (!sleep)
    {
#if 0        
        // Power up M30 first
        IOWRREG_U32(QBM3_0_MB0, SCAN_POWER_UP | 0x01);
        IOWRREG_U32(QBM3_0_MBMC0, 0);
        // Get response (mailbox1)
        tStart = IORDREG_U32(HRTCNT1);
        while (1)
        {
            u32 vals[QM3_MAX_ARGS];
            u32 header, msgtype, msgid, msgcount;
            u32 i;
          
            entries = IORDREG_U32(QBM3_0_MBS1);
            msgcnt = (entries >> 16) & 0x1F;        
            entries &= 0x1F;
            if (! msgcnt || ! entries)
            {
                if ((IORDREG_U32(HRTCNT1) - tStart) > WAITFORSLEEP_TIMEOUT_US)
                {
                    // Timeout, failed
                    quasarOutStr("\nWait for sleep timeout !!!, QBM3_0_MBS1: ");   
                    quasarOutHex(IORDREG_U32(QBM3_0_MBS1));                   
                    break;
                }
                
                continue;
            }
            header = IORDREG_U32(QBM3_0_MB1);  
            entries--;
            msgtype  = header >> 24;
            msgid    = (header >> 16) & 0xFF;
            msgcount = (header & 0xFF); /* number of dwords, excluding header */

            if (entries < msgcount)
            {
                // Protocol error
                quasarOutStr("\nProtocol error !!!\n");  
                sleep = 0;
                break;
            }
            
            // read msg arguments into buffer
            //
            for (i = 0; i < (msgcount - 1) && (entries > 0) && (i < QM3_MAX_ARGS); i++)
            {
                vals[i] = IORDREG_U32(QBM3_0_MB1);
                entries--;
            }
            // read last msg word and dec msg count
            //
            vals[i] = IORDREG_U32(QBM3_0_MBMC1);
            entries--;
            
            if ((msgtype != 0xAA) || (msgid != 0x07) || (vals[0] != SCA_RSP_OK))
            {
                // protocol error
                quasarOutStr("\nPesponse error !!!\n");  
                quasarOutHex(header);            
                quasarOutHex(vals[0]);             
            }
            break;
        }        
#endif        
        // wakeup immediately
        IOWRREG_U32(SLEEPM31INFO_LOCATION, 1);
        BIOSSetCp15Secure();
        quasarRestore();
    }

    IOWRREG_U32(GICD_CTLR, 0); // Disable GIC Distributor
    IOWRREG_U32(GICC_CTLR, 0); // Disable GIC CPU Interface
    asm volatile("isb\n");  

    // wait for turn off
    BIOSshutdownA7();
    asm volatile(
        "loopwfi:           \n\t"   \
        "   wfi             \n\t"   \
        "   b   loopwfi     \n\t");
}

// Saving running context
void qlop_sleep(u32 cpsr, u32 sp, u32 lr)
{
    u32 i, regval;
    void *pDest, *pSrc;
    u32 size, val;
    void __iomem *mapped_address;
	PSLEEPINFO psinfo=&sinfo;

    psinfo->spsr = cpsr;
    psinfo->sp = sp;       
   
    asm volatile("mrc     p15, #0, %0, c1, c0, #0" :"=r"(val));  
    printk(KERN_WARNING "Go to sleep, SCTLR=%08X, cpsr=0x%08X, sp=0x%08X, lr=0x%08X\n", val, cpsr, sp, lr);
    nioregs = 0;
	for (i=0; i<MAX_LOPIOREGS; i++) {
        mapped_address = ioremap(quasariomap[i].start, quasariomap[i].length);
        if (mapped_address) { 
            ioregs[nioregs].start = quasariomap[nioregs].start;
            ioregs[nioregs].end = quasariomap[i].start + quasariomap[i].length;
            ioregs[nioregs].base = mapped_address;
            //printk(KERN_WARNING "   iomap [%d]-- start=0x%08X, end=0x%08X, base=0x%08X\n", nioregs, ioregs[nioregs].start, ioregs[nioregs].end, (u32)ioregs[nioregs].base); 
            nioregs++;
        }
        else {
            printk(KERN_WARNING "   !!! Could not map io region -- start=0x%08X, size=%d\n", quasariomap[i].start, quasariomap[i].length);
        }
	} 
    
    //INTOFF();         /* don't interrupt me! */

    SLEEP_INFO("CPUrunInfo.spsr=0x%08X, CPUrunInfo.sp=0x%08X\n", psinfo->spsr, psinfo->sp);

#ifdef CONFIG_SOC_QUASAR_M30_POWER_DOWN    
    // Keep M30 registers
    qlomap_readl(QBM3_0_IPC, &psinfo->m30reg[0]);
    qlomap_readl(QBM3_0_ISP, &psinfo->m30reg[1]);
    qlomap_readl(QBM3_0_PAGE0, &psinfo->m30reg[2]);
    qlomap_readl(QBM3_0_PAGE1, &psinfo->m30reg[3]);
    qlomap_readl(QBM3_0_PAGE2, &psinfo->m30reg[4]);
    qlomap_readl(QBM3_0_PAGE3, &psinfo->m30reg[5]);
    qlomap_readl(QBM3_0_PPAGE, &psinfo->m30reg[6]);
    qlomap_readl(QBM3_0_M3IEN, &psinfo->m30reg[7]);
    //SLEEP_INFO("QBM3_0_IPC:   0x%08X, QBM3_0_ISP:   0x%08X\n", psinfo->m30reg[0], psinfo->m30reg[1]);
    //SLEEP_INFO("QBM3_0_PAGE0: 0x%08X, QBM3_0_PAGE1: 0x%08X\n", psinfo->m30reg[2], psinfo->m30reg[3]);
    //SLEEP_INFO("QBM3_0_PAGE2: 0x%08X, QBM3_0_PAGE3: 0x%08X\n", psinfo->m30reg[4], psinfo->m30reg[5]);
    //SLEEP_INFO("QBM3_0_PPAGE: 0x%08X, QBM3_0_M3IEN: 0x%08X\n", psinfo->m30reg[6], psinfo->m30reg[7]);
#endif
    // Keep M30 registers
    qlomap_readl(QBM3_1_IPC, &psinfo->m31reg[0]);
    qlomap_readl(QBM3_1_ISP, &psinfo->m31reg[1]);
    qlomap_readl(QBM3_1_PAGE0, &psinfo->m31reg[2]);
    qlomap_readl(QBM3_1_PAGE1, &psinfo->m31reg[3]);
    qlomap_readl(QBM3_1_PAGE2, &psinfo->m31reg[4]);
    qlomap_readl(QBM3_1_PAGE3, &psinfo->m31reg[5]);
    qlomap_readl(QBM3_1_PPAGE, &psinfo->m31reg[6]);
    qlomap_readl(QBM3_1_M3IEN, &psinfo->m31reg[7]);
    
    // copy sleep info to target location
    pSrc = (void*)psinfo;
    pDest = (void*)qlomap_addr(SLEEPINFO_LOCATION);
    size = SLEEPINFO_LENGTH;
    memcpy(pDest, pSrc, size);
    SLEEP_INFO("Copy sleep info, src=0x%08X, dest=0x%08X, src[0]=0x%08X, dest[0]=0x%08X\n", (u32)pSrc, (u32)pDest, *((u32*)pSrc), *((u32*)pDest));

    // Set restore hander and stack
    qlomap_writel(A7GP_A7SRA0, (u32)quasarRestore);
    qlomap_readl(A7GP_A7SRA0, &val);
    SLEEP_INFO("Set Restore handler: ptr=0x%08X, A7GP_A7SRA0=0x%08X\n", (u32)quasarRestore, val);
    qlomap_writel(A7GP_A7SRSP0, (u32)SLEEPINFO_SP_LOCATION);
    qlomap_readl(A7GP_A7SRSP0, &val);
    SLEEP_INFO("Set Restore SP: sp=0x%08X, A7GP_A7SRSP0=0x%08X\n", SLEEPINFO_SP_LOCATION, val);

    // unmap ioregion
    for (i=0; i<nioregs; i++) {
        iounmap(ioregs[i].base);
    }
    nioregs = 0;
  
    // get current contents of CP15 control reg.
    asm volatile("mrc p15, #0, %0, c1, c0, #0" : "=r"(regval));
    // turn off DCache
    regval &= ~(CR_DCACHE_EN);
    asm volatile("mcr p15, #0, %0, c1, c0, #0" :: "r"(regval));
    mb();
	flush_cache_all();  

    // Consider the reference time on the colck source
    // TBD:
    // ???
    
    // go to sleep
    SLEEP_INFO("Go to low level sleep\n");
    quasarSleep();   

	/* should never here */
	BUG();
}

void quasar_lp_start(void)
{
    asm volatile("\n"
        "stmfd   sp!, {r0-r12,lr}   \n\t"   \
        "mrs     r0, cpsr           \n\t"   \
        "mov     r1, sp             \n\t"   \
        "mov     r2, lr             \n\t"   \
        "b       qlop_sleep         \n\t");
}

