/*
 * arch/arm/mach-quasar/quasar63xx_pm.c - QB63xx power management support
 *
 * Created by:	Nicolas Pitre, October 2012
 * Copyright:	(C) 2012-2013  Linaro Limited
 *
 * Some portions of this file were originally written by Achin Gupta
 * Copyright:   (C) 2012  ARM Limited
 *
 * Portions Copyright 2014, 2015 Linux Foundation.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

// #define DEBUG

#include <linux/delay.h>
#include <linux/init.h>
#include <linux/io.h>
#include <linux/kernel.h>
#include <linux/of_address.h>
#include <linux/of_irq.h>
#include <linux/spinlock.h>
#include <linux/errno.h>
#include <linux/irqchip/arm-gic.h>

#include <asm/mcpm.h>
#include <asm/proc-fns.h>
#include <asm/cacheflush.h>
#include <asm/cputype.h>
#include <asm/cp15.h>
#include <asm/cacheflush.h>

#include <linux/arm-cci.h>
#include "core.h"

#define QUASAR63XX_CLUSTERS		                1
#define QUASAR63XX_MAX_CPUS_PER_CLUSTER	        2

#define RSTGEN_BASE                             0x04108000
#define RSTGEN_SIZE                             0x000001B0
#define RSTGEN_PWR_ISOLATE_OFF                  0x00000160
#define RSTGEN_PWR_ON_A7_OFF                    0x00000170
#define RSTGEN_PWR_ON_A7_CPU0_OFF               0x00000174
#define RSTGEN_PWR_ON_A7_CPU1_OFF               0x00000178
#define RSTGEN_PWR_ISOLATE__A7_CPU1_N__MASK     0x00000004
#define RSTGEN_PWR_ISOLATE__A7_CPU1_N__INV_MASK 0xFFFFFFFB 
            
#define PWR_CHECK_MASK                          0x00010003
#define PWR_ON_SW                               0x00010003
#define PWR_ON_W1                               0x00000001
#define PWR_ON_W2                               0x00000002
#define PWR_ON_S1                               0x00010000

#define A7GP_BASE                               0x04310000
#define A7GP_SIZE                               0x00000080
#define A7GP_A7RSTCTR_OFF                       0x00000034
#define A7GP_A7SLPCSR_OFF                       0x00000038
#define A7GP_A7SRA0_OFF                         0x00000044
#define A7GP_A7SRA1_OFF                         0x00000048
#define A7GP_A7RSTCTR__CORE_CPU1__MASK          0x00000020
#define A7GP_A7RSTCTR__POR_CPU1__MASK           0x00000002  
#define A7GP_A7RSTCTR__CORE_CPU1__INV_MASK      0xFFFFFFDF
#define A7GP_A7RSTCTR__POR_CPU1__INV_MASK       0xFFFFFFFD
                                                                
static unsigned int quasar63xx_nr_cpus[QUASAR63XX_CLUSTERS];

static void quasar63xx_pm_set_boot_vector(void)
{
    void __iomem *mapped_address;

    mapped_address = ioremap(A7GP_BASE, A7GP_SIZE);     /* TODO: do something better here, maybe dl -related. */

    writel(quasar_secondary_startup, (void*)mapped_address+A7GP_A7SRA1_OFF);
    
    iounmap(mapped_address);
}

/**
 * quasar63xx_set_resume_addr_pm(() - set the jump address used for warm boot
 *
 * @cluster: mpidr[15:8] bitfield describing cluster affinity level
 * @cpu: mpidr[7:0] bitfield describing cpu affinity level
 * @addr: physical resume address
 */
static void quasar63xx_set_resume_addr_pm(u32 cluster, u32 cpu, u32 addr)
{
	if (quasar63xx_smp_entrypoint == NULL)
	{
		// Linux 4.19 MMU will set quasar63xx_smp_entrypoint(in .text section)
		// attribute as read only after secondary CPU boot.
		// So we skip the resume address assignment during suspend resume 
		quasar63xx_smp_entrypoint = addr;
	}
	mb();
	flush_cache_all();
	return;
}

static DEFINE_SPINLOCK(power_lock);   
/* static TODO: restore this */ void power_up_cpu_pm(unsigned int cpu)
{
	unsigned long reg_value;
	void __iomem* mapped_address;
    void __iomem* mapped_address_a7gpf;
    
	if (cpu != 1)
		return;

	mapped_address = ioremap(RSTGEN_BASE, RSTGEN_SIZE); /* TODO: do something better here, maybe dl -related. */
    // STEP.0. Check if the block has already powered on.
    reg_value = readl((void *)mapped_address + RSTGEN_PWR_ON_A7_CPU1_OFF);
    if(PWR_ON_SW == (reg_value & PWR_CHECK_MASK))
    {
        // DOMAIN_A7_CPU1_PD has already powered on. skip
        iounmap(mapped_address);
        return;
    }

    mapped_address_a7gpf = ioremap(A7GP_BASE, A7GP_SIZE); /* TODO: do something better here, maybe dl -related. */
    spin_lock(&power_lock);
    
    // ?? Assert nCOREPORESET LOW 
    writel(readl((void *)mapped_address_a7gpf + A7GP_A7RSTCTR_OFF) & 
           A7GP_A7RSTCTR__POR_CPU1__INV_MASK,
           (void *)mapped_address_a7gpf + A7GP_A7RSTCTR_OFF);
    
    // STEP.3-8. Power on
    // STEP.3. Change PWR_ON_BLOCKNAME to set the w(1) signal to 1.
    // STEP.4. Wait Tdiff1.
    writel(readl((void *)mapped_address + RSTGEN_PWR_ON_A7_CPU1_OFF) | PWR_ON_W1,
                (void *)mapped_address + RSTGEN_PWR_ON_A7_CPU1_OFF);
    udelay(2);
    // STEP.5. Change PWR_ON_BLOCKNAME to set the w(2) signal to 1.
    // STEP.6. Wait Tdiff2.
    writel(readl((void *)mapped_address + RSTGEN_PWR_ON_A7_CPU1_OFF) | PWR_ON_W2,
           (void *)mapped_address + RSTGEN_PWR_ON_A7_CPU1_OFF);
    udelay(2);
    // STEP.7. Change PWR_ON_BLOCKNAME to set the s(1) signal to 1.
    // STEP.8. Wait Tstart.
    writel(readl((void *)mapped_address + RSTGEN_PWR_ON_A7_CPU1_OFF) | PWR_ON_S1,
           (void *)mapped_address + RSTGEN_PWR_ON_A7_CPU1_OFF);
    udelay(3);

    // STEP.9. Block is now powered up, isolated and held in reset.
    // STEP.10. Turn on all clocks into the block.
    // no clocks to enable
    // STEP.11. Change RSTGEN_PWR_ISOLATE.BLOCKNAME_N to 1 to conenct the block.
    writel(readl((void *)mapped_address + RSTGEN_PWR_ISOLATE_OFF) | RSTGEN_PWR_ISOLATE__A7_CPU1_N__MASK,
           (void *)mapped_address + RSTGEN_PWR_ISOLATE_OFF);

    // STEP.12. Deassert the block reset.
    // release POR CPU1 reset
    // release CORE CPU1 resets
    writel(readl((void *)mapped_address_a7gpf + A7GP_A7RSTCTR_OFF) | 
            A7GP_A7RSTCTR__CORE_CPU1__MASK | A7GP_A7RSTCTR__POR_CPU1__MASK,
            (void *)mapped_address_a7gpf + A7GP_A7RSTCTR_OFF);
            
    spin_unlock(&power_lock); 
    iounmap(mapped_address_a7gpf);
    iounmap(mapped_address);
}

void power_down_cpu_pm(unsigned int cpu)
{
	void __iomem* mapped_address;
    void __iomem* mapped_address_a7gpf;

	if (cpu != 1) {
		return;
	}

    mapped_address = ioremap(RSTGEN_BASE, RSTGEN_SIZE); /* TODO: do something better here, maybe dl -related. */
    mapped_address_a7gpf = ioremap(A7GP_BASE, A7GP_SIZE); /* TODO: do something better here, maybe dl -related. */
    
    spin_lock(&power_lock);
	// A7 CPU1 power down sequence
    // apply the CPU1 resets
    writel(readl((void *)mapped_address_a7gpf + A7GP_A7RSTCTR_OFF) & 
            A7GP_A7RSTCTR__CORE_CPU1__INV_MASK & A7GP_A7RSTCTR__POR_CPU1__INV_MASK,
            (void *)mapped_address_a7gpf + A7GP_A7RSTCTR_OFF);
    // no clocks to disable
    // isolate CPU1
    writel(readl((void *)mapped_address + RSTGEN_PWR_ISOLATE_OFF) &
            RSTGEN_PWR_ISOLATE__A7_CPU1_N__INV_MASK,
            (void *)mapped_address + RSTGEN_PWR_ISOLATE_OFF);
    udelay(1);
    // powerdown
    writel(0x00, (void *)mapped_address + RSTGEN_PWR_ON_A7_CPU1_OFF);
        
    spin_unlock(&power_lock); 
    iounmap(mapped_address_a7gpf);   
    iounmap(mapped_address);
}

/* quasar63xx_cpu_in_wfi_pm(u32 cpu, u32 cluster)
 *
 * @cpu: mpidr[7:0] bitfield describing CPU affinity level within cluster
 * @cluster: mpidr[15:8] bitfield describing cluster affinity level
 *
 * @return: non-zero if and only if the specified CPU is in WFI
 *
 * Take care when interpreting the result of this function: a CPU might
 * be in WFI temporarily due to idle, and is not necessarily safely
 * parked.
 */

int quasar63xx_cpu_in_wfi_pm(u32 cpu, u32 cluster)
{
	unsigned long reg_value;
	void __iomem* mapped_address_a7gpf;
	unsigned long comparison_value;
	int in_wfi;

	if (cpu != 1)
		return 0;

    mapped_address_a7gpf = ioremap(A7GP_BASE, A7GP_SIZE); /* TODO: do something better here, maybe dl -related. */
	reg_value = readl((void *)mapped_address_a7gpf+A7GP_A7SLPCSR_OFF);
	comparison_value = 1 << (2+cpu);

	iounmap(mapped_address_a7gpf);
	
	in_wfi = reg_value & comparison_value;
	return in_wfi;
}

static void quasar63xx_pm_boot_secondary(unsigned int cpu) //, struct task_struct *idle)
{
	/* Tell boot ROM where to execute from. */
	quasar63xx_pm_set_boot_vector();
	power_up_cpu_pm(cpu);
}

static int quasar63xx_pm_power_up(unsigned int cpu, unsigned int cluster)
{
	pr_debug("%s: cpu %u cluster %u\n", __func__, cpu, cluster);
	if (cluster >= QUASAR63XX_CLUSTERS || cpu >= quasar63xx_nr_cpus[cluster]) {
		return -EINVAL;        
	}

	quasar63xx_set_resume_addr_pm(cluster, cpu,
				      virt_to_phys(mcpm_entry_point));
	quasar63xx_pm_boot_secondary(cpu);

	return 0;
}

static int quasar63xx_pm_cluster_power_up(unsigned int cluster)
{
    return 0;
}

static void quasar63xx_pm_powerdown_prepare(unsigned int cpu, unsigned int cluster)
{
	quasar63xx_cpu_in_wfi_pm(cpu, cluster);
}

static void quasar63xx_pm_cluster_powerdown_prepare(unsigned int cluster)
{
}

static void quasar63xx_pm_cpu_cache_disable(void)
{
	v7_exit_coherency_flush(louis);
}

static void quasar63xx_pm_cluster_cache_disable(void)
{
	v7_exit_coherency_flush(all);
	cci_disable_port_by_cpu(read_cpuid_mpidr());
}

static unsigned long const reset_constants[] = {0x11, 0x22};
static int quasar63xx_core_in_reset_pm(unsigned int cpu, unsigned int cluster)
{
	unsigned long reg_value;
	void __iomem* mapped_address_a7gpf;

	if (cpu != 1)
		return 0;

    mapped_address_a7gpf = ioremap(A7GP_BASE, A7GP_SIZE); /* TODO: do something better here, maybe dl -related. */
	reg_value = readl((void *)mapped_address_a7gpf+A7GP_A7RSTCTR_OFF);
	iounmap(mapped_address_a7gpf);  

	if ((reg_value & reset_constants[cpu]) == reset_constants[cpu]) {
		return 0;
	}
	else {
		return 1;
	}
}

#define POLL_MSEC 10
#define TIMEOUT_MSEC 1000

static int quasar63xx_pm_wait_for_powerdown(unsigned int cpu, unsigned int cluster)
{
	unsigned tries;

	pr_debug("%s: cpu %u cluster %u\n", __func__, cpu, cluster);
	BUG_ON(cluster >= QUASAR63XX_CLUSTERS || cpu >= QUASAR63XX_MAX_CPUS_PER_CLUSTER);

	for (tries = 0; tries < TIMEOUT_MSEC / POLL_MSEC; ++tries) {
		/*
		 * Only examine the hardware state if the target CPU has
		 * caught up at least as far as quasar63xx_pm_down():
		 */
		pr_debug("%s(cpu=%u, cluster=%u): RESET_CTRL = 0x%08X\n",
			 __func__, cpu, cluster,
			/* readl_relaxed(scc + RESET_CTRL) */ 0);

		/*
		 * We need the CPU to reach WFI, but the power
		 * controller may put the cluster in reset and
		 * power it off as soon as that happens, before
		 * we have a chance to see STANDBYWFI.
		 *
		 * So we need to check for both conditions:
		 */
		if (quasar63xx_core_in_reset_pm(cpu, cluster) ||
			quasar63xx_cpu_in_wfi_pm(cpu, cluster)) {
			power_down_cpu_pm(cpu);
			return 0; /* success: the CPU is halted */
		}

		/* Otherwise, wait and retry: */
		msleep(POLL_MSEC);
	}

	return -ETIMEDOUT; /* timeout */
}

static void quasar63xx_pm_suspend_prepare(unsigned int cpu, unsigned int cluster)
{
	quasar63xx_set_resume_addr_pm(cluster, cpu, virt_to_phys(mcpm_entry_point));
}

static void quasar63xx_pm_cpu_is_up(unsigned int cpu, unsigned int cluster)
{
	pr_debug("%s: cpu %u cluster %u\n", __func__, cpu, cluster);
	BUG_ON(cluster >= QUASAR63XX_CLUSTERS || cpu >= QUASAR63XX_MAX_CPUS_PER_CLUSTER);
    
	quasar63xx_set_resume_addr_pm(cluster, cpu, 0);
}

static void quasar63xx_pm_cluster_is_up(unsigned int cluster)
{
}

static const struct mcpm_platform_ops quasar63xx_pm_power_ops = {
	.cpu_powerup				= quasar63xx_pm_power_up,
	.cluster_powerup			= quasar63xx_pm_cluster_power_up,
	.cpu_suspend_prepare		= quasar63xx_pm_suspend_prepare,
	.cpu_powerdown_prepare		= quasar63xx_pm_powerdown_prepare,
	.cluster_powerdown_prepare 	= quasar63xx_pm_cluster_powerdown_prepare,
	.cpu_cache_disable			= quasar63xx_pm_cpu_cache_disable,
	.cluster_cache_disable		= quasar63xx_pm_cluster_cache_disable,
	.wait_for_powerdown			= quasar63xx_pm_wait_for_powerdown,
	.cpu_is_up					= quasar63xx_pm_cpu_is_up,
	.cluster_is_up				= quasar63xx_pm_cluster_is_up,
};

/*
 * Enable cluster-level coherency, in preparation for turning on the MMU.
 */
static void __naked quasar63xx_pm_power_up_setup(unsigned int affinity_level)
{
	asm volatile (" \n"
//"   loop2: b loop2 \n"
"	cmp	r0, #1 \n"
"	bxne	lr \n"
"	b	cci_enable_port_for_self ");
}

static int __init quasar63xx_pm_init(void)
{
	unsigned int mpidr, cpu, cluster;
	int ret;

	quasar63xx_nr_cpus[0] = 2; /// TODO: do this dynamically

	if (!cci_probed()) {
		return -ENODEV;
	}

	mpidr = read_cpuid_mpidr();
	cpu = MPIDR_AFFINITY_LEVEL(mpidr, 0);
	cluster = MPIDR_AFFINITY_LEVEL(mpidr, 1);

	pr_debug("%s: cpu %u cluster %u\n", __func__, cpu, cluster);
	if (cluster >= QUASAR63XX_CLUSTERS || cpu >= quasar63xx_nr_cpus[cluster]) {
		pr_err("%s: boot CPU is out of bound!\n", __func__);
		return EINVAL;
	}

	ret = mcpm_platform_register(&quasar63xx_pm_power_ops);
	if (!ret) {
		mcpm_sync_init(quasar63xx_pm_power_up_setup);
		pr_info("Quasar63xx power management initialized\n");
	}
	else {
		pr_info("Quasar63xx power management failed, bad return %d from mcpm_platform_register()\n", (int)mcpm_platform_register);
	}

	return ret;
}

early_initcall(quasar63xx_pm_init);
