/*
 * arch/arm/mach-quasar/platsmp.c
 *
 * Copyright (c) 2014, 2015 Linux Foundation.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */

#include <linux/init.h>
#include <linux/kernel.h>

#include <linux/delay.h>
#include <linux/jiffies.h>
#include <linux/io.h>
#include <linux/smp.h>
#include <asm/sizes.h>
#include <asm/cacheflush.h>
#include <asm/smp_scu.h>

#include <linux/of.h>
#include <linux/of_platform.h>
#include "quasar.h"


#define RSTGEN_BASE                             0x04108000
#define RSTGEN_SIZE                             0x000001B0
#define RSTGEN_PWR_ISOLATE_OFF                  0x00000160
#define RSTGEN_PWR_ON_A7_OFF                    0x00000170
#define RSTGEN_PWR_ON_A7_CPU0_OFF               0x00000174
#define RSTGEN_PWR_ON_A7_CPU1_OFF               0x00000178
#define RSTGEN_PWR_ISOLATE__A7_CPU1_N__MASK     0x00000004
#define RSTGEN_PWR_ISOLATE__A7_CPU1_N__INV_MASK 0xFFFFFFFB 
            
#define PWR_CHECK_MASK                          0x00010003
#define PWR_ON_SW                               0x00010003
#define PWR_ON_W1                               0x00000001
#define PWR_ON_W2                               0x00000002
#define PWR_ON_S1                               0x00010000

#define A7GP_BASE                               0x04310000
#define A7GP_SIZE                               0x00000080
#define A7GP_A7RSTCTR_OFF                       0x00000034
#define A7GP_A7SLPCSR_OFF                       0x00000038
#define A7GP_A7SRA0_OFF                         0x00000044
#define A7GP_A7SRA1_OFF                         0x00000048
#define A7GP_A7RSTCTR__CORE_CPU1__MASK          0x00000020
#define A7GP_A7RSTCTR__POR_CPU1__MASK           0x00000002  
#define A7GP_A7RSTCTR__CORE_CPU1__INV_MASK      0xFFFFFFDF
#define A7GP_A7RSTCTR__POR_CPU1__INV_MASK       0xFFFFFFFD
   
static void quasar63xx_set_boot_vector(void)
{
    void __iomem *mapped_address;

    mapped_address = ioremap(A7GP_BASE, A7GP_SIZE);     /* TODO: do something better here, maybe dl -related. */

    writel(quasar_secondary_startup, (void*)mapped_address+A7GP_A7SRA1_OFF);
    
    iounmap(mapped_address);
}

static DEFINE_SPINLOCK(power_lock);

void quasar63xx_power_up_cpu(unsigned int cpu)
{
	unsigned long reg_value;
	void __iomem* mapped_address;
    void __iomem* mapped_address_a7gpf;
	if (cpu != 1)
		return;

	mapped_address = ioremap(RSTGEN_BASE, RSTGEN_SIZE); /* TODO: do something better here, maybe dl -related. */
    // STEP.0. Check if the block has already powered on.
    reg_value = readl((void *)mapped_address + RSTGEN_PWR_ON_A7_CPU1_OFF);
    if(PWR_ON_SW == (reg_value & PWR_CHECK_MASK))
    {
        // DOMAIN_A7_CPU1_PD has already powered on. skip
        iounmap(mapped_address);
        return;
    }

    mapped_address_a7gpf = ioremap(A7GP_BASE, A7GP_SIZE); /* TODO: do something better here, maybe dl -related. */
    spin_lock(&power_lock);
    
    // ?? Assert nCOREPORESET LOW 
    writel(readl((void *)mapped_address_a7gpf + A7GP_A7RSTCTR_OFF) & 
           A7GP_A7RSTCTR__POR_CPU1__INV_MASK,
           (void *)mapped_address_a7gpf + A7GP_A7RSTCTR_OFF);
    
    // STEP.3-8. Power on
    // STEP.3. Change PWR_ON_BLOCKNAME to set the w(1) signal to 1.
    // STEP.4. Wait Tdiff1.
    writel(readl((void *)mapped_address + RSTGEN_PWR_ON_A7_CPU1_OFF) | PWR_ON_W1,
                (void *)mapped_address + RSTGEN_PWR_ON_A7_CPU1_OFF);
    udelay(2);
    // STEP.5. Change PWR_ON_BLOCKNAME to set the w(2) signal to 1.
    // STEP.6. Wait Tdiff2.
    writel(readl((void *)mapped_address + RSTGEN_PWR_ON_A7_CPU1_OFF) | PWR_ON_W2,
           (void *)mapped_address + RSTGEN_PWR_ON_A7_CPU1_OFF);
    udelay(2);
    // STEP.7. Change PWR_ON_BLOCKNAME to set the s(1) signal to 1.
    // STEP.8. Wait Tstart.
    writel(readl((void *)mapped_address + RSTGEN_PWR_ON_A7_CPU1_OFF) | PWR_ON_S1,
           (void *)mapped_address + RSTGEN_PWR_ON_A7_CPU1_OFF);
    udelay(3);

    // STEP.9. Block is now powered up, isolated and held in reset.
    // STEP.10. Turn on all clocks into the block.
    // no clocks to enable
    // STEP.11. Change RSTGEN_PWR_ISOLATE.BLOCKNAME_N to 1 to conenct the block.
    writel(readl((void *)mapped_address + RSTGEN_PWR_ISOLATE_OFF) | RSTGEN_PWR_ISOLATE__A7_CPU1_N__MASK,
           (void *)mapped_address + RSTGEN_PWR_ISOLATE_OFF);

    // STEP.12. Deassert the block reset.
    // release POR CPU1 reset
    // release CORE CPU1 resets
    writel(readl((void *)mapped_address_a7gpf + A7GP_A7RSTCTR_OFF) | 
            A7GP_A7RSTCTR__CORE_CPU1__MASK | A7GP_A7RSTCTR__POR_CPU1__MASK,
            (void *)mapped_address_a7gpf + A7GP_A7RSTCTR_OFF);
            
    spin_unlock(&power_lock); 
    iounmap(mapped_address_a7gpf);
    iounmap(mapped_address);
}

void quasar63xx_power_down_cpu(unsigned int cpu)
{
	void __iomem* mapped_address;
    void __iomem* mapped_address_a7gpf;

	if (cpu != 1) {
		return;
	}

    mapped_address = ioremap(RSTGEN_BASE, RSTGEN_SIZE); /* TODO: do something better here, maybe dl -related. */
    mapped_address_a7gpf = ioremap(A7GP_BASE, A7GP_SIZE); /* TODO: do something better here, maybe dl -related. */
    
    spin_lock(&power_lock);
	// A7 CPU1 power down sequence
    // apply the CPU1 resets
    writel(readl((void *)mapped_address_a7gpf + A7GP_A7RSTCTR_OFF) & 
            A7GP_A7RSTCTR__CORE_CPU1__INV_MASK & A7GP_A7RSTCTR__POR_CPU1__INV_MASK,
            (void *)mapped_address_a7gpf + A7GP_A7RSTCTR_OFF);
    // no clocks to disable
    // isolate CPU1
    writel(readl((void *)mapped_address + RSTGEN_PWR_ISOLATE_OFF) &
            RSTGEN_PWR_ISOLATE__A7_CPU1_N__INV_MASK,
            (void *)mapped_address + RSTGEN_PWR_ISOLATE_OFF);
    udelay(1);
    // powerdown
    writel(0x00, (void *)mapped_address + RSTGEN_PWR_ON_A7_CPU1_OFF);
        
    spin_unlock(&power_lock); 
    iounmap(mapped_address_a7gpf);   
    iounmap(mapped_address);
}

/*
 * Write pen_release in a way that is guaranteed to be visible to all
 * observers, irrespective of whether they're taking part in coherency
 * or not.  This is necessary for the hotplug code to work reliably.
 */
static void write_pen_release(int val)
{
	pen_release = val;
	smp_wmb();
	sync_cache_w(&pen_release);
}

static DEFINE_SPINLOCK(boot_lock);

static void quasar_secondary_init(unsigned int cpu)
{
	/*
	 * let the primary processor know we're out of the
	 * pen, then head off into the C entry point
	 */
	write_pen_release(-1);

	/*
	 * Synchronise with the boot thread.
	 */
	spin_lock(&boot_lock);
	spin_unlock(&boot_lock);
}

static int quasar_boot_secondary(unsigned int cpu, struct task_struct *idle)
{
	unsigned long timeout;

	/*
	 * set synchronisation state between this boot processor
	 * and the secondary one
	 */
	spin_lock(&boot_lock);

	/*
	 * set boot vector of the secondary cpu and power it up
	 */
	quasar63xx_set_boot_vector();
	quasar63xx_power_up_cpu(cpu);

	/*
	 * The secondary processor is waiting to be released from
	 * the holding pen - release it, then wait for it to flag
	 * that it has been released by resetting pen_release.
	 *
	 * Note that "pen_release" is the hardware CPU ID, whereas
	 * "cpu" is Linux's internal ID.
	 */
	write_pen_release(cpu);

	timeout = jiffies + (6 * HZ);
	while (time_before(jiffies, timeout)) {
		smp_rmb();
		if (pen_release == -1)
			break;

		udelay(10);
	}
	/*
	 * now the secondary core is starting up let it run its
	 * calibrations, then wait for it to finish
	 */
	spin_unlock(&boot_lock);
	return pen_release != -1 ? -ENOSYS : 0;
}


/*
 * Initialise the CPU possible map early - this describes the CPUs
 * which may be present or become present in the system.
 */
static void __init quasar_smp_init_cpus(void)
{
	unsigned int i, ncores;

	ncores = 2;
	for (i = 0; i < ncores; i++)
		set_cpu_possible(i, true);
}

void __ref quasar_cpu_die(unsigned int cpu)
{
}

struct smp_operations quasar_smp_ops __initdata = {
	.smp_init_cpus		= quasar_smp_init_cpus,
	.smp_secondary_init	= quasar_secondary_init,
	.smp_boot_secondary	= quasar_boot_secondary,
#ifdef CONFIG_HOTPLUG_CPU
	.cpu_die		= quasar_cpu_die,
#endif
};

