/*
 * If distributed as part of the Linux kernel, this code is licensed under the
 * terms of the GPL v2.
 *
 * Otherwise, the following license terms apply:
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1) Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2) Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3) The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESSED OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
#ifndef _DEV_PCI_FCUVAR_H_
#define	_DEV_PCI_FCUVAR_H_

#include <linux/gw/chelsea_io.h> 
// #include "opt_fcu.h"
// #include <machine/bus.h>
// #include <dev/pci/fcureg.h>
#include <linux/delay.h>
// #include "fcureg.h"
#include <linux/gw/fcureg.h>
#include <asm/io.h>

/* Makefileに書くとエラーになるので退避 */
#define FCU_DEBUG 1
#define FCU_DEBUG_IOCTL 1
#define FCUD_DEBUG_IOCTL 1

/*
 * FBI interrupt cause
 */
typedef struct fbi_intr_cause {
	u_long	intr;				/* 割込み要因レジスタ */
	u_char	cmd;				/* 割込み要因データ( 0) */
	u_char	ack;				/* 割込み要因データ(+1) */
	u_char	rdy;				/* 割込み要因データ(+2) */
	u_char	dma3;				/* 割込み要因データ(+3) */
	u_char	dma4;				/* 割込み要因データ(+4) */
} fbi_intr_t;

/*
 * sc_flag operation (sc_flag専用)
 */
#define FLAG_SET(flag)	(sc->sc_flags |= (flag))
#define FLAG_CLR(flag)	(sc->sc_flags &= ~(flag))
#define FLAG_CHK(flag)	(sc->sc_flags & (flag))

/*
 * bit operation
 */
#define bit_set(a, b)		((a) |= (1<<(b)))
#define bit_clr(a, b)		((a) &= ~(1<<(b)))
#define bit_is_set(a, b)	((a) & (1<<(b)))
#define bit_is_clr(a, b)	(((a) & (1<<(b))) == 0)


/********************************************************************/
/*
 * SIMACの不具合対策(1) ----------------------------------------------
 * SIMAC 1.0(ES1), SIMAC 1.1(ES2)
 *
 * lpux_bus_space_[read|write]_[1,2,4]()を定義する
 *
 * CPUからPCI空間をreadする時にウソのデータが読める不具合の暫定対策
 * rapiドライバと同じ対策としてread前にwaitを入れる
 * waitは2usecとする
 * write前にもwaitが必要になった
 * (上記内容はmips1.5対応時にカットした。構造は将来の為保持している)
 */

#define LPUX_BUS_SPACE_READ_DEFINE(BYTES, TYPE)				\
static __inline TYPE							\
lpux_bus_space_read_##BYTES(void *addr)					\
{									\
	return (ioread##BYTES(addr));					\
}

#define LPUX_BUS_SPACE_WRITE_DEFINE(BYTES, TYPE)			\
static __inline void							\
lpux_bus_space_write_##BYTES(TYPE value, void *addr)			\
{									\
	iowrite##BYTES(value, addr);					\
}


LPUX_BUS_SPACE_READ_DEFINE(8, u8)
LPUX_BUS_SPACE_READ_DEFINE(16, u16)
LPUX_BUS_SPACE_READ_DEFINE(32, u32)
LPUX_BUS_SPACE_WRITE_DEFINE(8, u8)
LPUX_BUS_SPACE_WRITE_DEFINE(16, u16)
LPUX_BUS_SPACE_WRITE_DEFINE(32, u32)

#undef LPUX_BUS_SPACE_READ_DEFINE
#undef LPUX_BUS_SPACE_WRITE_DEFINE



/*
 * SIMACの不具合対策(2) ----------------------------------------------
 * SIMAC 1.0(ES1), SIMAC 1.1(ES2)
 *
 * lpux_bsw[1,2,4]_[io|mem]()を定義する
 *
 * CPUからPCI空間をwriteした直後にダミーのwrite動作が必要になった
 * I/O空間:	バイトレーンレジスタに0Lをwriteすることでダミー動作とする
 * メモリ空間:	同じデータを2度書きすることでダミー動作とする
 * (上記内容はmips1.5対応時にカットした。構造は将来の為保持している)
 */
/*
#define LPUX_BSW_IO_DEFINE(BYTES, TYPE)					\
static __inline void							\
lpux_bsw##BYTES##_io(TYPE value, void *addr)			\
{									\
	iowrite##BYTES(value, addr);					\
}
*/

#define LPUX_BSW_IO_DEFINE(BYTES, TYPE)					\
static __inline void							\
lpux_bsw##BYTES##_io(TYPE value, void *addr)					\
{									\
	iowrite##BYTES(value, addr);					\
}

#define LPUX_BSW_MEM_DEFINE(BYTES, TYPE)				\
static __inline void							\
lpux_bsw##BYTES##_mem(TYPE value, void *addr)			\
{									\
	iowrite##BYTES(value, addr);					\
}

LPUX_BSW_IO_DEFINE(32, u32)		/* decl lpux_bsw4_io() */
LPUX_BSW_MEM_DEFINE(8, u8)		/* decl lpux_bsw1_mem() */
LPUX_BSW_MEM_DEFINE(16, u16)		/* decl lpux_bsw2_mem() */
LPUX_BSW_MEM_DEFINE(32, u32)		/* decl lpux_bsw4_mem() */

#undef LPUX_BSW_IO_DEFINE
#undef LPUX_BSW_MEM_DEFINE



/*
 * SIMACの不具合対策(3) ----------------------------------------------
 *
 * 不具合というよりも仕様
 * 割り込みクリア後に割り込み信号がディアサートまでの時間を確保する
 * ために1usecのディレイを入れる
 *
 * FCU_LPUX_SIMAC_INTRCLR_DELAYが(!0)ならば適用する
 */

#define lpux_bsw4_io_intr_clear(sc, ofs, val)			\
	lpux_bsw4_io_intr_clear_body((sc)->io_vaddr, (sc)->irqno, (ofs), (val))

static __inline void 
lpux_bsw4_io_intr_clear_body(unsigned long io_vaddr, unsigned int irqno, long offset, u32 value)
{

#ifdef  FCU_LPUX_SIMAC_INTRCLR_DELAY
//      s = splhigh();
//      lpux_bsw4_io(space, handle, offset, htobe32(value));
//      DELAY(1);
//      splx(s);
        disable_irq(irqno);
        lpux_bsw32_io((u32)(value), (void *)(io_vaddr + offset));
        udelay(1);
        enable_irq(irqno);
#else
        lpux_bsw32_io((u32)(value), (void *)(io_vaddr + offset));
#endif
}

#define PRINT_ALERT(fmt, ...)   pr_alert(fmt, ##__VA_ARGS__)
#define PRINT_ERR(fmt, ...)     pr_err(fmt, ##__VA_ARGS__)

#define PRINT_NOTICE( fmt, ...) pr_notice(fmt, ##__VA_ARGS__)
#define PRINT_INFO(fmt, ...)    pr_info(fmt, ##__VA_ARGS__)
#define PRINT_DEBUG(fmt, ...)   pr_debug(fmt, ##__VA_ARGS__)















/*
 * IO空間アクセス bus_space_read()/bus_space_write()
 */
#define bsr4_io(sc, ofs)						\
	lpux_bus_space_read_32((void *)(sc->io_vaddr + (ofs)))
/*
	htobe32(lpux_bus_space_read_4((sc)->sc_iot, (sc)->sc_ioh, (ofs)))
*/
#define bsw4_io(sc, ofs, val)						\
	lpux_bsw32_io((val), (void *)(sc->io_vaddr + (ofs)))
/*
	lpux_bsw4_io((sc)->sc_iot, (sc)->sc_ioh, (ofs), htobe32(val))
*/

/*
 * DMA割込みマスクリセット
 */
#define bsw4_io_intrmask_all(sc)					\
	bsw4_io(sc, FBI_OFS_IO_DMAMODE, 0)

/*
 * DMA全チャンネルを停止
 */
#define bsw4_io_dmaabort_all(sc)					\
	bsw4_io(sc, FBI_OFS_IO_DMACTL, FBI_BIT_DMACTL_DMAALL)

/*
 * 割込み要因ビットチェック for FBI_OFS_IO_INTRCAUSE
 */
#define FBI_IS_DPRAM(x)		((x) & FBI_BIT_INTRCAUSE_DPRAM)
#define FBI_IS_TOFACE(x)	((x) & FBI_BIT_INTRCAUSE_TO_DP)
#define FBI_IS_DMA1(x)		((x) & FBI_BIT_INTRCAUSE_DMA1)
#define FBI_IS_DMA2(x)		((x) & FBI_BIT_INTRCAUSE_DMA2)
#define FBI_IS_DMA3A(x)		((x) & FBI_BIT_INTRCAUSE_DMA3A)
#define FBI_IS_DMA4A(x)		((x) & FBI_BIT_INTRCAUSE_DMA4A)
#define FBI_IS_DMA3B(x)		((x) & FBI_BIT_INTRCAUSE_DMA3B)
#define FBI_IS_DMA4B(x)		((x) & FBI_BIT_INTRCAUSE_DMA4B)
#define FBI_IS_DMAALL(x)	((x) & FBI_BIT_INTRCAUSE_DMALL)


/*
 * レディビットチェック for FBI_OFS_IO_RESETCTL
 */
#define FBI_IS_PCIREADY(x)	((x) & FBI_BIT_RESETCTL_PCIRDY)
#define FBI_IS_FCUREADY(x)	((x) & FBI_BIT_RESETCTL_FCURDY)


/*
 * DMA転送数
 */
#define FBI_DMA12_SIZE(w, l)						\
	(((w) & FBI_BIT_DMA12WIDTH_MASK) |				\
	 (((l - 1) & FBI_BIT_DMA12LINE_MASK) << FBI_SHIFT_DMA12SIZE))
#define FBI_DMA12_WIDTH(w)						\
	((w) & FBI_BIT_DMA12WIDTH_MASK)
#define FBI_DMA12_LINE(l)						\
	(((l + 1) >> FBI_SHIFT_DMA12SIZE) & FBI_BIT_DMA12LINE_MASK)
#define FBI_DMA34_SIZE(x)						\
	((x) & FBI_BIT_DMA34SIZE_MASK)


/*
 * メモリ空間アクセス bus_space_read()/bus_space_write()
 */
/*
 * ベースアドレスに対するオフセット
 */
/* FCUエリアのベース */
#define FBI_OFS_DP_FCUBASE	(FBI_OFS_MM_DPRAM_BASE)
/* PCI(SCU)エリアのベース */
#define FBI_OFS_DP_BASE_SIZE	(0x00000200)
#define FBI_OFS_DP_PCIBASE	(FBI_OFS_MM_DPRAM_BASE + FBI_OFS_DP_BASE_SIZE)


/*
 * DPRAMのマッピング
 * FCUBASEとPCIBASEに対する共通なoffset値
 */

#define FBI_OFS_DP_INTR_TOP			(0x00000000)
#define FBI_DP_INTR_SIZE			(0x00000010)
#define FBI_OFS_DP_INTR_WRITE		(FBI_OFS_DP_INTR_TOP + 0x0000010)
#define FBI_OFS_DP_INTR_ACK			(FBI_OFS_DP_INTR_WRITE + 4)
#define FBI_OFS_DP_INTR_READY		(FBI_OFS_DP_INTR_ACK   + 4)
#define FBI_OFS_DP_INTR_DMA3		(FBI_OFS_DP_INTR_READY + 4)
#define FBI_OFS_DP_INTR_DMA4		(FBI_OFS_DP_INTR_DMA3  + 4)
#define FBI_OFS_DP_PACKET			(0x00000080)
#define FBI_OFS_DP_STATUS			(0x00000100)

#if 0
/* FBIモニタ用旧定義 */
#define FBI_OFS_DP_INTR_WRITE_ORG	(FBI_OFS_DP_INTR_TOP)
#define FBI_OFS_DP_INTR_ACK_ORG		(FBI_OFS_DP_INTR_WRITE_ORG + 1)
#define FBI_OFS_DP_INTR_READY_ORG	(FBI_OFS_DP_INTR_ACK_ORG   + 1)
#define FBI_OFS_DP_INTR_DMA3_ORG	(FBI_OFS_DP_INTR_READY_ORG + 1)
#define FBI_OFS_DP_INTR_DMA4_ORG	(FBI_OFS_DP_INTR_DMA3_ORG  + 1)

/* 変換マクロ（新定義→旧定義）*/
#define FBI_OFS_DP_INTR_CHG(x)		(((x) - FBI_OFS_DP_INTR_WRITE) / 4)
#endif

/*
 * DPRAMアクセス bus_space_read()/bus_space_write()
 */
#define bsr1_fcubase(sc, ofs)						\
	lpux_bus_space_read_8((void *)(sc->mem_vaddr + FBI_OFS_DP_FCUBASE + (ofs)))
#define bsr2_fcubase(sc, ofs)						\
	lpux_bus_space_read_16((void *)(sc->mem_vaddr + FBI_OFS_DP_FCUBASE + (ofs)))
#define bsr4_fcubase(sc, ofs)						\
	lpux_bus_space_read_32((void *)(sc->mem_vaddr + FBI_OFS_DP_FCUBASE + (ofs)))
#define bsw1_fcubase(sc, ofs, val)					\
	lpux_bsw8_mem((u8)val,						\
			(void *)(sc->mem_vaddr + FBI_OFS_DP_FCUBASE + (ofs)))
#define bsw2_fcubase(sc, ofs, val)					\
	lpux_bsw16_mem((u16)(val),					\
			(void *)(sc->mem_vaddr + FBI_OFS_DP_FCUBASE + (ofs)))
#define bsw4_fcubase(sc, ofs, val)					\
	lpux_bsw32_mem((u32)(val),					\
			(void *)(sc->mem_vaddr + FBI_OFS_DP_FCUBASE + (ofs)))
#define bsr1_pcibase(sc, ofs)						\
	lpux_bus_space_read_8((void *)(sc->mem_vaddr + FBI_OFS_DP_PCIBASE + (ofs)))
#define bsr2_pcibase(sc, ofs)						\
	lpux_bus_space_read_16((void *)(sc->mem_vaddr + FBI_OFS_DP_PCIBASE + (ofs)))
#define bsr4_pcibase(sc, ofs)						\
	lpux_bus_space_read_32((void *)(sc->mem_vaddr + FBI_OFS_DP_PCIBASE + (ofs)))
#define bsw1_pcibase(sc, ofs, val)					\
	lpux_bsw8_mem((u8)val,					\
			(void *)(sc->mem_vaddr + FBI_OFS_DP_PCIBASE + (ofs)))
#define bsw2_pcibase(sc, ofs, val)					\
	lpux_bsw16_mem((u16)(val),					\
			(void *)(sc->mem_vaddr + FBI_OFS_DP_PCIBASE + (ofs)))
#define bsw4_pcibase(sc, ofs, val)					\
	lpux_bsw32_mem((u32)(val),					\
			(void *)(sc->mem_vaddr + FBI_OFS_DP_PCIBASE + (ofs)))
/*
	lpux_bsw4_mem((sc)->sc_memt,					\
				(sc)->sc_memh,					\
				(FBI_OFS_DP_PCIBASE + (ofs)),			\
				htobe32(val))
*/

/*
 * 割込み要因セット(1をwrite)
 */
#define bsw1_pcibase_intr(sc, ofs)					\
do {									\
	bsw1_pcibase(sc, ofs, 1);					\
/*	bsw1_pcibase(sc, FBI_OFS_DP_INTR_CHG(ofs), 1);			*/ \
} while(0)

/*
 * 割込み要因クリア(0をwrite)
 */
#define bsw1_fcubase_intrclear(sc, ofs)					\
do {									\
	bsw1_fcubase(sc, ofs, 0);					\
/*	bsw1_fcubase(sc, FBI_OFS_DP_INTR_CHG(ofs), 0);			*/ \
} while(0)

/*
 * 割込み要因クリア初期化用
 */
#define bsw4_fcubase_intrclear(sc, ofs)					\
	bsw4_fcubase(sc, ofs, 0)
#define bsw4_pcibase_intrclear(sc, ofs)					\
	bsw4_pcibase(sc, ofs, 0)


/**
 * macros for counting and rounding
 * linux環境では見当たらなかったので、netbsdからコピー
 */
#ifndef howmany
#define howmany(x, y)	(((x)+((y)-1))/(y))
#endif

#endif	/* !_DEV_PCI_FCUVAR_H_ */


