/*
 * If distributed as part of the Linux kernel, this code is licensed under the
 * terms of the GPL v2.
 *
 * Otherwise, the following license terms apply:
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1) Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2) Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3) The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESSED OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <linux/types.h>
#include <stdarg.h>
#include <linux/linkage.h>
#include <linux/printk.h>
#include <linux/device.h>
#include <linux/fs.h>
#include <asm/io.h>
#include <linux/string.h>
#include <linux/ctype.h>
#include <linux/err.h>
#include <linux/sched.h>
#include <linux/pci.h>
#include <linux/interrupt.h>
#include <linux/uaccess.h>
#include <linux/module.h>
//#include <asm/hw_irq.h>
#include <linux/irq.h>
#include <linux/irqdomain.h>
#include "fcuvar.h"
// #include "fcureg.h"
#include <linux/gw/fcureg.h>
#include "fcu_subr.h"			/* for plist_t */
// #include "fcucom.h"
#include <linux/gw/fcucom.h>
#include "fax_share.h"		/* XXX */
// #include "gw-irq.h"		/* ヘッダ提供されるまでの暫定 */


#define DRIVER_NAME	"fcu"		/* ドライバ名 */

extern struct devlist fax_devlist[2];


/*
 * plist_t count
 */
#define PLIST_CNT	8			/* packet FIFOの個数 */

#ifdef LOWPOWER2
/*
 * 省エネ状態定義
 */
#define FCU_LP2_WAKEUP	0	/* 起動状態 */
#define FCU_LP2_DOWN	1	/* LP2モード状態 */
#endif /* LOWPOWER2 */


/*
 * packet size buffer
 */
typedef struct pkt_buf {
	u_char	buf[FCUCMDSIZE];
} pkt_buf_t;


struct fcu_softc {
	fax_driver_header_t		sc_header;
	struct pci_dev                  *pci_dev;
	u_short			sc_flags;	/* flags */
#define FSF_ISOPEN		0x0001	/* open されている */
#define FSF_RSLEEP		0x0002	/* 受信データ待ち sleep 中 */
#define FSF_WSLEEP		0x0004	/* 送信バッファ待ち sleep 中 */
#define FSF_RSELWANT	0x0008	/* 受信データselect要求 */
#define FSF_WSELWANT	0x0010	/* 送信fifo空きselect要求 */
#define FSF_WRITING		0x0020	/* 送信データ書き込み中 */
#define FSF_ACKREQ		0x0040	/* ACK保留中 */
#define FSF_INITIAL		0x0080	/* 初期化処理中 */
#define FSF_FCURDY		0x0100	/* FCU READY確認割込み起動済 */
#define FSF_RDYINTR		0x0200	/* READY確認割込み受信済 */
#define FSF_REBOOT		0x0400	/* FCUリブート検出 */
	irqreturn_t		(*fcud_intr)(void *);	/* fcud intr handler */
	plist_t			sc_rpl;		/* read  FIFO */
	plist_t			sc_wpl;		/* write FIFO */
	pkt_buf_t		sc_rbuf;	/* read  work buffer */
	pkt_buf_t		sc_wbuf;	/* write work buffer */
	pkt_buf_t		sc_irbuf;	/* intr read  work buffer */
	pkt_buf_t		sc_iwbuf;	/* intr write work buffer */
	fax_share_param_t	*sc_share;
	wait_queue_head_t	sc_rpl_q;
	wait_queue_head_t	sc_wpl_q;
	unsigned long		io_vaddr;
	unsigned long		io_addr;
	unsigned long		io_len;
	unsigned long		mem_vaddr;
	unsigned long		mem_addr;
	unsigned long		mem_len;
	unsigned int		irqno;
};

typedef struct fcu_softc	fcu_t;

fcu_t g_fcu_softc;
EXPORT_SYMBOL(g_fcu_softc);
static fcu_t *sc = &g_fcu_softc;


#ifdef USE_PCI_CONF_RECOVER
/*
 * コンフィグ保存領域
 */
static pciconf_t fcu_pci_conf;
#endif	/* !USE_PCI_CONF_RECOVER */

#ifdef FCU_STR_SUPPORT

/*
 * 省エネ対応したFCU(FACE3.5)装着有無情報
 * 1:あり
 * 0:なし
 */
static int is_fcu_face35 = 0;
#endif	/* !FCU_STR_SUPPORT */

#ifdef LOWPOWER2
/*
 * LowPower２モード中かどうかの状態
 * FCU_LP2_WAKEUP	:起動状態
 * FCU_LP2_DOWN		:LP2モード状態
 */
static int fcu_lp2_state = FCU_LP2_WAKEUP;
#endif /* LOWPOWER2 */


#ifdef FCU_STR_SUPPORT
int	fcu_chaeck_face35(void);
void	fcu_pci_conf_recover(void);
#endif	/* !FCU_STR_SUPPORT */

#ifdef LOWPOWER2
static void	fcu_lp2_down(void *);
static void	fcu_lp2_wakeup(void *);
#endif /* LOWPOWER2 */


/*
 * prototypes
 */
static int fcuopen( struct inode *inode, struct file *filp );
static int fcuclose( struct inode *inode, struct file *filp );
static ssize_t fcuread( struct file* p_file, char* buf, size_t count, loff_t* f_pos );
static ssize_t fcuwrite( struct file* p_file, const char* buf, size_t count, loff_t* f_pos );
static long fcuioctl( struct file *filp, unsigned int cmd, unsigned long arg);
/*
 * static prototypes
 */
static int    fcud_intr_establish(irqreturn_t (*)(void *));
static irqreturn_t fcud_intr_dummy(void *);

struct file_operations fcu_FileOps = {
 .owner			= THIS_MODULE,
 .open			= fcuopen,
 .release		= fcuclose,
 .read			= fcuread,
 .write			= fcuwrite,
 .unlocked_ioctl= fcuioctl,
 .compat_ioctl= fcuioctl,
};
EXPORT_SYMBOL(fcu_FileOps);

#ifdef FCU_STR_SUPPORT
int		fcu_chaeck_face35(void);
void	fcu_pci_conf_recover(void);
#endif	/* !FCU_STR_SUPPORT */


#ifdef	FCU_DEBUG_IOCTL
static void fcu_debug_0(fcu_t *);
static void fcu_debug_ioctl_func(fcu_t *, int);
#endif	/* !FCU_DEBUG_IOCTL */

/*
extern int fcudstart(struct pci_dev*, void*, void*, unsigned int);
extern irqreturn_t fcudintr(void*);
*/
static __inline void fcu_fgate_assert_bugfix(fcu_t *sc);
static __inline void fcu_read_intrcause(fcu_t *sc, fbi_intr_t *ic);
static __inline void fcu_intr_ack(fcu_t *sc);
static __inline void fcu_intr_cmd(fcu_t *sc);
static __inline void fcu_intr_ready(fcu_t *sc);
irqreturn_t fcuintr(int irq, void *dev_id);

static  char    sys_str_flag    = ECOMODE_NOT_STR;      /* システムSTR移行フラグ 0=STR中  1=STR復帰中    *//* @15S_murata_016    */

/*
 * @SRAM_INIT
 * reboot_flagを設けた意図について。
 *
 * FCUからリブートが通知された際に、FCUDのwrite処理の中でリブートフラグがクリアされてしまう。
 * その為、後続のread処理にてリブートフラグが参照できなくなる。
 * それにより、read処理でなにもしないため、上位Mdにリブートが挙がらず、FAXが起動しない。
 *
 * 本事象を回避するため、read処理で参照するリブートフラグを別に設ける。
 */
static  char    reboot_flag     = REBOOT_FLAG_FALSE;                /* リブートフラグ  REBOOT_FLAG_FALSE=非リブート REBOOT_FLAG_FALSE=リブート中  *//* @SRAM_INIT     */
/********************************************************************/

/*
 * fcud attach routine
 */
static int fcud_intr_establish(irqreturn_t (*func)(void *))
{

	if (func == NULL) {			/* NULLなら登録しない */
		PRINT_ERR("[OS]fcu:%s: fcud_intr_establish function is null\n", sc->sc_header.sc_name);
		return (1);
	}

	sc->fcud_intr = func;
	return (0);
}

static irqreturn_t 
fcud_intr_dummy(void *arg)
{

	/* 念のために全割り込みクリア */
	bsw4_io(sc, FBI_OFS_IO_DMAINTRCLR, FBI_BIT_DMAINTRCLR_DALL);
	return IRQ_HANDLED;
}
	

/********************************************************************/
/* PCI BASE領域全クリア初期化用 */
static __inline void
init_pcibase_intr(fcu_t *sc)
{
	unsigned long ofs;
	for(ofs = 0x00000000; ofs < FBI_OFS_DP_BASE_SIZE; ofs += 4) {
		bsw4_pcibase_intrclear(sc, FBI_OFS_DP_INTR_TOP + ofs);
	}
}

static __inline void
fcu_write_dpram(fcu_t *sc, u_char *ptr)
{
	int i;
	u_int *p = (u_int *)ptr;		/* big endian */

	u_int *be_chg = (u_int *)ptr;

	/*
	 * DPRAMのバイトレーン制御対応
	 */
//	for (i = 0; i < FCUCMDSIZE; i += 4, be_chg++) {
//		*be_chg = cpu_to_be32(*be_chg);
//	}

	for (i = 0; i < FCUCMDSIZE; i += 4, p++) {
		bsw4_pcibase(sc, FBI_OFS_DP_PACKET + i, *p);
	}
	bsw1_pcibase_intr(sc, FBI_OFS_DP_INTR_WRITE);
	bsw4_io(sc, FBI_OFS_IO_INTRCTL, FBI_BIT_INTRCTL_INTR);
}

static __inline void
fcu_read_dpram(fcu_t *sc, u_char *ptr)
{
	int i;
	u_int *p = (u_int *)ptr;		/* big endian */

	u_int *be_chg = (u_int *)ptr;

	for (i = 0; i < FCUCMDSIZE; i += 4, p++)
		*p = bsr4_fcubase(sc, FBI_OFS_DP_PACKET + i);
	bsw1_pcibase_intr(sc, FBI_OFS_DP_INTR_ACK);
	bsw4_io(sc, FBI_OFS_IO_INTRCTL, FBI_BIT_INTRCTL_INTR);

	/*
	 * DPRAMのバイトレーン制御対応
	 */
//	for (i = 0; i < FCUCMDSIZE; i += 4, be_chg++) {
//		*be_chg = cpu_to_be32(*be_chg);
//	}

}

/********************************************************************/
static __inline void
fcu_wakup_reader(fcu_t *sc)
{
	PRINT_DEBUG("[OS]fcu:fcu_wakup_reader\n");
	if (FLAG_CHK(FSF_RSLEEP)) {
		FLAG_CLR(FSF_RSLEEP);
		PRINT_DEBUG("[OS]fcu:fcu_wakup_reader2\n");
		wake_up_interruptible(&sc->sc_rpl_q);
	}
}

static __inline void
fcu_wakup_writer(fcu_t *sc)
{
	PRINT_DEBUG("[OS]fcu:fcu_wakup_writer\n");
	if (FLAG_CHK(FSF_WSLEEP)) {
		FLAG_CLR(FSF_WSLEEP);
		PRINT_DEBUG("[OS]fcu:fcu_wakup_writer2\n");
		wake_up_interruptible(&sc->sc_wpl_q);
	}
}

/**
 * FCUドライバの管理情報を初期化する関数
 */
static void fcu_initial(void){

	/**
	 * 全体をゼロクリア
	 */
	memset(sc, 0, sizeof(fcu_t));

	/**
	 * ゼロクリア以外に必要な初期化
	 */
	FLAG_SET(FSF_INITIAL);
	sys_str_flag    = ECOMODE_NOT_STR;              /* システムSTRを復帰にする    *//* @15S_murata_016    */
	reboot_flag     = REBOOT_FLAG_FALSE;                        /* 非リブートに設定         *//* @SRAM_INIT         */
	init_waitqueue_head(&sc->sc_rpl_q);
	init_waitqueue_head(&sc->sc_wpl_q);

	/* ダミーの割り込みハンドラをセット */
	sc->fcud_intr = fcud_intr_dummy;

}

/********************************************************************/

/*
 * attach
 */
int fcu_attach(void *self){

	int error = -1;
	int ret;
	
#ifdef USE_PCI_CONF_RECOVER	
	pciconf_t *cf = &fcu_pci_conf;
#endif	/* !USE_PCI_CONF_RECOVER */
	
	PRINT_INFO("[OS]fcu:fcu_attach: fcu initialization is started...\n");
	
	// init parameter 
	fcu_initial();
	
	// get shared parameter
	sc->sc_share = get_fax_share_param();
	sc->pci_dev = sc->sc_share->pci_dev;
	
	// enable pci
	error = pci_enable_device(sc->pci_dev);
	if (error){
		PRINT_ERR("[OS]fcu:%s: can't enable pci device, error=%d\n", DRIVER_NAME, error);
		goto pci_enable_error;
	}

	// enable bus_master
	pci_set_master(sc->pci_dev);

#if 0
	// io mapping
	sc->io_addr = pci_resource_start(sc->pci_dev, 0);
	sc->io_len = pci_resource_len(sc->pci_dev, 0);

	PRINT_INFO("[OS]fcu:io_addr=%lx, io_len=%ld\n", sc->io_addr, sc->io_len);
	
	if(!request_mem_region(sc->io_addr, sc->io_len, DEV_NAME)){
		PRINT_ERR("[OS]fcu:%s: request_region_io error =%d\n", DRIVER_NAME, error);
		error = -1;
		goto io_request_error;
	}
	
	sc->io_vaddr = (unsigned long)ioremap_nocache(sc->io_addr, sc->io_len);
	if(!sc->io_vaddr){
		PRINT_ERR("[OS]fcu:%s: remap_io error\n", DRIVER_NAME);
		error = -1;
		goto io_remap_error;
	}
	PRINT_INFO("[OS]fcu:io_vaddr=%lx\n", sc->io_vaddr);
#endif

	// memory mapping
	sc->mem_addr = pci_resource_start(sc->pci_dev, 0);
	sc->mem_len = pci_resource_len(sc->pci_dev, 0);
	
	PRINT_INFO("[OS]fcu:mem_addr=%lx, mem_len=%ld\n", sc->mem_addr, sc->mem_len);

	if(!request_mem_region(sc->mem_addr, sc->mem_len, DEV_NAME)){
		PRINT_ERR("[OS]fcu:%s: request_region_mem error =%d\n", DRIVER_NAME, error);
		error = -1;
		goto mem_request_error;
	}
	
	sc->mem_vaddr = (unsigned long)ioremap_nocache(sc->mem_addr, sc->mem_len);
	if(!sc->mem_vaddr){
		PRINT_ERR("[OS]fcu:%s: remap_mem error\n", DRIVER_NAME);
		error = -1;
		goto mem_remap_error;
	}
	PRINT_INFO("[OS]fcu:mem_vaddr=%lx\n", sc->mem_vaddr);

	sc->io_addr  = sc->mem_addr;
	sc->io_len   = sc->mem_len;
	sc->io_vaddr = sc->mem_vaddr;

	
	/*
	 * DPRAM割り込み要因エリアをクリア
	 */
	init_pcibase_intr(sc);
	
	/*
	 * DMA割り込みマスクとバイトレーンをリセット
	 */
	PRINT_INFO("[OS]fcu:bytelane set=%lx + %lx\n", sc->io_vaddr, FBI_OFS_IO_BYTELANE);
	bsw4_io(sc, FBI_OFS_IO_BYTELANE, FBI_BIT_BYTELANE_TARGET | FBI_BIT_BYTELANE_DPRAM);
	bsw4_io_intrmask_all(sc);

	/*
	 * DMAを停止
	 */
	bsw4_io_dmaabort_all(sc);
	

	
#if 0	/* test */
	{
		u16 vendor_id;
		u16 device_id;
		u16 sub_vendor_id;
		u16 sub_device_id;
		ret = pci_read_config_byte(sc->pci_dev, PCI_VENDOR_ID, &vendor_id);
		PRINT_DEBUG("[OS]fcu:ret=%d vendor_id=%lx\n", ret, vendor_id);
		ret = pci_read_config_byte(sc->pci_dev, PCI_DEVICE_ID, &device_id);
		PRINT_DEBUG("[OS]fcu:ret=%d device_id=%lx\n", ret, device_id);
		ret = pci_read_config_byte(sc->pci_dev, PCI_SUBSYSTEM_VENDOR_ID, &sub_vendor_id);
		PRINT_DEBUG("[OS]fcu:ret=%d sub_vendor_id=%lx\n", ret, sub_vendor_id);		
		ret = pci_read_config_byte(sc->pci_dev, PCI_SUBSYSTEM_ID, &sub_device_id);		
		PRINT_DEBUG("[OS]fcu:ret=%d sub_device_id=%lx\n", ret, sub_device_id);		
	}
#endif
	
	/*
	 * 割り込みハンドラを登録
	 */
//	sc->irqno = GW_IRQ_BASE + GW_IRQ_FCU_INTA;
	sc->irqno = sc->pci_dev->irq;
	PRINT_DEBUG("[OS]fcu:irqno=%d\n", sc->irqno);
	error = request_irq(sc->irqno, fcuintr, IRQF_SHARED, DEV_NAME, (void *)sc->pci_dev);
	if (error){
		PRINT_ERR("[OS]fcu:%s: request_irq fail, error %d\n", DRIVER_NAME, error );
		goto request_irq_error;
	}
	
	/*
	 * fifo allocate
	 */
	if (plalloc(&sc->sc_rpl, PLIST_CNT, sizeof(pkt_buf_t), sc->irqno)
			== -1) {
		PRINT_ALERT("[OS]fcu:%s: couldn't allocate plist:r\n", sc->sc_header.sc_name);
		goto fifo_allocate_error;
	}
	if (plalloc(&sc->sc_wpl, PLIST_CNT, sizeof(pkt_buf_t), sc->irqno) // forMT
			== -1) {
		PRINT_ALERT("[OS]fcu:%s: couldn't allocate plist:w\n", sc->sc_header.sc_name);
		plfree(&sc->sc_rpl);
		goto fifo_allocate_error;
	}
	
#ifdef USE_PCI_CONF_RECOVER	
	/*
	 * PCIコンフィグレジスタの保存処理
	 */
	// TODO 第二引数と第三引数で何を渡せばよいか確認中
	// pci_conf_save(cf, pa->pa_pc, pa->pa_tag);
	pci_conf_save(cf,0,0);
#endif	/* !USE_PCI_CONF_RECOVER */
#ifdef FCU_STR_SUPPORT
	/*
	 * FCU(FACE3.5)装着有無確認 
	 * 省エネ未対応FCUが装着されることはないので、FCUがattachされた場合には必ず1（省エネ対応FCU装着）を返すようにする
	 */
	is_fcu_face35 = 1;
#endif	/* !FCU_STR_SUPPORT */
	
#ifdef LOWPOWER2
	/* LP2フック関数の登録 */
	/* LP2移行時呼ぶ関数の登録 */
 	lp2downhook_establish(fcu_lp2_down, sc, 0);
	/* LP2復帰時呼ぶ関数の登録 */
	lp2wakeuphook_establish(fcu_lp2_wakeup, sc, 0);
#endif /* LOWPOWER2 */

	
	// キャラクタデバイス系の処理は共通部に移管
	
	// 共通部に値をセット
	sc->sc_share->io_vaddr = sc->io_vaddr;
	sc->sc_share->io_addr = sc->io_addr;
	sc->sc_share->io_len = sc->io_len;
	sc->sc_share->mem_vaddr = sc->mem_vaddr;
	sc->sc_share->mem_addr = sc->mem_addr;
	sc->sc_share->mem_len = sc->mem_len;
	sc->sc_share->irqno = sc->irqno;
	sc->sc_share->fax_intr_establish = fcud_intr_establish;
	
	PRINT_INFO("[OS]fcu:fcu attached\n");

#if 0 // for MT 
	PRINT_DEBUG("[OS]fcu:pcidev = %p\n" ,(void *)sc->pci_dev );
	PRINT_DEBUG("[OS]fcu:sc_flags = %d\n" ,sc->sc_flags );
	PRINT_DEBUG("[OS]fcu:sc_rpl cnt = %d\n" ,plcnt(&sc->sc_rpl));
	PRINT_DEBUG("[OS]fcu:sc_wpl cnt = %d\n" ,plemp(&sc->sc_wpl));
	PRINT_DEBUG("[OS]fcu:io_vaddr = %lx\n" ,sc->io_vaddr );
	PRINT_DEBUG("[OS]fcu:io_addr = %lx\n" ,sc->io_addr );
	PRINT_DEBUG("[OS]fcu:io_len = %lx\n" ,sc->io_len );
	PRINT_DEBUG("[OS]fcu:mem_vaddr = %lx\n" ,sc->mem_vaddr );
	PRINT_DEBUG("[OS]fcu:mem_addr = %lx\n" ,sc->mem_addr );
	PRINT_DEBUG("[OS]fcu:mem_len = %lx\n" ,sc->mem_len );
	PRINT_DEBUG("[OS]fcu:irqno = %d\n" ,sc->irqno);
#endif

	return 0;
	
fifo_allocate_error:
	free_irq(sc->irqno, NULL);
request_irq_error:
	iounmap((void *)sc->mem_vaddr);
mem_remap_error:
	release_mem_region(sc->mem_addr, sc->mem_len);
mem_request_error:
	iounmap((void *)sc->io_vaddr);
io_remap_error:
	release_mem_region(sc->io_addr, sc->io_len);
io_request_error:
	pci_disable_device(sc->pci_dev);
pci_enable_error:
	return error;
	
}
EXPORT_SYMBOL(fcu_attach);

/*
 * detach
 */
int fcu_detach(void *self){

	PRINT_INFO("[OS]fcu:%s: fcu detach\n", DRIVER_NAME);
	
	free_irq(sc->irqno, NULL);
	iounmap((void *)sc->mem_vaddr);
	release_mem_region(sc->mem_addr, sc->mem_len);
	iounmap((void *)sc->io_vaddr);
	release_mem_region(sc->io_addr, sc->io_len);
	pci_disable_device(sc->pci_dev);
	memset(sc, 0, sizeof(fcu_t));
	return 0;
}
EXPORT_SYMBOL(fcu_detach);

/*
 * open
 */
static int fcuopen( struct inode* inode, struct file* p_file )
{

	PRINT_INFO("[OS]fcu:%s: [fcuopen start]\n", DRIVER_NAME);
	
	if ((p_file->f_mode & (FMODE_READ|FMODE_WRITE)) != (FMODE_READ|FMODE_WRITE)){
		PRINT_ERR("[OS]fcu:mode error\n");
		return (-EACCES);		/* needs O_RDWR */
	}

	if (FLAG_CHK(FSF_ISOPEN)){		/* already opened */
		PRINT_ERR("[OS]fcu:already open\n");
		return (-EBUSY);
	}

	FLAG_SET(FSF_ISOPEN);
	PRINT_DEBUG("[OS]fcu:%s: [fcuopen end]\n", DRIVER_NAME);

	return (0);
}
EXPORT_SYMBOL(fcuopen);


/*
 * close
 */
static int fcuclose( struct inode* inode, struct file* p_file )
{
	PRINT_INFO("[OS]fcu:%s: [fcuclose start]\n", DRIVER_NAME);

	disable_irq(sc->irqno);

	/*
	 * レディビットをリセット
	 */
	bsw4_io(sc, FBI_OFS_IO_RESETCTL, 0);	/* FBI bug : half => long */

	/*
	 * plistの初期化
	 */
	plinit(&sc->sc_rpl);
	plinit(&sc->sc_wpl);
	
#ifdef FCU_HANDLE_UNEXPECTED_NOTRDY
	/* 
	 * OPENフラグのクリア
	 */
	FLAG_CLR(FSF_ISOPEN);
#else	
	/*
	 * sc_flagsのリセット
	 */
	sc->sc_flags = FSF_INITIAL;
#endif
	
	enable_irq(sc->irqno);
	PRINT_DEBUG("[OS]fcu:%s: [fcuclose end]\n", DRIVER_NAME);

	return (0);
}
EXPORT_SYMBOL(fcuclose);


/*
 * write
 */
static ssize_t fcuwrite( struct file* p_file, const char* buf, size_t count, loff_t* f_pos )
{
	int error;

	PRINT_DEBUG("[OS]fcu:%s: device write\n", DRIVER_NAME);

	if (count < FCUCMDSIZE){
		PRINT_ERR("[OS]fcu:command size error %d\n", count);
		return (-EINVAL);
	}

	disable_irq(sc->irqno);

	error = copy_from_user((caddr_t)sc->sc_wbuf.buf, buf, FCUCMDSIZE);
	if (error) {
		PRINT_ERR("[OS]fcu:copy_from_user error\n");
		enable_irq(sc->irqno);
		return (error);
	}

    if( reboot_flag == REBOOT_FLAG_TRUE ) {        /* FCUがrebootした */          /* @SRAM_INIT               */
    	reboot_flag = REBOOT_FLAG_FALSE;
		PRINT_ERR("[OS]fcu:fcu_reboot\n");
		enable_irq(sc->irqno);
		return (-EBUSY);
	}

	/*
	 * if FSF_WRITING is off, no packet in fifo.
	 */
	if (FLAG_CHK(FSF_WRITING)||(sys_str_flag==ECOMODE_STR)) {		/* write to fifo */ /* @15S_murata_016  */
		PRINT_DEBUG("[OS]fcu:write to fifo\n");
		while (plput(&sc->sc_wpl, sc->sc_wbuf.buf) == -1) {
			FLAG_SET(FSF_WSLEEP);
			enable_irq(sc->irqno);
			if ((wait_event_interruptible(sc->sc_wpl_q, !(FLAG_CHK(FSF_WSLEEP)))) != 0){
				PRINT_ERR("[OS]fcu:wait error\n");
				return (-EINTR);
			}
			disable_irq(sc->irqno);
			if( reboot_flag == REBOOT_FLAG_TRUE ) {    /* FCUがrebootした */
				reboot_flag = REBOOT_FLAG_FALSE;
				PRINT_ERR("[OS]fcu:fcu_reboot\n");
				enable_irq(sc->irqno);
				return (-EBUSY);
			}
			/* 2007/06/08 @maruyama 0001 */ 
			if ( !FLAG_CHK(FSF_WRITING) && (sys_str_flag==ECOMODE_NOT_STR) && plemp(&sc->sc_wpl)== PLIST_CNT ) {	/* FIFOが空なら直接書き込み	*/ /* @15S_murata_016  */
				PRINT_DEBUG("[OS]fcu:write_direct\n");
				FLAG_SET(FSF_WRITING);
				fcu_write_dpram(sc, sc->sc_wbuf.buf);
				break;
			}
		}
	} else {				/* write direct */
		PRINT_DEBUG("[OS]fcu:write direct\n");
		FLAG_SET(FSF_WRITING);
		fcu_write_dpram(sc, sc->sc_wbuf.buf);
	}

	enable_irq(sc->irqno);

	return (0);
}
EXPORT_SYMBOL(fcuwrite);


/*
 * read
 */
static ssize_t fcuread( struct file* p_file, char* buf, size_t count, loff_t* f_pos )
{
	int error;

	PRINT_DEBUG("[OS]fcu:%s: device read\n", DRIVER_NAME);

	if (count < FCUCMDSIZE) {
		PRINT_ERR("[OS]fcu:command size error%d\n", count);
		return (-EINVAL);
	}

	disable_irq(sc->irqno);

	if (FLAG_CHK(FSF_REBOOT)) {		/* FCUがrebootした */
		PRINT_ERR("[OS]fcu:fcu_reboot\n");
		enable_irq(sc->irqno);
		return (-EBUSY);
	}
	
	if ( sys_str_flag == ECOMODE_STR ){                             /* @15S_murata_016  */
		/* STR中ならI/O Errorを返す    */
		enable_irq(sc->irqno);
		return ( -EIO );
	}

	while (plget(&sc->sc_rpl, sc->sc_rbuf.buf) == -1) {
		PRINT_DEBUG("[OS]fcu:wait for read\n");
		FLAG_SET(FSF_RSLEEP);
		enable_irq(sc->irqno);
		if ((wait_event_interruptible(sc->sc_rpl_q, !(FLAG_CHK(FSF_RSLEEP)))) != 0){
			PRINT_ERR("[OS]fcu:wait error\n");
			return (-EINTR);
		}
		disable_irq(sc->irqno);
		if (FLAG_CHK(FSF_REBOOT)) {	/* FCUがrebootした */
			PRINT_ERR("[OS]fcu:fcu_reboot\n");
			enable_irq(sc->irqno);
			return (-EBUSY);
		}
		
		if ( sys_str_flag == ECOMODE_STR ){                             /* @15S_murata_016  */
		/* STR中ならI/O Errorを返す    */
			enable_irq(sc->irqno);
			return ( -EIO );
		}
		
	}
	/*
	 * requested packet read
	 */
	if (FLAG_CHK(FSF_ACKREQ) && plemp(&sc->sc_rpl) > 0) {
		PRINT_DEBUG("[OS]fcu:read exist\n");
		fcu_read_dpram(sc, sc->sc_irbuf.buf);
		plput(&sc->sc_rpl, sc->sc_irbuf.buf); /* no error */
		FLAG_CLR(FSF_ACKREQ);
	}

	error = copy_to_user(buf, (caddr_t)sc->sc_rbuf.buf, FCUCMDSIZE);
	if( error ){
		PRINT_ERR("[OS]fcu:copy_to_user error\n");
		enable_irq(sc->irqno);
		return (-EFAULT);
	}

	enable_irq(sc->irqno);

	return (FCUCMDSIZE);
}
EXPORT_SYMBOL(fcuread);

/*
 * FBI不具合対策
 * スキャン開始時FGATEアサート状態を回避する処理
 */
static __inline void
fcu_fgate_assert_bugfix(fcu_t *sc)
{
	/*
	 * (1)DMA1 FIFOをダミーライト
	 * (2)スキャナデータ転送終了コマンド1回目
	 * (3)スキャナデータ転送終了コマンド2回目
	 * (念のために割り込み禁止にして処理する)
	 */
	lpux_bsw32_mem(0, (void *)(sc->mem_vaddr + FBI_OFS_MM_DMA1_BASE));
	bsw4_io(sc, FBI_OFS_IO_SCANENDCMD, 0);
	bsw4_io(sc, FBI_OFS_IO_SCANENDCMD, 0);
}

/*
 * ready check
 */
static __inline u_long
fcu_get_ready(fcu_t *sc)
{
	u_long ret = FCU_NOTREADY;
	u_long rdybit;

	rdybit = bsr4_io(sc, FBI_OFS_IO_RESETCTL);/* FBI bug : half => long */
	PRINT_DEBUG("[OS]fcu:fcu_get_ready: reset control register %lx.\n", rdybit);

	/*
	 * 自分のレディをセットしていない状態
	 */
	if (!FBI_IS_PCIREADY(rdybit)){
		PRINT_INFO("[OS]fcu:%s: SCU is not ready yet!\n", DRIVER_NAME );
		return (ret);
	}
	disable_irq(sc->irqno);

	/*
	 * FCUレディを確認していない
	 */
	if (!FLAG_CHK(FSF_FCURDY) && FBI_IS_FCUREADY(rdybit)) {
		PRINT_INFO("[OS]fcu:%s: check FCU ready!\n", DRIVER_NAME );
		FLAG_SET(FSF_FCURDY);		/* FCUレディを確認した */

		/*
		 * FBI不具合対策
		 */
		fcu_fgate_assert_bugfix(sc);

		/*
		 * FCUレディ割込みをセット
		 */
		bsw1_pcibase_intr(sc, FBI_OFS_DP_INTR_READY);
		/*
		 * DPRAM割込み起動
		 */
		bsw4_io(sc, FBI_OFS_IO_INTRCTL, FBI_BIT_INTRCTL_INTR);
	}

	/*
	 * FCUレディ割込み起動後 && SCUレディ確認割込み受信後
	 */
	if (FLAG_CHK(FSF_FCURDY|FSF_RDYINTR) == (FSF_FCURDY|FSF_RDYINTR)){
		ret = FCU_READY;
	} else {
		PRINT_INFO("[OS]fcu:%s : FCURDY_%s", DRIVER_NAME, FLAG_CHK(FSF_FCURDY)   ? "ON\n"   : "OFF\n");
		PRINT_INFO("[OS]fcu:%s : RDYINTR_%s", DRIVER_NAME, FLAG_CHK(FSF_RDYINTR)  ? "ON\n"  : "OFF\n");
	}

	/*
	 * 両方の割込みが揃った
	 */
	if (ret == FCU_READY) {
		FLAG_CLR(FSF_INITIAL);		/* 初期化中状態終わり */
		FLAG_CLR(FSF_REBOOT);		/* FCUリブート検出終わり */
		if( reboot_flag == REBOOT_FLAG_TRUE ) {
			PRINT_INFO("[OS]fcu:%s : REBOOT_FLAG DOWN\n", DRIVER_NAME);
			reboot_flag = REBOOT_FLAG_FALSE;
		}
	}

	enable_irq(sc->irqno);

	return (ret);
}


/*
 * ioctl
 */
static long fcuioctl( struct file* p_file, unsigned int cmd, unsigned long arg )
{

	
	u_int fcuready;
	u_int sys_str_flag_tmp;
	int error = 0;
	PRINT_DEBUG( "[OS]fcu:%s: fcuioctl\n", DRIVER_NAME);

	switch (cmd) {
	case FIOCSETRDY:
		/* FBI bug : half => long */
		PRINT_INFO("[OS]fcu:[fcuioctl][%s] FIOCSETRDY\n", DRIVER_NAME);
		bsw4_io(sc, FBI_OFS_IO_RESETCTL, FBI_BIT_RESETCTL_PCIRDY);
		break;
	case FIOCGETRDY:
		PRINT_DEBUG("[OS]fcu:[fcuioctl][%s] FIOCGETRDY\n", DRIVER_NAME);
		fcuready = (u_int)fcu_get_ready(sc);
		PRINT_DEBUG("[OS]fcu:fcu_get_ready val=%d\n", fcuready);
		error = copy_to_user((u_int *)arg, &fcuready, sizeof(u_int));
		if( error ){
			PRINT_ERR("[OS]fcu:[fcuioctl][%s] Invalid argument\n", DRIVER_NAME);
			return (error);
		}
		break;
	case FIOCGETSTR:                                            /* @15S_murata_016  */
		/* STR中なら0、STR中でない場合は1を返す   */
		PRINT_DEBUG("[OS]fcu:[fcuioctl][%s] FIOCGETSTR\n", DRIVER_NAME);
		sys_str_flag_tmp = (u_int)sys_str_flag;
		PRINT_DEBUG("[OS]fcu:fcu_get_str val=%d\n", sys_str_flag_tmp);
		error = copy_to_user((u_int *)arg, &sys_str_flag_tmp, sizeof(u_int));
		if( error ){
			PRINT_ERR("[OS]fcu:[fcuioctl][%s] Invalid argument\n", DRIVER_NAME);
			return (error);
		}
		break;		
	case FIOCDEBUG:
		PRINT_INFO("[OS]fcu:[fcuioctl][%s] FIOCDEBUG\n", DRIVER_NAME);
#ifdef	FCU_DEBUG_IOCTL
		fcu_debug_ioctl_func(sc, *(int *)arg);
#endif	/* !FCU_DEBUG_IOCTL */
		break;
	default:
		return (-EINVAL);
	}

	return (error);
}
EXPORT_SYMBOL(fcuioctl);

/*
 * intr
 */
static __inline void fcu_read_intrcause(fcu_t *sc, fbi_intr_t *ic)
{
	/*
	 * 全データがセットされない場合があるようになったので初期化
	 */
	memset(ic, 0, sizeof(*ic));
	/*
	 * 割り込み要因レジスタを読んでクリアする期間は割り込み禁止する
	 */
	ic->intr = bsr4_io(sc, FBI_OFS_IO_INTRCAUSE);
	PRINT_DEBUG("[OS]fcu:intrcause=%lx\n", ic->intr);
	/*
	 * DPRAM割り込みの場合だけDPRAM割り込みクリア
	 */
	if (FBI_IS_DPRAM(ic->intr))
		lpux_bsw4_io_intr_clear(sc, FBI_OFS_IO_INTRCTL,
			FBI_BIT_INTRCTL_INTRCLR);

	/* printf("intr = %lx \n",ic->intr); */
	/*
	 * DPRAMエリアを読むのをDPRAM割り込み発生時だけにした
	 */
	if (FBI_IS_DPRAM(ic->intr)) {
#if 1
		ic->cmd  = bsr1_fcubase(sc, FBI_OFS_DP_INTR_WRITE);
		ic->ack  = bsr1_fcubase(sc, FBI_OFS_DP_INTR_ACK);
		ic->rdy  = bsr1_fcubase(sc, FBI_OFS_DP_INTR_READY);
		ic->dma3 = bsr1_fcubase(sc, FBI_OFS_DP_INTR_DMA3);
		ic->dma4 = bsr1_fcubase(sc, FBI_OFS_DP_INTR_DMA4);
# else 
		ic->cmd  = bsr1_fcubase(sc, FBI_OFS_DP_INTR_WRITE_ORG);
		ic->ack  = bsr1_fcubase(sc, FBI_OFS_DP_INTR_ACK_ORG);
		ic->rdy  = bsr1_fcubase(sc, FBI_OFS_DP_INTR_READY_ORG);
		ic->dma3 = bsr1_fcubase(sc, FBI_OFS_DP_INTR_DMA3_ORG);
		ic->dma4 = bsr1_fcubase(sc, FBI_OFS_DP_INTR_DMA4_ORG);
#endif
		if (ic->cmd) {
			PRINT_DEBUG("[OS]fcu:[Interrupt] FCU command\n");
			bsw1_fcubase_intrclear(sc, FBI_OFS_DP_INTR_WRITE);
		}
		if (ic->ack) {
			PRINT_DEBUG("[OS]fcu:[Interrupt] FCU ack check\n");
			bsw1_fcubase_intrclear(sc, FBI_OFS_DP_INTR_ACK);
		}
		if (ic->rdy) {
			PRINT_DEBUG("[OS]fcu:[Interrupt] FCU rdy check\n");
			bsw1_fcubase_intrclear(sc, FBI_OFS_DP_INTR_READY);
		}
		if (ic->dma3) {
			PRINT_DEBUG("[OS]fcu:[Interrupt] FCU dma3 check\n");
			bsw1_fcubase_intrclear(sc, FBI_OFS_DP_INTR_DMA3);
		}
		if (ic->dma4) {
			PRINT_DEBUG("[OS]fcu:[Interrupt] FCU dma4 check\n");
			bsw1_fcubase_intrclear(sc, FBI_OFS_DP_INTR_DMA4);
		}
	}
	PRINT_DEBUG("[OS]fcu:fbi intrcause2\n");
}


/* ACK割込み */
static __inline void fcu_intr_ack(fcu_t *sc)
{
	PRINT_DEBUG("[OS]fcu:fcu_intr_ack was invoked.\n");
	/*
	 * 次のpacketがあれば続けて送信
	 */
	if (plget(&sc->sc_wpl, sc->sc_iwbuf.buf) == 0) {
		fcu_write_dpram(sc, sc->sc_iwbuf.buf);
		fcu_wakup_writer(sc);
		PRINT_DEBUG("[OS]fcu:[fcu_intr_ack] Send next packet\n");
	} else
		FLAG_CLR(FSF_WRITING);		/* packetが無くなった */
}


/* パケットライト割込み */
static __inline void fcu_intr_cmd(fcu_t *sc)
{
	PRINT_DEBUG("[OS]fcu:fcu_intr_cmd was invoked.\n");
	/*
	 * fifoに空きがあればreadする(と同時にACKを返す)
	 * fifoがfullだったらFSF_ACKREQをsetするだけ
	 */
	if (plemp(&sc->sc_rpl) > 0) {
		fcu_read_dpram(sc, sc->sc_irbuf.buf);
		plput(&sc->sc_rpl, sc->sc_irbuf.buf);
		fcu_wakup_reader(sc);
	} else
		FLAG_SET(FSF_ACKREQ);		/* fifoが一杯で受けられない */
}


/* SCUレディ確認割込み */
static __inline void fcu_intr_ready(fcu_t *sc)
{
	u_long rdybit;

	rdybit = bsr4_io(sc, FBI_OFS_IO_RESETCTL); /* FBI bug : half => long */
	PRINT_INFO("[OS]fcu:rdybit: %lx\n", rdybit);

	if (FBI_IS_PCIREADY(rdybit)) {
		/*
		 * 初期化中ならば立ち上げ処理
		 */
		if (FLAG_CHK(FSF_INITIAL))
			FLAG_SET(FSF_RDYINTR);	/* 割込み受けたよフラグ */
		/*
		 * 通常動作中ならばFCUのリブート
		 */
		else {
			PRINT_INFO("[OS]fcu:reboot detect\n");
			/*
			 * パケットwrite中
			 */
			if (FLAG_CHK(FSF_WRITING)) {
				FLAG_CLR(FSF_WRITING);	/* フラグクリア */
				plinit(&sc->sc_wpl);	/* write plist初期化 */
			}
			FLAG_SET(FSF_INITIAL);	/* 初期化中にする */
			FLAG_CLR(FSF_FCURDY);	/* FCUレディ確認前にする */
			FLAG_SET(FSF_REBOOT);	/* FCUリブート検出にする */
			reboot_flag = REBOOT_FLAG_TRUE;     /* リブート中にする     *//* @SRAM_INIT     */
			fcu_wakup_reader(sc);	/* sleepしてたら起こす */
			fcu_wakup_writer(sc);	/* sleepしてたら起こす */
			/*
			 * FBI不具合対策
			 */
			fcu_fgate_assert_bugfix(sc);
		}
	} else
		PRINT_ERR("[OS]fcu:%s: SCU is not ready yet!\n", DRIVER_NAME);
}

irqreturn_t fcuintr(int irq, void *dev_id)
{
	fbi_intr_t ic;
	
	PRINT_DEBUG("[OS]fcu:fcuintr_intrcause called\n");
	
#ifdef LOWPOWER2
	/* LP2モード中は処理せず 0 を返す */
	if(fcu_lp2_state == FCU_LP2_DOWN){
		return(0);
	}
#endif /* LOWPOWER2 */

	fcu_read_intrcause(sc, &ic);

	PRINT_DEBUG("[OS]fcu:fcuintr_intrcause=%lx\n", ic.intr);
	/* 該当割込み以外の場合は処理せず 0 を返す */
	if((ic.intr & FBI_BIT_INTRCAUSE_ALL) == 0){
		return(IRQ_NONE);
	}

	/*
	 * DPRAM割り込み時だけ処理する
	 */
	if (FBI_IS_DPRAM(ic.intr)) {
		if (ic.ack)
			fcu_intr_ack(sc);
		if (ic.cmd)
			fcu_intr_cmd(sc);
		if (ic.rdy)
			fcu_intr_ready(sc);
	}

	/*
	 * fcudの割込みハンドラを呼び出す
	 * sc->fcud_intrはNULLにならない
	 */
	if (FBI_IS_DMAALL(ic.intr) || ic.dma3 || ic.dma4)
		(*sc->fcud_intr)(&ic);

	/*
	 * 割込みを処理したのでIRQ_HANDLEDを返す
	 */
	return IRQ_HANDLED;
}

int fcu_pci_runtime_suspend(struct device* dev){
	PRINT_DEBUG("[OS]fcu:fcu_pci_runtime_suspend called\n");
	sys_str_flag = ECOMODE_STR;       /* STR中に設定      *//* @15S_murata_016    */
	FLAG_CLR(FSF_FCURDY);
	return 0;
}
EXPORT_SYMBOL(fcu_pci_runtime_suspend);

int fcu_pci_runtime_resume(struct device* dev){
	PRINT_DEBUG("[OS]fcu:fcu_pci_runtime_resume called\n");
	sys_str_flag = ECOMODE_NOT_STR;     /* STR復帰に設定 */  
	bsw4_io(sc, FBI_OFS_IO_BYTELANE, FBI_BIT_BYTELANE_TARGET | FBI_BIT_BYTELANE_DPRAM);
	FLAG_CLR(FSF_WRITING);              /* ACKを受けて解除される書き込み中フラグをクリア */  /* @15S_murata_016  */
	return 0;
}
EXPORT_SYMBOL(fcu_pci_runtime_resume);

#ifdef FCU_STR_SUPPORT/*
 * FCU(FACE3.5)装着有無確認
 * 0:FACE3.5なし
 * 1:FACE3.5あり
 */
int
fcu_chaeck_face35(void){
	PRINT_DEBUG("[OS]fcu:fcu_chaeck_face35 called\n");
    /* 新FCU-fcu_chaeck_face35の戻り値は常にONで返すようにする。 */
    is_fcu_face35 = 1;
	return (is_fcu_face35);
}
EXPORT_SYMBOL(fcu_chaeck_face35);

/*
 * コンフィグレジスタを書き戻す
 */
void
fcu_pci_conf_recover(void){
	PRINT_DEBUG("[OS]fcu:fcu_pci_conf_recover called\n");
#ifdef USE_PCI_CONF_RECOVER
	/* コンフィグレジスタ取得 */
	pciconf_t* cf = &fcu_pci_conf;
	/* コンフィグレジスタ書き戻し */
	pci_conf_recover(cf);
#endif	/* !USE_PCI_CONF_RECOVER */
}
#endif	/* !FCU_STR_SUPPORT */
EXPORT_SYMBOL(fcu_pci_conf_recover);

#ifdef LOWPOWER2
/*
 * 状態をLP2モード中にする
 */
static void
fcu_lp2_down(void *arg){
	fcu_lp2_state = FCU_LP2_DOWN;
}

/*
 * 状態を起動中にする
 */
static void
fcu_lp2_wakeup(void *arg){
	fcu_lp2_state = FCU_LP2_WAKEUP;
}
#endif /* LOWPOWER2 */



/********************************************************************/
#ifdef	FCU_DEBUG_IOCTL
static void
fcu_debug_0(fcu_t *sc)
{
	fbi_intr_t ic;
	u_long rdybit;

	rdybit = bsr4_io(sc, FBI_OFS_IO_RESETCTL);
	printk("[OS]fcu:rdy=%x\n", (int)rdybit);


	printk("[OS]fcu:\n[$Id: fcu.c,v 1.11 2011/12/08 00:31:26 takubo Exp $]\n");

	printk("[OS]fcu:[softc]\n");

	printk("[OS]fcu:[sc_flags]\n");
	printk("[OS]fcu:%s", FLAG_CHK(FSF_ISOPEN)   ? "FSF_ISOPEN\n"   : "");
	printk("[OS]fcu:%s", FLAG_CHK(FSF_RSLEEP)   ? "FSF_RSLEEP\n"   : "");
	printk("[OS]fcu:%s", FLAG_CHK(FSF_WSLEEP)   ? "FSF_WSLEEP\n"   : "");
	printk("[OS]fcu:%s", FLAG_CHK(FSF_RSELWANT) ? "FSF_RSELWANT\n" : "");
	printk("[OS]fcu:%s", FLAG_CHK(FSF_WSELWANT) ? "FSF_WSELWANT\n" : "");
	printk("[OS]fcu:%s", FLAG_CHK(FSF_WRITING)  ? "FSF_WRITING\n"  : "");
	printk("[OS]fcu:%s", FLAG_CHK(FSF_ACKREQ)   ? "FSF_ACKREQ\n"   : "");
	printk("[OS]fcu:%s", FLAG_CHK(FSF_INITIAL)  ? "FSF_INITIAL\n"  : "");
	printk("[OS]fcu:%s", FLAG_CHK(FSF_FCURDY)   ? "FSF_FCURDY\n"   : "");
	printk("[OS]fcu:%s", FLAG_CHK(FSF_RDYINTR)  ? "FSF_RDYINTR\n"  : "");
	printk("[OS]fcu:%s", FLAG_CHK(FSF_REBOOT)   ? "FSF_REBOOT\n"   : "");

	printk("[OS]fcu:[plist]\n");
	printk("[OS]fcu:plist r cnt : %d\n", plcnt(&sc->sc_rpl));
	printk("[OS]fcu:plist w emp : %d\n", plemp(&sc->sc_wpl));

	ic.intr = bsr4_io(sc, FBI_OFS_IO_INTRCAUSE);
	ic.cmd  = bsr1_fcubase(sc, FBI_OFS_DP_INTR_WRITE);
	ic.ack  = bsr1_fcubase(sc, FBI_OFS_DP_INTR_ACK);
	ic.rdy  = bsr1_fcubase(sc, FBI_OFS_DP_INTR_READY);
	ic.dma3 = bsr1_fcubase(sc, FBI_OFS_DP_INTR_DMA3);
	ic.dma4 = bsr1_fcubase(sc, FBI_OFS_DP_INTR_DMA4);

	printk("[OS]fcu:[intr cause & DPRAM]\n");
	printk("[OS]fcu:ic.intr=%x\n", (int)ic.intr);
	printk("[OS]fcu:    cmd=%x\n", (int)ic.cmd);
	printk("[OS]fcu:    ack=%x\n", (int)ic.ack);
	printk("[OS]fcu:    rdy=%x\n", (int)ic.rdy);
	printk("[OS]fcu:   dma3=%x\n", (int)ic.dma3);
	printk("[OS]fcu:   dma4=%x\n", (int)ic.dma4);

	printk("[OS]fcu:[bsr4_io(sc,FBI_OFS_IO_DMACTL)]\n");
	printk("[OS]fcu:%08x \n",bsr4_io(sc,FBI_OFS_IO_DMACTL));
	
	printk("[OS]fcu:sys_str_flag=%d\n",sys_str_flag);
	printk("[OS]fcu:reboot_flag=%d\n",reboot_flag);
	
	

}


static void fcu_debug_ioctl_func(fcu_t *sc, int arg)
{
	switch (arg) {
	default:
	case 0:
		fcu_debug_0(sc);
		break;
	}
}

#endif	/* !FCU_DEBUG_IOCTL */

