/*
 * If distributed as part of the Linux kernel, this code is licensed under the
 * terms of the GPL v2.
 *
 * Otherwise, the following license terms apply:
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1) Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2) Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3) The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESSED OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
/*
 * common data structure and control functions for fcu and fcud
 */
#include <linux/kernel.h>
#include <linux/printk.h>
#include <linux/fs.h>		/* file or file_operation */
#include <linux/slab.h>		/* for kmalloc */
#include <linux/cdev.h>
#include <linux/wait.h>
#include <linux/device.h>	/* for class etc... */
#include <linux/spinlock.h>	/* for class etc... */
#include <linux/sysctl.h>	/* for sysctl etc... */
#include <linux/module.h>
#include "fcuvar.h"
#include "fax_share.h"

#ifdef	DRIVER_DEBUG
static int __init init_sysctl(void);
static void __exit cleanup_sysctl(void);
#endif
 
static fax_share_param_t share_param;
static bool initdone = false;

static int __init  fax_init(void);
static void fax_exit(void);

extern int fcu_attach(void *);
extern int fcu_detach(void *);
extern int fcud_attach(void *);
extern int fcud_detach(void *);

extern int fcu_pci_runtime_suspend(struct device*);
extern int fcu_pci_runtime_resume(struct device*);

/* inspired from NetBSD device cd structure */
extern struct file_operations fcu_FileOps;
extern struct file_operations fcud_FileOps;
extern struct fcu_softc g_fcu_softc;
extern struct fcud_softc g_fcud_softc;
struct devlist fax_devlist[2] = {
	/* minor = 0 : fcu */
	{
		.softc = (void*)(&g_fcu_softc),
		.ops = &fcu_FileOps,
		.name = "fcu",
		.attachfunc = fcu_attach,
		.detachfunc = fcu_detach,
	},
	/* minor = 1 : fcud */
	{
		.softc = (void*)(&g_fcud_softc),
		.ops = &fcud_FileOps,
		.name = "fcud",
		.attachfunc = fcud_attach,
		.detachfunc = fcud_detach,
	},
};
EXPORT_SYMBOL(fax_devlist);

/**
 * pci device table of fcu
 */
static struct pci_device_id fcu_pci_tbl[] = {
/*	    { PCI_DEVICE(PCI_VENDOR_RICOH, PCI_PRODUCT_RICOH_FBI2_FACE3) }, */
	    { PCI_DEVICE(PCI_VENDOR_RICOH, PCI_PRODUCT_RICOH_FACE6) },
	    { }
};

/**
 * pci powerhook table
 */
static const struct dev_pm_ops fcu_pci_pm_ops = {
	.suspend_noirq = fcu_pci_runtime_suspend,
	.resume_noirq = fcu_pci_runtime_resume,
	.runtime_idle = NULL,
};

static void init_fax_share_param(void)
{
	if( initdone == true ){
		return;
	}
	
	memset(&share_param, 0, sizeof(share_param));

	initdone = true;
}

fax_share_param_t * get_fax_share_param(void)
{
	if(initdone == false){
		init_fax_share_param();
	}
	return &share_param;
}
EXPORT_SYMBOL(get_fax_share_param);

/*
 * probe
 */
static int
fcuprobe(struct pci_dev *pdev, const struct pci_device_id *ent)
{
	int error = -1, major, minor;
	fax_driver_header_t *header;
	dev_t device;
	struct class *fax_class;
	struct device *dev;
	
	PRINT_INFO("[OS]fcu:%s: fcuprobe_start\n", DEV_NAME);
	
	/* memory initial */
	init_fax_share_param();

	/* save pci_dev */
	share_param.pci_dev = pdev;
	
	/* allocate device class */
	fax_class = class_create(THIS_MODULE, DEV_NAME);
	if(IS_ERR(fax_class)){
		PRINT_ERR("[OS]fcu:%s: can't create class\n", DEV_NAME);
		error = PTR_ERR(fax_class);
		goto class_create_error;	
	}
	
	/* allocate device file,2 means fcu and fcud */
	error = alloc_chrdev_region(&device, 0, 2, DEV_NAME);
	if (error) {
		PRINT_ERR( "[OS]fcu:%s: can't alloc major%d, error=%d\n", DEV_NAME, MAJOR(device), error);
		goto chrdev_alloc_error;
	}
	/* driver specific initialization */
	for( minor = 0; minor < ARRAY_SIZE(fax_devlist); minor++ ){
		if (fax_devlist[minor].attachfunc(fax_devlist[minor].softc)!= 0){
			PRINT_ERR( "[OS]fcu:%s: attacherror minor=%d, error=%d\n", DEV_NAME, minor, error);
			goto attach_error;
		}
	}
	/* register minor device interface to char driver framework*/
	major = MAJOR(device);
	
	for( minor = 0; minor < ARRAY_SIZE(fax_devlist); minor++ ){
		/* header must be first member of each softc */
		header = fax_devlist[minor].softc;
		/* set major level device */
		header->sc_dev = MKDEV(major, minor);
		/* set class structure */
		header->sc_class = fax_class;
		header->sc_name = fax_devlist[minor].name;
		/* add character device to kernel */
		cdev_init(&header->sc_cdev, fax_devlist[minor].ops);
		header->sc_cdev.owner = THIS_MODULE;
		error = cdev_add(&header->sc_cdev, header->sc_dev, 1);
		if( error ){
			PRINT_ERR("[OS]fcu:%s: can't register cdev(%d)d, error=%d\n", DEV_NAME, MINOR(header->sc_dev), error);
			goto cdev_add_error;
		}
		dev = device_create(fax_class, NULL, header->sc_dev, NULL, fax_devlist[minor].name);
		if (IS_ERR(dev)){
			PRINT_ERR("[OS]fcu:%s: can't create cdev(%d)d\n", DEV_NAME, MINOR(header->sc_dev));
			goto device_create_error;
		}
	}
	PRINT_INFO("[OS]fcu:%s: fcuprobe_end\n", DEV_NAME);

#if 0 // for MT
	header = fax_devlist[0].softc;
	PRINT_DEBUG("[OS]fcu:fcu header sc_name = %s\n",header->sc_name );
	header = fax_devlist[1].softc;
	PRINT_DEBUG("[OS]fcu:fcud header sc_name = %s\n", header->sc_name );
	PRINT_DEBUG("[OS]fcu:share irqno = %d\n", share_param.irqno);
	PRINT_DEBUG("[OS]fcu:share io_addr = %lx\n", share_param.io_addr);
	PRINT_DEBUG("[OS]fcu:share io_len = %lx\n", share_param.io_len);
	PRINT_DEBUG("[OS]fcu:share mem_vaddr = %lx\n", share_param.mem_vaddr);
	PRINT_DEBUG("[OS]fcu:share mem_len = %lx\n", share_param.mem_len);
	PRINT_DEBUG("[OS]fcu:share io_addr = %lx\n", share_param.io_addr);
	PRINT_DEBUG("[OS]fcu:share mam_addr = %lx\n", share_param.mem_addr);
	PRINT_DEBUG("[OS]fcu:share pci_dev = %p\n", share_param.pci_dev);
#endif	
	
	return 0;

device_create_error:
cdev_add_error:
	for( minor-- ; minor > 0; minor-- ){
		header = fax_devlist[minor].softc;
		device_destroy(header->sc_class, header->sc_dev);
		cdev_del(&header->sc_cdev);
	}
attach_error:
	unregister_chrdev_region(device, ARRAY_SIZE(fax_devlist));
chrdev_alloc_error:
	class_destroy(fax_class);
class_create_error:
	return error;
}

static void
fcuremove(struct pci_dev *dev){

	// shouldn't happen? 
	PRINT_INFO("[OS]fcu:%s: fcu remove\n", DEV_NAME);

	fax_exit();
}

/**
 * pci information for kernel registration
 */
static struct pci_driver fcu_driver = {
        .name           = DEV_NAME,
        .probe          = fcuprobe,
        .remove         = fcuremove,
        .id_table       = fcu_pci_tbl,
	.driver		= {
	   .pm = &fcu_pci_pm_ops,
	},
};

	
static int __init  fax_init(void)
{
	PRINT_INFO("[OS]fcu:%s: fax_init\n", DEV_NAME);
	 
	// regist pci driver 
	return pci_register_driver(&fcu_driver);
}
EXPORT_SYMBOL(fax_init);

static void fax_exit(void)
{
	int minor;
	fax_driver_header_t *header;

	PRINT_INFO("[OS]fcu:%s: fcuExit\n", DEV_NAME);

	for( minor = 0; minor < ARRAY_SIZE(fax_devlist); minor++ ){
		header = fax_devlist[minor].softc;
		device_destroy(header->sc_class, header->sc_dev);
		cdev_del(&header->sc_cdev);
	}
	unregister_chrdev_region(header->sc_dev, ARRAY_SIZE(fax_devlist));
	class_destroy(header->sc_class);
	
	for( minor = 0; minor < ARRAY_SIZE(fax_devlist); minor++ ){
		fax_devlist[minor].detachfunc(fax_devlist[minor].softc);
	}
}
EXPORT_SYMBOL(fax_exit);

module_init(fax_init);
module_exit(fax_exit);

MODULE_AUTHOR("Tadashi Sato <tadashi.s1.satoh@jp.ricoh.com>");
MODULE_DESCRIPTION("fcud - Driver for FAX board");
MODULE_LICENSE("GPL v2");


