/*
 * Main API entry point
 *
 * Copyright (c) 2000-2003 Johannes Erdfelt <johannes@erdfelt.com>
 *
 * This library is covered by the LGPL, read LICENSE for details.
 */

#include <stdlib.h>	/* getenv */
#include <stdio.h>	/* stderr */
#include <errno.h>

#include <dev/usb/usb.h>

#include "usbi.h"

#include <string.h>
#include <pthread.h>
#include <unistd.h> /* close */
#include <fcntl.h>  /* open */
#include <sys/ioctl.h> /* ioctl */

int usb_debug = 2;	/* Set the debugging message level from 0 to 3. */
struct usb_bus usb_busses;

#define	USB_BUS_NAME	"/dev/usb0"

int usb_find_busses(void)
{
  /* ICCCTL-18:2foCXT|[gΉ */
  /* nothing */
  return 0;
}

int usb_find_devices(void)
{
  /* ICCCTL-18:2foCXT|[gΉ */
  /* nothing */
  return 0;
}

void usb_set_debug(int level)
{
  if (usb_debug || level)
    fprintf(stderr, "usb_set_debug: Setting debugging level to %d (%s)\n",
	level, level ? "on" : "off");

  usb_debug = level;
}

void usb_init(void)
{
  /* ICCCTL-18:2foCXT|[gΉ */
  if (!strcmp(usb_busses.dirname,USB_BUS_NAME)) {
    /* already initialized */
    return;
  }

  if (getenv("USB_DEBUG"))
    usb_set_debug(atoi(getenv("USB_DEBUG")));

  /* usb_bus\̏̏ */
  memset(&usb_busses, 0, sizeof(usb_busses));
  strcpy(usb_busses.dirname, USB_BUS_NAME);

  usb_busses.dirname[sizeof(usb_busses.dirname) - 1] = '\0';
  usb_busses.prev = NULL;
  usb_busses.next = NULL;
}

usb_dev_handle *usb_open(struct usb_device *dev)
{
  usb_dev_handle *udev;

  udev = malloc(sizeof(*udev));
  if (!udev)
    return NULL;

  udev->fd = -1;
  udev->device = dev;
  udev->bus = dev->bus;
  udev->config = udev->interface = udev->altsetting = -1;

  if (usb_os_open(udev) < 0) {
    free(udev);
    return NULL;
  }

  return udev;
}

int usb_get_string(usb_dev_handle *dev, int index, int langid, char *buf,
	size_t buflen)
{
  /*
   * We can't use usb_get_descriptor() because it's lacking the index
   * parameter. This will be fixed in libusb 1.0
   */
  return usb_control_msg(dev, USB_ENDPOINT_IN, USB_REQ_GET_DESCRIPTOR,
			(USB_DT_STRING << 8) + index, langid, buf, buflen, 1000);
}

int usb_get_string_simple(usb_dev_handle *dev, int index, char *buf, size_t buflen)
{
  char tbuf[256];
  int ret, langid, si, di;

  /*
   * Asking for the zero'th index is special - it returns a string
   * descriptor that contains all the language IDs supported by the
   * device. Typically there aren't many - often only one. The
   * language IDs are 16 bit numbers, and they start at the third byte
   * in the descriptor. See USB 2.0 specification, section 9.6.7, for
   * more information on this. */
  ret = usb_get_string(dev, 0, 0, tbuf, sizeof(tbuf));
  if (ret < 0)
    return ret;

  if (ret < 4)
    return -EIO;

  langid = tbuf[2] | (tbuf[3] << 8);

  ret = usb_get_string(dev, index, langid, tbuf, sizeof(tbuf));
  if (ret < 0)
    return ret;

  if (tbuf[1] != USB_DT_STRING)
    return -EIO;

  if (tbuf[0] > ret)
    return -EFBIG;

  for (di = 0, si = 2; si < tbuf[0]; si += 2) {
    if (di >= (buflen - 1))
      break;

    if (tbuf[si + 1])	/* high byte */
      buf[di++] = '?';
    else
      buf[di++] = tbuf[si];
  }

  buf[di] = 0;

  return di;
}

int usb_close(usb_dev_handle *dev)
{
  int ret;

  ret = usb_os_close(dev);
  free(dev);

  return ret;
}

struct usb_device *usb_device(usb_dev_handle *dev)
{
  return dev->device;
}

void usb_free_dev(struct usb_device *dev)
{
  usb_destroy_configuration(dev);
  free(dev);
}

struct usb_bus *usb_get_busses(void)
{
  return &usb_busses;
}

void usb_free_bus(struct usb_bus *bus)
{
  free(bus);
}

/* ICCCTL-18:2foCXT|[gΉ */
/* foCXfoCXXg֒ǉ */
int usb_add_device(char *device_name)
{
  int dfd;
  char add_device_name[20];
  struct usb_device *dev;
  usb_dev_handle *dev_handle;

  if (!device_name) {
    if (usb_debug >= 1) {
      fprintf(stderr, "usb_add_device: add_driver_name is null\n");
    }
    return -1;
  }

  /* libusbhʒmꂽdevice_nameɁAfoCXfBXNv^擾 */
  memset(add_device_name, 0x00, sizeof(add_device_name));

#if __FreeBSD__
  snprintf(add_device_name, sizeof(add_device_name) - 1, "/dev/%s", device_name);
#else
  snprintf(add_device_name, sizeof(add_device_name) - 1, "/dev/%s.00", device_name);
#endif

  /* Open its control endpoint */
  dfd = open(add_device_name, O_RDONLY);
  if (dfd < 0) {
    if (usb_debug >= 2) {
	  fprintf(stderr, "usb_add_device: couldn't open device %s: %s\n",
	  		add_device_name, strerror(errno));
    }
    return -1;
  }

  dev = malloc(sizeof(*dev));
  if (!dev) {
    if (usb_debug >= 1) {
      fprintf(stderr, "usb_add_device: Not enough memory for dev \n");
    }
	close(dfd);
    USB_ERROR_R(-ENOMEM);
  }

  memset((void *)dev, 0, sizeof(*dev));
  dev->bus = &usb_busses;

  /* we need to report the device name as /dev/ugenx NOT /dev/ugenx.00
   * This seemed easier than having 2 variables...
   */
#if (__NetBSD__ || __OpenBSD__)
  snprintf(add_device_name, sizeof(add_device_name) - 1, "/dev/%s", device_name);
#endif

  strncpy(dev->filename, add_device_name, sizeof(dev->filename) - 1);
  dev->filename[sizeof(dev->filename) - 1] = '\0';

  if (ioctl(dfd, USB_GET_DEVICE_DESC, &dev->descriptor) < 0) {
    close(dfd);
    free(dev);
    USB_ERROR_STR(-errno, "couldn't get device descriptor for %s: %s",
				add_device_name, strerror(errno));
  }

  close(dfd);

  USB_LE16_TO_CPU(dev->descriptor.bcdUSB);
  USB_LE16_TO_CPU(dev->descriptor.idVendor);
  USB_LE16_TO_CPU(dev->descriptor.idProduct);
  USB_LE16_TO_CPU(dev->descriptor.bcdDevice);

  /* Some ports fetch the descriptors on scanning (like Linux) so we */
  /*  don't need to fetch them again */
  dev_handle = usb_open(dev);
  if (!dev_handle) {
    if (usb_debug >= 1) {
      fprintf(stderr, "usb_add_device: usb_open failed \n");
    }
	free(dev);
    return -1;
  }
  usb_fetch_and_parse_descriptors(dev_handle);
  usb_close(dev_handle);

  /* usb_device\̂̃Xg֐VKɒǉ */
  LIST_ADD(usb_busses.devices, dev);

  return 0;
}

/* ICCCTL-18:2foCXT|[gΉ */
/* foCXhCofoCXXg폜*/
int usb_remove_device(char *device_name)
{
  char remove_device_name[20];
  struct usb_device *dev = NULL;

  if (!device_name) {
    if (usb_debug >= 1) {
		fprintf(stderr, "usb_remove_device: remove_driver_name is null\n");
    }
    return -1;
  }

  memset(remove_device_name, 0x00, sizeof(remove_device_name));
#if (__NetBSD__ || __OpenBSD__)
  snprintf(remove_device_name, sizeof(remove_device_name) - 1, "/dev/%s", device_name);
#endif

  /* usb_device\̂̃Xg폜 */
  for (dev = usb_busses.devices; dev; dev = dev->next) {
    if (!strcmp(dev->filename, remove_device_name)) {
      LIST_DEL(usb_busses.devices, dev);
	  usb_free_dev(dev);
      break;
    }
  }

  return 0;
}

/* ICCCTL-18:2foCXT|[gΉ */
void usb_init_device(void)
{
  /* nothing */
}

