/*
 * Copyright (c) 2010 RICOH Co. Ltd. All rights reserved.
 */

#ifdef USE_WITH_PTL
#include <pthread.h>

pthread_mutex_t	mfprintf_mutex = PTHREAD_MUTEX_INITIALIZER;
pthread_cond_t	mfprintf_cond = PTHREAD_COND_INITIALIZER;
#endif

#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <sys/uio.h>
#include <errno.h>
#include <fcntl.h>
#include <stdarg.h>
#include <unistd.h>
#include <string.h>

#include "mfprintf.h"
#include "mfpr_shm.h"
#include "mfpr_if.h"
#include "mfpr_logboard.h"


static mfpr_t	_mfcreate_shm __P((key_t, int));
static int	_mfdelete_shm __P((mfpr_t, int));
static void	_mfkdoprnt_shm __P((void (*)__P((int,mfpr_t)),
				     mfpr_t, const char *, va_list));
static void	_mfputs_shm __P((const char *, mfpr_t));
static void	_mfsetflags_shm __P((mfpr_t, int, int));

static mfpr_t	_mfcreate_logb __P((key_t, int));
static int	_mfdelete_logb __P((mfpr_t, int));
static void	_mfkdoprnt_logb __P((void (*)__P((int,mfpr_t)),
				     mfpr_t, const char *, va_list));
static void	_mfputs_logb __P((const char *, mfpr_t));
static void	_mfsetflags_logb __P((mfpr_t, int, int));

static void	_mfsputchar __P((int, mfpr_t));
static void	_mfsputc __P((int, mfpr_t));
static void	_mfstamp __P((mfpr_t));

static void	_mfhexdump __P((void (*)__P((const char *, mfpr_t)),
				mfpr_t, u_char *, int));
static char *	_mfsprtn __P((char *, u_long, int, int));

static const struct _mf_funcptr *	_mfsettarget __P((void));





static int _mf_devfd = -1;

static const struct _mf_funcptr _mf_funcptr_shm = {	
	_mfcreate_shm,
	_mfdelete_shm,
	_mfkdoprnt_shm,
	_mfputs_shm,
	_mfsetflags_shm
};
static const struct _mf_funcptr _mf_funcptr_logb = {	
	_mfcreate_logb,
	_mfdelete_logb,
	_mfkdoprnt_logb,
	_mfputs_logb,
	_mfsetflags_logb
};



int
_mfextension(command, argc, argv, retval)
	int	command;
	int	argc;
	void	*argv[];
	int	*retval;
{
	static struct _mf_funcptr	*funcptr = NULL;

	int	error = 0;

#ifdef USE_WITH_PTL
	pthread_mutex_lock(&mfprintf_mutex);
#endif

	switch (command) {
	case EXT_STATUS: {
		mfpr_t	mf = (mfpr_t)argv[0];
		struct mf_status *ms = (struct mf_status *)argv[1];

		if (mf->mf_magic1 != MF_MAGIC1 || mf->mf_magic2 != MF_MAGIC2) {
			error++;
			*retval = EXT_ENOEXEC;		
			break;
		}
		ms->ms_version = mf->mf_version;
		ms->ms_time = mf->mf_time;
		ms->ms_pid = mf->mf_pid;
		ms->ms_key = mf->mf_key;
		ms->ms_head = (char *)mf + sizeof (*mf);
		ms->ms_tail = (char *)mf + mf->mf_size;
		ms->ms_wpos = (char *)mf + mf->mf_wpos;
		break;
	}
	case EXT_MFCREATE: {
		key_t	key = (key_t)argv[0];
		int	size = (int)argv[1];

		if (funcptr == NULL) {
			funcptr = (struct _mf_funcptr *)_mfsettarget();
		}

		*retval = (int)funcptr->f_create(key, size);
		break;
	}
	case EXT_MFDELETE: {
		mfpr_t	mf = (mfpr_t)argv[0];
		int	del = (int)argv[1];

		*retval = funcptr->f_delete(mf, del);
		break;
	}
	case EXT_VMFPRINTF: {
		mfpr_t	mf = (mfpr_t)argv[0];
		char	*fmt = (char *)argv[1];
//		va_list	ap = (va_list)argv[2];
		char*   ap = (char*)argv[2];
       
		if (mf->mf_flags & MF_DISABLE)
			break;
		funcptr->f_kdoprnt(_mfsputchar, mf, fmt, ap);
		break;
	}
	case EXT_MFHEXDUMP: {
		mfpr_t	mf = (mfpr_t)argv[0];
		u_char	*addr = (u_char *)argv[1];
		int	len = (int)argv[2];

		if (mf->mf_flags & MF_DISABLE)
			break;
		_mfhexdump(funcptr->f_puts, mf, addr, len);
		break;
	}
	case EXT_MFPUTS: {
		mfpr_t	mf = (mfpr_t)argv[0];
		char	*s = (char *)argv[1];

		if (mf->mf_flags & MF_DISABLE)
			break;
		funcptr->f_puts(s, mf);
		break;
	}
	case EXT_MFFLAGS: {
		mfpr_t	mf = (mfpr_t)argv[0];
		int	flag = (int)argv[1];
		int	val = (int)argv[2];

		funcptr->f_setflags(mf, flag, val);
		break;
	}
	default:
		error++;
		*retval = EXT_ENOSYS;
		break;
	}

#ifdef USE_WITH_PTL
	pthread_cond_signal(&mfprintf_cond);
	pthread_mutex_unlock(&mfprintf_mutex);
#endif

	return ((error)? -1 : 0);
}




static const struct _mf_funcptr *
_mfsettarget()
{

	_mf_devfd = open(MF_LOGBDEV, O_WRONLY);
	if ((_mf_devfd != -1) ||
	    ( (_mf_devfd == -1) && ((errno == EACCES) || (errno == EPERM)) )) {
		return (&_mf_funcptr_logb);
	} else {
		return (&_mf_funcptr_shm);
	}
}







static mfpr_t
_mfcreate_shm(key, size)
	key_t	key;
	int	size;
{
	int	shm;
	mfpr_t	mf;
	struct timeval	tv;

	if (size > 4096)
		size &= ~(4096 - 1);

	shm = shmget(key, size, size? (IPC_CREAT | 0660) : 0);
	if (shm == -1)
		return (NULL);
	mf = (mfpr_t)shmat(shm, 0, 0);
	if (mf == (mfpr_t)(-1)) {
		if (size)
			(void) shmctl(shm, IPC_RMID, 0);
		return (NULL);
	}

	if (!size)
		return (mf);

	gettimeofday(&tv, NULL);

	mf->mf_magic1 = MF_MAGIC1;
	mf->mf_magic2 = MF_MAGIC2;
	mf->mf_version = MF_VERSION;
	mf->mf_flags = 0;
	mf->mf_time = tv.tv_sec;
	mf->mf_key = key;
	mf->mf_size = size;
	mf->mf_pid = getpid();
	mf->mf_wpos = sizeof (*mf);

	return (mf);
}


static int
_mfdelete_shm(mf, del)
	mfpr_t	mf;
	int	del;
{
	int	shm;

	if ((shm = shmget(mf->mf_key, mf->mf_size, 0)) == -1)
		return (-1);
	if (shmdt((void *)mf) == -1)
		return (-1);
	if (del && shmctl(shm, IPC_RMID, 0) == -1)
		return (-1);
	return (0);
}


static void
_mfkdoprnt_shm(put, mf, fmt, ap)
	void (*put)__P((int, mfpr_t));
	mfpr_t	mf;
	const char *fmt;
	va_list ap;
{
	register char	*p;
	register int	ch;
	unsigned long	ul;
	int		lflag;
	int		base;
	int		width;
	char		padc;
	char		buf[(sizeof(long) * NBBY / 3) + 1];

	for (;;) {
		while ((ch = *fmt++) != '%') {
			if (ch == '\0')
				return;
			put(ch, mf);
		}
		lflag = 0;
		width = 0;
		padc = ' ';

reswitch:	switch (ch = *fmt++) {
		case '\0':
			return;

		case '0':
			padc = '0';
			goto	reswitch;

		case '1': case '2': case '3':
		case '4': case '5': case '6':
		case '7': case '8': case '9':
			for (width = 0; ; ++fmt) {
				width = width * 10 + (ch - '0');
				ch = *fmt;
				if (ch < '0' || ch > '9')
					break;
			}
			goto	reswitch;

		case 'l':
			lflag = 1;
			goto reswitch;
		case 'c':
			ch = va_arg(ap, int);
			put(ch & 0x7f, mf);
			break;
		case 's':
			p = va_arg(ap, char *);
			while ((ch = *p++))
				put(ch, mf);
			break;

		case 'd':
			ul = lflag ? va_arg(ap, long) : va_arg(ap, int);
			base = 10;
			if ((long)ul < 0) {
				put('-', mf);
				ul = -(long)ul;
			}
			goto	donumber;

		case 'o':
			ul = lflag ? va_arg(ap, u_long) : va_arg(ap, u_int);
			base = 8;
			goto	donumber;

		case 'u':
			ul = lflag ? va_arg(ap, u_long) : va_arg(ap, u_int);
			base = 10;
			goto	donumber;

		case 'x':
			ul = lflag ? va_arg(ap, u_long) : va_arg(ap, u_int);
			base = 16;
donumber:
			p = buf;
			do {
				*p++ = "0123456789abcdef"[ul % base];
				width--;
			} while (ul /= base);
			while (width-- > 0)
				put(padc, mf);
			do {
				put(*--p, mf);
			} while (p > buf);
			break;

		default:
			put('%', mf);
			if (lflag)
				put('l', mf);
			put(ch, mf);
		}
	}
	va_end(ap);
}


static void
_mfputs_shm(s, mf)
	const char *s;
	mfpr_t	mf;
{
	while (*s) {
		_mfsputchar(*s, mf);
		s++;
	}
}


static void
_mfsetflags_shm(mf, flag, val)
	mfpr_t mf;
	int flag;
	int val;
{
	switch (val) {
	case 0:
		mf->mf_flags &= ~flag;
		break;
	case 1:
		mf->mf_flags |= flag;
		break;
	default:
		break;
	}

	return;
}



static void
_mfsputchar(c, mf)
	int	c;
	mfpr_t	mf;
{
#define	DOSTAMP	(MF_TIMESTAMP | MF_1STCOLUMN)

	if ((mf->mf_flags & DOSTAMP) == DOSTAMP) {
		_mfstamp(mf);
		mf->mf_flags &= ~MF_1STCOLUMN;
	}
	_mfsputc(c, mf);
	if (c == '\n')
		mf->mf_flags |= MF_1STCOLUMN;
}


static void
_mfsputc(c, mf)
	int	c;
	mfpr_t	mf;
{
	*((u_char *)mf + mf->mf_wpos) = (u_char)c;
	if (++mf->mf_wpos >= mf->mf_size) {
		if (mf->mf_flags & MF_ONESHOT)
			mf->mf_wpos = mf->mf_size - 1;		
		else
			mf->mf_wpos = sizeof (*mf);
	}
}


static void
_mfstamp(mf)
	mfpr_t	mf;
{
	struct timeval	tv;
	char	buf[10+1+6+1+1];	
	char	*bp;

	gettimeofday(&tv, NULL);
	bp = _mfsprtn(buf, (u_long)tv.tv_sec, 10, 10);
	*bp++ = '.';
	bp = _mfsprtn(bp, (u_long)tv.tv_usec, 10, 6);
	*bp++ = ':';
	*bp = '\0';
	for (bp = buf; *bp; bp++)
		_mfsputc(*bp, mf);
}






static mfpr_t
_mfcreate_logb(key, size)
	key_t	key;
	int	size;
{
	mfpr_t mf;
	struct logb_cmd_create ioarg;  
	int ret;

	mf = _mfcreate_shm(key,size);
	if (mf == NULL) {
		return (mf);
	}
	if (size) {
		_mfputs_shm(MF_SHMMSG, mf);
	}

	if (_mf_devfd == -1) {
		return (mf);
	}

	ioarg.id = key;
	ioarg.size = size;
	ret = ioctl(_mf_devfd, LOGB_CREATE, &ioarg);
	if (ret) {
		;
	}

	return (mf);
}



static int
_mfdelete_logb(mf, del)
	mfpr_t	mf;
	int	del;
{
	return (_mfdelete_shm(mf, del));
}

static void
_mfkdoprnt_logb(put, mf, fmt, ap)
	void (*put)__P((int, mfpr_t));
	mfpr_t	mf;
	const char *fmt;
	va_list ap;
{


#define _MF_DATAFLUSH()                                           \
	     \
	do {                                                      \
		write(_mf_devfd, &data_buf,                       \
		      sizeof(data_buf.id) + bufsize);             \
		buf_p = data_buf.data;                            \
		bufsize = 0;                                      \
	} while (0)

#define _MF_DATAWRITE(c)                                          \
                                                       \
	do {                                                      \
		*buf_p++ = (c);                                   \
		bufsize++;                                        \
		if (buf_p == guard) {                             \
			_MF_DATAFLUSH();                          \
		}                                                 \
	} while (0)


	struct logb_logdata data_buf;
	char *guard = (char *)&data_buf + sizeof(data_buf);  
	char		*buf_p = data_buf.data;
	int		bufsize = 0;
	register char	*p;
	register int	ch;
	unsigned long	ul;
	int		lflag;
	int		base;
	int		width;
	char		padc;
	char		buf[(sizeof(long) * NBBY / 3) + 1];

	if (_mf_devfd == -1) {
		return;
	}
	data_buf.id = mf->mf_key;


	for (;;) {
		while ((ch = *fmt++) != '%') {
			if (ch == '\0') {
				_MF_DATAFLUSH();
				return;
			}
			_MF_DATAWRITE((char)ch);
		}
		lflag = 0;
		width = 0;
		padc = ' ';

reswitch:	switch (ch = *fmt++) {
		case '\0':
			_MF_DATAFLUSH();
			return;	

		case '0':
			padc = '0';
			goto	reswitch;

		case '1': case '2': case '3':
		case '4': case '5': case '6':
		case '7': case '8': case '9':
			for (width = 0; ; ++fmt) {
				width = width * 10 + (ch - '0');
				ch = *fmt;
				if (ch < '0' || ch > '9')
					break;
			}
			goto	reswitch;

		case 'l':
			lflag = 1;
			goto reswitch;
		case 'c':
			ch = va_arg(ap, int);
			_MF_DATAWRITE((char)ch & 0x7f);
			break;
		case 's':
			p = va_arg(ap, char *);
			_MF_DATAFLUSH();
			_mfputs_logb(p, mf);
			break;

		case 'd':
			ul = lflag ? va_arg(ap, long) : va_arg(ap, int);
			base = 10;
			if ((long)ul < 0) {
				_MF_DATAWRITE('-');
				ul = -(long)ul;
			}
			goto	donumber;

		case 'o':
			ul = lflag ? va_arg(ap, u_long) : va_arg(ap, u_int);
			base = 8;
			goto	donumber;

		case 'u':
			ul = lflag ? va_arg(ap, u_long) : va_arg(ap, u_int);
			base = 10;
			goto	donumber;

		case 'x':
			ul = lflag ? va_arg(ap, u_long) : va_arg(ap, u_int);
			base = 16;
donumber:
			p = buf;
			do {
				*p++ = "0123456789abcdef"[ul % base];
				width--;
			} while (ul /= base);
			while (width-- > 0)
				_MF_DATAWRITE(padc);
			do {
				_MF_DATAWRITE(*--p);
			} while (p > buf);
			break;

		default:
			_MF_DATAWRITE('%');
			if (lflag)
				_MF_DATAWRITE('l');
			_MF_DATAWRITE(ch);
		}
	}
	va_end(ap);
}


static void
_mfputs_logb(s, mf)
	const char *s;
	mfpr_t	mf;
{
	struct logb_logdata data_buf;
	const char *ptr = s;
	int rest_len = strlen(s);
	int wr_len, ret;

	if (_mf_devfd == -1) {
		return;
	}

	data_buf.id = mf->mf_key;

	while (rest_len > 0) {
		wr_len = (rest_len > LOGB_BUFSIZE) ? LOGB_BUFSIZE : rest_len;
		strncpy(data_buf.data, s, wr_len);
		ret = write(_mf_devfd, &data_buf, sizeof(data_buf.id) + wr_len);
		s += wr_len;
		rest_len -= wr_len;
	}

	return;
}



static void
_mfsetflags_logb(mf, flag, val)
	mfpr_t mf;
	int flag;
	int val;
{
	struct logb_cmd_ctl ioarg;  

	_mfsetflags_shm(mf, flag, val);

	if (_mf_devfd == -1) {
		return;
	}

	ioarg.id = mf->mf_key;

	switch (flag) {
	case MFPRINTF_STOP:
		ioarg.cmd = LOGB_STOP;
		break;
	case MFPRINTF_TIMESTAMP:
		ioarg.cmd = LOGB_TIMESTAMP;
		break;
	case MFPRINTF_SINGLEBUFFER:
		ioarg.cmd = LOGB_TYPE;
		break;
	default:
		return;
	}

	ioarg.cmd_arg = val;

	ioctl(_mf_devfd, LOGB_LOGCTL, &ioarg);
	return;
}




static void
_mfhexdump(func, mf, addr, len)
	void	(*func)__P((const char *,mfpr_t));
	mfpr_t	mf;
	u_char	*addr;
	int	len;
{
	char	buf[80];
	char	*bp;
	int	done = 0;

	while (done < len) {
		bp = _mfsprtn(buf, (u_long)addr, 16, 8);
		*bp++ = ':';
		do {
			*bp++ = ' ';
			bp = _mfsprtn(bp, (u_long)(*addr), 16, 2);
			addr++;
			done++;
		} while ((done < len) && (done & 0xf));
		*bp++ = '\n';
		*bp = '\0';
		func(buf, mf);	
	}
}


static char *
_mfsprtn(bp, ul, base, zfill)
	char	*bp;
	u_long	ul;
	int	base;
	int	zfill;
{
	char	*p;
	char	buf[16];

	p = buf;
	do {
		*p++ = "0123456789abcdef"[ul % base];
	} while (ul /= base);
	while ((p - buf) < zfill)
		*p++ = '0';
	do {
		*bp++ = *--p;
	} while (p > buf);
	return (bp);
}


