
/*
 * USB3 stub device for Pegmatite SoC
 *
 * Provides VBUS status as a GPIO and enables USB2 interrupt.
 *
 * Copyright (C) 2015 Lexmark International Inc.
 *
 * This file is licensed under the terms of the GNU General Public
 * License version 2.  This program is licensed "as is" without any
 * warranty of any kind, whether express or implied.
 *
 */

#include <linux/module.h>
#include <linux/io.h>
#include <linux/gpio.h>
#include <linux/platform_device.h>
#include <linux/interrupt.h>
#include <linux/clk.h>
#include <linux/of_platform.h>

#define DRIVER_NAME "mv-pegmatite-u3d"

#define PEGMATITE_DEV_INFO_REG_OFFSET	0
#define DEV_SPEED_MASK	0x3
#define DEV_SPEED_SHIFT	16
#define DEV_USB2_ONLY	2

#define PEGMATITE_GBL_STS_OFFSET	0x2c
#define VBUS_PRESENT		BIT(2)

#define PEGMATITE_INT_STS_OFFSET	0xd8
#define VBUS_INT_STS		BIT(5)

#define PEGMATITE_INT_ENB_OFFSET	0xdc
#define USB2_INT_EN		BIT(0)
#define VBUS_INT_EN		BIT(5)

struct pegmatite_u3d {
	spinlock_t lock;
	int irq;
	void __iomem *iomem;
	struct clk *clk;
	struct gpio_chip gpio;
	struct irq_domain *domain;
};

static irqreturn_t pegmatite_u3d_irq(int irq, void *dev)
{
	struct pegmatite_u3d *u3d = dev;
	u32 val;

	val = readl(u3d->iomem + PEGMATITE_INT_STS_OFFSET);
	if (val & VBUS_INT_STS) {
		int vbus_irq = irq_find_mapping(u3d->domain, 0);

		if (vbus_irq > 0)
			generic_handle_irq(vbus_irq);
		return IRQ_HANDLED;
	}

	return IRQ_NONE;
}

static void pegmatite_u3d_init(struct pegmatite_u3d *u3d)
{
	u32 val;

	/* USB2 mode only */
	val = readl(u3d->iomem + PEGMATITE_DEV_INFO_REG_OFFSET);
	val &= ~(DEV_SPEED_MASK << DEV_SPEED_SHIFT);
	val |= (DEV_USB2_ONLY << DEV_SPEED_SHIFT);
	writel(val, u3d->iomem + PEGMATITE_DEV_INFO_REG_OFFSET);

	/*
	 * USB2 will be shared at the top level, so we'll enable but ignore USB2 IRQs.
	 * The VBUS IRQ will be handled as a GPIO
	 */
	writel(USB2_INT_EN, u3d->iomem + PEGMATITE_INT_ENB_OFFSET);
}

static int pegmatite_u3d_gpio_get(struct gpio_chip *chip, unsigned offset)
{
	struct pegmatite_u3d *u3d =
		container_of(chip, struct pegmatite_u3d, gpio);
	u32 val;

	BUG_ON(offset != 0);

	val = readl(u3d->iomem + PEGMATITE_GBL_STS_OFFSET);

	return !!(val & VBUS_PRESENT);
}

static int pegmatite_u3d_gpio_direction_input(struct gpio_chip *chip, unsigned offset)
{
	return 0;
}

static int pegmatite_u3d_gpio_to_irq(struct gpio_chip *chip, unsigned pin)
{
	struct pegmatite_u3d *u3d =
		container_of(chip, struct pegmatite_u3d, gpio);

	BUG_ON(pin != 0);
	return irq_create_mapping(u3d->domain, pin);
}

static void pegmatite_u3d_gpio_irq_ack(struct irq_data *d)
{
	struct pegmatite_u3d *u3d = irq_data_get_irq_chip_data(d);

	writel(VBUS_INT_STS, u3d->iomem + PEGMATITE_INT_STS_OFFSET);
}

static void pegmatite_u3d_gpio_irq_enable(struct irq_data *d)
{
	struct pegmatite_u3d *u3d = irq_data_get_irq_chip_data(d);
	unsigned long flags;
	u32 val;

	spin_lock_irqsave(&u3d->lock, flags);
	val = readl(u3d->iomem + PEGMATITE_INT_ENB_OFFSET);
	val |= VBUS_INT_EN;
	writel(val, u3d->iomem + PEGMATITE_INT_ENB_OFFSET);
	spin_unlock_irqrestore(&u3d->lock, flags);
}

static void pegmatite_u3d_gpio_irq_disable(struct irq_data *d)
{
	struct pegmatite_u3d *u3d = irq_data_get_irq_chip_data(d);
	unsigned long flags;
	u32 val;

	spin_lock_irqsave(&u3d->lock, flags);
	val = readl(u3d->iomem + PEGMATITE_INT_ENB_OFFSET);
	val &= ~VBUS_INT_EN;
	writel(val, u3d->iomem + PEGMATITE_INT_ENB_OFFSET);
	spin_unlock_irqrestore(&u3d->lock, flags);
}

static int pegmatite_u3d_gpio_irq_set_type(struct irq_data *d, unsigned int flow_type)
{
	if (flow_type & IRQ_TYPE_EDGE_BOTH)
		return IRQ_SET_MASK_OK;
	else
		return -EINVAL;
}

static struct irq_chip pegmatite_u3d_gpio_irq_chip = {
	.name		= DRIVER_NAME,
	.irq_ack	= pegmatite_u3d_gpio_irq_ack,
	.irq_enable	= pegmatite_u3d_gpio_irq_enable,
	.irq_disable	= pegmatite_u3d_gpio_irq_disable,
	.irq_set_type	= pegmatite_u3d_gpio_irq_set_type,
};

static int pegmatite_gpio_irq_domain_map(struct irq_domain *d, unsigned int irq,
					 irq_hw_number_t hwirq)
{
	struct pegmatite_u3d *u3d = d->host_data;

	irq_set_chip_and_handler(irq, &pegmatite_u3d_gpio_irq_chip, handle_edge_irq);
	irq_set_chip_data(irq, u3d);
	set_irq_flags(irq, IRQF_VALID);

	return 0;
}

static const struct irq_domain_ops pegmatite_gpio_irq_domain_ops = {
	.map = pegmatite_gpio_irq_domain_map,
};

static int pegmatite_u3d_probe(struct platform_device *pdev)
{
	struct pegmatite_u3d *u3d;
	struct resource *res;
	int ret;

	u3d = devm_kzalloc(&pdev->dev, sizeof(*u3d), GFP_KERNEL);
	if (!u3d)
		return -ENOMEM;
	platform_set_drvdata(pdev, u3d);
	spin_lock_init(&u3d->lock);

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	u3d->iomem = devm_ioremap_resource(&pdev->dev, res);
	if (!res) {
		dev_err(&pdev->dev, "failed to map I/O memory\n");
		return -ENODEV;
	}

	u3d->clk = devm_clk_get(&pdev->dev, NULL);
	if (IS_ERR(u3d->clk)) {
		dev_err(&pdev->dev, "failed to get clk %ld\n", PTR_ERR(u3d->clk));
		return PTR_ERR(u3d->clk);
	}

	ret = clk_prepare_enable(u3d->clk);
	if (ret) {
		dev_err(&pdev->dev, "failed to enable clk %d\n", ret);
		return ret;
	}

	pegmatite_u3d_init(u3d);

	u3d->domain = irq_domain_add_linear(pdev->dev.of_node, 1,
					    &pegmatite_gpio_irq_domain_ops,
					    u3d);
	if (!u3d->domain) {
		dev_err(&pdev->dev, "failed to add irq domain\n");
		ret = -ENODEV;
		goto err;
	}

	u3d->gpio.label = dev_name(&pdev->dev);
	u3d->gpio.dev = &pdev->dev;
	u3d->gpio.get = pegmatite_u3d_gpio_get;
	u3d->gpio.direction_input = pegmatite_u3d_gpio_direction_input;
	u3d->gpio.to_irq = pegmatite_u3d_gpio_to_irq;
	u3d->gpio.base = -1;
	u3d->gpio.ngpio = 1;
	u3d->gpio.of_node = pdev->dev.of_node;

	u3d->irq = platform_get_irq(pdev, 0);
	if (u3d->irq <= 0) {
		dev_err(&pdev->dev, "no IRQ\n");
		ret = -ENODEV;
		goto err;
	}

	ret = request_irq(u3d->irq, pegmatite_u3d_irq, IRQF_SHARED, DRIVER_NAME, u3d);
	if (ret) {
		dev_err(&pdev->dev, "Request IRQ %d failed\n", u3d->irq);
		u3d->irq = -1;
		goto err;
	}

	ret = gpiochip_add(&u3d->gpio);
	if (ret) {
		dev_err(&pdev->dev, "gpiochip_add failed: %d\n", ret);
		goto err;
	}

	return 0;
err:
	if (u3d->irq > 0)
		free_irq(u3d->irq, u3d);
	if (u3d->domain)
		irq_domain_remove(u3d->domain);
	clk_disable_unprepare(u3d->clk);
	return ret;
}

static int pegmatite_u3d_remove(struct platform_device *pdev)
{
	struct pegmatite_u3d *u3d = platform_get_drvdata(pdev);

	free_irq(u3d->irq, u3d);
	gpiochip_remove(&u3d->gpio);
	irq_domain_remove(u3d->domain);
	clk_disable_unprepare(u3d->clk);

	return 0;
}

static const struct of_device_id pegmatite_u3d_of_match[] = {
	{ .compatible = "marvell,pegmatite-u3d" },
	{},
};
MODULE_DEVICE_TABLE(of, pegmatite_u3d_of_match);

static struct platform_driver pegmatite_u3d_driver = {
	.probe	= pegmatite_u3d_probe,
	.remove	= pegmatite_u3d_remove,
	.driver	= {
		.owner	= THIS_MODULE,
		.name	= DRIVER_NAME,
		.of_match_table	= pegmatite_u3d_of_match,
	},
};

module_platform_driver(pegmatite_u3d_driver);
MODULE_ALIAS("platform:" DRIVER_NAME);
MODULE_LICENSE("GPL v2");
MODULE_AUTHOR("Tyler Hall <tyhall@lexmark.com>");
MODULE_DESCRIPTION("Pegmatite USB3 UDC");
