/*
 * Marvell Pegmatite SATA PHY driver
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/clk.h>
#include <linux/delay.h>
#include <linux/io.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/phy/phy.h>
#include <linux/platform_device.h>

#define REV_B 2
#define REV_C 4

struct mv_pegmatite_sata_phy {
	struct phy *phy;
	struct clk *phyclk;
	void __iomem *phyregs;
	void __iomem *vsregs;
	bool enb_reva_errata;
	int rev;
};

static unsigned int mv_pegmatite_phy_read(void *reg, unsigned int phyreg) {
	writel(phyreg, reg);
	return readl(reg + 4);
}

static void mv_pegmatite_phy_write(void *reg, unsigned int phyreg, unsigned int val) {
	writel(phyreg, reg);
	writel(val, reg + 4);
}

static void mv_pegmatite_phy_write_mask(void *reg, unsigned int phyreg, unsigned int andmask, unsigned int ormask) {
	unsigned int val;

	writel(phyreg, reg);

	val = readl(reg + 4);
	val &= andmask;
	val |= ormask;

	writel(val, reg + 4);
}

static int mv_pegmatite_sata_phy_init(struct phy *phy)
{
	struct mv_pegmatite_sata_phy *sata_phy = phy_get_drvdata(phy);
	void __iomem *reg = sata_phy->phyregs;
	void __iomem *vdrreg = sata_phy->vsregs;
	unsigned int val;

	/* Hard Reset all registers in the PHY */
	mv_pegmatite_phy_write(reg, 0x252, 0xE208);
	udelay(1);

	/* Write to SATA Control register via VSA/VSD to clear bit 6 to set PU_PLL */
	val = mv_pegmatite_phy_read(vdrreg, 0);
	val |= (1 << 6);
	mv_pegmatite_phy_write(vdrreg, 0, val);

	/*
	 * Power down all PLLs until everything is set up.
	 * Set the PHY mode to SATA
	 * Set the RefClk to the correct frequency
	 */
	mv_pegmatite_phy_write_mask(reg, 0x201, 0x8F00, 1);

	/* Set RefClk Select - not in sim (4F) */
	if (sata_phy->enb_reva_errata)
		mv_pegmatite_phy_write_mask(reg, 0x24F, 0xFFFF, (1 << 10));

	/* These are both reserved fields that were set in the loopback sim */
	mv_pegmatite_phy_write(reg, 0x222, 0x1016);
	mv_pegmatite_phy_write(reg, 0x25C, 0xA8AD);

	/* These are setings for Rx calibration and the Rx regulator from the loopback sim */
	mv_pegmatite_phy_write(reg, 0x241, 0x9608);
	mv_pegmatite_phy_write(reg, 0x242, 0x2436);

	/* Set some drive strengths from the SATA loopback vector - modified the Tx power */
	if (sata_phy->rev < REV_C)
		mv_pegmatite_phy_write_mask(reg, 0x2A5, 0xFFFF, (0x3 << 2));

	mv_pegmatite_phy_write(reg, 0x20D, 0xBA70);
	mv_pegmatite_phy_write(reg, 0x20F, 0xAAF0);
	mv_pegmatite_phy_write(reg, 0x211, 0x0CE2);

	/* These are the Rx settings used in the loopback sim */
	mv_pegmatite_phy_write(reg, 0x212, 0x1149);
	mv_pegmatite_phy_write(reg, 0x210, 0x0949);
	mv_pegmatite_phy_write(reg, 0x20E, 0x0949);
	mv_pegmatite_phy_write(reg, 0x310, 0x04CF);
	mv_pegmatite_phy_write(reg, 0x312, 0x04CF);
	mv_pegmatite_phy_write(reg, 0x314, 0x04CF);

	/*
	 * Set Tx/Rx PHY Gen
	 * Disable Auto RX_INIT
	 * Set TX driver to idle
	 */
	mv_pegmatite_phy_write(reg, 0x226, 0x0922);

	/* Set PHY Gen Max (25) */
	mv_pegmatite_phy_write_mask(reg, 0x225, 0xFFFFF3FF, 0x00000800);

	/* Set PHY bus width to 40 (23) */
	mv_pegmatite_phy_write_mask(reg, 0x223, 0xFFFFF3FF, 0x00000800);

	/* Set bit 12 of 0202 - Max PLL Rate */
	mv_pegmatite_phy_write_mask(reg, 0x202, 0xFFFFFFFF, (1 << 12));

	/* Set the edge select to single edge detect */
	mv_pegmatite_phy_write_mask(reg, 0x42, 0xFFFFFEFF, 0x00000000);

	/* Set Register 0xFE to 02F3 - PIN_SQ_OOB_SEL override to 1 */
	mv_pegmatite_phy_write(reg, 0x2FE, 0x02F3);

	/* Try changing the squelch setting */
	mv_pegmatite_phy_write_mask(reg, 0x206, 0xF0FF, 0x0500);

	val = mv_pegmatite_phy_read(vdrreg, 0);
	val &= 0xFFFFFFBF;
	mv_pegmatite_phy_write(vdrreg, 0, val);

	/* Power up the PLLs */
	mv_pegmatite_phy_write(reg, 0x201, 0xFC01);

	/* Check that the PLL is locked */
	do {
		val = mv_pegmatite_phy_read(reg, 0x201);
		val = ((val >> 8) & 0x1);
	} while (val == 0);

	/* Check that Tx is ready */
	do {
		val = mv_pegmatite_phy_read(reg, 0x223);
		val = ((val >> 4) & 0x1);
	} while (val == 0);

	/* Check that Rx is ready */
	do {
		val = mv_pegmatite_phy_read(reg, 0x223);
		val = ((val >> 5) & 0x1);
	} while (val == 0);

	/* Turn on automatic Rx Init */
	mv_pegmatite_phy_write(reg, 0x226, 0x0122);

	/*
	 * Issue a Soft Reset to the PHY
	 * This should start the autocalibration
	 */
	mv_pegmatite_phy_write(reg, 0x252, 0xE008);
	mv_pegmatite_phy_write(reg, 0x252, 0xE408);
	mv_pegmatite_phy_write(reg, 0x252, 0xE008);

	/* Added instead of doing manual Rx init, just enable the Tx driver */
	mv_pegmatite_phy_write(reg, 0x226, 0x0022);

	return 0;
}

static struct phy_ops mv_pegmatite_sata_phy_ops = {
	.init		= mv_pegmatite_sata_phy_init,
	.owner		= THIS_MODULE,
};

static int mv_pegmatite_sata_phy_probe(struct platform_device *pdev)
{
	struct mv_pegmatite_sata_phy *sata_phy;
	struct device *dev = &pdev->dev;
	struct resource *res;
	struct phy_provider *phy_provider;
	int ret = 0;
	void *__iomem rev_reg;

	sata_phy = devm_kzalloc(dev, sizeof(*sata_phy), GFP_KERNEL);
	if (!sata_phy)
		return -ENOMEM;

	res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "phyregs");
	sata_phy->phyregs = devm_ioremap_resource(dev, res);
	if (IS_ERR(sata_phy->phyregs))
		return PTR_ERR(sata_phy->phyregs);

	res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "vsregs");
	sata_phy->vsregs = devm_ioremap(dev, res->start, resource_size(res));
	if (IS_ERR(sata_phy->vsregs))
		return PTR_ERR(sata_phy->vsregs);

	res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "revregs");
	rev_reg = ioremap(res->start, resource_size(res));
	if (IS_ERR(rev_reg))
		return PTR_ERR(rev_reg);
	sata_phy->rev = readl(rev_reg);
	iounmap(rev_reg);

	dev_set_drvdata(dev, sata_phy);

	sata_phy->phyclk = devm_clk_get(dev, "phy_clk");
	if (IS_ERR(sata_phy->phyclk)) {
		dev_err(dev, "failed to get clk for PHY\n");
		return PTR_ERR(sata_phy->phyclk);
	}

	ret = clk_prepare_enable(sata_phy->phyclk);
	if (ret < 0) {
		dev_err(dev, "failed to enable source clk\n");
		return ret;
	}

	sata_phy->phy = devm_phy_create(dev, dev->of_node,
					&mv_pegmatite_sata_phy_ops, NULL);
	if (IS_ERR(sata_phy->phy)) {
		clk_disable_unprepare(sata_phy->phyclk);
		dev_err(dev, "failed to create PHY\n");
		return PTR_ERR(sata_phy->phy);
	}

	sata_phy->enb_reva_errata = !of_property_read_bool(pdev->dev.of_node, "disable-reva-errata");

	phy_set_drvdata(sata_phy->phy, sata_phy);

	phy_provider = devm_of_phy_provider_register(dev,
					of_phy_simple_xlate);
	if (IS_ERR(phy_provider)) {
		clk_disable_unprepare(sata_phy->phyclk);
		return PTR_ERR(phy_provider);
	}

	return 0;
}

static const struct of_device_id mv_pegmatite_sata_phy_of_match[] = {
	{ .compatible = "marvell,pegmatite-sata-phy" },
	{ },
};
MODULE_DEVICE_TABLE(of, mv_pegmatite_sata_phy_of_match);

static struct platform_driver mv_pegmatite_sata_phy_driver = {
	.probe	= mv_pegmatite_sata_phy_probe,
	.driver = {
		.of_match_table	= mv_pegmatite_sata_phy_of_match,
		.name  = "marvell,sata-phy",
		.owner = THIS_MODULE,
	}
};
module_platform_driver(mv_pegmatite_sata_phy_driver);

MODULE_DESCRIPTION("Marvell Pegmatite SATA PHY driver");
MODULE_LICENSE("GPL");
