/*
 * PCIe support for Pegmatite platforms.
 *
 * This file is licensed under the terms of the GNU General Public
 * License version 2.  This program is licensed "as is" without any
 * warranty of any kind, whether express or implied.
 *
 */
#ifndef __MACH_PEGMATITE_PCIE_H
#define __MACH_PEGMATITE_PCIE_H

#define PCIE_APPERATURE_SIZE        		SZ_32M
#define PCIE_INBOUND_APPERATURES		3
#define PCIE_OUTBOUND_APPERATURES		8
#define PCIE_LCL_CFG_OFFSET			0x8000

/* Only needed for emulation */
#define FPGAD_PCIE_CONFIG			0xfd000068

#define PCIE_FABRIC10_2x2_MODE			(1 << 7)
#define PCIE_FABRIC10_1x4_AND_1x2_MODE		(1 << 6)

#define DBG_ERROR				(1 << 0)	// general errors
#define DBG_WARN				(1 << 1)
#define DBG_INFO    				(1 << 2)
#define DBG_HW_ERR  				(1 << 3)	// errors as reported by hw
#define DBG_PME					(1 << 4)	// power managment
#define DBG_RESOURCE				(1 << 5)	// memory and io reource
#define DBG_STATE				(1 << 6)	// general driver state
#define DBG_BUS					(1 << 7)	// bus topology (bus, dev, fun)
#define DBG_LINK				(1 << 8)	// link status
#define DBG_CFG					(1 << 9)	// config access (rd/wr)
#define DBG_ABORT				(1 << 10)	// abort handler
#define DBG_MB				 	(1 << 11)	// mailbox
#define DBG_MSI				 	(1 << 12)	// msi
#define DBG_INTX		 		(1 << 13)	// intx

#define PCIE_DRIVER_DEBUG			(DBG_ERROR /* | DBG_INFO */ | DBG_HW_ERR)

#define PCIE_PRINTK(a,fmt,args...)		if (a & PCIE_DRIVER_DEBUG) printk(fmt, ##args)

#define PCIE_CAP_LINK_STATUS_LINK_TRAINING	(1 << 27)
#define PCIE_CAP_LINK_STATUS_DLL_LINK_ACTIVE	(1 << 29)
#define PCIE_CAP_LINK_STATUS_LINK_UP_MASK	(PCIE_CAP_LINK_STATUS_LINK_TRAINING | PCIE_CAP_LINK_STATUS_DLL_LINK_ACTIVE)
#define PCIE_CAP_LINK_STATUS_LINK_UP		PCIE_CAP_LINK_STATUS_DLL_LINK_ACTIVE
#define PCIE_CAP_LINK_STATUS_NEG_WIDTH_SH	20
#define PCIE_CAP_LINK_STATUS_NEG_WIDTH		(0x3F << 20)
#define PCIE_CAP_LINK_STATUS_CURR_LINK_SPEED	(0x0F << 16)

#define PCIE_PORT_LINK_CTRL_FAST_LINK_TIMING	(1 << 7)    /* 1= Fast Link Training enable */
#define PCIE_PORT_LINK_CTRL_HOT_RESET		(1 << 3)    /* 1= Hot Reset, generate a high pulse to start a hot reset sequence */

typedef struct pcie_cfg_s
{
	unsigned short VenID;                /* 0x00 -- Vendor ID*/
	unsigned short DevID;                /* 0x02 -- Device ID*/
	unsigned short Cmd;                  /* 0x04 -- Command */
	unsigned short Stat;                 /* 0x06 -- Status */
	unsigned int ClassRevID;           /* 0x08 -- Class Code and Revision ID */
	unsigned char CacheLineSize;        /* 0x0C -- Cache Line Size */
	unsigned char LatTimer;             /* 0x0D -- Latency Timer */
	unsigned char HdrType;              /* 0x0E -- Header Type */
	unsigned char BIST;                 /* 0x0F -- BIST */
	unsigned int BAR0;                 /* 0x10 -- Base Address 0 */
	unsigned int BAR1;                 /* 0x14 -- Base Address 1 */
	unsigned char PrimBusNum;           /* 0x18 -- Primary Bus Number */
	unsigned char SecBusNum;            /* 0x19 -- Secondary Bus Number */
	unsigned char SubBusNum;            /* 0x1A -- Subordinate Bus Number */
	unsigned char SecLatTimer;          /* 0x1B -- Secondary Latency Timer */
	unsigned char IOBase;               /* 0x1C -- I/O Base */
	unsigned char IOLimit;              /* 0x1D -- I/O Limit */
	unsigned short SecStat;              /* 0x1E -- Secondary Status */
	unsigned short MemBase;              /* 0x20 -- Memory Base */
	unsigned short MemLimit;             /* 0x22 -- Memory Limit */
	unsigned short PrefMemBase;          /* 0x24 -- Prefetchable Memory Base */
	unsigned short PrefMemLimit;         /* 0x26 -- Prefetchable Memory Limit */
	unsigned int PrefBaseUpper;        /* 0x28 -- Prefetchable Base Upper 32 */
	unsigned int PrefLimitUpper;       /* 0x2C -- Prefetchable Limit Upper 32 */
	unsigned short IOBaseUpper;          /* 0x30 -- I/O Base Upper 16 */
	unsigned short IOLimitUpper;         /* 0x32 -- I/O Limit Upper 16 */
	unsigned char CapPtr;               /* 0x34 -- Capabilities Pointer */
	unsigned char rsrvd35_37[0x38-0x35]; /* 0x35-0x37 */
	unsigned int ExpROMBaseAddr;       /* 0x38 -- Expansion ROM Base Address */
	unsigned char IntLine;              /* 0x3C -- Interrupt Line */
	unsigned char IntPin;               /* 0x3D -- Interrupt Pin */
	unsigned short BridgeCtrl;          /* 0x3E -- Bridge Control */
	unsigned int LPMCap;               /* 0x40 -- Local Power Management Capabilities */
	unsigned int LPMCSR;               /* 0x44 -- Local Power Management Control and Status */
	unsigned int rsrvd48_4c[(0x50-0x48)/4]; /* blank */
	unsigned int LMsiCtrl;             /* 0x50 -- Local MSI Capabilities Control */
	unsigned int LMsiAddr;             /* 0x54 -- Local MSI Capabilities Address */
	unsigned int rsrvd58;              /* blank */
	unsigned int LMsiData;             /* 0x5c-- Local MSI Capabilities Data */
	unsigned int rsrvd60_6c[(0x70-0x60)/4]; /* blank */
	unsigned int LPCIECap;             /* 0x70 -- Local PCIE Capabilities */
	unsigned int LDevCap;              /* 0x74 -- Local Device Capabilities */
	unsigned int LDevContStatus;       /* 0x78 -- Local Device Control and Status */
	unsigned int LDevLinkCap;          /* 0x7c -- Local Device Link Capabilities */
	unsigned int LLinkContStatus;      /* 0x80 -- Local Link Control and Status */
	unsigned int rsrvd84_88[(0x8c-0x84)/4]; /* blank */
	unsigned int LRootCtrl;            /* 0x8c -- Local Root Control */
	unsigned int LRootStat;            /* 0x90 -- Local Root Status */
	unsigned int LDevCap2;             /* 0x94 -- Local Device Capabilities 2 */
	unsigned int LDevContStatus2;      /* 0x98 -- Local Device Control and Status 2 */
	unsigned int LDevLinkCap2;         /* 0x9c -- Local Device Link Capabilities 2 */
	unsigned int LLinkContStatus2;     /* 0xa0 -- Local Link Control and Status 2 */
	unsigned int rsrvda4_fc[(0x100-0xa4)/4]; /* blank */
	unsigned int LAERCap;              /* 0x100 -- Local AER Capabilities */
	unsigned int LUCorrStat;           /* 0x104 -- Local Uncorrectable Error Status */
	unsigned int LUCorrMask;           /* 0x108 -- Local Uncorrectable Error Mask */
	unsigned int LUCorrSev;            /* 0x10c -- Local Uncorrectable Error Severity */
	unsigned int LCorrStat;            /* 0x110 -- Local Correctable Error Status */
	unsigned int LCorrMask;            /* 0x114 -- Local Correctable Error Mask */
	unsigned int LAERCapCtrl;          /* 0x118 -- Local AER Capabilities and Control */
	unsigned int LHdrLog[4];           /* 0x11c-0x128 -- Local Header Log */
	unsigned int LRootErrCmd;          /* 0x12c -- Local Root Error Command */
	unsigned int LRootErrStat;         /* 0x130 -- Local Root Error Status */
	unsigned int LErrSrcID;            /* 0x134 -- Local Error Source Identification */
	unsigned int LTLPPrefixLog[4];     /* 0x138-0x144 -- Local TLP Prefix Log */
	unsigned int rsrvd148_6fc[(0x700-0x148)/4]; /* blank */
	unsigned int LPortTimer;           /* 0x700 -- Local Port Logic Ack Latency and Replay Timer */
	unsigned int LMsgReg;              /* 0x704 -- Local Port Logic Message */
	unsigned int LForceLink;           /* 0x708 -- Local Port Logic Message */
	unsigned int LAckFreq;             /* 0x70c -- Local Port Logic Ack Frequency */
	unsigned int LPortLink;            /* 0x710 -- Local Port Link Control */
	unsigned int rsrvd714_100c[(0x1010-0x714)/4]; /* blank */
	unsigned int LCfgBAR0Mask;         /* 0x1010 -- Local Config BAR0 Mask */
	unsigned int LCfgBAR1Mask;         /* 0x1014 -- Local Config BAR1 Mask */
} pci_cfg_t, *Pci_cfg_t;

typedef struct rc_msi_irq_s
{
	unsigned int Mask;
	unsigned int Stat;
	unsigned int Clr;
} rc_msi_irq_t;

typedef struct rc_intx_irq_s
{
	unsigned int En;
	unsigned int Cnt;
} rc_intx_irq_t;

#define PCIE_APP_CTRL_REG_LTSSM_EN		(1 << 15) /* 0x8000 */

#define PCIE_APP_STATUS_REG_LINK_STATUS_MASK    0x1F  /* bits 4:0 - Link State */
#define PCIE_APP_STATUS_REG_LINK_UP             (1 << 7) /* bit 7 - Link Up */
#define PCIE_APP_STATUS_REG_LINK_STATUS_MASK_2  0x9F /* bits 7 & 4:0 - Link Up & Link State */
#define PCIE_APP_STATUS_REG_LINK_UP_2           0x91

#define PCIE_APP_INT_ERROR_MASK			0xf1c0e919	/* all but UR */
#define PCIE_APP_INT_PME_MASK			0x0c001000

#define PCIE_CFG_CTRL_FUNC_NUM_SH		0
#define PCIE_CFG_CTRL_DEV_NUM_SH		3
#define PCIE_CFG_CTRL_BUS_NUM_SH		8
#define PCIE_CFG_CTRL_TYPE_1			(1<<16)

typedef struct pci_app_regs_s
{
	unsigned int ACtrl;                /* 0x00 -- Application Control */
	unsigned int AStat;                /* 0x04 -- Application Status */
	unsigned int AIntP;                /* 0x08 -- Application Interrupt Pending */
	unsigned int AIntEn;               /* 0x0c -- Application Interrupt Enable */
	unsigned int AIntAck;              /* 0x10 -- Application Interrupt Acknowledge */
	unsigned int InAddrXlt[3];         /* 0x14-0x1c -- Inbound Address Translation 0 */
	unsigned int BAR0;                 /* 0x20 -- BAR0 */
	unsigned int WinMiss;              /* 0x24 -- Window Miss */
	rc_msi_irq_t RC_MSI_IRQ[4];        /* 0x28-0x54 -- RC MSI Interrupt0-3 (Mask, Stat, Clr) */
	rc_intx_irq_t RC_INTX_IRQ[4];       /* 0x58-0x74 -- RC INTA-D Interrupt (En; Cnt) */
	unsigned int EP_INTA_Gen;          /* 0x78 -- EP INTA Generate */
	unsigned int EP_MSI_Gen;           /* 0x7c -- EP MSI Generate */
	unsigned int EP_MSI_Gen_En;        /* 0x80 -- EP MSI Generate Enabled */
	unsigned int MB_WR_En[8];          /* 0x84-0xa0 -- Mailbox Write Interrupt Enable 0-7 */
	unsigned int MB_WR[8];             /* 0xa4-0xc0 -- Mailbox Write 0-7 */
	unsigned int MB_WR_Stat[8];        /* 0xc4-0xe0 -- Mailbox Write Status 0-7 */
	unsigned int MB_WR_Ack[8];         /* 0xe4-0x100 -- Mailbox Write Interrupt Acknowledge 0-7 */
	unsigned int rsrvd104_120[(0x124-0x104)/4]; /* blank */
	unsigned int AckCount;             /* 0x124 -- received ACK count */
	unsigned int NakCount;             /* 0x128 -- received NAK count */
	unsigned int WrAckCount;           /* 0x12c -- write ACK count register */
	unsigned int WrNakCount;           /* 0x130 -- write NAK count register */
	unsigned int rsrvd134_100c[(0x1010-0x134)/4]; /* blank */
	unsigned int OutAddrXlt[8];        /* 0x1010-0x102c -- Outbound Address Translation 0 */
	unsigned int CfgControl;           /* 0x1030 -- Configuration TLP Settings */
	unsigned int PhyLaneNum;           /* 0x1034 -- Application PHY Lane Number */
	unsigned int RefClkBuf;            /* 0x1038 -- Reference Clock Buffer Control */
	unsigned int rsrvd103c_1050[(0x1050-0x103c)/4]; /* blank */
	unsigned int REV0;                 /* 0x1050 -- IP Tagging Revision 0 */
	unsigned int REV1;                 /* 0x1054 -- IP Tagging Revision 1 */
	unsigned int rsrvd1058_2000[(0x2000-0x1058)/4]; /* blank */

   /*
    * Phy registers start here.  Select each PHY with PhyLaneNum above first.
    */
	unsigned int CID_REG0;             /* 0x2000 -- Chip ID */
	unsigned int POWER_REG0;           /* 0x2004 -- Power and PLL Control Register */
#define	POWER_REG0_RSRVD_15          (1 << 15)
#define	POWER_REG0_PU_PLL            (1 << 14)
#define	POWER_REG0_PU_RX             (1 << 13)
#define	POWER_REG0_PU_TX             (1 << 12)
#define	POWER_REG0_RSRVD_11          (1 << 11)
#define	POWER_REG0_PU_DFE            (1 << 10)
#define POWER_REG0_PHY_MODE_PCIE     (3 <<  5)
#define POWER_REG0_REFCLK_SEL_100MHZ (0 <<  0)

	unsigned int CAL_REG0;             /* 0x2008 -- KVCO Calibration Control Register */
	unsigned int rsrvd200c_2034[(0x2034-0x200c)/4]; /* blank */
	unsigned int appReg_2034;          /* 0x2034 -- undocumented magic register */
	unsigned int appReg_2038;          /* 0x2038 -- undocumented magic register */
	unsigned int appReg_203c;          /* 0x203c -- undocumented magic register */
	unsigned int appReg_2040;          /* 0x2040 -- undocumented magic register */
	unsigned int appReg_2044;          /* 0x2044 -- undocumented magic register */
	unsigned int appReg_2048;          /* 0x2048 -- undocumented magic register */
	unsigned int rsrvd204c_208c[(0x208c-0x204c)/4]; /* blank */
	unsigned int LOOPBACK_REG0;        /* 0x208c -- Digital Loopback Control Register */
	unsigned int appReg_2090;          /* 0x2090 -- undocumented magic register */
	unsigned int appReg_2094;          /* 0x2094 -- undocumented magic register */
	unsigned int appReg_2098;          /* 0x2098 -- undocumented magic register */
	unsigned int rsrvd209c_2104[(0x2104-0x209c)/4]; /* blank */
	unsigned int appReg_2104;          /* 0x2104 -- undocumented magic register */
	unsigned int appReg_2108;          /* 0x2108 -- undocumented magic register */
	unsigned int rsrvd210c_2120[(0x2120-0x210c)/4]; /* blank */
	unsigned int appReg_2120;          /* 0x2120 -- undocumented magic register */
	unsigned int MISC_REG49;           /* 0x2124 -- COMPHY Register 0x49 */
#define LANE_ALIGN_REQ0 MISC_REG49

	unsigned int rsrvd2128_213c[(0x213c-0x2128)/4]; /* blank */
	unsigned int MISC_FREQ_SEL;        /* 0x213c -- COMPHY Register 0x4f - Misc Freq Select */
	unsigned int rsrvd2140;            /* blank */
	unsigned int MISC_REG51;           /* 0x2144 -- COMPHY Register 0x51 */
#define INTERFACE_REQ2 MISC_REG51

	unsigned int appReg_2148;           /* 0x2148 -- undocumented magic register */
	unsigned int rsrvd214c_2170[(0x2170-0x214c)/4]; /* blank */
	unsigned int appReg_2170;           /* 0x2170 -- undocumented magic register */
	unsigned int rsrvd2174_2600[(0x2600-0x2174)/4]; /* blank */
	unsigned int LANE_CFG0;            /* 0x2600 -- Lane Configuration 0 */
	unsigned int LANE_CFG1;            /* 0x2604 -- Lane Configuration 1 */
	unsigned int LANE_STATUS0;         /* 0x2608 -- Lane Status 0 */
	unsigned int LANE_STATUS1;         /* 0x260c -- Lane Status 1 */
	unsigned int LANE_STATUS2;         /* 0x2610 -- Lane Status 2 */
	unsigned int LANE_BEACON;          /* 0x2614 -- Lane Beacon Control */
	unsigned int rsrvd2618_262c[(0x262c-0x2618)/4]; /* blank */
	unsigned int LANE_TXCMN_CFG;       /* 0x262c -- Lane TX Common Mode Configuration Register */
	unsigned int rsrvd2630_269c[(0x269c-0x2630)/4]; /* blank */
	unsigned int LANE_EQ_CFG0;         /* 0x269c -- Lane TX Common Mode Configuration Register */
	unsigned int rsrvd26a0_2700[(0x2700-0x26a0)/4]; /* blank */
	unsigned int GLOB_PHY_REV;         /* 0x2700 -- Global Phy PIPE Revision */
	unsigned int GLOB_CLK_CTRL;        /* 0x2704 -- Global Clock Control */
#define GLOB_CLK_CTRL_MULTICAST    (1 << 7)
#define GLOB_CLK_CTRL_REFCLK_DIV_2 (2 << 4)
#define GLOB_CLK_CTRL_FIXED_PCLK   (1 << 2)
#define GLOB_CLK_CTRL_SOFT_RESET   (1 << 0)

	unsigned int GLOB_TEST_CTRL;       /* 0x2708 -- Global Test Control */
	unsigned int GLOB_CLK_SRC_LO;      /* 0x270c -- Clock Source Low */
#define GLOB_CLK_SRC_LO_PLL_READY_DELAY(_n_)  (_n_ << 4)
#define GLOB_CLK_SRC_LO_BUNDLE_PERIOD_32_NSEC (1 << 2)
#define GLOB_CLK_SRC_LO_BUNDLE_PERIOD_FIXED   (1 << 1)
#define GLOB_CLK_SRC_LO_PCLK_FROM_PLL         (1 << 0)

	unsigned int GLOB_CLK_SRC_HI;      /* 0x2710 -- Clock Source High */

#define CLK_SRC_HI_LANE_MASTER (1 << 2)
#define CLK_SRC_HI_LANE_BREAK  (1 << 1)
#define CLK_SRC_HI_LANE_START  (1 << 0)

	unsigned int GLOB_TRIGGER;         /* 0x2714 -- Global Trigger */
	unsigned int GLOB_MISC_CTRL;       /* 0x2718 -- Global Miscellaneous Control */
	unsigned int GLOB_CLKREQ_CTRL;     /* 0x271c -- CLKREQ# Timing Control */
	unsigned int GLOB_DP_CFG;          /* 0x2720 -- Datapath Configuration */
	unsigned int rsrvd2724_2740[(0x2740-0x2724)/4]; /* blank */
	unsigned int appReg_2740;          /* 0x2740 -- undocumented magic register */
	unsigned int appReg_2744;          /* 0x2744 -- undocumented magic register */
	unsigned int GLOB_COUNTER_CTRL;    /* 0x2748 -- Counter Type Register */
	unsigned int GLOB_COUNTER_LOW;     /* 0x274c -- Counter Low Register */
	unsigned int GLOB_COUNTER_HI;      /* 0x2750 -- Counter High Register */
	unsigned int rsrvd2754_275c[(0x275c-0x2754)/4]; /* blank */
	unsigned int GLOB_LOW_FREQ_CFG;    /* 0x275c -- Low Frequency Configuration Register */
} pci_app_regs_t, *Pci_app_regs_t;

struct pegmatite_pcie_controller {
	struct platform_device *pdev;
	void __iomem *pcie_core_base;
	void __iomem *pcie_lclcfg_base;
	void __iomem *pcie_app_base;
	int irq;
	int endpoint;
	spinlock_t pcie_lock;
	unsigned int pcie_mem_space;
	unsigned int pcie_mem_size;
	unsigned int pcie_apperature_size;
	unsigned int pcie_app_base_physical;
	volatile unsigned int *pcie_reset_control;
};

int
   pegmatite_msi_irq_handler (struct pegmatite_pcie_controller *controller);

#endif /* __MACH_PEGMATITE_PCIE_H */
