/*
 * tsen_pegmatite.c - Driver for pegmatite CPU core temperature monitoring
 *
 * based on existing via-cputemp.c, which is
 *
 * Copyright (C) 2007 Rudolf Marek <r.marek@assembler.cz>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA.
 */

#include <linux/module.h>
#include <linux/delay.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/hwmon.h>
#include <linux/sysfs.h>
#include <linux/hwmon-sysfs.h>
#include <linux/err.h>
#include <linux/platform_device.h>
#include <linux/cpu.h>
#include <linux/io.h>

/* Register Definitions */
#define TSEN_CONFIGURATION	0x0
#define TSEN_CONTROL		0x4
#define TSEN_ADC_CONTROL	0x8
#define TSEN_STATUS		0x10

/* Control Masks */
#define TSEN_RESET_MASK		0x1
#define TSEN_ENABLE_MASK	0x2
#define TSEN_START_MASK		0x4

/* Calibration Masks */
#define TSEN_CAL_MASK		0x0800
#define TEMP_VAL_MASK   	0x01000000
#define TEMP_OUT_MASK   	0x00000fff

enum { SHOW_TEMP, SHOW_NAME };

struct tsen_pegmatite_data {
	struct device *hwmon_dev;
	const char *name;
	void __iomem *ioaddr;
};

/*
 * Sysfs stuff
 */
static ssize_t show_name(struct device *dev, struct device_attribute *devattr, char *buf)
{
	struct tsen_pegmatite_data *data = dev_get_drvdata(dev);
	return snprintf(buf, PAGE_SIZE, "%s\n", data->name);
}

static signed long  _tsen_get_temp_value(void __iomem *ioaddr) {
	uint32_t val = 0;
	if (ioaddr == NULL)
		return -1;

	/* Wait for a valid temperature reading */
	do {
		val = readl(ioaddr + TSEN_STATUS);
	} while (!(val & TEMP_VAL_MASK));

 	val = readl(ioaddr + TSEN_STATUS);
	val &= TEMP_OUT_MASK;
    /* Read the temperature 
	* The function for getting the junction temp is
	* Tj = (.3904*temp_out[11:0])-281.8
	* Since we can't do floating point math in the kernel, 
	* have been multiplied by 10000 and then further manipulated
	* when printing the results (see function call in probe function).
	*/
	return ((3904*val)-2818000);
}

static ssize_t show_temp(struct device *dev, struct device_attribute *devattr, char *buf)
{
	struct tsen_pegmatite_data *data = dev_get_drvdata(dev);
	void __iomem *ioaddr = data->ioaddr;
	unsigned long temp = _tsen_get_temp_value(ioaddr);
	return snprintf(buf, PAGE_SIZE, "%lu.%02lu\n", temp / 10000, temp % 10000);
}
static SENSOR_DEVICE_ATTR(temp, S_IRUGO, show_temp, NULL, SHOW_TEMP);
static SENSOR_DEVICE_ATTR(name, S_IRUGO, show_name, NULL, SHOW_NAME);

static struct attribute *tsen_pegmatite_attributes[] = {
	&sensor_dev_attr_name.dev_attr.attr,
	&sensor_dev_attr_temp.dev_attr.attr,
	NULL
};

static const struct attribute_group tsen_pegmatite_group = {
	.attrs = tsen_pegmatite_attributes
};

static int tsen_pegmatite_probe(struct platform_device *pdev)
{
	struct tsen_pegmatite_data *data;
	struct resource *res;
	int err = 0;
	uint32_t val = 0;

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!res) {
		return -ENODEV;
	}

	data = devm_kzalloc(&pdev->dev, sizeof(*data), GFP_KERNEL);
	if (!data) {
		return -ENOMEM;
	}
	
	data->ioaddr = devm_ioremap_resource(&pdev->dev, res);
	if (IS_ERR(data->ioaddr)) {
		return -ENOMEM;
	}

	data->name = "tsen";

	/* Do the initial calibration sequence */
	/* Reset */
	val = readl(data->ioaddr + TSEN_CONTROL);
	val &= ~TSEN_RESET_MASK;
	writel(val, data->ioaddr + TSEN_CONTROL);

	/* Set enable high to power up */
	val = readl(data->ioaddr + TSEN_CONTROL);
	val |= TSEN_ENABLE_MASK;
	writel(val, data->ioaddr + TSEN_CONTROL);

	/* Wait at least 4 clock cycles */
	udelay(1);

	/* Kick off calibration */
	val = readl(data->ioaddr + TSEN_CONFIGURATION);
	val |= TSEN_CAL_MASK;
	writel(val, data->ioaddr + TSEN_CONFIGURATION);

	/* start conversion/measurement */

	val = readl(data->ioaddr + TSEN_CONTROL);
	val |= TSEN_START_MASK;
	writel(val, data->ioaddr + TSEN_CONTROL);

	/* Wait for a valid temperature reading */
	do {
		val = readl(data->ioaddr + TSEN_STATUS);
	} while (!(val & TEMP_VAL_MASK));

	/* Read the temperature */
	val = readl(data->ioaddr + TSEN_STATUS);
	val &= TEMP_OUT_MASK;

	/* Initial calibration sequence complete */

	platform_set_drvdata(pdev, data);
	err = sysfs_create_group(&pdev->dev.kobj, &tsen_pegmatite_group);
	data->hwmon_dev = hwmon_device_register(&pdev->dev);
	if (IS_ERR(data->hwmon_dev)) {
		err = PTR_ERR(data->hwmon_dev);
		dev_err(&pdev->dev, "Class registration failed (%d)\n",
			err);
		return -ENOMEM;
	}

	return 0;
}

static int tsen_pegmatite_remove(struct platform_device *pdev)
{
	struct tsen_pegmatite_data *data = platform_get_drvdata(pdev);
	hwmon_device_unregister(data->hwmon_dev);
	sysfs_remove_group(&pdev->dev.kobj, &tsen_pegmatite_group);
	platform_set_drvdata(pdev, NULL);
	return 0;
}

static const struct of_device_id tsen_of_match[] = {
	{ .compatible = "marvell,pegmatite-tsen", },
	{},
};

static struct platform_driver tsen_pegmatite_driver = {
	.driver = {
		.owner = THIS_MODULE,
		.name = "pegmatite-tsen",
		.of_match_table = tsen_of_match,
	},
	.probe = tsen_pegmatite_probe,
	.remove = tsen_pegmatite_remove,
};

module_platform_driver(tsen_pegmatite_driver);

MODULE_ALIAS("platform:pegmatite-tsen");
MODULE_DEVICE_TABLE(of, tsen_of_match);
MODULE_DESCRIPTION("pegmatite CPU temperature sensor");
MODULE_LICENSE("GPL");
