/*******************************************************************************
Copyright (C) Marvell International Ltd. and its affiliates

This software file (the "File") is owned and distributed by Marvell
International Ltd. and/or its affiliates ("Marvell") under the following
alternative licensing terms.  Once you have made an election to distribute the
File under one of the following license alternatives, please (i) delete this
introductory statement regarding license alternatives, (ii) delete the two
license alternatives that you have not elected to use and (iii) preserve the
Marvell copyright notice above.

********************************************************************************
Marvell Commercial License Option

If you received this File from Marvell and you have entered into a commercial
license agreement (a "Commercial License") with Marvell, the File is licensed
to you under the terms of the applicable Commercial License.

********************************************************************************
Marvell GPL License Option

If you received this File from Marvell, you may opt to use, redistribute and/or
modify this File in accordance with the terms and conditions of the General
Public License Version 2, June 1991 (the "GPL License"), a copy of which is
available along with the File in the license.txt file or by writing to the Free
Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 or
on the worldwide web at http://www.gnu.org/licenses/gpl.txt.

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE IMPLIED
WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE ARE EXPRESSLY
DISCLAIMED.  The GPL License provides additional details about this warranty
disclaimer.
********************************************************************************
Marvell BSD License Option

If you received this File from Marvell, you may opt to use, redistribute and/or
modify this File under the following licensing terms.
Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    *   Redistributions of source code must retain the above copyright notice,
	    this list of conditions and the following disclaimer.

    *   Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.

    *   Neither the name of Marvell nor the names of its contributors may be
		used to endorse or promote products derived from this software without
		specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*******************************************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <inttypes.h>

#include "config_marvell.h"  	/* Required to identify SOC and Board */
#include "bin_hdr_twsi.h"
#include "mvUart.h"

#include "bootstrap_os.h"


/************************************************************************************
* Name:		suspendWakeup -
* Desc:	 	Detectes suspend to RAM state, reads the return PC
* 		and performs registers write according to a list.
* 		The data is stored in the following format
* 		boot_info_addr + 0x0 = magic word
* 		boot_info_addr + 0x4 = resume pc
* 		boot_info_addr + 0x8 + 8*n = register addres (0xFFFFFFFF means end list)
*		boot_info_addr + 0xC + 8*n = register value
* Args:	 	None.
* Notes:
* Returns:	None.
*/

#define BOOT_INFO_ADDR 		(0x2800)
#define SUSPEND_MAGIC_WORD 	(0xDEADB002)
#define BOOT_MAGIC_WORD2	(0x1f2e3d4c)
#define REGISTER_LIST_END 	(0xFFFFFFFF)
#define MAX_DRAM_WINDOWS        (4)

#define MAX_SANE_PAIR_COUNT     (0x100)   /* just a sanity check.  If too small, it certainly can be increased with no down-side */

/*
 * The trampoline in cache.S must be built as ARM code since we're jumping to
 * an even address and the linker doesn't know about what we're doing since
 * we've copied the code ourselves. Using 0x1000, as we know that is one of
 * the pages that is reserved for DRAM training and has already been trashed.
 * The kernel must not use it.
*/
#define TRAMPOLINE_ADDR ((trampoline_func_t)0x1000)

typedef void (*resume_func_t)(void);
typedef void (*trampoline_func_t)(resume_func_t);

extern void *trampoline_start;
extern void *trampoline_end;

MV_STATUS suspendWakeup(void)
{
#if defined(MV88F78X60)
	int *boot_info = (int*)(BOOT_INFO_ADDR);
	int i;
	int  magic_word;
	resume_func_t resumeFunc = NULL;
	trampoline_func_t trampoline = TRAMPOLINE_ADDR;
	int *reg_addr, reg_value;

	magic_word =  *(boot_info);

	if(magic_word == SUSPEND_MAGIC_WORD)
	{
	   int  magic_word2 = *(boot_info+1);

	   if (magic_word2 != BOOT_MAGIC_WORD2)
	      return MV_OK;

	   /* Clear magic word to avoid succesive resumes */
	   (*boot_info++) =  0x0;
	   (*boot_info++) =  0x0;  /* word2 */

	   /*
	    * Resume function is next.
	    */
	   resumeFunc = (void *)(*boot_info++);

	   DEBUG_INIT_C("Detected suspend-to-RAM. Returning to OS: 0x", (int) resumeFunc,8);

	   /*
	    * The original code just assumed a list of addr/value pairs and looked for a terminating address.
	    * The way that the values are stored in the kernel, exposed a problem with this.
	    *
	    * The complication here is that if we are modifying the Window registers and we are still
	    * actively reading from said window, then the DDR contents will disappear.  This is because
	    * the prior DDR training is using window set 0, but the LXK code instead will be using
	    * window set 2.  The values that are currently being saved in DDR will have 0 disabled.  When
	    * we 'restore' the window 0 value, the DDR contents are no longer visible.
	    *
	    * The fix is to pre-read all the values into a local array in the D$.  We can then program
	    * them all at once with no dependency on the DRAM contents.
	    */
	   {
	      int
		 base_reg_addr [MAX_DRAM_WINDOWS],
		 base_reg_value[MAX_DRAM_WINDOWS],
		 ctrl_reg_addr [MAX_DRAM_WINDOWS],
		 ctrl_reg_value[MAX_DRAM_WINDOWS],
		 win;

	      for (win = 0; win < MAX_DRAM_WINDOWS; win++) {
		 base_reg_addr [win] = (*boot_info++);
		 base_reg_value[win] = (*boot_info++);

		 ctrl_reg_addr [win] = (*boot_info++);
		 ctrl_reg_value[win] = (*boot_info++);
	      }

	      /*
	       * We can now write all address/values pairs for the DRAM WINDOW controls from the D$ contents.
	       */
	      for (win = 0; win < MAX_DRAM_WINDOWS; win++) {

		 /* BASE */
		 reg_addr  = (int *) base_reg_addr [win];
		 reg_value =         base_reg_value[win];

		 *reg_addr = reg_value;

		 /* CTRL */
		 reg_addr  = (int *) ctrl_reg_addr [win];
		 reg_value =         ctrl_reg_value[win];

		 *reg_addr = reg_value;
	      }
	   }

	   /*
	    * We now process all free-form address/value pairs until the end of list marker is read.
	    *
	    * Put a sanity check in to keep from getting any possibility of an infinite loop.  If DDR
	    * is corrupted, we're likely sunk anyway, but try to recover.
	    */
	   for (i=0; i < MAX_SANE_PAIR_COUNT; i++) {

	      reg_addr  = (*boot_info++);
	      reg_value = (*boot_info++);

	      /*
	       * Check for end of list marker, and fixup the pairCount.
	       */
	      if (reg_addr == REGISTER_LIST_END ) {

		 /*
		  * end of list.
		  */
		 break;
	      }

	      *reg_addr = reg_value;
	   }

	   /*
	    * If we have a pairCount that is exactly MAX_PAIRS, then we may not have read all of the values.
	    * We want to make sure we don't have this issue.
	    */
	   if (reg_addr != REGISTER_LIST_END ) {
	      DEBUG_INIT_S("WARNING: Malformed addr/value sets during resume operation.");
	      return(MV_OK);
	   }

	   /*
	    * Copy our trampoline code to DRAM. This will disable the MMU and
	    * jump to the location in r0.
	    */
	   uint32_t *start = (uint32_t *)&trampoline_start;
	   uint32_t *end = (uint32_t *)&trampoline_end;
	   uint32_t *dest = (uint32_t *)trampoline;

	   for (i = 0; &start[i] < end; i++) {
		   dest[i] = start[i];
	   }
	   trampoline(resumeFunc);
	}
#endif
	return MV_OK;
}

