
#include <common.h>
#include <stdarg.h>
#include <i2c.h>
#include <asm/io.h>
#include <lcd.h>
#include <mv_mmp_panel.h>
#include <stdio_dev.h>
#include <lcd.h>
#include <mv_mmp_panel.h>
#include <video_font.h>
#include "mv_mmp_eeprom_info.h"
#include "atmelpnl_switch.h"
#include "mv_mmp_panel_common.h"

#define I2CADDR_PANEL_BB (0x42 >> 1)

static vga_panel_data vpd;

#ifndef CONFIG_PANEL_I2C_BUS
#error CONFIG_PANEL_I2C_BUS must be defined in board config
#endif
static mv_mmp_panel_i2c_info_t i2c_info = {
	.bus_num = -1,
	.dev_addr = I2CADDR_PANEL_BB,
	.read_retries = 0,
	.write_retries = 0,
	.delay = 1000,
	.protocol = PROTOCOL_GEN1,
};

static int hw_ver = 0;
static unsigned char fwmajor = 0;
static unsigned char fwminor = 0;

/*-----------------------------------------------------------------------------
 *
 * Name:        mv_mmp_panel_vga_extclkreq
 *
 * Description: Report whether the display requires external free-running clock.
 *
 * Input:       None
 *
 * Output:
 *
 * Notes:       Clock may still be required internally by lcd block even if not
 *              used externally.
 *---------------------------------------------------------------------------*/
mv_mmp_panel_ext_clk_requirement_t mv_mmp_panel_vga_extclkreq(void)
{
	return MV_MMP_PANEL_EXT_CLK_USED;
}

/*
 * RESET OUT - LVDS/HAPTICS/USB
 *
 * Eeprom checksummming was added in 2.3.
 * Pio control was fixed in fw 2.2 and 3.2, but 1.x, 2.0, 2.1, and 3.0 are broken.
 * If either is something older, continue the old way.
 * Otherwise, just use the stored infomation and return early.
 */
static int mv_mmp_vga_display_enable(vga_panel_info *vpi, int enable) {
    if((!mv_mmp_eeprom_version_blank(vpi) && mv_mmp_eeprom_version_ge(2, 3, vpi)) &&
             ((fwmajor > 3) ||
          (fwmajor == 3 && fwminor >=2 ) ||
          (fwmajor == 2 && fwminor >=2 ))) {
        int rc;

       /* display disable sequence */
       rc = mv_mmp_panel_disp_ctrl(&i2c_info, vpi, SECONDARY_RESET_OUT, enable);
       udelay(5*1000);
       rc |= mv_mmp_panel_disp_ctrl(&i2c_info, vpi, PRIMARY_RESET_OUT, enable);
       if (rc)
          MV_MMPPANELDBG("%s:%d: communication error or no response deasserting reset\n",
                                  __func__, __LINE__);

       return 1;
    }

    return 0;
}

/* set overrides for old hardware versions and panels with old EEPROM */
/* only EEPROM v2.0 and up has "reset out" information */
static void mv_mmp_vga_handle_old_eeproms(vga_panel_info *vpi, uint8_t *pio, uint8_t *pio_block) {
    int override = 0;

    switch(vpi->panel_type) {
       case PANEL_ARUBA:
          if (hw_ver > 0) { /* ARU01C and beyond */
              if (vpi->version_major < 2) {
                  override = 1;
                  *pio_block = 2; *pio = 2; /* PC2 */
              }
          } else { /* ARU01A */
              override = 1;
              *pio_block = 1; *pio = 4; /* PB4 */
          }
          break;
       case PANEL_BVI:
          if (hw_ver > 0) { /* BVI01B and beyond */
              if (vpi->version_major < 2) {
                  override = 1;
                  *pio_block = 2; *pio = 7; /* PC7 */
              }
          } else { /* BVI01A */
              override = 1;
              *pio_block = 1; *pio = 4; /* PB4 */
          }
          break;
       case PANEL_CUBA:
          if (vpi->version_major < 2) {
              override = 1;
              *pio_block = 2; *pio = 2; /* PC2 */
          }
          break;
       case PANEL_KEYWEST:
       case PANEL_BARBADOS4:
       case PANEL_BARBADOS7:
       case PANEL_BARBADOS10:
       default:
          if (vpi->version_major < 2) {
              override = 1;
              *pio_block = 1; *pio = 4; /* PB4 */
          }
          break;
    } /* switch */

    /* can't put this before set gpios because this is a dynamic gpio */
    if (override != 1) {
      *pio = PIO(vpi->flags2);
      *pio_block = PIO_BLOCK(vpi->flags2);
    }
}

/*-----------------------------------------------------------------------------
 *
 * Name:        initGPIOs
 *
 * Description: Initializes the microcontroller's GPIOs
 *
 * Input:       None
 *
 * Output:      None
 *
 * Notes:
 *---------------------------------------------------------------------------*/
static void initGPIOs (void)
{
  uint8_t msg[2];
  uint8_t i, pio, pio_block;
  uint8_t outB, outC;
  uint8_t oerA = 0x00, oerB = 0x00, oerC = 0x00, oerD = 0x00;
  vga_panel_info *vpi = (vga_panel_info *)&vpd.vpi;

  /* STEP 1 - STATIC CONFIGURATION */

  /* GPIOA */
  SET_BIT(oerA, 0, OER_INPUT);   /* ADC input 0 */
  SET_BIT(oerA, 1, OER_INPUT);   /* ADC input 1 */
  SET_BIT(oerA, 2, OER_DEFAULT); /* unused */
  SET_BIT(oerA, 3, OER_INPUT);   /* DISP_ON */
  SET_BIT(oerA, 4, OER_OUTPUT);  /* _PNP-TS2 */
  SET_BIT(oerA, 5, OER_OUTPUT);  /* NPN-TS2 */
  SET_BIT(oerA, 6, OER_OUTPUT);  /* _PNP-TS1 */
  SET_BIT(oerA, 7, OER_OUTPUT);  /* NPN-TS1 */

  /* GPIOB */
  SET_BIT(oerB, 0, OER_DEFAULT); /* dynamically configured, below */
  SET_BIT(oerB, 1, OER_DEFAULT); /* dynamically configured, below */
  SET_BIT(oerB, 2, OER_DEFAULT); /* dynamically configured, below */
  SET_BIT(oerB, 3, OER_DEFAULT); /* dynamically configured, below */
  SET_BIT(oerB, 4, OER_DEFAULT); /* dynamically configured, below */
  SET_BIT(oerB, 5, OER_INPUT);   /* MOSI */
  SET_BIT(oerB, 6, OER_OUTPUT);  /* MISO */
  SET_BIT(oerB, 7, OER_INPUT);   /* SCK */

  /* GPIOC */
  SET_BIT(oerC, 0, OER_OUTPUT);  /* SCL */
  SET_BIT(oerC, 1, OER_OUTPUT);  /* SDA */
  SET_BIT(oerC, 2, OER_DEFAULT); /* unused - possible reset out location */
  SET_BIT(oerC, 3, OER_DEFAULT); /* dynamically configured, below */
  SET_BIT(oerC, 4, OER_DEFAULT); /* dynamically configured, below */
  SET_BIT(oerC, 5, OER_DEFAULT); /* dynamically configured, below */
  SET_BIT(oerC, 6, OER_DEFAULT); /* dynamically configured, below */
  SET_BIT(oerC, 7, OER_DEFAULT); /* dynamically configured, below */

  /* GPIOD */
  SET_BIT(oerD, 0, OER_DEFAULT); /* unused */
  SET_BIT(oerD, 1, OER_DEFAULT); /* unused */
  SET_BIT(oerD, 2, OER_OUTPUT);  /* FP_IRQ/INTN */
  SET_BIT(oerD, 3, OER_OUTPUT);  /* LED_RED */
  SET_BIT(oerD, 4, OER_OUTPUT);  /* PWM_KP */
  SET_BIT(oerD, 5, OER_OUTPUT);  /* PWM_DIM - backlight */
  SET_BIT(oerD, 6, OER_OUTPUT);  /* LED_GREEN */
  SET_BIT(oerD, 7, OER_DEFAULT); /* unused */

  /* STEP 2 - DYNAMIC CONFIGURATION */

  /* KEY-SCAN */
  /* ROWS - GPIOB */
  for (i = 0; i < vpd.vpi.keyscan_rows; i++) {
    SET_BIT(oerB, i, OER_INPUT);
  }
  /* COLUMNS - GPIOC - START AT PC3 */
  for (i = 0; i < vpd.vpi.keyscan_cols; i++) {
    SET_BIT(oerC, i + 3, OER_INPUT);
  }

  /* SET GPIOS */
  msg[0] = MCU_GPIOA_OER_REG;
  msg[1] = oerA;
  mv_mmp_panel_i2c_write(&i2c_info, msg, sizeof(msg));

  msg[0] = MCU_GPIOB_OER_REG;
  msg[1] = oerB;
  mv_mmp_panel_i2c_write(&i2c_info, msg, sizeof(msg));

  msg[0] = MCU_GPIOC_OER_REG;
  msg[1] = oerC;
  mv_mmp_panel_i2c_write(&i2c_info, msg, sizeof(msg));

  msg[0] = MCU_GPIOD_OER_REG;
  msg[1] = oerD;
  mv_mmp_panel_i2c_write(&i2c_info, msg, sizeof(msg));

  if (mv_mmp_vga_display_enable(vpi, 0))
      return;

  mv_mmp_vga_handle_old_eeproms(vpi, &pio, &pio_block);

  /* get pio block from EEPROM */
  msg[0] = mv_mmp_common_gpio_map[pio_block].oer_reg;
  /* read current value */
  mv_mmp_panel_i2c_read(&i2c_info, msg[0], 1, &msg[1], sizeof(msg[1]));
  /* set appropriate bit */
  SET_BIT(msg[1], pio, OER_OUTPUT);
  mv_mmp_panel_i2c_write(&i2c_info, msg, sizeof(msg));

    /* TRI-STATE UNUSED PIOS */
    /* we want to disable the internal pull-ups on the unused PIOs */
    /* they're already set up as inputs, above, we just need to zero the appropriate bits in the PORT registers */

    /* TRI-STATE UNUSED KEY-SCAN PIOS */
    /* we specifically care about bar01g PC7 - RSTSW-N */

    mv_mmp_panel_i2c_read(&i2c_info, MCU_GPIOB_OUT_REG, 1, &outB, sizeof(outB));
    mv_mmp_panel_i2c_read(&i2c_info, MCU_GPIOC_OUT_REG, 1, &outC, sizeof(outC));

    /* ROWS @ PB0-4 */
    for (i = 4; i >= vpd.vpi.keyscan_rows; i--) {
        SET_BIT(outB, i, 0);
    }

    /* COLUMNS @ PC3-7 */
    for (i = 7; i >= vpd.vpi.keyscan_cols + 3; i--) {
        SET_BIT(outC, i, 0);
    }

    msg[0] = MCU_GPIOB_OUT_REG;
    msg[1] = outB;
    mv_mmp_panel_i2c_write(&i2c_info, msg, sizeof(msg));

    msg[0] = MCU_GPIOC_OUT_REG;
    msg[1] = outC;
    mv_mmp_panel_i2c_write(&i2c_info, msg, sizeof(msg));

    /* TRI-STATE OTHER UNUSED PIOS */
    /* these are unused across the board */

    msg[0] = MCU_GPIOD_OUT_REG;
    mv_mmp_panel_i2c_read(&i2c_info, msg[0], 1, &msg[1], sizeof(msg[1]));
    msg[1] &= ~0x81; /* bar01g PD0 - RSTHW-N */
                     /* bar01g PD7 - STRB */
    mv_mmp_panel_i2c_write(&i2c_info, msg, sizeof(msg));

}

/*
 * LED INFO BYTE
 * 0 - PIO
 * 1 - PIO
 * 2 - PIO
 * 3 - PIO block
 * 4 - PIO block
 * 5 - drive value
 * 6 - reserved
 * 7 - reserved
 */
#define LED_PIO(x) ((x) & 0x07)
#define LED_PIO_BLOCK(x) (((x) & 0x18) >> 3)
#define LED_DRIVE_VALUE(x) (((x) & 0x20) >> 5)

#define LED_GENERATE_MASK_ON(gpio, info) ( LED_DRIVE_VALUE(info) ? ((1 << LED_PIO(info)) | gpio) : (~(1 << LED_PIO(info)) & gpio) )
#define LED_GENERATE_MASK_OFF(gpio, info) ( LED_DRIVE_VALUE(info) ? (~(1 << LED_PIO(info)) & gpio) : ((1 << LED_PIO(info)) | gpio) )

static void initLEDOn(uint8_t green_info, uint8_t red_info)
{
  unsigned char
    dat,
    msg[2];
  int cmd;

  /*
   * Read the GPIOD_OER register
   */
  cmd = MCU_GPIOD_OER_REG;
  mv_mmp_panel_i2c_read(&i2c_info, cmd, 1, &dat, sizeof(dat));

  dat = ~dat;
  dat |= 0x48;
  dat = ~dat;

  /*
   * Write to the GPIOD_OER register
   */
  msg[0] = MCU_GPIOD_OER_REG;
  msg[1] = dat;
  mv_mmp_panel_i2c_write(&i2c_info, msg, sizeof(msg));

  cmd = MCU_GPIOD_OUT_REG;
  mv_mmp_panel_i2c_read(&i2c_info, cmd, 1, &dat, sizeof(dat));

  /* turn LED green */
  dat = LED_GENERATE_MASK_ON(dat, green_info);
  dat = LED_GENERATE_MASK_OFF(dat, red_info);

  /* turn on cavelight */
  dat |= 0x10;

  msg[0] = MCU_GPIOD_OUT_REG;
  msg[1] = dat;
  mv_mmp_panel_i2c_write(&i2c_info, msg, sizeof(msg));

  msg[0] = MCU_LED_CONTROL;
  msg[1] = 0x01; /* green on */
  mv_mmp_panel_i2c_write(&i2c_info, msg, 2);
}

/*-----------------------------------------------------------------------------
 *
 * Name:        backlightEnable
 *
 * Description: turn backlight off or on using current duty cycle
 *
 * Input:       None
 *
 * Output:      None
 *
 * Notes:
 *---------------------------------------------------------------------------*/
static int backlightEnable(int enable) {
  unsigned char
    msg[2],dat;
  int cmd, rc;

  if(!enable) {
            /* disable PWM because it doesn't quite go to zero */
            cmd = MCU_ALT_FUNC_CONFIG_REG;
            rc = mv_mmp_panel_i2c_read(&i2c_info, cmd, 1, &dat, 1);
            if(rc)
	    	return rc;
	    dat &= ~0x02; /* PWM enable bit */
            msg[0] = cmd;
            msg[1] = dat;
            rc = mv_mmp_panel_i2c_write(&i2c_info, msg, 2);
            if(rc)
	    	return rc;

            /* drive PWM_DIM low */
            cmd = MCU_GPIOD_OUT_REG;
            rc = mv_mmp_panel_i2c_read(&i2c_info, cmd, 1, &dat, 1);
            if(rc)
	    	return rc;
	    SET_BIT(dat, 5, 0);
            rc = mv_mmp_panel_i2c_write(&i2c_info, msg, 2);
            if(rc)
	    	return rc;

            cmd = MCU_GPIOD_OER_REG;
            rc = mv_mmp_panel_i2c_read(&i2c_info, cmd, 1, &dat, 1);
            if(rc)
	    	return rc;
	    SET_BIT(dat, 5, OER_OUTPUT);
            msg[0] = cmd;
            msg[1] = dat;
            rc = mv_mmp_panel_i2c_write(&i2c_info, msg, 2);
            if(rc)
	    	return rc;
  }
  else {
            /* enable PWM */
            cmd = MCU_ALT_FUNC_CONFIG_REG;
            rc = mv_mmp_panel_i2c_read(&i2c_info, cmd, 1, &dat, 1);
            if(rc)
	    	return rc;
            dat |= 0x02; /* PWM enable bit */
            msg[0] = cmd;
            msg[1] = dat;
            rc = mv_mmp_panel_i2c_write(&i2c_info, msg, 2);
            if(rc)
	    	return rc;
  }

  return 0;
}

/*-----------------------------------------------------------------------------
 *
 * Name:        panelEnable
 *
 * Description: Enable video and keyscan
 *
 * Input:       None
 *
 * Output:      None
 *
 * Notes:
 *---------------------------------------------------------------------------*/
static void panelEnable(void) {
  unsigned char
    msg[2],dat;
  int cmd;
  uint8_t i, outB, pio, pio_block;
  vga_panel_info *vpi = (vga_panel_info *)&vpd.vpi;

  /* set shutdown and display on both to 1 for legacy compatibility */
  cmd = MCU_GPIOA_OUT_REG;
  mv_mmp_panel_i2c_read(&i2c_info, cmd, 1, &dat, 1);

  dat |= 1<<4;
  msg[0] = MCU_GPIOA_OUT_REG;
  msg[1] = dat;
  mv_mmp_panel_i2c_write(&i2c_info, msg, 2);

  /* ENABLE VIDEO AND KEY-SCAN - DRIVE HIGH */

  /* KEY-SCAN ROWS - GPIOB */
  outB = 0x00; /* default */
  for (i = 0; i < vpd.vpi.keyscan_rows; i++) {
    SET_BIT(outB, i, 1);
  }
  msg[0] = MCU_GPIOB_OUT_REG;
  msg[1] = outB;
  mv_mmp_panel_i2c_write(&i2c_info, msg, 2);

  /*
   * RESET OUT - LVDS/HAPTICS/USB
   *
   * Eeprom checksummming was added in 2.3.
   * Pio control was fixed in fw 2.2 and 3.2, but 1.x, 2.0, 2.1, and 3.0 are broken.
   * If either is something older, continue the old way.
   * Otherwise, just use the stored infomation and return early.
   */

    if (PIO_ENABLED(vpd.vpi.fpga_rstsw_n)) {
        pio = PIO(vpd.vpi.fpga_rstsw_n);
        pio_block = PIO_BLOCK(vpd.vpi.fpga_rstsw_n);

        /* configure as output */
        msg[0] = mv_mmp_common_gpio_map[pio_block].oer_reg;
        mv_mmp_panel_i2c_read(&i2c_info, msg[0], 1, &msg[1], 1);
        msg[1] &= ~(0x01 << pio);
        mv_mmp_panel_i2c_write(&i2c_info, msg, 2);

        /* drive high */
        msg[0] = mv_mmp_common_gpio_map[pio_block].out_reg;
        mv_mmp_panel_i2c_read(&i2c_info, msg[0], 1, &msg[1], 1);
        msg[1] |= (0x01 << pio);
        mv_mmp_panel_i2c_write(&i2c_info, msg, 2);

        udelay(5*1000);
    }

    if (PIO_ENABLED(vpd.vpi.fpga_rsthw_n)) {
        pio = PIO(vpd.vpi.fpga_rsthw_n);
        pio_block = PIO_BLOCK(vpd.vpi.fpga_rsthw_n);
        /* need to pulse RSTHW-N, it's pulled up initially */

        /* configure as output */
        msg[0] = mv_mmp_common_gpio_map[pio_block].oer_reg;
        mv_mmp_panel_i2c_read(&i2c_info, msg[0], 1, &msg[1], 1);
        msg[1] &= ~(0x01 << pio);
        mv_mmp_panel_i2c_write(&i2c_info, msg, 2);

        /* drive low */
        msg[0] = mv_mmp_common_gpio_map[pio_block].out_reg;
        mv_mmp_panel_i2c_read(&i2c_info, msg[0], 1, &msg[1], 1);
        msg[1] &= ~(0x01 << pio);
        mv_mmp_panel_i2c_write(&i2c_info, msg, 2);

        udelay(1); /* 500ns requirement */

        /* drive high */
        msg[0] = mv_mmp_common_gpio_map[pio_block].out_reg;
        mv_mmp_panel_i2c_read(&i2c_info, msg[0], 1, &msg[1], 1);
        msg[1] |= (0x01 << pio);
        mv_mmp_panel_i2c_write(&i2c_info, msg, 2);

        udelay(5*1000);
    }

    if (mv_mmp_vga_display_enable(vpi, 1))
        return;

    mv_mmp_vga_handle_old_eeproms(vpi, &pio, &pio_block);

  /* get pio block from EEPROM */
  msg[0] = mv_mmp_common_gpio_map[pio_block].out_reg;
  /* read current value */
  mv_mmp_panel_i2c_read(&i2c_info, msg[0], 1, &msg[1], sizeof(msg[1]));
  /* set appropriate bit */
  SET_BIT(msg[1], pio, 1);
  mv_mmp_panel_i2c_write(&i2c_info, msg, 2);

  return;
}

void initCavelight(void)
{
   unsigned char msg[2];
   /* Turn the cave light on to DIM. */
   msg[0] = MCU_CAVE_CONTROL;
   msg[1] = 0x82;
   mv_mmp_panel_i2c_write(&i2c_info, msg, 2);
}

int mv_mmp_panel_vga_probe(void)
{
  char porKeys[11];
  unsigned char *dest, data[3];
  int cmd, rc;
  vga_panel_info *vpi = (vga_panel_info *)&vpd.vpi;
  const char *panel_name = "Unrecognized";

  MV_MMPPANELDBG("%s:%d\n", __func__, __LINE__);
  rc = mv_mmp_panel_i2c_probe(&i2c_info);
  if(rc) {
     MV_MMPPANELDBG("%s:%d: no response\n", __func__, __LINE__);
     return 1;
  }
  rc = mv_mmp_panel_reset_mcu(&i2c_info);
  if (rc) {
     MV_MMPPANELDBG("%s:%d: communication error or no response rc: %d \n", __func__, __LINE__, rc);
     return 1;
  }

  /* Get the firmware version */
  cmd = MCU_GET_FWREV;
  rc = mv_mmp_panel_i2c_read(&i2c_info, cmd, 1, data, 3);
  fwmajor = data[0];
  fwminor = data[1];
  hw_ver = data[2];

  /* Get the panel info from the eeprom */
  data[0] = MCU_EEPROM_ADDR_REG;
  data[1] = 0x00;
  if(!rc)
  	mv_mmp_panel_i2c_write(&i2c_info, data, 2);

  cmd = MCU_EEPROM_DATA_REG;
  dest = (unsigned char *)vpi;
  if(!rc)
  	rc = mv_mmp_panel_i2c_read(&i2c_info, cmd, 1, dest, sizeof(vga_panel_info));

  if (rc) {
     MV_MMPPANELDBG("%s:%d: communication error or no response rc: %d \n", __func__, __LINE__, rc);
     return 1;
  }

  printf("%s:%d: panel uicc type %d, hw vers %d, eeprom v%d.%d\n", __func__, __LINE__,
  					vpi->panel_type, hw_ver,
					vpi->version_major, vpi->version_minor);

  printf("flags1=0x%02x, flags2=0x%02x, lvds type %d\n", vpi->flags1, vpi->flags2,
  									vpi->lvds_type);

  if((vpi->panel_type == PANEL_NOT_DETECTED) || (vpi->panel_type >= PANEL_NOT_RECOGNIZED))
  {
     return 1;
  }

  if(mv_mmp_eeprom_version_blank(vpi)) {
     printf("%s:%d: Uicc eeprom corrupted\n",__func__, __LINE__);
     return -1;
  }

  /*
   * mv_mmp_eeprom_checksum will do its own supported revision check.
   */
  rc = mv_mmp_eeprom_checksum(vpi);
  if(rc < 0) {
     printf("%s:%d: checksum failed\n", __func__, __LINE__);
     return 1;
  }

  /*
   * Initialize the panel
   */
  {
    /*
     * Initializes the microcontroller's GPIOs
     */
    initGPIOs();

    /*
     * Initializes the PWM for backlight control and set the BL
     */
    backlightEnable(0);

    /*
     * Initializes the key scanning functionality
     */
    mv_mmp_panel_keyscan_init(&i2c_info, vpi->keyscan_rows, vpi->keyscan_cols);

    /*
     * Initializes the GPIOs fof the LED controls and set the LED
     */
    initLEDOn(vpi->led_info_green, vpi->led_info_red);

    /* enable display and keyscan but leave backlight off */
    panelEnable();

    initCavelight();

    sprintf(porKeys, "0x%08x", (unsigned int)mv_mmp_panel_vga_buttons());
    setenv("PORKEYS", porKeys);
  }

  vpd.xres = ((vpi->xres_msb & 0xff) << 8) | (vpi->xres_lsb & 0xff);
  vpd.yres = ((vpi->yres_msb & 0xff) << 8) | (vpi->yres_lsb & 0xff);

  vpd.vga_type= MV_MMP_PANEL_TYPE_INVALID;

  /* TODO: determine display subtypes */
  switch((PANEL_TYPES)vpi->panel_type)
  {
     case PANEL_KEYWEST:
        panel_name = "Keywest 4.3\"";
        vpd.vga_type = MV_MMP_PANEL_TYPE_480X272;
        break;
     case PANEL_BVI:
        panel_name = "BVI 4.3\"";
        if(hw_ver < 1) {
           vpd.vga_type = MV_MMP_PANEL_TYPE_480X272_FIXW;
        }
        else {
           vpd.vga_type = MV_MMP_PANEL_TYPE_480X272;
        }
        break;
     case PANEL_ARUBA:
        panel_name = "Aruba 7.0\"";
        vpd.vga_type= MV_MMP_PANEL_TYPE_800X480;
        if(hw_ver < 1) {
           vpd.vga_type = MV_MMP_PANEL_TYPE_800X480_FIXW;
        }
        else {
           vpd.vga_type = MV_MMP_PANEL_TYPE_800X480;
        }
        break;
     case PANEL_BARBADOS4:
        panel_name = "Barbados 4.3\"";
        {
            int tmp;
            tmp = (vpd.xres + vpi->hfp + vpi->hpw + vpi->hbp);
            tmp *= (vpd.yres + vpi->vfp + vpi->vpw + vpi->vbp);
            if(tmp > 200000) {
                printf("Alternate timing detected for 4.3\" panel\n");
                vpd.vga_type= MV_MMP_PANEL_TYPE_480X272_PROTO;
            }
            else {
                vpd.vga_type= MV_MMP_PANEL_TYPE_480X272;
            }
        }
        if(hw_ver < 1)
           printf("Warning! %s not supported\n", panel_name);
        break;
     case PANEL_BARBADOS10:
        panel_name = "Barbados 10.2\"";
        vpd.vga_type= MV_MMP_PANEL_TYPE_800X480_SCALED;
        break;
     case PANEL_BARBADOS7:
        panel_name = "Barbados 7.0\"";
        vpd.vga_type= MV_MMP_PANEL_TYPE_800X480;
        if(hw_ver < 1)
           printf("Warning! %s not supported\n", panel_name);
        break;
     case PANEL_CUBA:
        if((vpd.xres == 480) && (vpd.yres == 272)) {
           panel_name = "Cuba 4.3\"";
           vpd.vga_type = MV_MMP_PANEL_TYPE_480X272;
        }
        else if((vpd.xres == 800) && (vpd.yres == 480)) {
          panel_name = "Cuba 7.0\"";
          vpd.vga_type= MV_MMP_PANEL_TYPE_800X480;
        }
        else {
           vpd.vga_type= MV_MMP_PANEL_TYPE_INVALID;
        }
        break;
     default:
        vpd.vga_type= MV_MMP_PANEL_TYPE_INVALID;
        break;
  }
  printf("%s Panel Detected\n", panel_name);

  vpd.sequential = (mv_mmp_eeprom_version_ge(2, 1, vpi) && !(vpi->flags1 & PANEL_FLAGS1_RGB));

  /* panel_info defaults to the maximum legal values, which predetermine the buffer size */
  if((vpd.yres < 0) || (vpd.yres > panel_info.vl_row) ||
        (vpd.xres < 0) || (vpd.xres > panel_info.vl_col)) {
     printf("%s:%d: unsupported display size rows=%d cols=%d\n", __func__, __LINE__,
            vpd.yres, vpd.xres);
     printf("Uicc eeprom corrupted\n");
     return -1;
  }

  if(PANEL_FLAGS1_PIXEL_DEPTH(vpi->flags1) != PIXEL_DEPTH_16) {
     printf("%s:%d: unsupported color depth, flags1=0x%02x\n", __func__, __LINE__,
            vpi->flags1);
     printf("Uicc eeprom corrupted\n");
     return -1;
  }

  /* update the defaults to real values for common/lcd.c*/
  panel_info.vl_row = vpd.yres;
  panel_info.vl_col = vpd.xres;

  /* hardware info for mv_mmp_vga_mv61fb */
  panel_info.priv = &vpd;

  lcd_line_length = (panel_info.vl_col * NBITS (panel_info.vl_bpix)) / 8;

  vpd.backlight_enable = backlightEnable;

  printf("Big Brother %d.%d Detected\n", fwmajor, fwminor);

  return(0);
}

void mv_mmp_panel_vga_status(MV_MMP_PANEL_STATUS msg, va_list ap)
{
   /*
    * Updating smpn displays is slow, and newlines will do it automatically.
    * Only need to explicitly call panel_info.refresh() for cases without a
    * printf is not terminated by '\n'. It will be a null pointer for other
    * displays, so always check it.
    */
   int data1, data2;

   if(msg != MV_MMP_PANEL_BLANK)
      stdio_deregister("lcd");

   switch (msg) {

   case MV_MMP_PANEL_BLANK:
      lcd_clear();
      if(panel_info.refresh)
      	(*panel_info.refresh)();
      break;

   case MV_MMP_PANEL_DRAMTEST_DONE:
      lcd_puts("\nDramTest Complete\n");
      break;

   case MV_MMP_PANEL_DRAMTEST_START:
      lcd_clear();
      lcd_puts("Starting Dram Test\n");
      break;

   case MV_MMP_PANEL_DRAMTEST_DIAG:
      lcd_clear();
      lcd_puts(" Dram Testing\n");
      break;

   case MV_MMP_PANEL_MEMSIZE_CPUSPEED:
      data1 = va_arg(ap, int);
      data2 = va_arg(ap, int);
      if (data2) { /* cpuspeed too */
      	lcd_clear();
      	lcd_printf("%4d MB %4d Mhz", data1, data2);
      }
      else { /* only memsize */
      	lcd_printf("\r%4d MB", data1);
      }
      if(panel_info.refresh)
      	(*panel_info.refresh)();
      break;

   case MV_MMP_PANEL_SCROLLING_DOT:
   case MV_MMP_PANEL_EARLY_MEMORY_TEST:
      break;

   /*
    * PANEL ERROR MESSAGES:
    */
   case MV_MMP_PANEL_ASIC1_FAIL:
      lcd_clear();
      lcd_puts("957 Service System Board\n");
      break;

   case MV_MMP_PANEL_BAD_MBRSIZE:
      lcd_clear();
      lcd_puts("900 Service MBR Size\n");
      break;

   case MV_MMP_PANEL_NO_MBR:
      lcd_clear();
      lcd_puts("900 Service No MBR\n");
      break;

   case MV_MMP_PANEL_DECOMPRESS_ERROR:
      lcd_clear();
      lcd_puts("Decompress Code Error\n");
      break;

   case MV_MMP_PANEL_DECOMPRESS_FAIL:
      lcd_clear();
      lcd_puts("Decompression Failure\n");
      break;

   case MV_MMP_PANEL_DRAMTEST_FAIL:
      data1 = va_arg(ap, int);
      lcd_clear();
      lcd_printf("Failure Addr: 0x%x\n", data1);
      break;

   case MV_MMP_PANEL_ECC_ERROR:  /* Found Non Correctable Error */
      lcd_clear();
      lcd_puts("958 Service Multibit ECC\n");
      break;

   case MV_MMP_PANEL_ECC_FAIL:
      lcd_clear();
      lcd_puts("900 Service ECC Failure\n");
      break;

   case MV_MMP_PANEL_FLASH_ERROR:
      lcd_clear();
      lcd_puts("Flash Error No empty blocks\n");
      break;

   case MV_MMP_PANEL_MEMORY_ERROR:
      data1 = va_arg(ap, int);
      lcd_clear();
      lcd_printf("96%d Service Memory Error - %c", data1 >> 8, data1);
      if(panel_info.refresh)
      	(*panel_info.refresh)();
      break;

   case MV_MMP_PANEL_NO_DRAM:
      lcd_clear();
      lcd_puts("No DRAM Installed\n");
      break;

   case MV_MMP_PANEL_NO_NVRAM:
      lcd_clear();
      lcd_puts("No Nvram Found\n");
      break;

   case MV_MMP_PANEL_ROMCRC_ERROR:
      data1 = va_arg(ap, int);
      lcd_printf("955 Service\nCode CRC P %d", data1);
      if(panel_info.refresh)
      	(*panel_info.refresh)();
      break;

   case MV_MMP_PANEL_EXCESSIVE_MEMORY_ERROR:
      data1 = va_arg(ap, int);
      lcd_clear();
      lcd_printf("55 Unsupported Memory in slot %d\n", data1);
      break;

   case MV_MMP_PANEL_UNSUPPORTED_FIRMWARE:
      lcd_clear();
      lcd_puts("40 Unsupported Firmware Card\n");
      break;

   case MV_MMP_PANEL_UNSUPPORTED_FLASHCARD:
      data1 = va_arg(ap, int);
      lcd_clear();
      lcd_printf("55 Unsupported Flash in Slot %d\n", data1);
      break;

   case MV_MMP_PANEL_WATCHDOG_ERROR:
      lcd_clear();
      lcd_puts("System Timeout\n");
      lcd_puts("To continue, power off,wait 10 seconds, power on\n");
      break;

   case MV_MMP_PANEL_FLS_STR:
      {
         char *str = va_arg(ap, char *);
         lcd_clear();
         lcd_puts(str);
         if(panel_info.refresh)
            (*panel_info.refresh)();
      }
      break;

   case MV_MMP_PANEL_ENGINELESS_STRING:
      {
      	int i=0, max;
      	const char *string = "Engineless Mode. ";

      	lcd_clear();
      	max = ((vpd.yres/VIDEO_FONT_HEIGHT)*(vpd.xres/VIDEO_FONT_WIDTH))/(strlen(string));
      	for(i=0; i<max;i++) {
           lcd_puts(string);
      	}
      	if(panel_info.refresh)
           (*panel_info.refresh)();
      }
      break;

   case MV_MMP_PANEL_VERSIONS:
      {
         char *rip, *eng;
         int englen;
         rip = va_arg(ap, char *);
         eng = va_arg(ap, char *);
         lcd_position_cursor(0, 0);
         /* If you want to try to put the versions at the bottom
          * the line below should do it if uncommented
          * console_row=(vpd.yres/VIDEO_FONT_HEIGHT) - 1;
          */
         if(rip) {
            lcd_puts(rip);
         }
         if(eng) {
            englen = strlen(eng);
            lcd_position_cursor((vpd.xres/VIDEO_FONT_WIDTH)-englen, 0);
            lcd_puts(eng);
         }
         if(panel_info.refresh)
            (*panel_info.refresh)();
         }
      break;

   case MV_MMP_PANEL_AUTH_FAIL:
      lcd_clear();
      lcd_puts("959.20 Service System Board\n");
      break;

   default:
      lcd_puts("900 Service Fatal Boot err\n");
      break;
   }
}

unsigned long long mv_mmp_panel_vga_buttons(void)
{
   unsigned int buttons;
   const key_map_t *map = mv_mmp_common_key_map;
   
   buttons = mv_mmp_panel_read_buttons(&i2c_info, map, 0, 0);

   return (long long) mv_mmp_panel_buttons_alias(buttons);
}

int mv_mmp_panel_vga_type(void)
{
       if(vpd.vga_type == (unsigned char)MV_MMP_PANEL_TYPE_INVALID)
               return (int)MV_MMP_PANEL_TYPE_INVALID;

       else
               return (int)vpd.vga_type;
}

char *mv_mmp_panel_vga_subtype(void)
{
	/*
	 * As a placeholder, just keep it simple.
	 * TODO: Build the subtype string in the probe function switch
	 * to handle the display_type field.
	 */
	if(vpd.sequential)
		return "seq";
	else
		return "";
}
