/*----------------------------------------------------------------------------
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */
#include <common.h>
#include <stdarg.h>
#include <i2c.h>
#include <asm/arch/gpio.h>
#include <asm/arch/board_priv.h>
#include "mv_mmp_panel.h"
#include "mv_mmp_panel_common.h"

static panel_info_t panel_info;

enum {
    APATYPE_NONE = 0x00,
    APATYPE_TIANMA,
    APATYPE_POWERTIP,
};

#define     I2C_NUM_RETRIES                10
static unsigned int read_led_panel_buttons(void);
static unsigned int read_led_panel_buttons_only(unsigned int porkeys);

#define I2CADDR_PANEL_2L (0x4A >> 1)
#define I2CADDR_PANEL_LED (0x46 >> 1)

#define KEY_ROWS_LED 2
#define KEY_COLS_LED 2
#define KEY_ROWS_2LINE 6
#define KEY_COLS_2LINE 2

#if 0
#define PRINTF(_s_, ...) printf("{%s:%s} " _s_, __BASE_FILE__, __func__, ##__VA_ARGS__)
#else
#define PRINTF(_s_, ...)
#endif

#ifndef CONFIG_PANEL_I2C_BUS
#error CONFIG_PANEL_I2C_BUS must be defined in board config
#endif
static mv_mmp_panel_i2c_info_t i2c_info = {
	.bus_num = -1,
	.read_retries = I2C_NUM_RETRIES,
	.write_retries = I2C_NUM_RETRIES,
	.delay = 1000,
	.protocol = PROTOCOL_GEN1,
};

static const key_map_t PORKeyMap_led[] =
{
   {1<<0,  0x00}, /*   BTN_CANCEL_KEY_CHAR*/
   {1<<1,  0x08}, /*   BTN_START_KEY_CHAR */
};

static const key_map_t PORKeyMap_2line[] =
{
   {1<<0,  0x00}, /*   BTN_CANCEL_KEY_CHAR*/
   {1<<1,  0x20}, /*   BTN_LEFT_KEY_CHAR  */
   {1<<2,  0x18}, /*   BTN_OK_KEY_CHAR    */
   {1<<3,  0x28}, /*   BTN_RIGHT_KEY_CHAR */
   {1<<4,  0x10}, /*   BTN_HOME_KEY_CHAR  */
   {1<<5,  0x08}, /*   BTN_BACK_KEY_CHAR  */
};

int dram_test_led_state;
int is_scrolling_enabled;
static unsigned char *DispBuff = NULL;

void clear_non_status_leds(void)
{
    PRINTF("Set led state to 1.\n");
    dram_test_led_state = 1;
}

void shift_non_status_led(void)
{
    unsigned char msg[2];
    int rc;

    if (dram_test_led_state != 0) {
        switch(dram_test_led_state % 5) {

        case 1:
            // Set LED 1
            PRINTF("Set LED 1.\n");
            msg[1] = 0x78;
            break;
        case 2:
            // Set LED 2
            PRINTF("Set LED 2\n");
            msg[1] = 0x74;
            break;
        case 3:
            // Set LED 3
            PRINTF("Set LED 3\n");
            msg[1] = 0x6C;
            break;
        case 4:
            // Set LED 4
            PRINTF("Set LED 4\n");
            msg[1] = 0x5C;
            break;
        case 0:
            // Set LED 5
            PRINTF("Set LED 5\n");
            msg[1] = 0x3C;
            break;
        default:
            //NO operation.
            break;
        }

        // Sequence represents the LED state on the panel. Each LED
        // is represented by 1 bit(ON/OFF). Now set that panel via I2C
        // on atmel. NOTE, LED is ACTIVE LOW, so we'll need to set it
        // to zero in order for it to light up. Cheers!
        msg[0] = MCU_GPIOD_OUT_REG;
        rc = mv_mmp_panel_i2c_write(&i2c_info, msg, 2);

    }

    dram_test_led_state++;
}

void LEDSvcErr(int secondary, int tertiary)
{
    unsigned char msg[2];
    int pattern[5];
    unsigned int btn_pressed = 0;
    unsigned int switch_click_count = 0;
    unsigned int exit_click_count = 0;
    int delay = 0;
    int current = 0;
    int stat = 1;
    int retry = 0;

    memset(pattern, 0, sizeof(pattern));
    while (1) {
        btn_pressed = read_led_panel_buttons_only(btn_pressed);
        if (btn_pressed == 2) {
            retry = 6;
            while (retry) {
                btn_pressed = read_led_panel_buttons_only(btn_pressed);
                if (btn_pressed == 0) {
                    switch_click_count++;
                    delay++;
                    break;
                }
                mdelay(50);
                retry--;
            }
        } else if (btn_pressed == 3) {
            retry = 6;
            while (retry) {
                btn_pressed = read_led_panel_buttons_only(btn_pressed);
                if (btn_pressed == 1) {
                    PRINTF("Click exit!\n");
                    exit_click_count++;
                    delay++;
                    break;
                }
                mdelay(50);
                retry--;
            }
        } else {
            mdelay(300);
            if (delay > 0) {
                delay++;
            }
            if (delay == 5) {
                // Next double click occured too late.
                switch_click_count = 0;
                exit_click_count = 0;
                delay = 0;
            }
        }

        if (exit_click_count == 2) {
            // Do not exit on primary code.
            if (current != 0) {
                break;
            } else {
                exit_click_count = 0;
                delay = 0;
            }
        } else if (switch_click_count == 2) {
            switch_click_count = 0;
            delay = 0;
            if (current == 0) {
                current = secondary;
            } else if (current == secondary) {
                current = tertiary;
            } else {
                current = 0;
            }
            PRINTF("switching...\n");
        }

        if (stat == 1) {
            msg[1] = current;
            stat = 0;
        } else {
            msg[1] = 0xFC;
            stat = 1;
        }
        msg[0] = MCU_GPIOD_OUT_REG;
        mv_mmp_panel_i2c_write(&i2c_info, msg, 2);
    } // while (1)

    printf("Exiting svcerr\n");
}

static void led_report_status(int type, va_list ap)
{
    int data1;
    unsigned char msg[2];

    switch (type) {
        /*
         * PANEL STATUS MESSAGES:
         */
    case MV_MMP_PANEL_BLANK:
        break;

    case MV_MMP_PANEL_DRAMTEST_DONE:
        PRINTF("DRAMTEST Done!\n");
        msg[1] = 0xFC;
        msg[0] = MCU_GPIOD_OUT_REG;
        mv_mmp_panel_i2c_write(&i2c_info, msg, 2);
        PRINTF("Re-enabling boot led...");
        mv_mmp_panel_i2c_read(&i2c_info, MCU_ALT_FUNC_CONFIG_REG,1, &msg[1], 1);
        msg[0] = MCU_ALT_FUNC_CONFIG_REG;
        msg[1] |= 0x04;
        mv_mmp_panel_i2c_write(&i2c_info, msg, 2);
        PRINTF("done.\n");
        break;

    case MV_MMP_PANEL_DRAMTEST_START:
        PRINTF("DRAMTEST Start!\n");
        PRINTF("Disabling boot led........");
        mv_mmp_panel_i2c_read(&i2c_info, MCU_ALT_FUNC_CONFIG_REG,1, &msg[1], 1);
        msg[0] = MCU_ALT_FUNC_CONFIG_REG;
        msg[1] &= 0xFB;
        mv_mmp_panel_i2c_write(&i2c_info, msg, 2);
        PRINTF("done.\n");
        break;
        // Same routine for both?
    case MV_MMP_PANEL_MEMSIZE_CPUSPEED:
    case MV_MMP_PANEL_DRAMTEST_DIAG:

        // Alight LEDs 2-6 (excluding status LED) each time
        // the memory size increases.
        data1 = va_arg(ap, int);
        if (data1 == 0) {
            PRINTF("Clear non status leds.\n");
            clear_non_status_leds();
        } else {
            PRINTF("Shift status leds.\n");
            shift_non_status_led();
        }
        break;
    case MV_MMP_PANEL_SCROLLING_DOT:
    case MV_MMP_PANEL_EARLY_MEMORY_TEST:
        break;

        /*
         * PANEL ERROR MESSAGES:
         */
    case MV_MMP_PANEL_ECC_ERROR:   /* Found Non Correctable Error */
        /* 958 Service */
        /* Multibit ECC */
        LEDSvcErr(0x68, 0x9C);
        break;

    case MV_MMP_PANEL_NO_NVRAM:
        // 952 Error?
        LEDSvcErr(0x68,0xAC);
        break;

    case MV_MMP_PANEL_ROMCRC_ERROR:
        // 955 Error?
        LEDSvcErr(0x68,0xA8);
        break;

    case MV_MMP_PANEL_MEMORY_ERROR:
    case MV_MMP_PANEL_EXCESSIVE_MEMORY_ERROR:
        LEDSvcErr(0x64,0xBC);
        break;

    case MV_MMP_PANEL_WATCHDOG_ERROR:
        /* blinking light pattern 111111, 010101, 101010 */
        /* Watchdog Error */
        LEDSvcErr(0xA8, 0x54);
        break;

    case MV_MMP_PANEL_ASIC1_FAIL:
    case MV_MMP_PANEL_AUTH_FAIL:
         LEDSvcErr(0x68, 0xA4);
         break;

    case MV_MMP_PANEL_BAD_MBRSIZE:
    case MV_MMP_PANEL_NO_MBR:
    case MV_MMP_PANEL_DECOMPRESS_ERROR:
    case MV_MMP_PANEL_DECOMPRESS_FAIL:
    case MV_MMP_PANEL_DRAMTEST_FAIL:
    case MV_MMP_PANEL_ECC_FAIL:
    case MV_MMP_PANEL_FLASH_ERROR:
    case MV_MMP_PANEL_NO_DRAM:
    case MV_MMP_PANEL_UNSUPPORTED_FIRMWARE:
    case MV_MMP_PANEL_UNSUPPORTED_FLASHCARD:
    default:
        /* 900 Service */
        /* Fatal Error */
        LEDSvcErr(0x7C, 0xBC);
        break;
    }
}





#define DELAY_FOR_PANEL()   udelay(1000)
#define LCD_2LINE_ALLPAGES  8

/*-----------------------------------------
  Enums for the ST7567 LCD controller
  registers settings.
  ----------------------------------------*/
enum _PanelST7567InstructionMode {
    TWO_LINE_COMMAND_MODE,
    TWO_LINE_DATA_MODE,
    TWO_LINE_UNKNOWN_MODE
};


typedef enum _PanelST7567InstructionMode PanelST7567InstructionMode;

/*------------------------------------------------------------------------------
 * Name:    panel_i2c_msg() (DEPRECATED)
 *
 * Description: Updated version of i2cMsgH() wrapper
 *
 * Notes: Though updated for clarity, this function is DEPRECATED!
 *        It is preserved in this file for now only because removing it will
 *        be tedious.
 *        This function should not be used or ported for any new code.
 *        Instead, the mv_mmp_panel_i2c_read() and mv_mmp_panel_i2c_write()
 *        wrapper functions should be called directly.
 *-----------------------------------------------------------------------------*/
static int
panel_i2c_msg(unsigned char *msg, int msglen,
    unsigned char *dat, int datlen, unsigned char *rcv, int rcvlen)
{
    int rc = 0;
    unsigned char buf[msglen + datlen];

    memcpy(buf, msg, msglen);
    memcpy(buf + msglen, dat, datlen);

    if(msglen + datlen)
       rc = mv_mmp_panel_i2c_write(&i2c_info, buf, msglen + datlen);

    if(rc)
        return rc;

    if(rcvlen && rcv)
        rc = mv_mmp_panel_i2c_read(&i2c_info, 0, 0, rcv, rcvlen);

    return rc;
}


/*------------------------------------------------------------------------------
 * Name:    PanelWriteInstructionST7567()
 *
 * Description: Sends data to the TLS8202 LCD controller through the SPI
 *              interface of the ATMEL I/O Expander
 *
 * Input:       None
 *
 * Output:      None
 *
 * Notes:       Refer to the Serial Peripheral Interface section in the ATMEL
 *              I/O Expander spec for the details
 *-----------------------------------------------------------------------------*/

static void
PanelWriteInstructionST7567(PanelST7567InstructionMode imode,
                PanelST7567InstructionMode transferbyte)
{
    unsigned char
     cmd, dat;

    DELAY_FOR_PANEL();

    /*
       1.) The A0 is used to determine whether the serial
       data input is display or command data. Set A0
       to the correct level prior to start of of transmission.
     */

    /* Read first the PB6 port (A0 - PB6) */
    cmd = MCU_GPIOB_OUT_REG;
    panel_i2c_msg(&cmd, 1, NULL, 0, &dat, 1);

    /* Set the status of A0 based on the instruction mode */
    if (TWO_LINE_COMMAND_MODE == imode) //Please declare " TWO_LINE_COMMAND_MODE this on panel_2line_lcd.h at
    {
        /* CLEAR A0 to signal command mode instruction. */
        dat = 0xBF; //A0_LCD_LOW;
    } else if (TWO_LINE_DATA_MODE == imode) {
        /* SET A0 to signal data mode instruction. */
        dat = 0xFF; //A0_LCD_HIGH;
    }
    DELAY_FOR_PANEL();
    panel_i2c_msg(&cmd, 1, &dat, 1, NULL, 0);
    DELAY_FOR_PANEL();
    cmd = MCU_SPI_TDR_REG;
    dat = transferbyte;
    panel_i2c_msg(&cmd, 1, &dat, 1, NULL, 0);
    DELAY_FOR_PANEL();
    /* Read the SPI Status Register */
    cmd = MCU_SPI_SR_REG;
    panel_i2c_msg(&cmd, 1, NULL, 0, &dat, 1);

    DELAY_FOR_PANEL();
    cmd = MCU_IRQ_MASK_REG;
    panel_i2c_msg(&cmd, 1, NULL, 0, &dat, 1);
    dat |= 0x04;
    DELAY_FOR_PANEL();
    panel_i2c_msg(&cmd, 1, &dat, 1, NULL, 0);

}

/*--------------------------------------------------------------------------------------
 *                          Sitronix ST7567 65 x 132 Dot Matrix
 *                        LCD Controller/Driver Command Settings
 *
 * Note: Please refer to the Sitronix ST7567 LCD Controller Specification
 ---------------------------------------------------------------------------------------*/

#define     LCD_2LINE_DISPLAY_INVERSED          0   /* Inverse display */
//1     /* Normal display */

enum _LcdST7567Instruction {
    TWO_LINE_INSTRUCTION_DATA = 0x20000000, /* Mark data command. */
    TWO_LINE_INSTRUCTION_BASIC = 0x40000000,    /* Mark basic instruction */
    TWO_LINE_INSTRUCTION_SET_MASK = (TWO_LINE_INSTRUCTION_DATA |
                     TWO_LINE_INSTRUCTION_BASIC),
    TWO_LINE_INSTRUCTION_BIT0 = 0x01,   /* Bit( 0 ) of instruction. */
    TWO_LINE_INSTRUCTION_BIT1 = 0x02,   /* Bit( 1 ) of instruction. */
    TWO_LINE_INSTRUCTION_BIT2 = 0x04,   /* Bit( 2 ) of instruction. */
    TWO_LINE_INSTRUCTION_BIT3 = 0x08,   /* Bit( 3 ) of instruction. */
    TWO_LINE_INSTRUCTION_BIT4 = 0x10,   /* Bit( 4 ) of instruction. */
    TWO_LINE_INSTRUCTION_BIT5 = 0x20,   /* Bit( 5 ) of instruction. */
    TWO_LINE_INSTRUCTION_BIT6 = 0x40,   /* Bit( 6 ) of instruction. */
    TWO_LINE_INSTRUCTION_BIT7 = 0x80,   /* Bit( 7 ) of instruction. */
    TWO_LINE_INSTRUCTION_BITS_MASK = (TWO_LINE_INSTRUCTION_BIT0 |
                      TWO_LINE_INSTRUCTION_BIT1 |
                      TWO_LINE_INSTRUCTION_BIT2 |
                      TWO_LINE_INSTRUCTION_BIT3 |
                      TWO_LINE_INSTRUCTION_BIT4 |
                      TWO_LINE_INSTRUCTION_BIT5 |
                      TWO_LINE_INSTRUCTION_BIT6 |
                      TWO_LINE_INSTRUCTION_BIT7),

    /*-------------------------------
         Display ON/OFF
     -------------------------------*/

    TWO_LINE_DISPLAY_CONTROL_BIT_D = 0x01,  /* Bit(0), D0 */
    TWO_LINE_DISPLAY_CONTROL_BASE = 0xAE,   /* BASE VALUE */
    TWO_LINE_DISPLAY_CONTROL_MASK = (TWO_LINE_DISPLAY_CONTROL_BIT_D |
                     TWO_LINE_DISPLAY_CONTROL_BASE),
    TWO_LINE_DISPLAY_CONTROL = (TWO_LINE_DISPLAY_CONTROL_BASE |
                    TWO_LINE_INSTRUCTION_BASIC),

    /*-------------------------------
       Set Start Line
      -------------------------------*/

    TWO_LINE_SET_START_LINE_BIT_S0 = 0x01,  /* Bit(0), S0 */
    TWO_LINE_SET_START_LINE_BIT_S1 = 0x02,  /* Bit(1), S1 */
    TWO_LINE_SET_START_LINE_BIT_S2 = 0x04,  /* Bit(2), S2 */
    TWO_LINE_SET_START_LINE_BIT_S3 = 0x08,  /* Bit(3), S3 */
    TWO_LINE_SET_START_LINE_BIT_S4 = 0x10,  /* Bit(4), S4 */
    TWO_LINE_SET_START_LINE_BIT_S5 = 0x20,  /* Bit(5), S5 */
    TWO_LINE_SET_START_LINE_BASE = 0x40,    /* Base */
    TWO_LINE_SET_START_LINE_MASK = (TWO_LINE_SET_START_LINE_BIT_S0 |
                    TWO_LINE_SET_START_LINE_BIT_S1 |
                    TWO_LINE_SET_START_LINE_BIT_S2 |
                    TWO_LINE_SET_START_LINE_BIT_S3 |
                    TWO_LINE_SET_START_LINE_BIT_S4 |
                    TWO_LINE_SET_START_LINE_BIT_S5 |
                    TWO_LINE_SET_START_LINE_BASE),
    TWO_LINE_SET_START_LINE = (TWO_LINE_SET_START_LINE_BASE |
                   TWO_LINE_INSTRUCTION_BASIC),

    /*-------------------------------
       Set Page Address
    -------------------------------*/

    TWO_LINE_PAGE_ADDR_BIT_Y0 = 0x01,   /* Bit(0), D0 */
    TWO_LINE_PAGE_ADDR_BIT_Y1 = 0x02,   /* Bit(1), D1 */
    TWO_LINE_PAGE_ADDR_BIT_Y2 = 0x04,   /* Bit(2), D2 */
    TWO_LINE_PAGE_ADDR_BIT_Y3 = 0x08,   /* Bit(3), D3 */
    TWO_LINE_PAGE_ADDR_BASE = 0xB0, /* Base */
    TWO_LINE_PAGE_ADDR_MASK = (TWO_LINE_PAGE_ADDR_BIT_Y0 |
                   TWO_LINE_PAGE_ADDR_BIT_Y1 |
                   TWO_LINE_PAGE_ADDR_BIT_Y2 |
                   TWO_LINE_PAGE_ADDR_BIT_Y3 | TWO_LINE_PAGE_ADDR_BASE),
    TWO_LINE_PAGE_ADDR = (TWO_LINE_PAGE_ADDR_BASE | TWO_LINE_INSTRUCTION_BASIC),

    /*-------------------------------
       Set Column Address MSB
    -------------------------------*/

    TWO_LINE_COLADDR_MSB_BIT_X4 = 0x01, /* Bit(0), D0 */
    TWO_LINE_COLADDR_MSB_BIT_X5 = 0x02, /* Bit(1), D1 */
    TWO_LINE_COLADDR_MSB_BIT_X6 = 0x04, /* Bit(2), D2 */
    TWO_LINE_COLADDR_MSB_BIT_X7 = 0x08, /* Bit(3), D3 */
    TWO_LINE_COLADDR_MSB_BASE = 0x10,   /* Base */
    TWO_LINE_COLADDR_MSB_MASK = (TWO_LINE_COLADDR_MSB_BIT_X4 |
                     TWO_LINE_COLADDR_MSB_BIT_X5 |
                     TWO_LINE_COLADDR_MSB_BIT_X6 |
                     TWO_LINE_COLADDR_MSB_BIT_X7 |
                     TWO_LINE_COLADDR_MSB_BASE),
    TWO_LINE_COLLADDR_MSB = (TWO_LINE_COLADDR_MSB_BASE | TWO_LINE_INSTRUCTION_BASIC),

    /*-------------------------------
        Set Column Address LSB
    -------------------------------*/

    TWO_LINE_COLADDR_LSB_BIT_X0 = 0x01, /* Bit(0), D0 */
    TWO_LINE_COLADDR_LSB_BIT_X1 = 0x02, /* Bit(1), D1 */
    TWO_LINE_COLADDR_LSB_BIT_X2 = 0x04, /* Bit(2), D2 */
    TWO_LINE_COLADDR_LSB_BIT_X3 = 0x08, /* Bit(3), D3 */
    TWO_LINE_COLADDR_LSB_BASE = 0x00,   /* Base */
    TWO_LINE_COLADDR_LSB_MASK = (TWO_LINE_COLADDR_LSB_BIT_X0 |
                     TWO_LINE_COLADDR_LSB_BIT_X1 |
                     TWO_LINE_COLADDR_LSB_BIT_X2 |
                     TWO_LINE_COLADDR_LSB_BIT_X3 |
                     TWO_LINE_COLADDR_LSB_BASE),
    TWO_LINE_COLLADDR_LSB = (TWO_LINE_COLADDR_LSB_BASE | TWO_LINE_INSTRUCTION_BASIC),

    /*-------------------------------
         Write Data
     -------------------------------*/

    TWO_LINE_WRITE_DATA_D0 = 0x01,  /* Bit(0), D0 */
    TWO_LINE_WRITE_DATA_D1 = 0x02,  /* Bit(1), D1 */
    TWO_LINE_WRITE_DATA_D2 = 0x04,  /* Bit(2), D2 */
    TWO_LINE_WRITE_DATA_D3 = 0x08,  /* Bit(3), D3 */
    TWO_LINE_WRITE_DATA_D4 = 0x10,  /* Bit(4), D4 */
    TWO_LINE_WRITE_DATA_D5 = 0x20,  /* Bit(5), D5 */
    TWO_LINE_WRITE_DATA_D6 = 0x40,  /* Bit(6), D6 */
    TWO_LINE_WRITE_DATA_D7 = 0x80,  /* Bit(7), D7 */
    TWO_LINE_WRITE_DATA_BASE = 0x00,    /* BASE VALUE */
    TWO_LINE_WRITE_DATA_MASK = (TWO_LINE_WRITE_DATA_D0 |
                    TWO_LINE_WRITE_DATA_D1 |
                    TWO_LINE_WRITE_DATA_D2 |
                    TWO_LINE_WRITE_DATA_D3 |
                    TWO_LINE_WRITE_DATA_D4 |
                    TWO_LINE_WRITE_DATA_D5 |
                    TWO_LINE_WRITE_DATA_D6 |
                    TWO_LINE_WRITE_DATA_D7 | TWO_LINE_WRITE_DATA_BASE),
    TWO_LINE_WRITE_DATA = (TWO_LINE_WRITE_DATA_BASE | TWO_LINE_INSTRUCTION_DATA),

    /*-------------------------------
       SEG Direction
     -------------------------------*/

    TWO_LINE_SEG_DIRECTION_MX = 0x01,   /* MX = 1 Reverse Direction */
    TWO_LINE_SEG_DIRECTION_BASE = 0xA0, /* MX = 0 Normal Direction BASE VALUE */
    TWO_LINE_SEG_DIRECTION_MASK = (TWO_LINE_SEG_DIRECTION_MX |
                       TWO_LINE_SEG_DIRECTION_BASE),
    TWO_LINE_SEG_DIRECTION = (TWO_LINE_SEG_DIRECTION_BASE |
                  TWO_LINE_INSTRUCTION_BASIC),

    /*-------------------------------
       Inverse Display
     -------------------------------*/

    TWO_LINE_INV_DISP_INV = 0x01,   /* INV = 1 Inverse Display */
    TWO_LINE_INV_DISP_BASE = 0xA6,  /* INV = 0 Normal Display */
    TWO_LINE_INV_DISP_MASK = (TWO_LINE_INV_DISP_INV | TWO_LINE_INV_DISP_BASE),
    TWO_LINE_INV_DISP = (TWO_LINE_INV_DISP_BASE | TWO_LINE_INSTRUCTION_BASIC),

    /*-------------------------------
       All Pixel ON
     -------------------------------*/

    TWO_LINE_ALL_PIXEL_ON_AP = 0x01,    /* AP = 1 Set all pixel ON */
    TWO_LINE_ALL_PIXEL_ON_BASE = 0xA4,  /* AP = 0 Normal Display */
    TWO_LINE_ALL_PIXEL_ON_MASK = (TWO_LINE_ALL_PIXEL_ON_AP |
                      TWO_LINE_ALL_PIXEL_ON_BASE),
    TWO_LINE_ALL_PIXEL_ON = (TWO_LINE_ALL_PIXEL_ON_BASE | TWO_LINE_INSTRUCTION_BASIC),

    /*-------------------------------
       BIAS SELECT
     -------------------------------*/

    TWO_LINE_BIAS_SELECT_BS = 0x01, /* BS = 1 - 1/7 (AT 1/65 DUTY) */
    TWO_LINE_BIAS_SELECT_BASE = 0xA2,   /* BS = 0 - 1/9 (AT 1/65 DUTY) */
    TWO_LINE_BIAS_SELECT_MASK = (TWO_LINE_BIAS_SELECT_BS | TWO_LINE_BIAS_SELECT_BASE),
    TWO_LINE_BIAS_SELECT = (TWO_LINE_BIAS_SELECT_BASE | TWO_LINE_INSTRUCTION_BASIC),

    /*------------------------------------------------------------------
       Read-modify-Write - Column address increment: Read:+0, Write:+1
     -------------------------------------------------------------------*/

    TWO_LINE_READ_MODIFY_WRITE = (0xE0 | TWO_LINE_INSTRUCTION_BASIC),

    /*------------------------------------
       END - Exit Read-modify-Write mode
     ------------------------------------*/

    TWO_LINE_END = (0xEE | TWO_LINE_INSTRUCTION_BASIC),

    /*------------------------------------
        RESET - Software Reset
     ------------------------------------*/

    TWO_LINE_RESET = (0xE2 | TWO_LINE_INSTRUCTION_BASIC),

    /*----------------------------------------------
        COM DIRECTION - Set output direction of COM
      ---------------------------------------------*/

    TWO_LINE_COM_DIR_MY = 0x08, /* MY = 1 - REVERSE DIRECTION */
    TWO_LINE_COM_DIR_BASE = 0xC0,   /* MY = 0 - NORMAL DIRECTION */
    TWO_LINE_COM_DIR_MASK = (TWO_LINE_COM_DIR_MY | TWO_LINE_COM_DIR_BASE),
    TWO_LINE_COM_DIR = (TWO_LINE_COM_DIR_BASE | TWO_LINE_INSTRUCTION_BASIC),

    /*-----------------------------------------------------------
          Power Control - Control built-in power circuit ON/OFF
     ----------------------------------------------------------*/

    TWO_LINE_POWER_CONTROL_BIT_VF = 0x01,   /* Bit(0), D0 */
    TWO_LINE_POWER_CONTROL_BIT_VR = 0x02,   /* Bit(1), D1 */
    TWO_LINE_POWER_CONTROL_BIT_VB = 0x04,   /* Bit(2), D2 */
    TWO_LINE_POWER_CONTROL_BASE = 0x28, /* Base */
    TWO_LINE_POWER_CONTROL_MASK = (TWO_LINE_POWER_CONTROL_BIT_VF |
                       TWO_LINE_POWER_CONTROL_BIT_VR |
                       TWO_LINE_POWER_CONTROL_BIT_VB |
                       TWO_LINE_POWER_CONTROL_BASE),
    TWO_LINE_POWER_CONTROL = (TWO_LINE_POWER_CONTROL_BASE |
                  TWO_LINE_INSTRUCTION_BASIC),

    /*------------------------------------------------------
      Regulation Ratio - Select regulation resistor ratio
    -------------------------------------------------------*/

    TWO_LINE_REGULATION_RATIO_RR0 = 0x01,   /* Bit(0), RR0 */
    TWO_LINE_REGULATION_RATIO_RR1 = 0x02,   /* Bit(1), RR1 */
    TWO_LINE_REGULATION_RATIO_RR2 = 0x04,   /* Bit(2), RR2 */
    TWO_LINE_REGULATION_RATIO_BASE = 0x20,  /* Base */
    TWO_LINE_REGULATION_RATIO_MASK = (TWO_LINE_REGULATION_RATIO_RR0 |
                      TWO_LINE_REGULATION_RATIO_RR1 |
                      TWO_LINE_REGULATION_RATIO_RR2 |
                      TWO_LINE_REGULATION_RATIO_BASE),
    TWO_LINE_REGULATION_RATIO = (TWO_LINE_REGULATION_RATIO_BASE |
                     TWO_LINE_INSTRUCTION_BASIC),

    /*-------------------------------------------------------------
     SET EV - (Double command.. - Set Electronic Volume (EV) level
    --------------------------------------------------------------*/

    TWO_LINE_SET_EV_INST1 = (0x81 | TWO_LINE_INSTRUCTION_BASIC),    /* 1st instruction */

    TWO_LINE_SET_EV_EV0 = 0x01, /* Bit(0), EV0 */
    TWO_LINE_SET_EV_EV1 = 0x02, /* Bit(1), EV1 */
    TWO_LINE_SET_EV_EV2 = 0x04, /* Bit(2), EV2 */
    TWO_LINE_SET_EV_EV3 = 0x08, /* Bit(3), EV3 */
    TWO_LINE_SET_EV_EV4 = 0x10, /* Bit(4), EV4 */
    TWO_LINE_SET_EV_EV5 = 0x20, /* Bit(5), EV5 */
    TWO_LINE_SET_EV_BASE = 0x00,    /* Base */
    TWO_LINE_SET_EV_MASK = (TWO_LINE_SET_EV_EV0 |
                TWO_LINE_SET_EV_EV1 |
                TWO_LINE_SET_EV_EV2 |
                TWO_LINE_SET_EV_EV3 |
                TWO_LINE_SET_EV_EV4 |
                TWO_LINE_SET_EV_EV5 | TWO_LINE_SET_EV_BASE),
    TWO_LINE_SET_EV = (TWO_LINE_SET_EV_BASE | TWO_LINE_INSTRUCTION_BASIC),

    /*-------------------------------------------------------------
      SET BOOSTER - (Double command.. - Set BOOSTER level
     --------------------------------------------------------------*/

    TWO_LINE_SET_BOOSTER_INST1 = (0xF8 | TWO_LINE_INSTRUCTION_BASIC),   /* 1st instruction */

    TWO_LINE_SET_BOOSTER_BL = 0x01, /* BL = 1 - 5X */
    TWO_LINE_SET_BOOSTER_BASE = 0x00,   /* BL = 0 - 4X - Base */
    TWO_LINE_SET_BOOSTER_MASK = (TWO_LINE_SET_BOOSTER_BL | TWO_LINE_SET_BOOSTER_BASE),
    TWO_LINE_SET_BOOSTER = (TWO_LINE_SET_BOOSTER_BASE | TWO_LINE_INSTRUCTION_BASIC),

    /*-------------------------------
       Power Save Normal Mode
     -------------------------------*/

    TWO_LINE_PWRSAVE_NORMAL_MODE1 = (0xAE | TWO_LINE_INSTRUCTION_BASIC),
    TWO_LINE_PWRSAVE_NORMAL_MODE2 = (0xA5 | TWO_LINE_INSTRUCTION_BASIC),

    /*-------------------------------
        Power Save Mode
     -------------------------------*/

    TWO_LINE_PWRSAVE_MODE1 = (0xA4 | TWO_LINE_INSTRUCTION_BASIC),
    TWO_LINE_PWRSAVE_MODE2 = (0xAF | TWO_LINE_INSTRUCTION_BASIC),

    /*-------------------------------
            No Operation
     -------------------------------*/

    TWO_LINE_NO_OPERATION = (0xE3 | TWO_LINE_INSTRUCTION_BASIC),

};

typedef struct fonts_entry
    {
        unsigned char key;
        unsigned char value[16];
    } font_twoline_entries;


static const font_twoline_entries font_twoline[] = {
    {0x20,{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,}},
    {0x21,{0x00, 0x00, 0x00, 0xf8, 0x00, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x1b, 0x00, 0x00, 0x00, 0x00,}},
    {0x22,{0x00, 0x00, 0x78, 0x00, 0x00, 0x78, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,}},
    {0x23,{0x40, 0x40, 0xc0, 0x78, 0x40, 0xc0, 0x78, 0x40,
           0x02, 0x1e, 0x03, 0x02, 0x1e, 0x03, 0x02, 0x02,}},
    {0x24,{0x00, 0x70, 0x88, 0xfc, 0x08, 0x10, 0x00, 0x00,
           0x00, 0x08, 0x10, 0x3f, 0x11, 0x0e, 0x00, 0x00,}},
    {0x25,{0x70, 0x88, 0x88, 0x70, 0xc0, 0x30, 0x08, 0x00,
           0x00, 0x10, 0x0c, 0x03, 0x0e, 0x11, 0x11, 0x0e,}},
    {0x26,{0x00, 0x70, 0x88, 0x48, 0x30, 0x00, 0x00, 0x00,
           0x0e, 0x11, 0x10, 0x11, 0x12, 0x0c, 0x13, 0x00,}},
    {0x27,{0x00, 0x00, 0x00, 0x78, 0x00, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,}},
    {0x28,{0x00, 0x00, 0x00, 0xc0, 0x30, 0x08, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x0f, 0x30, 0x40, 0x00, 0x00,}},
    {0x29,{0x00, 0x00, 0x08, 0x30, 0xc0, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x40, 0x30, 0x0f, 0x00, 0x00, 0x00,}},
    {0x2a,{0x00, 0x50, 0x20, 0xf8, 0x20, 0x50, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,}},
    {0x2b,{0x00, 0x00, 0x00, 0xe0, 0x00, 0x00, 0x00, 0x00,
           0x01, 0x01, 0x01, 0x0f, 0x01, 0x01, 0x01, 0x00,}},
    {0x2c,{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x20, 0x18, 0x00, 0x00, 0x00, 0x00,}},
    {0x2d,{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x02, 0x02, 0x02, 0x02, 0x00, 0x00,}},
    {0x2e,{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00,}},
    {0x2f,{0x00, 0x00, 0x00, 0x80, 0x60, 0x18, 0x00, 0x00,
           0x00, 0x18, 0x06, 0x01, 0x00, 0x00, 0x00, 0x00,}},
    {0x30,{0x00, 0xf0, 0x08, 0x08, 0x08, 0x08, 0xf0, 0x00,
           0x00, 0x0f, 0x10, 0x10, 0x10, 0x10, 0x0f, 0x00,}},
    {0x31,{0x00, 0x00, 0x10, 0x08, 0xf8, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x00,}},
    {0x32,{0x00, 0x10, 0x08, 0x08, 0x08, 0x88, 0x70, 0x00,
           0x00, 0x18, 0x14, 0x12, 0x11, 0x10, 0x10, 0x00,}},
    {0x33,{0x00, 0x10, 0x08, 0x88, 0x88, 0x88, 0x70, 0x00,
           0x00, 0x08, 0x10, 0x10, 0x10, 0x10, 0x0f, 0x00,}},
    {0x34,{0x00, 0x00, 0x80, 0x60, 0x18, 0xf8, 0x00, 0x00,
           0x00, 0x06, 0x05, 0x04, 0x04, 0x1f, 0x04, 0x00,}},
    {0x35,{0x00, 0xf8, 0x88, 0x88, 0x88, 0x88, 0x08, 0x00,
           0x00, 0x08, 0x10, 0x10, 0x10, 0x10, 0x0f, 0x00,}},
    {0x36,{0x00, 0xf0, 0x88, 0x88, 0x88, 0x88, 0x10, 0x00,
           0x00, 0x0f, 0x10, 0x10, 0x10, 0x10, 0x0f, 0x00,}},
    {0x37,{0x00, 0x08, 0x08, 0x08, 0x88, 0x68, 0x18, 0x00,
           0x00, 0x00, 0x18, 0x06, 0x01, 0x00, 0x00, 0x00,}},
    {0x38,{0x00, 0x70, 0x88, 0x88, 0x88, 0x88, 0x70, 0x00,
           0x00, 0x0f, 0x10, 0x10, 0x10, 0x10, 0x0f, 0x00,}},
    {0x39,{0x00, 0xf0, 0x08, 0x08, 0x08, 0x08, 0xf0, 0x00,
           0x00, 0x08, 0x11, 0x11, 0x11, 0x11, 0x0f, 0x00,}},
    {0x3a,{0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00,}},
    {0x3b,{0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x20, 0x18, 0x00, 0x00, 0x00, 0x00,}},
    {0x3c,{0x00, 0x00, 0x80, 0x80, 0x40, 0x40, 0x20, 0x00,
           0x00, 0x01, 0x02, 0x02, 0x04, 0x04, 0x08, 0x00,}},
    {0x3d,{0x00, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x00,
           0x00, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x00,}},
    {0x3e,{0x00, 0x20, 0x40, 0x40, 0x80, 0x80, 0x00, 0x00,
           0x00, 0x08, 0x04, 0x04, 0x02, 0x02, 0x01, 0x00,}},
    {0x3f,{0x00, 0x10, 0x08, 0x08, 0x88, 0x70, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x1b, 0x00, 0x00, 0x00, 0x00,}},
    {0x40,{0xc0, 0x30, 0x88, 0x48, 0x48, 0xc8, 0x10, 0xe0,
           0x07, 0x18, 0x23, 0x24, 0x24, 0x27, 0x14, 0x03,}},
    {0x41,{0x00, 0x00, 0xe0, 0x18, 0x18, 0xe0, 0x00, 0x00,
           0x18, 0x07, 0x02, 0x02, 0x02, 0x02, 0x07, 0x18,}},
    {0x42,{0x00, 0xf8, 0x88, 0x88, 0x88, 0x88, 0x70, 0x00,
           0x00, 0x1f, 0x10, 0x10, 0x10, 0x10, 0x0f, 0x00,}},
    {0x43,{0x00, 0xe0, 0x10, 0x08, 0x08, 0x08, 0x10, 0x00,
           0x00, 0x07, 0x08, 0x10, 0x10, 0x10, 0x08, 0x00,}},
    {0x44,{0x00, 0xf8, 0x08, 0x08, 0x08, 0x10, 0xe0, 0x00,
           0x00, 0x1f, 0x10, 0x10, 0x10, 0x08, 0x07, 0x00,}},
    {0x45,{0x00, 0xf8, 0x88, 0x88, 0x88, 0x88, 0x08, 0x00,
           0x00, 0x1f, 0x10, 0x10, 0x10, 0x10, 0x10, 0x00,}},
    {0x46,{0x00, 0xf8, 0x88, 0x88, 0x88, 0x88, 0x08, 0x00,
           0x00, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,}},
    {0x47,{0x00, 0xe0, 0x10, 0x08, 0x08, 0x08, 0x10, 0x00,
           0x00, 0x07, 0x08, 0x10, 0x11, 0x11, 0x0f, 0x00,}},
    {0x48,{0x00, 0xf8, 0x80, 0x80, 0x80, 0x80, 0xf8, 0x00,
           0x00, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x00,}},
    {0x49,{0x00, 0x00, 0x00, 0xf8, 0x00, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x00,}},
    {0x4a,{0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x00, 0x00,
           0x00, 0x0c, 0x10, 0x10, 0x10, 0x0f, 0x00, 0x00,}},
    {0x4b,{0x00, 0xf8, 0x00, 0x80, 0x40, 0x20, 0x10, 0x08,
           0x00, 0x1f, 0x01, 0x01, 0x02, 0x04, 0x08, 0x10,}},
    {0x4c,{0x00, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
           0x00, 0x1f, 0x10, 0x10, 0x10, 0x10, 0x10, 0x00,}},
    {0x4d,{0xf8, 0x38, 0xc0, 0x00, 0xc0, 0x38, 0xf8, 0x00,
           0x1f, 0x00, 0x07, 0x18, 0x07, 0x00, 0x1f, 0x00,}},
    {0x4e,{0x00, 0xf8, 0x30, 0xc0, 0x00, 0x00, 0xf8, 0x00,
           0x00, 0x1f, 0x00, 0x00, 0x03, 0x0c, 0x1f, 0x00,}},
    {0x4f,{0x00, 0xe0, 0x10, 0x08, 0x08, 0x10, 0xe0, 0x00,
           0x00, 0x07, 0x08, 0x10, 0x10, 0x08, 0x07, 0x00,}},
    {0x50,{0x00, 0xf8, 0x08, 0x08, 0x08, 0x08, 0xf0, 0x00,
           0x00, 0x1f, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00,}},
    {0x51,{0xe0, 0x10, 0x08, 0x08, 0x08, 0x10, 0xe0, 0x00,
           0x07, 0x08, 0x10, 0x10, 0x30, 0x48, 0x47, 0x00,}},
    {0x52,{0x00, 0xf8, 0x08, 0x08, 0x08, 0x88, 0x70, 0x00,
           0x00, 0x1f, 0x01, 0x01, 0x03, 0x0c, 0x10, 0x00,}},
    {0x53,{0x00, 0x70, 0x88, 0x88, 0x08, 0x08, 0x10, 0x00,
           0x00, 0x08, 0x10, 0x10, 0x11, 0x11, 0x0e, 0x00,}},
    {0x54,{0x08, 0x08, 0x08, 0xf8, 0x08, 0x08, 0x08, 0x00,
           0x00, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x00,}},
    {0x55,{0x00, 0xf8, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x00,
           0x00, 0x0f, 0x10, 0x10, 0x10, 0x10, 0x0f, 0x00,}},
    {0x56,{0x18, 0xe0, 0x00, 0x00, 0x00, 0xe0, 0x18, 0x00,
           0x00, 0x00, 0x07, 0x18, 0x07, 0x00, 0x00, 0x00,}},
    {0x57,{0xf8, 0x00, 0xc0, 0x38, 0xc0, 0x00, 0xf8, 0x00,
           0x01, 0x1e, 0x03, 0x00, 0x03, 0x1e, 0x01, 0x00,}},
    {0x58,{0x08, 0x30, 0x40, 0x80, 0x40, 0x30, 0x08, 0x00,
           0x10, 0x0c, 0x02, 0x01, 0x02, 0x0c, 0x10, 0x00,}},
    {0x59,{0x08, 0x30, 0xc0, 0x00, 0xc0, 0x30, 0x08, 0x00,
           0x00, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x00,}},
    {0x5a,{0x00, 0x08, 0x08, 0x08, 0xc8, 0x28, 0x18, 0x00,
           0x00, 0x18, 0x14, 0x13, 0x10, 0x10, 0x10, 0x00,}},
    {0x5b,{0x00, 0x00, 0x00, 0xf8, 0x08, 0x08, 0x00, 0x00,
           0x00, 0x00, 0x00, 0xff, 0x80, 0x80, 0x00, 0x00,}},
    {0x5c,{0x00, 0x18, 0x60, 0x80, 0x00, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x01, 0x06, 0x18, 0x00, 0x00,}},
    {0x5d,{0x00, 0x00, 0x08, 0x08, 0xf8, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x80, 0x80, 0xff, 0x00, 0x00, 0x00,}},
    {0x5e,{0x80, 0x40, 0x30, 0x08, 0x30, 0x40, 0x80, 0x00,
           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,}},
    {0x5f,{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
           0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80,}},
    {0x60,{0x00, 0x00, 0x00, 0x08, 0x10, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,}},
    {0x61,{0x00, 0x80, 0x40, 0x40, 0x40, 0x40, 0x80, 0x00,
           0x00, 0x0c, 0x12, 0x12, 0x12, 0x0a, 0x1f, 0x00,}},
    {0x62,{0x00, 0xf8, 0x80, 0x40, 0x40, 0x40, 0x80, 0x00,
           0x00, 0x1f, 0x08, 0x10, 0x10, 0x10, 0x0f, 0x00,}},
    {0x63,{0x00, 0x80, 0x40, 0x40, 0x40, 0x40, 0x80, 0x00,
           0x00, 0x0f, 0x10, 0x10, 0x10, 0x10, 0x08, 0x00,}},
    {0x64,{0x00, 0x80, 0x40, 0x40, 0x40, 0x80, 0xf8, 0x00,
           0x00, 0x0f, 0x10, 0x10, 0x10, 0x08, 0x1f, 0x00,}},
    {0x65,{0x00, 0x80, 0x40, 0x40, 0x40, 0x40, 0x80, 0x00,
           0x00, 0x0f, 0x12, 0x12, 0x12, 0x12, 0x0b, 0x00,}},
    {0x66,{0x00, 0x00, 0x40, 0xf0, 0x48, 0x48, 0x08, 0x00,
           0x00, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x00,}},
    {0x67,{0x00, 0x80, 0x40, 0x40, 0x40, 0xc0, 0x40, 0x00,
           0x00, 0x6b, 0x94, 0x94, 0x94, 0x93, 0x60, 0x00,}},
    {0x68,{0x00, 0xf8, 0x80, 0x40, 0x40, 0x40, 0x80, 0x00,
           0x00, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x00,}},
    {0x69,{0x00, 0x00, 0x40, 0x40, 0xc8, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x00,}},
    {0x6a,{0x00, 0x00, 0x40, 0x40, 0xc8, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x80, 0x80, 0x7f, 0x00, 0x00, 0x00,}},
    {0x6b,{0x00, 0xf8, 0x00, 0x00, 0x80, 0x40, 0x00, 0x00,
           0x00, 0x1f, 0x02, 0x03, 0x04, 0x08, 0x10, 0x00,}},
    {0x6c,{0x00, 0x00, 0x00, 0xf8, 0x00, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x0f, 0x10, 0x10, 0x00, 0x00,}},
    {0x6d,{0xc0, 0x80, 0x40, 0x80, 0x80, 0x40, 0x80, 0x00,
           0x1f, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x1f, 0x00,}},
    {0x6e,{0x00, 0xc0, 0x80, 0x40, 0x40, 0x40, 0x80, 0x00,
           0x00, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x00,}},
    {0x6f,{0x00, 0x80, 0x40, 0x40, 0x40, 0x40, 0x80, 0x00,
           0x00, 0x0f, 0x10, 0x10, 0x10, 0x10, 0x0f, 0x00,}},
    {0x70,{0x00, 0xc0, 0x80, 0x40, 0x40, 0x40, 0x80, 0x00,
           0x00, 0xff, 0x08, 0x10, 0x10, 0x10, 0x0f, 0x00,}},
    {0x71,{0x00, 0x80, 0x40, 0x40, 0x40, 0x80, 0xc0, 0x00,
           0x00, 0x0f, 0x10, 0x10, 0x10, 0x08, 0xff, 0x00,}},
    {0x72,{0x00, 0x00, 0xc0, 0x80, 0x40, 0x40, 0x00, 0x00,
           0x00, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x00,}},
    {0x73,{0x00, 0x80, 0x40, 0x40, 0x40, 0x80, 0x00, 0x00,
           0x00, 0x09, 0x12, 0x12, 0x12, 0x0c, 0x00, 0x00,}},
    {0x74,{0x00, 0x00, 0x40, 0xf0, 0x40, 0x40, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x0f, 0x10, 0x10, 0x00, 0x00,}},
    {0x75,{0x00, 0xc0, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x00,
           0x00, 0x0f, 0x10, 0x10, 0x10, 0x08, 0x1f, 0x00,}},
    {0x76,{0x00, 0xc0, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x00,
           0x00, 0x00, 0x07, 0x18, 0x18, 0x07, 0x00, 0x00,}},
    {0x77,{0xc0, 0x00, 0x00, 0xc0, 0x00, 0x00, 0xc0, 0x00,
           0x03, 0x1c, 0x07, 0x00, 0x07, 0x1c, 0x03, 0x00,}},
    {0x78,{0x00, 0xc0, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00,
           0x00, 0x18, 0x05, 0x02, 0x05, 0x18, 0x00, 0x00,}},
    {0x79,{0x00, 0xc0, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x00,
           0x00, 0x80, 0x83, 0x6c, 0x18, 0x07, 0x00, 0x00,}},
    {0x7a,{0x00, 0x40, 0x40, 0x40, 0x40, 0xc0, 0x00, 0x00,
           0x00, 0x18, 0x14, 0x12, 0x11, 0x10, 0x00, 0x00,}},
    {0x7b,{0x00, 0x00, 0x00, 0xf0, 0x08, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x02, 0x7d, 0x80, 0x00, 0x00, 0x00,}},
    {0x7c,{0x00, 0x00, 0x00, 0xfc, 0x00, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0x00,}},
    {0x7d,{0x00, 0x00, 0x00, 0x08, 0xf0, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x80, 0x7d, 0x02, 0x00, 0x00,}},
    {0x00,{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,}}
    };

/*---------------------------------------
 * Structure containing the different
 * info needed for the command processing
 *--------------------------------------*/
typedef enum _LcdST7567Instruction
 LcdST7567Instruction;

struct lcd_ST7567_instruction {
    LcdST7567Instruction instruction;
    LcdST7567Instruction type;
    LcdST7567Instruction mode;
};

typedef struct {
    unsigned short int width_of_characters;
    unsigned short int payload_size;
    unsigned short int row;
    unsigned short int reserved;
    unsigned char *payload;
} Panel2lineDataType;

/*------------------------------------------
 * Structure for parsing of the instruction
 *-----------------------------------------*/

static struct lcd_ST7567_instruction ParseInstruction2Line(LcdST7567Instruction setting)
{
    /*Declare a variable of type structure */
    struct lcd_ST7567_instruction InstructionType;

    /*Get the instruction to write to controller */
    InstructionType.instruction = (setting & TWO_LINE_INSTRUCTION_BITS_MASK);
    /* Get the type of instruction - Basic or Extended */
    InstructionType.type = (setting & TWO_LINE_INSTRUCTION_SET_MASK);

    /*Return the instruction and command type as a structure */
    return (InstructionType);
}

/*------------------------------------------
 * Structure for the commmand processing
 *-----------------------------------------*/

static struct lcd_ST7567_instruction
LcdGetInstructionType2Line(LcdST7567Instruction instruction)
{
    /* Declare a variable of type structure */
    struct lcd_ST7567_instruction InstructionType;

    /*Get the parsed intruction type */
    InstructionType = ParseInstruction2Line(instruction);

    /* Instruction: BASIC COMMAND */
    /* Check if the instruction is a BASIC instruction */
    if (InstructionType.type & TWO_LINE_INSTRUCTION_BASIC) {
        /* Defaults to command mode */
        InstructionType.mode = TWO_LINE_COMMAND_MODE;
        InstructionType.type &= TWO_LINE_INSTRUCTION_BASIC;

    } else {
        /* If the instruction is a DATA instruction */
        if (InstructionType.type & TWO_LINE_INSTRUCTION_DATA) {
            InstructionType.mode = TWO_LINE_DATA_MODE;
            InstructionType.type &= TWO_LINE_INSTRUCTION_DATA;
        } else
            InstructionType.mode = TWO_LINE_COMMAND_MODE;
    }

    /* Return the instruction info as a structure  */
    return (InstructionType);

}

/*-----------------------------------------------------------------------------
 *
 *  Function: LcdSetST7567Controller
 *
 *  Description: Pre-processes the instruction before setting the ST7567
 *               LCD controller.
 *
 *  Input:       data - instruction/setting of type enum
 *
 *  Output:      NONE
 *
 *-----------------------------------------------------------------------------*/

static void LcdSetST7567Controller(LcdST7567Instruction data)
{
    /* Declare a variable of type structure */
    struct lcd_ST7567_instruction ParsedInstruction;

    /* Get the pre-processed instruction */
    ParsedInstruction = LcdGetInstructionType2Line(data);

    /* Write to the LCD controller */
    PanelWriteInstructionST7567(ParsedInstruction.mode,
                    ParsedInstruction.instruction);
}

/*------------------------------------------------------------------------------
 * Name:    panel_2line_lcd_reset()
 *
 * Description: Resets the 2-line LCD controller
 *
 * Input:       None
 *
 * Output:      None
 *
 * Notes:       RST_LCD - PB7
 *-----------------------------------------------------------------------------*/

static void panel_2Line_lcd_reset(void)
{
    unsigned char
     cmd, dat;

    /* Read first the GPIOB output status register */
    cmd = MCU_GPIOB_OUT_REG;
    panel_i2c_msg(&cmd, 1, NULL, 0, &dat, 1);

    /* Mask out the desired output port status
       Set to LOW */
    dat &= 0x7F;

    DELAY_FOR_PANEL();
    panel_i2c_msg(&cmd, 1, &dat, 1, NULL, 0);

    DELAY_FOR_PANEL();

    panel_i2c_msg(&cmd, 1, NULL, 0, &dat, 1);

    dat |= 0x80;

    DELAY_FOR_PANEL();
    panel_i2c_msg(&cmd, 1, &dat, 1, NULL, 0);

}

/*------------------------------------------------------------------------------
 * Name:    panel_2LineClearLCD
 *
 * Description:
 *
 * Input:       None
 *
 * Output:      None
 *
 *-----------------------------------------------------------------------------*/
static void panel_2LineClearLCD(unsigned char page)
{
    unsigned char l;
    unsigned char data[2], cmd;

    PRINTF("Executing the Lcd clear routine\n");

    cmd = MCU_SPI_SR_REG;
    data[0] = 0x68;     //re-enable NSS in case it was previously disabled
    panel_i2c_msg(&cmd, 1, data, 1, NULL, 0);

    for (l = 0; l < page; l++) {
        if ((l == 0) | (l == 8))
            LcdSetST7567Controller(TWO_LINE_PAGE_ADDR | 0x00);
        else
            LcdSetST7567Controller(TWO_LINE_PAGE_ADDR | l);

        LcdSetST7567Controller(TWO_LINE_COLLADDR_MSB | 0x00);
        LcdSetST7567Controller(TWO_LINE_COLLADDR_LSB | 0x00);

        //Set the A0 to high for data instruction
        cmd = MCU_GPIOB_OUT_REG;
        data[0] = 0xFF;
        panel_i2c_msg(&cmd, 1, data, 1, NULL, 0);

        //automatic clearing of the display RAM - MSB count
        cmd = MCU_SPI_ZERO_MSB;
        data[0] = 0x03;

        //automatic clearing of the display RAM - LSB count
        data[1] = 0xFF;
        panel_i2c_msg(&cmd, 1, data, 2, NULL, 0);
    }
    cmd = MCU_SPI_SR_REG;
    data[0] = 0xCF;     // multibyte: disable; nss: disable; spi: enable
    panel_i2c_msg(&cmd, 1, data, 1, NULL, 0);

    PRINTF("Done Executing the Lcd clear routine\n");
    //Clear buffer
    memset(DispBuff, 0, sizeof(DispBuff));
    return;

}


void panel_ClearAllLCD(void)
{
    panel_2LineClearLCD(LCD_2LINE_ALLPAGES);
}


/*------------------------------------------------------------------------------
 * Name:    panel_Init2LineLcd
 *
 * Description:
 *
 * Input:       None
 *
 * Output:      None
 *
 * Notes:
 *-----------------------------------------------------------------------------*/
static void panel_Init2LineLcd(void)
{
    unsigned char rr, ev;

    PRINTF("panel_Init2LineLcd\n");
    /* Reset the 2-line LCD */
    /*
       1) LCD controller initialization
     */
    panel_2Line_lcd_reset();
    DELAY_FOR_PANEL();
    LcdSetST7567Controller((TWO_LINE_POWER_CONTROL |
                TWO_LINE_POWER_CONTROL_BIT_VF |
                TWO_LINE_POWER_CONTROL_BIT_VR |
                TWO_LINE_POWER_CONTROL_BIT_VB));

    /*
       2) Set Bias Select to 1/6 - BS = 1
     */

    DELAY_FOR_PANEL();
    LcdSetST7567Controller(TWO_LINE_BIAS_SELECT
                   /* | TWO_LINE_BIAS_SELECT_BS */ );

    /*
       3) Select the BOOSTER to BL = 1 / 5x
     */

    DELAY_FOR_PANEL();

    LcdSetST7567Controller(TWO_LINE_SET_BOOSTER_INST1); /*First Instruction */

    LcdSetST7567Controller((TWO_LINE_SET_BOOSTER |  /*BL = 1 / 5x Boost Level */
                TWO_LINE_SET_BOOSTER_BL));

    /*
       3.) Set to Inverse Display - (white -> Black , Black -> white)
     */
    DELAY_FOR_PANEL();
    LcdSetST7567Controller((TWO_LINE_INV_DISP));    //|

    /*
       4) Set the display mode: Reverse or Normal
     */

#if LCD_2LINE_DISPLAY_INVERSED  // Upside down display

    /* Set the Initial COM register to 36 */

    DELAY_FOR_PANEL();
    /* Set the SEG to 1: Reverse Direction
       SEG output reverse 132 --> 0 */

    LcdSetST7567Controller((TWO_LINE_SEG_DIRECTION | TWO_LINE_SEG_DIRECTION_MX));

#else               // Normal display
    /* Set the COM to 1: Reverse Direction
       COM output reverse 64 --> 0 */

    DELAY_FOR_PANEL();
    LcdSetST7567Controller((TWO_LINE_COM_DIR | TWO_LINE_COM_DIR_MY));
#endif

    /* RR = Regulation Ratio */
    DELAY_FOR_PANEL();
    rr = panel_info.apa_rr & 0x07;
    LcdSetST7567Controller(TWO_LINE_REGULATION_RATIO | rr);

    /* EV = Electric Volume */
    DELAY_FOR_PANEL();
    ev = panel_info.apa_ev & 0x3f;
    LcdSetST7567Controller(TWO_LINE_SET_EV_INST1);
    LcdSetST7567Controller(TWO_LINE_SET_EV | ev);

    /*
       8) Set the Power Control = 0x2F (0010_1111)
       Internal voltage follower circuit is ON
       Internal voltage regulator circuit is ON
       Internal voltage converter circuit is ON
     */

    DELAY_FOR_PANEL();
    LcdSetST7567Controller((TWO_LINE_POWER_CONTROL |
                TWO_LINE_POWER_CONTROL_BIT_VF |
                TWO_LINE_POWER_CONTROL_BIT_VR |
                TWO_LINE_POWER_CONTROL_BIT_VB));

    /*
       9) Turn the Display ON = 0xAF (1010_1111)
     */

    DELAY_FOR_PANEL();
    panel_2LineClearLCD(8);
    DELAY_FOR_PANEL();
    LcdSetST7567Controller((TWO_LINE_DISPLAY_CONTROL |
                TWO_LINE_DISPLAY_CONTROL_BIT_D));
}


/*-----------------------------------------------------------------------------
 *
 * Name:        panel_Init2LineGPIOs
 *
 * Description: Initializes the microcontroller's GPIOs
 *
 * Input:       None
 *
 * Output:      None
 *
 * Notes:
 *---------------------------------------------------------------------------*/

static int panel_Init2LineGPIOs(void)
{
    unsigned char cmd, data;
    int rc;
    udelay(1000);

    /* Init the GPIOs for the 2-Line and 4-Line op-panel */

    /* PB0 should be set as an Interrupt Output
       PB0 -> INTN

       PB1 should be set as a PWM output
       PB1 -> PWM (for Backlight)
     */

    PRINTF("Initializing the GPIOs\n");
    cmd = MCU_ALT_FUNC_CONFIG_REG;
    rc = panel_i2c_msg(&cmd, 1, NULL, 0, &data, 1);

    if (rc == -1) {
        return rc;
    }

    data |= 0x03;  //ALT_FUNC_CONFIG_BIT0 | ALT_FUNC_CONFIG_BIT1;
    rc = panel_i2c_msg(&cmd, 1, &data, 1, NULL, 0);

    /* PB2 should be set as output port
       PB2 -> CS_LCD
       PB6 should be set as output port
       PB6 -> A0

       PB7 should be set as output port
       PB7 -> RST_LCD
     */

    /* Read the GPIOB_OER register */
    cmd = MCU_GPIOB_OER_REG;
    rc = panel_i2c_msg(&cmd, 1, NULL, 0, &data, 1);
    if (rc == -1) {
        return rc;
    }

    data = ~data;        //(~(GPIOB_OER_BIT2) | ~(GPIOB_OER_BIT6) | ~(GPIOB_OER_BIT7))
    data |= 0xC2;
    data = ~data;
    rc = panel_i2c_msg(&cmd, 1, &data, 1, NULL, 0);

    return rc;
}

void display_2line_LCD(unsigned char *p_buff)
{
    unsigned char pageaddr;
    unsigned char temp;
    unsigned char cmd, dat;

    cmd = MCU_SPI_SR_REG;
    dat = 0x68;
    panel_i2c_msg(&cmd, 1, &dat, 1, NULL, 0);

    pageaddr = 0;
    temp = 0;
    while (pageaddr < 4) {
        /* 1) Send the page address */
        LcdSetST7567Controller(TWO_LINE_PAGE_ADDR | pageaddr);

        /* 2) Send column address */
        LcdSetST7567Controller(TWO_LINE_COLLADDR_MSB | temp);
        LcdSetST7567Controller(TWO_LINE_COLLADDR_LSB | temp);

        /* 3) Start writing  */
        cmd = MCU_GPIOB_OUT_REG;
        panel_i2c_msg(&cmd, 1, NULL, 0, &dat, 1);
        dat |= 0xFC;
        panel_i2c_msg(&cmd, 1, &dat, 1, NULL, 0);
        cmd = MCU_SPI_TDR_REG;
        panel_i2c_msg(&cmd, 1, p_buff, 128, NULL, 0);
        pageaddr++;

        p_buff += 128;
    }

    cmd = MCU_SPI_SR_REG;
    dat = 0xCF;
    panel_i2c_msg(&cmd, 1, &dat, 1, NULL, 0);
}

/*------------------------------------------------------------------------------
 * Name:    panel_2LineLcdWrite
 *
 * Description: This routine sends data to the front-panel but uses no RAM.
 *
 * Input:   row - row address
 *          col - column address
 *          msg - string to display
 *
 * Output:  0 - SUCCESS
 *          < 0 - FAIL
 *
 * Note: This function is only used for 2-Line and 4-Line LCDs
 *-----------------------------------------------------------------------------*/
static void panel_2LineLcdWrite(int row, int col, const char *msg)
{
    const char *dat;
    int font_index;
    unsigned char *pbuff;
    int col_offset;
    int row_offset;

    dat = msg;
    pbuff = DispBuff;
    col_offset = col * 8; // 8 pixels wide
    row_offset = row * 256; // 256 pixels = 1 row
    while ((*dat) != '\0') {
        //search font table
        font_index = 0;
        while (font_twoline[font_index].key != 0x00) {
            if (font_twoline[font_index].key == (*dat)) {
                memcpy(&pbuff[col_offset + row_offset], font_twoline[font_index].value, 8);
                memcpy(&pbuff[col_offset + row_offset + 128], &font_twoline[font_index].value[8], 8);
                break;
            }
            font_index++;
        }
        col_offset += 8;
        dat++;
    }

    display_2line_LCD(DispBuff);
}


static char PeriodBuff[17];
void panel_PanelScrollingDot(void)
{
    /*
     * Don't go off end of buffer.  -1 accounts for NULL termination
     */

    if (strlen(PeriodBuff) == sizeof(PeriodBuff) - 1) {
        panel_2LineClearLCD(4);
        memset(PeriodBuff, 0, sizeof(PeriodBuff));
    }

    strcat(PeriodBuff, ".");
    panel_2LineLcdWrite(0, 0, PeriodBuff);
}


typedef enum {
    LED_STATE_ON,
    LED_STATE_OFF,
    LED_STATE_BLINK,
} LedState_e;


void panel_LcdSetLed(unsigned char state)
{
    unsigned char dat, cmd;

    /* Read the GPIOD_OER register */
    cmd = MCU_GPIOD_OER_REG;
    panel_i2c_msg(&cmd, 1, NULL, 0, &dat, 1);
    dat &= 0x3F;        // set as output port PD7 & PD6
    /* Write to the GPIOD_OER register */
    panel_i2c_msg(&cmd, 1, &dat, 1, NULL, 0);

    cmd = MCU_GPIOD_OUT_REG;
    panel_i2c_msg(&cmd, 1, NULL, 0, &dat, 1);

    if (state)
        /* Turn green led ON */
        dat |= 0x40;
    else
        /* Turn green led OFF */
        dat &= 0x00;

    panel_i2c_msg(&cmd, 1, &dat, 1, NULL, 0);
}

static void panel_InitSpi(void)
{
    unsigned char cmd, data[2];
    int rc, retries;

    data[0] = 0x1F; // Slow down the clock to 125KHz. We're seeing glitches in the display.
    data[1] = 0x78;

    /* Write to SPI_CLK CONFIG and SPI_SR register
       The register address auto-increments */
    cmd = MCU_SPI_CLK_CONFIG_REG;
    retries = I2C_NUM_RETRIES;
    while (retries) {
        rc = panel_i2c_msg(&cmd, 1, data, 2, NULL, 0);
        if (rc == -1) {
            retries--;
        } else {
            break;
        }
    }
}

int read_eeprom(void) {
    unsigned char cmd, dat;
    int rc = 0;

    cmd = MCU_EEPROM_ADDR_REG;
    dat = 0x00;
    panel_i2c_msg(&cmd, 1, &dat, 1, NULL, 0);

    memset(&panel_info, 0xff, sizeof(panel_info));
    cmd = MCU_EEPROM_DATA_REG;
	rc = mv_mmp_panel_i2c_read(&i2c_info, cmd, 1, (unsigned char *)&panel_info, sizeof(panel_info));
    if (rc)
        printf("mv_mmp_panel_i2c_read failed\n");

    if (panel_info.version_major == 0xff && panel_info.version_minor == 0xff) {
        rc = 1; /* empty EEPROM */
    }

    return(rc);
}

/*-----------------------------------------------------------------------------
 *
 * Description: This routine performs the necessary action to reset the panel.
 *              1) Initialize the key scanning
 *              2) Initialize the SPI for LCD control
 *              3) Initializes the LCD controller
 *              4) Initialize PWM for backlight
 *              5) Initialize GPIOs for LEDs
 *
 * Input:       None
 *
 * Output:      None
 *
 * Notes:
 *---------------------------------------------------------------------------*/
static int panel_2line_ResetAll(void)
{
    int rc, empty;
    PRINTF("Panel reset All.\n");

    /* NOTE: PLEASE DON'T RE-ARRANGE THE ORDER!!!
       rlc: Need to revisit this implementation... */

    /* 1) Reset the ATtiny48 mcu */
    rc = mv_mmp_panel_reset_mcu(&i2c_info);;
    if (rc != 0)
        return 1;

    empty = read_eeprom();
    if (empty == 1) {
        panel_info.apa_type = APATYPE_TIANMA;
        panel_info.apa_rr = 0x03; /* 0b011 */
        panel_info.apa_ev = 0x12; /* 0b010010 */
    }

    /* 2) Initializes the microcontroller's GPIOs */
    rc = panel_Init2LineGPIOs();
    if (rc != 0)
        return 1;

    /* 3) Initializes the GPIOs fof the LED controls and set the LED */
    panel_LcdSetLed(1);

    /* 4) Initializes the SPI for the LCD control */

    panel_InitSpi();

    /* 6) Initializes the LCD controller */
    panel_Init2LineLcd();

    return rc;
}


static void lcd_report_status(int type, va_list ap)
{
    int data1, data2;

    switch (type) {
        /*
         * PANEL STATUS MESSAGES:
         */
    case MV_MMP_PANEL_BLANK:
        panel_ClearAllLCD();
        memset(PeriodBuff, 0, sizeof(PeriodBuff));
        break;

    case MV_MMP_PANEL_DRAMTEST_DONE:
        break;

    case MV_MMP_PANEL_DRAMTEST_START:
        panel_LcdSetLed(LED_STATE_BLINK);
        panel_2LineClearLCD(4); // clear lines 1 & 2
        break;
        // Same routine for both?
    case MV_MMP_PANEL_MEMSIZE_CPUSPEED:
    case MV_MMP_PANEL_DRAMTEST_DIAG:
        {

            char buf[20];

            data1 = va_arg(ap, int);
            data2 = va_arg(ap, int);

            if (data2)  /* cpuspeed too */ {
                sprintf(buf, "%3d MB %3d Mhz", data1, data2);
            } else    /* only memsize */ {
                sprintf(buf, "%4d MB", data1);
            }

            panel_2LineClearLCD(4);
            panel_2LineLcdWrite(0, 0, buf);
        }
        is_scrolling_enabled = 0;
        break;

    case MV_MMP_PANEL_SCROLLING_DOT:
        if (is_scrolling_enabled == 1) {
            panel_PanelScrollingDot();
        }
        break;

    case MV_MMP_PANEL_EARLY_MEMORY_TEST:
        break;

        /*
         * PANEL ERROR MESSAGES:
         */
    case MV_MMP_PANEL_ASIC1_FAIL:
        panel_2LineClearLCD(4); // clear lines 1 & 2
        panel_2LineLcdWrite(0, 0, "957 Service");
        panel_2LineLcdWrite(1, 0, "    System Board");
        break;

    case MV_MMP_PANEL_BAD_MBRSIZE:
        panel_2LineClearLCD(4); // clear lines 1 & 2
        panel_2LineLcdWrite(0, 0, "900 Service");
        panel_2LineLcdWrite(1, 0, "    MBR Size");
        break;

    case MV_MMP_PANEL_NO_MBR:
        panel_2LineClearLCD(4); // clear lines 1 & 2
        panel_2LineLcdWrite(0, 0, "900 Service");
        panel_2LineLcdWrite(1, 0, "    No MBR");
        break;

    case MV_MMP_PANEL_DECOMPRESS_ERROR:
        panel_2LineClearLCD(4); // clear lines 1 & 2
        panel_2LineLcdWrite(0, 0, "Decompress");
        panel_2LineLcdWrite(1, 0, "Code Error");
        break;

    case MV_MMP_PANEL_DECOMPRESS_FAIL:
        panel_2LineClearLCD(4); // clear lines 1 & 2
        panel_2LineLcdWrite(0, 1, "Decompression");
        panel_2LineLcdWrite(1, 4, "Failure");
        break;

    case MV_MMP_PANEL_DRAMTEST_FAIL:
        {
            char
             line1[17];

            data1 = va_arg(ap, int);
            panel_2LineClearLCD(4); // clear lines 1 & 2
            panel_2LineLcdWrite(0, 0, "Failure");
            sprintf(line1, "Addr: 0x%x", data1);
            panel_2LineLcdWrite(1, 0, line1);
        }
        break;

    case MV_MMP_PANEL_ECC_ERROR:   /* Found Non Correctable Error */
        panel_2LineClearLCD(4); // clear lines 1 & 2
        panel_2LineLcdWrite(0, 0, "958 Service");
        panel_2LineLcdWrite(1, 0, "    Multibit ECC");
        break;

    case MV_MMP_PANEL_ECC_FAIL:
        panel_2LineLcdWrite(0, 0, "900 Service");
        panel_2LineLcdWrite(1, 0, "ECC Failure");
        break;

    case MV_MMP_PANEL_FLASH_ERROR:
        panel_2LineClearLCD(4); // clear lines 1 & 2
        panel_2LineLcdWrite(0, 0, "Flash Error");
        panel_2LineLcdWrite(1, 0, "No empty blocks");
        break;

    case MV_MMP_PANEL_MEMORY_ERROR:
        {
            char buf[30];
            data1 = va_arg(ap, int);
            panel_2LineClearLCD(4); // clear lines 1 & 2
            sprintf(buf, "96%d Service", data1 >> 8);
            panel_2LineLcdWrite(0, 0, buf);

            sprintf(buf, "Memory Error - %c", data1);
            panel_2LineLcdWrite(1, 0, buf); /* insert correct subcode */
        }
        break;

    case MV_MMP_PANEL_NO_DRAM:
        panel_2LineClearLCD(4); // clear lines 1 & 2
        panel_2LineLcdWrite(0, 0, "No DRAM");
        panel_2LineLcdWrite(1, 0, "Installed");
        break;

    case MV_MMP_PANEL_NO_NVRAM:
        panel_2LineClearLCD(4); // clear lines 1 & 2
        panel_2LineLcdWrite(0, 4, "No Nvram");
        panel_2LineLcdWrite(1, 7, "Found");
        break;

    case MV_MMP_PANEL_ROMCRC_ERROR:
        {
            char
             errstr[16];

            data1 = va_arg(ap, int);
            sprintf(errstr, "Code CRC P %d", data1);

            panel_2LineClearLCD(4); // clear lines 1 & 2
            panel_2LineLcdWrite(0, 0, "955 Service");
            panel_2LineLcdWrite(1, 0, errstr);
        }
        break;

    case MV_MMP_PANEL_EXCESSIVE_MEMORY_ERROR:
        {
            char
             errstr[24];

            data1 = va_arg(ap, int);
            sprintf(errstr, "Memory in slot %d", data1);

            panel_2LineClearLCD(4); // clear lines 1 & 2
            panel_2LineLcdWrite(0, 0, "55 Unsupported");
            panel_2LineLcdWrite(1, 0, errstr);

        }
        break;

    case MV_MMP_PANEL_UNSUPPORTED_FIRMWARE:
        panel_2LineClearLCD(4); // clear lines 1 & 2
        panel_2LineLcdWrite(0, 0, "40 Unsupported");
        panel_2LineLcdWrite(1, 0, "   Firmware Card");
        break;

    case MV_MMP_PANEL_UNSUPPORTED_FLASHCARD:
        {
            char line1[16];

            data1 = va_arg(ap, int);
            panel_2LineClearLCD(4); // clear lines 1 & 2
            panel_2LineLcdWrite(0, 0, "55 Unsupported");
            sprintf(line1, "Flash in Slot %d", data1);
            panel_2LineLcdWrite(1, 0, line1);
        }
        break;

    case MV_MMP_PANEL_WATCHDOG_ERROR:
        panel_2LineClearLCD(4); // clear lines 1 & 2
        panel_2LineLcdWrite(0, 1, "System Timeout");
        break;

    case MV_MMP_PANEL_FLS_STR:
        {
            char line[17];
            char *str = va_arg(ap, char *);
            int i;
            panel_2LineClearLCD(4); // clear lines 1 & 2
            memset(line, 0x20,16);
            line[16] = '\0';
            for(i=0;i<16;i++)
            {
               line[i]=str[i];
               if(str[i]=='\0')
               {
                  panel_2LineLcdWrite(0, 0, line);
                  return;
               }
               if(str[i]=='\n')
               {
                  panel_2LineLcdWrite(0, 0, line);
                  break;
               }
            }
            panel_2LineLcdWrite(0, 0, line);
            memset(line, 0x20,16);
            line[16] = '\0';
            for(i=16;i<32;i++)
            {
               line[i-16]=str[i];
               if(str[i]=='\0')
               {
                  panel_2LineLcdWrite(1, 0, line);
                  return;
               }
               if(str[i]=='\n')
               {
                  panel_2LineLcdWrite(1, 0, line);
                  break;
               }
            }
            panel_2LineLcdWrite(1, 0, line);
        }
        break;

    case MV_MMP_PANEL_ENGINELESS_STRING:
        panel_2LineClearLCD(2); // clear lines 1 & 2
        panel_2LineLcdWrite(0, 0, "Engineless Mode");
        break;
    case MV_MMP_PANEL_VERSIONS:
        break;

    default:
        panel_2LineClearLCD(4); // clear lines 1 & 2
        panel_2LineLcdWrite(0, 0, "900 Service");
        panel_2LineLcdWrite(1, 0, "Fatal Boot err");
        break;
    }
}

static int panel_type = -1;

void mv_mmp_panel_2x16v1_status(MV_MMP_PANEL_STATUS msg, va_list ap)
{
    if (panel_type == MV_MMP_PANEL_TYPE_ATMEL_LED) {
        PRINTF("LED Panel report status\n");
        led_report_status(msg, ap);
    } else if (panel_type == MV_MMP_PANEL_TYPE_ATMEL_2LINE) {
        PRINTF("2 Line LCD Panel report status\n");
        lcd_report_status(msg, ap);
    }
}

static unsigned int read_led_panel_buttons_only(unsigned int porkeys)
{
   const key_map_t *map = PORKeyMap_led;

   PRINTF("Reading LED POR Keys.......................\n");

   return mv_mmp_panel_read_buttons(&i2c_info, map, porkeys, 1);
}

static unsigned int read_led_panel_buttons(void)
{
   unsigned int porkeys;
   unsigned char cover_d;
   unsigned char bin_full;
   const key_map_t *map = PORKeyMap_led;

   PRINTF("Reading LED POR Keys.......................\n");

   porkeys = mv_mmp_panel_read_buttons(&i2c_info, map, 0, 0);

   // cover detect is BLOCKE(1). See card02.c
   cover_d = gpio_sample(COVER_DETECT);
   // The bin_full gpio is low when the bin is full (latch is up)
   bin_full = gpio_sample(BIN_FULL);

   if ((porkeys == 3) && (cover_d == 0) && (bin_full == 1)) {
        porkeys = ((1 << 0) | (1 << 1) | (1 << 3) | (1 << 5));
    } else if ((porkeys == 3) && (cover_d == 1) && (bin_full == 1)) {
        porkeys = ((1 << 5) | (1 << 1) | (1 << 3));
    } else if((porkeys == 1) && (cover_d == 0) && (bin_full == 1)) {
        porkeys = ((1 << 1) | (1 << 2));
    } else if((porkeys == 2) && (cover_d == 0) && (bin_full == 1)) {
        porkeys = ((1 << 2) | (1 << 3));
    } else if ((porkeys == 1) && (cover_d == 1) && (bin_full == 0)) {
        porkeys = (1 << 5);
    } else if ((porkeys == 1) && (cover_d == 1) && (bin_full == 1)) {
        porkeys = ((1 << 5)  | (1 << 2));
    } else if((porkeys == 2) && (cover_d == 1) && (bin_full == 1)) {
        porkeys = (1 << 4);
    } else {
        porkeys = 0;
    }

   return porkeys;
}

static unsigned int read_2line_panel_buttons(void)
{
   const key_map_t *map = PORKeyMap_2line;
   
   return mv_mmp_panel_read_buttons(&i2c_info, map, 0, 0);
}

int panel_lb_led_keyscan_init(void)
{
   int rc = 0;
   unsigned char msg[2];

   rc = mv_mmp_panel_keyscan_init(&i2c_info, KEY_ROWS_LED, KEY_COLS_LED);
   if(rc)
      return rc;

   msg[0] = MCU_ALT_FUNC_CONFIG_REG;
   msg[1] = 0x01;
   rc = mv_mmp_panel_i2c_write(&i2c_info, msg, 2);
   if(rc)
      return rc;

   msg[0] = MCU_GPIOD_OER_REG;
   msg[1] = 0x03; //Enable PORT PD2-PD7 Output enable register;
   rc = mv_mmp_panel_i2c_write(&i2c_info, msg, 2);
   if(rc)
      return rc;

   msg[0] = MCU_GPIOD_OUT_REG;
   msg[1] = 0xBC;
   rc = mv_mmp_panel_i2c_write(&i2c_info, msg, 2);

   return rc;
}

int panel_lb_2line_keyscan_init(void)
{
    int rc = 0;
    unsigned char msg[2];

    rc = mv_mmp_panel_keyscan_init(&i2c_info, KEY_ROWS_2LINE, KEY_COLS_2LINE);
    if (rc)
        return rc;

    msg[0] = MCU_ALT_FUNC_CONFIG_REG;
    msg[1] = 0x03; //ENABLE_ALL_ALTERNATE_FUNCTION_REGISTER;
    rc = mv_mmp_panel_i2c_write(&i2c_info, msg, 2);
    return rc;
}

int mv_mmp_panel_2x16v1_probe(void)
{
    int rc;
    char porKeys[11];

    MV_MMPPANELDBG("%s:%d\n", __func__, __LINE__);
  
    /* First, try initializing it as a 2line panel. */
    i2c_info.dev_addr = I2CADDR_PANEL_2L;
    rc = mv_mmp_panel_i2c_probe(&i2c_info);
    if (!rc) {
        rc = panel_2line_ResetAll();
        is_scrolling_enabled = 1;
        if(!rc) {
            rc = panel_lb_2line_keyscan_init();
            if(!rc) {
                panel_type = MV_MMP_PANEL_TYPE_ATMEL_2LINE;
                sprintf(porKeys, "0x%08x", (unsigned int)read_2line_panel_buttons());
                setenv("PORKEYS", porKeys);
                printf("2line panel detected\n");
            }
        }
    }

    if(rc) {
        unsigned char msg[2];

        /* If 2line init failed, try initializing it as an LED panel */

        i2c_info.dev_addr = I2CADDR_PANEL_LED;
        rc = mv_mmp_panel_i2c_probe(&i2c_info);
        if(rc)
            goto done;
        rc = mv_mmp_panel_reset_mcu(&i2c_info);
        if(rc)
            goto done;
        rc = panel_lb_led_keyscan_init();
        if(rc)
            goto done;
        PRINTF("Enabling boot led...");
        rc = mv_mmp_panel_i2c_read(&i2c_info, MCU_ALT_FUNC_CONFIG_REG,1, &msg[1], 1);
        if(rc)
            goto done;
        msg[0] = MCU_ALT_FUNC_CONFIG_REG;
        msg[1] |= 0x04;
        rc = mv_mmp_panel_i2c_write(&i2c_info, msg, 2);
        if(rc)
            goto done;
        panel_type = MV_MMP_PANEL_TYPE_ATMEL_LED;
        sprintf(porKeys, "0x%08x", (unsigned int)read_led_panel_buttons());
        setenv("PORKEYS", porKeys);
        printf("LED panel detected.\n");

        DispBuff = malloc(640);
    }
    
done:
    if(rc) {
        MV_MMPPANELDBG("%s:%d communication error or no response\n", __func__, __LINE__);
        return 1;
    }
    else {
        return 0;
    }
}

unsigned long long mv_mmp_panel_2x16v1_buttons(void)
{
    if (panel_type == MV_MMP_PANEL_TYPE_ATMEL_LED)
        return (unsigned long long)read_led_panel_buttons();
    else if (panel_type == MV_MMP_PANEL_TYPE_ATMEL_2LINE)
        return (unsigned long long)read_2line_panel_buttons();
    else
        return 0;
}

int mv_mmp_panel_2x16v1_type(void)
{
    if (panel_type != -1)
        return panel_type;
    return 0;
}


