#include <common.h>
//#include <asm/io.h>
#include <asm/arch/gpio.h>
#include "mv61fb.h"
#include <lcd.h>
#include <lxk_panel.h>
#include "lxk_eeprom_info.h"
#include "lcd_smpn_TM024HDH49.h"
#include <asm/arch/board_priv.h>
#include "lxk_panel_common.h"

#define REF_RATE_HZ (60)

#if 0
#define TARGET_FREQ_HZ_SLOW	(30000000)
#define TARGET_FREQ_HZ_FAST	(40000000)
#else
#define TARGET_FREQ_HZ_SLOW	(32400000)
#define TARGET_FREQ_HZ_FAST	(43200000)
#endif

#define MAX_ERRS_REPORTED 16
static int smpn_console_active = 0;
static int smpn_console_busy = 0;
static int smpn_err_count = 0;

#ifdef __LXK_VGA_MV61FB_DEBUG__
#if 1
#define SMPNPRINTF(...) printf(__VA_ARGS__)
#else
#define SMPNPRINTF(...)
#endif

/* These will be disabled after init. */
#define SMPNPRINTFINIT(...) if(!smpn_console_active) printf(__VA_ARGS__)

#else
#define SMPNPRINTF(...)
#define SMPNPRINTFINIT(...)
#endif

/* report errors to serial console only, though smpn_update will prevent recursion */
#define SMPNPRINTFERR(...) if(smpn_err_count <= MAX_ERRS_REPORTED) {  \
				serial_printf(__VA_ARGS__); smpn_err_count++; }

#define IGNORE_FIFO_TIMEOUT

struct dbg_state {
	uint32_t 
		dbg_slvtop,
		dbg_txctrl,
		dbg_gratop,
		dbg_muxtop,
		dbg_isa;
};

static struct dbg_state dbg_init = { 0, 0, 0, 0, 0};
static struct dbg_state dbg_good = { 0, 0, 0, 0, 0};
static struct dbg_state dbg_idle = { 0, 0, 0, 0, 0};
static struct dbg_state dbg_busy = { 0, 0, 0, 0, 0};
static struct dbg_state dbg_done = { 0, 0, 0, 0, 0};
static struct dbg_state dbg_tout = { 0, 0, 0, 0, 0};

static void smpn_get_state(void *reg_base, struct dbg_state *hwstate)
{
	uint32_t tmp;
	uint32_t sepxlcnt_state;
	uint32_t sepxlcnt_masked_state;
	
	/* save initial select state and pre-mask it */
	sepxlcnt_state = readl(reg_base + LCD_PN_SEPXLCNT);
	sepxlcnt_masked_state = sepxlcnt_state &= ~(CFG_DBG_RD_INDEX_MASK);
	
	tmp = sepxlcnt_masked_state | CFG_DBG_RD_INDEX(DBG_SLVTOP_SEL);
	writel(tmp, (reg_base + LCD_PN_SEPXLCNT));
	dmb();
	hwstate->dbg_slvtop = readl(reg_base + LCD_SLV_DBG);
	
	tmp = sepxlcnt_masked_state | CFG_DBG_RD_INDEX(DBG_TXCTRL_SEL);
	writel(tmp, (reg_base + LCD_PN_SEPXLCNT));
	dmb();
	hwstate->dbg_txctrl = readl(reg_base + LCD_SLV_DBG);
	
	tmp = sepxlcnt_masked_state | CFG_DBG_RD_INDEX(DBG_GRATOP_SEL);
	writel(tmp, (reg_base + LCD_PN_SEPXLCNT));
	dmb();
	hwstate->dbg_gratop = readl(reg_base + LCD_SLV_DBG);

	tmp = sepxlcnt_masked_state | CFG_DBG_RD_INDEX(DBG_MUXTOP_SEL);
	writel(tmp, (reg_base + LCD_PN_SEPXLCNT));
	dmb();
	hwstate->dbg_muxtop = readl(reg_base + LCD_SLV_DBG);
	
	tmp = sepxlcnt_masked_state | CFG_DBG_RD_INDEX(DBG_ISA_SEL);
	writel(tmp, (reg_base + LCD_PN_SEPXLCNT));
	dmb();
	hwstate->dbg_isa = readl(reg_base + LCD_SLV_DBG);
	
	/* restore initial select state */
	writel(sepxlcnt_state, (reg_base + LCD_PN_SEPXLCNT));
	dmb();
}

static void smpn_show_state(struct dbg_state *hwstate, char *state_type)
{
	serial_printf("smpn hw state %s:\n", state_type ? state_type : "");
	serial_printf("dbg_slvtop = 0x%08x\n", hwstate->dbg_slvtop);
	serial_printf("dbg_txctrl = 0x%08x\n", hwstate->dbg_txctrl);
	serial_printf("dbg_gratop = 0x%08x\n", hwstate->dbg_gratop);
	serial_printf("dbg_muxtop = 0x%08x\n", hwstate->dbg_muxtop);
	serial_printf("dbg_isa    = 0x%08x\n", hwstate->dbg_isa);
	serial_printf("\n");
}

static void smpn_dump_saved_hwstate(void)
{
	smpn_show_state(&dbg_init, "dbg_init");
	smpn_show_state(&dbg_good, "dbg_good");
	smpn_show_state(&dbg_busy, "dbg_busy");
	smpn_show_state(&dbg_done, "dbg_done");
	smpn_show_state(&dbg_idle, "dbg_idle");
	smpn_show_state(&dbg_idle, "dbg_tout");
}

/*
 * smart_wait_fifo - wait for command transfer in progress to finish
 * @cmds: pointer to command/data sequence array
 * @n_cmds: number of array entries
 *
 * For now, implement without using interrupt overhead. This function will usually
 * be called for screen updates, which use a short/fast command sequence followed by
 * dma data.
 */  
static int smart_wait_fifo(void *reg_base) {
	unsigned timer;
	int rc = 0;
	int sanity = 1;
#if 1   
	uint32_t tmp, tmp2;
   
	/* Ensure fifo status is selected for LCD_SLV_DBG register */
	tmp = readl(reg_base + LCD_PN_SEPXLCNT);
	tmp &= ~(CFG_DBG_RD_INDEX_MASK);
	tmp |= CFG_DBG_RD_INDEX(DBG_SLVTOP_SEL);
	writel(tmp, (reg_base + LCD_PN_SEPXLCNT));

	dmb();
#endif
	/* sanity check */
	tmp = readl(reg_base + LCD_SLV_DBG);
	tmp2 = ((tmp & SLV_SPACECNT_MASK) >> SLV_SPACECNT_SHIFT) +
	      ((tmp & SLV_DATACNT_MASK) >> SLV_DATACNT_SHIFT);
	if(tmp2 != LCD_SLV_FIFO_DEPTH)
		sanity = 0;
	
	if(!sanity) {
		SMPNPRINTFERR("smpn:%s:%d: Warning: invalid fifo status 0x%08x, "
							"must use blind wait.\n", 
      							__func__, __LINE__, tmp);
		timer = 1000;
		while(timer--)
			udelay(1);
		return 0;
	}
	
	timer = 1000;
	while (1) {
		dmb();
		tmp = readl(reg_base + LCD_SLV_DBG);
		if(!(tmp & SLV_DATACNT_MASK))
			break;
 		timer--;
      		if(!timer) {
         		rc = -1;
	 		break;
      		}
      		udelay(1);
	}

	if(rc) {	
		/* save debug info */
		smpn_get_state(reg_base, &dbg_tout);
	
		SMPNPRINTFERR("smpn:%s:%d: Error: timeout waiting for fifo, last read 0x%08x\n",
      							__func__, __LINE__, tmp);
		smpn_dump_saved_hwstate();
		lxk_mv61fb_dump_regs(reg_base); /* ifdef __LXK_VGA_MV61FB_DEBUG__ */ 
	}
	return rc;
}

/**
 * smpn_check_dma_busy - verify dma is really finished
 * @regbase: base address of lcd control regs
 */  
static int smpn_check_dma_busy(void *reg_base)
{
	uint32_t tmp;
	uint32_t sepxlcnt_state;
	int ret = 0;
	int gra_dma_enabled;

	dmb();
	
	tmp = readl(reg_base + LCD_SPU_DMA_CTRL0);
	gra_dma_enabled = (tmp & CFG_GRA_ENA_MASK);
	
#if 0
	int vsync_enabled;

	tmp = readl(reg_base + LCD_SPU_DMA_CTRL1);
	vsync_enabled = ((tmp & CFG_VSYNC_TRIG_MASK) == CFG_VSYNC_TRIG(1));

	/* vsync support is not fully implemented yet */
	tmp = readl(reg_base + LCD_SPU_DMA_CTRL1);
	vsync_enabled = ((tmp & CFG_VSYNC_TRIG_MASK) == CFG_VSYNC_TRIG(1)) ? 1 : 0;

	/* if vsync trigger mode is enabled, must assume dma could be busy */
	if(vsync_enabled) {
		if(gra_dma_enabled) {
			ret = 1;
			goto check_dma_done;
		}
		else {
			/* illegal state */
		        SMPNPRINTFERR("smpn:%s:%d: illegal state\n", __func__, __LINE__);
			ret = -1;
			goto check_dma_done;
		}
	}
#else
	if(!gra_dma_enabled) {
		ret = 0;
		goto check_dma_done;
	}	
#endif
	/* 
	 * Determine whether a transaction is still in progress.
	 */ 
	
	/* Select graphics channel fifo status in LCD_SLV_DBG register */
	sepxlcnt_state = readl(reg_base + LCD_PN_SEPXLCNT);
	tmp =  sepxlcnt_state & ~(CFG_DBG_RD_INDEX_MASK);
	tmp |= CFG_DBG_RD_INDEX(DBG_GRATOP_SEL);
	writel(tmp, (reg_base + LCD_PN_SEPXLCNT));

	dmb();
		
	tmp = readl(reg_base + LCD_SLV_DBG);
	if(tmp & GRA_AFF_EMPTY_MASK) {
		/* 
		 * Verify that empty fifo was not caused by an underrun.
		 * To avoid potential race, need to recheck underrun after checking fifo.
		 */
		tmp = readl(reg_base + SPU_IRQ_ISR);
		if(tmp & GRA_FF_UNDERFLOW_MASK) {
			/* Underrun can contaminate the GRA_AFF_EMPTY flag */
		        SMPNPRINTFERR("smpn:%s:%d: illegal state\n", __func__, __LINE__);
			ret = -1;
		}
		else {
			/* there is no dma in progress */
			ret = 0;
		}
	}
	else {
		if(gra_dma_enabled) {
			/* report dma is still busy */
			ret = 1;
		}
		else {
			/* illegal state */
		        SMPNPRINTFERR("smpn:%s:%d: illegal state\n", __func__, __LINE__);
			ret = -1;
		}
	}
	
	/* restore the LCD_SLV_DBG mux */
	writel(sepxlcnt_state, (reg_base + LCD_PN_SEPXLCNT));

	dmb();
		
check_dma_done:
	return ret;
}

/*
 * smart_wait_dma - wait for dma in progress to finish
 * @cmds: pointer to command/data sequence array
 * @n_cmds: number of array entries
 *
 */  
static int smart_wait_dma(void *reg_base) {
	unsigned timer;
	int err = 0;
	int underrun = 0;
	int timeout1 = 0;
	int timeout2 = 0;
	uint32_t tmp = 0;
	int first = 0;
	int flags = 0;
	int mask = TWC_FRAMEDONE_MASK | GRA_FRAME_IRQ0_MASK;     
   
	SMPNPRINTFINIT("smpn:%s:%d\n", __func__, __LINE__);

	/* irq will be cleared when read in polled mode */
	timer = 50000;

	dmb();
	
	while (timer--) {
		tmp = readl(reg_base + SPU_IRQ_ISR);
		
		flags |= tmp & mask;
		if(!first)
			first = flags;
		
		/* don't stop for underrun, but complain */
		if(tmp & GRA_FF_UNDERFLOW_MASK)
			underrun = 1;
		
		/* wait until we see both flags */
		if(flags == mask)
			break;
		udelay(1);
	}
	
	/* save debug info */
	smpn_get_state(reg_base, &dbg_done);

	if(!timer)
		timeout1 = 1;
   
	/* verify the transaction is really finished */
	timer = 100;
	while (timer--) {
		tmp = smpn_check_dma_busy(reg_base);
		if(tmp < 1)	/* bail out if idle or error */
			break;
		udelay(1);
	}
	if(!timer)
		timeout2 = 1;
   
	if(tmp || timeout1 || timeout2 || underrun)
		err = -1;
   
	SMPNPRINTF("smpn:%s:%d: first = 0x%08x, flags = 0x%08x\n",
						 __func__, __LINE__, first, flags);
	if(underrun)
		SMPNPRINTFERR("smpn:%s:%d: dma underrun reported\n", __func__, __LINE__);
	if(timeout1)
		SMPNPRINTFERR("smpn:%s:%d: dma timeout waiting for dma\n", __func__, __LINE__);
	if(timeout2)
		SMPNPRINTFERR("smpn:%s:%d: dma buffer flush timeout\n", __func__, __LINE__);
   
	return err;
}

/*
 * smart_command_send - Stuff a command/data sequence into the fifo and wait.
 * @cmds: pointer to command/data sequence array
 * @n_cmds: number of array entries
 *
 * For now, implement without using interrupts. This function will usually
 * be called only for screen updates, which use a short command sequence that completes
 * very quickly.
 */  
static int smart_command_send(void *reg_base, const unsigned short *cmds, int n_cmds) {
	int i;
	unsigned short cmd;
	unsigned int entry;
	int err = 0;
	uint32_t tmp;
	int gra_ena_state;
	
        SMPNPRINTFINIT("smpn:%s:%d\n", __func__, __LINE__);
	
	if (!cmds || !n_cmds)
		return 0;
	
	dmb();
	
	/* Ensure slave path is enabled */
	tmp = readl(reg_base + LCD_SPU_SMPN_CTRL);
	if (!(tmp & CFG_SMPN_ENA_MASK) || !(tmp & CFG_SLV_ENA_MASK)) {
		LXKPANELDBG("%s:%d: smart panel not enabled\n", __func__, __LINE__);
		return -1;
	}

	/* The fifo should always be empty on entry, but just in case .... */
	err = smart_wait_fifo(reg_base);
	if(err) {
        	SMPNPRINTFERR("smpn:%s:%d:Error: smart_wait_fifo returned %d\n",
						 __func__, __LINE__, err);
#ifndef IGNORE_FIFO_TIMEOUT
		return err;
#else
		err = 0;
#endif
	}
	
	/* Must disable dma to use command fifo. */
	tmp = readl(reg_base + LCD_SPU_DMA_CTRL0);
	gra_ena_state  = tmp & CFG_GRA_ENA_MASK;
	tmp &= ~(CFG_GRA_ENA_MASK);
	writel(tmp, (reg_base + LCD_SPU_DMA_CTRL0));
	dmb();

	/*
	 * Two commands are written into each fifo word. An odd number of 
	 * commands requires padding with a bus NOP. Delays do not consume
	 * a command slot in the fifo, so a command + delay pair must also
	 * be padded and written before executing the delay.
	 *
	 * The cpu write buffer must be flushed after each fifo write so 
	 * the fifo status read will be accurate.
	 */	
	for (i = 0; i < n_cmds; ) {
		if(!(readl(reg_base + LCD_SLV_DBG) &
							SLV_SPACECNT_MASK)) {
			/*
			 * If the fifo is full, wait for it to empty. This
			 * should only happen during initialization, if ever.
			 */
			err = smart_wait_fifo(reg_base);
			if(err) {
        			SMPNPRINTFERR("smpn:%s:%d:Error: smart_wait_fifo "
							"returned %d\n",
							 __func__, __LINE__, err);
#ifndef IGNORE_FIFO_TIMEOUT
				return err;
#else
				err = 0;
#endif
			}
		}
		
		cmd = cmds[i] & 0x0ffff;
		SMPNPRINTFINIT("smpn:%s:%d: cmds[%d] = 0x%04x\n", __func__, __LINE__, i, cmd);
		
		i++;
		
		/* delay slot here does not consume fifo space */
		if ((cmd & SMART_DELAY_MASK) == SMART_DELAY_MASK) {
			mdelay(cmd & 0x0ff);
			continue;
		}
		
		entry = cmd & 0x0ffff;
		
		if ((i) < n_cmds) {
			cmd = cmds[i] & 0x0ffff;
		        SMPNPRINTFINIT("smpn:%s:%d: cmds[%d] = 0x%04x\n", __func__, __LINE__, i, cmd);
		}
		else {
			cmd = SMART_NOP & 0x0ffff;
		}
		
		i++;
		
		/* 
		 * delay slot here will consume fifo space for a nop so previous 
		 * command will flush before the delay
		 */
		if ((cmd & SMART_DELAY_MASK) == SMART_DELAY_MASK) {
			cmd = (SMART_NOP & 0x0ffff);
			entry |= ((unsigned int)(cmd & 0x0ffff) << SMART_CMD_SHIFT);
			SMPNPRINTFINIT("smpn:%s:%d: Writing 0x%08x to fifo\n", __func__, __LINE__, entry);
			writel(entry, (reg_base + LCD_SLV_PORT));
			//EIEIO();
			mdelay(cmd & 0x0ff);
			continue;
		}

		entry |= ((unsigned int)(cmd & 0x0ffff) << SMART_CMD_SHIFT);
		SMPNPRINTFINIT("smpn:%s:%d: Writing 0x%08x to fifo\n", __func__, __LINE__, entry);
		writel(entry, (reg_base + LCD_SLV_PORT));
		//EIEIO();
	}
	
	dmb();
	
	err = smart_wait_fifo(reg_base);
	if(err) {
#ifndef IGNORE_FIFO_TIMEOUT
        	SMPNPRINTFERR("smpn:%s:%d:Error: smart_wait_fifo returned %d\n",
						 __func__, __LINE__, err);
		err = smart_wait_fifo(reg_base);
#else
        	SMPNPRINTFERR("smpn:%s:%d:Error: smart_wait_fifo returned %d\n",
						 __func__, __LINE__, err);
		err = 0;
#endif

		if(err) {
        		SMPNPRINTFERR("smpn:%s:%d:Error: smart_wait_fifo returned %d\n",
							 __func__, __LINE__, err);
#ifndef IGNORE_FIFO_TIMEOUT
			return err;
#else
			err = 0;
#endif
		}
		else {
        		SMPNPRINTFINIT("smpn:%s:%d\n", __func__, __LINE__);
		}
	}

	tmp = readl(reg_base + LCD_SPU_DMA_CTRL0);
	tmp |= gra_ena_state;
	writel(tmp, (reg_base + LCD_SPU_DMA_CTRL0));
	dmb();

	return 0; 
}

/*
 * smpn_update - Prepare the display for frame data and send it
 *
 * This needs to block while booting.
 */
static void smpn_update(void)
{
	lb_vga_panel_data *vpd;
	lcd_smpn_info_t *sinfo;
	int err = 0;
	u32 tmp;

	/*
	 * Debug printfs in this driver can lead to infinitely recursive calls.
	 * Bailing out here should limit recursion to one benign level.
	 * There may still be another console device such as a uart that
	 * is also handling printfs, so just return quietly. 
	 */
	 
	if(smpn_console_busy)
		return;
	else
		smpn_console_busy = 1;
	
        SMPNPRINTFINIT("smpn:%s:%d\n", __func__, __LINE__);

	vpd = (lb_vga_panel_data *)panel_info.priv;
	if(!vpd) {
        	SMPNPRINTFERR("smpn:%s:%d:Error: panel_info not found\n",
						 __func__, __LINE__);
		goto smpn_update_done;
	}
		
	sinfo = (lcd_smpn_info_t *)vpd->smpn_info;
	if(!sinfo) {
        	SMPNPRINTFERR("smpn:%s:%d:Error: smpn_info not found\n",
						 __func__, __LINE__);
		goto smpn_update_done;
	}
	
	err = smpn_check_dma_busy(vpd->reg_base);
	if(err) {
        	SMPNPRINTFERR("smpn:%s:%d:Error: smpn_check_dma_busy returned %d\n",
						 __func__, __LINE__, err);
		err = 0;
	}

	err = smart_wait_fifo(vpd->reg_base);
	if(err) {
        	SMPNPRINTFERR("smpn:%s:%d:Error: smart_wait_fifo returned %d\n",
						 __func__, __LINE__, err);
#ifndef IGNORE_FIFO_TIMEOUT
		goto smpn_update_done;
#else
		err = 0;
#endif
	}

	err = smart_command_send(vpd->reg_base, sinfo->frame_update, sinfo->frame_update_size);
	if(err) {
        	SMPNPRINTFERR("smpn:%s:%d:Error: smart_command_send returned %d\n",
						 __func__, __LINE__, err);
		err = smart_command_send(vpd->reg_base, sinfo->frame_update, 
						sinfo->frame_update_size);
		if(err) {
        		SMPNPRINTFERR("smpn:%s:%d:Error: smart_command_send returned %d\n",
						 __func__, __LINE__, err);
			goto smpn_update_done;
		}
	}
	
	/* kick off the frame data dma */
	tmp = readl(vpd->reg_base + LCD_SPU_DMA_CTRL1);
	tmp &= ~(CFG_FRAME_TRIG_MASK);
	writel(tmp, (vpd->reg_base + LCD_SPU_DMA_CTRL1));
	dmb();
	udelay(1);
	tmp |= CFG_FRAME_TRIG(1);
	writel(tmp, (vpd->reg_base + LCD_SPU_DMA_CTRL1));
	dmb();
	udelay(1);
	tmp &= ~(CFG_FRAME_TRIG_MASK);
	writel(tmp, (vpd->reg_base + LCD_SPU_DMA_CTRL1));
	dmb();
	
	/* save debug info */
	smpn_get_state(vpd->reg_base, &dbg_busy);
	
	err = smart_wait_dma(vpd->reg_base);
	if(err) {
        	SMPNPRINTFERR("smpn:%s:%d:Error: smart_wait_dma returned %d, "
						"must use blind wait\n",
						 __func__, __LINE__, err);
		
		mdelay(15);
	}

	/* Allow time for dma fifo to flush */
	udelay(10);

	/* save debug info */
	smpn_get_state(vpd->reg_base, &dbg_idle);

	/* verify the command fifo is empty */
	err = smart_wait_fifo(vpd->reg_base);
	if(err) {
        	SMPNPRINTFERR("smpn:%s:%d:Error: smart_wait_fifo returned %d\n",
						 __func__, __LINE__, err);
#ifndef IGNORE_FIFO_TIMEOUT
		goto smpn_update_done;
#else
		err = 0;
#endif
	}
	
smpn_update_done:
	smpn_console_busy = 0;
	
}

/*
 * smpn_display_init - Initialize the Smart panel display.
 *
 */
static int smpn_display_init(void *reg_base, lcd_smpn_info_t *sinfo)
{
   int err = 0;
   
   err = smart_command_send(reg_base, sinfo->power_on, sinfo->power_on_size); 
   if(!err && sinfo->rotate)
      smart_command_send(reg_base, sinfo->rotate, sinfo->rotate_size); 
   if(!err)
      err = smart_command_send(reg_base, sinfo->panel_on, sinfo->panel_on_size); 
   return err;
}

static int __smart_timing(unsigned min_time_ns, int refclk_ns)
{
	int ticks;
	
	ticks = (int)((min_time_ns) / refclk_ns);
	
	if((ticks * refclk_ns) < (min_time_ns)) 
		ticks++;
	
	/* 0 = 1 clock */
	ticks--;
	
	ticks = max(ticks, CFG_ISA_TIME_MIN);
	ticks = min(ticks, CFG_ISA_TIME_MAX);
	
	return (ticks);
}

/* 
 * Counters start at 0, and there is always a dead clock between cycles.
 * Low time = (ticks + 1) * clock period.
 * High time = (ticks + 2) * clock period
 * Smpn min cycle time is generally greater than min low + min high.
 * If the smpn output clock will be used for serdes, minimize the frequency,
 * otherwise try to balance the duty cycle to maximize the strobe width.
 */
static void setup_smart_timing(void *reg_base, lcd_smpn_info_t *sinfo, int clk_div, int smpnclk)
{
	int rxhigh, rxlow, txhigh, txlow;
	u32 tmp;
	int tmpi;
	int refclk_ns;
	int core_mhz;
	
	/* 
	 * Get the actual reference clock period in nanosecs.
	 * The order  of operations is important to prevent overflow.
	 */
	core_mhz = board_get_core_freq_hz();
	refclk_ns = (clk_div + 1) * (1000000000 / board_get_core_freq_hz());

	tmpi = smpnclk ? sinfo->rxlow : max(sinfo->rxlow, (sinfo->rxcycle/2));
	rxlow = __smart_timing(tmpi, refclk_ns);
	/* adjust high time request for actual low time and dead clock */
	tmpi = max(sinfo->rxhigh, (sinfo->rxcycle - ((rxlow + 1) * refclk_ns))); 
	if(tmpi > refclk_ns)
		tmpi -= refclk_ns;
	else
		tmpi = 0;
	rxhigh = __smart_timing(tmpi, refclk_ns); 

	tmpi = smpnclk ? sinfo->txlow : max(sinfo->txlow, (sinfo->txcycle/2));
	txlow = __smart_timing(tmpi, refclk_ns);
	/* adjust high time request for actual low time and dead clock */
	tmpi = max(sinfo->txhigh, (sinfo->txcycle - ((txlow + 1) * refclk_ns))); 
	if(tmpi > refclk_ns)
		tmpi -= refclk_ns;
	else
		tmpi = 0;
	txhigh = __smart_timing(tmpi, refclk_ns); 

	tmp = readl(reg_base + LCD_SPU_SMPN_CTRL);
	tmp &= ~(CFG_ISA_RXLOW_MASK | CFG_ISA_RXHIGH_MASK |
				CFG_ISA_TXLOW_MASK | CFG_ISA_TXHIGH_MASK);
	tmp |= CFG_ISA_RXLOW(rxlow) | CFG_ISA_RXHIGH(rxhigh);
	tmp |= CFG_ISA_TXLOW(txlow) | CFG_ISA_TXHIGH(txhigh);

	writel(tmp, (reg_base + LCD_SPU_SMPN_CTRL));
	dmb();
	
	SMPNPRINTF("core_mhz=%d, clk_div+1=%d, refclk_ns=%d, txlow=%d, txhigh=%d\n", 
						core_mhz, (clk_div + 1), refclk_ns, 
						txlow, txhigh); 
}

/*
 * get_lcd_smpn_info - Get the parameters and command sequences.
 *
 */
static lcd_smpn_info_t *get_lcd_smpn_info(int smpn_type)
{
	lb_vga_panel_data *vpd = (lb_vga_panel_data *)panel_info.priv;
	
	/* assume there will inevitably be more types later */
	switch(vpd->smpn_type) {
	case LXK_PANEL_SMPN_TYPE_TM024HDH49:
		return ((lcd_smpn_info_t *)&smpn_info_TM024HDH49);
	default:
	   return NULL; 
	}
}

int lxk_mv61fb_smart_panel_init(void *reg_base, void *buffer)
{
	char string[32];
	lb_vga_panel_data *vpd = NULL;
	lcd_smpn_info_t *sinfo;
	unsigned xtotal, ytotal;
	unsigned tmp;
	unsigned bufsize;
	int clk_div;
	int smpnclk; /* 1 = smpn output clock will be used for serdes */

	vpd = (lb_vga_panel_data *)panel_info.priv;

	if(!vpd || !reg_base || !panel || !buffer) {
		LXKPANELDBG("%s:%d: no panel data found\n", __func__, __LINE__);
		return -1;
	}
	
	smpnclk = vpd->use_smpnclk;
	
	sinfo = get_lcd_smpn_info(vpd->smpn_type);
	vpd->smpn_info = (void *)sinfo;
	vpd->reg_base = reg_base;
		
	xtotal = vpd->xres;
	ytotal = vpd->yres;
	
	bufsize = panel_info.vl_row*panel_info.vl_col*(NBITS(panel_info.vl_bpix)/8);
	tmp = lxk_vram_size();
	if(bufsize > tmp) {
		printf("%s:%d: Error, display frame size 0x%08x exceeds frame buffer "
					"size 0x%08x\n", __func__, __LINE__, 
					bufsize, tmp);
		return -1;
	}	
	sprintf(string, "0x%08x@0x%p", tmp, buffer);
	setenv("vram",string);

	/*
	 * Configure default register values.
	 */
	writel( 0x000000ff, reg_base + LCD_CFG_SCLK_DIV);
	writel( 0x00000000, reg_base + LCD_SPU_DMA_CTRL0);
	writel( 0x00000000, reg_base + LCD_SPU_DMA_CTRL1);
	writel( 0x00000000, reg_base + LCD_SPU_SMPN_CTRL);
	writel( 0x00000000, reg_base + LCD_SPU_DUMB_CTRL);
	writel( 0x00000000, reg_base + SPU_IOPAD_CONTROL);
	writel( 0x00000000, reg_base + SPU_IRQ_ENA);
	writel( 0x00000050, reg_base + LCD_TOP_CTRL);
	writel( 0x00000000, reg_base + SPU_IRQ_ISR);
	writel( 0x00000000, reg_base + LCD_SPU_BLANKCOLOR);
	writel( 0x00000000, reg_base + LCD_SPU_GRA_OVSA_HPXL_VLN);
	writel( 0x00000000, reg_base + LCD_SPU_SRAM_PARA1);

	writel( (unsigned int)buffer, reg_base + LCD_CFG_GRA_START_ADDR0);
	writel( (unsigned int)buffer, reg_base + LCD_CFG_GRA_START_ADDR1);

	/* start and end values must be non-zero */
	writel( 0x00010001, reg_base + LCD_PN_SEPXLCNT);

	dmb();
	
	/*
	 * configure the frame buffer to represent full-screen on the panel.
	 */
	tmp = vpd->yres << 16;
	tmp |= vpd->xres & 0x0ffff;
	writel(tmp, reg_base + LCD_SPU_V_H_ACTIVE);
	writel(tmp, reg_base + LCD_SPU_GRA_HPXL_VLN);
	writel(tmp, reg_base + LCD_SPU_GZM_HPXL_VLN);
	
	/* apparently not used for smpn, but match known good settings */
	tmp = 0;
	writel(tmp, reg_base + LCD_SPUT_V_H_TOTAL);

	if(smpnclk)
		clk_div = lxk_mv61fb_get_clk_div(TARGET_FREQ_HZ_SLOW);
	else
		clk_div = lxk_mv61fb_get_clk_div(TARGET_FREQ_HZ_FAST);

	writel(clk_div, reg_base + LCD_CFG_SCLK_DIV);

	/* Reset the data path and state machines. */
	writel(0, (reg_base + LCD_SOFT_RESET));
	udelay(1);
	writel(CFG_SW_RESET(1), (reg_base + LCD_SOFT_RESET));
	udelay(1);
	writel(0, (reg_base + LCD_SOFT_RESET));
	udelay(100);

   	/* enable lvds serializer */
   	gpio_direction(VIDEO_SIG, 1);
   	gpio_set(VIDEO_SIG, 1);
	
	tmp = CFG_DMAFORMAT(PIX_FMT_RGB565); /* format of src data in buffer */
	tmp |= CFG_ARBFAST_ENA(1);
	tmp |= CFG_GRA_ENA(1);
	writel(tmp, reg_base + LCD_SPU_DMA_CTRL0);
	
	tmp = CFG_FRAME_TRIG(0);
	tmp |= CFG_VSYNC_TRIG(VSYNC_MODE_WRITE_TRIG); /* write triggers dma */
	tmp |= CFG_PN_ALPHA_MODE(ALPHA_DISABLED);     /* disable Alpha channel */
	tmp |= CFG_PN_ALPHA(0xff);                    /* no effect when disabled, but stay consistent with sim */
	tmp |= CFG_PXLCMD(PXLCMD_OPAQUE_VALUE);       /* per spec */
	writel(tmp, reg_base + LCD_SPU_DMA_CTRL1);
	
	lxk_mv61fb_default_gamma(reg_base);

	tmp = vpd->xres * (NBITS(panel_info.vl_bpix)/8);
	writel( tmp, reg_base + LCD_CFG_GRA_PITCH);
	
	dmb();
	
	/* this will modify LCD_SPU_SMPN_CTRL */
	setup_smart_timing(reg_base, sinfo, clk_div, smpnclk);

	dmb();
	
	tmp = readl(reg_base + LCD_SPU_SMPN_CTRL);
	tmp &= (CFG_ISA_RXLOW_MASK | CFG_ISA_RXHIGH_MASK |
				CFG_ISA_TXLOW_MASK | CFG_ISA_TXHIGH_MASK);
	tmp |= CFG_IORDY_MSK(0);
	tmp |= CFG_SLV_ONLY(0);
	tmp |= CFG_SMPNMODE(SMART_MODE_RGB565_2CYCLE);
	tmp |= CFG_CMD32OR16B(0);
	tmp |= CFG_SWAPBYTES(1);
	tmp |= CFG_SMPN_SEL(0);
	tmp |= CFG_68S80SB(0);
	tmp |= CFG_KEEPXFER(1);
	tmp |= CFG_SMPN_ENA(1);
	tmp |= CFG_SLV_ENA(1);
	writel(tmp, (reg_base + LCD_SPU_SMPN_CTRL));
	//wmb();
	
	/* pulse the reset */
	mdelay(200);
	writel((tmp | CFG_SMPN_RSTB_MASK), (reg_base + LCD_SPU_SMPN_CTRL));
	//wmb();
	mdelay(200);
	writel(tmp, (reg_base + LCD_SPU_SMPN_CTRL));
	//wmb();
	
	tmp = CFG_IOPADMODE(PIN_MODE_SMART_8_GPIO);
	writel(tmp, reg_base + SPU_IOPAD_CONTROL);

	/* set up for polling the frame done and underrun interrupts */
	tmp = GRA_FRAME_IRQ0_ENA(1);
	tmp |= TWC_FRAMEDONE_ENA(1);
	tmp |= GRA_FF_UNDERFLOW(1);
	writel(tmp, reg_base + SPU_IRQ_RSR);
	//writel(tmp, reg_base + SPU_IRQ_ENA);

	lxk_mv61fb_paint_buffer(buffer, bufsize); 	

	dmb();
	
	smpn_display_init(reg_base, sinfo);
	
	dmb();
	
	lxk_mv61fb_dump_regs(reg_base);
	
	/* save debug info */
	smpn_get_state(reg_base, &dbg_init);

	smpn_update();
	
	mdelay(10);
	
	/* save debug info */
	smpn_get_state(reg_base, &dbg_good );

	vpd->backlight_enable(1);
	
	/* 
	 * Must disable any printfs in the smpn_update() call chain
	 * before registering it to avoid recursion.
	 */
	smpn_console_active = 1;
	panel_info.refresh = smpn_update;
	
	return 0;
}
