#include <common.h>
#include <asm/arch/gpio.h>
#include <asm/arch/GPIO_regstructs.h>
#include <asm/arch/GPIO_regmasks.h>

/* Base addresses for the gpio blocks */
static const struct gpio_block_config_t *gpioConfig;
static uint32_t gpioConfigSize;

/* Mapping of signal to gpio */
static const struct gpio_mapping_t *gpioMapping;
static uint32_t gpioMappingSize;

static int32_t get_gpio_from_signal(enum signal_id_t id) {
    uint32_t i;

    for (i=0; i<gpioMappingSize; ++i) {
        if (gpioMapping[i].id == id)
            return (int32_t)gpioMapping[i].gpio;
    }

    return -1;
}

static volatile struct GPIO_REGS_s *get_block_from_gpio(int32_t gpio) {
    int index = gpio / 32;

    if (index > gpioConfigSize)
        return NULL;

    /* Some pins in a block may not be valid */
    if (!(gpioConfig[index].valid & (1 << (gpio & 0x1f))))
        return NULL;

    return ((volatile struct GPIO_REGS_s *)gpioConfig[index].base);
}

int gpio_direction(enum signal_id_t signal, int direction) {
    int32_t gpio;
    volatile struct GPIO_REGS_s *gpio_block;

    gpio = get_gpio_from_signal(signal);
    if (gpio < 0)
        return -1;

    gpio_block = get_block_from_gpio(gpio);
    if (!gpio_block)
        return -1;

    if (direction)
        gpio_block->OE |= (1 << (gpio & 0x1f));
    else
        gpio_block->OE &= ~(1 << (gpio & 0x1f));

    return 0;
}

int gpio_sample(enum signal_id_t signal) {
    int32_t gpio;
    volatile struct GPIO_REGS_s *gpio_block;

    gpio = get_gpio_from_signal(signal);
    if (gpio < 0)
        return -1;

    gpio_block = get_block_from_gpio(gpio);
    if (!gpio_block)
        return 0;

    return !!(gpio_block->InData & (1 << (gpio & 0x1f)));
}

int gpio_set(enum signal_id_t signal, uint32_t value) {
    int32_t gpio;
    volatile struct GPIO_REGS_s *gpio_block;

    gpio = get_gpio_from_signal(signal);
    if (gpio < 0)
        return -1;

    gpio_block = get_block_from_gpio(gpio);
    if (!gpio_block)
        return -1;

    if (value)
        gpio_block->OutData |= (1 << (gpio & 0x1f));
    else
        gpio_block->OutData &= ~(1 << (gpio & 0x1f));

    return 0;
}

int gpio_mux(enum signal_id_t signal, uint32_t mux) {
    int32_t gpio;
    volatile struct GPIO_REGS_s *gpio_block;
    volatile uint32_t *pincfg;

    gpio = get_gpio_from_signal(signal);
    if (gpio < 0)
        return -1;

    gpio_block = get_block_from_gpio(gpio);
    if (!gpio_block)
        return -1;

    pincfg = (volatile uint32_t *)&gpio_block->PinCfg0;
    pincfg[gpio & 0x1f] = (pincfg[gpio & 0x1f] & ~0x7) | mux;

    return 0;
}

int gpio_init_pins(const struct gpio_mapping_t *lGpioMapping, uint32_t lGpioMappingSize, const uint8_t *pinSetup, uint32_t pinSetupSize) {
    uint32_t i;
    uint32_t func_sel;
    uint32_t index;
    uint32_t temp;
    volatile uint32_t *pincfg;
    volatile struct GPIO_REGS_s *gpio_block;

    gpioMapping     = lGpioMapping;
    gpioMappingSize = lGpioMappingSize;

    for (i=0; i<pinSetupSize; ++i) {
        gpio_block = get_block_from_gpio(i);
        if (gpio_block) {
            index = i & 0x1f;

            /*
             * If the pin function is GPIO then set the direction
             * and value.
             */
            switch ((pinSetup[i] & GPIO_PINCFG_FUNC_MASK) >> GPIO_PINCFG_FUNC_SHIFT) {
                case 0:
                    func_sel = 0;
                    gpio_block->OE &= ~(1 << index);
                    break;
                case 8:
                    func_sel = 0;
                    gpio_block->OutData &= ~(1 << index);
                    gpio_block->OE |= (1 << index);
                    break;
                case 9:
                    func_sel = 0;
                    gpio_block->OutData |= (1 << index);
                    gpio_block->OE |= (1 << index);
                    break;
                default:
                    func_sel = (pinSetup[i] & GPIO_PINCFG_FUNC_MASK) >> GPIO_PINCFG_FUNC_SHIFT;
                    break;
            }

            pincfg = (volatile uint32_t *)&gpio_block->PinCfg0;

            temp = GPIO_PINCFG0_FUNCSEL_REPLACE_VAL(pincfg[index], func_sel);
            temp = GPIO_PINCFG0_SLEW_REPLACE_VAL(temp, ((pinSetup[i] & GPIO_PINCFG_SR_MASK) >> GPIO_PINCFG_SR_SHIFT));
            pincfg[index] = GPIO_PINCFG0_PUEN_REPLACE_VAL(temp, ((pinSetup[i] & GPIO_PINCFG_PU_MASK) >> GPIO_PINCFG_PU_SHIFT));
        }
    }

    return 0;
}

int gpio_block_init(const struct gpio_block_config_t *lGpioConfig, uint32_t lGpioConfigSize) {
    gpioConfig     = lGpioConfig;
    gpioConfigSize = lGpioConfigSize;

    return 0;
}
