/*
 * (C) Copyright 2011
 * Marvell Semiconductors Ltd. <www.marvell.com>
 * Lei Wen <leiwen@marvell.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <common.h>
#include <errno.h>
#include <i2c.h>
#include <ddr_spd.h>
#include <mvmfp.h>
#include <asm/arch/mfp.h>
#include <fdt_support.h>
#include <asm/arch/gpio.h>
#include <asm/gpio.h>
#include <asm/io.h>
#include "ddr.h"
#include "asm/arch/regAddrs.h"
#include "asm/arch/apb_config_regstructs.h"
#include "MC_regstructs.h"
#include "MC_regmasks.h"
#include "fpga_regAddrs.h"
#include <mv_mmp_panel.h>
#ifdef CONFIG_GENERIC_MMC
#include <sdhci.h>
#include <asm/arch/cpu.h>
#endif
#ifdef CONFIG_CMD_NAND
#include <linux/err.h>
#include <nand.h>
#include "fpgaGpio.h"
#include "fpga_GPIO_regstructs.h"
#endif

//#define LIMIT_DRAM_TO_256MB ( 1 )
/*  RICOH : add  */
#define  FASTBOOT

DECLARE_GLOBAL_DATA_PTR;
short global_fix;

typedef struct 
{
    uint64_t start;
    uint64_t size;
} bank_info_t;

/**
 * Additional info about the board. To be stored in ddr3_spd_eeprom_t.cust field.
 */
typedef struct board_info_s {
  uint32_t magic_value;    // used to verify that this info is valid
  uint32_t version;        // version of this struct, to know whether the layout is what you expect
  uint32_t size;           // size of this struct – ie – how much data can be trusted?
  char     serial_num[16]; // the device serial number
  uint8_t  eth_mac[6];     // Holding spot for the mac address

  // potentially more items in the future, similar to static NVRAM section we had in the past
  // could hold serial numbers, etc…
} board_info_t;


static bank_info_t mc_banks[CONFIG_MC_BANKS] __attribute__ ((section (".data")));
static bank_info_t usable_banks[CONFIG_NR_DRAM_BANKS] __attribute__ ((section (".data")));

#define PEGMTITE_BRD_MAGIC_WORD 0x5045474D
#define PEGMTITE_BRD_VER_1_0    0x00010000

#define REV_A0 0x0000
#define REV_B0 0x0010
#define REV_B1 0x0011
#define REV_C0 0x0020

#define AP_CIU_BASE_GR2_REVA    (0xD4282400)
#define AP_CIU_BASE_POST_REVA   (0xD0610000)

#define DFT_ISLD_REV_ID_REG     (0xD401D800)
#define DFT_ISLD_GR2_REVA_VAL   (0x00010000)

#define board_is(board) ( 0 == memcmp(spd.mpart, board, strlen(board)))

int is_Gr2_RevA(void) {
    volatile uint32_t *DFT_ISLD_rev_id_reg = (uint32_t*)DFT_ISLD_REV_ID_REG;
    return (DFT_ISLD_GR2_REVA_VAL == *DFT_ISLD_rev_id_reg);
}

/*
 * The CIU moved between Gr2 Rev A and later.  Reading the Rev A location on later
 * parts, or reading the later location on Gr2 Rev A are both a no go.  Check the DFT
 * revision register to determine if this is Gr2 Rev A or not, and use that
 * information to determine where to read the chip id.
 */
static inline void* get_CIU_base(void)
{
    if (is_Gr2_RevA())
        return (void*)AP_CIU_BASE_GR2_REVA;
    else
        return (void*)AP_CIU_BASE_POST_REVA;
}

inline void* get_APB_CONFIG_base(void)
{
    if (is_Gr2_RevA())
        return (void*)AP_AP_APB_APB_CONFIG_BASE;
    else
        return (void*)(AP_AP_APB_APB_CONFIG_BASE - 8);
}

/*
 * If the APMU revision is 1.2, then this is Rev A.  Else, get the revision
 * from the Rev B chip id register.
 */
static inline uint16_t get_chip_id(void)
{
    volatile uint32_t *CIU_chip_id_reg = (uint32_t*) get_CIU_base();
    
    return *CIU_chip_id_reg & 0xFFFF;
}


static inline uint8_t get_chip_rev(void)
{
    volatile uint32_t *CIU_chip_id_reg = (uint32_t*) get_CIU_base();
    
    return ((*CIU_chip_id_reg) >> 16) & 0xFF;
}


bool isFPGA(void) {
    APB_CONFIG_REGS_t *apb_config_regs = (APB_CONFIG_REGS_t*)get_APB_CONFIG_base();

    /* SQU_REV0 will read as zero on FPGAs.  Safe to read on Si. */
    return (apb_config_regs->SQU_REV0 == 0);
}

unsigned int get_uart_clock(void) {
    if (isFPGA( ))
        return 13000000;

    return 11059200;
}

/*
 * Read data in EEPROM
 * \param spd the data structure to be filled with data
 * \bus the bus number
 *
 * \return 0 on success, not 0 on failure
 */
static int i2c_read_eeprom(ddr3_spd_eeprom_t *spd, int bus)
{
	int rc;
	int len = sizeof(ddr3_spd_eeprom_t);
    int eeprom_addr[] = { 0x50, 0x51, 0x54 };
    int i;

	for (i=0; i<sizeof(eeprom_addr)/sizeof(int); i++) {
		rc = i2c_read(eeprom_addr[i], 0, 1, (uint8_t*)spd, len);
		if (0 == rc) {
			printf("i2c_read_eeprom succeeded from 0x%X bus %d\n", eeprom_addr[i], bus);
			/* validate the data */
			if (0 != ddr3_spd_check(spd)) {
				printf("ddr3_spd_check failed on addr: 0x%02x\n",eeprom_addr[i]);
				rc = -EILSEQ;
                continue;
			}
			break;
		}
	}
	return rc;
}

static int i2c_read_spd(ddr3_spd_eeprom_t *spd)
{
    int rc, bus;

    bus  = i2c_get_bus_num( );
    i2c_set_bus_num(CONFIG_I2C_MULTI_BUS);
    rc = i2c_read_eeprom(spd, CONFIG_I2C_MULTI_BUS);
    i2c_set_bus_num(bus);

    if ((0 != rc) && (CONFIG_I2C_MULTI_BUS != bus)) {
    	/* try the default bus */
    	rc = i2c_read_eeprom(spd, bus);
    }

    return rc;
}

/**
 * Read board info from the EEPROM
 * \param board_info output
 * \return 0 on success, not 0 on failure
 */
static int i2c_read_board_info(struct board_info_s *board_info)
{
	int rc;
	ddr3_spd_eeprom_t spd;

	rc = i2c_read_spd(&spd);
	if (0 == rc) {
		int board_info_size = sizeof(struct board_info_s);
		memcpy(board_info, spd.msd, board_info_size);
		/* validate the board. First the magic word. */
		if (PEGMTITE_BRD_MAGIC_WORD != board_info->magic_value) {
			printf("board_info_s bad magic value: 0x%X expecting 0x%X\n",
					PEGMTITE_BRD_MAGIC_WORD);
			rc = -EILSEQ;
		}
		switch (board_info->version) {
		case PEGMTITE_BRD_VER_1_0:
			if (board_info_size != board_info->size) {
				printf("bad board_info_s size: %d expecting %d\n",
						board_info->size, board_info_size);
				rc = -EILSEQ;
			}
			break;
		default:
			rc = -EILSEQ;
			printf("bad board_info_s version: 0x%X\n", board_info->version);
			break;
		}
	}
	return rc;
}


struct board_info {
    const char *spd_mpart_name;
    const char *name;
    int onboard_dev;
    int dc_dev;
    int dc_blknum;
    int (*board_late_init)(void);
    int panel_reset;
    int panel_bus;
    int dac_reset;
    int buzzer;
    int hdd_reset;
    int isp_reset;
    int recovery;
};

/*
 * The enums in this list should match the definitions in
 * board_configs below.
 */
enum board_type {
    BOARD_UNDEFINED = 0,
    BOARD_GS2_DESTINY,
    BOARD_GR2_DESTINY,
    BOARD_TOC,
    BOARD_GLD2,
    BOARD_GLD4,
    BOARD_FFCr1,
    BOARD_FFCr2,
};

static const struct board_info board_configs[] = {
    [BOARD_UNDEFINED]   = { "",         "undefined",           0, 0, 0, NULL,  0, 0, -1, -1, -1, -1, -1 },
    [BOARD_GS2_DESTINY] = { "GS2_DESTY","mv6220-destiny-fpga", 0, 0, 0, NULL, 15, 2, -1, -1, -1, -1, -1 },
    [BOARD_GR2_DESTINY] = { "GR2_DESTY","mv6270-destiny-fpga", 0, 0, 0, NULL, 15, 2, -1, -1, -1, -1, -1 },
    [BOARD_TOC]         = { "TOC",      "mv6270-toc",          0, 2, 1, NULL, 15, 2, -1, -1, -1, -1, -1 },
    [BOARD_FFCr2]       = { "FFC_r2",   "mv6270-ffc_r2",       0, 2, 0, NULL, 15, 2, -1, -1, -1, -1, -1 },
    [BOARD_FFCr1]       = { "FFC_r1",   "mv6270-ffc_r1",       2, 2, 0, NULL, 15, 2, -1, -1, -1, -1, -1 },
    [BOARD_GLD2]        = { "GLD_2",    "mv6270-Golden2",      0, 2, 1, NULL, 15, 2, -1, -1, -1, -1, -1 },
    [BOARD_GLD4]        = { "GLD_4_TWN","mv6270-Golden4",      0, 2, 1, NULL, 15, 2, -1, -1, -1, -1, -1 },
};

static enum board_type gboard __attribute__ ((section (".data"))) = BOARD_UNDEFINED;

static enum board_type get_board_type(void) {
    if (gboard == BOARD_UNDEFINED) {
        gboard = BOARD_TOC; /* the default board */
        if (isFPGA( )) {
            if (0x6220 == get_chip_id())
                gboard = BOARD_GS2_DESTINY;
            else if(0x6270 == get_chip_id())
                gboard = BOARD_GR2_DESTINY;
        }
        else {
            ddr3_spd_eeprom_t spd;
            int rc;

            rc = i2c_read_spd(&spd);

            if ((rc == 0) && (spd.mem_type == SPD_MEMTYPE_DDR3)) {
                int i;

            	printf("%s mem_type=%d mpart=%s\n", __func__, spd.mem_type, spd.mpart);
                for (i=BOARD_TOC;i<(sizeof(board_configs)/sizeof(struct board_info));i++)
                {
            	    //printf("%s gboard testing: %s %d\n", __func__, board_configs[i].name,strlen(board_configs[i].spd_mpart_name));
                    if(board_is(board_configs[i].spd_mpart_name))
                    {
                        gboard = i;
                        break;
                    }
                }
            	printf("%s Configuring for: %s\n", __func__, board_configs[gboard].name);
            }
        }
    }

    return gboard;
}

#define CLKIN_PAD_PD    (1 << 3)
#define CLKOUT_PAD_PD   (1 << 1)
#define DATA_PAD_PD     (1 << 25)
/* Power down the SCIF0 and SCIF1 related LVDS pads */
static void powerdown_lvds_pads(void) {
    volatile uint32_t *scifx_lvds_reg;
    volatile uint32_t *scifx_clk_cntrl_reg;
    volatile uint32_t *apbc_asfar_reg;
    volatile uint32_t *apbc_assar_reg;
    int i;

    /* For RevB and beyond */
    if (!is_Gr2_RevA( )) {
        scifx_lvds_reg      = (volatile uint32_t *)0xD401E324;
        scifx_clk_cntrl_reg = (volatile uint32_t *)0xD401EBF0;
        apbc_asfar_reg      = (volatile uint32_t *)0xD4015000;
        apbc_assar_reg      = (volatile uint32_t *)0xD4015004;
        for(i = 0; i < 6; i++) {
            *scifx_lvds_reg |= DATA_PAD_PD;
            scifx_lvds_reg++;
        }
        for(i = 0; i < 2; i++) {
            /* to access the secure registers */
            *apbc_asfar_reg = 0xbaba;
            *apbc_assar_reg = 0xeb10;
            *scifx_clk_cntrl_reg = (CLKIN_PAD_PD | CLKOUT_PAD_PD);
            scifx_clk_cntrl_reg++;
        }
    }
}

static void power_up_srams(void) {
    int i;
    volatile unsigned int *ip = (volatile unsigned int *)AP_APMU_BASE;
    static const uint32_t apmu_sram_pwdn[] = {
        0x3c, 0x48, 0x50, 0x58, 0x5c, 0x68, 0x74, 0x80,
        0x8c, 0x94, 0x9c, 0xa4, 0xc0, 0xc8, 0xd0, 0xdc,
        0xe4, 0xec, 0xf4, 0x104,
    };
    static const uint32_t ipmu_sram_pwdn[] = {
        0x10, 0x18, 0x28, 0x30, 0x38, 0x48, 0x54, 0x5c,
        0x64, 0x70,
    };

    for (i=0 ; i<ARRAY_SIZE(apmu_sram_pwdn); ++i)
        ip[apmu_sram_pwdn[i]/4] = 0;

    ip = (volatile unsigned int *)IPS_IPMU_BASE;
    for (i=0 ; i<ARRAY_SIZE(ipmu_sram_pwdn); ++i)
        ip[ipmu_sram_pwdn[i]/4] = 0;
}



static void setup_pin_cfg(void) {
    u32 mfp_cfg[] = {
        MFP5_USB_HUB_POR,
/*  RICOH : del, we don't use I2C
        MFP16_I2C2_SDA,
        MFP17_I2C2_SCL,
*/
        MFP100_PCIE_CLK_OE,
        MFP113_SD_MMC1_CLK,
        MFP114_SD_MMC1_CMD,
        MFP118_SD_MMC1_DATA0,
        MFP119_SD_MMC1_DATA1,
        MFP120_SD_MMC1_DATA2,
        MFP121_SD_MMC1_DATA3,
        MFP122_SD_MMC1_DATA4,
        MFP123_SD_MMC1_DATA5,
        MFP124_SD_MMC1_DATA6,
        MFP125_SD_MMC1_DATA7,
        MFP152_SS1_RESET,
        MFP_EOC
    };

    mfp_config(mfp_cfg);
}

static uint32_t get_memAddrMap(int i) {
    uint32_t memAddrMap;
    struct MC_REGS_s *mc_reg = (struct MC_REGS_s *)CONFIG_MC_BASE;

    switch (i) {
        default:
        case 0:  memAddrMap = mc_reg->CH0_MMAP0; 
                 #if defined(LIMIT_DRAM_TO_256MB) && (LIMIT_DRAM_TO_256MB == 1)
                 memAddrMap = MC_CH0_MMAP0_AREA_LENGTH_REPLACE_VAL(memAddrMap, 12); // force down to 256MB
                 #endif
                 break;
        case 1:  memAddrMap = mc_reg->CH0_MMAP1; break;
        case 2:  memAddrMap = mc_reg->CH0_MMAP2; break;
    }

    return memAddrMap;
}

/******************************
 Routine:
 Description:
******************************/
static uint64_t dram_size_from_mask(uint32_t dram_mask) {
    return ((uint64_t)1 << ((dram_mask - 4) + 20));
}

static void read_mc_bank_info(void)
{
    int i;
    uint32_t mc_bank_config;

    memset(mc_banks, 0, sizeof(mc_banks));

    for (i = 0; i < ARRAY_SIZE(mc_banks); i++) {
        mc_bank_config = get_memAddrMap(i);

        if (mc_bank_config & MC_CH0_MMAP0_CS_VALID_MASK) {
            mc_banks[i].start = (mc_bank_config & MC_CH0_MMAP0_START_ADDRESS_MASK) | ((uint64_t)(mc_bank_config & MC_CH0_MMAP0_START_ADDRESS_HI_MASK) << 24);
            mc_banks[i].size  = dram_size_from_mask(MC_CH0_MMAP0_AREA_LENGTH_MASK_SHIFT(mc_bank_config));
            if ((i == 0) && (mc_banks[i].start == 0))
                mc_banks[i].size  = (uint64_t)(unsigned long)get_ram_size((long *)mc_banks[i].start, mc_banks[i].size);
        }
    }
}

static void fixup_banks_for_reserved_ram(void)
{
    int curbank_mc = 0;
    int curbank_usable = 0;
    bank_info_t reserved = {
        .start = 0x3fb00000,
        .size  = 0x00500000,
    };

    memset(usable_banks, 0, sizeof(usable_banks));
    for (curbank_mc = 0; curbank_mc < ARRAY_SIZE(mc_banks); curbank_mc++)
    {
        if ((mc_banks[curbank_mc].start <= reserved.start) && 
           ((mc_banks[curbank_mc].start + mc_banks[curbank_mc].size) >= (reserved.start + reserved.size)))
        {
            bank_info_t before, after;

            before.start = mc_banks[curbank_mc].start;
            before.size  = reserved.start - before.start;

            after.start  = reserved.start + reserved.size;
            after.size   = mc_banks[curbank_mc].start + mc_banks[curbank_mc].size - after.start;

            if (before.size > 0)
            {
                usable_banks[curbank_usable].start = before.start;
                usable_banks[curbank_usable].size  = before.size;
                curbank_usable++;
            }
            if (after.size > 0)
            {
                usable_banks[curbank_usable].start = after.start;
                usable_banks[curbank_usable].size  = after.size;
                curbank_usable++;
            }
        }
        else
        {
            usable_banks[curbank_usable].start = mc_banks[curbank_mc].start;
            usable_banks[curbank_usable].size  = mc_banks[curbank_mc].size;
            curbank_usable++;
        }
    }
}

void dram_init_banksize(void)
{
    int i;

    for (i = 0; i < CONFIG_NR_DRAM_BANKS; i++) {
        gd->bd->bi_dram[i].start = usable_banks[i].start;
        gd->bd->bi_dram[i].size  = usable_banks[i].size;
    }
}

int dram_init(void)
{
    int i;

    read_mc_bank_info();
    fixup_banks_for_reserved_ram();

    gd->ram_size = 0;
    for (i = 0; i < ARRAY_SIZE(mc_banks); ++i) {
        if (mc_banks[i].start == 0) {
            gd->ram_size = min(mc_banks[i].size, (3UL * 1024 * 1024 * 1024));
            break;
        }
    }

    return 0;
}

#define CORESIGHT_BASE 0xd4171000
#define GENERIC_TIMER_BASE 0xd40fd000
#define GENERIC_TIMER_BASE_FREQ_ID_OFFSET 0x20
static void enable_generic_timer_counter(void)
{
    int *coresight_addr = (int *)CORESIGHT_BASE;
    int *generic_timer_control_addr = (int *)GENERIC_TIMER_BASE;

    /*
     * On RevA the coresight timestamp generator
     * is the source of the generic timer count.
     * Enable it.
     */
    *coresight_addr = 1;

    /*
     * On RevB a counter in the ap apb is the
     * source of the generic timer count.
     *
     * Set the Base Frequency ID register with the
     * correct reference clock value and then enable it.
     */
    if (isFPGA( )) {
        *(generic_timer_control_addr + GENERIC_TIMER_BASE_FREQ_ID_OFFSET) = 13000000;
    } else {
        *(generic_timer_control_addr + GENERIC_TIMER_BASE_FREQ_ID_OFFSET) = CONFIG_SYS_HZ_CLOCK;
    }
    *generic_timer_control_addr = 1;
}

int board_early_init_f(void)
{
    /*
     * The uart clock should be 11.059 Mhz to reduce the error percentage
     * so use the fractional divider.
     */
    if (!isFPGA( )) {
        volatile unsigned int *uart0_fd = (volatile unsigned int *)(AP_APMU_BASE + 0x530);

        if (is_Gr2_RevA( ))
            uart0_fd = (volatile unsigned int *)(AP_APMU_BASE + 0x300);

        *uart0_fd = 0x90ae8000;
    }

    setup_pin_cfg( );
    power_up_srams( );

    /*
     * The ARM Generic Timers in the Cortex-A53 MPCore
     * need a reference counter input.  Enable it here.
     */
    enable_generic_timer_counter();

    return 0;
}

int board_init(void)
{
    gd->bd->bi_arch_number = 0xffffffff;
    gd->bd->bi_boot_params = CONFIG_SYS_TEXT_BASE + 0x3c00;

    if (isFPGA( )) {
        printf("FPGA Version: %02x:%02x:%02x:%02x\n",
             *(unsigned char *)FPGAA_CONFIG_BASE,
             *(unsigned char *)FPGAB_CONFIG_BASE,
             *(unsigned char *)FPGAC_CONFIG_BASE,
             *(unsigned char *)FPGAD_CONFIG_BASE
             );
    }

    return 0;
}

extern int get_board_suffix(void);
/*
 * Construct the boot command for the specified configuration.
 */
static int mmc_get_bootcmd_for_part(int dev_num, int part, disk_partition_t *pinfo, char *cmd, int len, block_dev_desc_t * dev_desc, int boot_recover) {
    /* If the device uses SSP partitioning then it is using CRAM */
    if (dev_desc->part_type == PART_TYPE_SSP) {
        int partition;
        ulong part_size;
        ulong part_sizeB;
        disk_partition_t pinfoB;
        const uint32_t main_addr = 0x100000;
        uint32_t cramfs_addr;
        const char format2[] = "setenv cramfsaddr %#x;"
                               "mmc dev %d;"
                               "mmc read %#x %#lx %#lx;"
                               "cramfsload %#x /main.img;"
                               "source %#x;"
                               "mmc read %#x %#lx %#lx;"
                               "cramfsload %#x /main.img;"
                               "source %#x;"
                               "loop.l 0xd0000000 1";

        if (strcmp((const char *)pinfo->name, "Kernel") != 0) /* The kernel partition is on partition 2 for both EXT2 and CRAM */
            return 1;

        /* Search for recovery partition */
        for (partition = 1; partition < 35; partition++) {
            if (get_partition_info(dev_desc, partition, &pinfoB) == 0) {
                if (strcmp((const char *)pinfoB.name, "RecoveryKernel") == 0 && pinfoB.data_len != 0) {
                    break;
                }
            }
        }

        /* If there is no recovery partition then just make it the same as the primary */
        if (partition == 35) {
            memcpy(&pinfoB, pinfo, sizeof(disk_partition_t));
        }

        /* Boot from the recovery partition */
        if (boot_recover) {
            memcpy(pinfo, &pinfoB, sizeof(*pinfo));
        }

        /* Load the kernel CRAM partition right in front of U-Boot allowing space for the malloc area */
        cramfs_addr = ((gd->start_addr_sp - max(pinfo->data_len, pinfoB.data_len)) >> 20) << 20;
        part_size = (pinfo->data_len + (dev_desc->blksz - 1)) / dev_desc->blksz;
        part_sizeB = (pinfoB.data_len + (dev_desc->blksz - 1)) / dev_desc->blksz;

        sprintf(cmd, format2, cramfs_addr,
                dev_num,
                cramfs_addr, pinfo->start, part_size,
                main_addr,
                main_addr,
                cramfs_addr, pinfoB.start, part_sizeB,
                main_addr,
                main_addr);
    }
    else {
#if 0  /*  RICOH : change  */
        const uint32_t uImage_addr = 0x400000;
        const uint32_t dtb_addr    = 0xf00000;
        enum board_type board = get_board_type( );
        struct board_info_s board_info;
        char set_eth_addr[50]; /* to hold "setenv ethaddr <eth_address>" */
        char serial_num[50]; /* to hold serial_num=<serial_number> */
        int mmcblk_num = 0;
        const char format[] = "%smmc dev %d;"
                              "ext2load mmc %d:%d %#x /boot/uImage;"
                              "ext2load mmc %d:%d %#x /boot/%s.dtb;"
                              "setenv bootargs $bootargs root=/dev/mmcblk%dp2 uio_pdrv_genirq.of_id=generic-uio rootwait%s;"
                              "bootm %#x - %0#x";

        if (part != 2) /* The kernel partition is on partition 2 for EXT2 */
            return 1;

        set_eth_addr[0] = 0;
        serial_num[0] = 0;

        if (0 == i2c_read_board_info(&board_info)) {
       		int i;
       		char *c;
       		char separator = ':';

       		strcpy(set_eth_addr, "setenv ethaddr ");
           	c = set_eth_addr + strlen(set_eth_addr);
           	for (i=0; i<6; i++, c += 3) {
           		if (5 == i) separator = ';';
           		sprintf(c, "%02x%c", (unsigned char)board_info.eth_mac[i], separator);
        	}

           	if (board_info.serial_num[0]) {
           		sprintf(serial_num, " serial_num=%s", board_info.serial_num);
           	}
        }

        if (dev_num != board_configs[board].onboard_dev)
        {
            if (dev_num < board_configs[board].dc_dev)
                mmcblk_num = 0;
            else
                mmcblk_num = 1;
        }
#else
#ifdef CONFIG_SD_BOOT
		const uint32_t uImage_addr = 0x400000;
		const uint32_t dtb_addr    = 0xf00000;
		char set_eth_addr[50]; /* to hold "setenv ethaddr <eth_address>" */
		char serial_num[50]; /* to hold serial_num=<serial_number> */
        int mmcblk_num = 0;
		const char format[] =
			"run load_img0;"
			"setenv bootargs_emmc $bootargs vt.global_cursor_default=0 loglevel=10 initrd=0x1100000,0x880000 rdinit=/init uio_pdrv_genirq.of_id=generic-uio rootwait coherent_pool=2M maxcpus=1 mem=%dm;"
			"setenv bootargs $bootargs_emmc;"
			"bootm 0x400000 - 0xf00000";
		int memsize = 302;
#else
		int loglevel = 1;
		int upc_lopow = 1;
		{
			int argc = 5;
			char * const argv[] = {"mmc", "read", "0x400000", "0x1000", "4"};
			int repeatable = 0;
			cmd_process(0, argc, argv, &repeatable, NULL);
			loglevel = ((*(unsigned char *)0x400400) == 0) ? 1 : (int)(*(unsigned char *)0x400400);
			if (loglevel > 15) {
				loglevel = 15;
			}
			upc_lopow = ((*(unsigned char *)0x400404) == 0) ? 1 : 0;
		}
		const uint32_t uImage_addr = 0x400000;
		const uint32_t dtb_addr    = 0xf00000;
		char set_eth_addr[50]; /* to hold "setenv ethaddr <eth_address>" */
		char serial_num[50]; /* to hold serial_num=<serial_number> */
        int mmcblk_num = 0;
		int memsize = 302;
		const char format[] =
			"run select_boot;"
			"setenv bootargs_emmc $bootargs vt.global_cursor_default=0 loglevel=%d initrd=0x1100000,0x400000 rdinit=/init uio_pdrv_genirq.of_id=generic-uio rootwait coherent_pool=2M maxcpus=1 mem=%dm upc_lowpow=%d;"
			"setenv bootargs $bootargs_emmc;"
			"bootm 0x400000 - 0xf00000";
		switch (get_board_suffix()) {
		case 1:/*TYPE_CHI_SFP*/
			memsize = 279;
			break;
		case 3:/*TYPE_VES_SFP:*/
			memsize = 303;
			break;
		case 2:/*TYPE_VES_MFP:*/
			memsize = 716;
			break;
		case 0:/*TYPE_CHI_MFP:*/
			memsize = 768;
			break;
		case 4:/*TYPE_CHI_MFP_S:*/
		case 5:/*TYPE_VES_MFP_S:*/
			memsize = 1740;
			break;
		case 6:/*TYPE_KONJIKI_SFP:*/
			memsize = (2048 - 1348 - 33);/* 33MB is R4Area */
			break;
		default:
			break;
		}
#endif  /*  CONFIG_SD_BOOT  */
        sprintf(cmd, format, loglevel, memsize, upc_lopow);
#endif
    }

    return 0;
}

static int sdmmc_get_bootcmd(int dev_num, int boot_recover) {
    block_dev_desc_t *dev_desc = NULL;
    struct mmc *mmc;
    char buf[300];

    /* Initialize needed devices */
    mmc = find_mmc_device(dev_num);
    if (mmc) {
        int rc;
        rc = mmc_init(mmc);
        if (rc)
        {
            return 0;
        }
    }

    setenv_ulong("MMC_RD_BLK", mmc->read_bl_len);
    setenv_ulong("MMC_WR_BLK", mmc->write_bl_len);

    sprintf(buf,"%llx",mmc->capacity);
    setenv("MMC_DEV_SIZE", buf);


    dev_desc = get_dev("mmc", dev_num);
    if (dev_desc) {
        int part;

        /*
         * Found a device -- go scan it for partitions
         * Restrict the search to the first few partitions.
         */
        for (part = 1; part < 5; part++) {
            disk_partition_t info;

            if (get_partition_info(dev_desc, part, &info) == 0) {
                if (info.size || info.data_len) {
                    /* Found a valid partition */
                    if (mmc_get_bootcmd_for_part(dev_num, part, &info, buf, sizeof(buf), dev_desc, boot_recover) == 0) {
                        setenv_ulong("MMC_DEVICE_NUM", dev_num);
                        setenv_ulong("bootCs", dev_num);
                        setenv("bootcmd", buf);
#ifndef FASTBOOT  /*  RICOH : del  */
                        printf("bootcmd: %s\n", buf);
#endif
                        return 1;
                    }
                }
            }
        }
    }

    return 0;
}

static void setup_bootcmd(int onboard, int dc, int ignore_dc, int boot_recover) {
    if(ignore_dc)
        /*
         * Ignore the daughtercard and just try to get the boot
         * command for the onboard
         */
        sdmmc_get_bootcmd(onboard, boot_recover);
    else {
        /*
         * Try getting the bootcmd for the daughtercard, if that
         * fails, try the onboard
         */
        if(0 == sdmmc_get_bootcmd(dc, boot_recover))
            sdmmc_get_bootcmd(onboard, boot_recover);
    }
}

int get_panel_bus(void) {
    enum board_type board;
    board = get_board_type( );
    return board_configs[board].panel_bus;
}

void mv_mmp_reset_panel(void) {
#if 0  /*  msugi : fastboot  */
    int i, panel_reset;
    enum board_type board;

    board = get_board_type( );
    panel_reset = board_configs[board].panel_reset;

    if (panel_reset == -1)
        return;

    gpio_request(panel_reset, "Panel Reset");
    gpio_direction_output(panel_reset, GPIO_LOW);
    mdelay(30);
    gpio_direction_output(panel_reset, GPIO_HIGH);
    for (i=0; i<20; i++)
       mdelay(100);
    gpio_free(panel_reset);
#endif
}

unsigned int board_get_core_freq_hz(void)
{
	static unsigned int core_freq = 0;
	volatile unsigned long *iopad_197 = 0xd401e34c;
	volatile unsigned long *gpio_pdrg = 0xd4019604;
	volatile unsigned long *gpio_plrg = 0xd4019600;

	if (core_freq == 0) {
		*iopad_197 = 0x00018040;
		*gpio_pdrg = 0x00000000;/* set all dir to in */
		udelay(50);
		core_freq = ((*gpio_plrg) & 0x00000020) ? (800 * 1000000) : (1200 * 1000000);
	}

	return core_freq;
}

struct audio_map {
    uint8_t addr;
    uint8_t val;
};

static const struct audio_map audio_settings[] = {
    { 0x00, 0x00 }, { 0x01, 0x01 }, { 0x04, 0x01 }, { 0x1b, 0xc0 },
    { 0x1c, 0x01 }, { 0x40, 0x0c }, { 0x00, 0x01 }, { 0x23, 0x11 },
    { 0x26, 0xff }, { 0x00, 0x00 }, { 0x44, 0x7f }, { 0x45, 0x00 },
    { 0x46, 0xb6 }, { 0x00, 0x01 }, { 0x20, 0x06 }, { 0x2a, 0x00 },
    { 0x22, 0x70 }, { 0x00, 0x00 }, { 0x19, 0x00 }, { 0x1a, 0x01 },
    { 0x33, 0x00 }, { 0x0b, 0x81 }, { 0x0c, 0x81 }, { 0x0d, 0x00 },
    { 0x0e, 0x01 }, { 0x00, 0x01 }, { 0x2a, 0x04 }, { 0x20, 0x86 },
    { 0x26, 0x90 }, { 0x00, 0x00 }, { 0x3f, 0x90 }, { 0x41, 0x20 },
    { 0x42, 0x20 }, { 0x40, 0x04 },
};

static void hdd_enable(void) {
    enum board_type board = get_board_type( );
    int hdd_reset = board_configs[board].hdd_reset;

    if (hdd_reset < 0)
        return;

    gpio_request(hdd_reset, "HDD Reset");
    gpio_direction_output(hdd_reset, GPIO_HIGH);
}

static void isp_enable(void) {
    enum board_type board = get_board_type( );
    int isp_reset = board_configs[board].isp_reset;

    if (isp_reset < 0)
        return;

    gpio_request(isp_reset, "ISP Reset");
    gpio_direction_output(isp_reset, GPIO_LOW);
    mdelay(10);
    gpio_direction_output(isp_reset, GPIO_HIGH);
    gpio_free(isp_reset);
}

static void setup_dac(void) {
    int i, oldbus;
    enum board_type board = get_board_type( );
    int dac_reset = board_configs[board].dac_reset;

    if (dac_reset < 0)
        return;

    /* Reset DAC */
    gpio_request(dac_reset, "DAC Reset");
    gpio_direction_output(dac_reset, GPIO_HIGH);
    mdelay(10);
    gpio_direction_output(dac_reset, GPIO_LOW);
    mdelay(10);
    gpio_direction_output(dac_reset, GPIO_HIGH);

    /* Setup the MIC input to go out */
    oldbus = i2c_get_bus_num( );
    i2c_set_bus_num(CONFIG_AUDIO_I2C_BUS);

    for (i=0; i<ARRAY_SIZE(audio_settings); ++i)
        i2c_write(CONFIG_AUDIO_I2C_ADDR,
                audio_settings[i].addr, sizeof(audio_settings[i].addr),
                (uint8_t *)&audio_settings[i].val, sizeof(audio_settings[i].val));

    i2c_set_bus_num(oldbus);
}

static int handle_recovery(enum board_type board) {
    int value;
    int recovery = board_configs[board].recovery;

    if (recovery == -1)
        return 0;

    gpio_request(recovery, "GR Recovery");
    gpio_direction_input(recovery);

    value = gpio_get_value(recovery);

    if (value)
        setenv_ulong("RECOVERY", 1);

    gpio_free(recovery);

    return !!value;
}

static void por_beep(void) {
    int i;
    enum board_type board = get_board_type( );
    int buzzer = board_configs[board].buzzer;

    if (buzzer < 0)
        return;

    gpio_request(buzzer, "DAC Reset");

    /* 2kHz tone for 100ms */
    for (i=0; i<200; ++i) {
        gpio_direction_output(buzzer, GPIO_HIGH);
        udelay(250);
        gpio_direction_output(buzzer, GPIO_LOW);
        udelay(250);
    }
}

void no_panel_beep_loop(void) {
    int i;

    // 5 1 second beeps (1/2 second on, 1/2 second off), 5 second pause, repeat
    while (1) {
        for (i=5; i; i--) {
            por_beep( );
            mdelay(900);
        }

        mdelay(5000);
    }
}

int board_late_init(void) {
    enum board_type board;
    char buf[300];
    int ignore_dc = 0;
    int boot_recover = 0;
    int i;
    const char *asic_rev = "";

    switch (get_chip_rev( )) {
        case REV_C0:
            asic_rev = "-revc";
            break;
        case REV_B1:
        case REV_B0:
            asic_rev = "-revb";
            break;
        default:
            break;
    }

    board = get_board_type( );
    sprintf(buf, "%s%s", board_configs[board].name, asic_rev);
    setenv("DTB", buf);

    if (board_configs[board].board_late_init)
        board_configs[board].board_late_init( );

    setup_dac( );
    por_beep( );
    hdd_enable( );
    isp_enable( );

    setenv_ulong("DRAM_SIZE", gd->ram_size);
    buf[0] = 0;
    for (i=0; i<ARRAY_SIZE(mc_banks); ++i) {
        sprintf(buf, "%s,%u", buf, (unsigned int)(mc_banks[i].size >> 20));
    }
    setenv("DRAM_CONFIG", &buf[1]); /* Remove leading comma */
    setenv_ulong("CPUMHZ", (unsigned long) board_get_core_freq_hz() / 1000000);
    setenv("CARDREV", "00a");
    setenv("initrd_high", "ffffffff");
    setenv("fdt_high", "ffffffff");

	/*  RICOH : add  */
	setenv("build_date", __DATE__" "__TIME__);

    if (isFPGA( )) {
        char *bootargs = getenv("bootargs");
        sprintf(buf, "%s FPGA=1", bootargs);
        setenv("bootargs", buf);
    }

    boot_recover = handle_recovery(board);

#ifdef CONFIG_MV_MMP_PANEL
    /*
     * Calling  mv_mmp_panel_init() here seems redundant but is harmless.
     * Called earlier during stdio_init()                      .
     * TBD: Is there any case that could bypass stdio_init() and require this call?
     */
    if (board_configs[board].panel_reset != -1) {
        char *keys;
        const char *ptr;

    mv_mmp_panel_init();

    mv_mmp_panel_status(MV_MMP_PANEL_BLANK);

        keys = getenv("PORKEYS");
        if (keys) {
            unsigned int por_keys = simple_strtoul(keys, NULL, 0);
            if (por_keys == NM_HOME)
                ignore_dc = 1;
            else if (por_keys == (NM_2 | NM_7 | NM_8))
                boot_recover = 1;
        }

    setenv_ulong("PANELTYPE_ENV", mv_mmp_panel_get_type());

    ptr = mv_mmp_panel_get_subtype();
        if(ptr)
            setenv("PANELSUBTYPE", (char *)ptr);
    }
#endif

	ignore_dc=1;
    setup_bootcmd(board_configs[board].onboard_dev, board_configs[board].dc_dev, ignore_dc, boot_recover);
    return 0;
}

#ifdef CONFIG_GENERIC_MMC
int board_mmc_init(bd_t *bd)
{
    u32 max_clk = 200000000;
    ulong mmc_base_address[CONFIG_SYS_MMC_NUM] = CONFIG_SYS_MMC_BASE;
    u8 i;

    if (is_Gr2_RevA( ) || isFPGA( ))
        max_clk = 52000000;

    for (i = 0; i < CONFIG_SYS_MMC_NUM; i++) {
        if (mv_sdh_init(mmc_base_address[i], max_clk, 0, SDHCI_QUIRK_WAIT_SEND_CMD))
            return 1;

        writel(CLK_GATE_ON | CLK_GATE_CTL \
               | WTC(WTC_DEF) | RTC(RTC_DEF),
               mmc_base_address[i] + SD_FIFO_PARAM);
    }

    writew(SDCLK_DELAY(0x1f) | SDCLK_SEL | WR_ENDIAN | RD_ENDIAN \
           | DMA_SIZE(DMA_FIFO_128) | BURST_SIZE(BURST_64),
           mmc_base_address[0] + SD_CLOCK_AND_BURST_SIZE_SETUP);

    return 0;
}
#endif

#ifdef CONFIG_CMD_NAND
extern int mv61x0_nand_init(struct nand_chip *nand);
int board_nand_init(struct nand_chip *nand) {
    volatile fGPIO_REGS_t *GPIOB1 = (fGPIO_REGS_t *)FPGAB_GPIO1_BASE;

    if (!nand->IO_ADDR_R)
        return -ENODEV;

    /* Turn off write protection on the Marvell DestinyFLASH NAND board */
    GPIOB1->PDR = GPIOB1->PDR | GPIOB1_TAG1_nWP;
    GPIOB1->PSR = GPIOB1->PSR | GPIOB1_TAG1_nWP;

    return mv61x0_nand_init(nand);
}
#endif

#ifdef CONFIG_OF_BOARD_SETUP
void ft_board_setup(void *blob, bd_t *bd)
{
    uint64_t dram_start[CONFIG_NR_DRAM_BANKS+1], dram_size[CONFIG_NR_DRAM_BANKS+1];
    int i, j = 0;

    for (i = 0; i < CONFIG_NR_DRAM_BANKS; ++i) {
        dram_start[j] = usable_banks[i].start;

        if ((usable_banks[i].start < 0x80000000)
            && ((usable_banks[i].start + usable_banks[i].size) > 0x80000000)) {
            dram_size[j] = 0x80000000 - usable_banks[i].start;
            ++j;

            dram_start[j] = 0x80000000ULL;
            dram_size[j] = usable_banks[i].size - 0x80000000;
        }
        else {
            dram_size[j] = usable_banks[i].size;
        }

        if (dram_start[j] >= 0x80000000)
            dram_start[j] |= ((uint64_t)1 << 35);

        ++j;
    }

    fdt_fixup_memory_banks(blob, dram_start, dram_size, j);
    ft_lcd_setup(blob, bd);
}
#endif
