#include <config.h>

#ifdef CONFIG_CRASH_KERNEL_START_MB
#include <u-boot/md5.h>
#ifdef CONFIG_OF_LIBFDT
#ifdef MV88F78X60
#include <common.h>
#include <fdt_support.h>
#endif
#endif

/*
 * This header is expected to be last full page of the crash kernel's reserved
 * memory region after a watchdog reset.
 *
 * Must match definition in linux/arch/arm/kernel/machine_kexec.c.
 */
struct kexec_on_watchdog_header {
	u32 magic; /* Must be 0xf7d5b391. */
	u16 version; /* Version number for this header. Must be 1. */
	u16 checksum; /* Exclusive or of all 16-bit words in this header. For
			 the purposes of computing the checksum, the value of
			 the checksum field is zero. */
	u64 start_address; /* Entry point into the kernel. */
	u64 mach_type; /* As specified by linux/arch/arm/tools/mach-types. */
	u64 boot_atags; /* Command line for the crash kernel. */
	u64 md5_start; /* First byte to hash. */
	u64 md5_length; /* Number of bytes to hash. */
	unsigned char md5[16]; /* MD5 digest of the crash kernel. */
};

/*
 * Exclusive or of all 16-bit words in header. Returns zero if header is valid.
 */
static u16 kexec_on_watchdog_checksum(struct kexec_on_watchdog_header *header)
{
	u16 checksum = 0;
	size_t i, length;

	length = sizeof(*header) / sizeof(header->checksum);
	for (i = 0; i < length; i++)
		checksum ^= *(((u16 *) header) + i);
	return checksum;
}

static int valid_kexec_on_watchdog_header(
		struct kexec_on_watchdog_header *header,
		uintptr_t crashk_start, uintptr_t crashk_end)
{
	uintptr_t md5_end;

	if (kexec_on_watchdog_checksum(header)) {
		printf("Watchdog header checksum invalid.");
		return 0;
	}
	if (header->start_address < crashk_start ||
			header->start_address > crashk_end) {
		printf("start_address outside of crash kernel "
		       "reserved memory region.\n");
		return 0;
	}
	if (header->boot_atags < crashk_start ||
			header->boot_atags > crashk_end) {
		printf("boot_atags outside of crash kernel "
		       "reserved memory region.\n");
		return 0;
	}
	if (header->md5_start < crashk_start ||
			header->md5_start > crashk_end) {
		printf("md5_start outside of crash kernel "
		       "reserved memory region.\n");
		return 0;
	}
	md5_end = header->md5_start + header->md5_length;
	if (md5_end < crashk_start || md5_end > crashk_end) {
		printf("md5_end outside of crash kernel "
		       "reserved memory region.\n");
		return 0;
	}
	if (header->md5_length > crashk_end - crashk_start) {
		printf("md5_length is larger than the crash kernel "
		       "reserved memory region.\n");
		return 0;
	}
	return 1;
}

/*
 * Check for the presence of the kexec_on_watchdog_header in DRAM. If it's
 * there, assume that we are booting after a watchdog reset, verify the
 * integrity of the header and the crash kernel, then jump to the crash kernel.
 * Otherwise, return and let the normal boot sequence continue.
 */
void kexec_on_watchdog(void) {
	uintptr_t crashk_start =
			CONFIG_CRASH_KERNEL_START_MB * 1024 * 1024;
	uintptr_t crashk_end =
			(CONFIG_CRASH_KERNEL_START_MB +
			 CONFIG_CRASH_KERNEL_SIZE_MB) *
			1024 * 1024 - 1; /* Inclusive */
	/* Last full page of the crash kernel's reserved memory region. */
	struct kexec_on_watchdog_header *header =
			(struct kexec_on_watchdog_header *)
			(((crashk_end + 1) & ~(0xfff)) - 4096);
	unsigned char md5_digest[16];

	if (header->magic != 0xf7d5b391)
		return;

	switch (header->version) {
	case 1:
		if (!valid_kexec_on_watchdog_header(header, crashk_start,
				crashk_end))
			return;

		printf("Watchdog header valid. Verifying crash kernel.\n");
		md5((unsigned char *)(uintptr_t) header->md5_start,
				header->md5_length, md5_digest);
		if (memcmp(header->md5, md5_digest, sizeof(header->md5)) == 0) {
			void (*kernel_entry)(int zero, int arch, uint params);

			printf("Crash kernel verified. "
			       "Booting crash kernel.\n");
			kernel_entry = (void (*)(int, int, uint))
					(uintptr_t) header->start_address;
#ifdef CONFIG_OF_LIBFDT
#ifdef MV88F78X60
			fdt_status_disabled_by_alias((void *)header->boot_atags, "/soc/pcie-controller");
#endif
#endif
			/* Jump to the crash kernel. */
			kernel_entry(0, header->mach_type, header->boot_atags);
		} else {
			printf("Crash kernel integrity check failed.");
		}
		break;
	default:
		printf("Unsupported watchdog header version: [%d]\n",
				header->version);
		break;
	}
}
#endif
