#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/input.h>
#include <linux/i2c.h>
#include <linux/gpio.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/module.h>
#include <linux/device.h>
#include <linux/cdev.h>
#include <linux/fs.h>
#include <linux/poll.h>
#include <linux/timer.h>
#include <linux/platform_device.h>
#include <linux/syscalls.h>
#include <linux/clk.h>
#include <linux/slab.h>
#include <linux/workqueue.h>
#include <linux/kthread.h>
#include <asm/gpio.h>

#include "svinput.h"
#include "sv_touchpanel.h"

extern int i2c_master_sendrecv(const struct i2c_client *client, const char *sndbuf, int sndcount, char *rcvbuf, int rcvcount);
extern int i2c_master_recvsend(const struct i2c_client *client, char *rcvbuf, int rcvcount, const char *sndbuf, int sndcount);
extern unsigned char calc_sum(unsigned char *buf, int sum_pos);

static int sv_pos_info_num = 0;
static int s_touchpanel_is_opened = SVINPUT_FALSE;
static struct i2c_client *s_client = NULL;
static int sv_calib_data_read_num = 0;
static int sv_calib_data_write_num = 0;
tpanel_calib_data_t g_tpanel_carib_data;

static DECLARE_WAIT_QUEUE_HEAD(tp_wait_queue);
static DECLARE_WAIT_QUEUE_HEAD(tp_calib_wait_queue);

static sv_tpanel_info_t cur_pos = {
	.pos_x = 0,
	.pos_y = 0,
	.press = 0,
	.info = 0,
};

static struct timeval tv = {
	.tv_sec = 0,
	.tv_usec = 0,
};

extern int is_ctl_ready;

void touchpanel_wqueue_handler(work_struct_t *work, svinput_slv_tp_cmd_t *slv_tp_cmd)
{
	svinput_work_t *ptr_work = NULL;
	static unsigned short prev_touchpanel_status = 0;

	ptr_work = (svinput_work_t*)work;

	do_gettimeofday(&tv);
	
	++(ptr_work->touch_cnt);
	if (!is_ctl_ready) {
		DPRINTK("Not Ready CTL. (%ld.%06ld)Touch count=>%d\n", tv.tv_sec, tv.tv_usec, ptr_work->touch_cnt);
		return;
	} else {
		DPRINTK("(%ld.%06ld)Touch count=>%d\n", tv.tv_sec, tv.tv_usec, ptr_work->touch_cnt);
	}
	
	cur_pos.press = (slv_tp_cmd->pres) ? 1 : 0;
	/* 押下イベントの場合にだけ座標を更新する */
	if (cur_pos.press) {
		cur_pos.pos_x = ((slv_tp_cmd->xh << 8) & 0xff00) | ((slv_tp_cmd->xl) & 0x00ff);
		cur_pos.pos_y = ((slv_tp_cmd->yh << 8) & 0xff00) | ((slv_tp_cmd->yl) & 0x00ff);
	}
	cur_pos.info = slv_tp_cmd->calib;
	/* 誰かが開いてる時 かつ 前回とステータスが異なる時 だけアップデート */
	if (s_touchpanel_is_opened) {
		if (cur_pos.press == prev_touchpanel_status){
			DPRINTK(KERN_EMERG "the same touchpanel status: pos_x=%d, pos_y=%d, press=%d, info=%d \n", cur_pos.pos_x, cur_pos.pos_y, cur_pos.press, cur_pos.info);
			return;
		}
		prev_touchpanel_status = cur_pos.press;
		sv_pos_info_num++;
		wake_up_interruptible(&tp_wait_queue);
	} else {
		sv_pos_info_num = 0;
	}

	return;
}

void touchpanel_wqueue_calib_handler(work_struct_t *work, uint8_t *_recv_packet)
{
	unsigned int cmd = _recv_packet[1];
	
	if (EVT_TPANEL_CALIB_REQ == cmd)
	{
		svinput_calib_cmd_t *calib_cmd = (svinput_calib_cmd_t*)_recv_packet;
		DPRINTK("calib_cmd->hdr   = %x \n", calib_cmd->hdr);
		DPRINTK("calib_cmd->cmd   = %x \n", calib_cmd->cmd);
		DPRINTK("carib_cmd->len   = %d \n", calib_cmd->len);
		DPRINTK("calib_cmd->ad1xh = %d \n", calib_cmd->ad1xh);
		DPRINTK("calib_cmd->ad1xl = %d \n", calib_cmd->ad1xl);
		DPRINTK("calib_cmd->ad1yh = %d \n", calib_cmd->ad1yh);
		DPRINTK("calib_cmd->ad1yl = %d \n", calib_cmd->ad1yl);
		DPRINTK("calib_cmd->ad2xh = %d \n", calib_cmd->ad2xh);
		DPRINTK("calib_cmd->ad2xl = %d \n", calib_cmd->ad2xl);
		DPRINTK("calib_cmd->ad2yh = %d \n", calib_cmd->ad2yh);
		DPRINTK("calib_cmd->ad2yl = %d \n", calib_cmd->ad2yl);
		DPRINTK("calib_cmd->ad3xh = %d \n", calib_cmd->ad3xh);
		DPRINTK("calib_cmd->ad3xl = %d \n", calib_cmd->ad3xl);
		DPRINTK("calib_cmd->ad3yh = %d \n", calib_cmd->ad3yh);
		DPRINTK("calib_cmd->ad3yl = %d \n", calib_cmd->ad3yl);
		DPRINTK("calib_cmd->ad4xh = %d \n", calib_cmd->ad4xh);
		DPRINTK("calib_cmd->ad4xl = %d \n", calib_cmd->ad4xl);
		DPRINTK("calib_cmd->ad4yh = %d \n", calib_cmd->ad4yh);
		DPRINTK("calib_cmd->ad4yl = %d \n", calib_cmd->ad4yl);
		DPRINTK("calib_cmd->ad5xh = %d \n", calib_cmd->ad5xh);
		DPRINTK("calib_cmd->ad5xl = %d \n", calib_cmd->ad5xl);
		DPRINTK("calib_cmd->ad5yh = %d \n", calib_cmd->ad5yh);
		DPRINTK("calib_cmd->ad5yl = %d \n", calib_cmd->ad5yl);
		
		g_tpanel_carib_data.calib      = 0;          /* 補正要求 1:補正前データを要求, 0:補正後データを要求. 書き込み時は RET値が入る */
		g_tpanel_carib_data.ad_data[0] = ((calib_cmd->ad1xh << 8) & 0xff00) | ((calib_cmd->ad1xl << 0) & 0x00ff);	/* [0]: AD1 x座標 */
		g_tpanel_carib_data.ad_data[1] = ((calib_cmd->ad1yh << 8) & 0xff00) | ((calib_cmd->ad1yl << 0) & 0x00ff);	/* [1]: AD1 y座標 */
		g_tpanel_carib_data.ad_data[2] = ((calib_cmd->ad2xh << 8) & 0xff00) | ((calib_cmd->ad2xl << 0) & 0x00ff);	/* [2]: AD2 x座標 */
		g_tpanel_carib_data.ad_data[3] = ((calib_cmd->ad2yh << 8) & 0xff00) | ((calib_cmd->ad3yl << 0) & 0x00ff);	/* [3]: AD2 y座標 */
		g_tpanel_carib_data.ad_data[4] = ((calib_cmd->ad3xh << 8) & 0xff00) | ((calib_cmd->ad3xl << 0) & 0x00ff);	/* [4]: AD3 x座標 */
		g_tpanel_carib_data.ad_data[5] = ((calib_cmd->ad3yh << 8) & 0xff00) | ((calib_cmd->ad3yl << 0) & 0x00ff);	/* [5]: AD3 y座標 */
		g_tpanel_carib_data.ad_data[6] = ((calib_cmd->ad4xh << 8) & 0xff00) | ((calib_cmd->ad4xl << 0) & 0x00ff);	/* [6]: AD4 x座標 */
		g_tpanel_carib_data.ad_data[7] = ((calib_cmd->ad4yh << 8) & 0xff00) | ((calib_cmd->ad4yl << 0) & 0x00ff);	/* [7]: AD4 y座標 */
		g_tpanel_carib_data.ad_data[8] = ((calib_cmd->ad5xh << 8) & 0xff00) | ((calib_cmd->ad5xl << 0) & 0x00ff);	/* [8]: AD5 x座標 */
		g_tpanel_carib_data.ad_data[9] = ((calib_cmd->ad5yh << 8) & 0xff00) | ((calib_cmd->ad5yl << 0) & 0x00ff);	/* [9]: AD5 y座標 */
		
		svinput_wake_up_interruptible(&tp_calib_wait_queue, &sv_calib_data_read_num);
	}
	if (EVT_TPANEL_CALIB_RET == cmd)
	{
		g_tpanel_carib_data.calib      = _recv_packet[3];
		svinput_wake_up_interruptible(&tp_calib_wait_queue, &sv_calib_data_write_num);
	}
	
	return;
}

int touchpanel_open(struct inode *inode, struct file *file)
{
	int rc = 0;
	FUNC_START;
	s_touchpanel_is_opened = SVINPUT_TRUE;
	FUNC_END;
	return rc;
}

int touchpanel_release(struct inode *inode, struct file *file)
{
	int rc = 0;
	FUNC_START;
	/* NOP touchpanelのreleaseが必要なケースはない。(外部のlibからreleaseされる事はある) */
	FUNC_END;
	return rc;
}

void touchpanel_probe(struct i2c_client *client, const struct i2c_device_id *idp)
{
	s_client = client;
	return;
}

ssize_t touchpanel_read(struct file *file, char *buffer, size_t count, loff_t *ppos)
{
	int rc = 0;

	FUNC_START;
	wait_event_interruptible(tp_wait_queue, (sv_pos_info_num != 0));
	rc = copy_to_user(buffer, &cur_pos, sizeof(sv_tpanel_info_t));
	DPRINTK("(%ld.%06ld)pos_x=%d, pos_y=%d, press=%d, info=%d \n", tv.tv_sec, tv.tv_usec, cur_pos.pos_x, cur_pos.pos_y, cur_pos.press, cur_pos.info);
	
	count = (rc == 0) ? sizeof(sv_tpanel_info_t) : 0;
	sv_pos_info_num--;
	if (sv_pos_info_num < 0) {
		sv_pos_info_num = 0;
	}
	FUNC_END;

	return count;
}

unsigned int touchpanel_poll(struct file *file, poll_table * wait)
{
	unsigned int rc = 0;

	FUNC_START;
	poll_wait(file, &tp_wait_queue, wait);

	if (sv_pos_info_num != 0) {
		rc = (POLLIN | POLLRDNORM);
	}

	FUNC_END;
	return rc;
}

long touchpanel_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	long rc = 0xffffffff;
	char recv_buf[I2C_RECV_BUF_SIZE] = {0};
	tpanel_calib_data_t tpanel_calib_data;
	int sum;

	svinput_calib_read_request_cmd_t calib_read_req = {
		.hdr = ST1,
		.cmd = 0x61,
		.len = 0,
		.sum = 0x61,
	};
	svinput_calib_cmd_t calib_read_res ={
		.hdr = ST2,
		.cmd = 0x62,
		.len = 20,
	};
	svinput_calib_cmd_t calib_write_req ={
		.hdr = ST1,
		.cmd = 0x63,
		.len = 20,
	};
	svinput_calib_write_response_cmd_t calib_write_res = {
		.hdr = ST2,
		.cmd = 0x64,
		.len = 1,
	};
	svinput_calib_request_cmd_t calib_req = {
		.hdr = ST1,
		.cmd = 0x65,
		.len = 1,
	};

	FUNC_START;

	if (!s_client) {
		rc = -1;
		goto EXIT_FUNC;
	}

	if ( !((cmd == CTLOPE_CALIB_READ) || (cmd == CTLOPE_CALIB_WRITE) || (cmd == CTLOPE_CALIB_REQUEST)) ) {
		rc = copy_from_user(&tpanel_calib_data, (void __user *)arg, sizeof(tpanel_calib_data_t));
		if (rc != 0) {
			rc = -1;
			goto EXIT_FUNC;
		}
	}

	switch (cmd) {
	/* 補正値の読み出し */
	case CTLOPE_CALIB_READ:
		DPRINTK(VT100_RED"CTLOPE_CALIB_READ\n"VT100_NORM);
		memset(&g_tpanel_carib_data, 0x00, sizeof(tpanel_calib_data_t));

		calib_read_req.hdr   = ST1;
		calib_read_req.cmd   = 0x61;
		calib_read_req.len   = 0;
		rc = i2c_master_sendrecv(s_client, (char*)(&calib_read_req), sizeof(calib_read_req), recv_buf, sizeof(recv_buf));
		if (rc == -1) {
			rc = -1;
			FORCE_DPRINTK(KERN_NOTICE "[svinput]Can't send balib_read command (-_-;\n");
			goto EXIT_FUNC;
		}
		if (recv_buf[0] != PAK) {
			rc = -1;
			FORCE_DPRINTK(KERN_NOTICE "[svinput]Can't recv slave ack(-_-;\n");
			goto EXIT_FUNC;
		}
		if(rc == 0) {
			int ret = 0;
			long timeout = HZ;
			ret = svinput_wait_event_interruptible_timeout(&tp_calib_wait_queue, &sv_calib_data_read_num , timeout);
			if (0 == ret) {
				/* timeout */
				FORCE_DPRINTK(KERN_NOTICE "[svinput]wait_event_interruptible_timeout Timeout! \n");
				rc = -1;
				goto EXIT_FUNC;
			} else {
				DPRINTK("g_tpanel_carib_data.calib      = %d \n", g_tpanel_carib_data.calib);
				DPRINTK("g_tpanel_carib_data.ad_data[0] = %d \n", g_tpanel_carib_data.ad_data[0]);
				DPRINTK("g_tpanel_carib_data.ad_data[1] = %d \n", g_tpanel_carib_data.ad_data[1]);
				DPRINTK("g_tpanel_carib_data.ad_data[2] = %d \n", g_tpanel_carib_data.ad_data[2]);
				DPRINTK("g_tpanel_carib_data.ad_data[3] = %d \n", g_tpanel_carib_data.ad_data[3]);
				DPRINTK("g_tpanel_carib_data.ad_data[4] = %d \n", g_tpanel_carib_data.ad_data[4]);
				DPRINTK("g_tpanel_carib_data.ad_data[5] = %d \n", g_tpanel_carib_data.ad_data[5]);
				DPRINTK("g_tpanel_carib_data.ad_data[6] = %d \n", g_tpanel_carib_data.ad_data[6]);
				DPRINTK("g_tpanel_carib_data.ad_data[7] = %d \n", g_tpanel_carib_data.ad_data[7]);
				DPRINTK("g_tpanel_carib_data.ad_data[8] = %d \n", g_tpanel_carib_data.ad_data[8]);
				DPRINTK("g_tpanel_carib_data.ad_data[9] = %d \n", g_tpanel_carib_data.ad_data[9]);
			}
			rc = copy_to_user((void __user *)arg, (const void *)&g_tpanel_carib_data, sizeof(tpanel_calib_data_t));
			if (rc != 0) {
				rc = -1;
				FORCE_DPRINTK(KERN_NOTICE "[svinput]Can't copy to user = %d  \n", rc);
				goto EXIT_FUNC;
			}
		}
		break;
			
	/* 補正値の書込み */
	case CTLOPE_CALIB_WRITE:
		DPRINTK(VT100_RED"CTLOPE_CALIB_WRITE\n"VT100_NORM);
		rc = copy_from_user(&tpanel_calib_data, (void __user *)arg, sizeof(tpanel_calib_data_t));
		if (rc != 0) {
			rc = -1;
			goto EXIT_FUNC;
		}
			
		calib_write_req.hdr   = ST1;
		calib_write_req.cmd   = 0x63;
		calib_write_req.len   = 20;
		calib_write_req.ad1xh = (tpanel_calib_data.ad_data[0] >> 8) & 0x00ff;
		calib_write_req.ad1xl = (tpanel_calib_data.ad_data[0] >> 0) & 0x00ff;
		calib_write_req.ad1yh = (tpanel_calib_data.ad_data[1] >> 8) & 0x00ff;
		calib_write_req.ad1yl = (tpanel_calib_data.ad_data[1] >> 0) & 0x00ff;
		calib_write_req.ad2xh = (tpanel_calib_data.ad_data[2] >> 8) & 0x00ff;
		calib_write_req.ad2xl = (tpanel_calib_data.ad_data[2] >> 0) & 0x00ff;
		calib_write_req.ad2yh = (tpanel_calib_data.ad_data[3] >> 8) & 0x00ff;
		calib_write_req.ad2yl = (tpanel_calib_data.ad_data[3] >> 0) & 0x00ff;
		calib_write_req.ad3xh = (tpanel_calib_data.ad_data[4] >> 8) & 0x00ff;
		calib_write_req.ad3xl = (tpanel_calib_data.ad_data[4] >> 0) & 0x00ff;
		calib_write_req.ad3yh = (tpanel_calib_data.ad_data[5] >> 8) & 0x00ff;
		calib_write_req.ad3yl = (tpanel_calib_data.ad_data[5] >> 0) & 0x00ff;
		calib_write_req.ad4xh = (tpanel_calib_data.ad_data[6] >> 8) & 0x00ff;
		calib_write_req.ad4xl = (tpanel_calib_data.ad_data[6] >> 0) & 0x00ff;
		calib_write_req.ad4yh = (tpanel_calib_data.ad_data[7] >> 8) & 0x00ff;
		calib_write_req.ad4yl = (tpanel_calib_data.ad_data[7] >> 0) & 0x00ff;
		calib_write_req.ad5xh = (tpanel_calib_data.ad_data[8] >> 8) & 0x00ff;
		calib_write_req.ad5xl = (tpanel_calib_data.ad_data[8] >> 0) & 0x00ff;
		calib_write_req.ad5yh = (tpanel_calib_data.ad_data[9] >> 8) & 0x00ff;
		calib_write_req.ad5yl = (tpanel_calib_data.ad_data[9] >> 0) & 0x00ff;
		calib_write_req.sum   = calc_sum((char*)(&calib_write_req), sizeof(calib_write_req) - 1);
			
		DPRINTK("calib_write_req.hdr   = %x \n", calib_write_req.hdr);
		DPRINTK("calib_write_req.cmd   = %x \n", calib_write_req.cmd);
		DPRINTK("calib_write_req.len   = %d \n", calib_write_req.len);
		DPRINTK("calib_write_req.ad1xh = %d \n", calib_write_req.ad1xh);
		DPRINTK("calib_write_req.ad1xl = %d \n", calib_write_req.ad1xl);
		DPRINTK("calib_write_req.ad1yh = %d \n", calib_write_req.ad1yh);
		DPRINTK("calib_write_req.ad1yl = %d \n", calib_write_req.ad1yl);
		DPRINTK("calib_write_req.ad2xh = %d \n", calib_write_req.ad2xh);
		DPRINTK("calib_write_req.ad2xl = %d \n", calib_write_req.ad2xl);
		DPRINTK("calib_write_req.ad2yh = %d \n", calib_write_req.ad2yh);
		DPRINTK("calib_write_req.ad2yl = %d \n", calib_write_req.ad2yl);
		DPRINTK("calib_write_req.ad3xh = %d \n", calib_write_req.ad3xh);
		DPRINTK("calib_write_req.ad3xl = %d \n", calib_write_req.ad3xl);
		DPRINTK("calib_write_req.ad3yh = %d \n", calib_write_req.ad3yh);
		DPRINTK("calib_write_req.ad3yl = %d \n", calib_write_req.ad3yl);
		DPRINTK("calib_write_req.ad4xh = %d \n", calib_write_req.ad4xh);
		DPRINTK("calib_write_req.ad4xl = %d \n", calib_write_req.ad4xl);
		DPRINTK("calib_write_req.ad4yh = %d \n", calib_write_req.ad4yh);
		DPRINTK("calib_write_req.ad4yl = %d \n", calib_write_req.ad4yl);
		DPRINTK("calib_write_req.ad5xh = %d \n", calib_write_req.ad5xh);
		DPRINTK("calib_write_req.ad5xl = %d \n", calib_write_req.ad5xl);
		DPRINTK("calib_write_req.ad5yh = %d \n", calib_write_req.ad5yh);
		DPRINTK("calib_write_req.ad5yl = %d \n", calib_write_req.ad5yl);
		DPRINTK("calib_write_req.sum   = %d \n", calib_write_req.sum);
			
		rc = i2c_master_sendrecv(s_client, (char*)(&calib_write_req), sizeof(calib_write_req), recv_buf, sizeof(recv_buf));
		if (rc == -1) {
			FORCE_DPRINTK(KERN_NOTICE "[svinput]Can't send calib_write command (-_-;\n");
			rc = -1;
			goto EXIT_FUNC;
		}
		if (recv_buf[0] != PAK) {
			FORCE_DPRINTK(KERN_NOTICE "[svinput]Can't recv slave ack(-_-;\n");
			rc = -1;
			goto EXIT_FUNC;
		}
		if(rc == 0) {
			int ret = 0;
			long timeout = HZ;
			ret = svinput_wait_event_interruptible_timeout(&tp_calib_wait_queue, &sv_calib_data_write_num, timeout);
			if (0 == ret) {
				/* timeout */
				FORCE_DPRINTK(KERN_NOTICE "[svinput]Can't send balib_read command (-_-;\n");
				rc = -1;
				goto EXIT_FUNC;
			} else {
				DPRINTK("g_tpanel_carib_data.ret        = %d \n", g_tpanel_carib_data.calib);
			}
			rc = copy_to_user((void __user *)&((tpanel_calib_data_t *)arg)->calib, (const void *)&g_tpanel_carib_data.calib, sizeof(g_tpanel_carib_data.calib));
			if (rc != 0) {
				FORCE_DPRINTK(KERN_NOTICE "[svinput]Can't copy to user = %d  \n", rc);
				rc = -1;
				goto EXIT_FUNC;
			}
		}
		break;
			
	/* 補正要求 */
	case CTLOPE_CALIB_REQUEST:
		DPRINTK(VT100_RED"CTLOPE_CALIB_REQUEST\n"VT100_NORM);
		rc = copy_from_user(&tpanel_calib_data, (void __user *)arg, sizeof(tpanel_calib_data_t));
		if (rc != 0) {
			rc = -1;
			goto EXIT_FUNC;
		}
			
		calib_req.hdr   = ST1;
		calib_req.cmd   = 0x65;
		calib_req.len   = 1;
		calib_req.calib = tpanel_calib_data.calib;
		calib_req.sum   = calc_sum((char*)(&calib_req), sizeof(calib_req) - 1);
		DPRINTK("calib_req.calib = %d \n", calib_req.calib);
			
		rc = i2c_master_sendrecv(s_client, (char*)(&calib_req), sizeof(calib_req), recv_buf, sizeof(recv_buf));
		if (rc == -1) {
			FORCE_DPRINTK(KERN_NOTICE "[svinput]Can't calib_req command (-_-;\n");
			rc = -1;
			goto EXIT_FUNC;
		}
		if (recv_buf[0] != PAK) {
			FORCE_DPRINTK(KERN_NOTICE "[svinput]Can't recv slave ack(-_-;\n");
			rc = -1;
			goto EXIT_FUNC;
		}
		break;
			
	default:
		DPRINTK("Unknown command => 0x%02x\n", cmd);
		break;
	}

 EXIT_FUNC:
	FUNC_END;
	return rc;
}

int mfpope_tpanel_calib_read_response(void *p, int len)
{
	
}

