#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/input.h>
#include <linux/i2c.h>
#include <linux/gpio.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/module.h>
#include <linux/device.h>
#include <linux/cdev.h>
#include <linux/fs.h>
#include <linux/poll.h>
#include <linux/timer.h>
#include <linux/platform_device.h>
#include <linux/syscalls.h>
#include <linux/clk.h>
#include <linux/slab.h>
#include <linux/workqueue.h>
#include <linux/kthread.h>
#include <asm/gpio.h>

#include "svinput.h"
#include "sv_buzzer.h"
//#include "ctlope_buzzer.h"

extern int i2c_master_sendrecv(const struct i2c_client *client, const char *sndbuf, int sndcount, char *rcvbuf, int rcvcount);
extern int i2c_master_recvsend(const struct i2c_client *client, char *rcvbuf, int rcvcount, const char *sndbuf, int sndcount);
extern unsigned char calc_sum(unsigned char *buf, int sum_pos);

static int s_buzzer_is_opened = SVINPUT_FALSE;
static struct i2c_client *s_client = NULL;
static buzzer_info_t s_buzzer_info;

static struct timeval tv = {
	.tv_sec = 0,
	.tv_usec = 0,
};

int buzzer_open(struct inode *inode, struct file *file)
{
	int rc = 0;
	s_buzzer_is_opened = SVINPUT_TRUE;
	return rc;
}

int buzzer_release(struct inode *inode, struct file *file)
{
	int rc = 0;
	s_buzzer_is_opened = SVINPUT_FALSE;
	return rc;
}

void buzzer_probe(struct i2c_client *client, const struct i2c_device_id *idp)
{
	s_client = client;
	return;
}

long buzzer_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	long rc = 0xffffffff;
	char recv_buf[I2C_RECV_BUF_SIZE] = {0};
	buzzer_info_t buz_info;
	buz_args_t buz_args;
	int sum;

	svinput_mas_buz_off_cmd_t buz_off = {
		.hdr = ST1,
		.cmd = 0x20,
		.len = 0,
		.sum = 0x20,
	};
	svinput_mas_buz_on_cmd_t buz_on = {
		.hdr = ST1,
		.cmd = 0x21,
		.len = 0,
		.sum = 0x21,
	};
	svinput_mas_buz_level_cmd_t buz_level = {
		.hdr = ST1,
		.cmd = 0x22,
		.len = 1,
	};	
	svinput_mas_buz_id_on_cmd_t buz_id_on = {
		.hdr = ST1,
		.cmd = 0x24,
		.len = 1,
	};

	svinput_mas_buz_org_ptn_cmd_t buz_org_ptn = {
		.hdr = ST1,
		.cmd = 0x26,
		.len = 11,
	};

//	FUNC_START;
	do_gettimeofday(&tv);
	
	if (!s_client) {
		rc = -1;
		goto EXIT_FUNC;
	}

	if ((cmd != BZ_IOC_BUZZER_OFF) && (cmd != BZ_IOC_BUZZER_ON) && (cmd != BZ_IOC_GET_BUZZER_INFO) && (cmd != BZ_IOC_BUZ_ORG_PTN)) {
		rc = copy_from_user(&buz_info, (void __user *)arg, sizeof(buzzer_info_t));
		if (rc != 0) {
			rc = -1;
			goto EXIT_FUNC;
		}
	}

	switch (cmd) {
	/* ブザーOFF */
	case BZ_IOC_BUZZER_OFF:
		DPRINTK("(%ld.%06ld)BZ_IOC_BUZZER_OFF\n", tv.tv_sec, tv.tv_usec);
		rc = i2c_master_sendrecv(s_client, (char*)(&buz_off), sizeof(buz_off), recv_buf, sizeof(recv_buf));
		if (rc == -1) {
			FORCE_DPRINTK(KERN_NOTICE "[svinput]Can't send buzzer command (-_-;\n");
		}
		if (recv_buf[0] != PAK) {
			FORCE_DPRINTK(KERN_NOTICE "[svinput]Can't recv slave ack(-_-;\n");
		}
		if(rc == 0){
			s_buzzer_info.state = BUZZER_OFF;
		}
		break;
	/* ブザーON */
	case BZ_IOC_BUZZER_ON:
		DPRINTK("(%ld.%06ld)BZ_IOC_BUZZER_ON\n", tv.tv_sec, tv.tv_usec);
		rc = i2c_master_sendrecv(s_client, (char*)(&buz_on), sizeof(buz_on), recv_buf, sizeof(recv_buf));
		if (rc == -1) {
			FORCE_DPRINTK(KERN_NOTICE "[svinput]Can't send buzzer command (-_-;\n");
		}
		if (recv_buf[0] != PAK) {
			FORCE_DPRINTK(KERN_NOTICE "[svinput]Can't recv slave ack(-_-;\n");
		}
		if(rc == 0){
			s_buzzer_info.state = BUZZER_ON;
		}
		break;
	/* ブザー・レベル制御 */
	case BZ_IOC_SET_VOLUME:
		DPRINTK("(%ld.%06ld)BZ_IOC_SET_VOLUME: %d\n", tv.tv_sec, tv.tv_usec, buz_info.level);
		if (buz_info.level < 0) {
			rc = -1;
			goto EXIT_FUNC;
		}
		if(buz_info.level >= BUZZER_LEVEL_MAX){
			buz_info.level = BUZZER_LEVEL_MAX;
		}
		buz_level.level = buz_info.level;
		buz_level.sum = ((buz_level.cmd + buz_level.len + buz_level.level) & 0x00ff);
		rc = i2c_master_sendrecv(s_client, (char*)(&buz_level), sizeof(buz_level), recv_buf, sizeof(recv_buf));
		if (rc == -1) {
			FORCE_DPRINTK(KERN_NOTICE "[svinput]Can't send buzzer command (-_-;\n");
		}
		if (recv_buf[0] != PAK) {
			FORCE_DPRINTK(KERN_NOTICE "[svinput]Can't recv slave ack(-_-;\n");
		}
		if(rc == 0){
			s_buzzer_info.level = buz_info.level;
		}
		break;
	/* 指定ブザーパターンでブザーON */
	case BZ_IOC_BUZ_ID_ON:
		DPRINTK("(%ld.%06ld)BZ_IOC_BUZ_ID_ON: 0x%02x\n", tv.tv_sec, tv.tv_usec, buz_info.id);
		if (buz_info.id < 0) {
			rc = -1;
			goto EXIT_FUNC;
		}
		else if(buz_info.id > BUZZER_ID_MAX){
			FORCE_DPRINTK(KERN_NOTICE "[svinput]Invalid argment\n");
		}
		buz_id_on.id = buz_info.id;
		buz_id_on.sum = ((buz_id_on.cmd + buz_id_on.len + buz_id_on.id) & 0x00ff);
		rc = i2c_master_sendrecv(s_client, (char*)(&buz_id_on), sizeof(buz_id_on), recv_buf, sizeof(recv_buf));
		if (rc == -1) {
			FORCE_DPRINTK(KERN_NOTICE "[svinput]Can't send buzzer command (-_-;\n");
			goto EXIT_FUNC;
		}
		else if(rc == 0){
			if (recv_buf[0] != PAK) {
				FORCE_DPRINTK(KERN_NOTICE "[svinput]Can't recv slave ack(-_-;\n");
				rc = -1;
				goto EXIT_FUNC;
			}
			s_buzzer_info.state = BUZZER_ON;
			s_buzzer_info.id = buz_info.id;
		}
		break;
	/* ブザーの状態情報取得 */
	case BZ_IOC_GET_BUZZER_INFO:
	//	DPRINTK("(%ld.%06ld)BZ_IOC_GET_BUZZER_INFO\n", tv.tv_sec, tv.tv_usec);
		rc = copy_to_user((void __user *)arg, &s_buzzer_info, sizeof(buzzer_info_t));
		if (rc != 0) {
			FORCE_DPRINTK(KERN_NOTICE "[svinput]Can't get buzzer_info(-_-;\n");
			rc = -1;
			goto EXIT_FUNC;
		}
		break;
	/* オリジナル・ブザー・パターン */
	case BZ_IOC_BUZ_ORG_PTN:
		DPRINTK("(%ld.%06ld)BZ_IOC_BUZ_ORG_PTN\n", tv.tv_sec, tv.tv_usec);
		rc = copy_from_user(&buz_args, (void __user *)arg, sizeof(buz_args_t));
		if (rc != 0) {
			rc = -1;
			goto EXIT_FUNC;
		}
		if ((buz_args.id != 12) && (buz_args.id != 13)) {
			DPRINTK("Invalid id => 0x%02x\n", buz_args.id);
			rc = -1;
			goto EXIT_FUNC;
		}
		buz_org_ptn.id		=	 buz_args.id;
		buz_org_ptn.m 		=	 buz_args.m;
		buz_org_ptn.level	=	 buz_args.level;
		buz_org_ptn.bptn0	=	 buz_args.bptn0;
		buz_org_ptn.bptn1	=	 buz_args.bptn1;
		buz_org_ptn.bptn2	=	 buz_args.bptn2;
		buz_org_ptn.bptn3	=	 buz_args.bptn3;
		buz_org_ptn.bptn4	=	 buz_args.bptn4;
		buz_org_ptn.bptn5	=	 buz_args.bptn5;
		buz_org_ptn.bptn6	=	 buz_args.bptn6;
		buz_org_ptn.bptn7	=	 buz_args.bptn7;
		sum 				=	 calc_sum((char*)(&buz_org_ptn), 13);
		buz_org_ptn.sum 	=	 (sum & 0x00ff);
		rc = i2c_master_sendrecv(s_client, (char*)(&buz_org_ptn), sizeof(buz_org_ptn), recv_buf, sizeof(recv_buf));
		if (rc == -1) {
			FORCE_DPRINTK(KERN_NOTICE "[svinput]Can't send buzzer command (-_-;\n");
		}
		else if(rc == 0){
			if (recv_buf[0] != PAK) {
				FORCE_DPRINTK(KERN_NOTICE "[svinput]Can't recv slave ack(-_-;\n");
				FORCE_DPRINTK(KERN_EMERG "[svinput]hdr=0x%02x\n", buz_org_ptn.hdr);
				FORCE_DPRINTK(KERN_EMERG "[svinput]cmd=0x%02x\n", buz_org_ptn.cmd);
				FORCE_DPRINTK(KERN_EMERG "[svinput]len=0x%02x\n", buz_org_ptn.len);
				FORCE_DPRINTK(KERN_EMERG "[svinput]id=0x%02x\n", buz_org_ptn.id);
				FORCE_DPRINTK(KERN_EMERG "[svinput]m=0x%02x\n", buz_org_ptn.m);
				FORCE_DPRINTK(KERN_EMERG "[svinput]level=0x%02x\n", buz_org_ptn.level);
				FORCE_DPRINTK(KERN_EMERG "[svinput]bptn0=0x%02x\n", buz_org_ptn.bptn0);
				FORCE_DPRINTK(KERN_EMERG "[svinput]bptn1=0x%02x\n", buz_org_ptn.bptn1);
				FORCE_DPRINTK(KERN_EMERG "[svinput]bptn2=0x%02x\n", buz_org_ptn.bptn2);
				FORCE_DPRINTK(KERN_EMERG "[svinput]bptn3=0x%02x\n", buz_org_ptn.bptn3);
				FORCE_DPRINTK(KERN_EMERG "[svinput]bptn4=0x%02x\n", buz_org_ptn.bptn4);
				FORCE_DPRINTK(KERN_EMERG "[svinput]bptn5=0x%02x\n", buz_org_ptn.bptn5);
				FORCE_DPRINTK(KERN_EMERG "[svinput]bptn6=0x%02x\n", buz_org_ptn.bptn6);
				FORCE_DPRINTK(KERN_EMERG "[svinput]bptn7=0x%02x\n", buz_org_ptn.bptn7);
				FORCE_DPRINTK(KERN_EMERG "[svinput]sum=0x%02x\n", buz_org_ptn.sum);
			}
			else{
				FORCE_DPRINTK(KERN_EMERG "[svinput]hdr=0x%02x\n", buz_org_ptn.hdr);
				FORCE_DPRINTK(KERN_EMERG "[svinput]cmd=0x%02x\n", buz_org_ptn.cmd);
				FORCE_DPRINTK(KERN_EMERG "[svinput]len=0x%02x\n", buz_org_ptn.len);
				FORCE_DPRINTK(KERN_EMERG "[svinput]id=0x%02x\n", buz_org_ptn.id);
				FORCE_DPRINTK(KERN_EMERG "[svinput]m=0x%02x\n", buz_org_ptn.m);
				FORCE_DPRINTK(KERN_EMERG "[svinput]level=0x%02x\n", buz_org_ptn.level);
				FORCE_DPRINTK(KERN_EMERG "[svinput]bptn0=0x%02x\n", buz_org_ptn.bptn0);
				FORCE_DPRINTK(KERN_EMERG "[svinput]bptn1=0x%02x\n", buz_org_ptn.bptn1);
				FORCE_DPRINTK(KERN_EMERG "[svinput]bptn2=0x%02x\n", buz_org_ptn.bptn2);
				FORCE_DPRINTK(KERN_EMERG "[svinput]bptn3=0x%02x\n", buz_org_ptn.bptn3);
				FORCE_DPRINTK(KERN_EMERG "[svinput]bptn4=0x%02x\n", buz_org_ptn.bptn4);
				FORCE_DPRINTK(KERN_EMERG "[svinput]bptn5=0x%02x\n", buz_org_ptn.bptn5);
				FORCE_DPRINTK(KERN_EMERG "[svinput]bptn6=0x%02x\n", buz_org_ptn.bptn6);
				FORCE_DPRINTK(KERN_EMERG "[svinput]bptn7=0x%02x\n", buz_org_ptn.bptn7);
				FORCE_DPRINTK(KERN_EMERG "[svinput]sum=0x%02x\n", buz_org_ptn.sum);
			}
			if(buz_org_ptn.m == 0){
				s_buzzer_info.state = BUZZER_ON;
			}
			else{
				s_buzzer_info.state = BUZZER_OFF;
			}
			s_buzzer_info.id = buz_args.id;
			s_buzzer_info.level = buz_args.level;
		}
		else{
			FORCE_DPRINTK(KERN_NOTICE "[svinput]unexpected return value:i2c_master_sendrecv\n");
		}
		break;
	default:
		DPRINTK("Unknown command => 0x%02x\n", cmd);
		break;
	}

 EXIT_FUNC:
//	FUNC_END;
	return rc;
}
			
void buzzer_wqueue_handler(svinput_slv_buz_end_cmd_t *slv_buz_end_cmd)
{
	do_gettimeofday(&tv);
	
	DPRINTK("(%ld.%06ld)buzzer sounding is end (ID:0x%02x)\n", tv.tv_sec, tv.tv_usec, slv_buz_end_cmd->id);
	s_buzzer_info.state = BUZZER_OFF;
	
	return;
}
