/*
 * Driver for the Marvell 61xx CDMA Controller
 *
 * Derived from include/linux/dw_dmac.h
 *
 * Copyright (C) 2007 Atmel Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */
#ifndef MV61_DMAC_H
#define MV61_DMAC_H

#include <linux/dmaengine.h>
#include <linux/platform_device.h>
#include <linux/string.h>

void mv61x0_cdma_platform_init(void);

/**
 * enum mv61_vdma_type - index for virtual dma controllers
 * @MV61_VDMA_OWNED:  Each virtual channel owns a physical channel
 * @MV61_VDMA_SHARED: Virtual channels share a pool of physical channels
 * @MV61_VDMA_MEMOPS: Virtual channel(s) for kernel use
 */
enum mv61_vdma_type {
	MV61_VDMA_OWNED = 0,
        MV61_VDMA_SHARED,
        MV61_VDMA_CYCLIC,
        MV61_VDMA_MEMOPS,
	MV61_VDMA_UNASSIGNED,
};

#define MV61_NR_VDMA_CONTROLLERS (MV61_VDMA_UNASSIGNED)

#define MV_VDMA_NAME	"mv61x0-vdma"
#define MV_CDMA_NAME	"mv61x0-cdma"

/**
 * struct mv61_dma_platform_data - Controller configuration parameters
 * @cdma_type: CDMA_MV61X0 = 1, CDMA_PEGMATITE = 2
 * @nr_channels: Number of channels supported by hardware (max 12)
 * @nr_pool_chans: Number of hardware channels per virtual controller
 * @nr_virt_chans: Number of virtual channels per virtual controller
 * @__mv61_dma: address of struct mv61_dma (not always in scope here)
 */
struct mv61_dma_platform_data {
	unsigned int	cdma_type;
	unsigned int	nr_channels;
	unsigned int	nr_pool_chans[MV61_NR_VDMA_CONTROLLERS];
	unsigned int	nr_virt_chans[MV61_NR_VDMA_CONTROLLERS];
};

/**
 * struct mv61_vdma_platform_data - Virtual dma controller parameters
 * @pool_device: real device providing channel pool
 */
struct mv61_vdma_platform_data {
	struct platform_device * pool_device;
};

#if 0 
/**
 * enum mv61_dma_endian - DMA slave byte lane placement
 * @MV61_DMA_SLAVE_ABCD_TO_XXXX: byte lanes in to byte lanes out
 */
enum mv61_dma_endian {
	MV61_DMA_ABCD_TO_ABCD,
	MV61_DMA_ABCD_TO_CDAB,
	MV61_DMA_ABCD_TO_BADC,
	MV61_DMA_ABCD_TO_DCBA,
};
#else
/* actual assignment TBD */
enum mv61_dma_endian {
	MV61_DMA_LITTLE_ENDIAN,
	MV61_DMA_BIG_ENDIAN,
};
#endif

/**
 * enum mv61_dma_xfer_width - DMA slave register access width
 * @MV61_DMA_XFER_WIDTH_8BIT: Do 8-bit slave register accesses
 * @MV61_DMA_XFER_WIDTH_16BIT: Do 16-bit slave register accesses
 * @MV61_DMA_XFER_WIDTH_32BIT: Do 32-bit slave register accesses
 */
enum mv61_dma_xfer_width {
	MV61_DMA_XFER_WIDTH_8BIT,
	MV61_DMA_XFER_WIDTH_16BIT,
	MV61_DMA_XFER_WIDTH_32BIT,
	MV61_DMA_XFER_WIDTH_RESERVED,
};

/**
 * enum mv61_dma_unit_size - data unit size,only affects endian swapping
 * @MV61_DMA_XFER_WIDTH_8BIT: Do 8-bit slave register accesses
 * @MV61_DMA_XFER_WIDTH_16BIT: Do 16-bit slave register accesses
 * @MV61_DMA_XFER_WIDTH_32BIT: Do 32-bit slave register accesses
 */
enum mv61_dma_dma_unit_size {
	MV61_DMA_UNIT_SIZE_8BIT,
	MV61_DMA_UNIT_SIZE_16BIT,
	MV61_DMA_UNIT_SIZE_32BIT,
	MV61_DMA_UNIT_SIZE_RESERVED,
};

/**
 * enum mv61_dma_flow_ctrl - dma handshake muxing
 * @MV61_DMA_MEMORY_TO_MEMORY: flow only limited by wrdelay and dram access 
 * @MV61_DMA_MEMORY_TO_PERIPHERAL: flow controlled by dest
 * @MV61_DMA_PERIPHERAL_TO_MEMORY: flow controlled by src
 * @MV61_DMA_PERIPHERAL_TO_PERIPHERAL: flow controlled by src and dest
 */
enum mv61_dma_flow_ctrl {
	MV61_DMA_MEMORY_TO_MEMORY,
	MV61_DMA_MEMORY_TO_PERIPHERAL,
	MV61_DMA_PERIPHERAL_TO_MEMORY,
	MV61_DMA_PERIPHERAL_TO_PERIPHERAL,
};

/**
 * enum mv61_burst_size - burst length in 32 bit words
 * @MV61_DMA_XFER_BURST1: 1 word bursts
 * @MV61_DMA_XFER_BURST4: 4 word bursts
 * @MV61_DMA_XFER_BURST8: 8 word bursts
 *
 * Must select 1 word bursts for transfer widths less than 32 bits.
 * 1 word bursts are a reasonable default for most peripherals.
 */
enum mv61_dma_burst_size {
	MV61_DMA_BURST1,
	MV61_DMA_BURST4,
	MV61_DMA_BURST8,
	MV61_DMA_BURST_RESERVED,
};

/**
 * enum mv61_timebase - used for request timeout
 */
enum mv61_timebase {
	MV61_TIMEBASE_1US,
	MV61_TIMEBASE_10US,
	MV61_TIMEBASE_100US,
	MV61_TIMEBASE_1MS,
	MV61_TIMEBASE_10MS,
	MV61_TIMEBASE_100MS,
	MV61_TIMEBASE_RESERVED1,
	MV61_TIMEBASE_RESERVED2,
};


/**
 * struct mv61_dma_slave - Controller-specific information for dma channel
 * @vtype: required virtual dma channel type
 * @wrdelay: usec to pause between bursts, 16 max, mainly used for mem to mem
 * @dest_endian: byte lane mapping from bus to dest
 * @dest_endian: byte lane mapping from src to bus
 * @flowctrl: dma handshake muxing type
 * @dest_pid: dma handshake muxing destination peripheral
 * @src_pid: dma handshake muxing source peripheral
 * @dest_addr_inc: addresses generally increment for memory but not peripherals
 * @src_addr_inc: addresses generally increment for memory but not peripherals
 * @dest_width: peripheral data width, use MV61_DMA_XFER_WIDTH_32BIT for memory
 * @src_width: peripheral data width, use MV61_DMA_XFER_WIDTH_32BIT for memory
 * @data_unit_size: only affects byte swapping
 * @dest_reg: physical address of destination peripheral
 * @src_reg: physical address of source peripheral
 * @timebase: timer tick period in uSec x power of 10  
 * @timer: max timer ticks between peripheral dma reqs, 1023 max, 0 = disabled
 * @wrap: aligned byte count for peripheral address to wrap
 * In most cases, this structure will probably be declared and populated in 
 * each peripheral's platform_device data (one per direction).
 */
struct mv61_dma_slave {
	int				vtype;
	int				wr_delay;
#if 0
	enum mv61_dma_endian		dest_endian;
	enum mv61_dma_endian		src_endian;
	bool				bigend;
#else
	bool				destendian;
	bool				srcendian;
#endif
	enum mv61_dma_flow_ctrl		flowctrl;
	int 				dest_pid;
	int 				src_pid;
	bool				dest_addr_inc;
	bool				src_addr_inc;
	enum mv61_dma_xfer_width	dest_width;
	enum mv61_dma_xfer_width	src_width;
	enum mv61_dma_dma_unit_size	data_unit_size;
	enum mv61_dma_burst_size	dest_burst;
	enum mv61_dma_burst_size	src_burst;
	dma_addr_t			dest_reg;
	dma_addr_t			src_reg;
	enum mv61_timebase		timebase;
	int				timer;
	int				wrap;
};


/**
 * __chan_to_plat_dev - get platform_device that is providing the channel
 * @chan: channel candidate offered by dmaengine's dma_request_channel()
 * @req_type: requested virtual channel type
 *
 * Assumes valid/sane name strings will be much shorter than 80 characters.
 */
static inline struct platform_device *__chan_to_plat_dev(struct dma_chan *chan) {
	return container_of(chan->device->dev, struct platform_device, dev);
}

/**
 * __chk_vdma_type - check candidate channel against requirements
 * @chan: channel candidate offered by dmaengine's dma_request_channel()
 * @req_type: requested virtual dma channel type
 *
 * Assumes valid names will be much shorter than 80 characters.
 */
static bool __chk_vdma_type(struct dma_chan *chan, enum mv61_vdma_type req_type) {
	struct platform_device	*ch_pdev = __chan_to_plat_dev(chan);
	
	if((ch_pdev->id == req_type) && ch_pdev->name)
		return (NULL != strstr(ch_pdev->name, "vdma"));
	else
		return false;
}

/**
 * filter - filter candidate dma channels and initialize if ok
 * @chan: channel candidate offered by dmaengine's dma_request_channel()
 * @slave: contains requested virtual channel type and peripheral data
 */
static bool __attribute__((used)) filter(struct dma_chan *chan, void *slave) {
	struct mv61_dma_slave *vslave = slave;

	if (__chk_vdma_type(chan, vslave->vtype)) {
		chan->private = vslave;
		return true;
	} else
		return false;
}

/******************* DMA API extensions ******************************/

/**
 * mv61vc_prep_slave_linear - debug use *ONLY*
 * @buf: physical address of a contiguous dma buffer
 * @len: transfer length in bytes
 *
 * All other arguments are the same as mv61c_prep_slave_sg(), aka
 * chan->device->device_prep_slave_sg(). For debug use ONLY, may be
 * removed later.
 */
struct dma_async_tx_descriptor *
mv61vc_prep_slave_linear (struct dma_chan *chan, dma_addr_t buf,
		unsigned int len, enum dma_transfer_direction direction,
		unsigned long flags);
#if 0
struct mv61_cyclic_desc {
	struct mv61_desc	**desc;
	unsigned long	periods;
	void		(*period_callback)(void *param);
	void		*period_callback_param;
};

struct mv61_cyclic_desc *mv61_dma_cyclic_prep(struct dma_chan *chan,
		dma_addr_t buf_addr, size_t buf_len, size_t period_len,
		enum dma_transfer_direction direction);
void mv61_dma_cyclic_free(struct dma_chan *chan);
int mv61_dma_cyclic_start(struct dma_chan *chan);
void mv61_dma_cyclic_stop(struct dma_chan *chan);

dma_addr_t mv61_dma_get_src_addr(struct dma_chan *chan);

dma_addr_t mv61_dma_get_dst_addr(struct dma_chan *chan);
#endif

/*
 * Provides basic scatterlist buffer support. For examples of more sophisticated
 * scatter-gather buffer management, see drivers/media/video/videobuf-dma-sg.c.
 */

struct dmabuf_sg {
	struct scatterlist	*sg;
	void 			*virt;		/* virtual address of buffer received from vmalloc */
						/* probably not physically contiguous */
	int			sg_ents;	/* number of entries in scatterlist */
	int			sglen;		/* number of entries mapped sg entries */
	unsigned long		bytes;
};

extern int dmabuf_vmalloc_to_sg(struct dmabuf_sg *buf);
extern int dmabuf_kmalloc_to_sg(struct dmabuf_sg *buf);
#endif /* MV61_DMAC_H */
