/*
 * Regulator support for Pegmatite platforms.
 *
 * This file is licensed under the terms of the GNU General Public
 * License version 2.  This program is licensed "as is" without any
 * warranty of any kind, whether express or implied.
 *
 */

#include <linux/io.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/regulator/driver.h>
#include <linux/regulator/machine.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/delay.h>
#include <linux/clk.h>
#include <linux/clk-provider.h>
#include <linux/regulator/of_regulator.h>
#include <linux/slab.h>
#include <linux/err.h>
#include <linux/clk-private.h>

#define TIMEOUT 10
#define MAX_CLK_DEPTH 10 

static int imagingIslandSRAMFix(struct regulator_dev *rdev);
static int islandPowerUp(struct regulator_dev *rdev);
static int islandPowerDown(struct regulator_dev *rdev);
extern bool __clk_is_prepared(struct clk *clk);

#define PICR_REG_OFFSET		0x0
#define PISR_REG_OFFSET		0x4
#define IPSBUSCLK_IMAGINGCONFIG_CLKGENCONFIG 0x158
#define IPSBUSCLK_IMAGINGCONFIG_CLKGENSTATUS 0x15C

#define CLKCONFIG_RSTN_MASK	0x1
#define CLKCONFIG_RSTN_SHIFT	0
#define CLKSTATUS_RSTN_MASK	0x1
#define CLKSTATUS_RSTN_SHIFT	0

#define PMU_ISOLATION		0x0
#define PMU_ISOLATION_MASK	0x1
#define ISOLATION_ENABLE	0
#define ISOLATION_DISABLE	1

#define PMU_BULKPWRUP		4
#define PMU_BULKPWRUP_MASK	0x30
#define SWITCH_FULL_OFF		0
#define POWER_RAMP		1
#define SWITCH_FULL_ON		3

struct pegmatite_reg_info {
	struct device		*dev;
	struct regulator_desc	desc;
	struct regulator_dev	*regulator;
	const char		*name;
	void   __iomem		*pmu_regs;
	struct clk		*island_clks[MAX_CLK_DEPTH];
	int num_clks;
	int state;
	bool disable_rbist;
	struct regulator_consumer_supply supply;
};

static unsigned int regRead(volatile unsigned int *regAddr)
{
	return(ioread32(regAddr));
};

static void regMaskWrite(volatile unsigned int * address, 
						unsigned int mask, unsigned int pos, 
						unsigned int value)
{
	unsigned int rdata;

	rdata = ioread32(address);
	rdata = ((rdata & ~mask) | ((value << pos) & mask));
	iowrite32(rdata, address);
};

static const struct of_device_id peg_dt_reg_id[] =
{
	{ .compatible = "pegmatite-reg", },
	{}
};
MODULE_DEVICE_TABLE(of, peg_dt_reg_id);

int pegmatite_reg_check(void)
{
	return 1;
}
EXPORT_SYMBOL(pegmatite_reg_check);
/* callback used by Kernel regulator core to enable regulator */
static int pegmatite_reg_enable(struct regulator_dev *rdev)
{
	struct pegmatite_reg_info *info = rdev_get_drvdata(rdev);
	int rc;

	pr_debug("%s:  o Powering up Island\n", __func__);
	islandPowerUp(rdev);

	/*
	 *  Rev B requires the following sequence to ensure that imaging
	 *  SRAMs are accessible.
	 */
	if (info->disable_rbist) {
		pr_debug("%s:  o Errata: Disabling RBIST for SRAMS\n", __func__);
		rc = imagingIslandSRAMFix(rdev);
		pr_debug("%s:  imagingIslandSRAMFix returned %d\n",__func__,rc);
	}

	info->state = 1;
	return 0;
}

/* callback used by Kernel regulator core to disable regulator */
static int pegmatite_reg_disable(struct regulator_dev *rdev)
{
	struct pegmatite_reg_info *info = rdev_get_drvdata(rdev);

	pr_debug("%s:  o Powering down Island\n", __func__);
	islandPowerDown(rdev);

	info->state = 0;
	return 0;
}

/* callback used by Kernel regulator core to determine state of regulator */
static int pegmatite_reg_is_enabled(struct regulator_dev *rdev)
{
	struct pegmatite_reg_info *info = rdev_get_drvdata(rdev);

	{/* RICOH patch at 2017/04/12 */
		/**
		 * Repiaring miss match of status and count
		 */
		if ((info->state == 0) && (rdev->use_count != 0)) {
			rdev->use_count = 0;
		} else if ((info->state != 0) && (rdev->use_count == 0)) {
			rdev->use_count = 1;
		}
	}

	return info->state;
}

static struct regulator_ops pegmatite_regulator_ops = {
	.enable     = pegmatite_reg_enable,
	.disable    = pegmatite_reg_disable,
	.is_enabled = pegmatite_reg_is_enabled,
};

static int pegmatite_reg_probe(struct platform_device *pdev)
{
	struct regulator_config config = {};
	struct regulator_init_data *init_data;
	struct device_node *np =   pdev->dev.of_node;
	struct pegmatite_reg_info *info;
	struct regulator_dev *rdev;
	struct regulation_constraints   *constraints;
	struct resource *res;
	int    init_on = 0;
	int    val;
	int    i;

	pr_debug("pegmatite_reg_probe called\n");

	info = devm_kzalloc(&pdev->dev,sizeof(struct pegmatite_reg_info),GFP_KERNEL);
	if (!info)
	{
		return -ENOMEM;
	}

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);

	if (!res)
		return -ENODEV;
	info->pmu_regs = devm_ioremap_resource(&pdev->dev, res);
	if (IS_ERR(info->pmu_regs)) {
		dev_warn(&pdev->dev, "Failed to get memory resource\n");
		return PTR_ERR(info->pmu_regs);
	}

	if (!of_property_read_u32(np, "init-on", &val))
	{
		init_on = val;
	}

	info->disable_rbist = of_property_read_bool(np, "disable-rbist");

	/* initialize regulator attributes */
	info->desc.name = "pegmatite-regulator";
	info->desc.ops   = &pegmatite_regulator_ops;
	info->desc.type  = REGULATOR_VOLTAGE;
	info->desc.owner = THIS_MODULE;

	info->num_clks = 0;
	for (i=0; i < MAX_CLK_DEPTH; i++)
	{
		info->island_clks[i] = of_clk_get(np,i);
		if (IS_ERR(info->island_clks[i])) {
			break;
		}
		else {
			info->num_clks++;
		}
	}
	pr_debug("%s:  o Found %d clocks...\n", __func__,info->num_clks);

	if (IS_ERR(info->island_clks[0])) {
		dev_err(&pdev->dev, "No clock declared\n");
		return -ENODEV;
	}

	config.dev = &pdev->dev;
	config.driver_data = info;
	config.of_node = pdev->dev.of_node;

	init_data = of_get_regulator_init_data(&(pdev->dev), np);
	info->name = of_get_property(np, "regulator-name", NULL);
	init_data->num_consumer_supplies = 1;
        init_data->consumer_supplies = &info->supply;
	info->supply.dev_name = of_get_property(np, "dev-name", NULL);
	info->supply.supply = of_get_property(np, "supply-name", NULL);

	constraints = &init_data->constraints;
	constraints->valid_modes_mask &= REGULATOR_MODE_NORMAL;
	constraints->valid_ops_mask &= REGULATOR_CHANGE_STATUS;

	config.init_data = init_data;

	/* register regulator with core */
	rdev = regulator_register(&info->desc, &config);
	if (IS_ERR(rdev)) {
		dev_err(&pdev->dev, "failed to register regulator %s\n", info->desc.name);
		return PTR_ERR(rdev);
	}

	info->regulator = rdev;

	platform_set_drvdata(pdev, rdev);

	if (init_on) {
		pegmatite_reg_enable(rdev);
	}

	return 0;
}

static int pegmatite_reg_remove(struct platform_device *pdev)
{
	struct regulator_dev *rdev = platform_get_drvdata(pdev);
	struct pegmatite_reg_info *info = rdev_get_drvdata(rdev);
	int i;

	for (i=0; i < info->num_clks; i++) {
		clk_put(info->island_clks[i]);
	}

	platform_set_drvdata(pdev, NULL);
	regulator_unregister(rdev);

	return 0;
}

static struct platform_driver pegmatite_reg_driver = {
	.driver = {
		.name  = "pegmatite-regulator",
		.owner = THIS_MODULE,
		.of_match_table = peg_dt_reg_id,
	},
	.probe     = pegmatite_reg_probe,
	.remove    = pegmatite_reg_remove,
};

static int __init pegmatite_reg_driver_platform_init(void)
{
	return platform_driver_register(&pegmatite_reg_driver);
}
module_init(pegmatite_reg_driver_platform_init);


static int imagingIslandSRAMFix(struct regulator_dev *rdev)
{

	struct clk *ipsbusdiv2clk;
	struct pegmatite_reg_info *info = rdev_get_drvdata(rdev);
	int i;

	ipsbusdiv2clk = of_clk_get(rdev->dev.of_node, 4);
	if (IS_ERR(ipsbusdiv2clk)) {
		dev_err(&rdev->dev, "No clock declared\n");
		return -ENODEV;
	}

	while (__clk_is_enabled(ipsbusdiv2clk))
		clk_disable_unprepare(ipsbusdiv2clk);

	/*
	 * Enable IPSBusClkDiv2
	 */
	clk_prepare_enable(ipsbusdiv2clk);

	/*
	 * Assert IPSBusClk_Imaging reset
	 */
	regMaskWrite(info->pmu_regs + IPSBUSCLK_IMAGINGCONFIG_CLKGENCONFIG,
		CLKCONFIG_RSTN_MASK,
		CLKCONFIG_RSTN_SHIFT,
		0);

	/* Poll until the status bit changed */
	i = 0;
	while((regRead(info->pmu_regs + IPSBUSCLK_IMAGINGCONFIG_CLKGENSTATUS) & CLKSTATUS_RSTN_MASK) != 0x0) {
		i++;
		if(i == TIMEOUT) {
			pr_err("%s: IMAGING Timeout reached\n", __func__);
			break;
		}
	}

	udelay(100);

	/*
	 * Disable IPSBusClkDiv2
	 */
	clk_disable_unprepare(ipsbusdiv2clk);

	udelay(100);

	/*
	 * De-assert IPSBusClk_Imaging reset
	 */
	regMaskWrite(info->pmu_regs + IPSBUSCLK_IMAGINGCONFIG_CLKGENCONFIG,
		CLKCONFIG_RSTN_MASK,
		CLKCONFIG_RSTN_SHIFT,
		1);

	/* Poll until the status bit changed */
	i = 0;
	while((regRead(info->pmu_regs + IPSBUSCLK_IMAGINGCONFIG_CLKGENSTATUS) & CLKSTATUS_RSTN_MASK) != 0x1) {
		i++;
		if(i == TIMEOUT) {
			pr_err("%s: IMAGING Timeout reached\n", __func__);
			break;
		}
	}

	clk_put(ipsbusdiv2clk);

	return 0;
};

static int islandPowerUp(struct regulator_dev *rdev)
{
	int i;
	struct pegmatite_reg_info *info = rdev_get_drvdata(rdev);

	/*
	*  Power up procedure
	*     -- Assert resets, poll
	*     -- wait 100 us
	*     -- turn on switches (2, 20us apart?), poll
	*     -- wait 100 us
	*     -- turn on clocks 10 us to propagate resets, then off
	*     -- wait 100 us
	*     -- un-isolate outputs, poll
	*     -- wait 100 us
	*     -- turn on island clocks
	*     -- de-assert resets, poll
	*/

	if (!info->pmu_regs) {
		pr_emerg("Island base address not initialized\n");
		return -1;
	}

	pr_debug("%s:  o Assert island resets by disabling clocks\n", __func__);
	for (i=0; i < info->num_clks; i++)
	{
		/* RICOH patch to suppress warning */
		if (info->island_clks[i]->enable_count > 0)
			clk_disable(info->island_clks[i]);
		if (info->island_clks[i]->prepare_count > 0)
			clk_unprepare(info->island_clks[i]);
	}

	udelay(100);
	pr_debug("%s:  o Turn on island power switches\n", __func__);
	regMaskWrite(info->pmu_regs + PICR_REG_OFFSET,
		PMU_BULKPWRUP_MASK,
		PMU_BULKPWRUP,
		POWER_RAMP);
	udelay(20);
	regMaskWrite(info->pmu_regs + PICR_REG_OFFSET,
		PMU_BULKPWRUP_MASK,
		PMU_BULKPWRUP,
		SWITCH_FULL_ON);

	/* Poll until the status bit changed */
	i = 0;
	while((regRead(info->pmu_regs + PISR_REG_OFFSET) & PMU_BULKPWRUP_MASK)
		!= PMU_BULKPWRUP_MASK) {
		i++;
		if(i == TIMEOUT) {
			pr_err("%s:  ! BULKPWRUP Timeout reached\n", __func__);
			break;
		}
	}

	udelay(100);

	pr_debug("%s:  o Turn island clocks on/off to propagte resets\n", __func__);

	for (i=0; i < info->num_clks; i++) {
		clk_prepare_enable(info->island_clks[i]);
	}

	udelay(10);

	for (i=0; i < info->num_clks; i++) {
		clk_disable(info->island_clks[i]);
	}

	udelay(100);

	pr_debug("%s:  o Un-isolate island outputs\n", __func__);
	regMaskWrite(info->pmu_regs + PICR_REG_OFFSET,
		PMU_ISOLATION_MASK,
		PMU_ISOLATION,
		ISOLATION_DISABLE);

	/* Poll until the status bit updated */
	i = 0;
	while((regRead(info->pmu_regs + PISR_REG_OFFSET) & PMU_ISOLATION_MASK)
		!= PMU_ISOLATION_MASK) {
		i++;
		if(i == TIMEOUT) {
			pr_err("%s:  ! ISOLATION_BUF Timeout reached\n", __func__);
			break;
		}
	}

	udelay(100);

	pr_debug("%s:  o Turn on island clocks\n", __func__);
	for (i=0; i < info->num_clks; i++) {
		clk_enable(info->island_clks[i]);
	}

	return 0;
};

static int islandPowerDown(struct regulator_dev *rdev)
{
	int i = 0;
	struct pegmatite_reg_info *info = rdev_get_drvdata(rdev);

	if (!info->pmu_regs) {
		pr_emerg("Island base address not initialized\n");
		return -1;
	}

	pr_debug("%s:  o Disabling island clocks, asserting reset\n", __func__);
	for (i=0; i < info->num_clks; i++)
	{
		while(__clk_is_enabled(info->island_clks[i])) {
			clk_disable_unprepare(info->island_clks[i]);
		}
	}

	udelay(100);

	pr_debug("%s:  o Isolate island outputs\n", __func__);
	regMaskWrite(info->pmu_regs + PICR_REG_OFFSET,
		PMU_ISOLATION_MASK,
		PMU_ISOLATION,
		ISOLATION_ENABLE);

	/* Poll until the status bit updated */
	i = 0;
	while((regRead(info->pmu_regs + PISR_REG_OFFSET) & PMU_ISOLATION_MASK)
		!= 0) {
		i++;
		if(i == TIMEOUT) {
			pr_err("%s:  ! ISOLATION_BUF Timeout reached\n", __func__);
			break;
		}
	}

	udelay(100);

	pr_debug("%s:  o Turn off Island power switches\n", __func__);
	regMaskWrite(info->pmu_regs + PICR_REG_OFFSET,
		PMU_BULKPWRUP_MASK,
		PMU_BULKPWRUP,
		POWER_RAMP);
	udelay(20);
	regMaskWrite(info->pmu_regs + PICR_REG_OFFSET,
		PMU_BULKPWRUP_MASK,
		PMU_BULKPWRUP,
		SWITCH_FULL_OFF);

	/* Poll until the status bit changed */
	i = 0;
	while((regRead(info->pmu_regs + PISR_REG_OFFSET) & PMU_BULKPWRUP_MASK)
		!= 0) {
		i++;
		if(i == TIMEOUT) {
			pr_err("%s:  ! BULKPWRDN Timeout reached\n", __func__);
			break;
		}
	}

	udelay(100);

	return 0;
};

MODULE_LICENSE("GPL");
