/*
 *  linux/arch/arm/mach-mv61x0/pwm.c
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 */

/********  Definititions & Macros for Module **********************************/


/********  Include Files ******************************************************/
#include <asm/io.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/types.h>
#include <linux/slab.h>
#include <linux/pwm.h>
#include <linux/err.h>
#include <linux/of.h>
#include <linux/clk.h>

/********  Variable Declarations for Module ***********************************/

static int vdebug = 0;
module_param(vdebug, int, S_IRUGO);
#define PRINTK(fmt...)  if (vdebug) printk(fmt)

#define DRIVER_NAME	"mv61x0_pwm"

#define NUM_PWMS	32

/* Spinlock for the global ioremapped memory pointer */
static DEFINE_SPINLOCK( pwm_lock );

/* Registers structure */
#define PWM_CFG		0x00
#define PWM_IDEL	0x04
#define PWM_C0		0x08
#define PWM_C1		0x0C

#define PWM_CFG_EN		(0x1 << 0)
#define PWM_CFG_HIGHLOW		(0x1 << 3)
#define PWM_CFG_TBSEL_SHIFT	4
#define PWM_CFG_TBSEL_MASK	(0xf << PWM_CFG_TBSEL_SHIFT)
#define PWM_IDELAY		0xffff
#define PWM_COUNT0		0xffff
#define PWM_COUNT1		0xffff

/**
 * struct mv61_pwm_chip - struct representing pwm chip
 *
 * @mmio_base: base address of pwm chip
 * @clk: pointer to clk structure of pwm chip
 * @chip: linux pwm chip representation
 * @dev: pointer to device structure of pwm chip
 * @sysclk_ns: clock period in nanoseconds
 */
struct mv61_pwm_chip {
	void __iomem *mmio_base;
	struct clk *clk;
	struct pwm_chip chip;
	int sysclk_ns;
};

static inline struct mv61_pwm_chip *to_mv61_pwm_chip(struct pwm_chip *chip)
{
	return container_of(chip, struct mv61_pwm_chip, chip);
}

static inline u32 mv61_pwm_readl(struct mv61_pwm_chip *chip, unsigned int num,
				  unsigned long offset)
{
	return readl_relaxed(chip->mmio_base + ((num + 1) << 8) + offset);
}

static inline void mv61_pwm_writel(struct mv61_pwm_chip *chip,
				    unsigned int num, unsigned long offset,
				    unsigned long val)
{
	writel_relaxed(val, chip->mmio_base + ((num + 1) << 8) + offset);
}

/*
 * PWM configuration function
 */
static int mv61_pwm_config(struct pwm_chip *chip, struct pwm_device *pwm,
			int duty_ns, int period_ns)
{
	unsigned long flags;
	struct mv61_pwm_chip *pc = to_mv61_pwm_chip(chip);
	struct pwm_device *pd = pwm;
	u32 duty_cnt_needed;
	u32 period_cnt_needed;
	u32 pwm_cfg;
	int i;
	int clk_ns = pc->sysclk_ns;

	/*
	 * Make sure the PWM offset is within bound
	 */
	if (pd->hwpwm < NUM_PWMS && pd->hwpwm >= 0) {
		/*
		 * Need to make sure that we have enough bits to get the duty
		 * requested from client.
		 */
		for (i = 0; i < 7; i++) {
			switch(i) {
			case 1: /* 1 us as timebase */
				duty_cnt_needed = duty_ns / 1000;
				period_cnt_needed = (period_ns - duty_ns) / 1000;
				break;
			case 2: /* 10 us as timebase */
				duty_cnt_needed = duty_ns / 10000;
				period_cnt_needed = (period_ns - duty_ns) / 10000;
				break;
			case 3: /* 100 us as timebase */
				duty_cnt_needed = duty_ns / 100000;
				period_cnt_needed = (period_ns - duty_ns) / 100000;
				break;
			case 4: /* 1 ms as timebase */
				duty_cnt_needed = duty_ns / 1000000;
				period_cnt_needed = (period_ns - duty_ns) / 1000000;
				break;
			case 5: /* 10 ms as timebase */
				duty_cnt_needed = duty_ns / 10000000;
				period_cnt_needed = (period_ns - duty_ns) / 10000000;
				break;
			case 6: /* 100 ms as timebase */
				duty_cnt_needed = duty_ns / 100000000;
				period_cnt_needed = (period_ns - duty_ns) / 100000000;
				break;
			default:
			case 0: /* system clock as timebase */
				PRINTK("sysclk_ns = %d\n", clk_ns);
				duty_cnt_needed = duty_ns / clk_ns;
				period_cnt_needed = (period_ns - duty_ns) / clk_ns;
				break;
			}

			/*
			* If there are enought bits for the PWM requested
			*/
			if (duty_cnt_needed < PWM_COUNT0 && period_cnt_needed < PWM_COUNT1) {

				spin_lock_irqsave(&pwm_lock, flags);
				/*
				 * Program the timebase needed to use. Always try to use the
				 * highest possible resolution
				 */
				pwm_cfg = mv61_pwm_readl(pc, pd->hwpwm, PWM_CFG);
				pwm_cfg &= ~(PWM_CFG_TBSEL_MASK);
				pwm_cfg |= ((i << PWM_CFG_TBSEL_SHIFT) & PWM_CFG_TBSEL_MASK);
				pwm_cfg |= PWM_CFG_HIGHLOW;
				mv61_pwm_writel(pc, pd->hwpwm, PWM_CFG, pwm_cfg);

				/*
				 * Program the duty cycles count and period count
				 */
				mv61_pwm_writel(pc, pd->hwpwm, PWM_C0, duty_cnt_needed);
				mv61_pwm_writel(pc, pd->hwpwm, PWM_C1, period_cnt_needed);

				spin_unlock_irqrestore(&pwm_lock, flags);

				return(0);
			}
		}
	}

	return(-1);
};

/*
 * mv61_pwm_enable - start a PWM output toggling
 */
static int mv61_pwm_enable(struct pwm_chip *chip, struct pwm_device *pwm)
{
	unsigned long flags;
	struct mv61_pwm_chip *pc = to_mv61_pwm_chip(chip);
	u32 cfg;

	/*
	 * Make sure the PWM offset is within bound
	 */
	if (pwm->hwpwm < NUM_PWMS && pwm->hwpwm >= 0) {
		/*
		 * Enable the PWM at base reg and unit's config reg
		 */
		cfg = readl_relaxed(pc->mmio_base);
		cfg |= (1 << pwm->hwpwm);
		writel_relaxed(cfg, pc->mmio_base);

		spin_lock_irqsave(&pwm_lock, flags);
		cfg = mv61_pwm_readl(pc, pwm->hwpwm, PWM_CFG);
		cfg |= PWM_CFG_EN;
		mv61_pwm_writel(pc, pwm->hwpwm, PWM_CFG, cfg);
		spin_unlock_irqrestore(&pwm_lock, flags);
	} else {
		return(-1);
	}

	return(0);
};

/*
 * mv61_pwm_disable - stop a PWM output toggling
 */
static void mv61_pwm_disable(struct pwm_chip *chip, struct pwm_device *pwm)
{
	unsigned long flags;
	struct mv61_pwm_chip *pc = to_mv61_pwm_chip(chip);
	u32 cfg;

	if (pwm->hwpwm < NUM_PWMS && pwm->hwpwm >= 0) {
		/*
		 * Disable the PWM
		 */
		cfg = readl_relaxed(pc->mmio_base);
		cfg &= ~(1<<pwm->hwpwm);
		writel_relaxed(cfg, pc->mmio_base);

		spin_lock_irqsave(&pwm_lock, flags);
		cfg = mv61_pwm_readl(pc, pwm->hwpwm, PWM_CFG);
		cfg &= ~(PWM_CFG_EN);
		mv61_pwm_writel(pc, pwm->hwpwm, PWM_CFG, cfg);
		spin_unlock_irqrestore(&pwm_lock, flags);
	}
};

static struct pwm_ops mv61_pwm_ops = {
	.config = mv61_pwm_config,
	.enable = mv61_pwm_enable,
	.disable = mv61_pwm_disable,
	.owner = THIS_MODULE,
};

/*
 * Called by the kernel to determine whether the hardware exists that is
 * managed by this driver.
 */
static int mv61_pwm_probe(struct platform_device *pdev)
{
	struct mv61_pwm_chip *pc;
	struct resource *r;
	int ret;
	int clk_ns;

	r = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!r) {
		dev_err(&pdev->dev, "no memory resources defined\n");
		return -ENODEV;
	}

	pc = devm_kzalloc(&pdev->dev, sizeof(*pc), GFP_KERNEL);
	if (!pc) {
		dev_err(&pdev->dev, "failed to allocate memory\n");
		return -ENOMEM;
	}

	pc->mmio_base = devm_ioremap_resource(&pdev->dev, r);
	if (IS_ERR(pc->mmio_base))
		return PTR_ERR(pc->mmio_base);

	pc->clk = devm_clk_get(&pdev->dev, NULL);
	if (IS_ERR(pc->clk))
		return PTR_ERR(pc->clk);

	ret = clk_prepare(pc->clk);
	if ( ret < 0 ) {
		dev_err(&pdev->dev, "unable to prepare clk\n");
		return ret;
	}

	/* Calculate and store system clock period in ns */
	clk_ns = 1000000000;
	do_div(clk_ns, clk_get_rate(pc->clk));
	pc->sysclk_ns = clk_ns;

	platform_set_drvdata(pdev, pc);

	pc->chip.dev = &pdev->dev;
	pc->chip.ops = &mv61_pwm_ops;
	pc->chip.base = -1;
	pc->chip.npwm = NUM_PWMS;

	ret = pwmchip_add(&pc->chip);
	if (ret < 0) {
		dev_err(&pdev->dev, "pwmchip_add() failed: %d\n", ret);
	}

	return ret;
}

/*
 * Called when a driver is unloaded.
 */
static int mv61_pwm_remove( struct platform_device *pdev )
{
	struct mv61_pwm_chip *pc = platform_get_drvdata(pdev);
	int i;

	/*
	* Clears out the enables bits.
	*/
	for (i = 0; i < NUM_PWMS; i++)
		pwm_disable(&pc->chip.pwms[i]);

	/* clk was prepared in probe, hence unprepare it here */
	clk_unprepare(pc->clk);

	return pwmchip_remove(&pc->chip);
}

static struct of_device_id mv61_pwm_of_match[] = {
	{ .compatible = "mrvl,mv61-pwm" },
	{ }
};

MODULE_DEVICE_TABLE(of, mv61_pwm_of_match);

/*
 * Defines a platform driver structure that can be passed to
 * platform_driver_register() to associate a driver with a named device.
 */
static struct platform_driver mv61_pwm_driver = {
	.probe  = mv61_pwm_probe,
	.remove = mv61_pwm_remove,
	.driver =
	{
		.name   = DRIVER_NAME,
		.owner  = THIS_MODULE,
		.of_match_table = mv61_pwm_of_match,
	},
};

module_platform_driver(mv61_pwm_driver);

MODULE_LICENSE("GPL");
#include <linux/vermagic.h>
MODULE_INFO(vermagic, VERMAGIC_STRING);
