
#include <linux/irq.h>
#include <linux/irqdomain.h>
#include <linux/of.h>
#include <linux/of_irq.h>
#include <linux/of_address.h>
#include <linux/export.h>
#include <linux/irqchip/chained_irq.h>
#include <asm/mach/irq.h>
#include <asm/io.h>
#include "irqchip.h"

#define SIC_IRQ_ENB         0
#define SIC_IRQ_ACK         0xc
#define SIC_IRQ_ENB_CLEAR   0x14
#define SIC_IRQ_STATUS      0x18

#define MAX_SIC_IRQS        32

struct sic_priv {
	void __iomem *base;
	unsigned int irq_base;
	unsigned int edge_mask;
	struct irq_domain *domain;
};

static struct sic_priv dec_sic_priv;

static void sic_ack_irq(struct irq_data *data) {
	struct sic_priv *priv = irq_data_get_irq_chip_data(data);
	unsigned int irq = data->irq;

	irq -= priv->irq_base;

	writel((1 << irq) , (priv->base + SIC_IRQ_ACK));
}

static void sic_mask_irq(struct irq_data *data) {
	struct sic_priv *priv = irq_data_get_irq_chip_data(data);
	unsigned int irq = data->irq;

	irq -= priv->irq_base;

	writel((1 << irq) , (priv->base + SIC_IRQ_ENB_CLEAR));
}

static void sic_unmask_irq(struct irq_data *orig_irq) {
	struct sic_priv *priv = irq_data_get_irq_chip_data(orig_irq);
	unsigned int irq = orig_irq->irq - priv->irq_base;

	writel((1 << irq), (priv->base + SIC_IRQ_ENB));

	if (!(priv->edge_mask & (1 << irq))) {
		/* Level-triggered interrupts need special handling here
		* because of the latching behavior of the SIC.  The initial
		* mask/ack (called by handle_level_interrupt) immediately
		* re-latched because the level interrupt was still active.
		* Therefore, we must re-ack here.  If the interrupt has
		* already been reasserted, the ack will have no affect for
		* a level-triggered interrupt, so this is safe. */
		sic_ack_irq(orig_irq);
	}
}

static void sic_enable_irq(struct irq_data *irq) {
	sic_ack_irq(irq);
	sic_unmask_irq(irq);
}

static void sic_disable_irq(struct irq_data *irq) {
	sic_mask_irq(irq);
}

static void sic_handler(unsigned int parent_irq, struct irq_desc *parent_desc) {
	const struct sic_priv *priv = irq_get_handler_data(parent_irq);
	unsigned int irq = priv->irq_base;
	unsigned int irq_stat = readl(priv->base + SIC_IRQ_STATUS);
	struct irq_chip *parent_chip = irq_get_chip(parent_irq);

	chained_irq_enter(parent_chip, parent_desc);
	while (irq_stat) {
	        if (irq_stat & 1) {
			generic_handle_irq(irq);
		}
	        ++irq;
		irq_stat >>= 1;
	}
	/* IRQ may have relatched since it is level triggered so clear again */
	chained_irq_exit(parent_chip, parent_desc);
}

static struct irq_chip sic_chip = {
	.name        = "dec_sic",
	.irq_ack     = sic_ack_irq,
	.irq_mask    = sic_mask_irq,
	.irq_unmask  = sic_unmask_irq,
	.irq_enable  = sic_enable_irq,
	.irq_disable = sic_disable_irq
};

static int sic_irqdomain_map(struct irq_domain *d, unsigned int irq,
                             irq_hw_number_t hwirq)
{
        irq_set_chip_and_handler(irq, &sic_chip, handle_level_irq);
        irq_set_chip_data(irq, &dec_sic_priv);
        set_irq_flags(irq, IRQF_VALID | IRQF_PROBE);
        return 0;
}

static struct irq_domain_ops sic_irqdomain_ops = {
        .map = sic_irqdomain_map,
};

int __init sic_of_init(struct device_node *node, struct device_node *parent)
{
	void __iomem *base;
	int top_irq;
	int i;
	int ret;
	int irq_base;
	u32 num_irqs;

	if (WARN_ON(!node))
		return -ENODEV;

	base = of_iomap(node, 0);
	WARN(!base, "unable to map sic dist registers\n");

	dec_sic_priv.base = base;
	top_irq = irq_of_parse_and_map(node, 0);

	ret = of_property_read_u32(node, "num_irqs", &num_irqs);
	if (ret){
		printk(KERN_INFO "SIC: num_irqs not found in device tree. \
			Defaulting to 32\n");
		num_irqs = 32;
	}
	if (num_irqs > MAX_SIC_IRQS || num_irqs < 0){
		printk(KERN_INFO "SIC: num_irqs outside expected range. \
			Defaulting to 32\n");
		num_irqs = 32;
	}

	irq_base = irq_alloc_descs(-1, 0, num_irqs, 0);
	if (irq_base < 0){
		return -ENOMEM;
	}
	else{
		dec_sic_priv.irq_base = irq_base;
	}

	dec_sic_priv.domain = irq_domain_add_legacy(node, num_irqs,
		dec_sic_priv.irq_base, 0, &sic_irqdomain_ops, &dec_sic_priv);

	for (i = 0; i < num_irqs; i++){
		irq_create_mapping(dec_sic_priv.domain, i);
	}

	irq_set_handler_data(top_irq, &dec_sic_priv);
	if (parent) {
		irq_set_chained_handler(top_irq, sic_handler);
	}
	return 0;
}
IRQCHIP_DECLARE(cortex_a15_sic, "arm,cortex-a15-sic", sic_of_init);
IRQCHIP_DECLARE(cortex_a9_sic, "arm,cortex-a9-sic", sic_of_init);
