/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms and conditions of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef MV_M2M_H
#define MV_M2M_H

#include <linux/io.h>

#define MV_M2M_NAME		"mv_m2m"
#define MV_M2M_POOL_SIZE	(PAGE_SIZE * 8)
#define MV_M2M_THRESHOLD	1
#define MV_M2M_MAX_BYTE_COUNT	((64 * 1024) - 1)

#define M2M_IRQ_TERM_CNT    0x01
#define M2M_IRQ_OUT_ORDER   0x02
#define M2M_IRQ_CHAIN_END   0x04
#define M2M_IRQ_CPU_OWNER   0x08
#define M2M_IRQ_ALL         0x0F

#define M2M_DESC_START_SOFT_RESET_SHIFT 1
#define M2M_DESC_CONTROL_LENGTH_MASK 0xffff

#define M2M_STATUS_IDLE_MASK 0x8000
#define M2M_STATUS_IDLE_SHIFT 15

#define THIRTY_TWO_MB       0x2000000
#define THIRTY_TWO_MB_MASK  (~(THIRTY_TWO_MB - 1))

/*
 * Build the descriptor control word
 *
 */
#define mv_m2m_hw_desc_ctl(own,xlate,len) ((own << 31) | (xlate << 30) | (len & M2M_DESC_CONTROL_LENGTH_MASK))

#define M2M_REG_MASKREAD(reg, mask, pos)        ((readl(reg) & mask) >> pos)
#define M2M_REG_MASKWRITE(reg, mask, pos, data) writel((readl(reg) & ~mask) \
                                                | ((data << pos) & mask), reg)

typedef struct M2M_REGS_s
{
	volatile uint32_t desc_start;  ///< 0x0 [W]: Link List Index Register
	volatile uint32_t current_desc_addr;  ///< 0x4 [R]: Current Link List Address Register
	volatile uint32_t status;  ///< 0x8 [R]: Status register
	volatile uint32_t IEN;  ///< 0xc [R/W]: Interrupt Enable
	volatile uint32_t IPEND;  ///< 0x10 [R]: Interrupt Pending
	volatile uint32_t IACK;  ///< 0x14 [W]: Interrupt Acknowledge
	volatile uint32_t IFOR;  ///< 0x18 [R/W]: Interrupt Force
	volatile uint32_t control;  ///< 0x1c [R]: Descriptor control register
	volatile uint32_t src_addr;  ///< 0x20 [R]: Descriptor Current Source Address
	volatile uint32_t dest_addr;  ///< 0x24 [R]: Descriptor Current Destination Address
	volatile uint32_t pcie_outbound;  ///< 0x28 [R]: Descriptor PCIe outbound Translate Value
	volatile uint32_t list_addr;  ///< 0x2c [R]: Descriptor Linked List Address
	volatile uint32_t chan0_dest_addr;  ///< 0x30 [R]: Channel 0 Destination Address
	volatile uint32_t chan1_dest_addr;  ///< 0x34 [R]: Channel 1 Destination Address
	volatile uint32_t chan2_dest_addr;  ///< 0x38 [R]: Channel 2 Destination Address
	volatile uint32_t chan3_dest_addr;  ///< 0x3c [R]: Channel 3 Destination Address
	volatile uint32_t chan4_dest_addr;  ///< 0x40 [R]: Channel 4 Destination Address
	volatile uint32_t chan5_dest_addr;  ///< 0x44 [R]: Channel 5 Destination Address
	volatile uint32_t chan6_dest_addr;  ///< 0x48 [R]: Channel 6 Destination Address
	volatile uint32_t chan7_dest_addr;  ///< 0x4c [R]: Channel 7 Destination Address
	volatile uint32_t debug;  ///< 0x50 [R]: Debug register
	volatile uint32_t pcie_xlate_reg_addr;  ///< 0x54 [R/W]: PCIe Translation Register Address
	volatile uint32_t REV0;  ///< 0x58 [R]: IP Tagging Revision 0
	volatile uint32_t REV1;  ///< 0x5c [R]: IP Tagging Revision 1
} M2M_REGS_t;

struct mv_m2m_device {
	void __iomem	     *m2m_base;
	struct mv_m2m_chan   *channel;
	int		     xlate_reg_addr;
	int		     pci_window;
	int		     pci_window_size;
	int		     remote_capable;
};

/**
 * struct mv_m2m_chan - internal representation of a XOR channel
 * @pending: allows batching of hardware operations
 * @lock: serializes enqueue/dequeue operations to the descriptors pool
 * @m2m_base: memory mapped register base
 * @irq: device irq
 * @chain: device chain view of the descriptors
 * @completed_slots: slots completed by HW but still need to be acked
 * @dma_desc_pool: allocated pool of dma descriptors
 * @dma_desc_pool_virt: virtual address of allocated pool of dma descriptors
 * @dmadev: dma device
 * @dmachan: dma channel
 * @last_used: place holder for allocation to continue from where it left off
 * @all_slots: complete domain of slots usable by the channel
 * @slots_allocated: records the actual size of the descriptor slot pool
 * @irq_thread: kernel thread where mv_m2m_slot_cleanup runs
 * @desc_addr: address of first descriptor in the pending chain
 */
struct mv_m2m_chan {
	int			pending;
	spinlock_t		lock; /* protects the descriptor slot pool */
	struct M2M_REGS_s	*m2m_base;
	int                     irq;
	struct list_head	chain;
	struct list_head	completed_slots;
	dma_addr_t		dma_desc_pool;
	void			*dma_desc_pool_virt;
	struct dma_device	dmadev;
	struct dma_chan		dmachan;
	struct mv_m2m_desc_slot	*last_used;
	struct list_head	all_slots;
	wait_queue_head_t	wq;
	int			slots_allocated;
	struct task_struct	*irq_thread;
	atomic_t		irqs_pending;
	u32			desc_addr;
	int			pci_window;
	int			pci_window_size;
	int			remote_capable;
};

/**
 * struct mv_m2m_desc_slot - software descriptor
 * @slot_node: node on the mv_m2m_chan.all_slots list
 * @chain_node: node on the mv_m2m_chan.chain list
 * @completed_node: node on the mv_m2m_chan.completed_slots list
 * @hw_desc: virtual address of the hardware descriptor chain
 * @group_head: first operation in a transaction
 * @in_use: whether the slot is in use
 * @unmap_len: transaction bytecount
 * @tx_list: list of slots that make up a multi-descriptor transaction
 * @async_tx: support for the async_tx api
 */
struct mv_m2m_desc_slot {
	struct list_head	slot_node;
	struct list_head	chain_node;
	struct list_head	completed_node;
	void			*hw_desc;
	struct mv_m2m_desc_slot	*group_head;
	u16			in_use;
	size_t			unmap_len;
	struct list_head	tx_list;
	struct dma_async_tx_descriptor	async_tx;
};

/**
 * struct mv_m2m_desc - hardware descriptor
 * @ctl: control field
 * @dma_src: transfer source address
 * @dma_dest: transfer destination address
 * @xlate: value to be written into tranlate register
 * @dma_next: address of next descriptor
 */
struct mv_m2m_desc {
	unsigned int ctl;
	unsigned int dma_src;
	unsigned int dma_dest;
	unsigned int xlate;
	unsigned int dma_next;
};

#define to_mv_m2m_chan(chan)		\
	container_of(chan, struct mv_m2m_chan, dmachan)

#define to_mv_m2m_slot(tx)		\
	container_of(tx, struct mv_m2m_desc_slot, async_tx)

#define mv_chan_to_devp(chan)           \
	((chan)->dmadev.dev)

#endif
