/*
 * Cyclic dma support for the Marvell 61xx CDMA Controller
 *
 * Derived from drivers/dma/TBD
 *
 * Copyright (C) 2007-2008 Atmel Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */
#include <linux/delay.h>
#include <linux/dmaengine.h>
#include <linux/dma-mapping.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/io.h>
#include <linux/mm.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/slab.h>
#include <linux/stat.h>
#include <linux/jiffies.h>
#include <asm/atomic.h>
#include "cdma.h"

/* #define CYCLIC_DEBUG */

/*
 * This provides cyclic dma support for the Marvell Central DMA Controller
 * used in the Marvell 88PA61x0.
 */


/**
 * mv61vc_cyclic_dma_tasklet_handler - provides cyclic support to tasklet handler
 * @mv61p: top physical dma control structure
 * @mv61v: top virtual dma control structure
 * @mv61vc: virtual channel control structure
 *
 * Called from mv61_dma_tasklet() in bottom half context.
 * Call with mv61p->biglock released.
 * Call with mv61vc->lock released.
 */
void mv61vc_cyclic_dma_tasklet_handler(struct mv61_dma 	*mv61p,
				struct mv61_vdma	*mv61v,
				struct mv61_vdma_chan	*mv61vc)
{
	unsigned long 			biglockflags;
	unsigned long 			lockvcflags;
	dma_async_tx_callback		callback = NULL;
	void				*param = NULL;
	u32 				interrupts;

	spin_lock_irqsave(&mv61p->biglock, biglockflags);
	spin_lock_irqsave(&mv61vc->lock, lockvcflags);
	interrupts = mv61vc->irqs;
	mv61vc->irqs = 0;
	if(interrupts & CDMA_INTPEND_TERMCNTPEND_MASK) {
		struct mv61_desc *desc = mv61vc_first_active(mv61vc);
		if(desc) {
			struct dma_async_tx_descriptor *txd = &desc->txd;

			callback = txd->callback;
			param = txd->callback_param;
		}
	}
	spin_unlock_irqrestore(&mv61vc->lock, lockvcflags);
	spin_unlock_irqrestore(&mv61p->biglock, biglockflags);

	if (callback)
		callback(param);

	if(interrupts & CDMA_INTPEND_TIMEOUTPEND_MASK)
		mv61vc_handle_error(mv61v, mv61vc);
}

/*
 * mv61vc_prep_dma cyclic - dmaengine API for preparing cyclic dma transaction
 * @chan: dmaengine API subset of channel control structure
 * @buf_addr: buffer physical address
 * @buf_len: buffer total buffer length in bytes
 * @period_len: bytes to transfer before callback
 * @direction: dma direction defined by linux/dma_mapping.h
 *
 * Note: The hardware requires the first descriptor to be directly loaded by
 * software, and the transaction descriptor was never intended to be auto-loaded.
 * To retrofit cyclic support with minimal disruption to existing code,
 * an extra struct mv61_chain link descriptor is added at the end of the chain
 * to replace the initial descriptor.
 * For example, the run sequence will be:
 * T->B->C->D->A->B->C->D->A->B->C->D, etc.
 * T = directly loaded transaction descriptor, dma from buf_addr, links to B
 * B = auto loaded link descriptor,dma from buf_addr + (1 * period), links to C
 * C = auto loaded link descriptor, dma from buf_addr + (2 * period), links to D
 * D = auto loaded link descriptor, dma from buf_addr + (3 * period), links to A
 * A = auto loaded link descriptor, dma from buf_addr, links to B
 */
struct dma_async_tx_descriptor *
mv61vc_prep_dma_cyclic(struct dma_chan *chan,
				dma_addr_t buf_addr, size_t buf_len,
				size_t period_len,
				enum dma_transfer_direction direction,
				unsigned long flags)
{
	struct mv61_vdma_chan	*mv61vc = NULL;
	struct mv61_vdma	*mv61v = NULL;
	struct mv61_desc	*desc = NULL;
	struct mv61_lli		*lli = NULL;
	struct mv61_chain	*prevlink = NULL;
	struct mv61_chain	*link;
	unsigned int		i;
	size_t			total_len = 0;
	enum mv61_dma_flow_ctrl	flowcontrol;
	unsigned int		periods;
	u32			mem;
	int			err;
	u32			tmp;

	if (unlikely(!chan))
		return NULL;

	mv61vc = dchan_to_mv61_vdma_chan(chan);
	mv61v = ddev_to_mv61_vdma(chan->device);

	if (unlikely(!buf_len || !period_len || (buf_len < period_len))) {
		__dev_vdbg(chan2dev(chan), "prep_dma_cyclic: length is zero!\n");
		return NULL;
	}

	if (unlikely(!dma_has_cap(DMA_CYCLIC, mv61v->dma.cap_mask))) {
		__dev_vdbg(chan2dev(chan), "prep_dma_cyclic: channel does not "
					"support cyclic dma, why are we here???\n");
		return NULL;
	}

	flowcontrol = MV61_CDMA_VAR_RD_FIELD(mv61vc->def.CFG,CDMA_CFG_FLOWCTRL);

	if(unlikely(flowcontrol == MV61_DMA_PERIPHERAL_TO_PERIPHERAL)) {
		__dev_vdbg(chan2dev(chan), "prep_dma_cyclic: channel does not "
					"support PERIPHERAL_TO_PERIPHERAL dma!\n");
		return NULL;
	}

	tmp = CDMA_CONTROL_TRANSSIZE_MASK >> CDMA_CONTROL_TRANSSIZE_SHIFT;
	if(period_len > tmp) {
		__dev_vdbg(chan2dev(chan), "prep_dma_cyclic: requested period "
						"0x%08x exceeds limit 0x%08x!\n",
						period_len, tmp);
		return NULL;
	}

	periods = buf_len / period_len;
	mem = (u32)buf_addr;

	for (i = 0; i <= periods; i++) {
		if (i == 0) {
			/* need initial transaction descriptor, will not be auto-loaded */
			__dev_vdbg(chan2dev(chan), "prep_dma_cyclic: trans desc\n");
			desc = mv61_desc_get(mv61vc);
			if (!desc) {
				dev_err(chan2dev(chan),
					"not enough descriptors available\n");
				goto err_desc_get;
			}
			desc->txregs.CFG = mv61vc->def.CFG;
			desc->txregs.FillValue = 0;
			desc->txregs.TimerControl = mv61vc->def.TimerControl;
			desc->txregs.intEn = 0;
			if (mv61v->mv61p->reva)
				MV61_CDMA_VAR_WR_FIELD(mv61vc->def.Control, CDMA_CONTROL_OWNWRITEDISABLE_A0, 1);
			else
				MV61_CDMA_VAR_WR_FIELD(mv61vc->def.Control, CDMA_CONTROL_OWNWRITEDISABLE, 1);
			if (mv61vc->def.TimerControl & CDMA_TIMERCONTROL_TIMERENABLE_MASK)
				MV61_CDMA_VAR_WR_FIELD(desc->txregs.intEn,
						CDMA_INTEN_TIMEOUTEN, 1);
			MV61_CDMA_VAR_WR_FIELD(desc->txregs.intEn,
						CDMA_INTEN_CHAINDONEEN, 1);
			MV61_CDMA_VAR_WR_FIELD(desc->txregs.intEn,
						CDMA_INTEN_CLEARCOMPLETEEN, 1);
			MV61_CDMA_VAR_WR_FIELD(desc->txregs.intEn,
						CDMA_INTEN_TERMCNTEN, 1);
			lli = &desc->lli;
			desc->chains = 1;
			err = mv61vc_prep_slave_lli(lli, mv61vc, direction, flowcontrol,
							mem, period_len, total_len);
			lli->NextCtrl |= DESCRIPTOR_NEXTCTRL_INT_MASK;
			if(err)
				goto err_cfg;
			__dev_vdbg(chan2dev(chan), "prep_dma_cyclic: desc = %p\n", desc);
		}
		else {
			/* these descriptors will always be auto-loaded */
			if(i == periods) {
				/* last descriptor will be an auto-loadable version
				 * of the transaction descriptor.
				 */
				mem = (u32)buf_addr;
			}

			__dev_vdbg(chan2dev(chan), "prep_dma_slave: lli desc\n");
			link = mv61_link_get(mv61vc);
			if (!link) {
				dev_err(chan2dev(chan),
					"not enough descriptors available\n");
				goto err_desc_get;
			}
			BUG_ON(!desc);
			desc->links++;
			if(prevlink) {
				prevlink->lli.NextCtrl = link->phys & DESCRIPTOR_NEXTCTRL_NEXT_MASK;
				prevlink->lli.NextCtrl |= DESCRIPTOR_NEXTCTRL_INT_MASK;

				/* the lli descriptor is always DMA_TO_DEVICE */
				dma_sync_single_for_device(chan2parent(chan),
						prevlink->phys,
						sizeof(prevlink->lli),
						DMA_TO_DEVICE);
				prevlink->synced = 1;
			} else {
				/* previous link was top of subchain */
				desc->lli.NextCtrl = link->phys & DESCRIPTOR_NEXTCTRL_NEXT_MASK;
				desc->lli.NextCtrl |= DESCRIPTOR_NEXTCTRL_INT_MASK;
			}
			list_add_tail(&link->chain_node, &desc->subchain);
			lli = &link->lli;
			prevlink = link;
			err = mv61vc_prep_slave_lli(lli, mv61vc, direction, flowcontrol,
							mem, period_len, 0);
			if(err)
				goto err_cfg;
		}

		if(i < periods) {
			desc->sublen += period_len;
			total_len +=period_len;
			mem +=period_len;
		}
	}

	/* last link replaces transaction descriptor after first pass, so copy its link pointer */
	if(likely(lli)) {
		lli->NextCtrl = desc->lli.NextCtrl;
	}

	if(likely(prevlink)) {
		dma_sync_single_for_device(chan2parent(chan),
				prevlink->phys, sizeof(prevlink->lli),
				DMA_TO_DEVICE);
		prevlink->synced = 1;
	}

	desc->len = total_len;

#ifdef CYCLIC_DEBUG
	mv61_tx_dump(desc);
#endif
	return (&desc->txd);

err_cfg:
err_desc_get:
	mv61_desc_put(desc);
	return NULL;
}
