/*
 * Marvell Pegmatite SoC ips timer handling.
 *
 * There are 4 timers in the ips apb that can be set up to generate
 * per timer interrupts.  We use them here to provide cpu local clockevents
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/init.h>
#include <linux/platform_device.h>
#include <linux/kernel.h>
#include <linux/clk.h>
#include <linux/timer.h>
#include <linux/clockchips.h>
#include <linux/cpu.h>
#include <linux/interrupt.h>
#include <linux/of.h>
#include <linux/of_irq.h>
#include <linux/of_address.h>
#include <linux/irq.h>
#include <linux/module.h>
#include <linux/sched_clock.h>

#define MAX_DELTA		(0xffffffff)
#define MIN_DELTA		(1)

#define TTCR(x)			((x * 0x10) + 0x4)
#define TCR(x)			((x * 0x10) + 0x8)
#define TIAR			0x20

struct pegmatite_timer_ips {
	struct clock_event_device evt;
	struct irqaction timer_irq;
	char name[16];
	int timer;
	int cpu;
	void __iomem *timer_base;
};

static DEFINE_PER_CPU(struct pegmatite_timer_ips, pegmatite_timer_ips_clockevent);

static irqreturn_t pegmatite_timer_ips_interrupt(int irq, void *dev_id)
{
	struct clock_event_device *evt = dev_id;
	struct pegmatite_timer_ips *ips = container_of(evt, struct pegmatite_timer_ips, evt);

	/*
	 * Clear pending interrupt status.
	 */
	__raw_writel(1 << ips->timer, ips->timer_base + TIAR);

	evt->event_handler(evt);

	return IRQ_HANDLED;
}

static int pegmatite_timer_ips_set_next_event(unsigned long delta,
				struct clock_event_device *evt)
{
	struct pegmatite_timer_ips *ips = container_of(evt, struct pegmatite_timer_ips, evt);

	/*
	 * Disable timer.
	 */
	writel(0, ips->timer_base + TCR(ips->timer));

	/*
	 * Setup new clockevent timer value.
	 */
	writel(delta, ips->timer_base + TTCR(ips->timer));

	/*
	 * Enable the timer for 1us period, non-continuous mode, and stop on terminal count
	 */
	writel(1, ips->timer_base + TCR(ips->timer));

	return 0;
}

static void pegmatite_timer_ips_set_mode(enum clock_event_mode mode,
			   struct clock_event_device *evt)
{
	struct pegmatite_timer_ips *ips = container_of(evt, struct pegmatite_timer_ips, evt);

	switch (mode) {
	case CLOCK_EVT_MODE_ONESHOT:
	case CLOCK_EVT_MODE_UNUSED:
	case CLOCK_EVT_MODE_SHUTDOWN:
		/*
		 * Disable timer.
		 */
		__raw_writel(0, ips->timer_base + TCR(ips->timer));
		writel(1 << ips->timer, ips->timer_base + TIAR);
		break;
	case CLOCK_EVT_MODE_RESUME:
	case CLOCK_EVT_MODE_PERIODIC:
		break;
	}
}
static void pegmatite_timer_ips_setup(struct pegmatite_timer_ips *ips)
{
	int cpu = smp_processor_id();
	struct clock_event_device *evt;
	struct irqaction *timer_irq;

	if(ips->cpu != cpu) {
		pr_err("No ips timer for cpu %d\n", cpu);
		return;
	}

	evt = &ips->evt;
	timer_irq = &ips->timer_irq;

	/*
	 * Create a unique name for the clockevent and irq
	 */
	snprintf(ips->name, sizeof(ips->name), "local timer %d", cpu);

	/*
	 * Initialize a clockevent localized to this cpu
	 */
	evt->name = ips->name;
	evt->features = CLOCK_EVT_FEAT_ONESHOT;
	evt->rating = 300;
	evt->set_next_event = pegmatite_timer_ips_set_next_event;
	evt->set_mode = pegmatite_timer_ips_set_mode;
	evt->cpumask = cpumask_of(cpu);

	/*
	 * Force the interrupt to this cpu only
	 */
	timer_irq->name = ips->name;
	timer_irq->flags = IRQF_TIMER | IRQF_IRQPOLL | IRQF_NOBALANCING;
	timer_irq->handler = pegmatite_timer_ips_interrupt;
	timer_irq->dev_id = evt;
	setup_irq(evt->irq, timer_irq);
	irq_force_affinity(evt->irq, cpumask_of(cpu));

	clockevents_config_and_register(evt, USEC_PER_SEC, MIN_DELTA, MAX_DELTA);
}

static void pegmatite_timer_ips_stop(struct pegmatite_timer_ips *ips)
{
	int cpu = smp_processor_id();
	struct clock_event_device *evt;
	struct irqaction *timer_irq;

	if(ips->cpu != cpu) {
		pr_err("No ips timer for cpu %d\n", cpu);
		return;
	}

	evt = &ips->evt;
	timer_irq = &ips->timer_irq;

	/*
	 * There is no way to unregister a clockevent, so just mark as unused
	 */
	evt->set_mode(CLOCK_EVT_MODE_UNUSED, evt);
	remove_irq(evt->irq, timer_irq);
}

static int pegmatite_timer_ips_cpu_notify(struct notifier_block *self,
				      unsigned long action, void *hcpu)
{
	switch (action & ~CPU_TASKS_FROZEN) {
	case CPU_STARTING:
		pegmatite_timer_ips_setup(this_cpu_ptr(&pegmatite_timer_ips_clockevent));
		break;
	case CPU_DYING:
		pegmatite_timer_ips_stop(this_cpu_ptr(&pegmatite_timer_ips_clockevent));
		break;
	}

	return NOTIFY_OK;
}

static struct notifier_block pegmatite_timer_ips_cpu_nb = {
	.notifier_call = pegmatite_timer_ips_cpu_notify,
};

static void __init pegmatite_timer_ips_init(struct device_node *np)
{
	struct clk *clk;
	int irq, ret, timer, i = 1;
	struct pegmatite_timer_ips *ips;
	struct device_node *child;
	void __iomem *timer_base;

	clk = of_clk_get(np, 0);
	if (IS_ERR(clk)) {
		ret = PTR_ERR(clk);
		pr_err("failed to get clock for ips timers (%d)\n", ret);
		goto out;
	}

	ret = clk_prepare_enable(clk);
	if (ret) {
		pr_err("failed to enable timer clock for ips timers (%d)\n",
		       ret);
		goto err_clk_enable;
	}

	timer_base = of_iomap(np, 0);
	if (!timer_base) {
		ret = -ENOMEM;
		goto err;
	}

	for_each_child_of_node(np, child) {
		if (!of_device_is_available(child)) {
			continue;
		}

		irq = irq_of_parse_and_map(child, 0);
		if(!irq)
			continue;

		ret = of_property_read_u32(child, "reg", &timer);
		if(ret)
			continue;

		ips = &per_cpu(pegmatite_timer_ips_clockevent, i);
		if(ips) {
			/* Disable timer and ensure IRQ line is clear */
			writel(0, timer_base + TCR(timer));
			writel(1 << timer, timer_base + TIAR);
			ips->evt.irq = irq;
			ips->timer = timer;
			ips->timer_base = timer_base;
			ips->cpu = i;
			i++;
		} else {
			/* We have more timers than cpus */
			break;
		}
	}

	ret = register_cpu_notifier(&pegmatite_timer_ips_cpu_nb);
	if (ret)
		pr_err("%s: notify register failed!! %d\n",__func__, ret);

	return;

err:
	clk_disable_unprepare(clk);
err_clk_enable:
	clk_put(clk);
out:
	pr_err("Failed to get timer from device tree with error:%d\n", ret);
}

CLOCKSOURCE_OF_DECLARE(pegmatite_timer_ips, "marvell,pegmatite-timer-ips", pegmatite_timer_ips_init);
