/*
 * clk-fan-in
 *
 * Controls multiple clk parents with a single virtual clock
 * Allows drivers that only know about a single clock to control multiple
 * clocks at once. Re-evaluate when genpd gets OF clock support
 * (3.19-3.20?)
 *
 * Copyright 2015 Lexmark International Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/kernel.h>
#include <linux/clkdev.h>
#include <linux/clk-provider.h>
#include <linux/of.h>

struct fan_in_clk {
	struct clk_hw hw;
	struct clk **clks;
	const char *parent_name;
	int num_clks;
};

#define to_fan_in_clk(p) container_of(p, struct fan_in_clk, hw)

static int fan_in_enable(struct clk_hw *hw)
{
	struct fan_in_clk *clk = to_fan_in_clk(hw);
	int i, rc = 0;

	for (i = 0; i < clk->num_clks; i++) {
		rc = clk_enable(clk->clks[i]);
		if (rc) {
			while (i-- > 0)
				clk_disable(clk->clks[i]);
			return rc;
		}
	}
	return rc;
}

static int fan_in_prepare(struct clk_hw *hw)
{
	struct fan_in_clk *clk = to_fan_in_clk(hw);
	int i, rc = 0;

	for (i = 0; i < clk->num_clks; i++) {
		rc = clk_prepare(clk->clks[i]);
		if (rc) {
			while (i-- > 0)
				clk_unprepare(clk->clks[i]);
			return rc;
		}
	}
	return rc;
}

static void fan_in_disable(struct clk_hw *hw)
{
	struct fan_in_clk *clk = to_fan_in_clk(hw);
	int i;

	for (i = 0; i < clk->num_clks; i++) {
		clk_disable(clk->clks[i]);
	}
}

static void fan_in_unprepare(struct clk_hw *hw)
{
	struct fan_in_clk *clk = to_fan_in_clk(hw);
	int i;

	for (i = 0; i < clk->num_clks; i++) {
		clk_unprepare(clk->clks[i]);
	}
}

static const struct clk_ops fan_in_clk_ops = {
	.enable		= fan_in_enable,
	.prepare	= fan_in_prepare,
	.disable	= fan_in_disable,
	.unprepare	= fan_in_unprepare,
};

static void __init of_fan_in_clk_setup(struct device_node *node)
{
	struct clk_init_data init;
	struct clk *clk;
	int i;
	int num_clks;
	struct fan_in_clk *fan_clk = kzalloc(sizeof(*fan_clk), GFP_KERNEL);

	if (WARN_ON(!fan_clk))
		goto err;

	num_clks = of_property_count_strings(node, "clock-names");
	if (WARN_ON(num_clks <= 0))
		goto err;
	fan_clk->clks = kmalloc(sizeof(fan_clk->clks[0]) * num_clks, GFP_KERNEL);
	if (WARN_ON(!fan_clk->clks))
		goto err;

	for (i = 0; i < num_clks; i++) {
		fan_clk->clks[i] = of_clk_get(node, i);
		if (WARN_ON(IS_ERR(fan_clk->clks[i])))
			goto err;
		fan_clk->num_clks = i+1;
	}

	fan_clk->parent_name = __clk_get_name(fan_clk->clks[0]);

	init.name = node->name;
	init.ops = &fan_in_clk_ops;
	/*
	 * Set the first clock as the parent. It will show up there in the tree
	 * rather than being an orphan. There's not a way to see the other
	 * clocks from user space.
	 */
	init.parent_names = &fan_clk->parent_name;
	init.num_parents = 1;
	init.flags = 0;
	fan_clk->hw.init = &init;

	clk = clk_register(NULL, &fan_clk->hw);
	if(WARN_ON(IS_ERR(clk)))
		goto err;

	of_clk_add_provider(node, of_clk_src_simple_get, clk);

	return;
err:
	if (fan_clk) {
		if (fan_clk->clks) {
			for (i = 0; i < fan_clk->num_clks; i++)
				clk_put(fan_clk->clks[i]);
			kfree(fan_clk->clks);
		}
		kfree(fan_clk);
	}
}

CLK_OF_DECLARE(fan_in_clock, "peg,clock-fan-in", of_fan_in_clk_setup);
