/*
 * Device Tree support for Armada 370 and XP platforms.
 *
 * Copyright (C) 2012 Marvell
 *
 * Lior Amsalem <alior@marvell.com>
 * Gregory CLEMENT <gregory.clement@free-electrons.com>
 * Thomas Petazzoni <thomas.petazzoni@free-electrons.com>
 *
 * This file is licensed under the terms of the GNU General Public
 * License version 2.  This program is licensed "as is" without any
 * warranty of any kind, whether express or implied.
 */

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/of_address.h>
#include <linux/of_platform.h>
#include <linux/of_irq.h>
#include <linux/clk-provider.h>
#include <linux/clocksource.h>
#include <linux/io.h>
#include <linux/irqchip.h>
#include <linux/dma-mapping.h>
#include <linux/sizes.h>
#include <linux/pci.h>
#include <asm/mach/arch.h>
#include <asm/mach/map.h>
#include <asm/mach/time.h>
#include "pegmatite.h"

#include <linux/platform_data/gpio_backlight.h>
#include <linux/input.h>
#include <linux/gpio.h>
#include <linux/gpio_keys.h>

// do this to allow low power idle .ko to hook into idle..
#include <asm/system_misc.h>
EXPORT_SYMBOL(arm_pm_idle);

static const struct of_device_id of_wdt_table[] = {
	{.compatible = "marvell,pegmatite-wdt"},
	{ /* end of list */ },
};

/* Create a dummy gpio backlight device
 * FIXME: Move this code to device tree
 */
static struct gpio_backlight_platform_data dummy_gpio_backlight_data = {
	.fbdev = NULL,
	.gpio = 197,
	.def_value = 1,
	.active_low = true,
	.name = "dummy-backlight",
};

static struct platform_device dummy_gpio_backlight_device = {
	.name = "gpio-backlight",
	.dev = {
		.platform_data = &dummy_gpio_backlight_data,
	},
};

static struct platform_device *platform_devices[] __initdata = {
	&dummy_gpio_backlight_device,
};

#define TWR 0x0
#define TTCR 0x4
#define TCR 0x8

#define WDT_ENABLE		0x80000000
#define WDT_CONTINUOUS_MODE	0x2
#define WDT_TIMER_ENABLE	0x1

/* These  DMA functions should behave the same as the generic ARM
 * functions, but with an additional sanity check to verify that we
 * don't try to DMA to DRAM above 2GB, because HW can't address that
 * memory.  Unfortunately the implementation for arm_dma_map_page is
 * static, so we can't just call them. */
static dma_addr_t pegmatite_map_page(struct device *dev, struct page *page,
				     unsigned long offset, size_t size,
				     enum dma_data_direction dir,
				     struct dma_attrs *attrs)
{
	dma_addr_t addr = pfn_to_dma(dev, page_to_pfn(page)) + offset;
	BUG_ON(page_to_phys(page) > 0x80000000);
	dma_sync_single_for_device(dev, addr, size, dir);
	return addr;
}

static int pegmatite_map_sg(struct device *dev, struct scatterlist *sg, int nents,
			    enum dma_data_direction dir, struct dma_attrs *attrs)
{
	int i;
	for (i=0; i<nents; i++)
		BUG_ON(page_to_phys(sg_page(&sg[i])) > 0x80000000);
	return arm_dma_map_sg(dev, sg, nents, dir, attrs);
}

static struct dma_map_ops pegmatite_dma_ops;

static void init_dma_ops (void)
{
	memcpy(&pegmatite_dma_ops, &arm_dma_ops, sizeof(pegmatite_dma_ops));
	pegmatite_dma_ops.map_page = pegmatite_map_page;
	pegmatite_dma_ops.map_sg = pegmatite_map_sg;
}

static int dma_notifier(struct notifier_block *nb,
			unsigned long event, void *__dev)
{
	struct device *dev = __dev;

	if (event != BUS_NOTIFY_ADD_DEVICE)
		return NOTIFY_DONE;
	set_dma_ops(dev, &pegmatite_dma_ops);

	return NOTIFY_OK;
}

static struct notifier_block dma_nb = {
	.notifier_call = dma_notifier,
};

static int __init dma_init(void)
{
	init_dma_ops();
	bus_register_notifier(&platform_bus_type,
			      &dma_nb);
	return 0;
}
postcore_initcall(dma_init);

#if IS_ENABLED(CONFIG_PCI)
static struct notifier_block pci_dma_nb = {
	.notifier_call = dma_notifier,
};

static int __init pci_dma_init(void)
{
	init_dma_ops();
	bus_register_notifier(&pci_bus_type,
			       &pci_dma_nb);
	return 0;
}
arch_initcall(pci_dma_init);
#endif

void pegmatite_restart(enum reboot_mode mode, const char *cmd)
{
        static void __iomem *wdt_reg;
	struct device_node *np;

	np = of_find_matching_node(NULL, of_wdt_table);
	if (np) {
		wdt_reg = of_iomap(np, 0);
	} else {
		pr_err("%s: Couldn't get watchdog address!! Can't reboot!!!!\n", __func__);
		while (1)
			;
	}

	/* Set the count register to something small */
	writel(0x00000055, wdt_reg + TTCR);

	/* Set timer enable, continuous mode, 1us timebase */
	writel(WDT_TIMER_ENABLE | WDT_CONTINUOUS_MODE, wdt_reg + TCR);

	/* Enable watchdog */
	writel(WDT_ENABLE, wdt_reg + TWR);

	while (1)
		;
}

static struct map_desc pegmatite_io_desc[] __initdata = {
	{
		.virtual	= (unsigned long) PEGMATITE_REGS_VIRT_BASE,
		.pfn		= __phys_to_pfn(PEGMATITE_REGS_PHYS_BASE),
		.length		= PEGMATITE_REGS_SIZE,
		.type		= MT_DEVICE,
	},
	{
		.virtual  	= (unsigned long) PEGMATITE_UPC_VIRT_BASE,
		.pfn      	= __phys_to_pfn(PEGMATITE_UPC_PHYS_BASE),
		.length 	= 0x000C0000,
		.type 		= MT_DEVICE
	},
};

void __init pegmatite_map_io(void)
{
	iotable_init(pegmatite_io_desc, ARRAY_SIZE(pegmatite_io_desc));
}

void __init pegmatite_timer_and_clk_init(void)
{
	of_clk_init(NULL);
	clocksource_of_init();
}

void __init pegmatite_init_early(void)
{
}

static void __init pegmatite_init_irq(void)
{
	irqchip_init();
}

static void __init pegmatite_dt_init(void)
{
	/* Add devices not supported by device tree */
	platform_add_devices(platform_devices, ARRAY_SIZE(platform_devices));

	of_platform_populate(NULL, of_default_bus_match_table, NULL, NULL);
}

static const char * const pegmatite_dt_compat[] = {
	"marvell,pegmatite",
	NULL,
};

DT_MACHINE_START(PEGMATITE_DT, "Marvell Pegmatite (Device Tree)")
#ifdef CONFIG_SMP
	.smp		= smp_ops(pegmatite_smp_ops),
#endif
	.init_machine	= pegmatite_dt_init,
	.map_io		= pegmatite_map_io,
	.init_early	= pegmatite_init_early,
	.init_irq	= pegmatite_init_irq,
	.init_time	= pegmatite_timer_and_clk_init,
	.restart	= pegmatite_restart,
	.dt_compat	= pegmatite_dt_compat,
#ifdef CONFIG_ZONE_DMA
	.dma_zone_size	= SZ_256M,
#endif
MACHINE_END

/* RICOH add for loadable kernel module */
void console_verbose_exp(void)
{
	console_verbose();
}
EXPORT_SYMBOL(console_verbose_exp);
