/*
 * CIU support for Pegmatite platforms.
 *
 * This file is licensed under the terms of the GNU General Public
 * License version 2.  This program is licensed "as is" without any
 * warranty of any kind, whether express or implied.
 *
 * The CPU interface controller provides various registers to configure
 * the various compute cores in the system
 */
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/of_address.h>
#include <linux/io.h>
#include <linux/smp.h>
#include <asm/smp_plat.h>
#include "apb_config.h"

static void __iomem *ciu_base;

#define CIU_CHIP_ID 0x0
#define REV_ID_SHIFT 16
#define REV_A0 0x0000
#define REV_B0 0x0010
#define REV_C0 0x0020

#define A53_CORE_0_CFG_CTRL 0x8
#define A53_CORE_1_CFG_CTRL 0x88
#define A53_CORE_2_CFG_CTRL 0xe4
#define A53_CORE_3_CFG_CTRL 0xe8

#define A53_CORE_X_CFG_CTRL_ARCH_MODE_MASK 0x8000000
#define A53_CORE_X_CFG_CTRL_VINITHI_MASK 0x1

#define MC_PRIORITY  0xb0
#define MC_PRIORITY2 0xb4

#define BOOT_CTRL 0xc4
#define REVA_BOOT_FROM_D1000000 (1 << 3)

#define REVB_ENABLE_REMAP (1 << 7)
#define REVB_HIVEC_NOT_LOVEC (1 << 5)
#define REVB_REMAP_D1000000 (1 << 4)
#define REVB_BOOT_FROM_D1000000 (REVB_REMAP_D1000000 | REVB_HIVEC_NOT_LOVEC | REVB_ENABLE_REMAP)

static const struct of_device_id of_ciu_table[] = {
	{.compatible = "marvell,pegmatite-ciu"},
	{ /* end of list */ },
};

static int is_reva(void) {
	int rev;
	rev = (readl(ciu_base + CIU_CHIP_ID) >> REV_ID_SHIFT);
	if(rev == REV_A0)
		return 1;

	return 0;
}

#ifdef CONFIG_SMP
int setup_boot_from_squ(unsigned int cpu)
{
	int *core_cfg_addr;
	int val;

	if (!ciu_base) {
		pr_warn("Can't boot CPU. CIU is uninitialized\n");
		return 1;
	}

	/* Select the correct CIU configuration control register for the requested cpu */
	switch(cpu) {
		case 0:
			core_cfg_addr = ciu_base + A53_CORE_0_CFG_CTRL;
			break;
		case 1:
			core_cfg_addr = ciu_base + A53_CORE_1_CFG_CTRL;
			break;
		case 2:
			core_cfg_addr = ciu_base + A53_CORE_2_CFG_CTRL;
			break;
		case 3:
			core_cfg_addr = ciu_base + A53_CORE_3_CFG_CTRL;
			break;
		default:
			printk(KERN_ERR "%s: invalid cpu for secondary boot %d\n", __func__, cpu);
			return -ENODEV;
	}

	val = readl(core_cfg_addr);

	/* Set VINITHI, this makes the reset vector in the SQU */
	val |= A53_CORE_X_CFG_CTRL_VINITHI_MASK;

	/* Make sure arch mode is not set so the core boots in aarch32 */
	val &= ~A53_CORE_X_CFG_CTRL_ARCH_MODE_MASK;

	writel(val, core_cfg_addr);

	/* Set the boot control register to SQU bank 0, 0xd1000000 */
	if(!is_fpga()) {
		if(is_reva()) {
			writel(REVA_BOOT_FROM_D1000000, ciu_base + BOOT_CTRL);
		} else {
			writel(REVB_BOOT_FROM_D1000000, ciu_base + BOOT_CTRL);
		}
	}

	return 0;
}
#endif

int __init pegmatite_ciu_init(void)
{
	struct device_node *np;

	np = of_find_matching_node(NULL, of_ciu_table);
	if (np) {
		u32 tmp;
		static void __iomem *bridge_base;

		pr_info("Initializing Pegmatite CIU\n");
		ciu_base = of_iomap(np, 0);
		bridge_base = of_iomap(np, 1);

		if (!of_property_read_u32(np, "priority", &tmp))
			writel(tmp, ciu_base + MC_PRIORITY);

		if (!of_property_read_u32(np, "priority2", &tmp))
			writel(tmp, ciu_base + MC_PRIORITY2);

		if (!of_property_read_u32(np, "cpu-out-fn-mod", &tmp))
			writel(tmp, bridge_base);

		iounmap(bridge_base);
	}

	return 0;
}

early_initcall(pegmatite_ciu_init);
