/*
**************************************************************************************
*
* Copyright (c) 2012-2014 Marvell International, Ltd. All Rights Reserved
*
**************************************************************************************
*
* Marvell GPL License
*
* This file licensed to you in accordance with the terms and conditions of the General
* Public License Version 2, June 1991 (the "GPL License"). You can redistribute it
* and/or modify it under the terms of the GPL License; either version 2 of the License,
* or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful, but WITHOUT ANY
* WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
* PARTICULAR PURPOSE. See the GPL License for more details.
*
* You should have received a copy of the GNU General Public License along with this
* program. If not, see http://www.gnu.org/licenses/.
*
**************************************************************************************
*/

#include <linux/version.h>
#include <linux/module.h>
#include <linux/types.h>

#include <linux/fs.h>
#include <linux/init.h>
#include <linux/kernel.h>

#include <linux/rwsem.h>
#include <linux/mutex.h>

#include <linux/uaccess.h>
#include <linux/interrupt.h>

#include <linux/slab.h>
#include <linux/dma-mapping.h>
#include <linux/dmapool.h>

#include <linux/cdev.h>
#include <linux/device.h>
#include <linux/platform_device.h>

#include <asm/string.h>   // for memset
#include <asm/io.h>       // for inl outl io access
#include <asm/div64.h>

/******************* F O R W A R D  D E C L A R A T I O N S *******************/

static int dma_alloc_open(struct inode *inode, struct file *filep);
static int dma_alloc_release(struct inode *inode, struct file *filep);
static int dma_alloc_read( struct file *filep, char __user *buf, size_t len, loff_t *ptr );
static int dma_alloc_write( struct file *filep, const char __user *buf, size_t len, loff_t *ptr);
static int dma_alloc_platform_probe(struct platform_device *pdev);
static int dma_alloc_platform_remove(struct platform_device *pdev);

#if 1 /* RICOH retry kmalloc */
extern int sgw_refrain_fcache(int );
#define REFRAIN_FCACHE_RATE 108 /* 8%(force) */
#endif

/******************** D R I V E R  D E C L A R A T I O N S ********************/


enum dma_alloc_operation_e {
    dma_operation_alloc,
    dma_operation_free,
    dma_operation_map,
    dma_operation_unmap,
    dma_operation_sync_for_cpu,
    dma_operation_sync_for_device,
    dma_operation_write_to_kernel,
    dma_operation_read_from_kernel,
    dma_operation_mmap_for_cpu,
    dma_operation_pool_alloc,
    dma_operation_pool_free,
    dma_operation_alloc_coherent,
    dma_operation_free_coherent,
#if 1 /* RICOH */
	dma_operation_alloc_no_retry,
#endif /* RICOH */
};

#define MAX_DMA_DESCRIPTOR_SIZE 64

// fileptr -> private data
struct dma_alloc_s
{
    enum dma_alloc_operation_e operation;
    size_t len;
    char *kv_addr;
    size_t hw_addr;
    void *v_addr;
    int direction;
    void  (*kfree_function)(struct dma_alloc_s *bb);
    int lines;
    void *kv_irb_adaptor;
};


typedef struct {
    int Minor;
    //int Opened; // singleton

    //struct cdev Cdev;
    //struct mutex Mutex;

} dma_alloc_dev_t;


static const struct file_operations dma_alloc_fops = {
    .owner          = THIS_MODULE,
    .open           = dma_alloc_open,
    .release        = dma_alloc_release,
    .read           = dma_alloc_read,
    .write          = dma_alloc_write,
};


typedef struct {
    int Major;
    int Minor;
    struct cdev Cdev;
    dev_t Dev;
    struct class *Class;
    struct platform_device *pdev; // = dma_alloc_dev
    dma_alloc_dev_t *dma_alloc_dev;
    struct dma_pool *pool;
} dma_alloc_drvr_t;

static dma_alloc_drvr_t dma_alloc_drvr = {
/*
  .fops = {
  .owner          = THIS_MODULE,
  .open           = dma_alloc_open,
  .release        = dma_alloc_release, // close
  .read           = dma_alloc_read,
  .write          = dma_alloc_write,
  },
*/
};

#define DRVNAME "dma_alloc"

static struct platform_driver dma_alloc_platform_driver = {
    .driver = {
        .owner = THIS_MODULE,
        .name =  DRVNAME,
    },
    .probe = dma_alloc_platform_probe,
    .remove = dma_alloc_platform_remove,
    // .suspend = dma_alloc_suspend,
    // .resume = dma_alloc_resume,
};


/******************** Direct Kernel API ********************/
/// Allocate an uncached memory block.
/// returns virtual address to uncached memory and a physical address for hardware direct access
/// note the CPU virtual address and the Hardware physical address may be different on some
/// platforms such as linux.
void * memMallocPhysical(
    void **physicalAddress, ///< return physical address may be different from virtual address.
    uint32_t size, ///< number of bytes to allocate
    uint32_t align,  ///< alignment in power of 2; ignored on Linux
    const char *file, const int line  ///< Debug; currently ignored on Linux
    )
{
    BUG_ON( size > MAX_DMA_DESCRIPTOR_SIZE ); // lets find the largest size we need.
    BUG_ON( !dma_alloc_drvr.pool );

    return dma_pool_alloc( dma_alloc_drvr.pool, GFP_DMA | GFP_KERNEL, (dma_addr_t*)physicalAddress );
}
EXPORT_SYMBOL(memMallocPhysical);

void *memFreePhysical( void *hw_addr, void *v_addr )
{
    dma_pool_free( dma_alloc_drvr.pool, v_addr, (dma_addr_t)hw_addr);
    return 0;
}
EXPORT_SYMBOL(memFreePhysical);

void *dma_buffer_map_single( struct dma_alloc_s *dma_alloc, int direction )
{
    dma_alloc->direction = direction;
    dma_alloc->hw_addr = dma_map_single( NULL,
                                         dma_alloc->kv_addr,
                                         dma_alloc->len,
                                         direction );
    return (void*)dma_alloc->hw_addr;
}
EXPORT_SYMBOL(dma_buffer_map_single);

void dma_buffer_unmap_single( struct dma_alloc_s *dma_alloc )
{
    dma_unmap_single( NULL,
                      dma_alloc->hw_addr,
                      dma_alloc->len,
                      dma_alloc->direction );
    dma_alloc->hw_addr = 0;
}
EXPORT_SYMBOL(dma_buffer_unmap_single);

struct dma_alloc_s *dma_free( struct dma_alloc_s *dma_alloc )
{
    if ( dma_alloc->kfree_function )
        dma_alloc->kfree_function( dma_alloc );
    else
        kfree( dma_alloc->kv_addr );
    kfree(dma_alloc);
    return 0;
}
EXPORT_SYMBOL(dma_free);

struct dma_alloc_s *dma_alloc( size_t len )
{
	int ret;

    struct dma_alloc_s *dma_alloc_block = (struct dma_alloc_s *) kmalloc(sizeof(struct dma_alloc_s), GFP_KERNEL);
    if (dma_alloc_block)
    {
		dma_alloc_block->kv_addr = kmalloc(len, GFP_DMA | GFP_KERNEL );

#if 1 /* RICOH retry kmalloc */
		while (dma_alloc_block->kv_addr == 0) {
			printk(KERN_ERR " %s: refrain_fcache %d(size:%d)\n", __FUNCTION__, REFRAIN_FCACHE_RATE, dma_alloc_block->len);
			ret = sgw_refrain_fcache(REFRAIN_FCACHE_RATE);

			/* retry kmalloc */
			dma_alloc_block->kv_addr = kmalloc(len, GFP_DMA | GFP_KERNEL );
			printk(KERN_ERR " refrain_fcache ret:%d res:%p\n", ret, dma_alloc_block->kv_addr);
		}
#endif /* RICOH retry kmalloc */

        if(dma_alloc_block->kv_addr)
        {
            dma_alloc_block->len = len;
            dma_alloc_block->hw_addr = 0xdeadbeef; // must lock first.
            dma_alloc_block->v_addr = (void*)0xdeadbeef; // must map from hw_addr.
	    dma_alloc_block->direction = 0;
            dma_alloc_block->kfree_function = 0;
            dma_alloc_block->lines = 0;
            dma_alloc_block->kv_irb_adaptor = 0;
        }
        else
        {
            kfree(dma_alloc_block);
            dma_alloc_block = 0;
        }
    }
    return dma_alloc_block;
}
EXPORT_SYMBOL(dma_alloc);

struct dma_alloc_s *dma_alloc_adopt_irb_lines( void *kv_data, size_t len, int lines,
                                               void (*kfree_function)(struct dma_alloc_s *),
                                               void *kv_irb_adaptor
    )
{
    struct dma_alloc_s *dma_alloc_block = (struct dma_alloc_s *) kmalloc(sizeof(struct dma_alloc_s), GFP_KERNEL);
    if (dma_alloc_block)
    {
        dma_alloc_block->kv_addr = kv_data;
        if(dma_alloc_block->kv_addr)
        {
            dma_alloc_block->len = len;
            dma_alloc_block->hw_addr = 0xdeadbeef; // must lock first.
            dma_alloc_block->v_addr = (void*)0xdeadbeef; // must map from hw_addr.
	    dma_alloc_block->direction = 0;
            dma_alloc_block->kfree_function = kfree_function;
            dma_alloc_block->lines = lines;
            dma_alloc_block->kv_irb_adaptor = kv_irb_adaptor;
        }
        else
        {
            kfree(dma_alloc_block);
            dma_alloc_block = 0;
        }
    }
    return dma_alloc_block;
}
EXPORT_SYMBOL(dma_alloc_adopt_irb_lines);

/******************** User level Char Dev Interface ********************/
static int dma_alloc_open(struct inode *inode, struct file *filep)
{
    int retval = -EBUSY;
    struct dma_alloc_s *dma_alloc_block;

    if (!filep->private_data) {
        filep->private_data = kmalloc(sizeof(struct dma_alloc_s), GFP_KERNEL);
        BUG_ON( filep->f_op != &dma_alloc_fops );
        retval = 0;
        ((struct dma_alloc_s*) filep->private_data)->operation = -1;
    }

    return(retval);
}

// close
static int dma_alloc_release(struct inode *inode, struct file *filep)
{
    if (filep->private_data) {
        kfree(filep->private_data);
        filep->private_data = 0;
    }
    return 0;
}

static int dma_alloc_read( struct file *filep, char __user *buf, size_t len, loff_t *ptr )
{
    int retval = len;
    struct dma_alloc_s *dma_alloc_block = (struct dma_alloc_s*) filep->private_data;

    if ( 0 == dma_alloc_block || -1 == dma_alloc_block->operation ) {
        retval = 0;
    }
    else if (dma_alloc_block->operation == dma_operation_read_from_kernel)      {
        if (copy_to_user(buf, dma_alloc_block->kv_addr, len)) {
            retval = -EFAULT;
        }
    }
    else if (copy_to_user(buf, dma_alloc_block, sizeof(struct dma_alloc_s))) {
        retval = -EFAULT;
    }
    dma_alloc_block->operation = -1; // write then read once.

    return retval;
}

static int dma_alloc_write( struct file *filep, const char __user *buf, size_t len, loff_t *ptr)
{
    struct dma_alloc_s *dma_alloc_block;
    int retval = 0;
	int ret;
	
    dma_alloc_block = (struct dma_alloc_s*) filep->private_data;

    BUG_ON( !dma_alloc_block ); // write without open?

    if ( dma_alloc_block->operation == dma_operation_write_to_kernel ) {
        // must be copying data into an existing kernel buffer
        // todo copy in blocks?
        // use offset to add to existing.
        if (len <= dma_alloc_block->len) {
            if (copy_from_user(dma_alloc_block->kv_addr, buf, len)) {
                retval = -EFAULT;
                return retval;
            }
            dma_alloc_block->operation = -1;
            return len;

        }
    }
    if (copy_from_user(dma_alloc_block, buf, sizeof(struct dma_alloc_s))) {
        retval = -EFAULT;
        return retval;
    }
    retval = sizeof(struct dma_alloc_s);

    switch (dma_alloc_block->operation) {
    case dma_operation_alloc :
		dma_alloc_block->kv_addr = kmalloc(dma_alloc_block->len, GFP_DMA | GFP_KERNEL );

#if 1 /* RICOH retry kmalloc */
		while (dma_alloc_block->kv_addr == 0) {
			printk(KERN_ERR " %s: refrain_fcache %d(size:%d)\n", __FUNCTION__, REFRAIN_FCACHE_RATE, dma_alloc_block->len);
			ret = sgw_refrain_fcache(REFRAIN_FCACHE_RATE);

			/* retry kmalloc */
			dma_alloc_block->kv_addr = kmalloc(dma_alloc_block->len, GFP_DMA | GFP_KERNEL );
			printk(KERN_ERR " refrain_fcache ret:%d ptr:%p\n", ret, dma_alloc_block->kv_addr);
		}
#endif /* RICOH retry kmalloc */
        dma_alloc_block->hw_addr = 0xdeadbeef; // must lock first.
        dma_alloc_block->v_addr = (void*)0xdeadbeef; // must map from hw_addr.
        dma_alloc_block->kfree_function = 0;
        break;

    case dma_operation_free :
        if (dma_alloc_block->kfree_function)
            dma_alloc_block->kfree_function(dma_alloc_block);
        else
            kfree( dma_alloc_block->kv_addr );
        dma_alloc_block->kv_addr = 0;
        break;

    case dma_operation_map :
        dma_alloc_block->hw_addr = dma_map_single( NULL,
                                                   dma_alloc_block->kv_addr,
                                                   dma_alloc_block->len,
                                                   dma_alloc_block->direction );
        break;

    case dma_operation_unmap :
        dma_unmap_single( NULL,
                          dma_alloc_block->hw_addr,
                          dma_alloc_block->len,
                          dma_alloc_block->direction );
        // dma_alloc_block->hw_addr = 0;
        break;

    case dma_operation_sync_for_cpu :
        dma_sync_single_for_cpu( NULL,
                                 dma_alloc_block->hw_addr,
                                 dma_alloc_block->len,
                                 dma_alloc_block->direction );
        break;

    case dma_operation_sync_for_device :
        dma_sync_single_for_device( NULL,
                                    dma_alloc_block->hw_addr,
                                    dma_alloc_block->len,
                                    dma_alloc_block->direction );
        break;

    case dma_operation_pool_alloc:
    {
        dma_alloc_block->kv_addr = memMallocPhysical((void**)&dma_alloc_block->hw_addr, dma_alloc_block->len, 32, 0, 0);
        printk( KERN_DEBUG "size %d hw %x kv %p\n",
                dma_alloc_block->len, dma_alloc_block->hw_addr, dma_alloc_block->kv_addr );
    }
    break;

    case dma_operation_pool_free:
        memFreePhysical((void*)dma_alloc_block->hw_addr, (void*)dma_alloc_block->v_addr);
        break;

    case dma_operation_alloc_coherent :
        dma_alloc_block->kv_addr = dma_alloc_coherent( 0, dma_alloc_block->len, &dma_alloc_block->hw_addr, GFP_DMA | GFP_KERNEL );
        break;

    case dma_operation_free_coherent :
        dma_free_coherent(0, dma_alloc_block->len, dma_alloc_block->kv_addr, dma_alloc_block->hw_addr);
        dma_alloc_block->len = 0;
        dma_alloc_block->kv_addr = 0;
        dma_alloc_block->hw_addr = 0;
        dma_alloc_block->v_addr = 0;
        break;

    case dma_operation_write_to_kernel :

        break;
    case dma_operation_read_from_kernel :
        if ( dma_alloc_block->kv_addr )
            ;// dma
        break;
#if 1 /* RICOH */
    case dma_operation_alloc_no_retry :
		dma_alloc_block->kv_addr = kmalloc(dma_alloc_block->len, GFP_DMA | GFP_KERNEL );
		if (!dma_alloc_block->kv_addr) {
			printk(KERN_ERR " %s: cannot allocate memory, and no retry (size:%d)\n", __FUNCTION__,  dma_alloc_block->len);
		}
        dma_alloc_block->hw_addr = 0xdeadbeef; // must lock first.
        dma_alloc_block->v_addr = (void*)0xdeadbeef; // must map from hw_addr.
        dma_alloc_block->kfree_function = 0;
        break;
#endif /* RICOH */
    default:
        retval = -EFAULT;
    }

    return retval ;
}

static int dma_alloc_platform_probe(struct platform_device *pdev)
{
    dma_alloc_dev_t *dev;

    if (!(dev = (dma_alloc_dev_t *)kzalloc(sizeof(dma_alloc_dev_t), GFP_KERNEL))) {
        dev_err(&pdev->dev, "dma_alloc kzalloc failed\n");
        return(-ENOMEM);
    }

    dma_alloc_drvr.pdev = pdev;
    dma_alloc_drvr.dma_alloc_dev = dev;

    platform_set_drvdata(pdev, dev);

#if LINUX_VERSION_CODE < KERNEL_VERSION(3, 16, 0)
    dma_alloc_drvr.pool = dma_pool_create("dma_alloc_device", 0, MAX_DMA_DESCRIPTOR_SIZE, 32, 0 );
#else
    dma_alloc_drvr.pool = dma_pool_create("dma_alloc_device", &pdev->dev, MAX_DMA_DESCRIPTOR_SIZE, 32, 0 );
#endif

    BUG_ON( !dma_alloc_drvr.pool );

    return 0;
}

static int dma_alloc_platform_remove(struct platform_device *pdev)
{
    // remove dev handle and class
    cdev_del(&dma_alloc_drvr.Cdev);
    device_destroy(dma_alloc_drvr.Class, MKDEV(dma_alloc_drvr.Major, dma_alloc_drvr.Minor));
    class_destroy(dma_alloc_drvr.Class);


    // reclaim any drv/dev-specific allocation
    if (pdev)
    {
        dma_alloc_dev_t *dev = platform_get_drvdata(pdev);
        /// remove dma_pool // todo:

        kfree(dev);
        platform_set_drvdata(pdev, NULL);
    }
    return 0;
}

static int __init dma_alloc_init(void)
{
    int err;

    dma_alloc_drvr.Minor = 1;
    dma_alloc_drvr.Class = class_create(THIS_MODULE, "dma_alloc");
    if (IS_ERR(dma_alloc_drvr.Class)) {
        printk(KERN_ERR "can't register dma_alloc character driver CLASS \n");
        return PTR_ERR(dma_alloc_drvr.Class);
    }

    if ((err = alloc_chrdev_region(&dma_alloc_drvr.Dev, 0, 1, "dma_alloc devices"))) {
        class_destroy(dma_alloc_drvr.Class);
        printk(KERN_ERR "can't register dma_alloc character driver %d\n", err);
        return(err);
    }

    dma_alloc_drvr.Major = MAJOR(dma_alloc_drvr.Dev);

    device_create(dma_alloc_drvr.Class, NULL, MKDEV(dma_alloc_drvr.Major, 1),
                  NULL, "dma_alloc");

    cdev_init( &dma_alloc_drvr.Cdev, &dma_alloc_fops);
    dma_alloc_drvr.Cdev.owner = THIS_MODULE;
    dma_alloc_drvr.Cdev.ops   = &dma_alloc_fops;

    if (cdev_add(&dma_alloc_drvr.Cdev, MKDEV(dma_alloc_drvr.Major, 1), 1) < 0) {
        printk(KERN_ERR "can't add jbig device driver\n");
        return(-1);
    }

    if ( (err = platform_driver_register(&dma_alloc_platform_driver) ) )
        goto exit;
    // sysfs ...

    printk(KERN_ERR "dma_alloc driver init \n");
    return 0;
exit:
    printk(KERN_ERR "dma_alloc driver init failure\n");
    return err;
}


static void __exit dma_alloc_exit(void)
{
    // cleanup dev handle
    dma_alloc_platform_remove(dma_alloc_drvr.pdev); // direct rather than through .remove method
    dma_alloc_drvr.pdev = NULL;

    // cleanup drv
    platform_driver_unregister(&dma_alloc_platform_driver);
    printk( KERN_ERR "%s done", __FUNCTION__ );
}

module_init(dma_alloc_init)
module_exit(dma_alloc_exit)

MODULE_AUTHOR("Copyright (c) 2012 - 2014 Marvell International Ltd. All Rights Reserved");
MODULE_DESCRIPTION("Marvell export dma interface to application space");
MODULE_VERSION("1.0");
MODULE_LICENSE("GPL");



