package jcifs.util.transport;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jcifs.UniAddress;
import jcifs.util.LogStream;

/**
 * This class simplifies communication for protocols that support
 * multiplexing requests. It encapsulates a stream and some protocol
 * knowledge (provided by a concrete subclass) so that connecting,
 * disconnecting, sending, and receiving can be syncronized
 * properly. Apparatus is provided to send and receive requests
 * concurrently.
 */

public abstract class Transport implements Runnable {

    static int id = 0;
    static LogStream log = LogStream.getInstance();

    public static int readn( InputStream in,
                byte[] b,
                int off,
                int len ) throws IOException {
        int i = 0, n = -5;

        while (i < len) {
            n = in.read( b, off + i, len - i );
            if (n <= 0) {
                break;
            }
            i += n;
        }

        return i;
    }

    /* state values
     * 0 - not connected
     * 1 - connecting
     * 2 - run connected
     * 3 - connected
     * 4 - error
     */
    int state = 0;
    
    // >> XgsdLocalFix
    // 2009/07/14
    // 接続が確立しているかどうか確認するためのList
    public static int CONN2LEN = 256;
    public static List CONNECTIONS2 = new ArrayList();
    	
    // public static List TIMEOUT_CONNECTIONS = new ArrayList(); // ADK様障害対応2011/07 WangNing Add	
    public static Map TIMEOUT_CONNECTIONS_EXPIRATION = new HashMap(); // TycoElectronics様障害対応 2014/08 Kamimoto Add

    public static UniAddress[] getAllByNameSorted(UniAddress[] addresses) {
		// DNSによって名前解決されたIPアドレス(InetAddress.getAllByName(String))をソートする.
		// まだ、接続の確立したSmbTransportが生成されていないIPアドレスは、配列の後の方にまわす.
    	// ADK障害対応のため、DCタイムアウトだったIPアドレスは配列の一番の後ろに保存
		// これにより、タイムアウト発生が低減される.
		if (addresses.length > 1) {
			// このケースでは、IPアドレスがDNSによってresolveされている.
			UniAddress[] _addresses = new UniAddress[addresses.length];
			List noConn = new ArrayList();
			List timeoutConn = new ArrayList(); // ADK障害対応　DCタイムアウトだったIPは一旦保存
			int index = 0;
			synchronized( CONNECTIONS2 ) {
//				synchronized(TIMEOUT_CONNECTIONS) {
				synchronized(TIMEOUT_CONNECTIONS_EXPIRATION) { // TycoElectronics様障害対応 
					for (int i = 0; i < addresses.length; i++) {
			    		if (CONNECTIONS2.contains(addresses[i].getHostAddress())) {
			    			_addresses[index] = addresses[i];
			    			index++;
//			    		} else if (TIMEOUT_CONNECTIONS.contains(addresses[i].getHostAddress())) { // ADK障害対応
			    		} else if (TIMEOUT_CONNECTIONS_EXPIRATION.containsKey(addresses[i].getHostAddress())) { // TycoElectronics様障害対応 
			    			timeoutConn.add(addresses[i]);                                        // ADK障害対応
			    		} else {
			    			noConn.add(addresses[i]);
			    		}
			    	}
				}
			}
			for (int i = 0; i < noConn.size(); i++) {
				_addresses[index] = (UniAddress) noConn.get(i);
				index++;
			}
			// TycoElectronics様障害対応 2014/08 Kamimoto Add
			// Expiration昇順に並び替える
			Comparator compExpirationAscending = new Comparator() {
				public int compare(Object obj1, Object obj2) {
					Long long1 = getExpiration(obj1);
					Long long2 = getExpiration(obj2);
					if (long1.longValue() > long2.longValue()) return 1;
					if (long1.longValue() < long2.longValue()) return -1;
					return 0;
				}
				private Long getExpiration(Object obj) {
					UniAddress addr = (UniAddress)obj;
					Object objValue = TIMEOUT_CONNECTIONS_EXPIRATION.get(addr.getHostAddress());
					Long longValue = (objValue instanceof Long) ? (Long)objValue : new Long(0);
					return longValue;
				}
			};
			Collections.sort(timeoutConn, compExpirationAscending);
			
			// ADK様障害対応2011/07 WangNing Add
			// DCタイムアウトだったIPアドレスは配列の一番の後ろに置く
			for (int i = 0; i < timeoutConn.size(); i++) {
				_addresses[index] = (UniAddress) timeoutConn.get(i);
				index++;
			}
			// ADK様障害対応2011/07
			addresses = _addresses;
		}
		return addresses;
	}
    // XgsdLocalFix <<

    String name = "Transport" + id++;
    Thread thread;
    TransportException te;

    protected HashMap response_map = new HashMap( 4 );

    protected abstract void makeKey( Request request ) throws IOException;
    protected abstract Request peekKey() throws IOException;
    protected abstract void doSend( Request request ) throws IOException;
    protected abstract void doRecv( Response response ) throws IOException;
    protected abstract void doSkip() throws IOException;
	
	// JCIFSバージョンアップ（V1.3.10⇒V1.3.15） 2011/06 WangNing Delete
	//public Object setupDiscoLock = new Object();
	
	//public void sendrecv( Request request,
    //                Response response,
    //                long timeout ) throws IOException {
    //    synchronized (response_map) {
	// JCIFSバージョンアップ（V1.3.10⇒V1.3.15）

    public synchronized void sendrecv( Request request,
                    Response response,
                    long timeout ) throws IOException {
            // JCIFSバージョンアップ（V1.3.10⇒V1.3.15） 2011/06 WangNing Delete
            //synchronized (response_map) {
            // JCIFSバージョンアップ（V1.3.10⇒V1.3.15）
            makeKey( request );
            response.isReceived = false;
            try {
                response_map.put( request, response );
                doSend( request );
                response.expiration = System.currentTimeMillis() + timeout;
                while (!response.isReceived) {
                	 // JCIFSバージョンアップ（V1.3.10⇒V1.3.15） 2011/06 WangNing Modify
                	//response_map.wait( timeout );
                    wait( timeout );
                     // JCIFSバージョンアップ（V1.3.10⇒V1.3.15）
                    timeout = response.expiration - System.currentTimeMillis();
                    if (timeout <= 0) {
                        throw new TransportException( name +
                                " timedout waiting for response to " +
                                request );
                    }
                }
            } catch( IOException ioe ) {
                if (log.level > 2)
                    ioe.printStackTrace( log );
                try {
                    disconnect( true );
                } catch( IOException ioe2 ) {
                    ioe2.printStackTrace( log );
                }
                throw ioe;
            } catch( InterruptedException ie ) {
                throw new TransportException( ie );
            } finally {
                response_map.remove( request );
            }
    }
    private void loop() {
        while( thread == Thread.currentThread() ) {
            try {
                Request key = peekKey();
                if (key == null)
                    throw new IOException( "end of stream" );
                // JCIFSバージョンアップ（V1.3.10⇒V1.3.15） 2011/06 WangNing Modify
                //synchronized (response_map) {
                synchronized (this) {
                // JCIFSバージョンアップ
                    Response response = (Response)response_map.get( key );
                    if (response == null) {
                        if (log.level >= 4)
                            log.println( "Invalid key, skipping message" );
                        doSkip();
                    } else {
                        doRecv( response );
                        response.isReceived = true;
                        // JCIFSバージョンアップ（V1.3.10⇒V1.3.15） 2011/06 WangNing Modify
                        //response_map.notifyAll();
                        notifyAll();
                        // JCIFSバージョンアップ
                    }
                }
            } catch( Exception ex ) {
                String msg = ex.getMessage();
                boolean timeout = msg != null && msg.equals( "Read timed out" );
                /* If just a timeout, try to disconnect gracefully
                 */
                boolean hard = timeout == false;

                if (!timeout && log.level >= 3)
                    ex.printStackTrace( log );

                try {
                    disconnect( hard );
                } catch( IOException ioe ) {
                    ioe.printStackTrace( log );
                }
            }
        }
    }

    /* Build a connection. Only one thread will ever call this method at
     * any one time. If this method throws an exception or the connect timeout
     * expires an encapsulating TransportException will be thrown from connect
     * and the transport will be in error.
     */

    protected abstract void doConnect() throws Exception;

    /* Tear down a connection. If the hard parameter is true, the diconnection
     * procedure should not initiate or wait for any outstanding requests on
     * this transport.
     */

    protected abstract void doDisconnect( boolean hard ) throws IOException;

    public synchronized void connect( long timeout ) throws TransportException {
        try {
            switch (state) {
                case 0:
                    break;
                case 3:
                    return; // already connected
                case 4:
                    state = 0;
                    throw new TransportException( "Connection in error", te );
                default:
                    TransportException te = new TransportException( "Invalid state: " + state );
                    state = 0;
                    throw te;
            }

            state = 1;
            te = null;
            thread = new Thread( this, name );
            thread.setDaemon( true );

            synchronized (thread) {
                thread.start();
                thread.wait( timeout );          /* wait for doConnect */

                switch (state) {
                    case 1: /* doConnect never returned */
                        state = 0;
                        thread = null;
                        throw new TransportException( "Connection timeout" );
                    case 2:
                        if (te != null) { /* doConnect throw Exception */
                            state = 4;                        /* error */
                            thread = null;
                            throw te;
                        }
                        state = 3;                         /* Success! */
                        return;
                }
            }
        } catch( InterruptedException ie ) {
            state = 0;
            thread = null;
            throw new TransportException( ie );
        } finally {
            /* This guarantees that we leave in a valid state
             */
            if (state != 0 && state != 3 && state != 4) {
                if (log.level >= 1)
                    log.println("Invalid state: " + state);
                state = 0;
                thread = null;
            }
        }
    }
    // JCIFSバージョンアップ（V1.3.10⇒V1.3.15） 2011/06 WangNing Modify
    public synchronized void disconnect( boolean hard ) throws IOException {
        switch (state) {
            case 0: /* not connected - just return */
                return;
            case 2:
                hard = true;
            case 3: /* connected - go ahead and disconnect */
                if (response_map.size() != 0 && !hard) {
                    break; /* outstanding requests */
                }
                doDisconnect( hard );
            case 4: /* in error - reset the transport */
                thread = null;
                state = 0;
                break;
            default:
                if (log.level >= 1)
                    log.println("Invalid state: " + state);
                thread = null;
                state = 0;
                break;
        }
    }
    // JCIFSバージョンアップ（V1.3.10⇒V1.3.15）
    public void run() {
        Thread run_thread = Thread.currentThread();
        Exception ex0 = null;

        try {
            /* We cannot synchronize (run_thread) here or the caller's
             * thread.wait( timeout ) cannot reaquire the lock and
             * return which would render the timeout effectively useless.
             */
            doConnect();
        } catch( Exception ex ) {
            ex0 = ex; // Defer to below where we're locked
            return;
        } finally {
            synchronized (run_thread) {
                if (run_thread != thread) {
                    /* Thread no longer the one setup for this transport --
                     * doConnect returned too late, just ignore.
                     */
                    if (ex0 != null) {
                    	// JCIFSバージョンアップ（V1.3.10⇒V1.3.15） 2011/06 WangNing Modify
                    	//ex0.printStackTrace();
                        if (log.level >= 2)
                            ex0.printStackTrace(log);
                    	// JCIFSバージョンアップ（V1.3.10⇒V1.3.15）
                    }
                    return;
                }
                if (ex0 != null) {
                    te = new TransportException( ex0 );
                }
                state = 2; // run connected
                run_thread.notify();
            }
        }

        /* Proccess responses
         */
        loop();
    }

    public String toString() {
        return name;
    }
    
    // TycoElectronics様障害対応  2014/08 Kamimoto Add
    // 対象AddressがTimeoutしたものかどうかを確認する
    public static boolean isTimeoutConnection(UniAddress address) {
    	Long timeout = (Long)TIMEOUT_CONNECTIONS_EXPIRATION.get(address.getHostAddress());
    	
    	// Timeout一覧に含まれていない
    	if (timeout == null) {
    		return false;
    	}
    	
    	// Timeoutの期限切れ
    	if (System.currentTimeMillis() > timeout.longValue()) {
    		return false;
    	}
    	
    	return true;
    }
}
