/*
 *
 *  Copyright (C) 2008-2009 RICOH Co.,LTD.
 *  All rights reserved.
 *
 *  affiliation	:DSS Development Center
 *  		 Document Solutions & Services Division
 * 
 *  purpose	: Provides a singleton registry for the card id & user info mappings.
 *
 */

package jp.co.ricoh.dsdk.ap.sample.localauth.auth;

import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.NoSuchElementException;
import java.util.StringTokenizer;

import jp.co.ricoh.dsdk.ap.sample.localauth.app.Activator;
import jp.co.ricoh.dsdk.ap.sample.localauth.util.FileUtil;
import jp.co.ricoh.dsdk.ap.sample.localauth.util.HexUtil;

/**
 * このクラスは、ログインユーザリストを管理するシングルトンクラスです。
 * リストには、IDと、それに対応するログインユーザ名、ログインパスワードが記録されています。
 * This is a singleton class that manages the user information list.
 * The list consists of card IDs and corresponding login user names and login passwords.
 */
public class LoginUserRegistry {
	
	private static LoginUserRegistry reg = null;
	private static final String USERFILE = "localauth.dat";
	private ArrayList users;
	
	/**
	 * LoginUserResigtryを取得します。
	 * Returns the LoginUsetRegistry.
	 */
	synchronized public static LoginUserRegistry getInstance() throws IOException {
		if (reg == null) {
			reg = new LoginUserRegistry();
			return reg;
		}
		else return reg;
	}
	
	/**
	 * LoginUserRegistryのインスタンスを生成します。
	 * Creates the LoginUserRegistroy instance.
	 */
	private LoginUserRegistry() throws IOException {
		users = new ArrayList();
		loadUsers();
	}
	
	/**
	 * インデクスを指定して、LoginUserを取得します。
	 * Returns the LoginUser with the given index.
	 */
	public LoginUser getUser(int index) {
		if (index < 0) return null;
		if (index > (users.size() - 1)) return null;
		return (LoginUser) ((LoginUser) users.get(index)).clone();
	}
	
	/**
	 * レジストリ内の全ログインユーザ数を取得します。
	 * Returns the number of login users in the registry. 
	 */
	public int getUserCounts() {
		return users.size();
	}
	
	/**
	 * 指定したIDを持つログインユーザのindexを取得します。
	 * Returns the index of the login user who has the given ID. 
	 */
	public int searchUser(byte[] id) {
		for (int i = 0; i < users.size(); i++) {
			LoginUser user = (LoginUser) users.get(i);
			if (Arrays.equals(user.getId(), id)) return i;
		}
		return -1;
	}
	
	/**
	 * LoginUserを指定して、ログインユーザを検索します。
	 * ログインユーザが見つかった場合、そのLoginUserインスタンスを返します。
	 * Look up a login user who is equal to the given LoginUser.
	 * If the login user exists, returns the LoginUser instance.
	 */
	public LoginUser searchUser(LoginUser user) {
		for (int i = 0; i < users.size(); i++) {
			LoginUser _user = (LoginUser) users.get(i);
			if (_user.equals(user)) return _user;
		}
		return null;
	}
	
	/**
	 * レジストリにログインユーザを追加します。
	 * Add a LoginUser to the registry. 
	 */
	public synchronized void addUser(LoginUser user) {
		if (searchUser(user.getId()) == -1) {
			users.add(user);
		} else {
		}
	}
	
	/**
	 * 指定したインデクスのログインユーザを、指定したLoginUserインスタンスで更新します。
	 * Updates the login user with the given index to the given LoginUser.
	 */
	public synchronized void updateUser(LoginUser user, int index) {
		if (index < 0) ;
		if (index > (users.size() - 1)) ;
		else {
			LoginUser _user = (LoginUser) users.get(index);
			_user.setId(user.getId());
			_user.setLoginName(user.getLoginName());
			_user.setLoginPassword(user.getLoginPassword());
		}
	}
	
	/**
	 * LoginUserを指定して、ログインユーザを削除します。
	 * Removes the login user who is equal to the given LoginUser.
	 */
	public synchronized void removeUser(LoginUser user) {
		LoginUser _user = searchUser(user);
		if (_user != null) {
			users.remove(_user);
		}
	}
	
	/**
	 * IDを指定して、ログインユーザを削除します。
	 * Removes the login user who is equal to the given LoginUser.
	 */
	public synchronized void removeUser(int index) {
		if (index < 0) ;
		if (index > (users.size() - 1)) ;
		users.remove(index);
	}
	
	/**
	 * ログインユーザをローカルのファイルからロードして、ログインユーザリストを生成します。
	 * ファイルの名前は、"localauth.dat"です。
	 * ファイルは、インストールパスにあります。
	 * 
	 * Loads login users from the local file and creates a login user list.
	 * The file name is "localauth.dat".
	 * The file is located in the INSTALL PATH.
	 */
	private void loadUsers() throws IOException {
		String authFilePath = Activator.filePath;
		File out = new File(authFilePath + File.separator + USERFILE);
		
		if (!out.exists()) {
			try {
				FileUtil.copyFile(getClass().getResourceAsStream(USERFILE), out);
				
			} catch (IOException e) {
				//e.printStackTrace();
				throw e;
			}
		}
		
		BufferedReader br = null;
		try {
			br = new BufferedReader(new FileReader(out));
			
			String line = null;
			while ((line = br.readLine())!= null) {				
				StringTokenizer st = new StringTokenizer(line, ",");
				
				byte[] id = HexUtil.hexStringToBytes(st.nextToken());
				
				String loginName = st.nextToken();
				
				String loginPassword = null;
				try {
					loginPassword = st.nextToken();
				} catch (NoSuchElementException e) {
					loginPassword = "";
				}
				
				users.add(new LoginUser(id, loginName, loginPassword));
			}
		
		} catch (FileNotFoundException e) {
			//e.printStackTrace();
			throw e;
			
		} catch (IOException e) {
			//e.printStackTrace();
			throw e;
				
		} finally {
			if (br != null) try {br.close();} catch (Exception e) {}
		}
	}
	
	/**
	 * 現時点のログインユーザリストで、ローカルのファイルを更新します。
	 * Updates the local file with the current login user list in the registry.
	 */
	public synchronized void saveUsers() throws IOException {
		String authFilePath = Activator.filePath;
		File out = new File(authFilePath + File.separator + USERFILE);
		
		BufferedOutputStream os = null;
		try {
			os = new BufferedOutputStream(new FileOutputStream(out));
			for (int i = 0; i < users.size(); i++) {
				LoginUser user = (LoginUser) users.get(i);
				String data = 
					HexUtil.bytesToHexString(user.getId()) + "," + 
					user.getLoginName() + "," + 
					user.getLoginPassword() + '\n';
				os.write(data.getBytes());
			}
			os.flush();
		} catch (FileNotFoundException e) {
			//e.printStackTrace();
			throw e;
		} catch (IOException e) {
			//e.printStackTrace();
			throw e;
		} finally {
			if (os != null) try {os.close();} catch (Exception e) {}
		}
	}
}
