/*
 *
 *  Copyright (C) 2008-2009 RICOH Co.,LTD.
 *  All rights reserved.
 *
 *  affiliation	:DSS Development Center
 *  		 Document Solutions & Services Division
 * 
 *  purpose	: Dummy SCF card service for PanelService sample.
 *
 */

package pssample;

import java.util.Arrays;

import jp.co.ricoh.dsdk.scard.framework.card.Card;
import jp.co.ricoh.dsdk.scard.framework.service.CardService;
import jp.co.ricoh.dsdk.scard.framework.service.CardServiceException;
import jp.co.ricoh.dsdk.scard.framework.service.UnsupportedCardException;
import jp.co.ricoh.dsdk.scard.option.security.CHVException;

/**
 * MockCardService class
 * This is a sample CardService which implements PIN verification and 
 * username and password acquisition.
 * This CardService does not communicate with the card to make this program simple.
 * PIN is "1234", username is "user", and password is "pass".
 * 
 * MockCardServiceクラス
 * このクラスは、PIN認証、ユーザ名・パスワード取得を実装するカードサービスのサンプルです。
 * 簡単のため、カードとの通信とは行いません。
 * PINは「1234」、ユーザ名は「user」、パスワードは「pass」で固定です。
 */
public class MockCardService extends CardService {
	
	/**
	 * The card PIN
	 * 
	 * カードのPIN番号
	 */
	private static final byte[] PIN = new byte[] {
		(byte)0x31, (byte)0x32, (byte)0x33, (byte)0x34
		};
	
	/**
	 * Indicates that the PIN verification succeeded
	 * 
	 * PIN認証が成功したかどうかを表すフラグ
	 */
	private boolean verified;
	
	/* (non-Javadoc)
	 * @see jp.co.ricoh.dsdk.scard.framework.service.CardService#initialize(jp.co.ricoh.dsdk.scard.framework.card.Card)
	 */
	public synchronized void initialize(Card card) throws UnsupportedCardException, CardServiceException {
		super.initialize(card);
		verified = false;
	}
	
	/**
	 * Verifies the given PIN.
	 * 
	 * PINを認証します。
	 */
	public void verifyPIN(byte[] pin) throws CardServiceException, CHVException {
		if (pin == null) throw new CHVException(null, 0);
		if (pin.length != 4) throw new CHVException(null, 0);
		if (!Arrays.equals(PIN, pin)) throw new CHVException(null, 0);
		verified = true;
		
	}
	
	/**
	 * Returns the username if the PIN verification succeeded.
	 * 
	 * PIN認証が行われていれば、ユーザ名を返します。
	 */
	public byte[] getUser() throws CardServiceException, CHVException {
		if (verified) return "user".getBytes();
		else throw new CHVException(null, 0);
	}
	
	/**
	 * Returns the password if the PIN verification succeeded.
	 * 
	 * PIN認証が行われていれば、パスワードを返します。
	 */
	public byte[] getPassword() throws CardServiceException, CHVException {
		if (verified) return "pass".getBytes();
		else throw new CHVException(null, 0);
	}
	
}
