/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2009-2014, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/


/**
 *
 * \file rotate_api.h
 *
 * \brief This file defines the API for the Rotate module.
 *
 */

#if !defined(__ROTATE_AND_PLACE_API_H__)
#define __ROTATE_AND_PLACE_API_H__

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <stdint.h>
#include <stdbool.h>

#include "BigBuffer.h"
#include "lassert.h"
#include "error_types.h"
#include "dma_buffer.h"
#include "logger.h"
#include "uio_lib_api.h"

struct rotate_and_place_s;
enum rotate_err_defs
{
	INVALID_DST_WIDTH,
	INVALID_DST_HEIGHT,
	INVALID_ROTATION,
};
/*
 *			Output Page
 *			<------total_width------->
 *			0------------------------^
 *			| image1 		|'
 *			| width			|'
 *			|(x,y)			|t
 *	offset_x=x	| *------h		|o
 *	offset_y=y	| |	|e		|t
 *	width and	| |	|i		|a
 *	height before	| |	|g		|l
 *	rotate or flip	| |     |h		|_
 *	of image1	| -------t		|h
 *			|			|e
 *			|			|i
 *			|<---strip_length------>|g
 *		       {|***********************|h
 * strip_height        {|      strip_data	|t
 * distance between    {|***********************|'
 * ****** and *****	|			|'
 *			-------------------------v
 *
 */

int is_valid_strip_height( int bpp, int proposed_strip_height,
					int image_width, int image_height);
/**
 * \brief Initialize the rotate system
 * 
 * @param output_width In pixels the width of the output buffers
 * @param output_height In lines the output height for the 
 *                      entire buffer
 * @param bpp The number of bits per pixel for the output buffer
 * @param strip_height The height of the strip in lines.  Must 
 *                     be the same for all strips.
 * @param num_strips The number of strips in the output. 
 *                   output_height = num_strips * strip_height;
 * @param out_buf_strip The actual allocated output strips. 
 *                      After an image is rotated, it's then
 *                      located in these strips.
 * 
 * @return struct rotate_and_place_s* A pointer to a handle for 
 *         the rotate system.
 */
struct rotate_and_place_s *rotate_and_place_start(
	int output_width,	/* in pixels */
	int output_height,	/* in lines of the total page */
	int bpp,		/* 1, 2, 8, 24, 32 bit per pixel */
	int strip_height,	/* is_valid_strip_height(...) */
	int num_strips,		/* num_strips :== height/strip_height */
	struct BigBuffer_s *out_buf_strip[]	/* pre allocated strip output
						 * memory
						 * out_buf_strip[num_strips] */
	);
/**
 * \brief inform that a new image is coming to be placed in the 
 *        output buffers.
 * This will specify the size of the image, where it is to 
 * start, and its rotation and flip settings 
 * 
 * 
 * @param rotate_and_place Handle returned from the 
 *                         rotate_and_place_start call.
 * @param offset_x The upper left x corner location of the image 
 *                 in the output buffers.
 * @param offset_y The upper left y corner location of the image 
 *                 in the output buffers.
 * @param rotate The rotate value for the image, allowed values 
 *               0,90,180,270
 * @param flip_x Specify if image will be flipped around the 
 *               x-axis.  0 = no, 1 = yes  Flip-x and flip-y are
 *               not allowed at the same time
 * @param flip_y Specify if the image will be flipped around the 
 *               y-axis.  0 = no, 1 = yes  Flip-x and flip-y are
 *               not allowed at the same time
 * @param page_width The width in pixels of the input image
 * @param page_height The width in lines of the input image. 
 *                    The number of lines added in
 *                    rotate_and_place_add_strip must not be
 *                    larger then this value.
 * 
 * @return int 
 * @retval 0 successful 
 * @retval -1 failed.  Due to parameter errors. 
 */
int rotate_and_place_add_image(
	struct rotate_and_place_s *rotate_and_place, /* handle returned from
						rotate_and_place_start(...) */
	int offset_x,	/* place this page onto the output at this location
			 * in pixels has alignment requirements */
	int offset_y,	/* place this page onto the output at this location
			 * in pixels has alignment requirements */
	int rotate,	/* rotation angle: 0, 90, 180, 270 : Clockwise */
	int flip_x,	/* 0: don't flip the image along X-Axis
			 * 1: Flip the image along X-Axis */
	int flip_y,	/* 0: don't flip the image along Y-Axis
			 * 1: Flip the image along Y-Axis */
	int page_width,	/* pre rotated page width in pixels */
	int page_height	/* pre rotated page height in pixels */
	);
/**
 * @brief take an input strip and rotate and place it into the 
 *        output strip.
 * 
 * 
 * @param rotate_and_place structure returned from 
 *                         rotate_and_place_add_image.
 * @param strip_height The number of lines in this strip
 * @param in_buf_strip The actual data in the strip, A BigBuffer
 *                      structure
 * 
 * @return int 0 successful, -1 error
 */
int rotate_and_place_add_strip(
	struct rotate_and_place_s *rotate_and_place,	/* handle returned from
						rotate_and_place_start(...) */
        uint32_t strip_height,                  /* strip height in lines*/
	struct BigBuffer_s *in_buf_strip	/* input strip buffer of
						 * strip_height to rotate
						 * strip-len :== (strip_height*
						 * 	     page_width*bpp)/8
						 */
	);

/* returns NULL as it will free the input structure after cleaning up */
struct rotate_and_place_s *rotate_and_place_close(
    struct rotate_and_place_s *rotate_and_place	/* handle returned from
						 * rotate_and_place_start(...)
						 * input context */
    );

/** usage: */

/**
	struct rotate_and_place_s *rotate_and_place = 0;
	int strip_height=64, tmp=0;
	tmp = is_valid_strip_height(1, strip_height);
	strip_height = tmp?tmp:strip_height;
	allocate out_buf_strip to pass to rotate_and_place_start 
	rotate_and_place = rotate_and_place_start( ... )

	rotate_and_place_add_image( ... );
	for num_strips { 
		if (need_to_cancel()) {
			canceling = true;
			goto exit_label;
		}
		rotate_and_place_add_strip( ... );
	}

	rotate_and_place_add_image( ... );
	
	for num_strips do
		rotate_add_strip_to_image( ... );

exit_label:
	rotate_and_place = rotate_and_place_close( ... );
	if (canceling)
		// do something like ack or send message...
*/

#endif /* __ROTATE_AND_PLACE_API_H__ */

