/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2009-2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/


/**
  *
  *This file contains hardware setup private definitions for Rotate
  *
 **/


#include <sys/mman.h>

#define _ALLOW_ROTATE_PRIV_INCLUSION_H
#define DBG_PRFX "ROTATE_PRIV: "

#include "rotate-setup-private.h"
#include "map_mem.h"
#include "logger.h"
#define DBG_PRFX "rotate: "
#define LOGGER_MODULE_MASK DEBUG_LOGGER_MODULE_PRINT | LOGGER_SUBMODULE_BIT( 18 )


#if defined(__linux__)
void rotate_uio_isr(int32_t interrupt_count, void *context);
#else
int rotate_isr(uint32_t intpos);
#endif

#if defined(__linux__)
static void* rotate_get_dev()
{
    static void *rotate_dev = NULL;

    DBG_PRINTF_INFO("%s %d: Entry\n", __func__, __LINE__);
    if (!rotate_dev &&
        (!(rotate_dev = uio_open_dev("rotate_uio"))))
    {
        DBG_PRINTF_INFO("%s %d: Entry\n", __func__, __LINE__);
        XASSERT(rotate_dev != 0, errno);
    }

    DBG_PRINTF_INFO("%s %d: Entry\n", __func__, __LINE__);
    return rotate_dev;
}

#define rotate_int_attach( priority, isr, cb_data ) \
		uio_int_attach( rotate_get_dev(), priority, isr, cb_data )
#define rotate_int_detach( ) uio_int_detach( rotate_get_dev() )
#define rotate_int_enable() uio_int_enable( rotate_get_dev() )
#define rotate_int_disable() uio_int_disable( rotate_get_dev() )
#else
#define rotate_get_dev void
#define rotate_int_attach( priority, isr, cb_data ) \
		intAttach( INTNUM_DAVINCI, cb_data, isr, priority )
#define rotate_int_detach( ) intDetach( INTNUM_DAVINCI )
#define rotate_int_disable( ) intDisable(INTNUM_DAVINCI)
#define uio_int_enable( ) intEnable(INTNUM_DAVINCI)
#endif	/* __linux__ */

#if defined(__linux__)
void rotate_uio_isr(int32_t interrupt_count, void *context)
{
    //uint32_t rotate_ipend;
    //uint32_t pa_ipend;
    uint32_t int_pending;
    static int32_t prev_interrupt_count = 0;
    DBG_PRINTF_INFO("%s %d: count %x context %x\n", __func__, __LINE__, interrupt_count, context);
    struct rotate_device *dev = (struct rotate_device *)context;


    if (dev->rotate_idma_core_regs->IIPR)
    {
        dev->rotate_idma_core_regs->IICR = 1;
        if (dev->notify_idma_interrupt)
        {
            DBG_PRINTF_INFO("%s %d: Entry\n", __func__, __LINE__);
            dev->notify_idma_interrupt(INT_IDMA_LEN_ERROR);
        }
    }

    if (dev->rotate_iudma_regs->UIPR)
    {
        DBG_PRINTF_INFO("%s %d: dev->rotate_iudma_regs->UIPR = 0x%x\n",
                        __func__, __LINE__,
                        dev->rotate_iudma_regs->UIPR);
        int_pending = dev->rotate_iudma_regs->UIPR;
        dev->rotate_iudma_regs->UICR = dev->rotate_iudma_regs->UIPR;
        if (dev->notify_iudma_interrupt)
        {
            DBG_PRINTF_INFO("%s %d: Entry\n", __func__, __LINE__);
            if (int_pending & ROTATE_IDMA_ROTATE_IDMA_UDMA_UIPR_DESC_MASK)
            {
                dev->notify_iudma_interrupt(INT_COMPLETE);
            }
            if (int_pending & ROTATE_IDMA_ROTATE_IDMA_UDMA_UIPR_OUTOFRANGE_MASK)
            {
                DBG_PRINTF_ERR("%s %d: iudma length error %x\n", int_pending);
                dev->notify_iudma_interrupt(INT_IUDMA_LEN_ERROR);
            }
        }
    }

    if (dev->rotate_odma_core_regs->IIPR)
    {
        dev->rotate_odma_core_regs->IICR = 1;
        if (dev->notify_odma_interrupt)
        {
            DBG_PRINTF_INFO("%s %d: Entry\n", __func__, __LINE__);
            dev->notify_odma_interrupt(INT_ODMA_LEN_ERROR);
        }
    }

    if (dev->rotate_oudma_regs->UIPR)
    {
        DBG_PRINTF_INFO("%s %d: dev->rotate_oudma_regs->UIPR = 0x%x\n",
                        __func__, __LINE__,
                        dev->rotate_oudma_regs->UIPR);
        int_pending = dev->rotate_oudma_regs->UIPR;
        // clear the interrupt
        dev->rotate_oudma_regs->UICR = dev->rotate_oudma_regs->UIPR;
        if (dev->notify_oudma_interrupt)
        {
            DBG_PRINTF_INFO("%s %d: Entry\n", __func__, __LINE__);
            if (int_pending & ROTATE_IDMA_ROTATE_IDMA_UDMA_UIPR_DESC_MASK)
            {
                dev->notify_oudma_interrupt(INT_COMPLETE);
            }
            if (int_pending & ROTATE_IDMA_ROTATE_IDMA_UDMA_UIPR_OUTOFRANGE_MASK)
            {
                DBG_PRINTF_ERR("%s %d: oudma length error %x\n", int_pending);
                dev->notify_oudma_interrupt(INT_IUDMA_LEN_ERROR);
            }
        }
    }

    if ((interrupt_count - prev_interrupt_count) > 1)
    {
        DBG_PRINTF_INFO("%s %d: Entry\n", __func__, __LINE__);
        DBG_MEMLOG_WARNING("%s dropping interrupts %d\n", __func__,
                           (interrupt_count - prev_interrupt_count));
    }

    DBG_PRINTF_INFO("%s %d: Entry\n", __func__, __LINE__);
    prev_interrupt_count = interrupt_count;

}
#else
int rotate_isr(uint32_t intpos)
{
    uint32_t rotate_ipend;
    uint32_t pa_ipend;
    static int32_t prev_interrupt_count = 0;
    DBG_PRINTF_INFO("%s %d: Entry\n", __func__, __LINE__);

    if ((interrupt_count - prev_interrupt_count) > 1)
    {
        DBG_PRINTF_INFO("%s %d: Entry\n", __func__, __LINE__);
        DBG_MEMLOG_WARNING("%s dropping interrupts %d\n", __func__,
                           (interrupt_count - prev_interrupt_count));
    }

    DBG_PRINTF_INFO("%s %d: Entry\n", __func__, __LINE__);
    prev_interrupt_count = interrupt_count;
}
#endif

/* memory map hardware registers using addresses from regAddrs.h
 * instead of the "linux" uio why since the number of blocks exceeds
 * what a single uio driver can have.
 */
error_type_t rotate_setupregs(struct rotate_device *device)
{
    int fd;
    int ret = SYS_OK;
    void *mapBase; /* page and structure addresses may be different,
            * we don't unmap which would require keeping the
            * base or computing it.
            */

    DBG_PRINTF_INFO("%s %d: Entry\n", __func__, __LINE__);
    if ((fd = open("/dev/mem", (O_RDWR | O_SYNC | O_CLOEXEC))) < 0)
    {
        DBG_PRINTF_CRIT("rotate_init()  open('dev/mem') failed ");
        ASSERT(fd == 0);
        ret = -ENODEV;
        DBG_PRINTF_ERR("%s %d: return ENODEV\n", __func__, __LINE__);
        goto fail;
    }

    if (MAP_FAILED == (device->rotate_core_regs = (ROTATE_CORE_REGS_t *)
                       mapMem(ROTATE_CORE_REG_BASE, sizeof(ROTATE_CORE_REGS_t),
                              &mapBase, fd)))
    {
        ret = errno;
        DBG_PRINTF_ERR("%s %d: core map failed\n", __func__, __LINE__);
        goto fail;
    }

    if (MAP_FAILED == (device->rotate_iudma_regs =
                       (ROTATE_IDMA_ROTATE_IDMA_UDMA_REGS_t *)
                       mapMem(ROTATE_IUDMA_REG_BASE,
                              sizeof(ROTATE_IDMA_ROTATE_IDMA_UDMA_REGS_t),
                              &mapBase, fd)))
    {
        ret = errno;
        DBG_PRINTF_ERR("%s %d: iudma map failed\n", __func__, __LINE__);
        goto fail_iudma;
    }

    if (MAP_FAILED == (device->rotate_idma_core_regs =
                       (ROTATE_IDMA_ROTATE_IDMA_CORE_REGS_t *)
                       mapMem(ROTATE_IDMA_CORE_REG_BASE,
                              sizeof(ROTATE_IDMA_ROTATE_IDMA_CORE_REGS_t),
                              &mapBase, fd)))
    {
        ret = errno;
        DBG_PRINTF_ERR("%s %d: idma core map failed\n", __func__, __LINE__);
        goto fail_idma;
    }

    if (MAP_FAILED == (device->rotate_oudma_regs =
                       (ROTATE_ODMA_ROTATE_ODMA_UDMA_REGS_t *)
                       mapMem(ROTATE_OUDMA_REG_BASE,
                              sizeof(ROTATE_ODMA_ROTATE_ODMA_UDMA_REGS_t),
                              &mapBase, fd)))
    {
        ret = errno;
        DBG_PRINTF_ERR("%s %d: oudma map failed\n", __func__, __LINE__);
        goto fail_oudma;
    }

    /*hex_dump_named((char*)device->rotate_oudma_regs,
            sizeof(ROTATE_ODMA_ROTATE_ODMA_UDMA_REGS_t), "rotate_oudma_core_reg_aftermap");*/

    if (MAP_FAILED == (device->rotate_odma_core_regs =
                       (ROTATE_ODMA_ROTATE_ODMA_CORE_REGS_t *)
                       mapMem(ROTATE_ODMA_CORE_REG_BASE,
                              sizeof(ROTATE_ODMA_ROTATE_ODMA_CORE_REGS_t),
                              &mapBase, fd)))
    {
        ret = errno;
        DBG_PRINTF_ERR("%s %d: odma core map failed\n", __func__, __LINE__);
        goto fail_odma;
    }
    /*hex_dump_named((char*)device->rotate_odma_core_regs,
            sizeof(ROTATE_ODMA_ROTATE_ODMA_CORE_REGS_t), "rotate_odma_core_reg_aftermap");*/

    DBG_PRINTF_INFO("%s %d: goto normal return\n", __func__, __LINE__);
    goto out;

fail_odma:
    unMapMem(device->rotate_oudma_regs,
             sizeof(ROTATE_ODMA_ROTATE_ODMA_UDMA_REGS_t));
fail_oudma:
    unMapMem(device->rotate_idma_core_regs,
             sizeof(ROTATE_IDMA_ROTATE_IDMA_CORE_REGS_t));
fail_idma:
    unMapMem(device->rotate_iudma_regs,
             sizeof(ROTATE_IDMA_ROTATE_IDMA_UDMA_REGS_t));
fail_iudma:
    unMapMem(device->rotate_core_regs, sizeof(ROTATE_CORE_REGS_t));
fail:
    device->rotate_core_regs = NULL;
    device->rotate_iudma_regs = NULL;
    device->rotate_idma_core_regs = NULL;
    device->rotate_oudma_regs = NULL;
    device->rotate_odma_core_regs = NULL;
out:
    DBG_PRINTF_INFO("%s %d: goto normal return\n", __func__, __LINE__);
    close(fd);
    return ret;
}

int uninitialize_rotate_hardware(struct rotate_device *device)
{
    int ret = -EINVAL;
    int int_ret = 0x00;

    DBG_PRINTF_INFO("%s %d: Entry\n", __func__, __LINE__);
    if (!device)
    {
        ret = -EINVAL;
        goto out;
    }

    if ((int_ret = (rotate_int_disable())))
    {
        DBG_PRINTF_ERR("%s %d: Interrupt disable failed ret = %d\n",
                       __func__, __LINE__, int_ret);
    } else
    {
        DBG_PRINTF_ERR("%s %d: Interrupt disabled ret = %d\n",
                       __func__, __LINE__, int_ret);
        if ((int_ret = (rotate_int_detach())))
        {
            DBG_PRINTF_ERR("%s %d: Interrupt detach failed ret = %d\n",
                           __func__, __LINE__, int_ret);
        } else
        {
            DBG_PRINTF_ERR("%s %d: Interrupt detached ret = %d\n",
                           __func__, __LINE__, int_ret);
        }
    }

    //uio_close_dev(rotate_dev);
//    uio_close_dev(rotate_get_dev());

out:
    DBG_PRINTF_INFO("%s %d: Entry\n", __func__, __LINE__);
    return ret;


}
int init_rotate_hardware(struct rotate_device *device)
{
	int ret = -EINVAL;
	int int_ret = 0x00;

	DBG_PRINTF_INFO("%s %d: Entry\n", __func__, __LINE__);
	if (!device) {
		ret = -EINVAL;
		goto out;
	}

	DBG_PRINTF_INFO("%s %d: Entry\n", __func__, __LINE__);
	if (SYS_OK != (ret = (int)rotate_setupregs(device)))
		goto out;

	DBG_PRINTF_INFO("%s %d: Entry\n", __func__, __LINE__);
        device->rotate_core_regs->SOFT_RESET = 1;
        device->rotate_oudma_regs->UCR = 0;
        device->rotate_oudma_regs->UICR = 0xffffffff;
        device->rotate_iudma_regs->UCR = 0;
        device->rotate_iudma_regs->UICR = 0xffffffff;
        device->rotate_idma_core_regs->IICR = 0xffffffff;
        device->rotate_odma_core_regs->IICR = 0xffffffff;
        device->rotate_idma_core_regs->IIER = 0;
        device->rotate_odma_core_regs->IIER = 0;
	DBG_PRINTF_INFO("%s %d: device %x\n", __func__, __LINE__, device);
	if ((int_ret=(rotate_int_attach(0, rotate_uio_isr, device)))) {
		DBG_PRINTF_ERR("%s %d: Interrupt attach failed ret = %d\n",
				__func__, __LINE__, int_ret);
	} else {
		DBG_PRINTF_INFO("%s %d: Interrupt attached ret = %d\n",
				__func__, __LINE__, int_ret);
		if ((int_ret=(rotate_int_enable()))) {
			DBG_PRINTF_ERR("%s %d: Interrupt enable failed ret = %d\n",
				__func__, __LINE__, int_ret);
		} else {
			DBG_PRINTF_INFO("%s %d: Interrupt enabled ret = %d\n",
				__func__, __LINE__, int_ret);
		}
	}

	DBG_PRINTF_INFO("%s %d: Entry\n", __func__, __LINE__);
	device->notify_idma_interrupt=NULL;
	device->notify_iudma_interrupt=NULL;
	device->notify_odma_interrupt=NULL;
	device->notify_oudma_interrupt=NULL;
out:
	DBG_PRINTF_INFO("%s %d: Entry\n", __func__, __LINE__);
	return ret;
}

