/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2015-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



#include <stdio.h>
#include <sys/types.h>
#include <dirent.h>
#include <time.h>
#include <sched.h>
#include <pthread.h>

#include "jpeghw_lib.h"
#include "jpeghw_api.h" // JPEGHW library

#include "jpeg_strip_adaptor.h"
#include "jpeg_page_adaptor.h"

#include "logger.h"
#include "dma_buffer.h"
#include "uio_lib_api.h"

#include "test_common.h"

#define DBG_PRFX "test_harness: "
#define LOGGER_MODULE_MASK DEBUG_LOGGER_MODULE_DEVICES | LOGGER_SUBMODULE_BIT(20)

#define MAX_FILE_IO_COUNT 3

//#define STRESS_VERBOSE 1
#define STRESS_USE_MEMORY_IO 1
//#define STRESS_NO_COMPARE 1

static uint32_t process_done_flag = 0;
struct file_blob_s {
	pthread_t file_thd_id; /**< file processing thread */

	int id;
	uint32_t count;

	char *file_data;
	int file_data_size;

#ifndef STRESS_NO_COMPARE
	char *compare_data;
	size_t compare_data_size;
#endif

	char page_name[256];
	uint32_t strip_max;
	uint32_t strip_min;
	uint32_t block_max;
	int quality;
	bool error;
	bool random;
	bool compress;
	bool is_last_strip;     //< eop marker
};

struct jpeg_jpeghw_file_page_adaptor {
	jpeg_page_adaptor_BaseClassMembers
	;
	//< put this macro in your subclass
	/// jpeghw
	union {
		struct jpeghw_common_struct common_info; 	//< jpeghw common
		struct jpeghw_compress_struct compress_info;   //< jpeghw compress
		struct jpeghw_decompress_struct decompress_info; //< jpeghw decompress
	};

	struct jpeghw_compression_mgr jcmgr;
	struct jpeghw_decompression_mgr jdmgr;
	struct jpeghw_error_mgr jerr;
	/// jpeghw end

	union {
		PPM_header_struct ppminfo;
		struct jpeghw_jpeg_header_struct jpeginfo;
	};

	// add my local page variables here.
	FILE* src_fp;

	FILE* dst_fp; /**< downstream dst stream handle */
	char *dst_data;
	size_t dst_data_size;
	char dst_filename[256];

	int quality;
	bool started;

	struct file_blob_s *file_blob;
};

struct jpeg_jpeghw_file_strip_adaptor {
	jpeg_strip_adaptor_BaseClassMembers
	;  // put this macro in your subclass

	char *filename; /** src filename */
};

// externs
extern void logger_init();
extern int pthread_tryjoin_np(pthread_t thread, void **retval);
extern int sched_getcpu(void);

// globals
#define JPEG_NUMBER_CONCURRENT_THREADS 5
#define JPEG_MEM_DST_MGR_BUFFER_SIZE (16 * 1024)   ///< might want to have this be configurable.

testFifo g_tblobFifoAvailable;
testFifo g_tblobFifoInProcess;
pthread_t g_tblobFifo_thd_id;

void mgr_progress_monitor(struct jpeghw_common_struct *info)
{
	if (info == NULL)
		return;

	struct jpeg_jpeghw_file_page_adaptor *jpa =
			(struct jpeg_jpeghw_file_page_adaptor*) info->client_context;
	DBG_ASSERT(jpa);

	if (info->type == e_JPEGHW_OBJ_COMPRESS)
	{

#ifdef STRESS_VERBOSE
		struct jpeghw_compress_struct * cinfo = (struct jpeghw_compress_struct *)info;
		printf("Compress Progress for %s: %u lines/%u bytes\n", jpa->dst_filename, cinfo->in_scanlines, cinfo->in_bytes);
#endif
	}
	else if (info->type == e_JPEGHW_OBJ_DECOMPRESS)
	{
#ifdef STRESS_VERBOSE
		struct jpeghw_decompress_struct * dinfo = (struct jpeghw_decompress_struct *)info;
		printf("Decompress Progress for %s: %u lines/%u bytes\n", jpa->dst_filename, dinfo->out_scanlines, dinfo->out_bytes);
#endif
	}
}

void cmgr_init(struct jpeghw_compress_struct * cinfo) {
	struct jpeg_jpeghw_file_page_adaptor *jpa =
			(struct jpeg_jpeghw_file_page_adaptor*) cinfo->common.client_context;

	DBG_ASSERT(jpa);
}

void cmgr_term(struct jpeghw_compress_struct * cinfo) {
	struct jpeg_jpeghw_file_page_adaptor *jpa =
			(struct jpeg_jpeghw_file_page_adaptor*) cinfo->common.client_context;

	DBG_ASSERT(jpa);
}

struct BigBuffer_s* cmgr_get_output_buffer(
		struct jpeghw_compress_struct * cinfo) {
	struct jpeg_jpeghw_file_page_adaptor *jpa =
			(struct jpeg_jpeghw_file_page_adaptor*) cinfo->common.client_context;

	DBG_ASSERT(jpa);

	struct BigBuffer_s* output_buffer = dma_buffer_malloc(0, JPEG_MEM_DST_MGR_BUFFER_SIZE);

	return output_buffer;
}

jpeghw_error_type_t cmgr_get_quant_table(struct jpeghw_compress_struct * cinfo, uint8_t quality, uint8_t *table, uint32_t *size)
{
	if (table == NULL)
	{
		printf("\n>>>>>%s() ERROR - table parameter is NULL!\n",
				__func__);

		// return invalid parameter
		return e_JPEGHW_ERR_INVALID_PARAMETERS;
	}

	if (size == NULL)
	{
		printf("\n>>>>>%s() ERROR - size parameter is NULL!\n",
				__func__);

		// return invalid parameter
		return e_JPEGHW_ERR_INVALID_PARAMETERS;
	}

	if  (quality > 100)
	{
		printf("\n>>>>>%s() ERROR - invalid quality parameter, received %u but expected 0-100!\n",
				__func__, (unsigned int)quality);

		// return invalid parameter
		return e_JPEGHW_ERR_INVALID_PARAMETERS;
	}

	if  (*size > sizeof(g_quantization_table[0]))
	{
		*size = sizeof(g_quantization_table[0]);
	}

	memcpy(table, &g_quantization_table[quality][0], *size);

	return e_JPEGHW_SUCCESS;
}

jpeghw_error_type_t cmgr_get_huff_table(struct jpeghw_compress_struct * cinfo, uint8_t table_index, bool ac, uint8_t *bits, uint32_t *bits_size, uint8_t *val, uint32_t *val_size)
{
	if (table_index > 1)
	{
		printf("\n>>>>>%s() ERROR - invalid table_index parameter, received %u but expected 0 or 1!\n",
				__func__, (unsigned int)table_index);

		return e_JPEGHW_ERR_INVALID_PARAMETERS;
	}

	if (bits && bits_size)
	{
		uint32_t size = (ac ? sizeof(g_ac_bits[table_index]) : sizeof(g_dc_bits[table_index]));

		if (*bits_size > size)
		{
			*bits_size = size;
		}

		if (ac)
		{
			memcpy(bits, g_ac_bits[table_index], *bits_size);
		}
		else
		{
    		memcpy(bits, g_dc_bits[table_index], *bits_size);
		}
	}

	if (val && val_size)
	{
		uint32_t size = (ac ? sizeof(g_ac_val[table_index]) : sizeof(g_dc_val[table_index]));

		if (*val_size > size)
		{
			*val_size = size;
		}

		if (ac)
		{
			memcpy(val, g_ac_val[table_index], *val_size);
		}
		else
		{
    		memcpy(val, g_dc_val[table_index], *val_size);
		}
	}

	return e_JPEGHW_SUCCESS;
}

bool cmgr_send_output_buffer(struct jpeghw_compress_struct * cinfo,
		struct BigBuffer_s* output_buffer, uint32_t bytes_in_buffer) {
	struct jpeg_jpeghw_file_page_adaptor *jpa =
			(struct jpeg_jpeghw_file_page_adaptor*) cinfo->common.client_context;

	DBG_ASSERT(jpa);
	DBG_ASSERT(output_buffer);

	char *data = dma_buffer_mmap_forcpu(output_buffer);
	if (data) {
		if (jpa->dst_fp) {
			fwrite(data, 1, bytes_in_buffer, jpa->dst_fp);
#ifndef STRESS_USE_MEMORY_IO
			jpa->dst_data_size += bytes_in_buffer;
#endif
		}
		dma_buffer_unmmap_forcpu(output_buffer);
	}

	BigBuffer_Free(output_buffer);

	return true;
}

void dmgr_init(struct jpeghw_decompress_struct * dinfo) {
	struct jpeg_jpeghw_file_page_adaptor *jpa =
			(struct jpeg_jpeghw_file_page_adaptor*) dinfo->common.client_context;

	DBG_ASSERT(jpa);

	if (jpa->src_fp) {
		// reset to beginning of file
		fseek(jpa->src_fp, 0, SEEK_SET);
	}
}

void dmgr_term(struct jpeghw_decompress_struct * dinfo) {
	struct jpeg_jpeghw_file_page_adaptor *jpa =
			(struct jpeg_jpeghw_file_page_adaptor*) dinfo->common.client_context;

	DBG_ASSERT(jpa);

}

struct BigBuffer_s* dmgr_get_input_buffer(
		struct jpeghw_decompress_struct * dinfo, uint32_t *bytes_in_buffer) {
	DBG_ASSERT(dinfo);
	DBG_ASSERT(bytes_in_buffer);

	struct jpeg_jpeghw_file_page_adaptor *jpa =
			(struct jpeg_jpeghw_file_page_adaptor*) dinfo->common.client_context;

	DBG_ASSERT(jpa);

	*bytes_in_buffer = 0;
	struct BigBuffer_s *input_buffer = dma_buffer_malloc(0,
			JPEG_MEM_DST_MGR_BUFFER_SIZE);
	if (input_buffer) {
		char *read_ptr = dma_buffer_mmap_forcpu(input_buffer);
		if (read_ptr) {
			*bytes_in_buffer = fread(read_ptr, 1, JPEG_MEM_DST_MGR_BUFFER_SIZE,
					jpa->src_fp);
		}

		dma_buffer_unmmap_forcpu(input_buffer);
	}

	return input_buffer;
}

bool mgr_scanline_timeout(
		struct jpeghw_common_struct *info, uint32_t time_in_seconds) {
	bool ret = false;

	DBG_ASSERT(info);

	struct jpeg_jpeghw_file_page_adaptor *jpa =
			(struct jpeg_jpeghw_file_page_adaptor*) info->client_context;

	DBG_ASSERT(jpa);

	struct file_blob_s* blob = jpa->file_blob;

	printf(".T%d-%d.", blob->id, time_in_seconds);
	fflush(stdout);

	if (time_in_seconds > 60)
	{
		printf("%s() T%d scanline timed out!\n", __func__, blob->id);
		ret = true;
	}

	return ret;
}

void my_error_exit(struct jpeghw_common_struct * cinfo) {
	printf("\n>>>>>jpeg internal exit\n");
	// do a longjmp here
	exit(-1);
}

void jpeg_jpeghw_file_attach_page(struct jpeg_page_adaptor *base_page,
		void *blob) {
	struct jpeg_jpeghw_file_page_adaptor* page =
			(struct jpeg_jpeghw_file_page_adaptor *) base_page;
	int fpos = 0;

	if (page == NULL) {
		DBG_PRINTF_ERR("%s() Invalid page parameter!\n", __func__);
		return;
	}

	struct file_blob_s *file_blob = (struct file_blob_s *) blob;

	if (file_blob == NULL) {
		DBG_PRINTF_ERR("%s() Invalid file_blob variable!\n", __func__);
		return;
	}

	DBG_PRINTF_INFO("%s() page %p file %p filename '%s'\n", __func__, page,
			blob, file_blob->page_name);

	page->started = false;

#ifndef STRESS_USE_MEMORY_IO
	page->src_fp = fopen(file_blob->file_data, "r");
#else
	page->src_fp = fmemopen(file_blob->file_data, file_blob->file_data_size, "r");
#endif
	if (page->src_fp) { // good file (stream)
		bool valid_stream = false;

		if (file_blob->compress) {
			fpos = parse_ppm_header(page->src_fp, &page->ppminfo);
			if (fpos != FAIL) {
				page->common_info.image_width = page->ppminfo.image_width;
				page->common_info.image_height = page->ppminfo.image_height;
				page->common_info.bytes_per_pixel =
						page->ppminfo.bytes_per_pixel;

				page->common_info.mcu_width =
						page->ppminfo.bytes_per_pixel == 1 ? 8 : 16;
				page->common_info.mcu_height = page->common_info.mcu_width;

				valid_stream = true;
			}
		} else {
			fpos = parse_jpeg_header(page->src_fp, &page->jpeginfo, true);
			if (fpos != FAIL) {
				page->common_info.image_width = page->jpeginfo.image_width;
				page->common_info.image_height = page->jpeginfo.image_height;
				page->common_info.bytes_per_pixel =
						page->jpeginfo.bytes_per_pixel;

				page->common_info.mcu_width = page->jpeginfo.mcu_width;
				page->common_info.mcu_height = page->jpeginfo.mcu_height;

				valid_stream = true;
			}
		}

		if (valid_stream)
		{
			char *fmt = NULL;
			jpeghw_set_defaults(&page->common_info);

			if (file_blob->compress)
			{
				fmt = "%s.%d.jpg";
			}
			else
			{
				fmt = "%s.%d.ppm";

				fpos = 0;
			}

			sprintf(page->dst_filename, fmt, file_blob->page_name, file_blob->count);

			fseek(page->src_fp, fpos, SEEK_SET);

#ifndef STRESS_USE_MEMORY_IO
			page->dst_fp = fopen(page->dst_filename, "w"); // create/truncate dst
			page->dst_data_size = 0;
#else
			page->dst_fp = open_memstream(&page->dst_data, &page->dst_data_size); // create/truncate dst
#endif
			DBG_ASSERT(page->dst_fp);

			if (file_blob->compress)
			{
				page->compress_info.quality = page->quality;
				jpeghw_start_compress(&page->compress_info);
			} else {
				jpeghw_start_decompress(&page->decompress_info);
			}

			page->started = true;

		} else {
			DBG_PRINTF_ERR("%s() '%s' NOT a valid stream format!\n", __func__,
					file_blob->page_name);
		}
	}
}

void jpeg_jpeghw_file_send_page(struct jpeg_page_adaptor *base_page, void *blob) {
	DBG_PRINTF_INFO("%s(%p)\n", __func__, base_page);

	if (!base_page || !blob)
		return;

	struct jpeg_jpeghw_file_page_adaptor* page =
			(struct jpeg_jpeghw_file_page_adaptor *) base_page;
	struct file_blob_s *file_blob = (struct file_blob_s *) blob;

	if (file_blob->compress) {
		// nothing to do
	} else {
		// write PPM header
		char header[256];

		sprintf(header, "%s\n%d %d\n255\n",
				(page->decompress_info.common.bytes_per_pixel == 1 ?
						"P5" : "P6"),
				page->decompress_info.common.image_width,
				page->decompress_info.common.image_height);

		if (page->dst_fp) {
			int len = strlen(header);
			fwrite(header, 1, len, page->dst_fp);
#ifndef STRESS_USE_MEMORY_IO
			page->dst_data_size += len;
#endif
		}

	}
}

void jpeg_jpeghw_file_close_page(struct jpeg_page_adaptor *base_page) {
	struct jpeg_jpeghw_file_page_adaptor *jpa =
			(struct jpeg_jpeghw_file_page_adaptor *) base_page;

	if (jpa == NULL || jpa->file_blob == NULL) {
		return;
	}

	struct file_blob_s* blob = jpa->file_blob;
	if (!blob) {
		return;
	}

	if (jpa->started) {
		// client could send info over the network about a page done
		if (blob->compress) {
			if (jpeghw_finish_compress(&jpa->compress_info)
					!= e_JPEGHW_SUCCESS)
			{
				printf("\n>>>>>%s() ERROR - compress failed for file %s!\n",
						__func__, blob->page_name);
				blob->error = true;
			}
		} else {
			if (jpeghw_finish_decompress(&jpa->decompress_info)
					!= e_JPEGHW_SUCCESS)
			{
				printf("\n>>>>>%s() ERROR - decompress failed for file %s!\n",
						__func__, blob->page_name);
				blob->error = true;
			}
		}
	} else {
		printf("\n>>>>>%s() ERROR - never started for file %s!\n", __func__,
				blob->page_name);
	}

	//printf("%s() closing %s file\n", __func__, jpa->dst_filename);
	if (jpa->src_fp) {
		fclose(jpa->src_fp);
	}

	if (jpa->dst_fp) {
		fflush(jpa->dst_fp);
		fclose(jpa->dst_fp);
	}
}

// update img height in results file with given src lines
int jpeg_jpeghw_fixup_page_length(struct jpeg_page_adaptor* base_page) {

	if (!base_page) {
		return FAIL;
	}

	uint32_t lines_encoded = base_page->strip_out_height;
	struct jpeg_jpeghw_file_page_adaptor* page =
			(struct jpeg_jpeghw_file_page_adaptor *) base_page;

	struct file_blob_s* blob = page->file_blob;
	if (!blob) {
		return FAIL;
	}

	if (lines_encoded != page->compress_info.common.image_height) {
		//todo: fixup the page length
		printf("\n>>>>>%s() WARNING - '%s' requires page length fixup, lines encoded = %u!\n",
				__func__, blob->page_name, (unsigned int)lines_encoded);

#ifndef STRESS_USE_MEMORY_IO
		FILE *dst_file = fopen(page->dst_filename, "r+"); // read/write dst
#else
		FILE *dst_file = fmemopen(page->dst_data, page->dst_data_size, "r+"); // read/write dst
#endif
		if (dst_file) { // good file (stream)
			lines_encoded = page_jpeg_fixup_page_length(dst_file, lines_encoded);

			fflush(dst_file);
			fclose(dst_file);
		}
	}

	return lines_encoded;
}

void jpeg_jpeghw_file_free_in_strip(struct jpeg_strip_adaptor* base_strip) {
	struct jpeg_jpeghw_file_strip_adaptor *strip =
			(struct jpeg_jpeghw_file_strip_adaptor *) base_strip;

	DBG_PRINTF_INFO("%s() strip_adaptor %p\n", __func__, strip);
}

void jpeg_recv_strip_jpeghw(struct jpeg_page_adaptor *base_page,
		struct jpeg_strip_adaptor *base_strip, void *blob) {

	if (!base_page || !base_strip || !blob)
		return;

	struct jpeg_jpeghw_file_strip_adaptor *strip =
			(struct jpeg_jpeghw_file_strip_adaptor *) base_strip;
	struct jpeg_jpeghw_file_page_adaptor* page =
			(struct jpeg_jpeghw_file_page_adaptor *) base_page;
	struct file_blob_s *file_blob = (struct file_blob_s *) blob;

	if (!page->started) {
		return;
	}

	strip->bytes_per_pixel = page->common_info.bytes_per_pixel;
	int mcu_height = page->common_info.bytes_per_pixel == 1 ? 8 : 16;
	strip->width_pix_in = page->common_info.image_width;
	strip->out_strip_width = page->common_info.image_width; /// stefan foo: this can be trimmed .
	strip->lines_in = page->common_info.image_height;
	strip->out_strip_height =
			(page->common_info.image_height + (mcu_height - 1))
					& ~(mcu_height - 1); // round up

	if (file_blob->strip_max == 0) {
		file_blob->strip_max = mcu_height; // default to MCU height for strips
	}

	uint32_t strip_count = 0;
	uint32_t byte_count = 0;

	do {
		uint32_t strip_lines = (
				file_blob->random ?
						(rand()
								% (file_blob->strip_max
										- file_blob->strip_min))
								+ file_blob->strip_min + 1 :
						file_blob->strip_max);
		uint32_t lines_in = MIN(strip_lines,
				(strip->lines_in - strip_count));
		uint32_t size = lines_in
				* (strip->width_pix_in * strip->bytes_per_pixel);
		struct BigBuffer_s* in_bb = dma_buffer_malloc(0, size);

		char *data = dma_buffer_mmap_forcpu(in_bb);
		if (data) {
			bool is_end_of_page =
					((strip_count + lines_in) < strip->lines_in) ?
							false : true;

			uint32_t bytes_read = fread(data, 1, size, page->src_fp);
			dma_buffer_unmmap_forcpu(in_bb);

			if (bytes_read != size)
			{
				printf("\n>>>>>%s() WARNING - bytes read were not what was expected, read %d bytes and expected %d bytes for '%s'!\n",
						__func__, (unsigned int)bytes_read, (unsigned int)size, file_blob->page_name);
				file_blob->error = true;

				is_end_of_page = true;
			}

			uint32_t lines = jpeghw_write_scanlines(&page->compress_info, in_bb,
					lines_in, is_end_of_page);

			if (lines < lines_in) {
				printf("\n>>>>>%s() WARNING - not all Scanlines (%u-%u) were processed for '%s'!\n",
						__func__, (unsigned int)lines, (unsigned int)lines_in, file_blob->page_name);
				file_blob->error = true;

				if (lines == 0) {
					BigBuffer_Free(in_bb);
				}
			} else {
				jpa_from_page_at_output_eoi(base_page, lines_in);
			}
		}

		strip_count += lines_in;
		byte_count += size;

	} while (strip_count < strip->lines_in && !file_blob->error);

	// after output with lines computation?   need progress object ?
	strip->free_in_strip(base_strip);
}

void jpeg_send_strip_jpeghw(struct jpeg_page_adaptor *base_page,
		struct jpeg_strip_adaptor *base_strip, void *blob) {
	if (!base_page || !base_strip || !blob)
		return;

	struct jpeg_jpeghw_file_strip_adaptor *strip =
			(struct jpeg_jpeghw_file_strip_adaptor *) base_strip;
	struct jpeg_jpeghw_file_page_adaptor* page =
			(struct jpeg_jpeghw_file_page_adaptor *) base_page;
	struct file_blob_s *file_blob = (struct file_blob_s *) blob;

	strip->bytes_per_pixel = page->decompress_info.common.bytes_per_pixel;
	int mcu_height = 48;
	strip->width_pix_in = page->decompress_info.common.image_width;
	strip->out_strip_width = page->decompress_info.common.image_width;
	strip->lines_in = page->decompress_info.common.image_height;
	strip->out_strip_height = (page->decompress_info.common.image_height
			+ (mcu_height - 1)) & ~(mcu_height - 1); // round up

	bool done = false;
	while (!done) {
		uint32_t buflen = mcu_height * page->jpeginfo.mcu_aligned_width
				* page->jpeginfo.bytes_per_pixel;

		struct BigBuffer_s* strip_bb = dma_buffer_malloc(0, buflen);
		if (strip_bb) {
			uint32_t bytes_read = 0;

			bytes_read = jpeghw_read_scanlines(&page->decompress_info,
					strip_bb, &done);
			if (bytes_read) {
				char *obuf = dma_buffer_mmap_forcpu(strip_bb);
				if (obuf) {
					if (page->dst_fp) {
						int rt = fwrite(obuf, 1, bytes_read, page->dst_fp);
                                                if (rt != bytes_read) {
                                                   printf("\n>>>>>%s() ERROR - Cannot write output all data to output file!\n", __func__);
                                                   printf("         reason: %s\n", strerror(errno));
                                                   done=true;
                                                }
#ifndef STRESS_USE_MEMORY_IO
						page->dst_data_size += bytes_read;
#endif
					}
					dma_buffer_unmmap_forcpu(strip_bb);
				}
			} else {
				if (!done) {
					printf("\n>>>>>%s() ERROR - no bytes were processed for '%s'!\n",
							__func__, file_blob->page_name);
					done = true;
				}
			}

			strip_bb = BigBuffer_Free(strip_bb);
		}
	}

	// after output with lines computation?   need progress object ?
	strip->free_in_strip(base_strip);
}

static int is_valid_pgm_file(const struct dirent *dir) {
	const char *s = dir->d_name;
	int len = strlen(s) - 4;

	if (len >= 0) {
		if (strncmp(s + len, ".ppm", 4) == 0) {
			return 1;
		} else if (strncmp(s + len, ".pgm", 4) == 0) {
			return 1;
		} else if (strncmp(s + len, ".jpg", 4) == 0) {
			return 1;
		}
	}

	return 0;
}

static struct jpeg_page_adaptor *construct_jpeg_jpeghw_file_page_adaptor(
		struct file_blob_s *blob, int quality) {
	struct jpeg_jpeghw_file_page_adaptor *jpa = 0;

	if (!blob)
		return 0;

	jpa = (struct jpeg_jpeghw_file_page_adaptor *) MEM_CALLOC(
			sizeof(struct jpeg_jpeghw_file_page_adaptor), 1);
	if (!jpa)
		return 0;

	jpa = (struct jpeg_jpeghw_file_page_adaptor*) construct_jpeg_page_adaptor(
			(struct jpeg_page_adaptor*) jpa);

	pid_t pid = getpid();
	bool warning = false;
	jpeghw_error_type_t ret = e_JPEGHW_ERROR;
	do {
		if (ret == e_JPEGHW_ERR_NOT_AVAILABLE) {
			if (!warning) {
				printf("\n>>>>>WARNING: Unable to acquire compressor/decompressor(%d), retrying...\n",
						(int) pid);
				warning = true;
			}

			posix_sleep_us(100);
		}

		if (blob->compress) {
			ret = jpeghw_create_compress(&jpa->compress_info,
					JPEGHW_CONFIG_FLAG_BLOCK_JPEG_CORE_AVAIL);
		} else {
			ret = jpeghw_create_decompress(&jpa->decompress_info,
					JPEGHW_CONFIG_FLAG_BLOCK_JPEG_CORE_AVAIL);
		}
	} while (ret == e_JPEGHW_ERR_NOT_AVAILABLE);

	if (ret == e_JPEGHW_SUCCESS) {
		int quality = blob->quality;

		jpa->file_blob = blob;

		jpa->quality = quality;

		if (jpa->common_info.type == e_JPEGHW_OBJ_COMPRESS) {
			jpa->adopt_from_page = jpeg_jpeghw_file_attach_page;
			jpa->jpeg_send_page = jpeg_jpeghw_file_send_page;
			jpa->page_close = jpeg_jpeghw_file_close_page;
			jpa->fixup_jpg_page_length = jpeg_jpeghw_fixup_page_length;

			// set cmgr callbacks
			jpa->compress_info.cmgr = &jpa->jcmgr;
			jpa->compress_info.cmgr->init = cmgr_init;
			jpa->compress_info.cmgr->get_output_buffer = cmgr_get_output_buffer;
			jpa->compress_info.cmgr->send_output_buffer =
					cmgr_send_output_buffer;
			jpa->compress_info.cmgr->term = cmgr_term;

			jpa->compress_info.cmgr->get_quant_table = cmgr_get_quant_table;
			jpa->compress_info.cmgr->get_huff_table = cmgr_get_huff_table;

		} else {
			jpa->adopt_from_page = jpeg_jpeghw_file_attach_page;
			jpa->jpeg_send_page = jpeg_jpeghw_file_send_page;
			jpa->page_close = jpeg_jpeghw_file_close_page;

			// set dmgr callbacks
			jpa->decompress_info.dmgr = &jpa->jdmgr;
			jpa->decompress_info.dmgr->init = dmgr_init;
			jpa->decompress_info.dmgr->get_input_buffer = dmgr_get_input_buffer;
			jpa->decompress_info.dmgr->term = dmgr_term;
		}

		// optional scanline timeout manager
		jpa->common_info.scanline_timeout = mgr_scanline_timeout;

		// optional error manager
		jpa->jerr.error_exit = my_error_exit;
		jpa->common_info.err = jpeghw_std_error(&jpa->jerr);
		jpa->common_info.progress_monitor = mgr_progress_monitor;

		// back pointer to allow callbacks to have context connection.
		jpa->common_info.client_context = (void *) jpa;

		if (blob->compress) {
#ifdef STRESS_VERBOSE
			int count = blob->count;
			int id = blob->id;
			int cpu = sched_getcpu();
			printf("Acquired JPEG core for compress (cpu:%d, pid:%d, thread:%d-%d, quality: %d%%)\n",
					(int)cpu, (int) pid, (int)id, (int)count, (int)quality);
#else
			printf(".C%d.", blob->id);
			fflush(stdout);
#endif
		} else {
#ifdef STRESS_VERBOSE
			int count = blob->count;
			int id = blob->id;
			int cpu = sched_getcpu();
			printf("Acquired JPEG core for decompress (cpu:%d, pid:%d, thread:%d-%d, quality: %d%%)\n",
					(int)cpu, (int) pid, (int)id, (int)count, (int)quality);
#else
			printf(".D%d.", blob->id);
			fflush(stdout);
#endif
		}
	} else {
		memFree(jpa);
		jpa = NULL;
	}

	return (struct jpeg_page_adaptor *) jpa;
}

static void destroy_jpeg_jpeghw_file_page_adaptor(
		struct jpeg_page_adaptor *page_adaptor) {
	struct jpeg_jpeghw_file_page_adaptor *jpa =
			(struct jpeg_jpeghw_file_page_adaptor *) page_adaptor;

	DBG_ASSERT(jpa);

	if (jpa) {
		struct file_blob_s* blob = jpa->file_blob;
		if (!blob) {
			return;
		}

		if (blob->compress) {
#ifdef STRESS_VERBOSE
			pid_t pid = getpid();
			int cpu = sched_getcpu();
			int id = blob->id;
			int count = blob->count;
			int quality = blob->quality;
			printf("Release JPEG core for compress (cpu:%d, pid:%d, thread:%d-%d, quality: %d%%)\n",
					(int)cpu, (int) pid, (int)id, (int)count, (int)quality);
#endif
		} else {
#ifdef STRESS_VERBOSE
			pid_t pid = getpid();
			int cpu = sched_getcpu();
			int id = blob->id;
			int count = blob->count;
			int quality = blob->quality;
			printf("Release JPEG core for decompress (cpu:%d, pid:%d, thread:%d-%d, quality: %d%%)\n",
					(int)cpu, (int) pid, (int)id, (int)count, (int)quality);
#endif
		}

		if (blob->compress) {
			jpeghw_destroy_compress(&jpa->compress_info);
		} else {
			jpeghw_destroy_decompress(&jpa->decompress_info);
		}

#ifndef STRESS_USE_MEMORY_IO
 #ifndef STRESS_NO_COMPARE
		if (blob->compare_data_size == 0)
		{
			blob->compare_data_size = jpa->dst_data_size;
		}
		else if (jpa->dst_data_size != blob->compare_data_size)
		{
			printf("\n>>>>>%s() ERROR - '%s' resulting size failure (%d != %d)!\n",
					__func__, blob->page_name, jpa->dst_data_size, blob->compare_data_size);

			exit(1);
		}
 #endif
#else
		if (jpa->dst_data)
		{
#ifndef STRESS_NO_COMPARE
			if (!blob->compare_data && jpa->dst_data_size)
			{
				blob->compare_data = MEM_MALLOC(jpa->dst_data_size);
				if (blob->compare_data)
				{
					memcpy(blob->compare_data, jpa->dst_data, jpa->dst_data_size);
					blob->compare_data_size = jpa->dst_data_size;
				}
			}
			else
			{
				if (jpa->dst_data_size == blob->compare_data_size)
				{
					if (memcmp(jpa->dst_data, blob->compare_data, blob->compare_data_size) != 0)
					{
						printf("\n>>>>>%s() ERROR - '%s' result compare failure!\n",
								__func__, blob->page_name);

						FILE *fp=fopen("ccerr.jpg", "w");
						fwrite(jpa->dst_data, 1, jpa->dst_data_size, fp);
						fflush(fp);
						fclose(fp);

						exit(1);
					}
				}
				else
				{
					printf("\n>>>>>%s() ERROR - '%s' resulting size failure (%zu != %zu)!\n",
							__func__, blob->page_name, jpa->dst_data_size, blob->compare_data_size);

					FILE *fp=fopen("scerr.jpg", "w");
					fwrite(jpa->dst_data, 1, jpa->dst_data_size, fp);
					fflush(fp);
					fclose(fp);

					exit(1);
				}
			}
#endif

			free(jpa->dst_data);

			jpa->dst_data_size = 0;
		}
#endif

		blob->count++;

#ifndef STRESS_USE_MEMORY_IO
		if (blob->count > MAX_FILE_IO_COUNT)
		{
			blob->count = 1;
		}
#endif
		memFree(jpa);
	}
}

static struct jpeg_strip_adaptor *construct_jpeg_jpeghw_file_strip_adaptor(
		struct file_blob_s *blob) {
	struct jpeg_jpeghw_file_strip_adaptor *jsa = 0;

	if (!blob)
		return 0;

	jsa = (struct jpeg_jpeghw_file_strip_adaptor *) MEM_MALLOC(
			sizeof(struct jpeg_jpeghw_file_strip_adaptor));
	if (!jsa)
		return 0;
	jsa = (struct jpeg_jpeghw_file_strip_adaptor*) construct_jpeg_strip_adaptor(
			(struct jpeg_strip_adaptor*) jsa);

	if (blob->compress) {
		jsa->recv_strip = jpeg_recv_strip_jpeghw;
	} else {
		jsa->send_strip = jpeg_send_strip_jpeghw;
	}

	jsa->free_in_strip = jpeg_jpeghw_file_free_in_strip; // default

	return (struct jpeg_strip_adaptor *) jsa;
}

static void destroy_jpeg_jpeghw_file_strip_adaptor(
		struct jpeg_strip_adaptor *base) {
	struct jpeg_jpeghw_file_strip_adaptor *jfsa =
			(struct jpeg_jpeghw_file_strip_adaptor *) base;

	DBG_ASSERT(jfsa);
	if (jfsa) {
		memFree(jfsa);
	}
}

static void* _jpeg_thread_completions(void *arg) {
	while (1) {
		struct file_blob_s *file_blob = testFifo_Remove(&g_tblobFifoInProcess,	true);
		DBG_ASSERT(file_blob);

		if (file_blob) {
			if (pthread_tryjoin_np(file_blob->file_thd_id, NULL) == 0)
			{
				testFifo_Add(&g_tblobFifoAvailable, file_blob, true);
			}
			else
			{
				testFifo_Add(&g_tblobFifoInProcess, file_blob, true);
			}
		} else {
			printf("\n>>>>>%s() ERROR - testFifo_Remove failed!\n", __func__);
			return NULL;
		}
	}
}

static void* _jpeg_process_file(void *arg) {
	struct file_blob_s *file_blob = (struct file_blob_s *) arg;
	struct jpeg_page_adaptor *page_adaptor = NULL;
	struct jpeg_strip_adaptor *strip_adaptor = NULL;

	testFifo_Add(&g_tblobFifoInProcess, file_blob, true);

	if (file_blob == NULL)
		return NULL;

//	printf("%s(%s) - Starting Thread 0x%x\n", __func__, file_blob->page_name,
//			file_blob->file_thd_id);

	{
		page_adaptor = construct_jpeg_jpeghw_file_page_adaptor(file_blob,
				file_blob->quality);
		strip_adaptor = construct_jpeg_jpeghw_file_strip_adaptor(file_blob);
		if (page_adaptor && strip_adaptor) {
			// setup new page
			jpeg_recv_page(page_adaptor, file_blob);

			if (file_blob->compress) {
				jpeg_recv_strip(page_adaptor, strip_adaptor, file_blob);
			} else {
				jpeg_send_strip(page_adaptor, strip_adaptor, file_blob);
			}

			jpa_from_page_close(page_adaptor);
			jpa_fixup_jpg_page(page_adaptor);
		}

		destroy_jpeg_jpeghw_file_strip_adaptor(strip_adaptor);
		destroy_jpeg_jpeghw_file_page_adaptor(page_adaptor);
	}

//	printf("%s(%s) - Ending Thread 0x%x\n", __func__, file_blob->page_name,
//			file_blob->file_thd_id);
	if (file_blob->block_max ==1 ) {
		process_done_flag = 1;
	}
	return NULL;
}

int main_jpeghw_adaptor(int argc, char *argv[]) {
	struct dirent **eps;
	int nFiles;
	int filearg = 0;
	int loop_max = 0;
	int strip_max = 0;
	int block_max = 2;
	int quality = 90;
	bool random = false;

	DBG_PRINTF_INFO("%s() argc %d argv[%d] %s\n", __func__, argc, 1, argv[1]);

	if (argc == 2) {
		filearg = 1;
	} else if (argc == 3) {
		filearg = 1;
		block_max = atoi(argv[2]);
	} else if (argc == 4) {
		filearg = 1;
		block_max = atoi(argv[2]);
		loop_max = atoi(argv[3]);
	} else if (argc == 5) {
		filearg = 1;
		block_max = atoi(argv[2]);
		loop_max = atoi(argv[3]);
		strip_max = atoi(argv[4]);
	} else if (argc == 6) {
		filearg = 1;
		block_max = atoi(argv[2]);
		loop_max = atoi(argv[3]);
		strip_max = atoi(argv[4]);
		quality = atoi(argv[5]);
	} else if (argc == 7 && strcmp(argv[6], "random") == 0) {
		filearg = 1;
		block_max = atoi(argv[2]);
		loop_max = atoi(argv[3]);
		strip_max = atoi(argv[4]);
		quality = atoi(argv[5]);
		random = true;
	} else {
		DBG_PRINTF_ERR(
				"usage mode files : program /dir/path/ <JPEG Block Max Number> \
                                <Loop Max> <Strip Max> <Quality>[random]\n");
		return 0;
	}

	if (block_max < 1) {
		DBG_PRINTF_ERR("ERROR: <JPEG Block Max Number> must be 1 or greater!\n");
		return 0;
	}

	if (loop_max < 0)
	{
		DBG_PRINTF_ERR("ERROR: <Loop Max> must be 0 or greater!\n");
		return 0;
	}

	if (random) {
		srand(time(NULL));
	}

	if (quality > 100)
	{
		quality = 100;
	}

	nFiles = scandir(argv[filearg], &eps, is_valid_pgm_file, alphasort);
	if (nFiles < 0) {
		DBG_PRINTF_ERR("ERROR: Couldn't open the directory!\n");
		return 0;
	}

	// fifo list of file_blobs
	testFifo_Init(&g_tblobFifoAvailable, nFiles);
	testFifo_Init(&g_tblobFifoInProcess, nFiles);

	int ret = pthread_create(&g_tblobFifo_thd_id, 0, _jpeg_thread_completions,
			(void *) NULL);
	if (ret != 0) {
		printf("\n>>>>>%s() Failed to create _jpeg_thread_completions thread - err = %d!!!\n",
				__func__, ret);
		return 0;
	}

	// store all files in the directory
	int nThreads = 0;
	int i;
	for (i=0; i<nFiles; i++)
	{
		char filename[256];
		char *strip_name = NULL;
		bool compress = true;

		memset(filename, 0, sizeof(filename));

		strip_name = eps[i]->d_name;

		int len = sizeof(filename) - 1;
		strncpy(filename, argv[filearg], len);

		len = sizeof(filename) - strlen(filename)
				- 1;
		strncat(filename, "/", len);

		len = sizeof(filename) - strlen(filename)
				- 1;
		strncat(filename, strip_name, len);

		DBG_PRINTF_INFO("%s\n", filename);

		// determine compress or decompress based on filename extension
		char *ext = strrchr(filename, '.');
		if (ext) {
			if (strcasecmp(ext, ".ppm") == 0) {
				compress = true;
			} else if (strcasecmp(ext, ".jpg") == 0) {
				compress = false;
			} else {
				printf("\n>>>>>%s() Invalid file format, skipping file %s!\n",
						__func__, filename);
				// skip file
				continue;
			}
		} else {
			printf("\n>>>>>%s() No file extension, skipping file %s!\n",
					__func__, filename);
			// skip file
			continue;
		}

		printf("%s%d = %s\n",
				(compress) ? "C" : "D",
						nThreads,
				filename);

		struct file_blob_s *blob_record = MEM_MALLOC(
				sizeof(struct file_blob_s));
		if (blob_record)
		{
			memset(blob_record, 0, sizeof(struct file_blob_s));
			blob_record->id = nThreads;

#ifndef STRESS_USE_MEMORY_IO
			blob_record->file_data_size = strlen(filename)+1;
			blob_record->file_data = MEM_MALLOC(blob_record->file_data_size);
			if (blob_record->file_data)
			{
				strncpy(blob_record->file_data, filename, blob_record->file_data_size);
			}
#else
			FILE *fp = fopen(filename, "r");
			if (fp)
			{
				fseek(fp, 0L, SEEK_END);
				blob_record->file_data_size = ftell(fp);

				if (blob_record->file_data_size)
				{
					int nRead = blob_record->file_data_size;

					fseek(fp, 0L, SEEK_SET);

					blob_record->file_data = MEM_MALLOC(blob_record->file_data_size);
					if (blob_record->file_data)
					{
						blob_record->file_data_size = fread(blob_record->file_data, 1, nRead, fp);
					}
				}

				fclose(fp);
			}
#endif

			if (blob_record->file_data)
			{
				len = strlen(strip_name) - 4;
				if (len > 0) {
					strncpy(blob_record->page_name, strip_name, len);
					blob_record->page_name[len] = 0;
				} else {
					strcpy(blob_record->page_name, strip_name);
				}

				blob_record->strip_max = 0;
				blob_record->strip_min = 0;
				blob_record->block_max = block_max;
				blob_record->quality = (random ? (rand() % quality) + 1 : quality);

				blob_record->random = random;
				blob_record->compress = compress;
				blob_record->error = false;

				if (strip_max > 0) {
					blob_record->strip_max = strip_max;
				}

				blob_record->is_last_strip = true; // we are sending entire file

				testFifo_Add(&g_tblobFifoAvailable, blob_record, false);
				nThreads++;
			}
			else
			{
				MEM_FREE_AND_NULL(blob_record);
			}

		} else {
			printf("\n>>>>>%s() ERROR - unable to allocate file_blob_s #%d!\n",
					__func__, i);
			return 0;
		}
	}

	int loop_count = 0;
	/* if loop_max == 0 causes infinite loop for running test indefinitely */
	while (loop_max == 0 || loop_count < loop_max) {
		i = 0;
		while (i < nThreads)
		{
			struct file_blob_s *file_blob = testFifo_Remove(
					&g_tblobFifoAvailable, true);
			if (file_blob) {

				int ret = pthread_create(&(file_blob->file_thd_id), 0,
						_jpeg_process_file, (void *) file_blob);
				if (ret != 0)
				{
					printf(
							">>>>>%s() Failed to create thread for %s - err = %d!!!\n",
							__func__, file_blob->page_name, ret);

					testFifo_Add(&g_tblobFifoAvailable, file_blob, true);
				}
			} else {
				printf(
						">>>>>%s() ERROR - unable to retrieve file_blob_s!\n",
						__func__);
			}

			//If only has one jpeg block,
		        //it is only allow to run one thread at one time
			if (block_max == 1) {
			while (process_done_flag == 0) {
			        posix_sleep_us(1000);
			 }
			      process_done_flag = 0;
			 }
			i++; // next file argument?

		}

		loop_count++;
	}

	i = 0;
	while (i < nThreads)
	{
		struct file_blob_s *blob_record = testFifo_Remove(&g_tblobFifoAvailable,
				true);

		if (blob_record)
		{
			if (blob_record->file_data)
				MEM_FREE_AND_NULL(blob_record->file_data);

#ifndef STRESS_NO_COMPARE
			if (blob_record->compare_data)
			{
				MEM_FREE_AND_NULL(blob_record->compare_data);
			}
#endif

			if (blob_record) {
				MEM_FREE_AND_NULL(blob_record);
			}
		}

		i++;
	}
	printf("\n");

	pthread_cancel(g_tblobFifo_thd_id);

	testFifo_Destroy(&g_tblobFifoAvailable);
	testFifo_Destroy(&g_tblobFifoInProcess);

	return 0;
}

/// test-sample() -- take a list of options/files and apply appropriate encode operations.
int main(int argc, char** argv) {
	memInitMemory(0, 1024 * 1024 * 128); // init memory pool uses dmaalloc.ko, implies 1 process for now.
	logger_init();  // controlled by /home/root/logger_config_file if present
	uio_lib_init(); // init uio uses /dev/uio*
	jpeghw_init();

	printf("Start test_stress\n");

	main_jpeghw_adaptor(argc, argv);

	printf("Completed test_stress\n");

	jpeghw_terminate();

	return 0;
}

// eof test-sample.c
