/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



#include <stdio.h>
#include <sys/types.h>
#include <dirent.h>

#include "jpeghw_api.h" // JPEGHW library
#include "jpeg_strip_adaptor.h"
#include "jpeg_page_adaptor.h"

#include "logger.h"
#include "dma_buffer.h"
#include "uio_lib_api.h"

#include "test_common.h"


#define JPEG_FILE_DEFAULT_IMAGE_WIDTH  4800 // 8 inch default
#define JPEG_FILE_DEFAULT_IMAGE_HEIGHT 8400 // 14 inch default
#define JPEG_MEM_DST_MGR_BUFFER_SIZE (16 * 1024)   ///< might want to have this be configurable.

#define DBG_PRFX "test_speed: "
#define LOGGER_MODULE_MASK DEBUG_LOGGER_MODULE_DEVICES | LOGGER_SUBMODULE_BIT(20)

struct page_blob_s {
	uint32_t width;
	uint32_t height;
	uint32_t bytes_per_pixel;

	int quality;
};

struct strip_blob_s {
	uint32_t lines;
	uint32_t lines_in;
	bool is_last_strip;     //< eop marker

	struct BigBuffer_s* bb;
};

struct jpeg_jpeghw_page_adaptor {
	jpeg_page_adaptor_BaseClassMembers
	;
	//< put this macro in your subclass
	/// jpeghw
	struct jpeghw_compress_struct compress_info;   //< jpeghw compress

	struct jpeghw_compression_mgr jcmgr;
	struct jpeghw_error_mgr jerr;
	/// jpeghw end

	PPM_header_struct ppminfo;

	struct BigBuffer_s *cache_buffer;

	bool error;
};

struct jpeg_jpeghw_strip_adaptor {
	jpeg_strip_adaptor_BaseClassMembers
	;  // put this macro in your subclass

	struct BigBuffer_s* bb; //< input data as dma_buffer.

};


// externs
extern void logger_init();
struct BigBuffer_s* dma_buffer_freefunc (struct BigBuffer_s*);

void cmgr_init(struct jpeghw_compress_struct * cinfo) {
	struct jpeg_jpeghw_page_adaptor *jpa =
			(struct jpeg_jpeghw_page_adaptor*) cinfo->common.client_context;

	DBG_ASSERT(jpa);
	if (jpa->cache_buffer)
		BigBuffer_Free(jpa->cache_buffer);

	jpa->cache_buffer = dma_buffer_malloc(0, JPEG_MEM_DST_MGR_BUFFER_SIZE);
}

void cmgr_term(struct jpeghw_compress_struct * cinfo) {
	struct jpeg_jpeghw_page_adaptor *jpa =
			(struct jpeg_jpeghw_page_adaptor*) cinfo->common.client_context;

	DBG_ASSERT(jpa);
	if (jpa->cache_buffer)
	{
		jpa->cache_buffer = BigBuffer_Free(jpa->cache_buffer);
	}
}

struct BigBuffer_s* cmgr_get_output_buffer(struct jpeghw_compress_struct * cinfo) {
	struct jpeg_jpeghw_page_adaptor *jpa =
			(struct jpeg_jpeghw_page_adaptor*) cinfo->common.client_context;

	DBG_ASSERT(jpa);

	if (jpa->cache_buffer == NULL)
	{
		jpa->cache_buffer = dma_buffer_malloc(0, JPEG_MEM_DST_MGR_BUFFER_SIZE);
	}

	return jpa->cache_buffer;
}

bool cmgr_send_output_buffer(struct jpeghw_compress_struct * cinfo, struct BigBuffer_s* output_buffer, uint32_t bytes_in_buffer) {
	struct jpeg_jpeghw_page_adaptor *jpa =
			(struct jpeg_jpeghw_page_adaptor*) cinfo->common.client_context;

	DBG_ASSERT(jpa);
	DBG_ASSERT(output_buffer);

	//jpa->cache_buffer = BigBuffer_Free(jpa->cache_buffer);

	return true;
}

void my_error_exit(struct jpeghw_common_struct * cinfo) {
	printf(">>>>>jpeg internal exit");
	// do a longjmp here
	exit(-1);
}

void jpeg_jpeghw_attach_page(struct jpeg_page_adaptor *base_page, void *blob) {
	struct jpeg_jpeghw_page_adaptor* page =
			(struct jpeg_jpeghw_page_adaptor *) base_page;
	struct page_blob_s *page_blob = (struct page_blob_s *) blob;

	if (page == NULL)
	{
		DBG_PRINTF_ERR("%s() Invalid page parameter!\n", __func__);
		return;
	}

	if (blob == NULL)
	{
		DBG_PRINTF_ERR("%s() Invalid file_blob variable!\n", __func__);
		return;
	}

	page->compress_info.common.image_width = page_blob->width;
	page->compress_info.common.image_height = page_blob->height;
	page->compress_info.common.bytes_per_pixel = page_blob->bytes_per_pixel;

	page->compress_info.common.mcu_width = page_blob->bytes_per_pixel == 1 ? 8 : 16;
	page->compress_info.common.mcu_height = page->compress_info.common.mcu_width;

	page->compress_info.quality = page_blob->quality;

	jpeghw_start_compress(&page->compress_info);
}

void jpeg_jpeghw_send_page(struct jpeg_page_adaptor *base_page, void *blob) {
	DBG_PRINTF_INFO("%s(%p)\n", __func__, base_page);

	// do nothing
}

void jpeg_jpeghw_close_page(struct jpeg_page_adaptor *base_page) {
	struct jpeg_jpeghw_page_adaptor *jpa =
			(struct jpeg_jpeghw_page_adaptor *) base_page;

	if (jpa == NULL)
	{
		return;
	}

	if (jpeghw_finish_compress(&jpa->compress_info) != e_JPEGHW_SUCCESS)
	{
		printf(">>>>>%s() ERROR - compress failed!\n", __func__);
		jpa->error = true;
	}
}

void jpeg_jpeghw_attach_from_strip(struct jpeg_page_adaptor *base_page,
		struct jpeg_strip_adaptor *base_strip, void *blob)
{
	if (!base_page || !base_strip || !blob)
		return;

	struct jpeg_jpeghw_strip_adaptor *jfsa =
			(struct jpeg_jpeghw_strip_adaptor *) base_strip;
	struct strip_blob_s *strip_blob = (struct strip_blob_s *) blob;

	jfsa->bb = strip_blob->bb;
	jfsa->lines_in = strip_blob->lines_in;
	jfsa->is_end_of_page = strip_blob->is_last_strip;
}


void jpeg_jpeghw_free_in_strip(struct jpeg_strip_adaptor* base_strip) {
	struct jpeg_jpeghw_strip_adaptor *strip =
			(struct jpeg_jpeghw_strip_adaptor *) base_strip;
	DBG_PRINTF_INFO("%s() strip_adaptor %p\n", __func__, strip);

	// do nothing
}

void jpeg_recv_strip_jpeghw(struct jpeg_page_adaptor *base_page,
		struct jpeg_strip_adaptor *base_strip, void *blob)
{
	if (!base_page || !base_strip || !blob)
		return;

	struct jpeg_jpeghw_strip_adaptor *strip =
			(struct jpeg_jpeghw_strip_adaptor *) base_strip;
	struct jpeg_jpeghw_page_adaptor* page =
			(struct jpeg_jpeghw_page_adaptor *) base_page;

	strip->adopt_bb_from_blob(base_page, base_strip, blob); // binds jsa->in_area.bb

	if (strip->bb)
	{
		uint32_t lines = jpeghw_write_scanlines(&page->compress_info, strip->bb,
							strip->lines_in, strip->is_end_of_page);

		if (lines < strip->lines_in) {
			printf(">>>>>%s() WARNING - not all Scanlines were processed!\n", __func__);
			page->error = true;

			if (lines == 0) {
				strip->bb = BigBuffer_Free(strip->bb);
			}
		} else {
			jpa_from_page_at_output_eoi(base_page, strip->lines_in);
		}

		strip->bb = NULL; // passed ownership downstream
	}

	// after output with lines computation?   need progress object ?
	strip->free_in_strip(base_strip);
}

static struct jpeg_jpeghw_page_adaptor *construct_jpeg_jpeghw_page_adaptor()
{
	struct jpeg_jpeghw_page_adaptor *jpa = 0;

	jpa = (struct jpeg_jpeghw_page_adaptor *) MEM_CALLOC(
			sizeof(struct jpeg_jpeghw_page_adaptor), 1);
	if (!jpa)
		return 0;

	jpa = (struct jpeg_jpeghw_page_adaptor*) construct_jpeg_page_adaptor(
			(struct jpeg_page_adaptor*) jpa);

	bool warning = false;
	jpeghw_error_type_t ret = e_JPEGHW_ERROR;
	do
	{
		if (ret == e_JPEGHW_ERR_NOT_AVAILABLE)
		{
			if (!warning)
			{
				printf("WARNING: Unable to acquire compressor, retrying...\n");
				warning = true;
			}

			posix_sleep_us(100);
		}

		ret = jpeghw_create_compress(&jpa->compress_info, JPEGHW_CONFIG_FLAG_BLOCK_JPEG_CORE_AVAIL);

	} while (ret == e_JPEGHW_ERR_NOT_AVAILABLE);

	if (ret == e_JPEGHW_SUCCESS)
	{
		jpa->adopt_from_page = jpeg_jpeghw_attach_page;
		jpa->jpeg_send_page = jpeg_jpeghw_send_page;
		jpa->page_close = jpeg_jpeghw_close_page;

		// set cmgr callbacks
		jpa->compress_info.cmgr = &jpa->jcmgr;
		jpa->compress_info.cmgr->init = cmgr_init;
		jpa->compress_info.cmgr->get_output_buffer = cmgr_get_output_buffer;
		jpa->compress_info.cmgr->send_output_buffer = cmgr_send_output_buffer;
		jpa->compress_info.cmgr->term = cmgr_term;

		/// todo: add error manage to prevent libjpeg from calling exit().
		jpa->jerr.error_exit = my_error_exit;
		jpa->compress_info.common.err = jpeghw_std_error(&jpa->jerr);

		// back pointer to allow callbacks to have context connection.
		jpa->compress_info.common.client_context = (void *)jpa;
	}
	else
	{
		memFree(jpa);
		jpa = NULL;
	}

	return (struct jpeg_jpeghw_page_adaptor *) jpa;
}

static void destroy_jpeg_jpeghw_page_adaptor(
		struct jpeg_jpeghw_page_adaptor *jpa) {

	DBG_ASSERT(jpa);

	if ( jpa  )
	{
		jpeghw_destroy_compress(&jpa->compress_info);

		memFree(jpa);
	}
}

static struct jpeg_jpeghw_strip_adaptor *construct_jpeg_jpeghw_strip_adaptor()
{
	struct jpeg_jpeghw_strip_adaptor *jsa = 0;

	jsa = (struct jpeg_jpeghw_strip_adaptor *) MEM_MALLOC(
			sizeof(struct jpeg_jpeghw_strip_adaptor));
	if (!jsa)
		return NULL;

	jsa = (struct jpeg_jpeghw_strip_adaptor*) construct_jpeg_strip_adaptor(
			(struct jpeg_strip_adaptor*) jsa);

	jsa->adopt_bb_from_blob = jpeg_jpeghw_attach_from_strip;
	jsa->recv_strip = jpeg_recv_strip_jpeghw;

	jsa->free_in_strip = jpeg_jpeghw_free_in_strip; // default

	return (struct jpeg_jpeghw_strip_adaptor *) jsa;
}

static void destroy_jpeg_jpeghw_strip_adaptor(struct jpeg_jpeghw_strip_adaptor *jfsa)
{
	DBG_ASSERT(jfsa);
	if (jfsa)
	{
		memFree(jfsa);
	}
}

static struct BigBuffer_s *bb_dont_free( struct BigBuffer_s *bb )
{
    return bb;
}


#define NUM_COMPRESSORS 2
#define NUM_STRIPS 2

int main_jpeghw_adaptor_strips(int argc, char *argv[]) {
	uint32_t num_compressors = 1;
	uint32_t loop_count = 100;
	uint32_t strip_width  = JPEG_FILE_DEFAULT_IMAGE_WIDTH;
	uint32_t strip_height = 16;
	uint32_t bpp = 1;
	int ncomp, cnt, lines;

	struct jpeg_jpeghw_page_adaptor *page_adaptor[NUM_COMPRESSORS];
	struct jpeg_jpeghw_strip_adaptor *strip_adaptor[NUM_COMPRESSORS];
	struct page_blob_s page_blob[NUM_COMPRESSORS];

	DBG_PRINTF_INFO("%s() argc %d argv[%d] %s\n", __func__, argc, 1, argv[1]);

	// optional setup a cached output buffer allocator
	// strip_adaptor->alloc_new_bb = dma_buffer_cached_malloc;

	// one directory per page with "page_..." sorted strips of pgm files

	if (argc >= 2 && argc <= 5)
	{
		if (strncmp(argv[1], "MONO", 4) == 0)
		{
			bpp = 1;
		}
		else if (strncmp(argv[1], "COLOR", 5) == 0)
		{
			bpp = 3;
		}
		else
		{
			DBG_PRINTF_ERR("ERROR : Invalid COLOR/MONO parameter!\n");
			return -1;
		}

		if (argc == 3) {
			num_compressors= atoi(argv[2]);
		}

		if (argc == 4) {
			num_compressors= atoi(argv[2]);
			loop_count= atoi(argv[3]);
		}

		if (argc == 5) {
			num_compressors= atoi(argv[2]);
			loop_count= atoi(argv[3]);
			strip_height = atoi(argv[4]);
		}

	} else {
		DBG_PRINTF_ERR("usage mode files : program  [COLOR|MONO] <num_compressors> <loop_count> <strip_height>\n");
		return 0;
	}

	if (num_compressors < 1)
	{
		DBG_PRINTF_ERR("ERROR : Invalid num_compressor value = %d\n", num_compressors);
		return -1;
	}

	if (num_compressors > NUM_COMPRESSORS)
	{
		DBG_PRINTF_ERR("ERROR : Only supports up to %d compressors\n", NUM_COMPRESSORS);
		return -1;
	}

	struct strip_blob_s strip_blob[NUM_COMPRESSORS][NUM_STRIPS];
	for (ncomp=0;ncomp<num_compressors;ncomp++)
	{
		memset(&page_blob[ncomp], 0, sizeof(struct page_blob_s));

		int i;
		for (i=0; i<NUM_STRIPS; i++)
		{
			memset(&strip_blob[ncomp][i], 0, sizeof(struct strip_blob_s));
			strip_blob[ncomp][i].lines = strip_height;

			uint32_t size = strip_width * bpp * strip_blob[ncomp][i].lines;

			strip_blob[ncomp][i].bb = dma_buffer_malloc( 0, size );
			strip_blob[ncomp][i].bb->freeFunc = bb_dont_free;
		}

		//printf("  - Create page[%d]...\n", ncomp);
		page_adaptor[ncomp] = construct_jpeg_jpeghw_page_adaptor();
		strip_adaptor[ncomp] = construct_jpeg_jpeghw_strip_adaptor();
	}

	int test;
	int test_cases = 5;
	for (test=0; test<test_cases; test++)
	{
		int quality = 90;
		uint8_t fill_chars_used = 0;
		uint8_t fill_char[4] = {0, 0, 0, 0};

		printf("Starting test %d - %s using %d compressor(s) ",
				test+1,
				(bpp == 1 ? "MONO" : "COLOR"),
				num_compressors);

		switch(test)
		{
			case 0:
				fill_chars_used = 1;
				fill_char[0] = 0xFF;
				break;

			case 1:
				fill_chars_used = 1;
				fill_char[0] = 0x00;
				break;

			case 2:
				fill_chars_used = 2;
				fill_char[0] = 0xA5;
				fill_char[1] = 0x5A;
				break;

			case 3:
				fill_chars_used = 4;
				fill_char[0] = 0xAA;
				fill_char[1] = 0x55;
				fill_char[2] = 0xA5;
				fill_char[3] = 0x5A;
				break;

			case 4:
				fill_chars_used = 0;
				break;

			default:
				DBG_PRINTF_ERR("ERROR: invalid test case");
				return -1;
		}

		if (fill_chars_used > 0)
		{
			int i;

			printf("and fill chars [ ");
			for (i=0; i<fill_chars_used; i++)
			{
				printf("0x%x ", fill_char[i]);
			}
			printf("]...\n");
		}
		else
		{
			printf("sequential data (0x00-0xFF)...\n");
		}

		for (ncomp=0;ncomp<num_compressors;ncomp++)
		{
			//printf("  - Setting up data buffers[%d]...\n", ncomp);
			memset(&page_blob[ncomp], 0, sizeof(struct page_blob_s));

			page_blob[ncomp].width = strip_width;
			page_blob[ncomp].height = strip_height * loop_count;
			page_blob[ncomp].bytes_per_pixel = bpp;
			page_blob[ncomp].quality = quality;

			int i;
			for (i=0; i<NUM_STRIPS; i++)
			{
				uint32_t size = strip_width * bpp * strip_blob[ncomp][i].lines;

				char *data = dma_buffer_mmap_forcpu(strip_blob[ncomp][i].bb);
				if (data)
				{
					if (fill_chars_used > 0)
					{
						int j, k;
						for (j=0, k=0; j<size; j++, k++)
						{
							if (k >= fill_chars_used)
							{
								k = 0;
							}

							data[j] = fill_char[k];
						}
					}
					else
					{
						int j;
						for (j=0; j<size; j++)
						{
							data[j] = (j % 0xFF);
						}
					}

					dma_buffer_unmmap_forcpu(strip_blob[ncomp][i].bb);
				}
			}
		}

		clock_t start_time = clock();

		for (ncomp=0;ncomp<num_compressors;ncomp++)
		{
			// setup new page
			//printf("  - Set up receive page[%d]...\n", ncomp);
			jpeg_recv_page((struct jpeg_page_adaptor *)page_adaptor[ncomp], (void *)&page_blob[ncomp]);
		}

		//printf("  - Sending strips...\n");
		uint32_t max_lines = strip_height * loop_count;
		cnt = 0;
		for (lines=0; lines<max_lines; )
		{
			int lines_in = strip_height;

			if ((lines + lines_in) > max_lines)
			{
				lines_in = max_lines - lines;
			}

			lines += lines_in;

			for (ncomp=0;ncomp<num_compressors;ncomp++)
			{
				struct strip_blob_s *strip_blob_ptr = &strip_blob[ncomp][cnt % NUM_STRIPS];

				strip_blob_ptr->lines_in = (lines_in < strip_blob_ptr->lines ? lines_in : strip_blob_ptr->lines);
				strip_blob_ptr->is_last_strip = lines >= max_lines ? true : false;

				jpeg_recv_strip((struct jpeg_page_adaptor *)page_adaptor[ncomp],
						(struct jpeg_strip_adaptor *)strip_adaptor[ncomp],
						strip_blob_ptr);
			}

			cnt++;
		}

		for (ncomp=0;ncomp<num_compressors;ncomp++)
		{
			//printf("  - Closing page[%d]...\n", ncomp);
			jpa_from_page_close((struct jpeg_page_adaptor *)page_adaptor[ncomp]);
		}

		clock_t end_time = clock();

		uint32_t pixs_compressed = 0;
		for (ncomp=0;ncomp<num_compressors;ncomp++)
		{
			//printf("  - Accumulate pixels[%d]...\n", ncomp);
			pixs_compressed += page_adaptor[ncomp]->compress_info.in_bytes;
		}

		double elapse_time_ms = end_time - start_time;
		elapse_time_ms = elapse_time_ms/1000;
		printf("Complete - Compressed %u pixels in %.2lf seconds (%.2lf MPix/sec)\n\n", pixs_compressed,
				elapse_time_ms/1000, ((double)(pixs_compressed/(elapse_time_ms/1000))/(1024*1024)));
	}

	for (ncomp=0;ncomp<num_compressors;ncomp++)
	{
		//printf("  - Destroy page[%d]...\n", ncomp);
		destroy_jpeg_jpeghw_strip_adaptor(strip_adaptor[ncomp]);
		destroy_jpeg_jpeghw_page_adaptor(page_adaptor[ncomp]);

		int i;
		for ( i = 0; i < NUM_STRIPS; i++ )
		{
			dma_buffer_freefunc( strip_blob[ncomp][i].bb );
		}
	}

	return 0;
}

/// test_speed() -- test the speed of the compressor
int main(int argc, char** argv) {
	memInitMemory(0, 1024 * 1024 * 128); // init memory pool uses dmaalloc.ko, implies 1 process for now.
	logger_init();  // controlled by /home/root/logger_config_file if present
	uio_lib_init(); // init uio uses /dev/uio*
	jpeghw_init();

	main_jpeghw_adaptor_strips(argc, argv);

	jpeghw_terminate();

	return 0;
}

// eof test_speed.c
